/*
 * Copyright (c) 1999 Thomas Grey All Rights Reserved.
 *
 * Thomas Grey grants you ("Licensee") a non-exclusive, royalty
 * free, license to use, modify and redistribute this software in
 * source and binary code form, provided that the following conditions 
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * This software is provided "AS IS," without a warranty of any
 * kind. ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND
 * WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY
 * EXCLUDED.  THOMAS GREY AND HIS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
 * MODIFYING OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO
 * EVENT WILL THOMAS GREY OR HIS LICENSORS BE LIABLE FOR ANY
 * LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE SOFTWARE, EVEN IF THOMAS GREY HAS BEEN
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 *
 * This software is not designed or intended for use in on-line
 * control of aircraft, air traffic, aircraft navigation or aircraft
 * communications; or in the design, construction, operation or
 * maintenance of any nuclear facility. Licensee represents and
 * warrants that it will not use or redistribute the Software for such
 * purposes.  
 */

import java.net.URL;
import java.net.MalformedURLException;
import java.io.InputStream;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.awt.image.IndexColorModel;
import java.util.StringTokenizer;

/**Abstract super-class from which all renderable objects must inherit. Classes to be used with the FORMAT parameter MUST use the FORMAT value prefixed with MDL and suffixed with Model. The format part MUST be lowercase.<p>
Eg PARAM NAME=FORMAT VALUE=quake2; the model class should be MDLquake2Model.class <p>
<I>Generally</I>: PARAM NAME=FORMAT VALUE=<I>somenewgame</I>; the model class should be called MDL<I>somenewgame</I>Model.class.
@author T.J.Grey <tjg1@ch.ic.ac.uk>
**/
public abstract class MDLModel implements Runnable{

	private Thread modelLoader;
//	BufferedInputStream modelStream;
	private boolean modelReady = false;
	static final int MDL_VER = 6;
	protected URL modelPath;
	private MDLModelTracker theBoss;
	private boolean modelError = false;
	private int[] loadframes;
	private int[] loadSkins;
	private int shiftY = 0;
// Methods for classes to use
/**Sets the ModelTracker which will be informed when readModelThread() returns.<b>Generally this method need not be overridden</b>.**/
	public void setModelTracker(MDLModelTracker boss){
	    theBoss = boss;
	}
	
/**Returns a string describing the frames loaded and the frames avalible. Used by MonitorWindow. <b>Generally this method need not be overridden</b>.**/
	public String getFramesString(){
	    return (getNumberOfFrames()+"/"+getTotalFrames());
	}
	
/**Returns a string describing the skins loaded and the skins avalible. Used by MonitorWindow. <b>Generally this method need not be overridden</b>.**/
	public String getSkinsString(){
	    return (getNumberOfSkins()+"/"+getTotalSkins());
	}

/**Returns true if an error has occured in the model. <b>Generally this should not be over-ridden</b>: use setModelError(boolean) instead.**/	
	public boolean getModelError(){
	    return modelError;
	}
	
/**Returns true if the model is ready to use. <b>Generally this should not be over-ridden as it will be handled automatically</b>.**/
	public boolean getModelReady(){
	    return modelReady;	
	}
	
/**Returns a string naming the author of the class and copyright etc etc. By all means override!**/
	public String getAuthorInfo(){
	    return("Super class Copyright T.J.Grey 1998");
	}
	
/**Returns a string naming the author of the class only- used to credit author in MDLHelpWindow. By all means override!**/
	public String getAuthorName(){
	    return("unknown");
	}
	
/**Called by MDLView or MDLMovie to cause the model to be loaded. <b>Generally this method should not be over-ridden</b>.
* Generally override the loadModelThread method to have your model loaded asyncronously.
*   @param path The document base of the applet
*   @param ldframes The value of the LOADFRAMES param
*   @param ldskins The value of the LOADSKINS param
*   @param shift The int value of the SHIFTY param
**/
	public void readModel(URL docBase,URL codeBase,String ldframes, String ldskins, int shift){
		modelReady = false;
		loadframes = intArrayFromString(ldframes,"LOADFRAMES");
		loadSkins = intArrayFromString(ldskins,"LOADSKINS");
		shiftY = shift;
		String mp = getParameter("MODELPATH");
		if (mp == null){
		    modelPath = docBase;
		}else{
		    try{
			modelPath = new URL(codeBase,mp);
		    }catch (MalformedURLException e){
			showStatus("Got this error: "+e+" : Check the MODELPATH is syntaxically correct");
			System.out.println("Got this error: "+e+" : Check the MODELPATH is syntaxically correct");
			throw new RuntimeException("Got this error: "+e+" : Check the MODELPATH is syntaxically correct");
		    }
		}
		modelLoader = new Thread(this, "MDLModelLoader");
		modelLoader.setPriority(Thread.MAX_PRIORITY);
		modelLoader.start();
	}
	
/** Called by MDLView/MDLMovie is their stop method is called. <b>Generally this should not be over-ridden.</b>**/
	public void stopCall(){
	    if (modelLoader != null && modelLoader.isAlive() ){
		modelLoader.stop();
		modelReady = false;
	    }
	}
	
/**<b>Do not touch</b> unless you are <b>SURE</b> you know what you are doing! Generally override the loadModelThread method to have your model loaded asyncronously.**/
	public void run(){
	    readModelThread(loadframes, loadSkins, shiftY);
	    modelReady = true;
	    theBoss.modelReady();
	    modelLoader = null;
	}
	
/**Returns the vertical shift in use via the SHIFTY parameter. <b>Generally this need not be over-ridden</b>.**/
	public int getYShift(){
	    return shiftY;
	}
// Absract methods
/** Return the version of the class. Return as a multiple of 100 eg. 413 is version 4.13**/
	public abstract int getVersion();
/**If the palette was specified as 'INTERNAL' then this method will be called. If you use an external palette then return null.**/
	public abstract IndexColorModel getInternalPalette();
/**This method must return the index of the current skin. What this REALLY means is upto you, eg For MDLquakeModel it returns the current skinGroup. However, the value must be useable in setCurrentSkin(int)**/
	public abstract int getCurrentSkinIndex();
/**This method selects a skin by its index- must be compatable with the value returned by getCurrentSkin().**/
	public abstract void setCurrentSkinIndex(int skin);
/**Returns the index of the current frame. The indexes MUST be in time order and the earliest frame must be 0. What this REALLY means is upto you, eg For MDLquakeModel it returns the current GroupFrame.**/
	public abstract int getCurrentFrameIndex();
/**Set the index of the current frame. Must be compatable with getCurrentFrameIndex(). 
The first frame MUST be accessable via setCurrentFrameIndex(0). @param fi The frame index to seek.**/
	public abstract void setCurrentFrameIndex(int fi);
/**Returns the total number of single frames of animation between two frame indices. ie, if I wanted to make
 a set of images for animating the model from <i>first</i> to <i>last</i> inclusive, how many pictures would I need. 
 For quake models some of the frames are infact groups of animation frames, this is a work around used by MDLMovie to allocate the right ammount of storage.
 @param first The index of the start frame
 @param last The index of the final frame.**/
	public abstract int getNumberOfAnimationFrames(int first, int last);
/**Returns the total number of frames which can be loaded for this model.**/
	public abstract int getTotalFrames();
/**Returns the total number of skins which can be loaded for this model.**/
	public abstract int getTotalSkins();
/**Must advance the model to the next animation frame such that all loaded frames may be seen eventually.**/
	public abstract void nextFrame();
/**Must move the model to the last animation frame- must complement nextFrame().**/
	public abstract void lastFrame();
/**Must advance the skin to the next skin avalible such that all useable skins can be visited eventually.**/
	public abstract void nextSkin();
/**Must retreat the skin to the last skin seen. Must be the complement of nextSkin().**/
	public abstract void lastSkin();
/** If there is a name assosiated with the current frame the return it. Otherwise return 'noname' or some such.**/
	public abstract String getFrameName();
/** Return the name of the model format eg. 'quake' or 'quake2'. Should be the class name without the MDL prefix and Model suffix.**/
	public abstract String getModelFormat();

/**Must return the radius of the bounding sphere of the model <b>for all frames</b>. Note that this value will be cached by the renderer (it 
will be read after model is read in!) so it can not change. For quake this is in the model file. 
For other models you may have to find it's value yourself. Remember, when calculating max radii from x*x+y*y+z*z, compare the squares
 and then square-root at the end!**/
	public abstract float getBoundingRadius();  // bounding radius
/** Must return the number of loaded skins**/
	public abstract int getNumberOfSkins();                // num skins
/**Must return the width of the skins in pixels.**/
	public abstract int getSkinWidth();                    // skin width
/**Must return the heigth of the skins in pixel.**/
	public abstract int getSkinHeight();                    // skin height
/** Must return the number of vertices in the model. Note that this value is cached by the renderer and so only read once, just after the model has been read.**/
	public abstract int getNumberOfVertices();
/** Must return the number of triangles in the model. Note that this value is cached by the renderer and so only read once, just after the model has been read.**/
	public abstract int getNumberOfTriangles();
/** Must return the number of loaded frames. Note that this value is cached by the renderer and so only read once, just after the model has been read.**/
	public abstract int getNumberOfFrames();
/**Return the index representing the colour of the indicated texel in the relevant palette.<p>
<b>Of all the methods this one must be as fast as possible as it is called hundereds of times pre frame</b>.
@param x The horizontal position on the skin.
@param y The vertical position on the skin.**/
	public abstract byte getTexel(short x, short y);
/**Return the vertex index (from 0 to numberOfVertices -1) of the given vertex of the given triangle. Called for every triangle as each frame is rendered.
@param triangleIndex The index of the triangle of interest (0 to (numberOfTriangles-1) )
@param vertex The vertex of the triangle- either 0, 1 or 2**/
	public abstract short getTriangleVertexIndex(int triangleIndex, int vertex);
/**Return the 3D x coordiante of the vertex of interest in the current animation frame. Positive X being the distance <b>from the model origin toward the viewer!</b><p>
ie. The greater X, the nearer the point is to the viewer [typically this would be -Z].
@param triangleIndex The index (from 0 to (numberOfVertices-1) ) of the vertex of interest.**/
	public abstract float getVertexX(int vertexIndex);
/**Return the 3D y coordiante of the vertex of interest in the current animation frame. Positive Y being the distance <b>from the model origin to the right as the model faces you.</b><p>
ie. The greater Y the further the point is to the right [typically this would be X].
@param vertexIndex The index (from 0 to (numberOfTriangles-1) ) of the vertex of interest.**/
	public abstract float getVertexY(int vertexIndex);
/**Return the 3D z coordiante of the vertex of interest in the current animation frame. Positive Z being the distance <b>from the model origin vertically up.</b><p>
ie. The greater Z the higher up the point is [typically this would be Y].
@param vertexIndex The index (from 0 to (numberOfTriangles-1) ) of the vertex of interest.**/
	public abstract float getVertexZ(int vertexIndex);
/**Return true is triangles facing the viewer list their vertices clockwise. Return false if triagles facing the viewer list their vertices anticlockwise. 
This value may be cached by the renderer and read only once.**/
	public abstract boolean getClockwiseCull();
/**Return the horizontal position of the specified vertex on the skin. Must be less than skin width.
@param triangle The triangle being rendered- sometimes required for vertices on seams.
@param vertexIndex The index of the vertex in the triangle- 0, 1 or 2**/
	public abstract short getSkinVertexS(int triangle, int vertexIndex);
/**Return the vertical position of the specified vertex on the skin. Must be less than skin height.
@param triangle The triangle being rendered- sometimes required for vertices on seams.
@param vertexIndex The index of the vertex in the triangle- 0, 1 or 2**/
	public abstract short getSkinVertexT(int triangle, int vertexIndex);
/**Get the ammount of time in milliseconds the current animation frame should be visible for.**/
	public abstract int getFrameDelayMillis();

// Methods for sub-classes
/**Put the code which loads the model here. This method will run asyronously. Subclasses should ensure that model loading is as complete as 
possible (ie. errors may cause an early abort) when this method returns.
@PARAM modelURL A URL object for the model- use openstream().
@PARAM loadFrames An array of int representing the indices of the frames to be loaded- the array is num_frames long. if (loadframes[0] == -1) then load all frames, 
if (loadframes[0] == -2) then load NO frames.
@PARAM loadSkins An array of ints, num_skins long, representing the index numbers of the skins to load
@PARAM Yshift Shift all points by this ammount in the vertical direction.**/
	protected abstract void readModelThread(int[] loadFrames, int[] loadSkins, int Yshift);
/**Fetchs a parameter from the applet. Used to allow direct access to MODELPATH etc.
@param parameterName The name of the parameter eg. "MODELPATH"
@returns The value or the parameter or null if it was not found
**/	
	protected String getParameter(String parameterName){
	    return theBoss.getParameter(parameterName);
	}
/**Allows subclasses to indicate an error has occured whilst trying to load the model.
@param state True to indicate an error has occured, false to turn the flag off.**/
	protected void setModelError(boolean state){
	    modelError = state;
	}
/**Send a string to the browser status line indicating progress. Call this often enough that users do not loose confidence that something is happening!**/
	protected void showStatus(String msg){
	    theBoss.showStatus(msg);
	}
	
/**Send a percentage completion to be shown in the loading screen. Generally this should indicate overall progress**/
	protected void showPercentage(int percentage){
	    theBoss.showPercentage(percentage);
	}
/**Cause the modelLoading thread to yield. Tricky to use this- call to often and modelLoading will crawl, call too occationally and the browser will freeze up.
Generally I would call this quite often.**/	
	protected void yield(){
	    modelLoader.yield();
	}
/**Fetch an InputStream for the following filename specified relative to the MODELPATH (or documentbase)**/
	protected InputStream getStreamForFile(String filename) throws MalformedURLException, IOException {
	    URL modelURL = new URL(modelPath,filename);
	    return modelURL.openStream();
	}
/**Receive and process unused keystroke. When a user press a key, the event is proceeded by MDLView, if it is not bound to any event eg. change 
skin etc then it will be passed to the model via this method. Key is the ascii code of the key- as standard in java. Return true if you want the renderer
to be asked to re-render the model. This method can be used to implement model format specific functions.**/	
	public boolean keyPressed(int key){
	    return false;
	}
/**Call to get the info from the EXTRA parameter. The EXTRA parameter is designed to allow format classes to have custom parameters. The applet 
will pass the contents of the EXTRA parameter as an array of strings through this method. Note that each String in the array represents a the contents of the square brackets used in the EXTRA parameter 
and that upto 30 sub-parameters are allowed. If no EXTRA parameter is set then null will be passed.<p>
eg PARAM NAME=EXTRA VALUE="[LOADWEAPONS=3,1,2,5][STARTWEAPON=5]" will give an array of length two containing "LOADWEAPONS=3,1,2,5" and "STARTWEAPON=5"**/
	protected String[] getExtraInfo(){
	    String param = theBoss.getExtraParameter();
	    if (param == null){
		return null;
	    }
	    String[] temp = new String[30];
	    int start;
	    int stop;
	    boolean repeat = true;
	    int num_strings = 0;
	    while (repeat){
		start = param.indexOf("[");
		stop = param.indexOf("]");
		if (start == -1 || stop == -1){
		    repeat = false;
		}else if (start > stop){
		    param= param.substring(stop+1);
		    repeat = true;
		}else{
		    temp[num_strings] = param.substring(start+1,stop);
		    num_strings++;
		    param = param.substring(stop+1);
		    repeat = true;
		}
	    }
	    String[] info = new String[num_strings];
	    for (int k=0; k<num_strings;k++){
		info[k] = temp[k];
	    }
	    return info;
	}
/**Converts a string in the LOADFRAMES format to an array of ints representing the indices in the array- see readModelThread info above.
@PARAM astring The string to be interpreted.
@PARAM message The name of the field (eg loadframes) to be put in any error messages**/
	protected int[] intArrayFromString(String astring, String message){
	    if (astring == null){
		int[] returnArray = new int[1];
		returnArray[0] = -1;
		return returnArray;	    
	    }
	    StringTokenizer st = new StringTokenizer(astring,",",false);
	// get number of frames
	    String numframes = st.nextToken();
	    int numberFrames;
	    try{
		numberFrames = (Integer.decode(numframes)).intValue();
	    }catch (NumberFormatException e){
		System.out.println("MDLView: Error in "+message+" string... loading all frames/skins");
		int[] returnArray = new int[1];
		returnArray[0] = -1;
		return returnArray;
	    }
	    if (numberFrames == 0){
		int[] returnArray = new int[1];
		returnArray[0] = (-2);
		return returnArray;
	    }
    // If there are no item indexs specifed but more than one item then inteprete as first N items
	    if (numberFrames > 0 && !(st.hasMoreTokens()) ){
		int[] returnArray = new int[numberFrames];
		for (int n=0; n < numberFrames; n++){
		    returnArray[n] = n;
		}
		return returnArray;
	    }
	    int[] tempArray = new int[numberFrames];
	    int count = 0;
	    while (st.hasMoreTokens()) {
		String token = st.nextToken();
//		System.out.println("TOKEN: "+token);
// Is this of the form x-y or x,?
		if(token.indexOf("-") >= 0){
	    // get all the bits but first an idiot trap
		    if (token.indexOf("-") == 0 || token.indexOf("-") == token.length()){
			System.out.println("MDLView: Error LOADFRAMES token starts/ends with hyphen not a number");
		    }else{
			String startIndexStr = token.substring(0,token.lastIndexOf("-"));
			String endIndexStr = token.substring(token.lastIndexOf("-")+1, token.length());
//			System.out.println("STARTSTR "+startIndexStr);
//			System.out.println("ENDSTR "+endIndexStr);
			int endIndex;
			int startIndex;
			try{
			    startIndex = (Integer.decode(startIndexStr)).intValue();
			    endIndex = (Integer.decode(endIndexStr)).intValue();
			    if (startIndex < endIndex && startIndex >= 0 && endIndex >= 0){
				for (int i = startIndex; i <= endIndex; i++){
				    tempArray[count] = i;
//			    System.out.println("MARK FRAME: "+i);
				    count++;
				}
			    }else{
				System.out.println("MDLView: "+message+", bad start/end values, "+startIndex+", "+endIndex+", items may be missing");
			    }
			}catch (NumberFormatException e){
			    System.out.println("MDLView: Error in "+message+" items may be missing");
			}
		    }
		}else{
	    // nice simple single specifier
		    try{
			int frameNum = (Integer.decode(token)).intValue();
			tempArray[count] = frameNum;
			count++;
		    }catch (NumberFormatException e){
			System.out.println("MDLView: Error in "+message+" items may be missing");
		    }
		}
	// Off to get next token	
	    }
	    // return finished array ensure it is full
	    int[] returnArray = new int[count];
	    for (int i = 0; i < count; i++){
		returnArray[i] = tempArray[i];
	    }
	    return returnArray;
	}

/**Converts 4 UNSIGNED bytes to one integer. Should be suitable for reading ints from model files. Read in as four bytes then convert here 
<b>nb.</b>For some formats you may need to reverse the order of the bytes so the first byte read is byte[3].**/
    protected int intFromFourBytes(byte[] intAsBytes){
	int returnInt = 0;
	returnInt = returnInt | ( (intAsBytes[0] & 255) << 24);
	returnInt = returnInt | ( (intAsBytes[1] & 255) << 16);
	returnInt = returnInt | ( (intAsBytes[2] & 255) << 8);
	returnInt = returnInt | (intAsBytes[3] & 255);
	return returnInt;
    }
/**Converts 4 UNSIGNED bytes to one float. Should be suitable for reading ints from model files- read in as four bytes then convert here- see above.**/
    protected float floatFromFourBytes(byte[] floatAsBytes){
/** Converts four bytes into a float (uses intFromFourBytes) **/
	float returnFloat = 0;
	int asInt;

	asInt = intFromFourBytes(floatAsBytes);
	returnFloat = java.lang.Float.intBitsToFloat(asInt);
	return returnFloat;
    }
/**Reads an int from the inputstream. This method assumes that the first byte read is the <b>least</b> significant byte.**/
    protected int readInt(InputStream stream) throws IOException{
	int i;
	byte[] fourBytes = new byte[4];
	for (i=0; i<4;i++){
	    fourBytes[3-i]=(byte)stream.read();
	}
	return intFromFourBytes(fourBytes);
    }
/**Reads a float from the inputstream. This method assumes that the first byte read is the <b>least</b> significant byte.**/
    protected float readFloat(InputStream stream) throws IOException{
	int i;
	byte[] fourBytes = new byte[4];
	for (i=0; i<4;i++){
	    fourBytes[3-i]=(byte)stream.read();
	}
	return floatFromFourBytes(fourBytes);
    }

/**Checks if the value of number is negative, where field is a string describing the number eg "model version". 
If so then returns true, prints a message and if errorNotWarning is true, sets the error flag to true and stops the applet.**/
    protected boolean checkNeg(boolean errorNotWarning, float number, String field){
	if (number < 0){
	    if (errorNotWarning){
		modelError = true;
		System.out.println("MDLView: Error "+field+" is negative");
		throw new RuntimeException("MDLView: Error "+field+" is negative");
	    }else{
		System.out.println("MDLView: Warning... "+field+" is negative");
	    }
	    return true;
	}
	return false;
    }
/**Check if the value of number is a short. Field is a string describing the number eg "number of vertices". 
If so then returns true and stops the applet.**/
    protected boolean checkShort(int number, String field){
	if (number > Short.MAX_VALUE){
	    modelError = true;
	    System.out.println("MDLView: Error "+field+" greater than "+Short.MAX_VALUE);
	    throw new RuntimeException("MDLView: Error "+field+" greater than "+Short.MAX_VALUE);
	}
	return false;
    }    
}