/*
 * OpenMod DynLevelInfo Creation Utility (aka. "Differ")
 * Written By: James Abbatiello (abbeyj@frag.com)
 * 
 * OpenMod is licensed under the terms of the GNU Library General Public
 * License.  More information on OpenMod and the terms of it's license may
 * be found in it's readme file, located in this game's "SYSTEM" directory.
 *  
 * "Differ" is licensed under the terms of the GNU General Public License.
 * Copyright (C) 1998  James Abbatiello (abbeyj@frag.com)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * The GNU General Public License may also be found at http://www.gnu.org
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <iostream.h>

// one element in a linked list
template <class T>
class linkedlistelem
{
public:
	T *elem;
	linkedlistelem<T> *prev;
	linkedlistelem<T> *next;

	linkedlistelem(T *e) : elem(e), prev(NULL), next(NULL) {}
	~linkedlistelem<T>() {delete elem;}
};

// a linked list
template <class T>
class linkedlist
{
public:
	linkedlistelem<T> *head, *tail;

	linkedlist() : head(NULL), tail(NULL) {}
	~linkedlist()
	{
		linkedlistelem<T> *walk, *nextwalk;
		walk = head;
		while (walk)
		{
			nextwalk = walk->next;
			delete walk;
			walk = nextwalk;
		}
	}
	void addelem(T *elem)
	{
		linkedlistelem<T> *newelem = new linkedlistelem<T>(elem);
		newelem->next = NULL;
		newelem->prev = tail;
		if (!head)
			head = newelem;
		if (tail)
			tail->next = newelem;
		tail = newelem;
	}
	void removeelem(linkedlistelem<T> *elem)
	{
		if (elem->prev)
			elem->prev->next = elem->next;
		else		// elem == head
			head = elem->next;
		if (elem->next)
			elem->next->prev = elem->prev;
		else		// elem == tail
			tail = elem->prev;
		delete elem;
	}
};

#if 0
class property
{
public:
	char name[64], value[64];
	property *next;

	property::property(char *n, char *v, property *n2) : next(n2)
	{
		strncpy(name, n, sizeof(name));
		strncpy(value, v, sizeof(value));
	}

	int property::operator==(property &other)
	{
		return (stricmp(name, other.name) == 0 && stricmp(value, other.value) == 0);
	}
	int property::operator!=(property &other)
	{
		return (!operator==(other));
	}
};


class entity
{
/*
	char class[64];
	char name[64];
	int physics;
	vector location;
	vector rotation;
	char ownerteam[64];
	char bsuperheal[64];
	int healingamount;
	int armorabsorption;
	int charge;
	int absorptionpriority;
	float respawntime;
	int ammoamount;
*/
public:
	property *head;
	entity *next;
	entity *prev;
	int ordinal;

	entity::entity(int o) : head(NULL), ordinal(o) {}
	entity::~entity()
	{
		property *walk, *newwalk;
		walk = head;
		while (walk)
		{
			newwalk = walk->next;
			delete walk;
			walk = newwalk;
		}
	}
	int entity::operator==(entity &other)
	{
		property *w1, *w2;
		w1 = head;
		w2 = other.head;
		while (w1 && w2)
		{
			if (*w1 != *w2)
			{
				return (false);
			}
			w1 = w1->next;
			w2 = w2->next;
		}
		if (w1 || w2)
			return (false);
		return(true);
	}
	int entity::operator!=(entity &other)
	{
		return(!operator==(other));
	}
	void entity::addproperty(char *name, char *value)
	{
		property *add = new property(name, value, head);
		head = add;
	}
	char *entity::getvalue(char *name)
	{
		property *walk = head;
		while (walk)
		{
			if (stricmp(walk->name, name) == 0)
				return (walk->value);
			walk = walk->next;
		}
		return (NULL);
	}

};
#endif

// one property (each entity has multiple properties)
class property
{
public:
	char name[64], value[64];

	property::property(char *n, char *v)
	{
		strncpy(name, n, sizeof(name));
		strncpy(value, v, sizeof(value));
	}

	int property::operator==(property &other)
	{
		return (stricmp(name, other.name) == 0 && stricmp(value, other.value) == 0);
	}
	int property::operator!=(property &other)
	{
		return (!operator==(other));
	}
};


// an entity
class entity
{
public:
	linkedlist<property> props;
	int ordinal;

	entity::entity(int o) :  ordinal(o) {}
	entity::~entity() {}
	int entity::operator==(entity &other)
	{
		linkedlistelem<property> *w1, *w2;
		w1 = props.head;
		w2 = other.props.head;
		// look thru the properties
		while (w1 && w2)
		{
			// if any two are not equal, we are not equal
			if (*w1->elem != *w2->elem)
			{
				return (false);
			}
			w1 = w1->next;
			w2 = w2->next;
		}
		// if there are a different number, then we are not equal
		if (w1 || w2)
			return (false);
		return(true);
	}
	int entity::operator!=(entity &other)
	{
		return(!operator==(other));
	}
	void entity::addproperty(char *name, char *value)
	{
		props.addelem(new property(name, value));
//		property *add = new property(name, value, head);
//		head = add;
	}
	char *entity::getvalue(char *name)
	{
		linkedlistelem<property> *walk = props.head;
		while (walk)
		{
			if (stricmp(walk->elem->name, name) == 0)
				return (walk->elem->value);
			walk = walk->next;
		}
		return (NULL);
	}

};



/*
		log("DIFF<START_______="$i$">");
		log("DIFF<VERSION_____="$DiffVersion$">");

		log("DIFF<CLASS_______="$A.Class$">");
		log("DIFF<NAME________="$A.Name$">");
		log("DIFF<PHYSICS_____="$GetEnum(enum'EPhysics', A.Physics)$">");
		log("DIFF<LOCATION____="$A.Location$">");
		log("DIFF<ROTATION____="$A.Rotation$">");
		
		
		if (A.IsA('PlayerStart'))
		{
			log("DIFF<PS_OWNERTEAM="$PlayerStart(A).ownerTeam$">");
			log("DIFF<PS_TEAMNUMBR="$PlayerStart(A).TeamNumber$">");
		}
		
		if (A.IsA('Health'))
		{
			log("DIFF<HL_SUPERHEAL="$Health(A).bSuperHeal$">");
			log("DIFF<HL_HEAL_AMT_="$Health(A).HealingAmount$">");
		}
		
		if (A.IsA('Inventory'))
		{
			log("DIFF<IV_AR_ABSORB="$Inventory(A).ArmorAbsorption$">");
			log("DIFF<IV_CHARGE___="$Inventory(A).Charge$">");
			log("DIFF<IV_ABSORB_PR="$Inventory(A).AbsorptionPriority$">");
			log("DIFF<IV_RESPNTIME="$Inventory(A).RespawnTime$">");
		}
		
		if (A.IsA('Ammo'))
		{
			log("DIFF<AM_AMMOAMNT_="$Ammo(A).AmmoAmount$">");
		}
		
		log("DIFF<END_________="$i$">");
		i++;
*/


void Usage()
{
	printf("Usage: differ <oldinputlog> <newinputlog>\n");
}

void Error(char *s)
{
	printf("Error: %s\n", s);
	exit(1);
}

int main(int argc, char *argv[])
{
	linkedlist<entity> entities[2];
//	entity *head[2] = {NULL, NULL};
	class linkedlistelem<class entity> *walk[2] = {NULL, NULL}, *nwalk=NULL;
	entity *ent = NULL;
	class linkedlistelem<class property> *llprop;
	int i;
	FILE *fp;
	char buff[256], sscriptlog[256], sdiff[256], name[256], value[256];

	if (argc != 3)
	{
		Usage();
		return (1);
	}

	// read everything in
	for (i=0; i<=1; i++)
	{
		fp = fopen (argv[i+1], "r");
		if (!fp)
			Error("opening input");

		while (true)
		{
//			if (!fscanf(fp, "%1024[^\n]\n", buff))
			if (!fgets(buff, sizeof(buff), fp))
				break;
			// ack, its like perl...
			sscanf(buff, "%s %[^<]<%[^=]=%[^>]> ", &sscriptlog, &sdiff, &name, &value);
			if (stricmp(sdiff, "DIFF") != 0)
				continue;	// not a diff line

			if (stricmp(name, "_START") == 0)
			{
				ent = new entity(atoi(value));
			} else if (stricmp(name, "_END") == 0)
			{
				entities[i].addelem(ent);
				ent = NULL;
			} else if (ent)
			{
				ent->addproperty(name, value);
			}
		}
		fclose(fp);
	}

	walk[0] = entities[0].head;
	while (walk[0])
	{
		walk[1] = entities[1].head;
		nwalk = walk[0]->next;
		while (walk[1])
		{
			if (*walk[0]->elem == *walk[1]->elem)
			{
				entities[0].removeelem(walk[0]);
				entities[1].removeelem(walk[1]);
				break;
			}
			walk[1] = walk[1]->next;
		}
		walk[0] = nwalk;
	}

	fp = fopen("diff.uc", "w");
	if (!fp)
		return(1);


	fprintf(fp, "//=============================================================================\n");
	fprintf(fp, "// DiffInfo.\n");
	fprintf(fp, "//=============================================================================\n");
	fprintf(fp, "class DiffInfo expands CTFLevelInfo;\n");
	fprintf(fp, "\n");
	fprintf(fp, "function bool InitializeLevel()\n");
	fprintf(fp, "{\n");
	fprintf(fp, "\tlocal Actor A;\n");
	fprintf(fp, "\n");
	fprintf(fp, "\tforeach AllActors(class'Actor', A)\n");
	fprintf(fp, "\t{\n");
	i = 0;
	walk[0] = entities[0].head;
	while (walk[0])
	{
		fprintf(fp, "\t\tif (A.Name == '%s')\n", walk[0]->elem->getvalue("_NAME"));
		fprintf(fp, "\t\t\tA.Destroy();\n");
//		fprintf(fp, "Remove entity %d: %s\n", walk[0]->elem->ordinal, walk[0]->elem->getvalue("NAME________"));
		i++;
		walk[0] = walk[0]->next;
	}
	fprintf(fp, "\t}\n");

//	fprintf(fp, "%d entities to be removed\n", i);
	i = 0;
	walk[1] = entities[1].head;
	while (walk[1])
	{
		fprintf(fp, "\n");
		fprintf(fp, "\tA = Spawn(class'%s');\n", walk[1]->elem->getvalue("_CLASS"));
		llprop = walk[1]->elem->props.head;
		while (llprop)
		{
			if (llprop->elem->name[0] != '_')
			{
				if (llprop->elem->value[0] == '$')
				{
					fprintf(fp, "\t%s(A).Set%s(%s);\n", 
								walk[1]->elem->getvalue("_CLASS"),
								llprop->elem->name,
								llprop->elem->value + 1);
				}
				else
				{
					fprintf(fp, "\t%s(A).%s = %s;\n", 
								walk[1]->elem->getvalue("_CLASS"),
								llprop->elem->name,
								llprop->elem->value);
				}

			}
			llprop = llprop->next;
		}
		i++;
		walk[1] = walk[1]->next;
	}

	fprintf(fp, "\n");
	fprintf(fp, "\treturn true;\n");
	fprintf(fp, "}\n");

//	fprintf(fp, "%d entities to be added\n", i);
	fclose(fp);
	getchar();	
	return (0);
}