//=============================================================================
// OpenModGameInfo.
//
// OpenMod is licensed under the terms of the GNU Library General Public
// License.  More information on OpenMod and the terms of it's license may
// be found in it's readme file, located in this game's "SYSTEM" directory.
//=============================================================================
class OpenModGameInfo expands GameInfo;


var() bool				bNoTeamStartFind;	// If this is set, OpenMod will not automatically
                                            // designate team spawn points

const MAX_CLASSES = 32;
var() name				AcceptedDLIClasses[32];		// What are the highest DynLevelInfo classes that
                                            		// should be accepted?  (Default = DynDMLevelInfo)

var	DynMusicInfo		MusicInfo;			// Information on the game's current music
var LevelPrefsInfo		LevelPrefs;

var name				FlagFinderClasses[8];
const MAX_FLAG_FINDERS = 8;

const FLAGNUM_FIRST		= 0;
const FLAGNUM_BLUE		= 0;
const FLAGNUM_GREEN		= 1;
const FLAGNUM_RED		= 2;
const FLAGNUM_YELLOW	= 3;
const FLAGNUM_LAST		= 3;
var class<Actor>		FlagClasses[4];
var class<Actor>		FlagBaseClasses[4];
var string				TeamNameStrings[4];
var name				TeamNames[4];
var byte				TeamIndexes[4];

var class<Actor>		DefaultItems[32];

// Ob1: This ensures that there will only be one GRI in game
//   by preventing one from being genereated when the class is
//   spawned.
function PreBeginPlay()
{
	// no setup needed for a helper game info class
}

function DynInfo LoadSingleClass( class<DynInfo> LoadClass, optional bool bDontInitialize )
{
	local DynInfo tmpDynInfo;
	
	if ( LoadClass != None )
	{
		log(class$": LoadSingleClass(): LoadClass="$LoadClass);
		
		tmpDynInfo = spawn(LoadClass);
		
		if ( tmpDynInfo != None )
			log(class$": LoadSingleClass(): Loaded, name="$tmpDynInfo.name);
		
		tmpDynInfo.SetOwner(self);
		
		if ( !bDontInitialize )
		{
			tmpDynInfo.PreInitialize();
			tmpDynInfo.Initialize();
		}
		
		return tmpDynInfo;
	}
	
	return None;
}


function bool LoadOMI( OpenModInfo OMI,
                       optional class OnlyLoad,
                       optional bool bOnlyLoadExactMatch,
                       optional bool bLoadGenericOnError)
{
	local int				i;
	local DynInfo			tmpDynInfo;
	local bool				bSuccessful;
	
	if ( OMI == None )
	{
		log("OpenMod: OpenModGameInfo: LoadOMI(): ERROR: OMI==None, function aborting.");
		return false;
	}
	
	
	for (i=0;i<MAX_CLASSES;i++)
	{
		if ( OnlyLoad == None ||
		     OnlyLoad == OMI.AvailableInfo[i] ||
		     ( !bOnlyLoadExactMatch && ClassIsChildOf(OnlyLoad, OMI.AvailableInfo[i]) ) )
		{
			if ( OMI.AvailableInfo[i] != None )
			{
				log(class$": LoadOMI(): Loading class="$OMI.AvailableInfo[i]);
				tmpDynInfo = spawn(OMI.AvailableInfo[i]);
				tmpDynInfo.SetOwner(self);
				
				tmpDynInfo.PreInitialize();
				tmpDynInfo.Initialize();
				bSuccessful = true;
			}
		}
	}
	
	return bSuccessful;
}


function OpenModInfo LoadFlagFinder()
{
	local int i, j;
	local OpenModInfo MyLevelInfo;
	local DynInfo tmpFlagFinder;

	for (i=0; i<MAX_FLAG_FINDERS; i++)
	{
		if ( FlagFinderClasses[i] != '' ||
		     FlagFinderClasses[i] != 'None' )
		{
			MyLevelInfo = Super(ObjectLoader).LoadObject( string(FlagFinderClasses[i]) );
			if ( MyLevelInfo != None )
			{
				for (j=0; j<MAX_CLASSES; j++)
				{
					if ( MyLevelInfo.AvailableInfo[j] != None )
					{
						log(class$": LoadLevelData(): Loading Flag Finder Module, class="$MyLevelInfo.AvailableInfo[j]);
						tmpFlagFinder = LoadSingleClass( MyLevelInfo.AvailableInfo[j], true );
						//tmpFlagFinder = spawn(MyLevelInfo.AvailableInfo[j], self);
						if ( tmpFlagFinder != None )
						{
							if ( tmpFlagFinder.IsA('DynLevelInfo') )
							{
								//log(class$": LoadLevelData(): Calling: j="$j$"  i="$i);
								if ( DynLevelInfo(tmpFlagFinder).IsValid() != false )
								{
									//log(class$": LoadLevelData(): Success: j="$j$"  i="$i);
									tmpFlagFinder.Destroy();
									return MyLevelInfo;
								} else {
									//log(class$": LoadLevelData(): Failure: j="$j$"  i="$i);
									//log(class$": LoadLevelData(): ERROR: IsValid() returned false");
								}
							} else {
								log(class$": LoadLevelData(): ERROR: tmpFlagFinder isn't a DynLevelInfo");
							}
						} else {
							log(class$": LoadLevelData(): ERROR: Unable to load class="$MyLevelInfo.AvailableInfo[j]);
							//
						}
					}
				}
			}
		}
	}
	
	return None;
}


function bool LoadLevelData()
{
	local OpenModInfo			MyLevelInfo;
	local DynInfo				tmpDynInfo[32];
	local int i, j;

	MyLevelInfo = LoadFlagFinder();

	if ( MyLevelInfo == None )
	{
		MyLevelInfo = Super(ObjectLoader).LoadObject(Level.Title);
	} else {
		log(class$": LoadLevelData(): bUseFlagFinder=true");
	}
		
	if ( MyLevelInfo == None )
	{
		log(class$": LoadLevelData(): WARNING: MyLevelInfo==None, looking for existing data");
		return LoadGenericLevelData();
	}
	
	j=0;
	for (i=0;i<MAX_CLASSES;i++)
	{
		//if (MyLevelInfo.AvailableInfo[i] != None)
		if (ClassIsChildOf(MyLevelInfo.AvailableInfo[i], class'DynLevelInfo'))
		{
			log("OpenMod: OpenModGameInfo: LoadLevelData(): Spawning Info: "$i$", "$MyLevelInfo.AvailableInfo[i]);
			
			tmpDynInfo[j] = spawn(MyLevelInfo.AvailableInfo[i]);
			if (tmpDynInfo[j] == none)
				log("OpenMod: OpenModGameInfo: LoadLevelData(): WARNING, "$MyLevelInfo.AvailableInfo[i]$" failed to spawn.");
			j++;
		}
	}

	if ( j == 0 )
	{
		log(class$": WARNING: No level data files found, looking for existing data.");
		return LoadGenericLevelData();
	}
	
	for (i=0; i<j; i++)
	{
		if (IsOfClasses(tmpDynInfo[i], AcceptedDLIClasses) && tmpDynInfo[i] != None)
		{
			log("OpenModGameInfo: LoadLevelData(): Calling Initialize() for: "$i$", "$tmpDynInfo[i].class);
			tmpDynInfo[i].SetOwner(self);
			tmpDynInfo[i].PreInitialize();
			tmpDynInfo[i].Initialize();
		}
	}
		
	return true;
}


function bool LoadGenericLevelData()
{
	local OpenModInfo MyLevelInfo;
	
	MyLevelInfo = spawn(class'OpenModInfo');
	if ( MyLevelInfo == None )
	{
		log(class$": LoadGenericLevelData(): ERROR: MyLevelInfo==None, function aborting");
		return false;
	}
	
	// DLL: A "DefaultLevelInfo" class will soon be added.
	MyLevelInfo.AvailableInfo[0] = class'DynCTFLevelInfo';
	LoadOMI(MyLevelInfo);
	
	return true;
}


// Sees if the specified actor is a member of any of the ClassNames[] array
function bool IsOfClasses(Actor tmpAct, name ClassNames[32])
{
	local int i;
	
	//log("OpenMod: OpenModGameInfo: IsOfClasses(): Testing class="$tmpAct.class);
	
	for (i=0; i<32; i++)
	{
		if (tmpAct.IsA(ClassNames[i]))
		{
			//log("OpenMod: OpenModGameInfo: IsOfClasses(): Found match at "$i);
			return true;
		}
	}
	
	return false;
}


// Strips a string of all spaces ("Hello World" becomes "HelloWorld")
function string StrStripSpaces( string str )
{
	local byte ch;
	local string outstr;
	local int pos, length;
	
	length = len(str);			// Get the string's length
	
	for (pos=0; pos<length; pos++)
	{
		ch = asc( mid(str, pos, 1) );
		
		if (ch != EInputKey.IK_Space) {
			outstr = outstr $ chr(ch);
			//log("SSS: outstr="$outstr$"  ch="$chr(ch));
		} else {
			//log("SSS: skipping");
		}	
	}
	
	return outstr;
}


function NavigationPoint FindPlayerStartSimple()
{
	local PlayerStart tmpPS;
	local int i, selection;
	
	foreach AllActors( class'PlayerStart', tmpPS )
	{
		i++;
	}
	
	Selection = RandRange(0, i);
	i = 0;
	foreach AllActors( class'PlayerStart', tmpPS )
	{
		if ( i == Selection )
			return tmpPS;
		else
			i++;
	}
}


function SetMusicVars( out string NewMusicStr,
                       out music NewMusic,
                       out byte NewMusicSection,
                       out EMusicTransition NewMusicTransition,
                       DynMusicInfo GetMusicFrom,
                       name MusicStyle )
{
	switch ( MusicStyle )
	{
		case 'Default':
			NewMusicStr = GetMusicFrom.DefaultMusic_MusicStr;
			NewMusic = GetMusicFrom.DefaultMusic_Music;
			NewMusicSection = GetMusicFrom.DefaultMusic_Section;
			NewMusicTransition = GetMusicFrom.DefaultMusic_Transition;
			break;
		case 'Suspense':
			NewMusicStr = GetMusicFrom.SuspenseMusic_MusicStr;
			NewMusic = GetMusicFrom.SuspenseMusic_Music;
			NewMusicSection = GetMusicFrom.SuspenseMusic_Section;
			NewMusicTransition = GetMusicFrom.SuspenseMusic_Transition;
			break;
		case 'Action':
			NewMusicStr = GetMusicFrom.ActionMusic_MusicStr;
			NewMusic = GetMusicFrom.ActionMusic_Music;
			NewMusicSection = GetMusicFrom.ActionMusic_Section;
			NewMusicTransition = GetMusicFrom.ActionMusic_Transition;
			break;
		default:
			log("OpenMod: OpenModGameInfo: SetMusicVars(): ERROR: Unknown MusicStyle");
	}
}


function byte GetTeamIndexForName( string str )
{
	local int i;
	
	for (i=FLAGNUM_FIRST; i<FLAGNUM_LAST; i++)
	{
		if ( TeamNameStrings[i] ~= str )
			return TeamIndexes[i];
	}
}

defaultproperties
{
     GameReplicationInfoClass=Class'Engine.GameReplicationInfo'
}
