//=============================================================================
// LevelDiffInfo.
//
// OpenMod is licensed under the terms of the GNU Library General Public
// License.  More information on OpenMod and the terms of its license may
// be found in it's readme file, located in this game's "SYSTEM" directory.
//=============================================================================
class LevelDiffInfo expands LevelInfo;

// DLL: Diff Syntax version 0.0.1
// (davidl@captured.com)
//
// <DIFF><command><data1><data2><data3>
// <DIFF> = Ignored.  Only used to signal the beginning of a line of output.
// <command> = How the rest of the fields (on that line) should be processed
// <data1> + <data2> + <data3> = Depends on the command.
//
// NOTE: *ALL* fields must exist for each command, however they can be empty.  (Many commands
//   do not need all the data fields.
//
// Field Definition Syntax: Any special characters must be preceded with a '\' character.
// Special Characters: \ < > , .
//
// Commands:
//
// version = The version of diff being used (This is *ALWAYS* the first command in a diff session.
//   data1 = Version (in the form #.#.#)
//
// logstart = The start of a log file.
//
// logend = The end of a log file.
//
// actorstart = The start of an actor's information
//   data1 = Actor number
//
// actorend = The end of an actor's information
//   data1 = Actor number
//
// var = Information on a variable
//   data1 = Variable Name (such as 'Charge')
//   data2 = Variable Value (such as 'Bob', 3, or 0.5)
//   data3 = Unrealscript Data Type (such as 'int', 'name', or 'String[32]')
// NOTE: data3 is not required unless the value of data1 (the variable name) hasn't
//   been defined by defname.  More on this below.
//
// defstruct = Define a new data structure (such as a 'vector')
//   data1 = Type name (such as 'Vector' or 'Rotator')
//   data2 = Type contents (such as 'float,float,float')
//   data3 = Type content names (such as '.X,.Y,.Z' or '.Pitch,.Yaw,.Roll');
//
// defname = Define a new name association
//   data1 = New [Variable] Name
//   data2 = Data type
//
// deffunc = Define a function
//   data1 = Function Name
//   data2 = Name associations (Which name associations (defined with defname) should use this function.)
//
// varp = Information that is part of a data structure (which was defined by defstruct).
//   data1 = Variable Name + Part (such as 'Location' or 'Rotation')
//   data2 = Variable Part (such as '.X' or '.Yaw')
//   data3 = Variable Value

// Example: Setting up 'Location'
// In UnrealScript: SetLocation( vector NewLocation );
//
// <DIFF><defstruct><vector><float,float,float><.X,.Y,.Z>
// <DIFF><defname><Location><vector><>
// <DIFF><deffunc><SetLocation><Location><>
//
// <DIFF><varp><Location><.X><1.0>
// <DIFF><varp><Location><.Y><1.0>
// <DIFF><varp><Location><.Z><1.0>

// Example: Setting up collision sizes
// In UnrealScript: SetCollisionSize( float NewRadius, float NewHeight );
//
// <DIFF><defname><CollisionRadius><float><>
// <DIFF><defname><CollisionHeight><float><>
// <DIFF><deffunc><SetCollisionSize><CollisionRadius,CollisionHeight><>
//
// <DIFF><var><CollisionHeight><1.0><>
// <DIFF><var><CollisionRadius><1.0><>
//
// NOTE: As shown above, the 'var' command does not need a data type passed if the variable name
//   (passed in data1) is a defined name association (as it's type is already defined.)  Ie, in
//   the case above, the parser will see that 'CollisionHeight' and 'CollisionRadius' are defined
//   name associations, each with the type float.  It will then fill in that needed type
//   information.  If a type was passed in (in one of the shown 'var' commands), it will simply
//   be ignored.

//
// Parser Information
//
// Pass 1:
// <DIFF> commands with the proper amount of fields are searched for and placed
//   into a large list, starting with the first <DIFF> command issued (which would
//   be the version number) and ending with the last <DIFF> command.
//
// Pass 2:
// The list is then checked to see if the commands exist and that their syntax is
//   correct, all according to the version number (given by the first command.)
//   Any duplicate 'version' commands, will be ignored.
//
// Pass 3:
// The list will be broken into different sections, each split by the placement of
//   the 'logstart' and 'logend' commands.
//
// Pass 4: (Each log-block starts here individually and without previous knowledge
//   of the others.)
// The definition blocks are each processed in separate passes (we'll just consider
//   it one big pass for the sake of documented clarity) in the order: defstruct,
//   defname, deffunc.
//
// Pass 5: The actors are split into individual lists (each with an internal data
//   structure containing the data.)  Note, all the var and varp commands are
//   processed here.  When they are, their names are checked against the definitions
//   defined by defname.
//

function PrintDiff()
{
	local Actor A;
	local int i;
	
	i = 0;
	
	log("<DIFF><version><0.0.1><><>");
	log("<DIFF><logstart><><><>");

	// Define types
	// Location
	log("<DIFF><defstruct><vector><float,float,float><.X,.Y,.Z>");
	log("<DIFF><defname><Location><vector><>");
	log("<DIFF><deffunc><SetLocation><Location><>");
	
	// Rotation
	log("<DIFF><defstruct><rotator><float,float,float><.Pitch,.Yaw,.Roll>");
	log("<DIFF><defname><Rotation><rotator><>");
	log("<DIFF><deffunc><SetRotation><Rotation><>");
	
	// Physics
	log("<DIFF><defname><Physics><EPhysics><>");
	log("<DIFF><deffunc><SetPhysics><Physics><>");
	
	// Collision
	log("<DIFF><defname><CollisionRadius><float><>");
	log("<DIFF><defname><CollisionHeight><float><>");
	log("<DIFF><deffunc><SetCollisionSize><CollisionRadius,CollisionHeight><>");
	
	foreach AllActors(class'Actor', A)
	{
		if (( A == self ) ||
		    ( A.IsA('PlayerPawn') ) ||
			( A.bStatic ) )
		{
			continue;
		}
	
		// Start this actor
		log("<DIFF><actorstart><"$i$"><><>");
		
		// General Variables: Variable Values
		log("<DIFF><var><Class><"$ A.Class $"><class>");
		log("<DIFF><var><Name><"$ A.Name $"><name>");
		
		log("<DIFF><var><Physics><"$ A.Physics $"><EPhysics>");
		
		log("<DIFF><varp><Location><.X><"$ A.Location.X $">");
		log("<DIFF><varp><Location><.Y><"$ A.Location.Y $">");
		log("<DIFF><varp><Location><.Z><"$ A.Location.Z $">");
		
		log("<DIFF><var><Rotation><.Pitch><"$ A.Rotation.Pitch $">");
		log("<DIFF><var><Rotation><.Roll><"$ A.Rotation.Roll $">");
		log("<DIFF><var><Rotation><.Yaw><"$ A.Rotation.Yaw $">");
		
		// General Variables: Collision Size
		if ( A.IsA('Trigger') )
		{
			log("<DIFF><var><CollisionRadius><"$ A.CollisionRadius $"><>");
			log("<DIFF><var><CollisionHeight><"$ A.CollisionHeight $"><>");
		}
		
		// Type-Specific Variables
		if ( A.IsA('LevelInfo') )
		{
			log("<DIFF><var><Title><"$ LevelInfo(A).Title $"><string[64]>");
		}
		
		if ( A.IsA('PlayerStart') )
		{
			log("<DIFF><var><OwnerTeam><"$ PlayerStart(A).OwnerTeam $"><name>");
			log("<DIFF><var><TeamNumber><"$ PlayerStart(A).TeamNumber $"><byte>");
		}
		
		if ( A.IsA('Health') )
		{
			log("<DIFF><var><bSuperHeal><"$ Health(A).bSuperHeal $"><bool>");
			log("<DIFF><var><HealingAmount><"$ Health(A).HealingAmount $"><int>");
		}
		
		if ( A.IsA('Inventory') )
		{
			log("<DIFF><var><ArmorAbsorption><"$ Inventory(A).ArmorAbsorption $"><int>");
			log("<DIFF><var><Charge><"$ Inventory(A).Charge $"><int>");
			log("<DIFF><var><AbsorptionPriority><"$ Inventory(A).AbsorptionPriority $"><int>");
			log("<DIFF><var><RespawnTime><"$ Inventory(A).RespawnTime $"><float>");
		}
		
		if ( A.IsA('Ammo') )
		{
			log("<DIFF><var><AmmoAmount><"$ Ammo(A).AmmoAmount $"><int>");
		}
		
		if ( A.IsA('Trigger') )
		{
			log("<DIFF><var><TriggerType><"$ Trigger(A).TriggerType $"><ETriggerType>");
			log("<DIFF><var><Message><"$ Trigger(A).TriggerType $"><string[80]>");
			log("<DIFF><var><bTriggerOnceOnly><"$ Trigger(A).bTriggerOnceOnly $"><bool>");
			log("<DIFF><var><bInitiallyActive><"$ Trigger(A).bInitiallyActive $"><bool>");
		}
		
		// End This Actor
		log("<DIFF><actorend><"$i$"><><>");
		
		// Set up for the next one
		i++;
	}
	
	log("<DIFF><logend><><><>");
}

defaultproperties
{
     bStatic=False
     bNoDelete=False
}
