//=============================================================================
// DynCTFLevelInfo.
//
// OpenMod is licensed under the terms of the GNU Library General Public
// License.  More information on OpenMod and the terms of its license may
// be found in it's readme file, located in this game's "SYSTEM" directory.
//=============================================================================
class DynCTFLevelInfo expands DynTGLevelInfo;


// Misc Variables
var() bool			bNoTeamStartFind;		// If this is set, OpenMod will not automatically designate team
                                            // start points
var() bool			bNoTeamStartDisabler;	// Should OpenMod not disable the team start point finder if
          			                     	// pre-set team start points are found?

var bool			bInitialized;

// Flag Variables
/*
const FLAG_BLUE		= class'CTFBlueFlagPH';
const FLAG_GREEN	= class'CTFGreenFlagPH';
const FLAG_RED		= class'CTFRedFlagPH';
const FLAG_YELLOW	= class'CTFYellowFlagPH';
*/
const FLAG_BLUE		= 'Blue';
const FLAG_GREEN	= 'Green';
const FLAG_RED		= 'Red';
const FLAG_Yellow	= 'Yellow';

const FLAGNUM_FIRST		= 0;
const FLAGNUM_BLUE		= 0;
const FLAGNUM_GREEN		= 1;
const FLAGNUM_RED		= 2;
const FLAGNUM_YELLOW	= 3;
const FLAGNUM_LAST		= 3;
var class<Actor>		FlagClasses[4];
var class<Actor>		FlagBaseClasses[4];
var string				TeamNames[4];
var byte				TeamIndexes[4];

const MAX_FLAGS			= 4;


function bool PreInitialize()
{
	local int i;
	local OpenModGameInfo OwnerOGI;

	if (!Super.PreInitialize())
		return false;

	OwnerOGI = OpenModGameInfo(Owner);
	if (OwnerOGI == none)
		log("OpenMod: PreInitialise(): WARNING: OwnerOGI == none.");

	for (i=OwnerOGI.FLAGNUM_FIRST; i<=OwnerOGI.FLAGNUM_LAST; i++)
	{
		FlagClasses[i] = OwnerOGI.FlagClasses[i];
		FlagBaseClasses[i] = OwnerOGI.FlagBaseClasses[i];
	}
	for (i=0; i<MAX_FLAGS; i++)
	{
		TeamNames[i] = OwnerOGI.TeamNameStrings[i];
		TeamIndexes[i] = OwnerOGI.TeamIndexes[i];
	}

	return true;
}


function bool InitializeLevel()
{
	local float				Distances[4], RefDistance;
	local CTFFlagPH			Flags[4], tmpFlag;
	local CTFFlagBasePH		tmpFlagBase;
	local int				i, j, a, b, numteams;
	local PlayerStartFinder PSFinder;
	local Actor				tmpActor;
	local PlayerStart		tmpPS;
	local string			CountString;
	local bool				bDontSpawnClasses;
	local CTFFlagPH			tmpFlagPH;
	local CTFFlagBasePH		tmpFlagBasePH;
	
	// DLL: This is here as a temporary safety check.  It can probably be removed,
	//   however I'm too lazy to find out.
	if (bInitialized)
		return false;
	bInitialized = true;
	
	// get the number of teams on this map
	numteams = GetNumTeams();
		
	// Ob1: bDontSpawnClasses will be set to true if no flag classes
	//   are found. 
	// FIXME: I could probably use a flag finder class for this.
	bDontSpawnClasses = FindExistingClasses();
	if (bDontSpawnClasses)
	{
		// Ob1: clean up place holders if flags are already there
		log("OpenMod: InitialiseLevel(): CTF classes already on level. Removing place holders.");
		foreach allactors(class'CTFFlagPH', tmpFlagPH)
			if (tmpFlagPH != none) tmpFlagPH.Destroy();
		foreach allactors(class'CTFFlagBasePH', tmpFlagBasePH)
			if (tmpFlagBasePH != none) tmpFlagBasePH.Destroy(); 
	}
	//log("OpenMod: InitialiseLevel(): bSpawnClasses="$bSpawnClasses);
	
	// spawn flags for current level
	if (!bDontSpawnClasses)
	{
		i = 0;
		foreach AllActors(class'CTFFlagPH', tmpFlag)
		{
			Flags[i++] = tmpFlag;
			//if (Flags[i-1] == None)
			//	log("InitializeLevel(): ERROR: Flags["$i-1$"] == None");

			switch(tmpFlag.TeamName)
			{
				case FLAG_BLUE:
					a = FLAGNUM_BLUE;
					break;
				case FLAG_GREEN:
					a = FLAGNUM_GREEN;
					break;
				case FLAG_RED:
					a = FLAGNUM_RED;
					break;
				case FLAG_YELLOW:
					a = FLAGNUM_YELLOW;
					break;
				default:
					log("OpenMod: DynCTFLevelInfo: InitializeLevel(): ERROR: Unknown team name: "$tmpFlag.class$", "$tmpFlagBase.TeamName);
			}
			if ( FlagClasses[a] != None )
			{
				//log("OpenMod: DynCTFLevelInfo: Initialize(): Spawning Flag: class="$FlagClasses[a]$", "$a);
				tmpActor = spawn(FlagClasses[a],,,tmpFlag.Location,tmpFlag.Rotation);
				//tmpActor.SetRotation(tmpFlag.Rotation);
			}
		}

		// spawn flagbases for current level
		foreach AllActors(class'CTFFlagBasePH', tmpFlagBase)
		{
			switch(tmpFlagBase.TeamName)
			{
				case FLAG_BLUE:
					a = FLAGNUM_BLUE;
					break;
				case FLAG_GREEN:
					a = FLAGNUM_GREEN;
					break;
				case FLAG_RED:
					a = FLAGNUM_RED;
					break;
				case FLAG_YELLOW:
					a = FLAGNUM_YELLOW;
					break;
				default:
					log("OpenMod: DynCTFLevelInfo: InitializeLevel(): ERROR: Unknown team name: "$tmpFlagBase.class$", "$tmpFlagBase.TeamName);
			}
			if ( FlagBaseClasses[a] != None )
			{
				//log("OpenMod: DynCTFLevelInfo: Initialize(): Spawning FlagBase: class="$FlagBaseClasses[a]$", "$a);
				tmpActor = spawn(FlagBaseClasses[a],,,tmpFlagBase.Location,tmpFlagBase.Rotation);
				//tmpActor.SetRotation(tmpFlag.Rotation);
			}
		}
	}
	//log("InitializeLevel(): Flags = i = "$i);
	if ( !bNoTeamStartDisabler )
	{
			if (CheckLevelStartPoints(numteams, CountString))
				bNoTeamStartFind = true;
			log(class$": InitializeLevel(): Initial start points counted. "$CountString);
			if (bNoTeamStartFind) 
				log(class$": InitializeLevel(): Pre-set team designated start points found.  Team-start finder disabled.");
	}

	// Assign the team spawn points by finding the distance between the flags and dividing by two.
	if (!bNoTeamStartFind)
	{
		for(j=0; j<i; j++)
		{
			// 'a' and 'b' are indexes to Flags[].  They will be the two flags compared.
			a = j;
			// This makes sure that Flags[i-1] (the last one in the array) and Flags[0] are compared
			if (j<(i-1)) b=j+1;
			else b=0;
			
			//log("InitializeLevel(): a: "$a$"  b: "$b);
			
			RefDistance += Super(OpenModInfo).FindDistance(Flags[a].Location, Flags[b].Location);
			//log("InitializeLevel(): RefDistance now: "$RefDistance);
		}
		
		// RefDistance will equal the average distance between flags, divided by 2
		if (i != 0)
			RefDistance /= i*2;
		
		//log("InitializeLevel(): RefDistance Final: "$RefDistance);
		
		// Now create a PlayerStartFinder point at each flag and then call it's FindPoints() function
		for(j=0; j<i; j++)
		{
			PSFinder = Spawn(class'PlayerStartFinder',,,Flags[j].Location);
			PSFinder.FindPoints(RefDistance, Flags[j].TeamName, GetTeamIndexForName( string(Flags[j].TeamName) ));
		}
		
		// Ob1: Ensure nearby player start points are assigned
        foreach allactors(class'CTFFlagPH', tmpFlag)
	        if (tmpFlag != none)
		        foreach tmpFlag.RadiusActors(class'Actor', tmpActor, 500)
        	 		if (tmpActor.IsA('PlayerStart') && (PlayerStart(tmpActor) != none) )
          			{
            			PlayerStart(tmpActor).ownerTeam = tmpFlag.Teamname;
            			PlayerStart(tmpActor).TeamNumber = GetTeamIndexForName(string(tmpFlag.teamname));
          			}
		
		// Uncomment the next line if you would like to view the team spawn points
		//ViewPoints();
	}
	
	// log the final start point count
	CheckLevelStartPoints(numteams, CountString);
	log(class$": InitializeLevel(): Final start point count: "$CountString);
	
	// DLL: The following line is temporary.
	return true;
}


function byte GetTeamIndexForName( string str )
{
	local int i;
	
	for (i=0; i<MAX_FLAGS; i++)
	{
		if ( TeamNames[i] ~= str )
			return TeamIndexes[i];
	}
}


function bool IsValid()
{
	local PlaceHolder tmpPH;
	
	foreach AllActors( class'PlaceHolder', tmpPH )
	{
		if ( tmpPH.IsA('CTFFlagPH') ||
		     tmpPH.IsA('CTFFlagBasePH') )
		{
			log(class$": IsValid(): Returning True");
			return true;
		}
	}
	
	log(class$": IsValid(): Returning False");
	return false;
}

// Ob1: counts the number of spawn points for each team and returns
//   false if any of the teams has only one. The out string var is 
//   used to get a text represetation of the search.
function bool CheckLevelStartPoints(int numteams, out string CountResultString)
{
	local name team;
	local navigationpoint N;
	local int r, b, g, y, maxteams;
	local dynplayerstart dps;
	
	r=0;
	b=0;
	g=0;
	y=0;
	
	// find OpenMod DynPlayerStarts
	foreach allactors(class'DynPlayerStart',dps)
	  if ((dps != none) && (dps.ownerTeam != 'None'))
		switch(dps.teamnumber)
		{
			case TeamIndexes[FLAGNUM_RED]: r++; break;
			case TeamIndexes[FLAGNUM_BLUE]: b++; break;
			case TeamIndexes[FLAGNUM_GREEN]: g++; break;
			case TeamIndexes[FLAGNUM_YELLOW]: y++; break;
		}
	
	// count the number of spawn points per team
	for ( N=Level.NavigationPointList; N!=None; N=N.nextNavigationPoint )
		if ( N.IsA('PlayerStart') && !((playerstart(N).ownerTeam == 'None') && (PlayerStart(N).TeamNumber == 0)) )
			switch(playerstart(N).ownerTeam)
			{
				case 'Red': r++; break;
				case 'Blue': b++; break;
				case 'Green': g++; break;
				case 'Yellow': y++; break;
			}
	
	
	// log the results
	if ( numteams == 2 )
		CountResultString = "Red: "$r$". Blue: "$b$".";
	else if ( numteams == 3 )
		CountResultString = "Red: "$r$". Blue: "$b$". Green: "$g$".";
	else if ( numteams == 4 )
		CountResultString = "Red: "$r$". Blue: "$b$". Green: "$g$". Yellow: "$y$".";
	

	// return false if any of the teams have less than two spawn points
	if (r < 2)
	  return false;
	else if (b < 2)
	  return false;
	else if ( (g < 2) && (numteams > 2) )
	  return false;
	else if ( (y < 2) && (numteams > 3) )
	  return false;
	
	// return true otherwise
	return true;
}


// Ob1: returns true if flag classes are found
function bool FindExistingClasses()
{
	local actor flag;
	local int count, i;
	
	count = 0;
	for (i=0; i<4; i++)
		foreach allactors(FlagClasses[i], flag)
			count++;
	
	if (count > 0)
		return true;
	else return false;
}

// Ob1: returns number of teams on level by counting CTFFlagPH classes
function int GetNumTeams()
{
	local int numteams;
	local CTFFlagPH tmpFlag;
	
	numteams = 0;
	foreach allactors(class'CTFFlagPH', tmpFlag)
		numteams++;
		
	return numteams;
}

defaultproperties
{
}
