//=============================================================================
// NnCTFGame.
//=============================================================================
class NnCTFGame expands TeamGame config(NnCTF);


// Relationships: 
// I thought I'd better lay out how things work somewhere.
// Flag spawning is the key to determining the number of teams on the level.
// When a flag gets spawned (either because it is in the map or thru the switch
// statement in SetInitialState() (see below)) it creates a FlagBase to go along with
// it.  It has a pointer to its FlagBase and the FlagBase has a pointer to its Flag.
// SetInitialState() creates an NnCTFTeamInfo to go with each Flag in the level, links
// them together, sets the TeamName, and initializes the Teams[] array with pointers
// to all the NnCTFTeamInfo actors.  The TeamName is stored in the NnCTFTeamInfo
// actor *only*.  The other actors (Flag and FlagBase) do not have a TeamName property
// to avoid redundancy.  They do, however, have a path that can be followed to get to
// their associated NnCTFTeamInfo actor so that they can find out what team they are on
// and what their TeamName is.  The names of the teams are only defined once 
// (in NnCTFGame).


//var TeamInfo Teams[16];
var name Teams[16];
//var NnCTFTeamInfo Teams[16];


var() config bool	bRunes;					// put runes in the level?
var() config bool	bDropRunesOnCapture;	// Should a team's runes be dropped and respawned after capturing
                 	                    	// a flag?
var() config class	RuneList[30];			// What types of runes should be included
var() config int	RuneCharges[30];		// How powerful each rune should be
var int				RuneChargesIndex;		// Used to assign charges to runes (see GetCharge())
const				MAX_RUNES = 30;			// The maximum number of runes
var() config int	MaxRuneCharge;			// The maximum rune charge any one player can have
var class			ValidRunes[30];			// A list of runes (without duplicate entries)
var() config byte	PresetRuneConfig;		// 0=4 runes @ 100% each, 1=8 runes at 50% each,
                 	                 		// 2=12 runes at 10% each.
var() config float	InvalidRuneTime;			// How long should runes stay invalid (0=infinity)
const				MAX_PRESET_RUNE_CONFIG = 2;

// Flag Variables
var() config bool	bAutoFlagReturn;		// should the flag be returned automatically?
var() config float	AutoFlagReturnTime; 	// The time before the flag auto-returns
var() config int	CaptureLimit;			// How many captures until a team wins
var() config int	BeenCapturedLimit;		// How many captures until a team loses
var() config bool	bCaptureWithFlag;		// does one's own flag need to be in base in order
											// for the flag to be captured?
var() config bool	bInvisibleBases;		// Should the bases be invisible?

var() config bool	bHook;					// Should each player be given a grappling hook?
var() config class	HookClass;				// The class of hook given to each player

// Login/Spectating vars
var() config bool	bAutoAssignTeams;		// Should each player automatically be assigned to a team?
var() config bool	bAllowSpectating;		// Is spectating allowed?
var() config bool 	bForceBaseSpawn;		// Force players to only start at base?

// Dynamic Team Sizing Vars
/*
var() config bool	bAutoTeamSize;			// Should the team size be dynamically adjusted?
var() config int	MaxTeamsCeiling;		// The most teams that can be playing (2, 3, or 4)
var() config int	MaxTeamsFloor;			// The fewest 
var() config int	MinPlayersPerTeam;		// The minimum number of players needed for a team
*/

// Misc. Variables
var() string	Version;				// The current version of NnCTF
var() string	ModPrefix;				// The shortened name of the CTF Mod (ie, NnCTF for NoName CTF)
var() bool			bNoTeamStartFind;		// If this is set, NnCTF will not automatically designate team
											// spawn points
var bool			bGameIsSetup;			// DLL: Don't use this variable.  It is too much of a hack.
var() config bool	bNoDebugMessages;		// Toggles the use of debugging messages
var NnGameReplicationInfo	NnGRI;			// A "quick access" variable to NnGameReplicationInfo(GameReplicationInfo)
var() config bool	bIgnoreLevelPrefs;		// Should the level specific preferences be ignored?
// not needed now team numbers are Unreal compatible
//var() config class<DynInfo>	TShieldBeltLoader;
//var() config class<DynInfo>	CTFCannonFixer;
var() config bool	bUseCustomSkins;		// Should custom skins be used?
var() config bool	bUseTargetingComputer;	// Should the targeting computer be enabled?
var() config bool	bBotsCanCarryFlags;		// Should bots be allowed to carry a flag, even though they
                                       		// currently do not know what to do with it?
// Grenade variables
var() config bool		bDisableGrenades;	// Should grenades be disabled
var() config float		GrenadesPerPlayer;	// How many grenades should there be for each player?
var() class<DynInfo>	GrenadeSpawnerClass;
var DynItemSetInfo		GrenadeSpawner;		// The object responsible for spawning grenades
var() int				PlayerCheckTime;	// How often should the grenade amts be checked?

// Bonus Variables
var() config float		teamCaptureBonus;	// how many points should a team get for a capture
var() config float		playerCaptureBonus;	// how many points should the capturing player get
var() config string		fragCarrierMessage;	// Message displayed after fragging the flag carrier
var() config float		fragCarrierBonus;	// Bonus given for fragging the flag carrier
var() config float		defendRadius;		// Radius for detecting what's a flag defend
var() config string		defendMessage;		// Message displayed about defenders
var() config float		defendBonus;		// Bonus given to defenders
var() config float		assistTime;			// How much time for an assist
var() config float		assistBonus;		// Bonus given to assisters

// Serpentine integration
var() config bool		bSerpentineWeapons;	// Should Serpentine weapons replace the usual ones?

// Music Information
var() config class<DynMusicInfo>	AvailableMusic[32];	// Available music selections
const MAX_AVAILABLE_MUSIC = 32;
var byte				LastFlagHolders[8];
var name				LastFlagStates[8];
var name				LastMusicSetTo[8];

// Persistent Scoring Information


// OpenMod stuff
var OpenModGameInfo		OpenModGI;
var() config name		MapConverters[8]; 	// CTF map format converters


// an enum would be nice here but they dont work as array indexes
// DLL: These items also have to be replicated to NnGameReplicationInfo.
//   This is handled in PreBeginPlay()
// Ob1: I changed the numbers around to get better compatibility with
//   current unreal team classes (eg. the new Unreal team shieldbelt)
const TC_First		= 0;
const TC_Red		= 0;
const TC_Blue		= 1;
const TC_Green		= 2;
const TC_Yellow		= 3;
const TC_Last		= 3;
const TC_Spectator	= 4;
const TC_None		= 255;
var() string		TeamName[5];
var() color			TeamColor[5];

const Player_FIRST			= 0;
const Player_FemaleOne		= 0;
const Player_FemaleTwo		= 1;
const Player_MaleOne		= 2;
const Player_MaleTwo		= 3;
const Player_MaleThree		= 4;
const Player_SkaarjPlayer	= 5;
const Player_LAST			= 4;


// debugging log
function dlog(string str)
{
	if (bNoDebugMessages)
		return;
	
	log(str);
}

// information log
function ilog(string str)
{
	log(str);
}

// AEon log
function alog(string str)
{
	log(str);
}


event InitGame( string Options, out string Error )
{
	local PlayerChecker PC;
	
	dlog("NnCTF: NnCTFGame: InitGame()");
	Super.InitGame( Options, Error );
	
	OpenModGI = spawn(class'OpenModGameInfo');  
	OpenModInit();
	
	if ( (GrenadeSpawner == none) && !bDisableGrenades )
	{
		log("NnCTF: InitGame(): GrenadeSpawnerClass='none': Disabling grenades");
		bDisableGrenades = true;
	}

	AdjustGrenadeAmt(0);
	
	if ( bRunes )
		RunesInit();
}


function InitGameReplicationInfo()
{
	local GameReplicationInfo GRI;
	local int i;
	
	log(class$": InitGameReplicationInfo()");
	Super.InitGameReplicationInfo();
	
	NnGRI = NnGameReplicationInfo(GameReplicationInfo);
	NnGRI.TC_None		= TC_None;
	NnGRI.TC_Spectator	= TC_Spectator;
	NnGRI.TC_First		= TC_First;
	NnGRI.TC_Red		= TC_Red;
	NnGRI.TC_Blue		= TC_Blue;
	NnGRI.TC_Green		= TC_Green;
	NnGRI.TC_Yellow		= TC_Yellow;
	NnGRI.TC_Last		= TC_Last;
	
	for (i=0; i<8; i++)
		NnGRI.FlagHolders[i] = TC_None;
	
	if ( NnGRI == None )
		log(class$": InitGameReplicationInfo(): ERROR: NnGRI == None");
	
	// DLL: For some reason, using a custom GameReplicationInfo class
	//   will end up with two GameReplicationInfo objects in the actor
	//   list, therefore screwing all preset info up (like the MOTD,
	//   server name, etc.)  This hack fixes that by simply copying
	//   info from the plain GameReplicationInfo class to the NnCTF
	//   specific NnGameReplicationInfo class.
	
	// Ob1: Have fixed this in OpenMod now. The extra GRI was being 
	//   created for the OpenModGI helper class.
	/*foreach AllActors(class'GameReplicationInfo', GRI)
	{
		if ( !GRI.IsA('NnGameReplicationInfo') )
		{
			NnGRI.ServerName = GRI.ServerName;
			NnGRI.ShortName = GRI.ShortName;
			NnGRI.AdminName = GRI.AdminName;
			NnGRI.AdminEmail = GRI.AdminEmail;
			NnGRI.Region = GRI.Region;
			NnGRI.ShowMOTD = GRI.ShowMOTD;
			NnGRI.MOTDLine1 = GRI.MOTDLine1;
			NnGRI.MOTDLine2 = GRI.MOTDLine2;
			NnGRI.MOTDLine3 = GRI.MOTDLine3;
			NnGRI.MOTDLine4 = GRI.MOTDLine4;
		}
	}*/
}



// Everything is OK as long as calling SetInitialState more than once is OK for
// anything spawned here.
function SetInitialState()
{
	local Flag F;
	local NnCTFTeamInfo TI;
	local int TeamNum;
	
	// Set up each of the teams
	foreach AllActors(class'Flag', F)
	{
		TeamNum = F.GetTeamNum();
		if (TeamNum < MaxTeams)
		{
			TI = CreateTeam(TeamName[TeamNum], TeamNum, TeamColor[TeamNum], false, F);
		}
		else
		{
			ilog("NnCTF: Eliminating " $ F.class $ " because it is over the MaxTeams limit");
			FreeStartPoints(F.TeamName);  // free up the teams playerstarts
			F.Destroy();
		}
		
		// set the auto return time
		if (bAutoFlagReturn)
			F.SetAutoReturnTime(AutoFlagReturnTime);
		else
			F.SetAutoReturnTime(0); // don't return automatically
	}
	
	// Set up the spectator team
	CreateTeam(TeamName[TC_Spectator], TC_Spectator, TeamColor[TC_Spectator], true, None);
	
	// Set up Runes
	//if (bRunes)
	//	RunesInit();
	
	Super.SetInitialState();
	
	// DLL: Just ignore the following line as it is a horrid hack.
	bGameIsSetup = true;
}


// Ob1: Called when a team is removed from play to free up playerstarts
function FreeStartPoints(name TeamName)
{
	local playerstart ps;
	
	foreach allactors(class'PlayerStart', ps)
	{
		if ((ps != none) && (ps.ownerTeam == TeamName))
		{
			ps.ownerTeam = 'None';
			ps.TeamNumber = 255;
		}
	}
}


function RunesInit()
{
	local Runes Rune;
	local int i, j;
	
	for (i=0; i<MAX_RUNES && RuneList[i]!=None; i++)
	{	
		if ( RuneList[i] != None )
		{
			for (j=0; j<MAX_RUNES; j++)
			{
				if ( RuneList[i]==ValidRunes[j] )
					break;
				if ( RuneList[i]!=ValidRunes[j] && ValidRunes[j]!=None )
					continue;
				if ( ValidRunes[j]==None && RuneList[i]!=None )
				{
					ValidRunes[j]=RuneList[i];
					break;
				}
			}
		
			Rune = spawn( class<Runes>(RuneList[i]) );
			if ( Rune != None )
			{
				Rune.Charge = GetNewCharge( RuneList[i] );
				Rune.InvalidTime = InvalidRuneTime;
				ilog("NnCTF: Spawning Rune: "$Rune.ItemName$" with charge "$Rune.Charge);
				Rune.MoveToPlayerStart();
			}
		}
	}
}


function OpenModInit()
{
	local OpenModInfo tmpOMI;
	local int i, j, num;
	local string result;
	local playerstart ps;
	
	dlog ("NnCTF: OpenMod setup code starting");
	
	// OpenMod: Give OpenMod info on the specific flag classes.
	OpenModGI.FlagClasses[OpenModGI.FLAGNUM_BLUE] = class'BlueFlag';
	OpenModGI.FlagClasses[OpenModGI.FLAGNUM_GREEN] = class'GreenFlag';
	OpenModGI.FlagClasses[OpenModGI.FLAGNUM_RED] = class'RedFlag';
	OpenModGI.FlagClasses[OpenModGI.FLAGNUM_YELLOW] = class'YellowFlag';
	
	OpenModGI.TeamNameStrings[OpenModGI.FLAGNUM_BLUE] = "Blue";
	OpenModGI.TeamNameStrings[OpenModGI.FLAGNUM_GREEN] = "Green";
	OpenModGI.TeamNameStrings[OpenModGI.FLAGNUM_RED] = "Red";
	OpenModGI.TeamNameStrings[OpenModGI.FLAGNUM_YELLOW] = "Yellow";
	
	OpenModGI.TeamNames[OpenModGI.FLAGNUM_BLUE] = 'Blue';
	OpenModGI.TeamNames[OpenModGI.FLAGNUM_GREEN] = 'Green';
	OpenModGI.TeamNames[OpenModGI.FLAGNUM_RED] = 'Red';
	OpenModGI.TeamNames[OpenModGI.FLAGNUM_YELLOW] = 'Yellow';
	
	OpenModGI.TeamIndexes[OpenModGI.FLAGNUM_BLUE] = TC_Blue;
	OpenModGI.TeamIndexes[OpenModGI.FLAGNUM_GREEN] = TC_Green;
	OpenModGI.TeamIndexes[OpenModGI.FLAGNUM_RED] = TC_Red;
	OpenModGI.TeamIndexes[OpenModGI.FLAGNUM_YELLOW] = TC_Yellow;
	
	// OpenMod: Make sure OpenMod knows that it can load anything up to a DynCTFLevelInfo class.
	OpenModGI.AcceptedDLIClasses[0] = 'DynCTFLevelInfo';
	OpenModGI.AcceptedDLIClasses[1] = 'LevelPrefsInfo';
	
	// Set up any CTF map format conversion modules (ie. RealCTF CTF map format to OpenMod CTF map format.)
	for (i=0; i<8; i++)
		if (MapConverters[i] != 'None')
			OpenModGI.FlagFinderClasses[i] = MapConverters[i];
	
	// OpenMod: Load up the level's data
	OpenModGI.LoadLevelData();
	
	num = GetNumTeams();
	// Ob1: Ensure that each team has more than one spawn point
	if (!CheckLevelStartPoints(num, result) && !(num < 2))
	{
		log("NnCTF: OpenModInit(): Too few team start points found on level. Spawning DynPlayerStarts.");
		SpawnDynPlayerStartPoints();
		CheckLevelStartPoints(num, result);
		log("NnCTF: OpenModInit(): New player start count: "$result);
	}	
	
	// Ob1: player start saftey check
	num = 0;
	if (!bSpawnInTeamArea && !bForceBaseSpawn)
	{
		foreach allactors(class'playerstart', ps)
			if (ps.ownerteam == 'None') num++;
		if (num < 2) bSpawnInTeamArea=true;
	}
	
	// OpenMod: Set up the dynamic music information
	tmpOMI = spawn(class'OpenModInfo');
	for (i=0; i<MAX_AVAILABLE_MUSIC; i++)
	{
		if (AvailableMusic[i] != None)
			j++;
	}
	i = RandRange(0, j);
	tmpOMI.AvailableInfo[0] = AvailableMusic[i];
	dlog(class$": SetInitialState(): Loading Music: Num="$i);
	dlog(class$": SetInitialState(): Loading Music: Class="$AvailableMusic[i]);
	
	// OpenMod: Make sure that TShieldBelts and TPowerShields are used
	//tmpOMI.AvailableInfo[1] = TShieldBeltLoader;
	
	// OpenMod: Make sure any CTFCannon objects get OpenMod-ified
	//tmpOMI.AvailableInfo[2] = CTFCannonFixer;
	
	// OpenMod: Load up the music and item info
	OpenModGI.LoadOMI(tmpOMI);
	
	// OpenMod: Set up any persistent item spawners
	if (!bDisableGrenades)
		GrenadeSpawner = DynItemSetInfo( OpenModGI.LoadSingleClass(GrenadeSpawnerClass) );

	dlog ("NnCTF: OpenMod setup code ending");
}

// Ob1: Spawn point check functions
function bool CheckLevelStartPoints(int numteams, optional out string CountResultString)
{
	local name team;
	local navigationpoint N;
	local int r, b, g, y, maxteams;
	local dynplayerstart dps;
	
	r=0;
	b=0;
	g=0;
	y=0;
	
	// find OpenMod DynPlayerStarts
	foreach allactors(class'DynPlayerStart',dps)
	  if ((dps != none) && (dps.ownerTeam != 'None'))
		switch(dps.teamnumber)
		{
			case TC_Red: r++; break;
			case TC_Blue: b++; break;
			case TC_Green: g++;	break;
			case TC_Yellow: y++; break;
		}
	
	// count the number of spawn points per team
	for ( N=Level.NavigationPointList; N!=None; N=N.nextNavigationPoint )
		//if ( N.IsA('PlayerStart') && !((playerstart(N).ownerTeam == 'None') && (PlayerStart(N).TeamNumber == TC_None)) )
		if ( N.IsA('PlayerStart') && (playerstart(N).ownerTeam != 'None'))
			switch(playerstart(N).ownerTeam)
			{
				case 'Red': r++; break;
				case 'Blue': b++; break;
				case 'Green': g++; break;
				case 'Yellow': y++;	break;
			}
	
	
	// log the results
	if ( numteams == 2 )
		CountResultString = "Red: "$r$". Blue: "$b$".";
	else if ( numteams == 3 )
		CountResultString = "Red: "$r$". Blue: "$b$". Green: "$g$".";
	else if ( numteams == 4 )
		CountResultString = "Red: "$r$". Blue: "$b$". Green: "$g$". Yellow: "$y$".";
	

	// return false if any of the teams have only one spawn point
	if (r < 2)
	  return false;
	else if (b < 2)
	  return false;
	else if ( (g < 2) && (numteams > 2) )
	  return false;
	else if ( (y < 2) && (numteams > 3) )
	  return false;
	
	// return true otherwise
	return true;
}

// Ob1: returns number of teams on level by counting flag classes
function int GetNumTeams()
{
	local int numteams;
	local Flag tmpFlag;
	
	numteams = 0;
	foreach allactors(class'Flag', tmpFlag)
		numteams++;
		
	return numteams;
}

function SpawnDynPlayerStartPoints()
{
	local Flag f;
	local Actor A;
	local DynPlayerStart tsp;
	local int i, TeamIndex;

	foreach AllActors( class 'Flag', f )
	{
	    if (f != none)
	    {
			// get the team number for the flag name
			for (i=0; i<5; i++)
				if (string(f.TeamName) == TeamName[i])
					TeamIndex = i;

	    	// Spawn a PlayerStart at each flag
	    	tsp=Spawn(class'DynPlayerStart',,,f.location);
			if(tsp==none)
			{
			  Log("NnCTF: Auto PlayerStart failed to spawn");
			  return;
			}      
			else
			{
			  tsp.teamnumber = TeamIndex;
			  tsp.ownerTeam = f.teamname;
			  log("NnCTF: Auto PlayerStart spawned for '"$ TeamName[TeamIndex] $"' team");  
			}
		}
    }	
}

function NnCTFTeamInfo CreateTeam(string	TName,
                                  int			TIndex,
                                  color			TColor,
                                  optional bool	bDontTabulate,
                                  optional Flag	TFlag)
{
	local NnCTFTeamInfo TI;
	
	TI = Spawn(class'NnCTFTeamInfo');
	
	TI.TeamName = TName;
	TI.TeamIndex = TIndex;
	TI.TeamColorR = TColor.R;
	TI.TeamColorG = TColor.G;
	TI.TeamColorB = TColor.B;
	
	if ( TFlag != None )
	{
		TFlag.TeamInfo = TI;
		TFlag.NnGRI = NnGRI;
		TI.Flag = TFlag;
		
		TFlag.SpawnFlagBase( bInvisibleBases );
	}
	
	NnGameReplicationInfo(GameReplicationInfo).Teams[TIndex] = TI;
	NnGRI.Teams[TIndex] = TI;
	if ( !bDontTabulate )
		NnGameReplicationInfo(GameReplicationInfo).NumTeams++;
	
	dlog("NnCTF: New Team:");
	dlog("NnCTF: TeamName: "$TI.TeamName);
	dlog("NnCTF: Index: "$TI.TeamIndex);
	dlog("NnCTF: Color: "$TI.TeamColorR$", "$TI.TeamColorG$", "$TI.TeamColorB);
	ilog("NnCTF: "$TI.TeamName$" team created");
	
	return TI;
}


function string GetRules()
{
	local string ResultSet;
	ResultSet = Super.GetRules();
	
	// Version
	ResultSet = ResultSet$"\\NnCTF Version\\"$Version;
	
	// AutoFlagReturn
	if ( bAutoFlagReturn )
		ResultSet = ResultSet$"\\AutoFlagReturn\\"$true;
	else
		ResultSet = ResultSet$"\\AutoFlagReturn\\"$false;
	
	// CaptureWithFlag
	if ( bCaptureWithFlag )
		ResultSet = ResultSet$"\\CaptureWithFlag\\"$true;
	else
		ResultSet = ResultSet$"\\CaptureWithFlag\\"$false;
	
	// GoalTeamScore
	ResultSet = ResultSet$"\\GoalTeamScore\\"$GoalTeamScore;
	
	// CaptureLimit
	ResultSet = ResultSet$"\\CaptureLimit\\"$CaptureLimit;
	
	// BeenCapturedLimit
	ResultSet = ResultSet$"\\BeenCapturedLimit\\"$BeenCapturedLimit;
}


/*
RWS: This function will assign a team to the player that is dlogging into this
map. The player's team is matched with a valid flagID. This makes sure that 
the player can't be apart of a team who's flag doesn't exist.
*/

function playerpawn Login
(
	string Portal,
	string Options,
	out string Error,
	class<playerpawn> SpawnClass
)
{
	local CTFPlayer		newPlayer;
	local int			i, j;
	local string	newOptions;
	local string	Pair;
	local string	Key;
	local string	Value;
	local string	newPortal;
	
	/*
	RWS: if we call the super.Login TeamGame.ChangeTeam will be called
	and player will assigned a team according to Team rules and not 
	CTF rules. We may need to move this class to expand DeathMatchGame
	instead of TeamGame
	*/
	 
	// find a team to spawn this person on
	if (bAutoAssignTeams || !bAllowSpectating)
		i = FindTeam();
	else {
		i = TC_Spectator;
	}
	
	newPortal = GetTeamNameForNum(i);
	
	// set Team in Options to be the new team
	newOptions = "";
	while (GrabOption(Options, Pair))
	{
		GetKeyValue(Pair, Key, Value);
		if (Key ~= "Team")
			newOptions = newOptions $ "?" $ Key $ "=" $ i;
		else
			newOptions = newOptions $ "?" $ Pair;
	}
	
	// Make sure the player is using an NnCTF player class
	switch (SpawnClass) 
	{
		case class'FemaleOne':
		case class'CTFFemaleOne':
			SpawnClass = class'CTFFemaleOne';
			break;
		case class'FemaleTwo':
		case class'CTFFemaleTwo':
			SpawnClass = class'CTFFemaleTwo';
			break;
		case class'MaleOne':
		case class'CTFMaleOne':
			SpawnClass = class'CTFMaleOne';
			break;
		case class'MaleTwo':
		case class'CTFMaleTwo':
			SpawnClass = class'CTFMaleTwo';
			break;
		case class'MaleThree':
		case class'CTFMaleThree':
			SpawnClass = class'CTFMaleThree';
			break;
		case class'SkaarjPlayer':
		case class'CTFSkaarjPlayer':
			SpawnClass = class'CTFSkaarjPlayer';
			break;
		default:
			SpawnClass = class'CTFMaleThree';
			break;
	}
	
	newPlayer = CTFPlayer( Super(DeathMatchGame).Login(newPortal, newOptions, Error, SpawnClass) );
	if (newPlayer == None)
		return None;
	
	// Set up the player's NnGRI value
	newPlayer.NnGRI = NnGameReplicationInfo(GameReplicationInfo);
	
	// Set up the player's Maximum Rune Charge
	NnPlayerReplicationInfo(newPlayer.PlayerReplicationInfo).MaxRuneCharge = MaxRuneCharge;
	
	// Set up the player's MOTD time
	//newPlayer.MOTDDisplayTime = MOTDDisplayTime;
	
	// Set up the player's ValidRunes list
	for (j=0; j<MAX_RUNES; j++)
	{
		if ( ValidRunes[j] != None )
			newPlayer.ValidRunes[j] = ValidRunes[j];
	}
	
	// Make sure that there are enough grenades on the level, now that
	// a new player is around
	CheckPlayerCount();	
	
	return newPlayer;
}


function bool RestartPlayer( pawn aPlayer )	
{	
	if ( Super.RestartPlayer(aPlayer) )
	{
		ResetInvalidRunes(aPlayer);
		return true;
	} else
		return false;
}



function Logout( pawn Exiting )
{
	if( Level.NetMode==NM_DedicatedServer || Level.NetMode==NM_ListenServer )
	{
		alog("L/"$Level.Title$"/"$Exiting.PlayerReplicationInfo.PlayerName$"/"); //AEon   
	}
	
	RemoveFromTeam(Exiting);
	
	// Make sure there are the correct amount of grenades on the level,
	// now that a player is exiting.
	CheckPlayerCount();
	
	Super.Logout(Exiting);
}


function Flag GetTeamFlagForName(String S)
{
	local NnCTFTeamInfo TI;
	TI = GetTeamInfoForName(S);
	if (TI != None)
		return(TI.Flag);
	else
		return(None);
}

function Flag GetTeamFlagForNum(int i)
{
	local NnCTFTeamInfo TI;
	TI = GetTeamInfoForNum(i);
	if (TI != None)
		return(TI.Flag);
	else
		return(None);
}

function int GetTeamNumForName(String S)
{ 
	local int i;
	
	for (i = TC_First; i <= TC_Last; i++)
	{
		if ((NnGRI.Teams[i] != None) &&
		    (S ~= NnGRI.Teams[i].TeamName) )
		{
			return i;
		}
	}
	
	if (S ~= NnGRI.Teams[TC_Spectator].TeamName)
		return TC_Spectator;

	log(class$": GetTeamNumForName(): Returning None, S="$S);
	return(TC_None);
}

function string GetTeamNameForNum(int i)
{
	if (NnGRI.Teams[i] != None)
		return(NnGRI.Teams[i].TeamName);
	else
		return("None");
}

function NnCTFTeamInfo GetTeamInfoForName(String S)
{
	return(NnGRI.Teams[GetTeamNumForName(S)]);
}

function NnCTFTeamInfo GetTeamInfoForNum(int i)
{
	return(NnGRI.Teams[i]);
}

function int GetFlagCount()
{
	local int i, fc;

	fc = 0;
	for (i = TC_First; i <= TC_Last; i++)
	{
		if (GetTeamFlagForNum(i) != None)
		fc++;
	}
	return(fc);
}

// Checks the scores+captures to see if the game should end
function CheckForEndGame()
{
	local PlayerReplicationInfo PRI;
	local int i;
	
	// Check to see if a specific player has hit the fraglimit
	if (FragLimit > 0)
	{
		foreach AllActors(class'PlayerReplicationInfo', PRI)
		{
			if (PRI.Score >= FragLimit)
				EndGame("fraglimit");
		}
	}
	
	// Check to see if the capture limit or the team limit has been hit 
	for (i=TC_First; i<=TC_Last; i++)
	{
		if (NnGRI.Teams[i] == None)
			continue;
		
		//dlog("NnCTF: NnCTFGame: CheckForEndGame(): Teams["$i$"].Captures="$NnCTFTeamInfo(Teams[i]).Captures);
		
		if ( (NnGRI.Teams[i].Captures >= CaptureLimit && CaptureLimit > 0) ||
		     (NnGRI.Teams[i].Score >= GoalTeamScore && GoalTeamScore > 0) ||
		     (NnGRI.Teams[i].BeenCaptured >= BeenCapturedLimit && BeenCapturedLimit > 0) )
			EndGame("scorelimit");
	}
}

// Adjusts a team's captures variable and then checks to see if the game has ended.
function AdjustCaptures( NnCTFTeamInfo Captors, NnCTFTeamInfo Captees, float adjamt )
{
	Captors.Captures += adjamt;
	Captees.BeenCaptured += adjamt;
	
	CheckForEndGame();
}

// JMA: Added the next few dozen lines
// helper function.  should always be called to adjust any score
function AdjustScore(pawn Other, float adjamt, optional bool bDontCheckForEndGame)
{
	dlog("NnCTF: NnCTFGame: AdjustScore(): "$Other.PlayerReplicationInfo.PlayerName$", "$adjamt);
	Other.PlayerReplicationInfo.Score += adjamt;
	GetTeamInfoForNum(Other.PlayerReplicationInfo.Team).Score += adjamt;

	if (!bDontCheckForEndGame)
		CheckForEndGame();
}

// returns true if A can see B.
// is there somewhere where this is already implemented?
function bool CanSee(pawn A, actor B)
{
	local Actor tmp;
	foreach A.VisibleActors(B.class, tmp)
	{
		if (tmp == B)
			return (true);
	}
	return(false);
}

// returns true if B is with a radius of A
// is there somewhere where this is already implemented?
function bool IsInRadius(pawn A, actor B, float radius)
{
	local Actor tmp;
	foreach A.RadiusActors(B.class, tmp, radius)
	{
		if (tmp == B)
			return (true);
	}
	return(false);
}

// Awards any frag-based bonuses
// (currently, you only get one bonus (the most important one))
// any problems with that?
function CheckBonuses(pawn killer, pawn Other)
{
	local NnFlagBase ourFlagBase;
	local Flag theirFlag;

	if (killer == None || Other == None || killer == Other)
		return;
	
	//log(class$": CheckBonuses(): Killer.Name="$Killer.Name);
	//log(class$": CheckBonuses(): Killer.PRI.Team="$Killer.PlayerReplicationInfo.Team);
	//log(class$": CheckBonuses(): Other.Name="$Other.Name);
	//log(class$": CheckBonuses(): Other.PRI.Team="$Other.PlayerReplicationInfo.Team);
	
	ourFlagBase = GetTeamFlagForNum(killer.PlayerReplicationInfo.Team).FlagHomeBase;
	theirFlag = GetTeamFlagForNum(Other.PlayerReplicationInfo.Team);
	
	if ( ourFlagBase == None )
		log(class$": CheckBonuses(): ERROR: ourFlagBase==None");
	if ( theirFlag == None )
		log(class$": CheckBonuses(): ERROR: theirFlag==None");
	
	// fragged the flag carrier
	if (theirFlag.Owner == Other)
	{
		killer.ClientMessage(fragCarrierMessage);
		GetTeamInfoForNum(killer.PlayerReplicationInfo.Team).SetAssistant(killer, assistTime);
		AdjustScore(killer, fragCarrierBonus);
		return;
	}

	// defended the base
	if (CanSee(killer, ourFlagBase) ||
	    CanSee(Other, ourFlagBase) ||
	    IsInRadius(killer, ourFlagBase, defendRadius) ||
	    IsInRadius(Other, ourFlagBase, defendRadius))
	{
		BroadcastMessage(killer.PlayerReplicationInfo.PlayerName $ defendMessage $ killer.PlayerReplicationInfo.TeamName $ " base", false);
		AdjustScore(killer, defendBonus);
		return;
	} 

	return;
}


function ResetInvalidRunes( Pawn Other )
{
	if ( NnPlayerReplicationInfo(Other.PlayerReplicationInfo).InvalidRunes != None )
	{
		if ( NnPlayerReplicationInfo(Other.PlayerReplicationInfo).InvalidRunes.NextInfo != None )
			NnPlayerReplicationInfo(Other.PlayerReplicationInfo).InvalidRunes.NextInfo.DestroyAllForward();
	}
	else
		log(class$": ResetInvalidRunes(): ERROR: InvalidRunes==None");
}


// All game rules for killing should be implemented here.
// We don't call any Super.Killed()
function Killed(pawn killer, pawn Other, name damageType)
{
	local string killed;
	local string message;
	
	// First, reset the invalid runes/
	ResetInvalidRunes(Other);
	
	// Now, handle the usual killed() stuff.
	Other.DieCount++;
	AEonKilled(killer, Other, damageType);
	if (Other.bIsPlayer)
	{
		killed = Other.PlayerReplicationInfo.PlayerName;
		if( killer==None || killer==other )
			message = KillMessage(damageType, Other);
		else 
			message = killer.KillMessage(damageType, Other);
		BroadcastMessage(killed$message, false);
	}
	
	CheckBonuses(killer, Other);    // ours
	
	if ((killer == Other) || (killer == none))
	{
		AdjustScore(Other, -1);
	}
	else if( (killer != None) && (killer.PlayerReplicationinfo.Team 
		!= other.PlayerReplicationinfo.Team) )
	{
		killer.killCount++;
		AdjustScore(killer, 1);
	}
	else if( (killer != None) && (killer.PlayerReplicationinfo.Team 
		== other.PlayerReplicationinfo.Team) )
	{
		// team frag penalty
		AdjustScore(killer, -1);
	}
	if (killer == None)
		return;
	
//  if ( (FragLimit > 0) && (killer.PlayerReplicationInfo.Score >= FragLimit) )
//    EndGame();
	CheckForEndGame();
/*
	if ( BotConfig.bAdjustSkill && (killer.IsA('PlayerPawn') || Other.IsA('PlayerPawn')) )
	{
		if ( killer.IsA('Bots') )
			Bots(killer).AdjustSkill(true);
		if ( Other.IsA('Bots') )
			Bots(Other).AdjustSkill(false);
	}
*/

	if ( (GoalTeamScore > 0) && (NnGRI.Teams[killer.PlayerReplicationInfo.Team].Score >= GoalTeamScore) )
		EndGame("TeamScoreLimit");

	/* JMA: ok, lets try rewriting this a bit ...
	// Display killing message
	Other.DieCount++;
	if (Other.bIsPlayer)
	{
		foreach killer.ChildActors(class 'NnCTFAPInfo', APInfo)
		{
			if (APInfo.bIsAssistant == false)
				APInfo = None;
		}
		if (APInfo == None)
		{
			killed = Other.PlayerName;
			if( killer==None || killer==other )
				message = KillMessage(damageType, Other);
			else 
				message = killer.KillMessage(damageType, Other);
			BroadcastMessage(killed$message, false);
		}
		else
		{
			message = Other.PlayerName $ " defended the " $ GetTeamNameForNum(Other.Team) $ " flag carrier";
			BroadcastMessage (message, false);
		}
	}
	
	// Adjust Scores
	if( killer == Other )
	{
		Other.Score -= 1;
		Teams[Other.Team].Score -= 1;
	}
	else if( killer != None )
	{
		killer.killCount++;
		killer.Score += 1;
		Teams[killer.Team].Score += 1;
	}
	
	if ( (GoalTeamScore > 0) && (Teams[killer.Team].Score >= GoalTeamScore) )
		EndGame("GoalTeamScore");
*/
}


function AEonKilled( pawn killer, pawn Other, name damageType )
{
	local string killerName;  //AEon
	local string logmsg;   //AEon
	local string datestamp;   //AEon
	local string zero;     //AEon
	local string killed;
	
	if (Other.bIsPlayer)
	{
		killed = Other.PlayerReplicationInfo.PlayerName;
		if (killer != none)
			killerName = killer.PlayerReplicationInfo.PlayerName; //AEon
	
		// AEon: Getting rid of stupid death messages! in dlog file!
		// Frag syntax: 
		//   "F/"<killed>"/"<killer>"/"<DamageType>"/"<Weapon ked>"/"<Weapon ker>"/"
		//    Suicides:     <killed>=<killer>
		//    Environment Deaths: <killer>=<killed>
		
		//Bulding a nice datestamp that will be easy to parse->
		datestamp = ""$Level.Year;
		if( Level.Month < 10 ) zero = "0"; else zero = "";
			datestamp = datestamp $ zero $Level.Month;
		if( Level.Day < 10 ) zero = "0"; else zero = "";
			datestamp = datestamp $ zero $Level.Day;
		if( Level.Hour < 10 ) zero = "0"; else zero = "";
			datestamp = datestamp $ zero $Level.Hour;
		if( Level.Minute < 10 ) zero = "0"; else zero = "";
			datestamp = datestamp $ zero $Level.Minute;
		if( Level.Second < 10 ) zero = "0"; else zero = "";
			datestamp = datestamp $ zero $Level.Second;   //AEon<-
		
		//Frag dlog, Date & Time, Map Name,
		logmsg = "F/"$datestamp$"/"$Level.Title$"/";  //AEon
		
		//Fragged, FraggerF,
		if( killer==None || killer==other )
		{
			if( killer==None )                   //AEon enviro deaths->
				logmsg = logmsg$killed$"/"$killed$"/";         
			else
				logmsg = logmsg$killerName$"/"$killerName$"/"; //AEon suicide<-
		}
		else
		{ 
			logmsg = logmsg$killed$"/"$killerName$"/";       //AEon
		} 
		
		logmsg = logmsg$damageType$"/"$           //AEon->
		Other.Weapon$"/"$killer.Weapon$"/"$
		Other.bAltFire$"/"$killer.bAltFire$"/"$
		Other.Health$"/"$killer.Health$"/"$
		Other.SelectedItem$"/"$killer.SelectedItem$"/";
		alog(logmsg);                     //AEon<-
	}
}


// Trys to find the team specified, but will return the smallest
// team (defined first by size and then by score) if the optional 
// parameter is omitted.  Returns TC_None if it can't find a team.
// Used for spawning, see GetTeamNumber() if you want a number
// for a given team name.
function int FindTeam(optional coerce string PlayerTeam)
{
	local int i;
	local NnCTFTeamInfo SmallestTeam;
	
	SmallestTeam = None;
	
	/*
	// First check if the player is joining the spectator team
	if ( PlayerTeam ~= GetTeamNameForNum(TC_Spectator) )
	{
		return(TC_Spectator);
	}
	*/
	
	if ( NnGRI == None )
		log(class$": FindTeam(): ERROR: NnGRI==None");
	
	//log(class$": FindTeam(): NnGRI.TC_First="$NnGRI.TC_First);
	//log(class$": FindTeam(): NnGRI.TC_Last="$NnGRI.TC_Last);
	
	for (i = TC_First; i <= TC_Last; i++)
	{
		if (NnGRI.Teams[i] != None)
		{
			/*
			if ( SmallestTeam == None )
				log(class$": FindTeam(): SmallestTeam==None");
			else {
				log(class$": FindTeam(): SmallestTeam.Size="$SmallestTeam.Size);
				log(class$": FindTeam(): SmallestTeam.Score="$SmallestTeam.Score);
				log(class$": FindTeam(): SmallestTeam.TeamName="$SmallestTeam.TeamName);
			}
			log(class$": FindTeam(): NnGRI.Teams["$i$"].Size="$NnGRI.Teams[i].Size);
			log(class$": FindTeam(): NnGRI.Teams["$i$"].Score="$NnGRI.Teams[i].Score);
			log(class$": FindTeam(): NnGRI.Teams["$i$"].TeamName="$NnGRI.Teams[i].TeamName);
			*/
			
			if ((SmallestTeam == None) || 
			    (SmallestTeam.Size > NnGRI.Teams[i].Size) || 
			    ((SmallestTeam.Size == NnGRI.Teams[i].Size) && (SmallestTeam.Score > NnGRI.Teams[i].Score)))
			{
				SmallestTeam = NnGRI.Teams[i];
			}

			if ( (NnGRI.Teams[i].TeamName ~= PlayerTeam) )
			{
				if (NnGRI.Teams[i].Size < MaxTeamSize)
				{
					return(i);
				}
				else
				{
					dlog ("NnCTF: FindTeam Error in Teams["$i$"].Size < MaxTeamSize: "$NnGRI.Teams[i].Size$"<"$MaxTeamSize);
					return(TC_None);
				}
			}
		}
	}
	
	if ( SmallestTeam.Size < MaxTeamSize )
	{
		log(class$": FindTeam(): Returning="$SmallestTeam.TeamIndex);
		return(SmallestTeam.TeamIndex);
	}
	else
	{
		//dlog ("NnCTF: FindTeam Error in SmallestTeam.Size: "$SmallestTeam.Size$"<"$MaxTeamSize$"!"$PlayerTeam);
		log(class$": FindTeam(): Returning=TC_Spectator");
		return (TC_Spectator);
	}
}

function bool ValidateTeamName( string TeamName )
{
	local NnCTFTeamInfo tmpTI;
	
	foreach AllActors(class'NnCTFTeamInfo', tmpTI)
	{
		if ( TeamName ~= tmpTI.TeamName )
			return true;
	}
	return false;
}

// Trys to put the player on the requested team.
function bool ChangeTeamStr(Pawn Other, coerce string PlayerTeam)
{
	local bool bReturn;
	
	//dlog("NnCTF: NnCTFGame: ChangeTeamStr(): Passed="$PlayerTeam);
	//dlog("NnCTF: NnCTFGame: ChangeTeamStr(): Passing="$GetTeamNumForName(PlayerTeam));
	bReturn = ChangeTeam(Other, GetTeamNumForName(PlayerTeam));
	return bReturn;
}

function bool ChangeTeam(Pawn Other, int NewTeam)
{
	local int i, s;
	local byte OldTeam;
	local pawn APlayer;
	local teaminfo SmallestTeam;

	/*
	dlog("NnCTF: NnCTFGame: ChangeTeam(): NewTeam: "$NewTeam);
	dlog("NnCTF: NnCTFGame: ChangeTeam(): Teams[NewTeam].TeamName: "$NnGRI.Teams[NewTeam].TeamName);
	dlog("NnCTF: NnCTFGame: ChangeTeam(): Teams[NewTeam].TeamIndex: "$NnGRI.Teams[NewTeam].TeamIndex);
	dlog("NnCTF: NnCTFGame: ChangeTeam(): Other.PRI.PlayerName: "$Other.PlayerReplicationInfo.PlayerName);
	dlog("NnCTF: NnCTFGame: ChangeTeam(): Other.PRI.TeamName: "$Other.PlayerReplicationInfo.TeamName);
	dlog("NnCTF: NnCTFGame: ChangeTeam(): Other.PRI.Team: "$Other.PlayerReplicationInfo.Team);
	dlog("NnCTF: NnCTFGame: ChangeTeam(): Teams[Other.PRI.Team].TeamName: "$NnGRI.Teams[Other.PlayerReplicationInfo.Team].TeamName );
	dlog("NnCTF: NnCTFGame: ChangeTeam(): bNoTeamChanges: "$bNoTeamChanges);
	*/
	
	//if (NewTeam > NnGRI.NumTeams) NewTeam = NnGRI.NumTeams;

	//log("ChangeTeam(): NnGRI.NumTeams == "$NnGRI.NumTeams);
	if ( !(Other.PlayerReplicationInfo.Team > NnGRI.NumTeams) 
		&& (Other.PlayerReplicationInfo.TeamName == NnGRI.Teams[Other.PlayerReplicationInfo.Team].TeamName) )
	{
		if ( bNoTeamChanges ) {
			dlog("NnCTF: NnCTFGame: ChangeTeam(): Error 1");
			return false;
		}
	}
	
	SmallestTeam = None;
	
	//  dlog("NnCTF: NnCTFGame: ChangeTeam(): MaxTeams: "$MaxTeams);
	//  dlog("NnCTF: NnCTFGame: ChangeTeam(): MaxTeamSize: "$MaxTeamSize);
	
	// First, check if the specified team is a legitimate one
	if ( NnGRI.Teams[NewTeam] == None )
	{
		dlog("NnCTF: NnCTFGame: ChangeTeam(): ERROR: NewTeam is not a legitimate team: "$NewTeam);
		return false;
	}

	// Now, see if the particular team will allow the player to join in
	if ( !IsAllowedOnTeam(Other, NewTeam) )
	{
		dlog("NnCTF: NnCTFGame: ChangeTeam(): "$Other.PlayerReplicationInfo.PlayerName$" failed to join the "$NnGRI.Teams[NewTeam].TeamName$" team.");
		return false;
	}
	
	// The player has passed the various checks.  Add them to the team
	/*
	if (NnGRI.Teams[Other.PlayerReplicationInfo.Team] != None )
	{
		dlog("NnCTF: NnCTFGame: ChangeTeam(): Removing "$Other.PlayerReplicationInfo.TeamName$" from team "$Other.PlayerReplicationInfo.Team);
		NnGRI.Teams[Other.PlayerReplicationInfo.Team].Size--;
	}
	*/
	//OldTeam = Other.PlayerReplicationInfo.Team;
	
	// Ob1: need to add a check so that a player wont be removed from a
	//      team when first logging in
	RemoveFromTeam(Other);
	AddToTeam(NewTeam, Other);
	
	//  dlog("NnCTF: NnCTFGame: ChangeTeam(): PlayerName: "$Other.PlayerReplicationInfo.PlayerName);
	//  dlog("NnCTF: NnCTFGame: ChangeTeam(): TeamName: "$Other.PlayerReplicationInfo.TeamName);
	//  dlog("NnCTF: NnCTFGame: ChangeTeam(): Team: "$Other.PlayerReplicationInfo.Team);
	
	return true;
/*
	for( i=0; i<MaxTeams; i++ ) {
		dlog("NnCTF: NnCTFGame: ChangeTeam(): Teams["$i$"].Size: "$Teams[i].Size);
		if ( (Teams[i].Size < MaxTeamSize) &&
		     ((SmallestTeam == None) || (SmallestTeam.Size > Teams[i].Size)) )
		{
			s = i;
			SmallestTeam = Teams[i];
		}
	}
	
	for( i=0; i<MaxTeams; i++ )
	{
		if ( i == NewTeam )
		{
			if (Teams[i].Size < MaxTeamSize)
			{
				AddToTeam(i, Other);
				return true;
			}
			else 
				break;
		}
	}
	
	if ( (SmallestTeam != None) && (SmallestTeam.Size < MaxTeamSize) )
	{
		AddToTeam(s, Other);
		return true;
	}
	
	dlog("NnCTF: NnCTFGame: ChangeTeam(): Error 2");
	return false;
	*/
}


// Checks to see if Other is allowed on the team NewTeam.
// DLL: At this point, it is assumed that the team 'NewTeam' exists in the
//   Teams[] array, so there is no need to check.
function bool IsAllowedOnTeam(Pawn Other, int NewTeam)
{
	if (NnGRI.Teams[NewTeam].Size >= MaxTeamSize)
	{
		dlog("NnCTF: NnCTFGame: IsAllowedOnTeam(): Team at the max. players limit");
		return false;
	}
	
	return true;
}


function RemoveFromTeam( pawn Exiting )
{
	local int OldTeam;
	
	OldTeam = Exiting.PlayerReplicationInfo.Team;
	if (OldTeam > NnGRI.numteams) return;
	
	if ( NnGRI.Teams[OldTeam] != None && OldTeam != NnGRI.TC_Spectator )
		NnGRI.Teams[OldTeam].Size--;
}


// Modified from UnrealShare.TeamGame
function AddToTeam( int num, Pawn Other )
{
	local texture NewSkin;
	local string NewSkinStr, PrefixStr, MidStr;
	local NnCTFTeamInfo aTeam;
	local Pawn P;
	local bool bSuccess;
	aTeam = NnGRI.Teams[num];

	// Make sure the player's NnPlayerReplicationInfo specific info is updated
	NnPlayerReplicationInfo(Other.PlayerReplicationInfo).TeamColorR = aTeam.TeamColorR;
	NnPlayerReplicationInfo(Other.PlayerReplicationInfo).TeamColorG = aTeam.TeamColorG;
	NnPlayerReplicationInfo(Other.PlayerReplicationInfo).TeamColorB = aTeam.TeamColorB;
	
	aTeam.Size++;
	Other.PlayerReplicationInfo.Team = num;
	Other.PlayerReplicationInfo.TeamName = aTeam.TeamName;
	bSuccess = false;
	
	if ( CTFPlayer(Other) != None )
		CTFPlayer(Other).ClientSetMusicStr(aTeam.CurrentMusicStr, aTeam.CurrentSection, aTeam.CurrentTransition);
	
	if ( Other.IsA('PlayerPawn') )
		Other.PlayerReplicationInfo.TeamID = 0;
	else
		Other.PlayerReplicationInfo.TeamID = 1;
	
	while ( !bSuccess )
	{
		bSuccess = true;
		for ( P=Level.PawnList; P!=None; P=P.nextPawn )
			if ( P.bIsPlayer && (P != Other) &&
                 (P.PlayerReplicationInfo.Team == Other.PlayerReplicationInfo.Team) &&
			     (P.PlayerReplicationInfo.TeamId == Other.PlayerReplicationInfo.TeamId) )
				bSuccess = false;
		if ( !bSuccess )
			Other.PlayerReplicationInfo.TeamID++;
	}
	
	BroadcastMessage(Other.PlayerReplicationInfo.PlayerName$NewTeamMessage$aTeam.TeamName, false);
	ilog(Other.PlayerReplicationInfo.PlayerName$NewTeamMessage$aTeam.TeamName);

	if ( aTeam.TeamIndex == NnGRI.TC_Spectator )
	{
		NewSkin = texture'DefaultTexture';
		Other.Skin = NewSkin;
	} else {
		PrefixStr = GetSkinPrefix(aTeam,Other);
		MidStr = GetSkinMid(Other);
		NewSkinStr = PrefixStr $ MidStr $ aTeam.TeamName;
		ilog("Skin Change: Player="$ Other.PlayerReplicationInfo.PlayerName $"  Skin="$ NewSkinStr);
		NewSkin = texture(DynamicLoadObject(NewSkinStr, class'texture'));
		
		if ( NewSkin != None )
			Other.Skin = NewSkin;
		else 
		{
			ilog("ERROR: with skin change.  Attempting to use default Unreal skins.");
			NewSkin = texture(DynamicLoadObject(GetItemName(string(Other.Mesh))$"Skins.T_"$aTeam.TeamName, class'Texture'));
			if ( NewSkin != None )
				Other.Skin = NewSkin;
			else
				ilog("ERROR, CRITICAL: with skin change.  Failure to use default Unreal skins!");
		}
	}
	
	// DLL: If the player changes either to or from spectator mode, the grenade amounts will have to be
	//   adjusted.
	CheckPlayerCount();
}

function string GetSkinMid( pawn Other )
{
	if ( !bUseCustomSkins )
		return "Skins.T_";
	else {
		switch (Other.class)
		{
			case class'FemaleOne':
			case class'CTFFemaleOne':
			case class'FemaleOneBot':
			
			case class'MaleOne':
			case class'CTFMaleOne':
			case class'MaleOneBot':
			
			case class'MaleTwo':
			case class'CTFMaleTwo':
			case class'MaleTwoBot':
			
			case class'MaleThree':
			case class'CTFMaleThree':
			case class'MaleThreeBot':
			
			case class'SkaarjPlayer':
			case class'CTFSkaarjPlayer':
			case class'SkaarjPlayerBot':
				return "SkinsNnCTF.CtfT_";
			
			case class'FemaleTwo':
			case class'CTFFemaleTwo':
			case class'FemaleTwoBot':
			
			default:
				return "Skins.T_";
		}
	}
}

function string GetSkinPrefix( NnCTFTeamInfo aTeam, pawn Other )
{
	switch (Other.class)
	{
		case class'FemaleOne':
		case class'CTFFemaleOne':
		case class'FemaleOneBot':
			return "Female1";
			
		case class'FemaleTwo':
		case class'CTFFemaleTwo':
		case class'FemaleTwoBot':
			return "Female2";
		
		case class'MaleOne':
		case class'CTFMaleOne':
		case class'MaleOneBot':
			return "Male1";
		
		case class'MaleTwo':
		case class'CTFMaleTwo':
		case class'MaleTwoBot':
			return "Male2";
		
		case class'MaleThree':
		case class'CTFMaleThree':
		case class'MaleThreeBot':
			return "Male3";
		
		case class'SkaarjPlayer':
		case class'CTFSkaarjPlayer':
		case class'SkaarjPlayerBot':
		
		default:
			return GetItemName(string(Other.Mesh));
	}
}


function NavigationPoint FindRandomPoint()
{
	local NavigationPoint tmpPoint;
	local int i, selection;
	
	foreach AllActors( class'NavigationPoint', tmpPoint )
	{
		if ( tmpPoint.IsA('PlayerStart') ||
		     tmpPoint.IsA('PathNode') )
		{
			i++;
		}
	}
	
	Selection = RandRange(0, i);
	i = 0;
	foreach AllActors( class'NavigationPoint', tmpPoint )
	{
		if ( tmpPoint.IsA('PlayerStart') ||
		     tmpPoint.IsA('PathNode') )
		{
			if ( i == Selection )
				return tmpPoint;
			else
				i++;
		}
	}
}


function NavigationPoint FindPlayerStart(optional byte Team, optional string incomingName)
{
	local PlayerStart Dest, Candidate[4], Best;
	local float Score[4], BestScore, NextDist;
	local pawn OtherPlayer;
	local int i, num;
	local Teleporter Tel;
	local bool bTeamStartsFound;
	local flag f;
	
	if( incomingName!="" )
		foreach AllActors( class 'Teleporter', Tel )
			if( string(Tel.Tag)~=incomingName )
				return Tel;
	
	//if (!bSpawnInitiallyInBase || incomingName == "" || Team == TC_Spectator || Team == 0 )
	/* Ob1: this code caused players not appear at own base
	if (!bSpawnInTeamArea)
	{
		incomingName = "None";    // find a non-team start
		Team = 0;
		dlog(class$": FindPlayerStart(): Looking for non-team start");
	}
	else
		dlog(class$": FindPlayerStart(): Looking for " $ incomingName $ " team start");
	*/
	
	num = 0;
	// find team start points
	foreach AllActors(class'PlayerStart', Dest)
	{
		dlog(class$": FindPlayerStart(): Testing="$Dest.Name$", Team="$Dest.TeamNumber$", ownerTeam="$Dest.ownerTeam);
		//if (incomingName == string(Dest.ownerTeam))  // find a start matching the correct mask
		if ( (bSpawnInTeamArea || bForceBaseSpawn) 
			&& ((Team == Dest.TeamNumber) || (incomingName == string(Dest.ownerTeam))) )
		{  
			if (num<4)
				Candidate[num] = Dest;
			else if (Rand(num) < 4)
				Candidate[Rand(4)] = Dest;
			num++;
			dlog(class$": FindPlayerStart(): team start found!, name="$Dest.Name);
		}
	}
	
	if (num != 0) bTeamStartsFound = true;
	else bTeamStartsFound = false;
	i=num;
	
	// add non-team points to list if no team-points were found
	if ( !bForceBaseSpawn )
		foreach AllActors( class'PlayerStart', Dest )
		{
			// so that another teams PlayerStart isn't picked
			// don't add non-team points if bForceBaseSpawn = true
			dlog(class$": FindPlayerStart(): Pass2: Testing="$Dest.Name$", Team="$Dest.TeamNumber$", ownerTeam="$Dest.ownerTeam);
			if ( /*(!bSpawnInTeamArea && ((Team == Dest.TeamNumber) || (string(Dest.ownerTeam)) ~= incomingName)) 
				||*/ (Dest.ownerTeam == 'None') || (Team == TC_Spectator))
			{
			  	if (num<4)
					Candidate[num] = Dest;
			  	else if (Rand(num) < 4)
					Candidate[Rand(4)] = Dest;
				dlog(class$": FindPlayerStart(): Pass2: candidate start found!, name="$Dest.Name);
				if ( ((num - i) == (3 - i)) && bTeamStartsFound && bSpawnInTeamArea)
				{
					dlog(class$": FindPlayerStart(): Pass2: "$string(3-i)$" non-team candidates added.");
					break; // so that there are team and non-team points
				}
				num++;
			}
		}
	
	if (!bNoDebugMessages)
	{
		for (i=0; i<4; i++)
			log(class$": FindPlayerStart(): candidate number="$i$", name="$candidate[i].Name);
		log(class$": FindPlayerStart(): num="$num);
	}
	
	// if we couldn't find any matching the mask, just look for any
	if ( num == 0 )
	{
		dlog(class$": FindPlayerStart(): WARNING: Couldn't find " $ incomingName $ " team start");
		foreach AllActors(class'PlayerStart', Dest)
		{
			if ((Dest.ownerTeam == 'None') || (Team == TC_Spectator))
			{
			  if (num<4)
				Candidate[num] = Dest;
			  else if (Rand(num) < 4)
				Candidate[Rand(4)] = Dest;
			  num++;
			}
		}
	}
	
	/*
	N = Level.NavigationPointList;
	While ( N != None )
	{
		if ( N.IsA('PlayerStart') )
		{
			if (num<4)
				Candidate[num] = PlayerStart(N);
			else if (Rand(num) < 4)
				Candidate[Rand(4)] = PlayerStart(N);
			num++;
		}
		N = N.nextNavigationPoint;
	}
	
	if (num == 0 )
		foreach AllActors( class 'PlayerStart', Dest )
		{
			if ( !bSpawnInTeamArea || (Team == Dest.TeamNumber) )
			{
				if (num<4)
					Candidate[num] = Dest;
				else if (Rand(num) < 4)
					Candidate[Rand(4)] = Dest;
				num++;
			}
		}
	*/  
	if (num > 4)
		num = 4;
	else if (num == 0)
	{
		dlog(class$": FindPlayerStart(): ERROR: num==0, function aborting");
		return None;
	}
	
	//assess candidates
	for (i=0;i<num;i++)
		Score[i] = 4000 * FRand(); //randomize
	
	OtherPlayer = Level.PawnList;
	while ( OtherPlayer != None )
	{
		if (OtherPlayer.bIsPlayer)
		{
			for (i=0;i<num;i++)
			{
				NextDist = VSize(OtherPlayer.Location - Candidate[i].Location);
				Score[i] += NextDist;
				if (NextDist < CollisionRadius + CollisionHeight)
					Score[i] -= 1000000.0;
				else if ( (NextDist < 1400) && (Team != OtherPlayer.PlayerReplicationInfo.Team) && OtherPlayer.LineOfSightTo(Candidate[i]) )
					Score[i] -= 2000.0;
				// don't want player to pop up in sight of enemy flag
				foreach candidate[i].visibleactors(class'Flag', f)
					if ((f != none) && ((GetTeamNumForName(string(f.teamname)) != Team) 
							/*|| (!bSpawnInTeamArea && !bForceBaseSpawn)*/) )
					{
						dlog(class$": FindPlayerStart(): Assess candidates: candidate="$candidate[i].name$" is within sight of the "$f.teamname$" flag.");
						Score[i] -= 10000.0;
					}
			}
		}
		OtherPlayer = OtherPlayer.NextPawn;
	}
	
	BestScore = Score[0];
	Best = Candidate[0];
	for (i=1;i<num;i++)
	{
		if (Score[i] > BestScore)
		{
			BestScore = Score[i];
			Best = Candidate[i];
		}
	}     
	
	//if (BestScore < -500000.0) //then would telefrag - but then how to handle re-try?
	//  return None;
	
	dlog(class$": FindPlayerStart(): Returning="$Best.Name);
	return Best;
}


// DLL: Prevent the player from accepting any items when they're being set up.
event AcceptInventory(pawn PlayerPawn)
{
	if ( CTFPlayer(PlayerPawn) != None )
		if ( CTFPlayer(PlayerPawn).bBeingSetup )
			return;
	
	Super.AcceptInventory(PlayerPawn);
}


function AddDefaultSpectatorInventory(pawn aPlayer)
{
	local NnTargetingComputer NnTC;
	
	if ( bUseTargetingComputer )
	{
		if ( aPlayer.IsA('PlayerPawn') ||
		     aPlayer.FindInventoryType(class'NnTargetingComputer') != None )
		{
			NnTC = Spawn(class'NnTargetingComputer',,,Location);
			if ( NnTC != None )
			{
				NnTC.bHeldItem = true;
				NnTC.GiveTo( aPlayer );
				NnTC.Activate();
				aPlayer.SelectedItem = NnTC;
			}
		}
	}
}


function AddDefaultInventory(pawn aPlayer)
{
	local Inventory Hook;
	local int i;
	local bool bUseHook;
	local SearchLight s;
	local NnTargetingComputer NnTC;
	
	if ( aPlayer.IsA('CTFPlayer') )
		if ( CTFPlayer(aPlayer).bSpectating )
		{
			AddDefaultSpectatorInventory(aPlayer);
			return;
		}
	Super.AddDefaultInventory(aPlayer);
	
	if (bHook)
	{
		if ( OpenModGI.LevelPrefs != None )
		{
			//dlog("NnCTF: NnCTFGame: AddDefaultInventory(): OpenModGI.LevelPrefs != None");
			//dlog("NnCTF: NnCTFGame: AddDefaultInventory(): bDisableGrapple="$OpenModGI.LevelPrefs.bDisableGrapple);
			//dlog("NnCTF: NnCTFGame: AddDefaultInventory(): bIgnoreLevelPrefs="$bIgnoreLevelPrefs);
			if ( !(OpenModGI.LevelPrefs.bDisableGrapple && !bIgnoreLevelPrefs) )
				bUseHook = true;
		}
		else
			bUseHook = true;
		
		if ( bUseHook )
		{
			dlog("NnCTF: NnCTFGame: AddDefaultInventory(): Using Hook, class="$HookClass);
		
			if ( aPlayer.FindInventoryType(HookClass) == None)
			{
				Hook = Spawn(class<Inventory>(HookClass),,,Location);
				if (Hook != None)
				{
					Hook.bHeldItem = true;
					Hook.GiveTo( aPlayer );
					//aPlayer.SelectedItem = Hook;
				}
			}
		}
	}
	
	if ( bUseTargetingComputer )
	{
		if ( aPlayer.IsA('PlayerPawn') ||
		     aPlayer.FindInventoryType(class'NnTargetingComputer') != None )
		{
			NnTC = Spawn(class'NnTargetingComputer',,,Location);
			if ( NnTC != None )
			{
				NnTC.bHeldItem = true;
				NnTC.GiveTo( aPlayer );
				NnTC.Activate();
				aPlayer.SelectedItem = NnTC;
			}
		}
	}
	
	if ( OpenModGI.LevelPrefs != None )
	{
		if ( OpenModGI.LevelPrefs.bDarkmatch )
		{
			s = Spawn(class'SearchLight',,, Location);
			if (s != None)
			{
				s.bHeldItem = true;
				s.GiveTo( aPlayer );
				s.Activate();
				aPlayer.SelectedItem = s;
			}
		}
	}
}


/*
function bool AddBot()
{
	BroadcastMessage("Bots are currently unsupported in NoName CTF", true);
	
	// DLL: AddBot must return true else it will continually be called
	return true;
}
*/

function bool AddBot()
{
	local NavigationPoint StartSpot;
	local bots NewBot;
	local int BotN, DesiredTeam;

	BotN = BotConfig.ChooseBotInfo();
	
	// Find a start spot.
	StartSpot = FindPlayerStart(0);
	if( StartSpot == None )
	{
		log("Could not find starting spot for Bot");
		return false;
	}

	// Try to spawn the player.
	NewBot = Spawn(BotConfig.GetBotClass(BotN),,,StartSpot.Location,StartSpot.Rotation);
	NewBot.PlayerReplicationInfo = Spawn(class'NnPlayerReplicationInfo', NewBot);

	if ( NewBot == None )
		return false;

	if ( (bHumansOnly || Level.bHumansOnly) && !NewBot.bIsHuman )
	{
		NewBot.Destroy();
		log("Failed to spawn bot");
		return false;
	}

	StartSpot.PlayTeleportEffect(NewBot, true);

	// Init player's information.
	BotConfig.Individualize(NewBot, BotN, NumBots);
	NewBot.ViewRotation = StartSpot.Rotation;
	NnPlayerReplicationInfo(NewBot.PlayerReplicationInfo).bNoCarryFlag = !bBotsCanCarryFlags;

	// broadcast a welcome message.
	BroadcastMessage( NewBot.PlayerReplicationInfo.PlayerName$EnteredMessage, true );

	AddDefaultInventory( NewBot );
	NumBots++;

	//DesiredTeam = BotConfig.GetBotTeam(BotN);
	DesiredTeam = FindTeam();
	if ( (DesiredTeam == 255) || !ChangeTeam(NewBot, DesiredTeam) )
	{
		ChangeTeam(NewBot, NextBotTeam);
		NextBotTeam++;
		if ( NextBotTeam >= MaxTeams )
			NextBotTeam = 0;
	}

	if ( bSpawnInTeamArea )
	{
		StartSpot = FindPlayerStart(newBot.PlayerReplicationInfo.Team);
		if ( StartSpot != None )
		{
			NewBot.SetLocation(StartSpot.Location);
			NewBot.SetRotation(StartSpot.Rotation);
			NewBot.ViewRotation = StartSpot.Rotation;
			NewBot.ClientSetRotation(NewBot.Rotation);
			StartSpot.PlayTeleportEffect( NewBot, true );
		}
	}

	return true;
}



function string GetVersionString()
{
	return ("Version: "$Version);
}


// Finds a charge for a rune of type RuneType
function int GetNewCharge(class RuneType)
{
	local int Last;
	
	do
	{
		Last = RuneChargesIndex++;
		if ( RuneChargesIndex >= MAX_RUNES )
			RuneChargesIndex = 0;
	} until ( RuneList[Last]==RuneType );
	
	return RuneCharges[Last];
}


// DLL: Set Team to 255 for all teams
function StartClientsMusic(string NewMusicStr, byte NewSection, EMusicTransition NewTransition, byte Team)
{
	local Pawn aPawn;
	local int i;
	
	//dlog("NnCTF: NnCTFGame: StartClientsMusic(): ...");
	
	if ( Team != 255 && NnGRI.Teams[Team] != None )
	{
		dlog("NnCTF: NnCTFGame: StartClientsMusic(): Setting Music for Team="$Team);
		NnGRI.Teams[Team].CurrentMusicStr = NewMusicStr;
		NnGRI.Teams[Team].CurrentSection = NewSection;
		NnGRI.Teams[Team].CurrentTransition = NewTransition;
	} else {
		dlog("NnCTF: NnCTFGame: StartClientsMusic(): Setting Music for all teams");
	}
	
	aPawn = Level.PawnList;
	
	while ( aPawn != None )
	{
		if ( aPawn.IsA('PlayerPawn' ))
		{
			if ( Team == 255 || Team == aPawn.PlayerReplicationInfo.Team )
				CTFPlayer(aPawn).ClientSetMusicStr( NewMusicStr, NewSection, NewTransition );
				//PlayerPawn(aPawn).ClientSetMusic( NewMusic, NewSection, 0, NewTransition );
				//playerpawn(apawn).ClientSetMusic( Song, SongSection, CdTrack, MTRAN_Fade );
		}   
		aPawn = aPawn.nextPawn;
	}
}


// PlaySoundAll(): Plays a particular sound, pSound, to all players.  It can also optionally
//   play a sound to only one particular team, in which case bTeamSound must be true and
//   TeamIndex must be set.
function PlaySoundAll( sound pSound,
                       optional ESoundSlot Slot,
                       optional float Volume,
                       optional bool bNoOverride,
                       optional bool bTeamSound,
                       optional byte TeamIndex )
{
	local int i;
	local Pawn aPawn;
	local float pVolume;
	
	if ( Volume == 0.0 )
		pVolume = 2.0;
	else
		pVolume = Volume;
	
	aPawn = Level.PawnList;
	
	while ( aPawn != None )
	{
		if ( aPawn.bIsPlayer && aPawn.IsA('PlayerPawn') )
		{
			if ( !bTeamSound || TeamIndex == aPawn.PlayerReplicationInfo.Team )
			{
				PlayerPawn(apawn).PlaySound(pSound,Slot,pVolume,bNoOverride,,);
				//PlayerPawn(aPawn).ClientHearSound(aPawn, 0, pSound, aPawn.Location, aPawn.Location);
			}
		}   
		aPawn = aPawn.nextPawn;
	}
} 


/*
function bool IsRelevant( actor Other )
{
	if (Super.IsRelevant(Other))
	{
//		if ( WeaponSet != None )
//		WeaponSet.DoReplacement();
		return true;
	} else {
		return false;
	}
}
*/


function ChangeTeamMusic( int team )
{
	local string NewMusicStr;
	local music NewMusic;
	local byte NewMusicSection, NewMusicTrack;
	local EMusicTransition NewMusicTransition;
	
	local int i;
	local Flag tmpFlag;
	local bool bTeamCarryingFlag, bTeamFlagInBase, bTeamReturningFlag, bIsAnyFlagCarried;
	local name SettingMusicTo;
	
	if ( team == 255 )
	{
		dlog("NnCTF: NnCTFGame: ChangeTeamMusic(): Team = 255, function aborting");
		return;
	}
	
	if ( team == NnGRI.TC_Spectator )
	{
		return;
	}
	
	if ( NnGRI.Teams[team] == None )
	{
		dlog("NnCTF: NnCTFGame: ChangeTeamMusic(): NnGRI.Teams["$team$"]==None, function aborting.");
		return;
	}
	
	for (i=NnGRI.TC_First; i<=NnGRI.TC_Last; i++)
	{
		if ( Team == NnGRI.FlagHolders[i] && Team != i )  // is team carrying a flag?  (and not their own)
		{
			bTeamCarryingFlag = true;
			dlog("NnCTF: NnCTFGame: ChangeTeamMusic(): Team "$i$" carrying flag (FlagHolders["$i$"]="$NnGRI.FlagHolders[i]);
		}
	}
	foreach AllActors( class'Flag', tmpFlag )
	{
		if ( tmpFlag.TeamInfo.TeamIndex == Team )     // is this the team's flag?
		{
			bTeamFlagInBase = ( tmpFlag.GetStateName() == 'InBase' );
			bTeamReturningFlag = ( tmpFlag.GetStateName() == 'Returning' );
		}
		if ( !bIsAnyFlagCarried )             // are any flags being carried?
		{
			bIsAnyFlagCarried = !( tmpFlag.IsInState('InBase') );
		}
	}
	
	SettingMusicTo = 'None';
	if ( !bTeamCarryingFlag && bTeamFlagInBase )
		SettingMusicTo = 'Default';
	if ( bIsAnyFlagCarried || (!bTeamCarryingFlag && bTeamReturningFlag) )
		SettingMusicTo = 'Suspense';
	if ( bTeamCarryingFlag || (!bTeamFlagInBase && !bTeamReturningFlag) )
	{
		SettingMusicTo = 'Action';
		dlog("NnCTF: NnCTFGame: ChangeTeamMusic(): Setting to 'Action': bTeamCarryingFlag="$bTeamCarryingFlag$", bTeamFlagInBase="$bTeamFlagInBase
			$", bTeamReturningFlag="$bTeamReturningFlag);
	}
	if ( SettingMusicTo == 'None' )
	{
		dlog(class$": ChangeTeamMusic(): ERROR: Confusion over music setting!");
	}
	
	if ( LastMusicSetTo[team] != SettingMusicTo )
	{
		//dlog(class$": ChangeTeamMusic(): bTeamCarryingFlag="$bTeamCarryingFlag);
		//dlog(class$": ChangeTeamMusic(): bTeamFlagInBase="$bTeamFlagInBase);
		//dlog(class$": ChangeTeamMusic(): bTeamReturningFlag="$bTeamReturningFlag);
		//dlog(class$": ChangeTeamMusic(): bIsAnyFlagCarried="$bIsAnyFlagCarried);
		dlog(class$": ChangeTeamMusic(): SettingMusicTo="$SettingMusicTo);
		OpenModGI.SetMusicVars( NewMusicStr, NewMusic, NewMusicSection, NewMusicTransition, OpenModGI.MusicInfo, SettingMusicTo );
		StartClientsMusic( NewMusicStr, NewMusicSection, NewMusicTransition, team );
		LastMusicSetTo[team] = SettingMusicTo;
	} else
	{
		dlog(class$": ChangeTeamMusic(): Not Setting Music, SettingMusicTo="$SettingMusicTo);
	}
}


function FlagChangedState( Flag ChangedFlag )
{ 
	local int i;
	//local byte ChangeMusic[8];      // This should be a bool array, but bool arrays aren't allowed.
	//local Flag tmpFlag;
	//local name FlagStates[8];
	
	//dlog("NnCTF: NnCTFGame: FlagChangedState(): ...");
	
	for (i=NnGRI.TC_First; i<=NnGRI.TC_Last; i++)
	{
		if ( NnGRI.Teams[i] != None )
		{
			ChangeTeamMusic(i);
		}
	}
}


function ClientResetMusic( CTFPlayer aPawn )
{
	local NnCTFTeamInfo aTeam;
	
	aTeam = NnGRI.Teams[aPawn.PlayerReplicationInfo.Team];
	
	aPawn.ClientSetMusicStr(aTeam.CurrentMusicStr, aTeam.CurrentSection, aTeam.CurrentTransition);
}


function CheckPlayerCount()
{
	local Pawn aPawn;
	local int PlayerCount, NumGrenades;
	
	// First, find out how many non-spectating players there are
	aPawn = Level.PawnList;
	while ( aPawn != None )
	{
		if ( aPawn.IsA('PlayerPawn') && aPawn.PlayerReplicationInfo.Team != TC_Spectator )
		{
			PlayerCount++;
		}
		
		aPawn = aPawn.nextPawn;
	}
	
	// Now, adjust the amount of grenades in the level
	NumGrenades = PlayerCount * GrenadesPerPlayer;
	AdjustGrenadeAmt( NumGrenades );
}


function AdjustGrenadeAmt( int NumGrenades )
{
	local int i, NumTypes, IndividualAmt;
	
	if ( (GrenadeSpawner == none) || bDisableGrenades )
		return;
	
	// First, find out how many types of grenades there are
	for (i=0; i<GrenadeSpawner.MAX_ITEMS; i++)
	{
		if ( GrenadeSpawner.NewItem[i] != None )
			NumTypes++;
	}
	
	if ( NumTypes != 0 )
		IndividualAmt = NumGrenades / NumTypes;
	else
		return;
	
	log(class$": AdjustGrenadeAmt(): IndividuualAmt="$IndividualAmt);
	
	for (i=0; i<NumTypes; i++)
	{
		PersistentItemSpawner(GrenadeSpawner).ItemSpawnMax[i] = IndividualAmt;
	}
}


function SetupRuneConfig()
{
	local int i, NumFilled, SetChargeTo;
	
	switch ( PresetRuneConfig )
	{
		case 0:
			ilog("Using Preset Rune Config: 4 Runes @ 100% Each");
			RuneList[0] = class'Haste';
			RuneList[1] = class'Regen';
			RuneList[2] = class'Resist';
			RuneList[3] = class'Strength';
			SetChargeTo = 100;
			NumFilled = 4;
			break;
		case 1:
			ilog("Using Preset Rune Config: 8 Runes @ 50% Each");
			RuneList[0] = class'Haste';
			RuneList[1] = class'Regen';
			RuneList[2] = class'Resist';
			RuneList[3] = class'Strength';
			RuneList[4] = class'Haste';
			RuneList[5] = class'Regen';
			RuneList[6] = class'Resist';
			RuneList[7] = class'Strength';
			SetChargeTo = 50;
			NumFilled = 8;
			break;
		case 2:
			ilog("Using Preset Rune Config: 12 Runes @ 10% Each");
			RuneList[0] = class'Haste';
			RuneList[1] = class'Regen';
			RuneList[2] = class'Resist';
			RuneList[3] = class'Strength';
			RuneList[4] = class'Haste';
			RuneList[5] = class'Regen';
			RuneList[6] = class'Resist';
			RuneList[7] = class'Strength';
			RuneList[8] = class'Haste';
			RuneList[9] = class'Regen';
			RuneList[10] = class'Resist';
			RuneList[11] = class'Strength';
		default:
			ilog("Using Custom Rune Config");
			return;
	}
	
	for (i=0; i<MAX_RUNES; i++)
	{
		if ( i < NumFilled )
		{
			RuneCharges[i] = SetChargeTo;
		} else {
			RuneList[i] = None;
			RuneCharges[i] = 0;
		}
		RuneCharges[i] = SetChargeTo;
	}
}


function PlayScoredEffect( byte TeamIndex )
{
	local NnCTFTeamInfo NnTI;
	
	NnTI = GetTeamInfoForNum(TeamIndex);
	NnTI.Flag.FlagHomeBase.PlayScored();
}


function PlayTeleportEffect( actor Incoming, bool bOut, bool bSound)
{
	if ( Pawn(Incoming) != None )
	{
		if ( Pawn(Incoming).PlayerReplicationInfo.Team != NnGRI.TC_Spectator )
		{
			Super.PlayTeleportEffect(Incoming, bOut, bSound);
		}
	}
}


function bool CanSpectate( pawn Viewer, actor ViewTarget )
{
	if ( bAllowSpectating )
		return true;
	
	return false;
}


function DropTeamRunes( NnCTFTeamInfo tmpTI )
{
	local Pawn aPawn;
	
	aPawn = Level.PawnList;
	
	while ( aPawn != None )
	{
		if ( aPawn.PlayerReplicationInfo.Team == tmpTI.TeamIndex )
		{
			if ( CTFPlayer(aPawn) != None )
			{
				CTFPlayer(aPawn).DropRunes(true);
				ResetInvalidRunes(aPawn);	// clear invalid rune list
			}
		}
		aPawn = aPawn.nextPawn;
	}
}


function ViewPoints( bool bEnable )
{
	local PlayerStart ps;
	local ViewablePoint vp;
	local string pointstr;
	local texture tmpTex;

	if ( bEnable )
	{
		foreach AllActors(class'PlayerStart', ps)
		{
			vp = spawn(class'ViewablePoint',,,ps.Location);
			if ( ps.ownerTeam != 'None' )
			{
				vp.tag = 'ViewPoint';
				pointstr = "UnrealShare." $ string(ps.ownerTeam) $ "Skull";
				log(class$": ViewPoints(): Loading, pointstr="$pointstr);
				tmpTex = texture( DynamicLoadObject(pointstr, class'texture') );
				if ( tmpTex != None )
					vp.Texture = tmpTex;
				else
					log(class$": ViewPoints(): WARNING: tmpTex==None");
			}
			vp.bHidden = false;
		}
	} else
	{
		foreach AllActors(class'ViewablePoint', vp)
		{
			if ( vp.tag == 'ViewPoint' )
			{
				vp.Destroy();
			}
		}
	}
}

defaultproperties
{
     bRunes=True
     bDropRunesOnCapture=True
     RuneList(0)=Class'NnCTF.Haste'
     RuneList(1)=Class'NnCTF.Regen'
     RuneList(2)=Class'NnCTF.Resist'
     RuneList(3)=Class'NnCTF.Strength'
     RuneCharges(0)=100
     RuneCharges(1)=100
     RuneCharges(2)=100
     RuneCharges(3)=100
     RuneCharges(4)=100
     RuneCharges(5)=100
     RuneCharges(6)=100
     RuneCharges(7)=100
     RuneCharges(8)=100
     RuneCharges(9)=100
     RuneCharges(10)=100
     RuneCharges(11)=100
     RuneCharges(12)=100
     RuneCharges(13)=100
     RuneCharges(14)=100
     RuneCharges(15)=100
     RuneCharges(16)=100
     RuneCharges(17)=100
     RuneCharges(18)=100
     RuneCharges(19)=100
     RuneCharges(20)=100
     RuneCharges(21)=100
     RuneCharges(22)=100
     RuneCharges(23)=100
     RuneCharges(24)=100
     RuneCharges(25)=100
     RuneCharges(26)=100
     RuneCharges(27)=100
     RuneCharges(28)=100
     RuneCharges(29)=100
     MaxRuneCharge=100
     bAutoFlagReturn=True
     AutoFlagReturnTime=45.000000
     CaptureLimit=5
     bCaptureWithFlag=True
     bAllowSpectating=True
     Version="0.5.2"
     ModPrefix="Nn"
     bNoDebugMessages=True
     bUseTargetingComputer=True
     GrenadesPerPlayer=2.000000
     GrenadeSpawnerClass=Class'uWeaponSpawner.uWeaponsSpawner'
     teamCaptureBonus=10.000000
     playerCaptureBonus=3.000000
     fragCarrierMessage="You fragged the flag carrier"
     fragCarrierBonus=3.000000
     defendRadius=100.000000
     defendMessage=" defends the "
     defendBonus=2.000000
     assistTime=30.000000
     assistBonus=2.000000
     AvailableMusic(0)=Class'OpenModMusicInfo.DMI_Crater'
     AvailableMusic(1)=Class'OpenModMusicInfo.DMI_DigSh_Vortex'
     AvailableMusic(2)=Class'OpenModMusicInfo.DMI_Dusk_Fifth'
     AvailableMusic(3)=Class'OpenModMusicInfo.DMI_Eversmoke_Seti'
     AvailableMusic(4)=Class'OpenModMusicInfo.DMI_Hub2'
     AvailableMusic(5)=Class'OpenModMusicInfo.DMI_Kran'
     AvailableMusic(6)=Class'OpenModMusicInfo.DMI_QueenSong'
     AvailableMusic(7)=Class'OpenModMusicInfo.DMI_Vortex_Warlord'
     MapConverters(0)=RealCTF
     TeamName(0)="Red"
     TeamName(1)="Blue"
     TeamName(2)="Green"
     TeamName(3)="Yellow"
     TeamName(4)="Spectators"
     TeamColor(0)=(R=255)
     TeamColor(1)=(G=128,B=255)
     TeamColor(2)=(G=195)
     TeamColor(3)=(R=195,G=195)
     bSpawnInTeamArea=True
     GoalTeamScore=150.000000
     FragLimit=150
     TimeLimit=30
     bHardCoreMode=False
     InitialBots=0
     bClassicDeathmessages=True
     DefaultPlayerClass=Class'NnCTF.CTFMaleThree'
     ScoreBoardType=Class'NnCTF.NnUnrealScoreBoard'
     GameMenuType=Class'NnCTF.NnTeamGameOptionsMenu'
     HUDType=Class'NnCTF.NnCTFHUD'
     MapListType=Class'NnCTF.NnMapList'
     BeaconName="NnCTF"
     GameReplicationInfoClass=Class'NnCTF.NnGameReplicationInfo'
}
