//=============================================================================
// Flag.
//=============================================================================
class Flag expands Pickup config(NnCTF)
	abstract;

/*
// the model
#exec MESH IMPORT MESH=NnCTFFlag ANIVFILE=MODELS\flag_a.3d DATAFILE=MODELS\flag_d.3d X=0 Y=0 Z=0
#exec MESH ORIGIN MESH=NnCTFFlag X=0 Y=240 Z=0 YAW=64 PITCH=0 ROLL=0
 
#exec MESH SEQUENCE MESH=NnCTFFlag SEQ=All   STARTFRAME=0 NUMFRAMES=15
#exec MESH SEQUENCE MESH=NnCTFFlag SEQ=Wave STARTFRAME=0 NUMFRAMES=15
 
#exec MESHMAP SCALE MESHMAP=NnCTFFlag X=0.1 Y=0.1 Z=0.2
 
// the textures for the subclasses
#exec TEXTURE IMPORT NAME=NnCTFBlueflag FILE=TEXTURES\FLAGS\BlueFlag.PCX GROUP="Flags" MIPS=OFF
#exec TEXTURE IMPORT NAME=NnCTFGreenFlag FILE=TEXTURES\FLAGS\GreenFlag.PCX GROUP="Flags" MIPS=OFF
#exec TEXTURE IMPORT NAME=NnCTFRedFlag FILE=TEXTURES\FLAGS\RedFlag.PCX GROUP="Flags" MIPS=OFF
#exec TEXTURE IMPORT NAME=NnCTFYellowFlag FILE=TEXTURES\FLAGS\YellowFlag.PCX GROUP="Flags" MIPS=OFF

// sounds
#exec AUDIO IMPORT FILE="Sounds\Captured\Buzzer.WAV" NAME="Buzzer" GROUP="Captured"
#exec AUDIO IMPORT FILE="Sounds\Stolen\CTFSiren2.WAV" NAME="CTFSiren1" GROUP="Stolen"
*/

//var() string	TeamName;

var NnFlagBase				FlagHomeBase;		// The home location of the flag
var() class<Actor>			FlagBaseSearchClass;
var() texture				FlagBaseTexture;

var vector					OriginalLocation;
var bool					bOriginalLocationSet;
	
var	NnCTFTeamInfo			TeamInfo;
var NnCTFGame				gameInfo;
var NnGameReplicationInfo	NnGRI;

var() int					autoReturnTime;
var int						TimeLeft;
var bool					bAutoReturn;
var bool					bInHurtZone;

var() sound					CapturedSound;
var() sound					ReturnedSound;
var() sound					ReturningSound;
var() sound					StolenSound;

var() name					TeamName;

var() class<FlagFX>			FlagFXClass;

// This variable ensures that Flag.Base is replicated properly
var Pawn					FlagHolder;

var name					StateName;


replication
{
	reliable if( Role==ROLE_Authority )
		TeamName, FlagHolder, NnGRI, StateName, flagHomeBase, 
		OriginalLocation, bOriginalLocationSet, SetAutoReturnTime;
	
	// Functions client can call.
	reliable if( Role<ROLE_Authority )
		InformGameOfState;  //, ChangeOwner;
			
	//unreliable if( Role<ROLE_Authority )
	//	PlaySoundAll;
}



// overridden in each state
function string GetStateText() {return("Unknown");}
// overridden in each subclass
function int GetTeamNum() {return(gameInfo.TC_None);}

function PreBeginPlay()
{
	OriginalLocation = self.Location;

	Super.PreBeginPlay();
	gameInfo = NnCTFGame(Level.Game);
	
	TimeLeft = 0;
	//GotoState('InBase');
}


function SpawnFlagBase( bool bInvisible )
{
	local Actor tmpPH;
	local vector SpawnLocation;
	local rotator SpawnRotation;
	local bool bFound;
	
	//log(class$": SpawnFlagBase()");

	if ( bInvisible )
	{
		SpawnLocation = self.Location;
	} else {
		//log(class$": SpawnFlagBase(): FlagBaseSearchClass="$FlagBaseSearchClass);
		foreach AllActors( FlagBaseSearchClass, tmpPH )
		{
			//log(class$": SpawnFlagBase(): tmpPH found!");
			SpawnLocation = tmpPH.Location;
			SpawnRotation = tmpPH.Rotation;
			bFound = true;
		}
	}
	
	if ( !bInvisible && !bFound )
	{
		log(class$": Flag base place holder not found, using invisible flag bases.");
		SpawnLocation = self.Location;
		bInvisible = true;
	}
	
	//log(class$": SpawnFlagBase(): bInvisible="$bInvisible);
	//log(class$": SpawnFlagBase(): SpawnLocation="$SpawnLocation);
	
	FlagHomeBase = Spawn(class'NnFlagBase',,, SpawnLocation, SpawnRotation);
	FlagHomeBase.OwnerFlag = self;
	FlagHomeBase.SetPhysics(PHYS_Falling);
	FlagHomeBase.SetVisibility( !bInvisible );
	
	if ( FlagHomeBase == None )
		log(class$": ERROR: FlagHomeBase==None");
	else if ( !bInvisible )
		FlagHomeBase.Skin = FlagBaseTexture;

}


function PostBeginPlay()
{
	Super.PostBeginPlay();
	LoopAnim('Wave',0.7,0.1);
}


function Destroyed()
{
	if (flagHomeBase != None)
		flagHomeBase.Destroy();
}


auto state InBase
{
	function BeginState()
	{
		NetPriority = 2.0;
		bAlwaysRelevant = true;
	}

	function Touch(actor Other)
	{
		local Pawn P;
		
		P = Pawn(Other);
		
		if ( P != None &&
		     P.bIsPlayer &&
		     P.PlayerReplicationInfo.TeamName != self.TeamInfo.TeamName &&
		     !NnPlayerReplicationInfo(P.PlayerReplicationInfo).bNoCarryFlag)
		{
			ChangeOwner(Other);
			GotoState('Stolen');
		}
	}

	function Landed( vector HitNormal )
	{
		if ( !bOriginalLocationSet )
		{
			OriginalLocation = self.Location;
			bOriginalLocationSet = true;
		}
	}

	function string GetStateText() {return("Secure");}
Begin:
	ChangeOwner(None);
	
	SetPhysics(PHYS_Falling);	
	//SetLocation(flagHomeBase.Location);
	SetLocation(OriginalLocation);
	SetRotation(self.Default.Rotation);
	
	InformGameOfState();
}


state NotInBase
{
	function BeginState()
	{	
		NetPriority = 2.0;
		bAlwaysRelevant = true;
		
		// DLL: If the flag is dropped, the clients are not fully informed
		// of the flag's current position.  Ie, the flag will appear to be in
		// one particular position (to each of the clients) while actually being
		// in another.  If simulated movement is turned off, this is fixed as the
		// server has to tell the clients of each new movement (the client can't
		// assume that the flag has or hasn't moved somewhere.)
		//
		// It seems as if this is a bug in the Unreal engine, but for now this
		// will suffice (as a hack-ish fix.)
		//
		// Note: RemoteRole is changed back to simulated in NotInBase.EndState()
		RemoteRole = ROLE_DumbProxy;
	
		if (Pawn(Owner) != None)
			BroadcastMessage(Pawn(Owner).PlayerReplicationInfo.PlayerName$" lost the "$self.TeamInfo.TeamName$" flag!", False);
		ChangeOwner(None);
		
		SetPhysics(PHYS_Falling);
		
		Enable('Timer');
	}
	
	function EndState()
	{
		RemoteRole = ROLE_SimulatedProxy;
		Disable('Timer');
	}
	
	function Timer()
	{
		if (bAutoReturn || bInHurtZone) 
			TimeLeft--;
		if ( (TimeLeft <= 0) && (bAutoReturn || bInHurtZone) )
			ReturnFlag();
	}
	
	function Touch(actor Other)
	{
		local Pawn P;
		
		P = Pawn(Other);
		
		if ( P != None &&
		     P.bIsPlayer &&
		     ( !NnPlayerReplicationInfo(P.PlayerReplicationInfo).bNoCarryFlag ||
		       gameInfo.bAutoFlagReturn ) )
		{
			if (P.PlayerReplicationInfo.TeamName == self.TeamInfo.TeamName)
			{
				if (gameInfo.bAutoFlagReturn)
				{
					ReturnFlag();
				}
				else
				{	
					ChangeOwner(Other);
					GotoState('Returning');
				}
			}
			else
			{
				ChangeOwner(Other);
				GotoState('Stolen');
			}
		}
	}
	
	// Reflect off Wall w/damping
	function HitWall (vector HitNormal, actor Wall)
	{
		Velocity = 0.5*(( Velocity dot HitNormal ) * HitNormal * (-2.0) + Velocity);
		If (VSize(Velocity) < 20)
		{
			bBounce = False;
			bStasis = false;
			SetPhysics(PHYS_None);
		}
	}	
	
	function string GetStateText() {return("Ground");}
	
	
Begin:
	if ( !(autoReturnTime <= 0) )
	{
		//SetTimer(autoReturnTime, true);
		TimeLeft = autoReturnTime;
		bAutoReturn = true;
	}
	else bAutoReturn = false;

	// Ob1: set return time if flag is in lava, slime, etc.
	if ( //(self.Region.Zone.bWaterZone && self.Region.Zone.bPainZone)
		self.Region.Zone.bKillZone || self.Region.Zone.bPainZone)
	{
		bInHurtZone = true;
		if (autoReturnTime <= 0)
			TimeLeft = 30;
	}
	else bInHurtZone = false;
	
	if (bAutoReturn || bInHurtZone)
		SetTimer(1.0, true);

	InformGameOfState();
}


// abstract
state Carried
{
	function BeginState()
	{
		NetPriority = 4.0;
		bAlwaysRelevant = false;
	
		SetFollow();
	}

	function EndState()
	{
		//Disable('Timer');
	}
	
	/*
	simulated function Timer()
	{
		local FlagFX FFX;			//+Gerbil!
		
		if (Level.NetMode!=NM_DedicatedServer)
		{
			FFX = Spawn(self.FlagFXClass);
			FFX.SetTimer(5.0, false);
		}
		
	}
	*/
	
	simulated function Tick(float deltatime)
	{
		Super.Tick(deltatime);
	
		//SetBase(Owner);
	
		// Check if Base is not the Flag carrier anymore
		if (self.Base != Pawn(Owner))
		{
			SetFollow();
		}

		// Check if the Flag carrier has died
		if (Pawn(Owner).Health <= 0)
		{
			DropFlag();
		}	
	}
	
	function string GetStateText() {return("Carried");}
}


state Stolen expands Carried
{
	function BeginState()
	{
		//log("NnCTF: Entering Stolen state");
		BroadcastMessage(Pawn(Owner).PlayerReplicationInfo.PlayerName$" got the "$self.TeamInfo.TeamName$" flag!", False);
		//PlaySound(StolenSound,,2.0);
		GameInfo.PlaySoundAll(StolenSOund, SLOT_MISC,, true);
		Super.BeginState();
	}

	function Touch( actor Other );

	function string GetStateText() {return("Stolen");}
Begin:
	InformGameOfState();
}


state Returning expands Carried
{
	function BeginState()
	{
		//log("NnCTF: Entering Returning State");
		BroadcastMessage(Pawn(Owner).PlayerReplicationInfo.PlayerName$" is returning the "$self.TeamInfo.TeamName$" flag.", False);
		Super.BeginState();
	}

	function Touch(actor Other);
/*	{
		if ( ValidTouch(Other) )
			ReturnFlag();
	}
*/
	function string GetStateText() {return("Returning");}
Begin:
	InformGameOfState();
}


// ChangeOwner();
// Changes ownership of the flag and sets up any other related properties (such as rotation)
function ChangeOwner(actor Other)
{	
	FlagHolder = Pawn(Other);

	// DLL: At this point Owner has not yet been changed.  Thus, Owner is
	//   the previous flag carrier (provided it isn't None)
	
	if (Other != None)
	{
		bCollideWorld = False;
		SetPhysics(PHYS_None);
		Disable('Touch');		// stop those nasty infinite touch bugs
		
		SetOwner(Other);

		NnGRI.FlagHolders[TeamInfo.TeamIndex] = Pawn(Owner).PlayerReplicationInfo.Team;

		SetFollow();
	} else
	{		
		bCollideWorld = True;
		SetPhysics(PHYS_Falling);
		Enable('Touch');
		
		SetOwner(None);
		
		NnGRI.FlagHolders[TeamInfo.TeamIndex] = 255;
		
		SetBase(None);
	}
	//log("NnCTF: Flag: ChangeOwner(): Owner Changed");
}


simulated function Tick(float DeltaTime)
{
	if ( FlagHolder != None && Base == None )
	{
		// DLL: This ensures that the flag properly (and smoothly) follows
		//   the owning player around.
		SetFollow();
	}
	else if ( StateName == 'InBase' )
	{
		// DLL: This ensures that when the flag is in base, clients will see
		//   the flag in base.
		SetLocation(OriginalLocation);
		SetRotation(self.default.Rotation);
	}
}


// DLL: SetFollow() assumes that the Flag's Owner property has been set
simulated function SetFollow()
{
	local rotator tmpRot;
	
	tmpRot = Pawn(Owner).Rotation;	// DLL: This *MUST* be set before Move() is called!
	
	SetLocation(Pawn(Owner).Location);	// Set the location of the Flag the same as for the flag carrier	
	Move( -18 * vector(tmpRot) );

	tmpRot.Pitch += 7500;
	tmpRot.Yaw += 32767;
	tmpRot.Roll += 3500;
	SetRotation(tmpRot);
	
	SetBase(Pawn(Owner));	// Attach the flag to the flag carrier	
	SetPhysics(PHYS_None);
}


function bool ValidTouch(actor Other)
{
	// Have to touch your own FlagBase
	return ((NnFlagBase(Other) != None) &&
	        (Pawn(Owner).PlayerReplicationInfo.TeamName == NnFlagBase(Other).OwnerFlag.TeamInfo.TeamName));
}


singular function BaseHit( actor Other )
{
	local Flag tmpFlag;
	
	// First, make sure that the base was only touched by a member of it's owning team.
	// (ie. the Blue base can only be touched by a blue player, etc.)
	if ( Pawn(Other).PlayerReplicationInfo.Team != TeamInfo.TeamIndex )
		return;
	
	foreach AllActors( class'Flag', tmpFlag )
	{
		if ( tmpFlag.Owner == Other )
		{
			if ( tmpFlag.IsInState('Stolen') &&
			     ( IsInState('InBase') || !gameInfo.bCaptureWithFlag ) )
			{
				tmpFlag.CaptureFlag();
			} else if ( tmpFlag.IsInState('Returning') )
			{
				tmpFlag.ReturnFlag();
			} else
			{
				log(class$": BaseHit(): ERROR: The flag did not do anything!, tmpFlag.class="$tmpFlag.class);
			}
		}
	}
}


function ReturnFlag()
{
	// DLL: ChangeOwner() is called here so that if Bump() is called [again] before
	//   the owner is changed (in the 'InBase' state's Begin: section), it will not
	//   get confused.
	ChangeOwner(None);
	
	BroadcastMessage("The "$self.TeamInfo.TeamName$" flag has been returned.", False);
	GameInfo.PlaySoundAll(ReturnedSound, SLOT_Misc, 10.0, true);
	GotoState('InBase');
}


function CaptureFlag()
{
	local UnrealIPlayer P;
	local Actor OldOwner;
	local Flag F;
	local int x;
	local Pawn Pn;
	local vector tmpLocation;
	local NnCTFTeamInfo CapturingTI;

	// DLL: ChangeOwner() is called here so that if Bump() is called [again] before
	//   the owner is changed (in the 'InBase' state's Begin: section), it will not
	//   get confused.
	OldOwner = Pawn(Owner);
	ChangeOwner(None);
	
	CapturingTI = gameInfo.GetTeamInfoForName(Pawn(OldOwner).PlayerReplicationInfo.TeamName);
	if ( CapturingTI == None )
		log(class$": CaptureFlag(): ERROR: CapturingTI==None");
	
	BroadcastMessage(Pawn(OldOwner).PlayerReplicationInfo.PlayerName$" captured the "$self.TeamInfo.TeamName$" team's flag!", False);
	//OldOwner.PlaySound(CapturedSound,SLOT_Misc,2.0,true);  // DLL: self.PlaySound() fails here.  why?
	GameInfo.PlaySoundAll(CapturedSound, SLOT_Misc,,true);
	
	GameInfo.PlayScoredEffect(CapturingTI.TeamIndex);
	
	// DLL: All calls to AdjustScore(), except for the last one, must set
	//   the optional bDontCheckForEndGame variable so that all scores can
	//   be properly adjusted first.
	
	// Give all the members of the capturing team their bonus points
	foreach AllActors( class'UnrealIPlayer', P)
	{
		if (P.PlayerReplicationInfo.TeamName == CapturingTI.TeamName)
		{
			gameInfo.AdjustScore(P, gameInfo.teamCaptureBonus, true);
		}
	}

	// award assist bonus
	//Pn = gameInfo.GetTeamInfoForNum(Pawn(OldOwner).PlayerReplicationInfo.Team).Assistant;
	Pn = CapturingTI.Assistant;
	if (Pn != None)
	{
		gameInfo.AdjustScore(P, gameInfo.assistBonus, true);
		BroadcastMessage(P.PlayerReplicationInfo.PlayerName $ " gets an assist ", false);
	}

	// bump up the capture count
	gameInfo.AdjustCaptures( CapturingTI, TeamInfo, 1);

	// Give the flag capturer bonus points
	gameInfo.AdjustScore(Pawn(OldOwner), gameInfo.playerCaptureBonus, false);
	
	// Check if all team runes need to be dropped
	if ( gameInfo.bDropRunesOnCapture )
		gameInfo.DropTeamRunes( CapturingTI );

	GotoState('InBase');
} 


function DropFlag()
{
	SetBase(none);	// The Flag carrier has died, unattach from the Flag
	SetPhysics(PHYS_Falling);	// Make the Flag not stick in mid-air
	Velocity = Pawn(Owner).Velocity;	// Make the Flag leap through the air along with the player
	GotoState('NotInBase');
}

// This is called to let NnCTF know about state changes.
function InformGameOfState()
{
	//log("NnCTF: Flag: InformGameOfState()");
	TeamInfo.FlagStateText = GetStateText();
	TeamInfo.FlagStateName = GetStateName();
	StateName = GetStateName();
	
	//log(class$": TeamInfo.FlagStateName="$TeamInfo.FlagStateName);
	
	gameInfo.FlagChangedState(self);
}

event float BotDesireability( pawn Bot )
{
	local Inventory AlreadyHas;
	local float desire;

	if ( !(Bot.PlayerReplicationInfo.TeamName ~= string(TeamName)) || !(IsInState('InBase')) )
		return 20;
	else
		return 0;
}

function SetAutoReturnTime(float num)
{
	autoReturnTime = num;
}

// Ob1: return flag to base if fallen out of world
event FellOutOfWorld()
{
	ReturnFlag();
}

defaultproperties
{
     autoReturnTime=45
     CapturedSound=Sound'NnCTFMedia.Captured.Buzzer'
     ReturnedSound=Sound'UnrealI.Pickups.Invisible'
     ReturningSound=Sound'UnrealShare.Pickups.TransA3'
     StolenSound=Sound'NnCTFMedia.Stolen.CTFSiren1'
     Mesh=LodMesh'NnCTFMedia.NnCTFFlag'
     bAlwaysRelevant=True
     LightType=LT_Steady
     LightBrightness=255
     LightSaturation=64
     LightRadius=20
     NetPriority=4.000000
}
