//=============================================================================
// CTFPlayer.
//=============================================================================
class CTFPlayer expands UnrealIPlayer
	config(NnCTF)
	abstract;


var NnGameReplicationInfo	 NnGRI;			// Contains various team/player info

var string		Version;

var bool			bBeingSetup;
var bool			bSpectating;
var bool			bDisplayTeamMenu;
var() int			SpectateAllowed;		// Can the player spectate?  (Received from server)
                                            // 0 = Data not retrieved
                                            // 1 = Yes
                                            // 2 = No
var PlayerPawn		Watching;				// The player you're watching
var() config name	SpectateModes[10];		// The order of states one can spectate in
var int				SpectatingOn;			// Which spectating mode one is in
var int				UnderWaterTimeBkup;		// A backup variable used when switching modes

//var int				NumFlags;				// The number of flags the player is carrying

//var	MessageInfo		PostOffice;				// Handles all incoming and outgoing talk/chat msgs

// Rune Information
var class			ValidRunes[10];			// A list of valid player runes
//var LinkedList		InvalidRunes;			// A list of specific runes which a player can't pick up
//var int				TotalRuneCharge;		// The total of all the player's rune charges
//var int				MaxRuneCharge;			// The maximum rune charge a player is allowed

// Sounds
var(CTFPlayerSounds) sound	HasteSnd;
var(CTFPlayerSounds) sound	RegenSnd;
var(CTFPlayerSounds) sound	ResistSnd;
var(CTFPlayerSounds) sound	StrengthSnd;
var(CTFPlayerSounds) sound	MessageSnd;

// HUD variables
var() config bool			bNoITSHUD;
var() config bool			bNoTeamInfo;
var() config bool			bNoRunesInfo;
var() config bool			bNoDynamicMusic;
var() config bool			bNoTargetingComputer;
var NnTargetingComputer		TargetingComputer;

// More UI stuff
var() color					PlayerColor;

// Music stuff
var(PlayerMusic) music				NewMusic;
var(PlayerMusic) byte				NewMusicSection;
var(PlayerMusic) byte				NewMusicTrack;
var(PlayerMusic) EMusicTransition	NewMusicTrans;

// Debugging Stuff
var PlayerStart				LastStart;		// Used in NextStart()

replication
{
	// Things the server should send to the client
	reliable if( Role==ROLE_Authority && bNetOwner )
		SpectateAllowed, Version, Watching, NnGRI,
		bNoITSHUD, bNoTeamInfo, bNoRunesInfo, bNoDynamicMusic,
		TargetingComputer;

	// Client-side functions
	reliable if ( Role == ROLE_Authority )
		ChangeTeamMenu, ClientSetMusicStr;
		
	// Functions client can call
	reliable if( Role<ROLE_Authority )
		Spectator, ChaseCam, ChangeToSpectator, Player, ServerRespawnPlayer,
		ServerCheckSpectate, ServerSendVersion, WatchNextPlayer, SpectatingFire,
		SpectatingAltFire, ServerGetDiff, GetDiff, ChangeTeamStr, ITSHUD, TeamInfo,
		RunesInfo, DynamicMusic, ShowMOTD, DropRunes, AboutNnCTF, NextStart,
		MoveToStart, ShowPoints, HidePoints;
}


function PreBeginPlay()
{
	Super.PreBeginPlay();
	//log("NnCTF: NnCTFPlayer: PreBeginPlay()");
	ServerCheckSpectate();
	//ServerSendVersion();
	
	SpectateModes[0] = 'Spectating';
	//SpectateModes[1] = 'WalkSpectating';
	
	//NnPlayerReplicationInfo(PlayerReplicationInfo).InvalidRunes = spawn(class'LinkedList');
}


// ************************************************************
// Authoring Functions (not for general play)
// ************************************************************

exec function NextStart()
{
	local PlayerStart First, tmpStart;
	
	if (!bAdmin)
		return;
	
	foreach AllActors(class'PlayerStart', tmpStart)
	{
		if ( First == None )
			First = tmpStart;
		
		if ( LastStart == None )
		{
			MoveToStart( tmpStart );
			return;
		} else if ( LastStart == tmpStart )
			LastStart = None;
	}
	
	MoveToStart( First );
}

function MoveToStart( PlayerStart MoveTo )
{
	log(class$": MoveToStart(): Moving to PlayerStart, name="$MoveTo.name);
	SetLocation( MoveTo.Location );
	SetRotation( MoveTo.Rotation );
	LastStart = MoveTo;
}

exec function ShowPoints()
{
	if (!bAdmin)
		return;
	
	if ( NnCTFGame(Level.Game) != None )
	{
		NnCTFGame(Level.Game).ViewPoints(true);
	}
}

exec function HidePoints()
{
	if (!bAdmin)
		return;
	
	if ( NnCTFGame(Level.Game) != None )
	{
		NnCTFGame(Level.Game).ViewPoints(false);
	}
}

exec function GetLL()
{
	local LinkedList tmpLL;
	
	if (!bAdmin)
		return;
	
	foreach AllActors(class'LinkedList', tmpLL)
	{
		log(class$": GetLL(): ---------------------------------");
		log(class$": GetLL(): name     ="$tmpLL.name);
		log(class$": GetLL(): dataactor="$tmpLL.DataActor);
	}
	log(class$": GetLL(): ---------------------------------");
}

exec function SetMusic( string SetTo )
{
	local music tmpMusic;

	if (!bAdmin)
		return;

	tmpMusic = music( DynamicLoadObject(SetTo, class'music') );
	NewMusic = tmpMusic;
	if ( NewMusicTrans == MTRAN_None )
		NewMusicTrans = MTRAN_FastFade;
	
	ClientSetMusic(NewMusic, 0, 0, NewMusicTrans);
}


exec function SetMusicL()
{
	if (!bAdmin)
		return;
		
	ClientSetMusic(NewMusic, NewMusicSection, NewMusicTrack, NewMusicTrans);
}


exec function GetDiff()
{
	if (!bAdmin)
		return;
	
	ServerGetDiff();
}


function ServerGetDiff()
{
	local LevelDiffInfo LDI;
	
	LDI = Spawn(class'LevelDiffInfo');
	LDI.PrintDiff();
}


exec function PrintValidRunes()
{
	local int i;
	
	if (!bAdmin)
		return;
	
	for (i=0; i<5; i++)
	{
		log(class$": PrintValidRunes(): ValidRunes["$i$"]="$ValidRunes[i]);
	}
}

// ************************************************************
// User Console Commands
// ************************************************************


exec function ITSHUD()
{
	bNoITSHUD = !bNoITSHUD;
}


exec function TeamInfo()
{
	bNoTeamInfo = !bNoTeamInfo;
}


exec function RunesInfo()
{
	bNoRunesInfo = !bNoRunesInfo;
}


exec function DynamicMusic()
{
	bNoDynamicMusic = !bNoDynamicMusic;
	
	// DLL: I need a way to set a client's music to nothing.  I'm currently investigating this.
	if ( bNoDynamicMusic )
		ClientSetMusicStr("", 0, MTRAN_Instant);
	else
		NnCTFGame(Level.Game).ClientResetMusic(self);
}


exec function AboutNnCTF()
{
	ClientMessage("NoName CTF Version "$NnCTFGame(Level.Game).Version);
	log("NoName CTF Version "$NnCTFGame(Level.Game).Version);
}


exec function SetTeam( coerce string S )
{
	ChangeTeamStr(S);
	SaveConfig();
}


exec function SetTeamMenu()
{
	ChangeTeamMenu();
	SaveConfig();
}


exec function ShowMOTD()
{
	UnrealHUD(myHUD).MOTDFadeOutTime = 255;
}


exec function DropRunes( optional bool bRespawn )
{
	local Inventory tmpItem;
	
	do
	{
		tmpItem = FindInventorySubType(class'Runes');
		if ( tmpItem != None )
		{
			//log(class$": DropRunes(): Found, name="$tmpItem.Name);
			Runes(tmpItem).bRespawnOnDestroy = bRespawn;
			tmpItem.Destroyed();
		}
	} until ( tmpItem == None );
}


exec function God()
{
	if (!bAdmin)
		ClientMessage("Command only allowed in admin mode");
	else
		Super.God();
}


exec function Fly()
{
	if (!bAdmin)
		ClientMessage("Command only allowed in admin mode");
	else
		Super.Fly();
}


exec function Ghost()
{
	if (!bAdmin)
		ClientMessage("Command only allowed in admin mode");
	else
		Super.Ghost();
}


exec function ChaseCam()
{
	if ( IsInState('ChaseCamming') )
		return;
	
	ClientMessage("ChaseCam Mode On");
	ChangeToSpectator(false, 'ChaseCamming');
	//GotoState('ChaseCamming');
}


exec function Spectator()
{
	if (IsInState('Spectating') || (SpectateAllowed == 2))
		return;
	
	ChangeToSpectator();
	//GotoState('Spectating');
}


// 220: Rewrite this
exec function Player()
{
	//log("NnCTF: NnCTFPlayer: Player(): Team="$PlayerReplicationInfo.Team);
	
	//if ( Team == NnCTFGame(Level.Game).TC_Spectator )
	if ( PlayerReplicationInfo.TeamName ~= NnCTFGame(Level.Game).GetTeamNameForNum( NnCTFGame(Level.Game).TC_Spectator ) )
	{
		ClientMessage("You must pick a team first!");
		return;
	}
	
	if ( IsInState('Spectating') )			// Is the player not already playing?
		ChangeToPlayer();
}



// ************************************************************
// Input+Action Commands
// ************************************************************


exec function Fire( optional float F )
{
	local inventory rune;
	local int i;
	
//	log ("NnCTF: Fire() bShowMenu == " $ bShowMenu $
//			" Level.Pauser == " $ Level.Pauser $ " Weapon == " $ Weapon);
	
	if ( bSpectating )
		SpectatingFire(F);
	else {
		Super.Fire(F);
	
		for (i=0 ; i<10 && ValidRunes[i] != None ; i++)
		{
			rune = FindInventoryType(ValidRunes[i]);
			if ( rune != None )
				break;
		}
		if ( rune != None )
		{
			//log(class$": Fire(): Rune Found, class="$rune.class);
			rune.Fire(F);
		}
	}
}


exec function AltFire( optional float F )
{
	local inventory rune;
	local int i;
	
	if ( bSpectating )
		SpectatingAltFire(F);
	else {
		Super.AltFire(F);
		
		for (i=0 ; i<10 && ValidRunes[i] != None ; i++)
		{
			rune = FindInventoryType(ValidRunes[i]);
			if ( rune != None )
				break;
		}
		if ( rune != None )
		{
			//log(class$": AltFire(): Rune Found, class="$rune.class);
			rune.AltFire(F);
		}
	}
}



// ************************************************************
// Spectator-Related Functions
// ************************************************************

// Called when a spectator hits the Fire button
function SpectatingFire( float F )
{
	log(class$": SpectatingFire("$F$")");
	if ( SpectateModes[SpectatingOn] == '' )
	{
		SpectatingOn = 0;
		if ( SpectateModes[0] == '' )
			return;
	}
	
	GotoState( SpectateModes[SpectatingOn] );
	SpectatingOn++;
}


function SpectatingAltFire( float F );


function WatchNextPlayer()
{
	local Pawn aPawn;
	
	aPawn = Level.PawnList;
	
	while ( aPawn != None )
	{
		if ( aPawn.bIsPlayer && !(aPawn.IsInState('Spectating')) && aPawn != Watching )
		{
			Watching = PlayerPawn(aPawn);
			return;
		}
		aPawn = aPawn.nextPawn;
	}
}


function ServerCheckSpectate()
{
	//log("NnCTF: NnCTFPlayer: ServerCheckSpectate(): bAutoAssignTeams: "$NnCTFGame(Level.Game).bAutoAssignTeams );
	//log("NnCTF: NnCTFPlayer: ServerCheckSpectate(): bAllowSpectating: "$NnCTFGame(Level.Game).bAllowSpectating );
	
	if ( NnCTFGame(Level.Game).bAutoAssignTeams || !NnCTFGame(Level.Game).bAllowSpectating ) {
		//log("NnCTF: NnCTFPlayer: ServerCheckSpectate(): Setting to 2");
		SpectateAllowed = 2;		// Player Mode
		return;
	}
	
	//log("NnCTF: NnCTFPlayer: ServerCheckSpectate(): Setting to 1");
	SpectateAllowed = 1;			// Spectator Mode
}


function ChangeToSpectator(optional bool bDontChangeTeam,
                           optional name NewState)
{	
	Health = 0;									// This ensures the flag gets dropped
	bSpectating = true;
	
	//log("NnCTF: CTFPlayer(): Changing To Spectator");
	//log(class$": ChangeToSpectator(): NewState="$NewState);
	
	if (!bDontChangeTeam)
		SetTeam( NnCTFGame(Level.Game).GetTeamNameForNum( NnCTFGame(Level.Game).TC_Spectator ) );
	
	ServerRespawnPlayer();
	ReducedDamageType = 'All';
	if ( string(NewState) == "None" ||
	     string(NewState) == "" )
		GotoState('Spectating');
	else
		GotoState(NewState);
	Level.Game.DiscardInventory(self);
	Level.Game.AddDefaultInventory(self);
	PlayerReplicationInfo.Score = 0;
	
	ClientMessage("Spectator Mode On");
}


function ChangeToPlayer()
{
	local int i;
			
	Health = default.Health;
	bSpectating = false;
	ServerRespawnPlayer();
	Level.Game.AcceptInventory(self);
	ReducedDamageType = '';
	PlayerReplicationInfo.Score = 0;
	StartWalk();
}


// ************************************************************
// Team Switching Functions
// ************************************************************

function ServerRespawnPlayer()
{
	//ResetInvalidRunes();

	if( Level.Game.RestartPlayer(self) )
	{
		Level.Game.StartPlayer(self);
		//Level.Game.RestartPlayer(self);
		ClientRestart();
	}
	else
		log("NnCTF: NnCTFPlayer: ServerRespawnPlayer(): ERROR!");
}


function ChangeTeam( int N )
{
	local NnGameReplicationInfo NnGRI;
	
	foreach AllActors( class'NnGameReplicationInfo', NnGRI );

	if ( NnGRI == None )
	{
		log("NnCTF: CTFPlayer: ChangeTeam(): ERROR: NnGRI == None, function halting");
		return;
	}
	
	ChangeTeamStr( NnGRI.GetTeamNameForNum(N) );
}


function ChangeTeamStr( coerce string S )
{
	local Flag F;
	local NnGameReplicationInfo tmpNnGRI, NnGRI;

	if ( S ~= PlayerReplicationInfo.TeamName ) {
		return;
	}

	foreach AllActors( class'NnGameReplicationInfo', tmpNnGRI )
	{
		NnGRI = tmpNnGRI;
	}

	if ( NnGRI == None )
	{
		log("NnCTF: CTFPlayer: ChangeTeamStr(): ERROR: NnGRI == None, function halting");
		return;
	}
	
	if ( !NnCTFGame(Level.Game).ValidateTeamName(S) )
	{
		ClientMessage(S$" is not a valid team name.");
		return;
	}
	
	if ( NnCTFGame(Level.Game).ChangeTeamStr(self, S) )
	{
		foreach ChildActors(class'Flag', F)
		{
			F.DropFlag();
		}
		
		//log(class$": PRI.Team="$PlayerReplicationInfo.Team);
		//log(class$": TC_Spectator="$NnGRI.TC_Spectator);
		if ( PlayerReplicationInfo.Team == NnGRI.TC_Spectator )
		{
			ChangeToSpectator(true);
			GotoState('Spectating');
		} else {
			ChangeToPlayer();								// Ensure the player gets respawned
		}
	} else {
		ClientMessage("You cannot change to the "$S$" team.");
	}
}


// JMA: note that this is a replicated function
// Note also that you are not expected to understand why its a replicated
// function.  Just accept it.  It'll make your life a whole lot easier.
// Trust me.
function ChangeTeamMenu()
{
	ShowSpecialMenu( "NnCTF.NnTeamSelectMenu" );
}


// ************************************************************
// Misc Functions
// ************************************************************


function Inventory FindInventorySubType( class DesiredClass, optional Inventory LastItem )
{
	local Inventory Inv;

	//log(class$": FindInventorySubType(): LastItem="$LastItem.Name);

	for( Inv=Inventory; Inv!=None; Inv=Inv.Inventory )
	{
		if ( LastItem != None ) {
			if ( Inv.Name == LastItem.Name ) {
				//log(class$": FindInventorySubType(): Inv==LastItem");
				LastItem = None;
			}
		}
		else if ( ClassIsChildOf(Inv.class, DesiredClass) )
			return Inv;
	}
	return None;
}

/*
function Died(pawn Killer, name damageType, vector HitLocation)
{
	ResetInvalidRunes();
	
	Super.Died(Killer, damageType, HitLocation);
}
*/

function ServerSendVersion()
{
	Version = NnCTFGame(Level.Game).Version;
}


function ClientSetMusicStr( string NewMusicStr, byte NewSection, EMusicTransition NewTransition )
{
	local music NewMusic;
	
	//log("NnCTF: CTFPlayer: ClientSetMusicStr(): NewMusicStr="$NewMusicStr);
	
	if ( NewMusicStr == "" )
		return;
	
	NewMusic = music( DynamicLoadObject(NewMusicStr, class'Music') );

	if ( NewMusic != None )	
		ClientSetMusic( NewMusic, NewSection, 0, NewTransition );
}


function ClientSetMusic( music NewSong, byte NewSection, byte NewCdTrack, EMusicTransition NewTransition )
{
	//log("NnCTF: CTFPlayer: ClientSetMusic(): NewSong="$NewSong);
	if ( NewSong != None )
		if ( !bNoDynamicMusic )
			Super.ClientSetMusic(NewSong, NewSection, NewCdTrack, NewTransition);
	else
		log("NnCTF: CTFPlayer: ClientSetMusic(): ERROR: NewSong==None");
}


// DLL: This code is based off that in Pawn and PlayerPawn.  It contains
//      a few minor adjustments though.
event Possess()
{
	local UnrealMeshMenu M;
	local byte i;
	local string str;
	//local MOTDInfo MI;

	if ( Level.Netmode == NM_Client )
	{
		// replicate client weapon preferences to server
		ServerSetHandedness(Handedness);
		for ( i=0; i<ArrayCount(WeaponPriority); i++ )
			ServerSetWeaponPriority(i, WeaponPriority[i]);
	}
	
	ServerUpdateWeapons();
		
	bIsPlayer = true;
	DodgeClickTime = FMin(0.3, DodgeClickTime);
	EyeHeight = BaseEyeHeight;
	NetPriority = 8;

	StartWalk();
	//log("NnCTF: CTFPlayer: Possess(): SpectateAllowed="$SpectateAllowed);
	if ( SpectateAllowed == 1 )
	{
		Spectator();
		ChangeTeamMenu();
	}
	
	if ( Level.Netmode == NM_Client )
	{
		M = spawn(class'UnrealMeshMenu');
		M.LoadAllMeshes();
		M.Destroy();
	}
		
}

// ************************************************************
// States
// ************************************************************

state Spectating expands CheatFlying
{
ignores SeePlayer, HearNoise, Bump, TakeDamage, ZoneChange,
	FootZoneChange, HeadZoneChange, Suicide;

	function BeginState()
	{
		// Make the player invisible and noncollideable
		bHidden = true;
		Visibility = 0;		// So bots can't see the spectator.
		UnderWaterTimeBkup = UnderWaterTime;
		UnderWaterTime = -1.0;	
		SetCollision(false, false, false);
		bCollideWorld = false;
		bBehindView = false;
		
		// Make sure the player is flying
		SetPhysics(PHYS_None);
		
		// Change the player's score
		//Score = 0;
		
		Super.BeginState();
	}
	
	function EndState()
	{
		bHidden = false;
		Visibility = default.Visibility;
		UnderWaterTime = UnderWaterTimeBkup;
		SetCollision(true, true, true);
		bCollideWorld = true;
	}
	
	exec function ActivateItem()
	{
		if ( SelectedItem != None )
			if ( SelectedItem.IsA('NnTargetingComputer') )
				Super.ActivateItem();
	}
}	


state ChaseCamming expands Spectating
{
	function BeginState()
	{
		Super.BeginState();
		WatchNextPlayer();
		SetTimer(2.0, false);
	}
	
	function Timer()
	{
		if ( Watching == None ) {
			SetTimer(2.0, false);
			return;
		}
		
		SetBase( Watching );
	}
	
	// Fire() provided by state: Spectating
}

defaultproperties
{
     SpectateModes(0)=Spectating
     MouseSensitivity=6.000000
     WeaponPriority(0)=DispersionPistol
     WeaponPriority(1)=AutoMag
     WeaponPriority(2)=Stinger
     WeaponPriority(3)=ASMD
     WeaponPriority(4)=Eightball
     WeaponPriority(5)=FlakCannon
     WeaponPriority(6)=GESBioRifle
     WeaponPriority(7)=Razorjack
     WeaponPriority(8)=Rifle
     WeaponPriority(9)=Minigun
     WeaponPriority(10)=None
     WeaponPriority(11)=None
     PlayerReplicationInfoClass=Class'NnCTF.NnPlayerReplicationInfo'
}
