//=============================================================================
// DPMSExampleGame.
//
// Example game code for using DynamicPlayer player classes.
//=============================================================================
class DPMSExampleGame extends DeathMatchGame;

event playerpawn Login
(
	string Portal,
	string Options,
	out string Error,
	class<playerpawn> SpawnClass
)
{
	local playerpawn NewPlayer;
	local class<PlayerMeshInfo> MeshInfoClass;
	local string InSkin, InFace;
	local int InTeam;

	NewPlayer = Super.Login(Portal, Options, Error, class'DPMSExamplePlayer');

	MeshInfoClass = GetMeshInfoClass(SpawnClass);
	log("MeshInfo:"@MeshInfoClass);
	log("SoundInfo:"@MeshInfoClass.default.DefaultSoundClass);

	InSkin	   = ParseOption ( Options, "Skin"    );
	InFace     = ParseOption ( Options, "Face"    );
	InTeam     = NewPlayer.PlayerReplicationInfo.Team;

	// set up player using spawnclass
	ChangeMeshClass(NewPlayer, SpawnClass);

	// set player skin
	MeshInfoClass.static.SetMultiSkin(NewPlayer, InSkin, InFace, InTeam);

	return NewPlayer;
}

// could use something like this to find a mesh class for a given class
function class<PlayerMeshInfo> GetMeshInfoClass( class<pawn> PlayerClass )
{
	switch (PlayerClass.Name)
	{
		case 'MaleOne':
			return class'MaleOneMeshInfo';
		case 'MaleTwo':
			return class'MaleTwoMeshInfo';
		case 'MaleThree':
			return class'MaleThreeMeshInfo';

		case 'FemaleOne':
			return class'FemaleOneMeshInfo';
		case 'FemaleTwo':
			return class'FemaleTwoMeshInfo';

		case 'SkaarjPlayer':
			return class'SkaarjPlayerMeshInfo';

		case 'NaliPlayer':
			return class'NaliPlayerMeshInfo';

		default:
			Log("GetMeshInfo(): WARNING, unable to find MeshInfo class for: "$PlayerClass);
			return class'MaleThreeMeshInfo';
	}
}

// change player class (doesn't need to be a playerpawn sub-class)
function ChangeMeshClass( playerpawn Other, class<pawn> NewClass )
{
	local class<PlayerMeshInfo> MeshInfoClass;
	local string PackageName, ItemName;

	// get mesh info for player class
	MeshInfoClass = GetMeshInfoClass(NewClass);

	if (MeshInfoClass == none)
	{
		Log("ChangePlayerClass(): Can't find MeshInfo class for: "$NewClass);
		return;
	}

	//Log("ChangePlayerClass(): New MeshInfo: "$MeshInfoClass);

	DynamicPlayer(Other).MeshInfo = MeshInfoClass;
	DynamicPlayer(Other).SoundInfo = MeshInfoClass.default.DefaultSoundClass;
	Other.SetCollisionSize(MeshInfoClass.default.CollisionRadius, MeshInfoClass.default.CollisionHeight);

	//Log("ChangePlayerClass(): Setting player skin...");
	MeshInfoClass.static.SetMultiSkin(Other, MeshInfoClass.default.DefaultSkinName, MeshInfoClass.default.DefaultFaceName, Other.PlayerReplicationInfo.Team);
}

//=============================================================================
// Other functions that could be used to change player info.

// change player mesh info
function ChangeMeshInfo( playerpawn Other, class<PlayerMeshInfo> NewMeshInfo, optional bool bDontSetCollision )
{
	if (NewMeshInfo == none)
		return;

	if (NewMeshInfo != none)
		DynamicPlayer(Other).MeshInfo = NewMeshInfo;

	if (!bDontSetCollision)
		Other.SetCollisionSize(NewMeshInfo.default.CollisionRadius, NewMeshInfo.default.CollisionHeight);
}

// change player sound info
function ChangeSoundInfo( playerpawn Other, class<PlayerSoundInfo> NewSoundInfo )
{
	if (NewSoundInfo == none)
		return;

	if (NewSoundInfo != none)
		DynamicPlayer(Other).SoundInfo = NewSoundInfo;
}

// change mesh and sound info
// "NewMeshInfo.default.DefaultSoundClass" will be used if NewSoundInfo not set
function ChangePlayerInfo( playerpawn Other, class<PlayerMeshInfo> NewMeshInfo, optional class<PlayerSoundInfo> NewSoundInfo )
{
	if (NewMeshInfo == none)
		return;

	DynamicPlayer(Other).MeshInfo = NewMeshInfo;
	Other.SetCollisionSize(NewMeshInfo.default.CollisionRadius, NewMeshInfo.default.CollisionHeight);

	if (NewSoundInfo != none)
		DynamicPlayer(Other).SoundInfo = NewSoundInfo;
	else DynamicPlayer(Other).SoundInfo = NewMeshInfo.default.DefaultSoundClass;
}


defaultproperties
{
}