//=============================================================================
// DynamicPlayer. (release 2 - UT)
// Author: Ob1-Kenobi (ob1@planetunreal.com)
//=============================================================================
class DynamicPlayer extends TournamentPlayer
	abstract;

var() class<PlayerMeshInfo> MeshInfo;
var() class<PlayerSoundInfo> SoundInfo;

var() string DefaultClassPackage; // default package used for SetMeshClass

replication
{
	// Things the server should send to the client.
	// Change these SERVER side. Call a function on the server to change them.
	//   eg. SomeGame(Level.Game).ChangeMeshClass(self, NewClass);
	reliable if( Role==ROLE_Authority )
		MeshInfo, SoundInfo;

	// Functions client can call.
	reliable if( Role<ROLE_Authority )
		ServerChangeMeshClass, SetMeshClass, SetSkin, SetFace;
}

exec function SetMeshClass(coerce string ClassName)
{
	local class<Pawn> NewClass;

	if (GetItemName(ClassName) == ClassName)
		NewClass = class<Pawn>(DynamicLoadObject(DefaultClassPackage$ClassName,class'Class'));
	else
		NewClass = class<Pawn>(DynamicLoadObject(ClassName,class'Class'));

	if (NewClass != none)
	{
		// change the class
		ServerChangeMeshClass(NewClass);
	}
	else ClientMessage("Unknown player class: "$ClassName);
}

exec function SetSkin(coerce string NewSkinName)
{
	local string OldSkin, MeshName, SkinName, FaceName;
	local string SkinDesc, TestName, Temp;
	local bool bFound;

	// get the mesh name
	MeshName = GetItemName(String(Mesh));

	SkinName = "None";
	FaceName = "None";
	TestName = "";
	bFound = false;

	// find the skin
	if (MeshInfo.default.bIsMultiSkinned)
	{
		while (true)
		{
			GetNextSkin(MeshName, SkinName, 1, SkinName, SkinDesc);

			if (TestName == SkinName)
				break;

			if (TestName == "")
				TestName = SkinName;

			if ((SkinDesc ~= NewSkinName) && (SkinDesc != ""))
			{
				SkinName = Left(SkinName, Len(SkinName) - 1);
				bFound = true;
				break;
			}

			if (NewSkinName ~= SkinName)
			{
				SkinName = Left(SkinName, Len(SkinName) - 1);
				bFound = true;
				break;
			}
		}

		// get a valid face for this skin
		if (bFound)
		{
			TestName = "";
			while ( True )
			{
				GetNextSkin(MeshName, FaceName, 1, FaceName, SkinDesc);

				if( FaceName == TestName )
					break;

				if( TestName == "" )
					TestName = FaceName;

				// Multiskin format
				if( SkinDesc != "")
				{
					Temp = GetItemName(FaceName);
					if(Mid(Temp, 5) != "" && Left(Temp, 4) == GetItemName(SkinName))
					{
						// valid face found so set skin
						FaceName = Left(FaceName, Len(FaceName) - Len(Temp)) $ Mid(Temp, 5);
						ServerChangeSkin(SkinName, FaceName, PlayerReplicationInfo.Team);
						return;
					}
				}
			}
		}
	}
	else
	{
		// try to set skin
		MeshInfo.static.GetMultiSkin(self, SkinName, FaceName);
		ServerChangeSkin(NewSkinName, FaceName, PlayerReplicationInfo.Team);
	}
}

exec function SetFace(coerce string NewFaceName)
{
	local string SkinName, MeshName, FaceName, FullFaceName;
	local string TestName, SkinDesc, Temp;

	if (!MeshInfo.default.bIsMultiSkinned)
		return;

	MeshName = GetItemName(String(Mesh));

	MeshInfo.static.GetMultiSkin(self, SkinName, FaceName);

	FullFaceName = "None";
	TestName = "";
	while ( True )
	{
		GetNextSkin(MeshName, FullFaceName, 1, FullFaceName, SkinDesc);
		if( FullFaceName == TestName )
			break;

		if( TestName == "" )
			TestName = FullFaceName;

		// Multiskin format
		if( SkinDesc != "")
		{
			Temp = GetItemName(FullFaceName);
			if(Mid(Temp, 5) != "" && Left(Temp, 4) == GetItemName(SkinName))
			{
				// valid face found so set skin
				FaceName = Left(FullFaceName, Len(FullFaceName) - Len(Temp)) $ Mid(Temp, 5);
				if (GetItemName(FaceName) ~= NewFaceName)
				{
					ServerChangeSkin(SkinName, FaceName, PlayerReplicationInfo.Team);
					return;
				}
			}
		}
	}
}


//=============================================================================
// Mesh, Skin & Sound functions
//=============================================================================
// General

// get the MeshInfo class for the current mesh
function class<PlayerMeshInfo> GetMeshInfoClass()
{
	// could make this a little more dynamic...
	return MeshInfo;
}

// spawn a carcass class at a specific location, rotation and velocity
// (called by MeshInfo)
// currently used by MeshInfo class
function carcass SpawnCarcassClass(class<Carcass> aCarcass, vector aLocation,
					optional rotator aRotation, optional bool bUseRotation)
{
	//Log("SpawnCarcassClass(): spawning Carcass: "$aCarcass$"    Location: "$aLocation$"   Rotation: "$aRotation$"   bUseRotation: "$bUseRotation);
	if (aCarcass == none)
		return none;

	if (bUseRotation)
		return spawn(aCarcass,,, aLocation, aRotation);
	else
		return spawn(aCarcass,,, aLocation);
}

//=============================================================================
// Mesh

// TODO: o Implement PropertyInfo classes for player

// player changed class
// Implement in sub class. (NewPlayerClass doesn't need to be a playerpawn class)
function ServerChangeMeshClass(class<Pawn> NewPlayerClass)
{
	/* Example mesh class changing code:

	if (NewPlayerClass.default.mesh == MeshInfo.default.PlayerMesh)
	{
		// Could check if mesh is already same here
		return;
	}

	// change class
	SomeGame(Level.Game).ChangeMeshClass(Self, NewPlayerClass);

	MeshInfo.static.CheckMesh(self); // force mesh to update

	UpdateVoicePack(); // check to see if voice pack needs to be changed

	PlayWaiting(); // stops the new mesh from looking frozen when changed
	*/
}

// update the voice pack if necessary
function UpdateVoicePack()
{
	if ((MeshInfo != none) && (VoicePackMetaClass != MeshInfo.default.VoicePackMetaClass))
	{
		VoicePackMetaClass = MeshInfo.default.VoicePackMetaClass;
		SetVoice(class<ChallengeVoicePack>(DynamicLoadObject(MeshInfo.default.VoiceType, class'Class')));
	}
}

//=============================================================================
// Overridden functions

simulated function bool AdjustHitLocation(out vector HitLocation, vector TraceDir)
{
	local float adjZ, maxZ;

	TraceDir = Normal(TraceDir);
	HitLocation = HitLocation + 0.4 * CollisionRadius * TraceDir;

	if ( (MeshInfo.static.StaticGetAnimGroup(AnimSequence) == 'Ducking') && (AnimFrame > -0.03) )
	{
		maxZ = Location.Z + 0.25 * CollisionHeight;
		if ( HitLocation.Z > maxZ )
		{
			if ( TraceDir.Z >= 0 )
				return false;
			adjZ = (maxZ - HitLocation.Z)/TraceDir.Z;
			HitLocation.Z = maxZ;
			HitLocation.X = HitLocation.X + TraceDir.X * adjZ;
			HitLocation.Y = HitLocation.Y + TraceDir.Y * adjZ;
			if ( VSize(HitLocation - Location) > CollisionRadius )
				return false;
		}
	}
	return true;
}

// From TournamentPlayer
function PlayHit(float Damage, vector HitLocation, name damageType, vector Momentum)
{
	local float rnd;
	local Bubble1 bub;
	local bool bServerGuessWeapon;
	local class<DamageType> DamageClass;
	local vector BloodOffset, Mo;
	local int iDam;

	if ( (Damage <= 0) && (ReducedDamageType != 'All') )
		return;

	//DamageClass = class(damageType);
	if ( ReducedDamageType != 'All' ) //spawn some blood
	{
		if (damageType == 'Drowned')
		{
			bub = spawn(class 'Bubble1',,, Location
				+ 0.7 * CollisionRadius * vector(ViewRotation) + 0.3 * EyeHeight * vect(0,0,1));
			if (bub != None)
				bub.DrawScale = FRand()*0.06+0.04;
		}
		else if ( (damageType != 'Burned') && (damageType != 'Corroded')
					&& (damageType != 'Fell') )
		{
			BloodOffset = 0.2 * CollisionRadius * Normal(HitLocation - Location);
			BloodOffset.Z = BloodOffset.Z * 0.5;
			if ( (DamageType == 'shot') || (DamageType == 'decapitated') || (DamageType == 'shredded') )
			{
				Mo = Momentum;
				if ( Mo.Z > 0 )
					Mo.Z *= 0.5;
				spawn(class 'UT_BloodHit',self,,hitLocation + BloodOffset, rotator(Mo));
			}
			else
				spawn(class 'UT_BloodBurst',self,,hitLocation + BloodOffset);
		}
	}

	rnd = FClamp(Damage, 20, 60);
	if ( damageType == 'Burned' )
		ClientFlash( -0.009375 * rnd, rnd * vect(16.41, 11.719, 4.6875));
	else if ( damageType == 'Corroded' )
		ClientFlash( -0.01171875 * rnd, rnd * vect(9.375, 14.0625, 4.6875));
	else if ( damageType == 'Drowned' )
		ClientFlash(-0.390, vect(312.5,468.75,468.75));
	else
		ClientFlash( -0.019 * rnd, rnd * vect(26.5, 4.5, 4.5));

	ShakeView(0.15 + 0.005 * Damage, Damage * 30, 0.3 * Damage);
	PlayTakeHitSound(Damage, damageType, 1);
	bServerGuessWeapon = ( ((Weapon != None) && Weapon.bPointing) || (MeshInfo.static.StaticGetAnimGroup(AnimSequence) == 'Dodge') );
	iDam = Clamp(Damage,0,200);
	ClientPlayTakeHit(hitLocation - Location, iDam, bServerGuessWeapon );
	if ( !bServerGuessWeapon
		&& ((Level.NetMode == NM_DedicatedServer) || (Level.NetMode == NM_ListenServer)) )
	{
		Enable('AnimEnd');
		BaseEyeHeight = Default.BaseEyeHeight;
		bAnimTransition = true;
		PlayTakeHit(0.1, hitLocation, Damage);
	}
}

function ClientPlayTakeHit(vector HitLoc, byte Damage, bool bServerGuessWeapon)
{
	local ChallengeHUD CHUD;

	CHUD = ChallengeHUD(myHUD);
	if ( CHUD != None )
		CHUD.SetDamage(HitLoc, damage);

	HitLoc += Location;
	if ( bServerGuessWeapon && ((MeshInfo.static.StaticGetAnimGroup(AnimSequence) == 'Dodge') || ((Weapon != None) && Weapon.bPointing)) )
		return;
	Enable('AnimEnd');
	bAnimTransition = true;
	BaseEyeHeight = Default.BaseEyeHeight;
	PlayTakeHit(0.1, HitLoc, Damage);
}

function Carcass SpawnCarcass()
{
	local carcass carc;

	carc = Spawn(MeshInfo.default.CarcassClass);
	if ( carc == None )
		return None;
	carc.Initfor(self);
	if (Player != None)
		carc.bPlayerCarcass = true;
	if ( !Level.Game.bGameEnded && (Carcass(ViewTarget) == None) )
		ViewTarget = carc; //for Player 3rd person views
	return carc;
}

function SpawnGibbedCarcass()
{
	local carcass carc;

	carc = Spawn(MeshInfo.default.CarcassClass);
	if ( carc != None )
	{
		carc.Initfor(self);
		carc.ChunkUp(-1 * Health);
	}
}

//=============================================================================
// Skin functions

static function GetMultiSkin( Actor SkinActor, out string SkinName, out string FaceName )
{
	local class<PlayerMeshInfo> MeshInfoClass;

	MeshInfoClass = DynamicPlayer(SkinActor).GetMeshInfoClass();

	if (MeshInfoClass == none)
		return;

	MeshInfoClass.static.GetMultiSkin(SkinActor, SkinName, FaceName);
}

static function SetMultiSkin(Actor SkinActor, string SkinName, string FaceName, byte TeamNum)
{
	local class<PlayerMeshInfo> MeshInfoClass;

	MeshInfoClass = DynamicPlayer(SkinActor).GetMeshInfoClass();

	if (MeshInfoClass == none)
		return;

	MeshInfoClass.static.SetMultiSkin(SkinActor, SkinName, FaceName, TeamNum);
}

function ServerChangeSkin( coerce string SkinName, coerce string FaceName, byte TeamNum )
{
	local string MeshName;

	MeshName = GetItemName(string(Mesh));
	if ( Level.Game.bCanChangeSkin )
		MeshInfo.static.SetMultiSkin(self, SkinName, FaceName, TeamNum);
}

//=============================================================================
// Sound Playing Functions (all overridden to use SoundInfo class for sounds)

// spawn a zone entry/exit actor
// (called by SoundInfo class)
function actor SpawnZoneActor(class<Actor> ActorClass, vector aLocation, optional rotator aRotation, optional bool bUseRotation)
{
	if (ActorClass == none)
		return none;

	if (bUseRotation)
		return spawn(ActorClass,,, aLocation, aRotation);
	else
		return spawn(ActorClass,,, aLocation);
}

// From Engine.Pawn
event FootZoneChange(ZoneInfo newFootZone)
{
	SoundInfo.static.FootZoneChange(self, newFootZone);
}

state FeigningDeath
{
	function Landed(vector HitNormal)
	{
		SoundInfo.static.PlayerLanded(self, HitNormal);
	}
}

// From BotPack.TournamentPlayer
function PlayDyingSound()
{
	SoundInfo.static.PlayDyingSound(self);
}

//Player Jumped
function DoJump( optional float F )
{
	SoundInfo.static.DoJump(self, F);
}

simulated function FootStepping()
{
	SoundInfo.static.FootStepping(self);
}

function PlayTakeHitSound(int damage, name damageType, int Mult)
{
	SoundInfo.static.PlayTakeHitSound(self, damage, damageType, Mult);
}

function Gasp()
{
	SoundInfo.static.Gasp(self);
}

// here to support UnrealI.MaleOne
function PlayMetalStep()
{
	SoundInfo.static.PlaySpecial(self, 'MetalStep');
}

// used by SkaarjTrooper mesh
function WalkStep()
{
	SoundInfo.static.PlaySpecial(self, 'WalkStep');
}

function RunStep()
{
	SoundInfo.static.PlaySpecial(self, 'RunStep');
}

//=============================================================================
// Animation Playing Functions

// TODO: o Implement MeshInfo.default.bCanHoldWeapon

function PlayTurning()
{
	MeshInfo.static.PlayTurning(self);
}

function TweenToWalking(float tweentime)
{
	MeshInfo.static.TweenToWalking(self, tweentime);
}

function PlayDodge(eDodgeDir DodgeMove)
{
	MeshInfo.static.PlayDodge(self, DodgeMove);
}

function PlayWalking()
{
	MeshInfo.static.PlayWalking(self);
}

function TweenToRunning(float tweentime)
{
	MeshInfo.static.TweenToRunning(self, tweentime);
}

function PlayRunning()
{
	MeshInfo.static.PlayRunning(self);
}

function PlayRising()
{
	MeshInfo.static.PlayRising(self);
}

function PlayFeignDeath()
{
	MeshInfo.static.PlayFeignDeath(self);
}

function PlayLanded(float impactVel)
{
	MeshInfo.static.PlayLanded(self, impactVel);
}

function PlayInAir()
{
	MeshInfo.static.PlayInAir(self);
}

function PlayDuck()
{
	MeshInfo.static.PlayDuck(self);
}

function PlayCrawling()
{
	MeshInfo.static.PlayCrawling(self);
}

function TweenToWaiting(float tweentime)
{
	MeshInfo.static.TweenToWaiting(self, tweentime);
}

function PlayRecoil(float Rate)
{
	MeshInfo.static.PlayRecoil(self, Rate);
}

function PlayFiring()
{
	MeshInfo.static.PlayFiring(self);
}

function PlayWeaponSwitch(Weapon NewWeapon)
{
	MeshInfo.static.PlayWeaponSwitch(self, NewWeapon);
}

function PlaySwimming()
{
	MeshInfo.static.PlaySwimming(self);
}

function TweenToSwimming(float tweentime)
{
	MeshInfo.static.TweenToSwimming(self, tweentime);
}

// from BotPack.TournamentMale
function PlayDying(name DamageType, vector HitLoc)
{
	MeshInfo.static.PlayDying(self, DamageType, HitLoc);
}

function PlayDecap()
{
	MeshInfo.static.PlayDecap(self);
}

function PlayGutHit(float tweentime)
{
	MeshInfo.static.PlayGutHit(self, tweentime);
}

function PlayHeadHit(float tweentime)
{
	MeshInfo.static.PlayHeadHit(self, tweentime);
}

function PlayLeftHit(float tweentime)
{
	MeshInfo.static.PlayLeftHit(self, tweentime);
}

function PlayRightHit(float tweentime)
{
	MeshInfo.static.PlayRightHit(self, tweentime);
}

// "GetAnimGroup()" functions from "Engine.Pawn"
exec function Taunt( name Sequence )
{
	MeshInfo.static.Taunt(self, Sequence);
}

function SwimAnimUpdate(bool bNotForward)
{
	MeshInfo.static.SwimAnimUpdate(self, bNotForward);
}

state PlayerSwimming
{
	function AnimEnd()
	{
		MeshInfo.static.SwimAnimEnd(self);
	}
}

// From Engine.PlayerPawn
state PlayerWalking
{
	function Dodge(eDodgeDir DodgeMove)
	{
		MeshInfo.static.Dodge(self, DodgeMove);
	}

	function AnimEnd()
	{
		MeshInfo.static.WalkingAnimEnd(self);
	}

	function ProcessMove(float DeltaTime, vector NewAccel, eDodgeDir DodgeMove, rotator DeltaRot)
	{
		MeshInfo.static.WalkingProcessMove(self, DeltaTime, NewAccel, DodgeMove, DeltaRot);
	}

	function PlayerMove( float DeltaTime )
	{
		MeshInfo.static.WalkingPlayerMove(self, DeltaTime);
	}
}

function PlayWaiting()
{
	MeshInfo.static.PlayWaiting(self);
}

function PlayChatting()
{
	MeshInfo.static.PlayChatting(self);
}

defaultproperties
{
	MeshInfo=None
	SoundInfo=None
	CollisionRadius=0.000000
	CollisionHeight=0.000000
}