//=============================================================================
// OSPWeapons.OSPDoubleMag
//		- Copyright (c), 1998 Brandon Reinhart
//
// Version: 1.0
// Author: Brandon "GreenMarine" Reinhart
// Contact: greenmarine@ittc.ukans.edu / http://www.planetquake.com/osp/
//   The latest version of this package is available at http://www.planetquake.com/osp/
// 
// Usage:
//     If you just want to play around with this object then type:
//			summon OSPWeapons.OSPDoubleMag
//     at the console.  That will create the OSPDoubleMag object.
//     If you want to use the OSPDoubleMag in a level, the add it to the level
//     like usual and deploy the OSPWeapons package with your level.
//     If you include OSPDoubleMag in a level, PLEASE READ THE KNOWN ISSUES
//     and LEGAL sections.
//
// Known Issues:
//   - This wasn't meant to be used in a level with a UnrealI.AutoMag object.
//	   If you pick up an AutoMag while you have none, one, or two OSPDoubleMags,
//     the weapons will get very weirded out.  I could fix it so that it only happens
//     when you have no OSPDoubleMags, but that would be unecessary work.
//     The point is, if you make a level that uses OSPDoubleMag, ONLY PUT OSPDoubleMag
//     OBJECTS IN IT!!!
//	 - Sometimes the second Mag flickers when you move around.  I'm not sure what
//     causes that yet, but I'll figure it out eventually.  It wasn't enough of a bug
//     to prevent this initial release of the OSPWeapons package because it doesn't
//     happen very often.
//
// Legal:
//	   The OSPWeapons package may only be transmitted by electronic means.
//     You may deploy the OSPWeapons package with your user created level
//     without notifying the author, but you MUST GIVE CREDIT TO THE AUTHOR.
//     You MAY NOT modify the OSPWeapons package, in whole or in part, without
//     express written permission from the author.  You MAY NOT include the OSPWeapons
//     package on a CD-ROM, in whole or in part, without express written permission
//     of the author.  If you deploy the OSPWeapons package, it must be the complete
//     package, as described in the package.manifest file.
//	   You MAY use the code within the OSPWeapons package as a basis for your
//     own classes as long as you GIVE CREDIT TO THE AUTHOR.
//     This package and every class in this package is Copyright (c) 1998, Brandon Reinhart
//
// About the Author:
//	   Brandon Reinhart, known as GreenMarine in the Quake/Quake2/Unreal scene is a 
//     dedicated mod hacker obsessed with game and graphics programming. Brandon has 
//	   authored the King of the Hill modification for Quake2 and currently has multiple 
//	   Unreal projects planned, in addition to a technology demonstration. Brandon is currently 
//	   looking for employment in the games industry. 
//
//=============================================================================
// Hopefully, this will help you get started in the world of UnrealScript.  I've tried
// to make my comments clear at each point, so you understand what everything is doing.
//
// I had two large challenges during the authoring of this class.  The first
// was discovered that bHidden makes items invisible.  Yes, I know its obvious,
// but for the longest time I was setting bHidden to true just before an operation
// that set it to false.  I couldn't figure out why my second gun wasn't showing up!
// The lesson learned is to trace the path of execution in your code very closely.
//
// The second big challenge was getting the Location and Rotation of the second
// gun to update correctly.  The key lay in finding the Tick() function.  This
// function is integral to operations that you want to transcend the normal time/state
// based operation of an object.  Originally, I was calling a modified version of
// InvCalcView that updated the position of the Slave gun whenever the first
// was updated.  (InvCalcView, a method of Inventory, is called for the PlayerPawn's
// main weapon every frame.)  This resulted in the slave gun dragging behind the
// player when he moved.  In Quake2 you see this happen when you try to update
// the position of an object at the beginning or middle of a frame.  Obviously,
// my slave was being moved at the wrong time during the frame pipeline!
//
// I recalled that Tim Sweeney's guide mentioned each object is updated during
// a game Tick.  I read these lines over and over and realized that if a Tick
// was like a server frame in Quake2, then there must be some master method
// that is called on every object each tick!  Unfortunately, Tim didn't say
// what function that was...  Only by looking in his examples and confirming in
// code, did I discover that the function had an obvious name: Tick()
//
// Tick() is called on every object, every tick, regardless of state.  Those are
// the two very things that make it a vitally important function.
// Every tick, in that you can piggy back performance critical operations on
// it.  In my case, I put the InvCalcView behavior in Tick() and now my left gun
// was moving correctly.  Important for regardless of state, in that your
// Tick() behavior can transcend the state based cases.
//
// Once these two challenges were met, the rest of the class was pretty easy.
// If you look at the code and my comments closely, you should be able to learn
// a lot about how to add a new weapon and how to deal with UnrealScript code.
// The authoring of this class has given me the inspiration to write a second
// tutorial (following up my Object Oriented Logic tutorial available at
// http://www.planetquake.com/osp/tuts/).  The new tutorial will be about state
// based programming and its applications in UnrealScript.  There is a lot of
// power in state based programming that the unexperienced programmer will not
// immediately realize.  Sweeney's implementation of UnrealScript is extremely
// versitile and powerful, but at the same time exceptionally subtle.
// I hope that this class and my tutorials help you become a better UnrealScript
// hacker.  If you ever have a question, either find me on EF-NET IRC's #UnrealED
// or #UnrealScript (I'm GMarine) or send me an email at the address above.
//=============================================================================

//=============================================================================
// Begin Class
class OSPDoubleMag expands AutoMag;

// These MESH directives are the same as the ones in AutoMag.  It wasn't clear at the
// time of this classes authoring if classes inherit #exec directives.  I assume that
// they do not, so I am repeating the directives here.
// A MESH directive is used for precaching the Mesh and defining animations.
// A TEXTURE directive is used for precaching a Texture and defining the texture's nature.
// A MESHMAP directive maps a texture onto a Mesh.
// A AUDIO directive is used for precaching a sound.
// Each have specific arguments...look each over carefully.

// pickup version
#exec MESH IMPORT MESH=AutoMagPickup ANIVFILE=MODELS\pislow_a.3D DATAFILE=MODELS\pislow_d.3D X=0 Y=0 Z=0
#exec MESH ORIGIN MESH=AutoMagPickup X=0 Y=0 Z=0 YAW=64
#exec MESH SEQUENCE MESH=AutoMagPickup SEQ=All  STARTFRAME=0  NUMFRAMES=1
#exec TEXTURE IMPORT NAME=Automa1 FILE=MODELS\pistol.PCX GROUP="Skins"
#exec MESHMAP SCALE MESHMAP=AutoMagPickup X=0.04 Y=0.04 Z=0.08
#exec MESHMAP SETTEXTURE MESHMAP=AutoMagPickup NUM=1 TEXTURE=Automa1

// 3rd person perspective version 
#exec MESH IMPORT MESH=auto3rd ANIVFILE=MODELS\pis3rd_a.3D DATAFILE=MODELS\pis3rd_d.3D X=0 Y=0 Z=0
#exec MESH ORIGIN MESH=auto3rd X=0 Y=-60 Z=-220 YAW=64 PITCH=0 ROLL=-72
#exec MESH SEQUENCE MESH=auto3rd SEQ=All  STARTFRAME=0  NUMFRAMES=5
#exec MESH SEQUENCE MESH=auto3rd SEQ=Still  STARTFRAME=0  NUMFRAMES=1
#exec MESH SEQUENCE MESH=auto3rd SEQ=Shoot  STARTFRAME=1  NUMFRAMES=4 RATE=30.0
#exec MESH SEQUENCE MESH=auto3rd SEQ=Shot2b  STARTFRAME=1  NUMFRAMES=4 RATE=30.0
#exec MESH SEQUENCE MESH=auto3rd SEQ=Shot2a  STARTFRAME=1  NUMFRAMES=4 RATE=30.0
#exec TEXTURE IMPORT NAME=Automa1 FILE=MODELS\pistol.PCX GROUP="Skins"
#exec OBJ LOAD FILE=textures\FireEffect18.utx PACKAGE=UNREALI.Effect18
#exec MESHMAP SCALE MESHMAP=auto3rd X=0.02 Y=0.02 Z=0.04
#exec MESHMAP SETTEXTURE MESHMAP=auto3rd NUM=1 TEXTURE=Automa1
#exec MESHMAP SETTEXTURE MESHMAP=auto3rd NUM=0 TEXTURE=Unreali.Effect18.FireEffect18

//  player view version
#exec MESH IMPORT MESH=AutoMagL ANIVFILE=MODELS\pistol_a.3D DATAFILE=MODELS\pistol_d.3D X=0 Y=0 Z=0
#exec MESH ORIGIN MESH=AutoMagL X=0 Y=0 Z=0 YAW=64 ROLL=-64
#exec MESH SEQUENCE MESH=AutoMagL SEQ=All     STARTFRAME=0  NUMFRAMES=190
#exec MESH SEQUENCE MESH=AutoMagL SEQ=Still   STARTFRAME=0  NUMFRAMES=1
#exec MESH SEQUENCE MESH=AutoMagL SEQ=Shoot   STARTFRAME=0  NUMFRAMES=4
#exec MESH SEQUENCE MESH=AutoMagL SEQ=Shoot0  STARTFRAME=6  NUMFRAMES=3
#exec MESH SEQUENCE MESH=AutoMagL SEQ=Shoot1  STARTFRAME=7  NUMFRAMES=7
#exec MESH SEQUENCE MESH=AutoMagL SEQ=Shoot2  STARTFRAME=15 NUMFRAMES=5
#exec MESH SEQUENCE MESH=AutoMagL SEQ=Shot2a  STARTFRAME=20  NUMFRAMES=5
#exec MESH SEQUENCE MESH=AutoMagL SEQ=Shot2b  STARTFRAME=29  NUMFRAMES=3
#exec MESH SEQUENCE MESH=AutoMagL SEQ=Shot2c  STARTFRAME=45  NUMFRAMES=5
#exec MESH SEQUENCE MESH=AutoMagL SEQ=Twiddle STARTFRAME=50  NUMFRAMES=25
#exec MESH SEQUENCE MESH=AutoMagL SEQ=Sway1   STARTFRAME=50  NUMFRAMES=2
#exec MESH SEQUENCE MESH=AutoMagL SEQ=Eject   STARTFRAME=75  NUMFRAMES=25
#exec MESH SEQUENCE MESH=AutoMagL SEQ=Down    STARTFRAME=100  NUMFRAMES=5
#exec MESH SEQUENCE MESH=AutoMagL SEQ=Select  STARTFRAME=105  NUMFRAMES=30 RATE=40 GROUP=Select
#exec MESH SEQUENCE MESH=AutoMagL SEQ=T1      STARTFRAME=135  NUMFRAMES=5
#exec MESH SEQUENCE MESH=AutoMagL SEQ=Twirl   STARTFRAME=140  NUMFRAMES=45
#exec MESH SEQUENCE MESH=AutoMagL SEQ=T2      STARTFRAME=185  NUMFRAMES=5
#exec TEXTURE IMPORT NAME=Automa1 FILE=MODELS\pistol.PCX GROUP="Skins" 
#exec OBJ LOAD FILE=textures\FireEffect18.utx PACKAGE=UNREALI.Effect18
#exec MESHMAP SCALE MESHMAP=AutoMagL X=0.007 Y=0.005 Z=0.01
#exec MESHMAP SETTEXTURE MESHMAP=AutoMagL NUM=1 TEXTURE=Automa1
#exec MESHMAP SETTEXTURE MESHMAP=AutoMagL NUM=0 TEXTURE=Unreali.Effect18.FireEffect18

// right handed player view version
#exec MESH IMPORT MESH=AutoMagR ANIVFILE=MODELS\pistol_a.3D DATAFILE=MODELS\pistol_d.3D unmirror=1
#exec MESH ORIGIN MESH=AutoMagR X=0 Y=0 Z=0 YAW=64 ROLL=-64
#exec MESH SEQUENCE MESH=AutoMagR SEQ=All     STARTFRAME=0  NUMFRAMES=190
#exec MESH SEQUENCE MESH=AutoMagR SEQ=Still   STARTFRAME=0  NUMFRAMES=1
#exec MESH SEQUENCE MESH=AutoMagR SEQ=Shoot   STARTFRAME=0  NUMFRAMES=4
#exec MESH SEQUENCE MESH=AutoMagR SEQ=Shoot0  STARTFRAME=6  NUMFRAMES=3
#exec MESH SEQUENCE MESH=AutoMagR SEQ=Shoot1  STARTFRAME=7  NUMFRAMES=7
#exec MESH SEQUENCE MESH=AutoMagR SEQ=Shoot2  STARTFRAME=15 NUMFRAMES=5
#exec MESH SEQUENCE MESH=AutoMagR SEQ=Shot2a  STARTFRAME=20  NUMFRAMES=5
#exec MESH SEQUENCE MESH=AutoMagR SEQ=Shot2b  STARTFRAME=29  NUMFRAMES=3
#exec MESH SEQUENCE MESH=AutoMagR SEQ=Shot2c  STARTFRAME=45  NUMFRAMES=5
#exec MESH SEQUENCE MESH=AutoMagR SEQ=Twiddle STARTFRAME=50  NUMFRAMES=25
#exec MESH SEQUENCE MESH=AutoMagR SEQ=Sway1   STARTFRAME=50  NUMFRAMES=2
#exec MESH SEQUENCE MESH=AutoMagR SEQ=Eject   STARTFRAME=75  NUMFRAMES=25
#exec MESH SEQUENCE MESH=AutoMagR SEQ=Down    STARTFRAME=100  NUMFRAMES=5
#exec MESH SEQUENCE MESH=AutoMagR SEQ=Select  STARTFRAME=105  NUMFRAMES=30 RATE=40 GROUP=Select
#exec MESH SEQUENCE MESH=AutoMagR SEQ=T1      STARTFRAME=135  NUMFRAMES=5
#exec MESH SEQUENCE MESH=AutoMagR SEQ=Twirl   STARTFRAME=140  NUMFRAMES=45
#exec MESH SEQUENCE MESH=AutoMagR SEQ=T2      STARTFRAME=185  NUMFRAMES=5
#exec MESHMAP SCALE MESHMAP=AutoMagR X=0.007 Y=0.005 Z=0.01
#exec MESHMAP SETTEXTURE MESHMAP=AutoMagR NUM=1 TEXTURE=Automa1
#exec MESHMAP SETTEXTURE MESHMAP=AutoMagR NUM=0 TEXTURE=Unreali.Effect18.FireEffect18

#exec AUDIO IMPORT FILE="Sounds\automag\cocking.WAV" NAME="Cocking" GROUP="AutoMag"
#exec AUDIO IMPORT FILE="Sounds\automag\shot.WAV" NAME="shot" GROUP="AutoMag"
#exec AUDIO IMPORT FILE="Sounds\flak\click.WAV" NAME="click" GROUP="flak"
#exec AUDIO IMPORT FILE="Sounds\automag\reload1.WAV" NAME="Reload" GROUP="Automag"

var() class<OSPDoubleMag> OSPDoubleMagClass;
var() OSPDoubleMag SlaveAutoMag;		// The left (second) OSPDoubleMag is a slave to the right.
var() bool MasterAutoMag;				// This switch defines whether the OSPDoubleMag is a master or slave
var() bool TwoMags;						// True if two mags are deployed

// We override the state Pickup so that we can handle special pickup cases.
// The first special pickup case is if the player only has one OSPDoubleMag.
// In that case, we want to spawn a Slave OSPDoubleMag and display it.
// The second case is when the player has both a Master and a Slave already.
// In this case, we want to add the OSPDoubleMag that was picked up and turn it
// into ammo...not a third arm :-)
auto state Pickup
{
	// When touched by an actor.
	function Touch( actor Other )
	{
		local Inventory oldItem;
		local Inventory Copy;
		
		// If touched by a player pawn, let him pick this up.
		if(ValidTouch(Other))
		{
			// FindInventoryType is very useful, it finds the first occurance of a class
			// in the Pawn's inventory and returns a reference to it if successful.
			// If the class isn't in the Pawn's inventory, it returns a none.
			oldItem = Pawn(Other).FindInventoryType(OSPDoubleMagClass);
			// If the item either isn't in the inventory or the player has two Mags,
			// we default to normal weapon pickup behavior.
			if ((oldItem == none) || (OSPDoubleMag(oldItem).TwoMags)) {
				SpawnCopy(Pawn(Other));
				Pawn(Other).ClientMessage(PickupMessage);
				PlaySound (PickupSound);		
				if ( Level.Game.Difficulty > 1 )
					Other.MakeNoise(0.1 * Level.Game.Difficulty);
			// Otherwise, we want to pull up this OSPDoubleMag as the Slave.
			} else {	
				PlaySound (PickupSound);		
				if ( Level.Game.Difficulty > 1 )
					Other.MakeNoise(0.1 * Level.Game.Difficulty);
				Copy = SpawnCopy(Pawn(Other));
				if (OSPDoubleMag(oldItem).SetSlaveAutoMag(Copy))
					OSPDoubleMag(Copy).SetupSlaveAutoMag();
			}			
		}
	}
}


// When HandlePickupQuery returns false, it means that either the object has
// handled the pickup behavior, or that the item just shouldn't be picked up.
// In our case, we want to signal that the object has handled pickup behavior
// when there is only one OSPDoubleMag up.
function bool HandlePickupQuery( inventory Item )
{
	if ((MasterAutoMag) && (SlaveAutoMag == None))
		return false;

	return Super.HandlePickupQuery(Item);
}

// This function sets the model its called on to appear on the right side of the
// screen as if setHand(-1) were called.
// It is interesting to note the following values:
// 				-1 Right Handed
//				 0 Center
//				 1 Left Handed
function bool SetRightHand()
{
	local Rotator newRot;

	// Set the Right handed mesh and correct the offset.	
	Mesh = mesh'AutoMagR';
	PlayerViewOffset.Y = Default.PlayerViewOffset.Y * -1;
	PlayerViewOffset.X = Default.PlayerViewOffset.X;
	PlayerViewOffset.Z = Default.PlayerViewOffset.Z;
	PlayerViewOffset *= 100;
	FireOffset.Y = Default.FireOffset.Y * -1;
	newRot = Rotation;
	newRot.Roll = Default.Rotation.Roll * -1;
	setRotation(newRot);
	SetLocation( Owner.Location + CalcDrawOffset() );
	
	return true;
}

// Same as the above, but instead it moves the model to the left side of the screen.
function bool SetLeftHand()
{
	local Rotator newRot;
	
	Mesh = mesh'AutoMagL';
	PlayerViewOffset.Y = Default.PlayerViewOffset.Y;
	PlayerViewOffset.X = Default.PlayerViewOffset.X;
	PlayerViewOffset.Z = Default.PlayerViewOffset.Z;
	PlayerViewOffset *= 100;
	FireOffset.Y = Default.FireOffset.Y;
	newRot = Rotation;
	newRot.Roll = Default.Rotation.Roll;
	setRotation(newRot);
	SetLocation(Owner.Location + CalcDrawOffset());
	
	return true;
}

// SetSlaveAutoMag prepares the original OSPDoubleMag for John Woo mode.  It assigns the
// Slave, moves the model, and outputs a little message.
function bool SetSlaveAutoMag(Inventory SlaveMag)
{
	// Set the SlaveAutoMag reference.
	SlaveAutoMag = OSPDoubleMag(SlaveMag);

	TwoMags = true;
	AmmoType.AddAmmo(PickupAmmoCount);
	Pawn(Owner).ClientMessage("Time to kick some ass John Woo style!");
	
	SetRightHand();
	PlayTwirl();
			
	return true;
}

// SetupSlaveAutoMag assigns the correct characteristics to the new SlaveAutoMag
function bool SetupSlaveAutoMag()
{
	TwoMags = True;				// We now have two mags...
	SlaveAutoMag = none;		// A Slave doesn't have any slaves...
	MasterAutoMag = false;		// This isn't the Master
//	InventoryGroup = 0;
	
	SetLeftHand();
		
	GoToState('Idle2');
	
	return true;
}

// We override GiveTo because we don't want to go into the Idle2 state until the
// slave is set up.  Idle2 is a sort of semiState, described below.
function GiveTo( pawn Other )
{
	Instigator = Other;
	BecomeItem();
	Other.AddInventory(Self);
}

// We want to override SpawnCopy so that WeaponSet isn't called.  When the SlaveAutoMag
// is created and given to the player's Pawn, the default SpawnCopy tries to make
// it the default Weapon.
function inventory SpawnCopy( pawn Other )
{
	local Weapon newWeapon;

	newWeapon = Weapon(Super.SpawnCopy(Other));
	newWeapon.Instigator = Other;
	newWeapon.GiveAmmo(Other);
	newWeapon.SetSwitchPriority(Other);
	if ((!Other.bNeverSwitchOnPickup) && (!Other.IsA('OSPDoubleMag')))
		newWeapon.WeaponSet(Other);
	newWeapon.AmbientGlow = 0;
	return newWeapon;
}

// Just in case WeaponSet is ever called on the SlaveAutoMag, we override.
// This will just return false if the game tries to set a OSPDoubleMag as the main weapon
// while one is already up.
function bool WeaponSet(Pawn Other)
{
	if (Other.IsA('OSPDoubleMag'))
		return false;
	return Super.WeaponSet(Other);
}

// We override setHand because we don't want to move the AutoMags if there are two up.
function setHand(float Hand)
{
	local rotator newRot;

	if (SlaveAutoMag == none) {
		if (Hand == -1) {
			SetRightHand();
		} else if (Hand == 0) {
			Super.SetHand(Hand);
			Mesh = mesh'AutoMagR';	
		} else if (Hand == 1) {
			SetLeftHand();
		}
	}
}

// This is the GOLDEN FUNCTION.  It is called every frame and it performs a set of operations
// before or after rendering (I don't know which, but Unreal removes the distinction, as
// opposed to Quake2, where moving something at the beginning of the frame isn't a good idea).
// Here, we use Tick to update the position of the left AutoMag, if it exists.
// I'm going to be working on a tutorial on Tick(), States, and state based behavior soon,
// which will clarify things a bit more.
event Tick(float DeltaTime)
{
	if (MasterAutoMag == false) {
		SetLocation( Owner.Location + CalcDrawOffset() );
		SetRotation( Pawn(Owner).ViewRotation );
	}
}

// The BringUp() behavior is a bit complex.  If the player has one OSPDoubleMag and
// he does weapon switching, everything is normal.  If he has two, then we want
// to BringUp() the slave whenver the Master gun is brought up.  That way, when
// you switch from two guns to one gun, or some other gun, the second gun comes
// back if you switch to two guns again.
function BringUp()
{
	if ((MasterAutoMag) && (SlaveAutoMag != none)) {
		SetRightHand();
		SlaveAutoMag.SetLeftHand();
		SlaveAutoMag.BringUp();
	}
	if ( Owner.IsA('PlayerPawn') )
		PlayerPawn(Owner).SetDesiredFOV(PlayerPawn(Owner).Default.FOVAngle);	
	bWeaponUp = false;
	PlaySelect();
	GotoState('Active');
}

// If the first gun is told to Fire, we want the second one to Fire also
function Fire(float Value)
{
	if ((MasterAutoMag) && (SlaveAutoMag != none))
		SlaveAutoMag.Fire(Value);
	Super.Fire(Value);
}

// If the first gun is told to AltFire, we want the second one to AltFire also
function AltFire( float Value )
{
	if ((MasterAutoMag) && (SlaveAutoMag != none))
		SlaveAutoMag.AltFire(Value);
	Super.AltFire(Value);
}

/////////////////////////////////////////////
// We override the Active state to include new PutDown behavior.  If the MasterAutoMag
// is put down, then we want to PutDown() the slave also.
// In addition, we want to twirl the Master whenever it is put into the Active
// state and there is a second gun.
state Active
{
	function Fire(float F) 
	{
		//bFireMem = true;
	}

	function AltFire(float F) 
	{
		//bAltFireMem = true;
	}

	function bool PutDown()
	{
		if ( bWeaponUp || (AnimFrame < 0.75) ) {
			if ((MasterAutoMag) && (SlaveAutoMag != none))
				SlaveAutoMag.GoToState('DownWeapon');
			GotoState('DownWeapon');
		} else
			bChangeWeapon = true;
		return True;
	}

	function BeginState()
	{
		bChangeWeapon = false;
	}

Begin:
// Do a twirl if there is a slave.
	if ((MasterAutoMag) && (SlaveAutoMag != none)) {
		FinishAnim();
		PlayPostSelect();
		FinishAnim();
		PlayTwirl();
	}
	FinishAnim();
	if ( bChangeWeapon )
		GotoState('DownWeapon');
	bWeaponUp = True;
	if (MasterAutoMag) {
		PlayPostSelect();
		FinishAnim();
	}
	Finish();
}

/////////////////////////////////////////////
// We override the DownWeapon state because we want the Slave to go into DownWeapon
// if the master does.
State DownWeapon
{
ignores Fire, AltFire;

	function bool PutDown()
	{
		return true; //just keep putting it down
	}

	function BeginState()
	{
		bChangeWeapon = false;
	}

Begin:
	if ((MasterAutoMag) && (SlaveAutoMag != none))
		SlaveAutoMag.GoToState('DownWeapon');
	TweenDown();
	FinishAnim();
	bOnlyOwnerSee = false;
	Pawn(Owner).ChangedWeapon();
}

///////////////////////////////////////////////////////
// We override normalfire to remove a self check in the original
// code that prevented the second gun from reloading.
state NormalFire
{
Begin:
	if (AnimSequence!='Shoot0') 
	{
		PlayAnim('Shoot',2.5, 0.02);
		FinishAnim();
	}
	PlayAnim('Shoot0',0.26, 0.04);	
	FinishAnim();
	if (ClipCount>15) PlaySound(Misc1Sound, SLOT_None, 3.5*Pawn(Owner).SoundDampening);	
	if ( bChangeWeapon )
		GotoState('DownWeapon');
	else if ( PlayerPawn(Owner) == None )
		Super.Finish();
	// This else if is changed so that the left gun reloads.  (Originally, the code
	// had any weapon that got to this point that wasn't the player's main weapon
	// go into the idle state.  I don't see why they did that...
	else if (AmmoType.AmmoAmount<=0)
		GotoState('Idle');
	else if (ClipCount>=20) GoToState('NewClip');
	else if ( /*bFireMem ||*/ Pawn(Owner).bFire!=0 ) Global.Fire(0);
	else if ( /*bAltFireMem ||*/ Pawn(Owner).bAltFire!=0 )Global.AltFire(0);		
	PlayAnim('Shoot2',0.8, 0.0);	
	FinishAnim();
	GoToState('Idle');
}

////////////////////////////////////////////////////////
// Same as above.
state AltFiring
{
ignores Fire, AltFire;

Begin:
	FinishAnim();
	PlayAnim('Shot2a', 1.2, 0.05);	
	FinishAnim();
Repeater:	
	if (AmmoType.UseAmmo(1)) 
	{
		if ( PlayerPawn(Owner) != None )
			PlayerPawn(Owner).ShakeView(ShakeTime, ShakeMag, ShakeVert);
		ClipCount++;	
		TraceFire(AltAccuracy);
		PlayAltFiring();	
		FinishAnim();
	}
	if ( AltAccuracy < 3 ) 
		AltAccuracy += 0.5;
	if (ClipCount>15) PlaySound(Misc1Sound, SLOT_None, 3.5*Pawn(Owner).SoundDampening);		
	if ( bChangeWeapon )
		GotoState('DownWeapon');
	else if ( /*bAltFireMem ||*/ (Pawn(Owner).bAltFire!=0) 
		&& AmmoType.AmmoAmount>0 && ClipCount<20)
	{
		if ( PlayerPawn(Owner) == None )
			Pawn(Owner).bAltFire = int( FRand() < AltReFireRate );
		//bFireMem = false;
		//bAltFireMem = false; 
		Goto('Repeater');	
	}
	PlayAnim('Shot2c', 0.7, 0.05);	
	FinishAnim();	
	PlayAnim('T2', 0.9, 0.05);	
	FinishAnim();
	Finish();
}

/////////////////////////////////////////////
// If the slave goes into Idle2, bring it up.  This is only used
// when the gun is first added to the Player's inventory.
// It ensures that the gun comes up initially.
State Idle2
{
	Begin:
	if (!MasterAutoMag) {
		BringUp();
	}		
}

// These two functions play the twirl effect.
// I guess the Unreal development team didn't want to include the Twirl in
// normal Unreal...it doesn't seem to be in the code anywhere.
// I don't know why...its kind of cool.
function TweenTwirl()
{
	TweenAnim('Twirl',0.001);
}

function PlayTwirl()
{
	PlayAnim('Twirl', 0.8, 0.0);
}

// We override PlaySelect to add the bHidden = false; statement.
// This makes sure that the second gun is shown if it is brought up
function PlaySelect()
{
	bHidden = false;
	Super.PlaySelect();
}

// We override the Finish() method because, like the Firing and AltFiring states,
// there was code here that prevented any weapon that wasn't the main weapon
// from reloading.
function Finish()
{
	if ( bChangeWeapon )
		GotoState('DownWeapon');
	else if ( PlayerPawn(Owner) == None )
		Super.Finish();
	// This else if is changed so that the left gun reloads.  (Originally, the code
	// had any weapon that got to this point that wasn't the player's main weapon
	// go into the idle state.  I don't see why they did that...
	else if (AmmoType.AmmoAmount<=0)
		GotoState('Idle');
	else if (ClipCount>=20) GoToState('NewClip');
	else if ( /*bFireMem ||*/ Pawn(Owner).bFire!=0 )
		Global.Fire(0);
	else if ( /*bAltFireMem ||*/ Pawn(Owner).bAltFire!=0 )
		Global.AltFire(0);
	else 
		GotoState('Idle');
}

// End Class
// OSPWeapons.OSPDoubleMag
// Copyright (c) 1998, Brandon Reinhart
//=============================================================================

defaultproperties
{
     OSPDoubleMagClass=Class'OSPWeapons.OSPDoubleMag'
     MasterAutoMag=True
}
