// ______________________________________________________________________
// Waypoint.cs	-	Waypointing functions/events
// ______________________________________________________________________
// The idea for much of the waypoint code came from |HH|Crunchy. I first
// learned how to set waypoints from DkS|c-Faze's Autowaypoint (which is
// based on Crunchy's Waypoint-to-flag carrier stuff).
// The functions in this file are, however, not copied from |HH|Crunchy's
// Flag waypoint or anything.
// Okay, that said, this script contains the following events:

// eventWaypointCancel()	 		 - Waypoint has been cancelled
// eventWaypointClient(%recipient, %client)	 - Waypoint issued to a client.
// eventWaypointLocation(%recipient, %x, %y)	 - Waypoint has been issued to an (x,y) location

// Waypoint::Cancel()
// Cancels the waypoint - not text or sound are displayed

// Waypoint::Client(%recipient, %target, %icon, %msg, %tag) -> triggers eventWaypointClient
// 	%recipient 	- the client who "recieves" the waypoint (must be on your team)
//	%target		- the client to set the waypoint to
//	%icon		- the icon to display on the command map (see defs below)
//	%msg		- the message to display as the command text (ie "Kill Player")
//	%tag		- the sound tag to attach to the message (see defs below)

// Waypoint::Flag(%recipient, %flagteam) -> triggers eventWaypointClient if flag is taken
// 	%recipient 	- the client who "recieves" the waypoint (must be on your team)
//	%flagteam	- the team who owns the flag to set a waypoint to

// Some constants for sounds used:
// "Constant" definitions ------------------------------------------------------

$WpSound::Acknowledged = "~wacknow";	// "Acknowledged!"
$WpSound::Declined = "~wobjxcmp";	// "Unable to complete Objective"
$WpSound::Completed = "~wobjcomp";	// "Objective completed"

$WpSound::AttackWP = "~wattway";	// "Attack enemy at waypoint
$WpSound::EscortFr = "~wescfr";		// "Escort Friendly to waypoint"

$WpIcon::None = 0;
$WpIcon::Attack = 1;
$WpIcon::Misc = 2;

// The code --------------------------------------------------------------------

// for waypoint-to-flag
include("Presto\\teamtrak.cs");
include("Presto\\event");

function Waypoint::Cancel() {	
	remoteEval(2048, CStatus, 0, "Waypoint Cancelled");	

	event::Trigger(eventWaypointCancel);
}

// Overridden from client.cs to allow waypoint events regardless 
// of how they were triggered (ie waypoint hotkeys)
// -------------------------------------------------------------
function commandDeclined()
{
	// Placed here to avoid binding problems with gui.
	remoteEval(2048,CStatus,0,"Unable to complete objective~wobjxcmp");

	event::Trigger(eventWaypointCancel);
}

function CommandCompleted()
{
	// Placed here to avoid binding problems with gui.
	remoteEval(2048,CStatus,0,"Objective Completed~wobjcomp");

	event::Trigger(eventWaypointCancel);
}

// Target last client who sent a message - client.cs
function targetClient()
{
	if (($lastClientMessage) && (client::getteam(getmanagerid()) != -1))
	{
		if(Client::getTeam(getManagerId()) == Client::getTeam($lastClientMessage))
			%cmd = "Escort " @ Client::getName($lastClientMessage) @ ".~wescfr";
		else
			%cmd = "Attack " @ Client::getName($lastClientMessage) @ ".~wattway";

		remoteEval(2048, "IssueTargCommand", 0, %cmd, $lastClientMessage - 2048, getManagerId());

		event::Trigger(eventWaypointClient, getmanagerID(), $lastClientMessage);

	}
}


// -------------------------------------------------------------

// %recipient is the client to receive the order
// %target is the client to set the wp to. %msg is the text to display
// %tag is the sound to play. %icon is the wp icon to display on the
// commander map (0 = none, 1 = attack, 2 = defend)
// Sorry 'bout the name change - I just thought this was more appropriate
// than Waypoint::Set
function Waypoint::Client(%recipient, %target, %icon, %msg, %tag) {	

	if (%recipient == 2048)
		%recipient = getmanagerid();

	// Make sure recipient is on our team, or is even in the game.
	if ((Client::GetTeam(%recipient) != Client::GetTeam(getmanagerID())) || (Client::GetName(%recipient) == "") || (client::getteam(getmanagerid()) == -1))
		return;

	remoteEval(2048, "IssueTargCommand", %icon, %msg @ Client::Getname(%target) @ %tag, %target - 2048, %recipient);

	event::Trigger(eventWaypointClient, %recipient, %target);
}

// Got the idea from |HH|Crunchy, but built the code myself
// (not sure how similar they are... haven't seen his yet :)
function Waypoint::Flag(%recipient, %flagteam) {

	if (%recipient == 2048)
		%recipient = getmanagerid();

	// Make sure recipient is on our team, or is even in the game.
	if ((Client::GetTeam(%recipient) != Client::GetTeam(getmanagerID())) || (Client::GetName(%recipient) == "") || (client::getteam(getmanagerid()) == -1))
		return;

	%location = Team::GetFlagLocation(%flagteam);

	// Check to make sure someone actually has the specified flag.
	if ((%location == $Trak::LocationHome) || (%location == $Trak::LocationField) || (%location == getmanagerID()) || (%location == "") && (client::getteam(getmanagerid()) != -1)) 
		return;

	%client = GetClientByName(%location);

	if (Client::getteam(%client) != client::getteam(getmanagerid()))
		remoteEval(2048, "IssueTargCommand", 1, "Kill Flag Carrier " @ %location @ "~wattway", %client - 2048, %recipient);
	else
		remoteEval(2048, "IssueTargCommand", 2, "Escort Flag Carrier " @ %location @ "~wescfr", %client - 2048, %recipient);

	event::Trigger(eventWaypointClient, %recipient, %client);
}

// For standardized world co-ordinates (0-1023)
function Waypoint::WorldLocation(%recipient, %x, %y, %icon, %msg, %tag)
{
	if (%recipient == 2048)
		%recipient = getmanagerid();

	// Make sure recipient is on our team, or is even in the game.
	if ((Client::GetTeam(%recipient) != Client::GetTeam(getmanagerID())) || (Client::GetName(%recipient) == ""))
		return;

	// |HH|Crunchy's Waypoint-to-location code.
	remoteEval(2048, "IssueCommand", %icon, %msg @ %tag, %x, %y, %recipient);

	event::Trigger(eventWaypointLocation, %recipient, %x, %y);
}

// For map-specific "map" co-ordinates (must have |HH|Crunchy's missioninfo/Objectivetrak to work)
function Waypoint::MapLocation(%recipient, %x, %y, %icon, %msg, %tag)
{
	// if we've loaded objective data, we can interpret the co-ords to world values
	// instead of map values :)	
	if (($Objective::Loaded) && ($ObjData::MultiX != 0) && ($ObjData::MultiY != 0)) {
		%x = (%x * $ObjData::MultiX) + $ObjData::OffsetX;
		%y = (%y * $ObjData::MultiY) + $ObjData::OffsetY;
	} else
		return -1;

	Waypoint::WorldLocation(%recipient, %x, %y, %icon, %msg, %tag);
}