//=======================================================================================
// Action.cs by Mental Trousers
//=======================================================================================
//
// This file is a direct rip of Writers Event.cs script, originally written by Presto. :)
// However it has been rearranged to enable scripts to attach to the action both before
// (pre) and after (post) the action is executed. By action I mean something that is
// triggered by pressing a key. The Options gui is a modified NewOpts gui (it still works
// perfectly with all versions of NewOpts). Why base it on a modified Event.cs
// script? Because it works !
//
// To use:
// The accompanying options gui must be used to trigger these functions. Key bindings
// defined using NewOpts can also use these functions, however the FearGui::CFGButton's
// should call action::trigger (%action, %p0 .....), and the appropriate actions
// defined using action::initPressed and action::initReleased. Other scripts may then
// attach to that action using either:
//
//  action::Attach::pre(%action, %function)
//  action::Attach (%action, %function) - exactly the same as action::Attach::post
//
// To detach a function from an action queue use
//
//  action::detach::pre(%action, %function)
//  action::detach (%action, %function) - exactly the same as action::detach::post
//
// etc etc. However
//  action::trigger (%action, %p0, %p1, %p2, %p3, %p4, %p5, %p6, %p7, %p8, %p9)
// will trigger the pre-action queue, the action itself and the post-action queue in that
// order. To trigger just the post-action queue, action::trigger::post must be called.
//
// Using the "pre" functions requires that action::completed (%action) is called when
// the attached function is finished executing. This is to maintain the order with
// which things happen ie all of the functions attached to action::trigger::pre are
// executed BEFORE the action itself and then action::trigger::post. eg
//
// I have had problems in the past with small functions executing very quickly and when
// function action::completed is called, things go wrong. This is cured by scheduling
// the function call for 0 seconds. Why this works I don't know, but it does and to save
// myself any hassles I always schedule action::completed now.
//
// if 3 functions are attached to the pre-action queue for jumpPressed, when the 1st is
// finished it will call action::completed (%action) to notify the handler that it is
// finished, as will the 2nd and 3rd. When all 3 are finished, then the action itself
// will trigger, then the post-action queue. If one of the attached functions doesn't
// perform the action::completed (%action) call, the action (and post-action queue) will
// not be triggered.
//
// Scripts are able to mute the action itself, however the any functions attached to
// that action wont be muted. Use action::mute (%action, %mute) where %mute is true or
// false.
//
// A number of actions have a limit placed on them to the number of functions that can
// be attached using the action::Attach::pre method. I have done this primarily to
// make people think about what they are doing before they go attaching stuff before
// the actions are triggered. Inefficient code before, for instance the fire action, can
// result in lag. Lag is bad (not the clan by that name though, they're nice guys :)
//
// This is set when the actions are first set up using action::initPressed. The second
// parameter of that function defines the number of allowed attachments in the
// action::trigger::pre queue, with -1 mean no limit. This limit can be overridden
// using action::overRideLimitOnPre (%action). However that function only allows one
// extra function to be attached. To attach more the override function must be called
// again. ie to attach 3 functions to an action that has a limit of zero,
// action::overRideLimitOnPre must be called 3 times. It's a kludge, but hopefully it
// will help limit the number of people saying "your script lags me" when it is someone
// elses scripts :)
//
// There also 5 special actions. I have not been able to trigger these from script, so
// there isn't a pre-action queue for them and you can not mute them at all. However,
// they do have post-action queues. These 5 are:
//   1) Scorelist
//   2) Game chat
//   3) Team chat
//   4) Hud configuration
//   5) Toggle chat window size
//
// Most of the functions here work exactly the same as in event.cs.
// Functions defined here (* means internal function only):
//	action::trigger (%action, %p0, %p1, %p2, %p3, %p4, %p5, %p6, %p7, %p8, %p9)
// *action::continue (%action, %p0, %p1, %p2, %p3, %p4, %p5, %p6, %p7, %p8, %p9)
//  action::completed (%action)
// *action::waitUntilFinished (%action, %p0, %p1, %p2, %p3, %p4, %p5, %p6, %p7, %p8, %p9)
//  action::overRideLimitOnPre (%action)
//  action::initPressed (%action, %preLimit, %str)
//  action::initReleased (%action, %str)
//  action::mute (%action, %mute)
//
//  action::trigger::pre (%action, %p0, %p1, %p2, %p3, %p4, %p5, %p6, %p7, %p8, %p9)
//  action::Attach::pre(%action, %function)
//  action::Detach::pre(%action, %function)
//  action::Exists::pre (%action)
//
//  action::trigger::post (%action, %p0, %p1, %p2, %p3, %p4, %p5, %p6, %p7, %p8, %p9)
//  action::Attach::post(%action, %function)
//  action::Detach::post(%action, %function)
//  action::Exists::post(%action)
//  action::GetCount::post(%action)
//
//  action::Attach(%action, %function)
//  action::Detach(%action, %function)
//  action::Exists(%action)
//  action::GetCount(%action) 
//
//=======================================================================================
// This is the one that is called when a key is pressed, it starts the ball rolling
//=======================================================================================
function action::trigger (%action, %p0, %p1, %p2, %p3, %p4, %p5, %p6, %p7, %p8, %p9)
{
	if ($action::count::pre[%action]>0 && !$action::[muteAll, %action])
		{
		action::trigger::pre (%action, %p0, %p1, %p2, %p3, %p4, %p5, %p6, %p7, %p8, %p9);
		action::waitUntilFinished (%action, %p0, %p1, %p2, %p3, %p4, %p5, %p6, %p7, %p8, %p9);
		}
	else
		action::continue (%action, %p0, %p1, %p2, %p3, %p4, %p5, %p6, %p7, %p8, %p9);
}

//=======================================================================================
// Here is where the actual action is executed
//=======================================================================================
function action::continue (%action, %p0, %p1, %p2, %p3, %p4, %p5, %p6, %p7, %p8, %p9)
{
	if (!$action::[mute, %action])
		eval ($action::[%action]);
	if ($action::count::post[%action]>0 && !$action::[muteAll, %action])
		action::trigger::post (%action, %p0, %p1, %p2, %p3, %p4, %p5, %p6, %p7, %p8, %p9);
}

//=======================================================================================
// Used to reduce $action::executed::pre[%action] by one. When it reaches zero, the
// action will be triggered
//=======================================================================================
function action::completed (%action)
{
	$action::executed::pre[%action]--;
}

//=======================================================================================
// Yes that schedule is correct, 25/1000ths of a second. It has to be that small because
// anything larger produces a noticable amount of latency between pressing the button and
// having the action happen if there are only one or two small scripts attached to the
// pre action queue.
//=======================================================================================
function action::waitUntilFinished (%action, %p0, %p1, %p2, %p3, %p4, %p5, %p6, %p7, %p8, %p9)
{
	if ($action::executed::pre[%action]>0)
		{
		%str= %action@", %p0, %p1, %p2, %p3, %p4, %p5, %p6, %p7, %p8, %p9";
		schedule ("action::waitUntilFinished ("@%str@");", 0.025);
		}
	else
		action::continue (%action, %p0, %p1, %p2, %p3, %p4, %p5, %p6, %p7, %p8, %p9);
}

//=======================================================================================
// Over rides the limit on the specified pre action queue - if there is one. The limit is
// there more to make people think before attaching anything in certain places. Writer
// says that decent code optimisation wouldn't produce any noticable latency, and I whole
// heartedly agree with him on that. However there are alot of people out there who are
// only starting on scripting or aren't experienced programmers and aren't familiar with
// how to write efficient code. The limit is there to make them think about it a little.
//=======================================================================================
function action::overRideLimitOnPre (%action)
{
	if ($action::[preLimit, %action]!=999)
		{
		$action::[preLimit, %action]++;
		return true;
		}
	else
		return false;
}

//=======================================================================================
// Initialise the actions for the key presses
//=======================================================================================
function action::initPressed (%action, %preLimit, %str)
{
	if (%str!="")
		{
		$action::[%action]=%str;
		$action::[mute, %action]=false;
		}
	else
		$action::[mute, %action]=true;

	if (%preLimit!=-1)
		$action::[preLimit, %action]=%preLimit;
	else
		$action::[preLimit, %action]=999;

	$action::[muteAll, %action]=false;

}

//=======================================================================================
// Initialise the actions for the key releases
//=======================================================================================
function action::initReleased (%action, %str)
{
	if (%str!="")
		{
		$action::[%action]=%str;
		$action::[mute, %action]=false;
		}
	else
		$action::[mute, %action]=true;
}

//=======================================================================================
// Mutes the action, but not the queues attached to that action
//=======================================================================================
function action::mute (%action, %mute)
{
	$action::[mute, %action]=%mute;
}

//=======================================================================================
// Mutes the action and the pre and post queues as well.
//=======================================================================================
function action::muteAll (%action, %mute)
{
	action::mute (%action, %mute);
	$action::[muteAll, %action]=%mute;
}

//***************************************************************************************
//***************************************************************************************
function action::trigger::pre (%action, %p0, %p1, %p2, %p3, %p4, %p5, %p6, %p7, %p8, %p9)
{
   if(!$action::count::pre[%action])
    {
        $action::returnCount::pre[%action] = 0;
        return %action;
    }

    // Call every attached function in turn
    for(%i = 1; %i <= $action::count::pre[%action]; %i++)
    {
        %function = $action::function::pre[%action, %i];
        if($action::isStatement::pre[%action, %i])
        {
            // This function is a statement so don't pass any parameters
            // Note that we store the return value in two different arrays so we can retrieve it
            // by index or by function
            $action::return::pre[%action, %i] = $action::return::pre[%action, %function] = eval(%function);
        }
        else
        {
            // This function is not a statement so pass the parameters to it
            // Note that we store the return value in two different arrays so we can retrieve it
            // by index or by function
            $action::return::pre[%action, %i] = $action::return::pre[%action, %function] = eval(%function @ "(%p0,%p1,%p2,%p3,%p4,%p5,%p6,%p7,%p8,%p9);" );
        }
    }

    // Set number of actions that returned...in this case, it's always all of them
    // Also set the number of actions to wait for before executing the action
    $action::executed::pre[%action]=$action::returnCount::pre[%action] = $action::count::pre[%action];
    
    return %action;
}

//=======================================================================================
function action::Attach::pre(%action, %function)
{
    // We will add this function only if it isn't already attached to this action
    if(!$action::index::pre[%action, %function])
    {
        // Increment count of functions attached to this action.
        // Since we'll be adding an element to the end of the $action::function array, we
        // can use the same value for the index into our $action::function array.
        // Storing that index in our $action::index array makes it easier to
        // quickly determine which element to access in the $action::function array 
        // when we only know the action and the function.  (Otherwise we'd have to
        // iterate through the $action::function array looking for a match.)

		if ($action::[preLimit, %action]>=$action::count::pre[%action])
			{
	        %index = $action::index::pre[%action, %function] = $action::count::pre[%action]++;
	
	        // Add this function to the action
	        $action::function::pre[%action, %index] = %function;
	
	        // Set flag if this function is a statement, clear it if it's just a function name
	    	$action::isStatement::pre[%action, %index] = (String::FindSubStr(%function, ";") != -1);
	
	        // If this is a new action, add it to the master action list
	        if(!action::Exists::pre(%action))
	        {
	            $action::list::pre[$action::totalactions::pre++] = %action;
	        }
	
	        return true;
			}
		else
			return false;
    }
    else
    {
        // We may want to test to see if our function was attached...if nothing else
        // this allows a script to know whether it's the first time it tried to attach
        // the function :)
        return false;
    }
}

//=======================================================================================
function action::Detach::pre(%action, %function)
{
    // Return false if function wasn't attached, might be useful
    if(!$action::index::pre[%action, %function])
        return false;

    // Move the last function in the action's function array into
    // the empty array element left behind by the detached function
    // Note that this does NOT preserve the order of the attached functions!
    $action::function::pre[%action, $action::index::pre[%action, %function]] = $action::function::pre[%action, $action::count::pre[%action]];
    $action::isStatement::pre[%action, $action::index::pre[%action, %function]] = $action::isStatement::pre[%action, $action::count::pre[%action]];

    // Clear flag to show this function isn't attached
    $action::index::pre[%action, %function] = 0;
    
    // Decrement the count of functions attached to this action
    $action::count::pre[%action]--;

    // Not really necessary, but just in case something weird happens,
    // don't let the count fall below zero
    if($action::count::pre[%action] < 0)
        $action::count::pre[%action] = 0;
        
    // Return true to say "function was attached, we detached it" :)
    return true;
}

//=======================================================================================
// Tells us if the action has at least one function attached to the pre-action queue
//=======================================================================================
function action::Exists::pre (%action)
{
    for(%i = 1; %i <= $action::totalactions::pre; %i++)
    {
        if(%action == $action::list::pre[%i])
            return true;
    }
    return false;
}


//=======================================================================================
// Tells us how many action are attached to the pre-action queue
//=======================================================================================
function action::GetCount::pre(%action) 
{
    if(%action = "")
        return $action::totalactions::pre;
    else
    	return $action::count::pre[%action];
}

//***************************************************************************************
// Starts all the functions that are triggered after the action it's self is executed
//=======================================================================================
function action::trigger::post (%action, %p0, %p1, %p2, %p3, %p4, %p5, %p6, %p7, %p8, %p9)
{
    if(!$action::count::post[%action])
    {
        $action::returnCount::post[%action] = 0;
        return %action;
    }

    // Call every attached function in turn
    for(%i = 1; %i <= $action::count::post[%action]; %i++)
    {
        %function = $action::function::post[%action, %i];
        if($action::isStatement::post[%action, %i])
        {
            // This function is a statement so don't pass any parameters
            // Note that we store the return value in two different arrays so we can retrieve it
            // by index or by function
            $action::return::post[%action, %i] = $action::return::post[%action, %function] = eval(%function);
        }
        else
        {
            // This function is not a statement so pass the parameters to it
            // Note that we store the return value in two different arrays so we can retrieve it
            // by index or by function
            $action::return::post[%action, %i] = $action::return::post[%action, %function] = eval(%function @ "(%p0,%p1,%p2,%p3,%p4,%p5,%p6,%p7,%p8,%p9);" );
        }
    }

    // Set number of actions that returned...in this case, it's always all of them
    $action::returnCount::post[%action] = $action::count::post[%action];
    
    return %action;
}

//=======================================================================================
function action::Attach::post(%action, %function)
{
    // We will add this function only if it isn't already attached to this action
    if(!$action::index::post[%action, %function])
    {
        // Increment count of functions attached to this action.
        // Since we'll be adding an element to the end of the $action::function array, we
        // can use the same value for the index into our $action::function array.
        // Storing that index in our $action::index array makes it easier to
        // quickly determine which element to access in the $action::function array 
        // when we only know the action and the function.  (Otherwise we'd have to
        // iterate through the $action::function array looking for a match.)
        %index = $action::index::post[%action, %function] = $action::count::post[%action]++;

        // Add this function to the action
        $action::function::post[%action, %index] = %function;

        // Set flag if this function is a statement, clear it if it's just a function name
    	$action::isStatement::post[%action, %index] = (String::FindSubStr(%function, ";") != -1);

        // If this is a new action, add it to the master action list
        if(!action::Exists::post(%action))
        {
            $action::list::post[$action::totalactions::post++] = %action;
        }

        return true;
    }
    else
    {
        // We may want to test to see if our function was attached...if nothing else
        // this allows a script to know whether it's the first time it tried to attach
        // the function :)
        return false;
    }
}

//=======================================================================================
function action::Detach::post(%action, %function)
{
    // Return false if function wasn't attached, might be useful
    if(!$action::index::post[%action, %function])
        return false;

    // Move the last function in the action's function array into
    // the empty array element left behind by the detached function
    // Note that this does NOT preserve the order of the attached functions!
    $action::function::post[%action, $action::index::post[%action, %function]] = $action::function::post[%action, $action::count::post[%action]];
    $action::isStatement::post[%action, $action::index::post[%action, %function]] = $action::isStatement::post[%action, $action::count::post[%action]];

    // Clear flag to show this function isn't attached
    $action::index::post[%action, %function] = 0;
    
    // Decrement the count of functions attached to this action
    $action::count::post[%action]--;

    // Not really necessary, but just in case something weird happens,
    // don't let the count fall below below zero
    if($action::count::post[%action] < 0)
        $action::count::post[%action] = 0;
        
    // Return true to say "function was attached, we detached it" :)
    return true;
}

//=======================================================================================
function action::Exists::post(%action)
{
    for(%i = 1; %i <= $action::totalactions::post; %i++)
    {
        if(%action == $action::list::post[%i])
            return true;
    }
    return false;
}

//=======================================================================================
function action::GetCount::post(%action) 
{
    if(%action = "")
        return $action::totalactions::post;
    else
    	return $action::count::post[%action];
}

//***************************************************************************************
// Exactly the same as the post action stuff, except that you don't have to type as much
// :P
//=======================================================================================
function action::Attach(%action, %function)
{
	action::Attach::post(%action, %function);
}

//=======================================================================================
function action::Detach(%action, %function)
{
	action::Detach::post(%action, %function);
}

//=======================================================================================
function action::Exists(%action)
{
	action::Exists::post(%action);
}

//=======================================================================================
function action::GetCount(%action) 
{
	action::GetCount::post(%action);
}

//=======================================================================================
// Used to unbind keys as there is no way to clear a keybinding without manually deleting
// it from config.cs
//=======================================================================================
function null()
{}

IDCTG_GS_PAGE_5  = "00141141", "";

include ("MT\\Library\\actions.cs");
include ("MT\\Library\\WriterCompat.cs");
