// Targeting Reticle Heads-Up Display
//
// By: Allan "Geist" Campbell
//		Greyshades
//
// December 14, 1998
// 
// Details:
//		An overlay targetting reticle for targets in the 3D world
//     
//     
// Modifications:
//		Dec 1998 LAC Adapted pseudo-code from Monolith (thanks guys!) 
//					 to current Shogo (v2.1) architecture.
//
// Developer list:
//		LAC		L. Allan Campbell (Geist)
//

#include "basedefs_de.h"
#include "cReticle.h"
#include "RiotClientShell.h"

extern CRiotClientShell* g_pRiotClientShell;

//cReticle Object

// Default Constructor for the Reticle
cReticle::cReticle()
{
}

// Initializer for the Reticle
DBOOL cReticle::Init(char *szFilename, HSURFACE hDisplaySurf, DBYTE nNumAnims, Anim_t* pAnimList)
{
	int		nFrames, nMostFrames = 0,
			i, j;
	Anim_t	*pAnim;

	CClientDE*	pClientDE = g_pRiotClientShell->GetClientDE();

	// Set the Primary Display surface
	m_hPrimaryDisplay = hDisplaySurf;

	// Save display screen stats
	pClientDE->GetSurfaceDims(hDisplaySurf, &m_DisplayStats.Width, &m_DisplayStats.Height);

	// Load the overlay pic
	m_hReticleOverlay = pClientDE->CreateSurfaceFromBitmap(szFilename);

	// Get Dims of overlay (for future expansion)
	pClientDE->GetSurfaceDims(m_hReticleOverlay, &m_OverlayStats.Width, &m_OverlayStats.Height);

	// Total number of modes/animations
	m_nNumAnims = nNumAnims;

	// Copy anim info
	for (i=0; i<nNumAnims; i++)
	{
		pAnim = &pAnimList[i];

		m_Anims[i].FrameWidth = pAnim->FrameWidth;
		m_Anims[i].FrameHeight = pAnim->FrameHeight;

		nFrames = m_Anims[i].NumFrames = pAnim->NumFrames;

		for (j=0; j<nFrames; j++)
		{
			m_Anims[i].FrameDuration[j] = pAnim->FrameDuration[j];
		}
	}

	// Init to first frame of first animation
	m_nCurrentAnim = 0;
	m_nCurrentFrame = 0;

	return DTRUE;
}

// Render the Reticle
DRESULT cReticle::Render()
{
	DRect		rDisplayRect;
	DVector		vScreenCoords;
	int			fw, fh, 
				sx, sy, 
				ex, ey, 
				offsetx, offsety,
				nScreenX, nScreenY;
	DFLOAT		fCurrentTime;

	CClientDE*	pClientDE = g_pRiotClientShell->GetClientDE();

	// Get current game time
	fCurrentTime = pClientDE->GetTime();

	// Test to see if time for a frame advance
	if (fCurrentTime > m_fAnimTimer + m_Anims[m_nCurrentAnim].FrameDuration[m_nCurrentFrame])
	{
		m_nCurrentFrame++;
		if (m_nCurrentFrame >= m_Anims[m_nCurrentAnim].NumFrames)
			m_nCurrentFrame = 0;		// Wrap to frame 0

		m_fAnimTimer = fCurrentTime;
	}

	fw = m_Anims[m_nCurrentAnim].FrameWidth;
	fh = m_Anims[m_nCurrentAnim].FrameHeight;

	// Calc starting x offset of current anim
	sx = 0;
	for (int i=0; i<m_nCurrentAnim; i++)
		sx += m_Anims[i].FrameWidth;

	// Calc starting y offset of current frame
	sy = (m_nCurrentFrame * fh);

	// Ending x, y are easy from here
	ex = sx + fw - 1;
	ey = sy + fh - 1;

	// Calc offset to middle of frame (from top/left of frame)
	offsetx = (fw / 2) - 1;
	offsety = (fh / 2) - 1;

	// Plug them in
	rDisplayRect.left = sx;
	rDisplayRect.top = sy;
	rDisplayRect.right = ex;
	rDisplayRect.bottom = ey;

	// Transform 3D world coords to 2D HUD coords.
	if (DTRUE == WorldToScreen(m_vTargetPos, nScreenX, nScreenY, 
							   g_pRiotClientShell->GetCamera()))
	{
		// On screen, so draw it
		pClientDE->DrawSurfaceToSurfaceTransparent(
			m_hPrimaryDisplay,
			m_hReticleOverlay,
			&rDisplayRect,
			nScreenX - offsetx, nScreenY - offsety,
			pClientDE->CreateColor(0, 0, 0, DTRUE)
			);

		return DE_OK;
	}
	else
	{
//		pClientDE->CPrint("ScreenCoords: OUTSIDE");
		return DE_OUTSIDE;
	}
}

// Convert a 3D position into a 2D screen coordinate using game camera
DBOOL cReticle::WorldToScreen(DVector &World, int &nScreenX, int &nScreenY, HLOCALOBJ hCamera)
{
	DMatrix		matCamera, matTemp,
				matRotX, matRotY, matRotZ;
	DVector		vNewPosition,
				vCameraPos,
				vTranslated;
	DRotation	rCameraRotation;
	DFLOAT		sx, cx,
				sy, cy,
				sz, cz,
				fXFOV, fYFOV,
				fMidWidth, fMidHeight;

	// Calc middle of screen
	fMidWidth = (float)(m_DisplayStats.Width / 2);
	fMidHeight = (float)(m_DisplayStats.Height / 2);

	// Get camera position
	g_pRiotClientShell->GetClientDE()->GetObjectPos(hCamera, &vCameraPos);

	// Reverse rotation around X axis
	sx = (float)sin(-g_pRiotClientShell->GetPitch());
	cx = (float)cos(-g_pRiotClientShell->GetPitch());

	// Reverse rotation around Y axis
	sy = (float)sin(-g_pRiotClientShell->GetYaw());
	cy = (float)cos(-g_pRiotClientShell->GetYaw());

	// Reverse rotation around Z axis
	sz = (float)sin(g_pRiotClientShell->GetCamCant() + g_pRiotClientShell->GetRoll());
	cz = (float)cos(g_pRiotClientShell->GetCamCant() + g_pRiotClientShell->GetRoll());

	// Setup Z rotation matrix (remember, column-major)
	MAT_SET(matRotZ,
		  cz,   sz, 0.0f, 0.0f,
		 -sz,   cz, 0.0f, 0.0f,
		0.0f, 0.0f, 1.0f, 0.0f,
		0.0f, 0.0f, 0.0f, 1.0f);

	// Setup X rotation matrix (remember, column-major)
	MAT_SET(matRotX,
		1.0f, 0.0f, 0.0f, 0.0f,
		0.0f,   cx,   sx, 0.0f,
		0.0f,  -sx,   cx, 0.0f,
		0.0f, 0.0f, 0.0f, 1.0f);

	// Setup Y rotation matrix (remember, column-major)
	MAT_SET(matRotY,
		  cy, 0.0f,  -sy, 0.0f,
		0.0f, 1.0f, 0.0f, 0.0f,
		  sy, 0.0f,   cy, 0.0f,
		0.0f, 0.0f, 0.0f, 1.0f);

	// Translate world position so that camera = (0,0,0)
	vTranslated = vCameraPos - World;	// DVector subtraction overloaded
	
	// Z axis is opposite from my calculations
	// FIXME: Adjust calculations to eliminate this step
	vTranslated.z = -vTranslated.z;

	// Build Transformation matrix (matCamera) = matRotZ * matRotX * matRotY
	// ** Order is significant!
	// FIXME: Too many unnecessary operations in matrix math. Optimize
	// to rotate 2 lines.
	MatMul(&matTemp, &matRotZ, &matRotX);
	MatMul(&matCamera, &matTemp, &matRotY);

	// Rotate around camera. After this step, vNewPosition is in a 
	// 3D system where the camera is at (0,0,0) and facing down the
	// Z axis.
	MatVMul(&vNewPosition, &matCamera, &vTranslated);

	// Squash the FOV to 45 degree angles
	g_pRiotClientShell->GetClientDE()->GetCameraFOV(hCamera, &fXFOV, &fYFOV);
	vNewPosition.x /= (float)tan(fXFOV/2);
	vNewPosition.y /= (float)tan(fYFOV/2);

	// Test to see if the point is inside the FOV on the camera
	if (vNewPosition.z > 1.0f &&				// in front of camera
		vNewPosition.x > -vNewPosition.z &&		// test left plane
		vNewPosition.y > -vNewPosition.z &&		// test bottom plane
		vNewPosition.x <  vNewPosition.z &&		// test right plane
		vNewPosition.y <  vNewPosition.z)		// test top plane
	{
		// Project to screen coords.
		nScreenX = int(fMidWidth - (fMidWidth * (vNewPosition.x / vNewPosition.z)));
		nScreenY = int(fMidHeight + (fMidHeight * (vNewPosition.y / vNewPosition.z)));

		return DTRUE;
	}
	else
	{
		return DFALSE;
	}
}

// Set animation mode
DBYTE cReticle::SetMode(DBYTE newmode)
{
	// Cap low values at 0
	if (newmode < 0)
		newmode = 0;

	// Cap high values at m_nNumAnims - 1
	if (newmode >= m_nNumAnims)
		newmode = m_nNumAnims - 1;

	m_nCurrentFrame = 0;

	return (m_nCurrentAnim = newmode);
}

// Cycle animation mode down
DBYTE cReticle::PrevMode()
{
	// m_nCurrentAnim is DBYTE (unsigned), so use a temp to test
	int temp = m_nCurrentAnim - 1;

	// Wrap
	if (temp < 0)
	{
		m_nCurrentAnim = m_nNumAnims - 1;
	}
	else
	{
		m_nCurrentAnim--;
	}

	m_nCurrentFrame = 0;

	return m_nCurrentAnim;
}

// Cycle animation mode up
DBYTE cReticle::NextMode()
{
	m_nCurrentAnim++;

	// Wrap
	if (m_nCurrentAnim >= m_nNumAnims)
		m_nCurrentAnim = 0;

	m_nCurrentFrame = 0;

	return m_nCurrentAnim;
}
