// ----------------------------------------------------------------------- //
//
// MODULE  : WeaponDefs.h
//
// PURPOSE : Definitions for weapon types
//
// CREATED : 9/35/97
//
// ----------------------------------------------------------------------- //

#ifndef __WEAPON_DEFS_H__
#define __WEAPON_DEFS_H__

#include "clientshell_de.h"
#include "RiotCommandIds.h"
#include "PlayerModeTypes.h"
#include "WeaponFXTypes.h"
#include "ProjectileFXTypes.h"
#include "basedefs_de.h"
#include "ModelFuncs.h"

#include <stdio.h>

#define WEAPON_MIN_IDLE_TIME	5.0f
#define WEAPON_MAX_IDLE_TIME	15.0f
#define WEAPON_KEY_FIRE			"FIRE_KEY"
#define WEAPON_KEY_SOUND		"SOUND_KEY"
#define WEAPON_SOUND_RADIUS		2000.0f
#define WEAPON_SOUND_DRYFIRE	1
#define WEAPON_SOUND_KEY		2
#define WEAPON_SOUND_FIRE		3


extern char s_FileBuffer[_MAX_PATH];

// This is defined in both "RiotObjectUtilities.h" and "ClientUtilities.h"
int GetRandom(int lo, int hi);

// new ev - 5/2/99 adding Shochi support, ie. added W_FIRED3 weapon state
enum WeaponState { W_IDLE, W_BEGIN_FIRING, W_FIRING, W_FIRED, W_FIRED2, 
				   W_END_FIRING, W_RELOADING, W_FIRING_NOAMMO, W_SELECT, 
				   W_DESELECT, W_FIRED3 };

enum ProjectileType { VECTOR=0, PROJECTILE, CANNON, MELEE };

// Defines....

enum RiotWeaponId {

	GUN_FIRST_ID = 0,
	GUN_FIRSTMECH_ID = 0,
	// new ev - 02/2/99 changing Pulse Rifle to Ambed Kato Burst
	GUN_AMBEDKATOBURST_ID = 0,
	GUN_LASERCANNON_ID	= 1,
	GUN_SPIDER_ID		= 2,
	// new ev - 12/3/99 putting Bullgut back in
	GUN_BULLGUT_ID	= 3,
	GUN_SNIPERRIFLE_ID	= 4,
	// new ev - 04/12/98 changing Juggernaut to Ambed DSG
	// new ev - 3/2/99 changing DSG to Shochi
	GUN_AMBEDSHOCHI_ID	= 5,
	GUN_SHREDDER_ID		= 6,
	// new ev - 04/12/98 changing Bullgut to Ambed MORT
	// new ev - 12/3/99  added Bullgut back so make the mort id 7
	GUN_AMBEDMORT_ID	= 7,
	GUN_AMBEDMORTFRAG_ID = 8,	// changed from GUN_UNUSED1_ID to allow the mort shell to fragment
	// new ev - 02/12/98 changing RedRiot to AMBED HWAR
	GUN_AMBEDHWAR_ID	= 9,
	GUN_ENERGYBATON_ID	= 10,
	GUN_ENERGYBLADE_ID	= 11,
	GUN_KATANA_ID		= 12,
	// new ev - 02/2/99 changing Mono Knife to Ambed Kato Axe
	GUN_AMBEDKATOAXE_ID = 13,
	GUN_LASTMECH_ID		= 13,

	GUN_FIRSTONFOOT_ID	= 14,
	// new ev - 12/1/99 changing COLT45 to AMBED EMS
	GUN_AMBEDEMS_ID		= 14,
	// new ev - 12/1/99 changing SHOTGUN to AMBED AGUMA
	GUN_AMBEDAGUMA_ID	= 15,
	GUN_ASSAULTRIFLE_ID = 16,
	GUN_ENERGYGRENADE_ID = 17,
	// new ev 28/1/99 changing KatoGrenade to Proximity Mines
	GUN_AMBEDPROXIMITYMINE_ID = 18,
	// new ev - 30/11/98 changing MAC10 to AMBED RASG
	GUN_AMBEDRASG_ID	= 19,
	GUN_TOW_ID			= 20,

	GUN_UNUSED2_ID		= 21,	// Left in to decrease impact on level designers

	// new ev - 20/1/99 changing SqueakyToy to AMBED MINIGUN
	GUN_AMBEDMINIGUN_ID = 22,
	// new ev 28/1/99 changing the Tanto to Drill Melee Weapon 
	GUN_AMBEDDRILL_ID	= 23,
	GUN_LASTONFOOT_ID	= 23,
	GUN_MAX_PLAYER		= 24,
	//GUN_MAX_NUMBER		= 23,

	// NOTE: the following weapons are not for player use

	GUN_MCACLAW_ID		= 24,	// mech claw melee weapon (eg. AssaultDoodleMech)
	GUN_OFCLAW_ID		= 25,	// on foot claw melee weapon (eg. Alien Stalker)
	GUN_BUGCLAW_ID		= 26,	// weaker claw weapon for the Bug (fast attack low damage)
	GUN_MAX_NUMBER		= 27,
	GUN_NONE			= 50
};


inline DBOOL FiredWeapon(WeaponState eState)
{
	if (eState == W_FIRED || eState == W_FIRED2 || eState == W_FIRED3) return DTRUE;

	return DFALSE;
}


inline DBOOL IsOneHandedWeapon(RiotWeaponId nWeaponId)
{
	DBOOL bRet = DFALSE;

	switch (nWeaponId)
	{
		case GUN_ENERGYBATON_ID :
		case GUN_ENERGYBLADE_ID :
		case GUN_KATANA_ID :
		case GUN_AMBEDKATOAXE_ID :
		case GUN_AMBEDDRILL_ID : // really would need 2 hands, but done for AI purposes!
		case GUN_MCACLAW_ID :
		case GUN_OFCLAW_ID :
		case GUN_BUGCLAW_ID	:
			bRet = DTRUE;
		break;

		default : break;
	}

	return bRet;
}


inline ProjectileType GetWeaponType(RiotWeaponId nWeaponId)
{
	ProjectileType eType = VECTOR;

	switch (nWeaponId)
	{
		case GUN_SNIPERRIFLE_ID :
		case GUN_AMBEDAGUMA_ID	:
		case GUN_ASSAULTRIFLE_ID :
		case GUN_AMBEDRASG_ID :
		case GUN_LASERCANNON_ID :
		case GUN_AMBEDMINIGUN_ID :
			eType = VECTOR;
		break;

		// new ev - 12/1/99 making the EMS a projectile
		case GUN_AMBEDEMS_ID :
		case GUN_AMBEDKATOBURST_ID :
		case GUN_BULLGUT_ID :
		case GUN_AMBEDMORT_ID :
		case GUN_SPIDER_ID :
		case GUN_ENERGYGRENADE_ID :
		case GUN_TOW_ID	:
		case GUN_AMBEDPROXIMITYMINE_ID : 
		// new ev - moved Shochi from cannon to projectile
		case GUN_AMBEDSHOCHI_ID : 
			eType = PROJECTILE;
		break;

		case GUN_AMBEDHWAR_ID :
		case GUN_SHREDDER_ID :
			eType = CANNON;
		break;

		case GUN_ENERGYBATON_ID :
		case GUN_ENERGYBLADE_ID :
		case GUN_KATANA_ID :
		case GUN_AMBEDKATOAXE_ID :
		case GUN_AMBEDDRILL_ID :
		case GUN_MCACLAW_ID :
		case GUN_OFCLAW_ID :
		case GUN_BUGCLAW_ID	:
			eType = MELEE;
		break;

		default : break;
	}

	return eType;
}

// Map command to weapon id.
inline int GetWeaponId(int nCommandId, DDWORD dwPlayerMode)
{
	int nWeaponId = -1;

	DBOOL bOnFoot = (dwPlayerMode == PM_MODE_FOOT) ? DTRUE : DFALSE;

	switch (nCommandId)
	{
		case COMMAND_ID_WEAPON_1 :
			if (bOnFoot) nWeaponId = GUN_AMBEDEMS_ID;
			else nWeaponId = GUN_AMBEDKATOBURST_ID;
		break;

		case COMMAND_ID_WEAPON_2 :
			if (bOnFoot)  nWeaponId = GUN_AMBEDAGUMA_ID;
			else nWeaponId = GUN_LASERCANNON_ID;
		break;

		case COMMAND_ID_WEAPON_3 :
			if (bOnFoot) nWeaponId = GUN_AMBEDRASG_ID;
			else nWeaponId = GUN_SPIDER_ID;
		break;

		case COMMAND_ID_WEAPON_4 :
			if (bOnFoot) nWeaponId = GUN_ASSAULTRIFLE_ID;
			else nWeaponId = GUN_AMBEDMORT_ID;
		break;

		case COMMAND_ID_WEAPON_5 :
			if (bOnFoot) nWeaponId = GUN_ENERGYGRENADE_ID;
			else nWeaponId = GUN_SNIPERRIFLE_ID;
		break;

		case COMMAND_ID_WEAPON_6 :
			if (bOnFoot) nWeaponId = GUN_AMBEDPROXIMITYMINE_ID;
			else nWeaponId = GUN_AMBEDSHOCHI_ID;
		break;

		case COMMAND_ID_WEAPON_7 :
			if (bOnFoot) nWeaponId = GUN_TOW_ID;
			else nWeaponId = GUN_SHREDDER_ID;
		break;

		case COMMAND_ID_WEAPON_8 :
			if (bOnFoot) nWeaponId = GUN_AMBEDMINIGUN_ID;
			else nWeaponId = GUN_AMBEDHWAR_ID;
		break;

		case COMMAND_ID_WEAPON_9 :
			if (bOnFoot) nWeaponId = GUN_NONE;
			else nWeaponId = GUN_BULLGUT_ID;
		break;

		case COMMAND_ID_WEAPON_10 :
		{
			switch (dwPlayerMode)
			{
				case PM_MODE_FOOT :
					nWeaponId = GUN_AMBEDDRILL_ID;
				break;

				case PM_MODE_MCA_AP :
					nWeaponId = GUN_ENERGYBATON_ID;
				break;

				case PM_MODE_MCA_UE :
					nWeaponId = GUN_ENERGYBLADE_ID;
				break;

				case PM_MODE_MCA_AO :
					nWeaponId = GUN_AMBEDKATOAXE_ID;
				break;

				case PM_MODE_MCA_SA :
					nWeaponId = GUN_KATANA_ID;
				break;

				case PM_MODE_MCA_DOKU :
					nWeaponId = GUN_AMBEDKATOAXE_ID;
				break;

				case PM_MODE_MCA_GCDM :
					nWeaponId = GUN_AMBEDKATOAXE_ID;
				break;

				default : break;
			}
		}
		break;

		default : 
		break;
	}

	return nWeaponId;
}

// Map weapon id to command id.
inline int GetCommandId(int nWeaponId)
{
	int nCommandId = -1;

	switch (nWeaponId)
	{
		case GUN_AMBEDEMS_ID :
		case GUN_AMBEDKATOBURST_ID :
			nCommandId = COMMAND_ID_WEAPON_1;
			break;

		case GUN_AMBEDAGUMA_ID :
		case GUN_LASERCANNON_ID :
			nCommandId = COMMAND_ID_WEAPON_2;
			break;

		case GUN_AMBEDRASG_ID :
		case GUN_SPIDER_ID :
			nCommandId = COMMAND_ID_WEAPON_3;
			break;

		case GUN_ASSAULTRIFLE_ID :
		case GUN_AMBEDMORT_ID :
			nCommandId = COMMAND_ID_WEAPON_4;
			break;

		case GUN_ENERGYGRENADE_ID :
		case GUN_SNIPERRIFLE_ID :
			nCommandId = COMMAND_ID_WEAPON_5;
			break;

		case GUN_AMBEDPROXIMITYMINE_ID :
		case GUN_AMBEDSHOCHI_ID :
			nCommandId = COMMAND_ID_WEAPON_6;
			break;

		case GUN_TOW_ID :
		case GUN_SHREDDER_ID :
			nCommandId = COMMAND_ID_WEAPON_7;
			break;

		case GUN_AMBEDMINIGUN_ID :
		case GUN_AMBEDHWAR_ID :
			nCommandId = COMMAND_ID_WEAPON_8;
			break;

		case GUN_BULLGUT_ID :
			nCommandId = COMMAND_ID_WEAPON_9;
			break;

		case GUN_ENERGYBATON_ID :
		case GUN_ENERGYBLADE_ID :
		case GUN_AMBEDKATOAXE_ID :
		case GUN_KATANA_ID :
		case GUN_AMBEDDRILL_ID :
			nCommandId = COMMAND_ID_WEAPON_10;
			break;

		default : break;
	}

	return nCommandId;
}

// get the maximum ammo for each weapon
inline DDWORD GetWeaponMaxAmmo(int nWeaponId)
{
	DDWORD nRet = 0;

	switch (nWeaponId)
	{
		case GUN_AMBEDKATOBURST_ID :
			nRet = 250;
		break;

		case GUN_SPIDER_ID :
			nRet = 50;
		break;

		case GUN_BULLGUT_ID :
			nRet = 100;
		break;

		case GUN_AMBEDMORT_ID :
			nRet = 40; // new ev - 18/1/99 was 500!
		break;

		case GUN_SNIPERRIFLE_ID :
			nRet = 400;
		break;

		case GUN_AMBEDSHOCHI_ID :
			nRet = 450;
		break;

		case GUN_SHREDDER_ID :
			nRet = 200;
		break;

		case GUN_AMBEDHWAR_ID :
			nRet = 10;
		break;

		case GUN_ENERGYBATON_ID :
		case GUN_ENERGYBLADE_ID :
		case GUN_AMBEDKATOAXE_ID :
		case GUN_KATANA_ID :
			nRet = 10000;
		break;


		// On-foot mode weapons...

		case GUN_AMBEDEMS_ID :
			nRet = 100; // new ev - 12/1/99 was 200
		break;

		case GUN_AMBEDAGUMA_ID	:
			nRet = 200; // new ev - 12/1/99 was 100
		break;

		case GUN_ASSAULTRIFLE_ID :
			nRet = 400;
		break;

		case GUN_ENERGYGRENADE_ID :
			nRet = 50;
		break;

		case GUN_AMBEDRASG_ID :
			nRet = 500; // new ev 3/12/98 was 400
		break;

		case GUN_TOW_ID	:
			nRet = 25;
		break;

		case GUN_LASERCANNON_ID :
			nRet = 200;
		break;

		case GUN_AMBEDMINIGUN_ID :
			nRet = 600;
		break;

		case GUN_AMBEDDRILL_ID :
			nRet = 10000; // don't run out of ammo!
		break;

		case GUN_AMBEDPROXIMITYMINE_ID :
			nRet = 20;
		break;

		// claw / melee weapons
		case GUN_MCACLAW_ID :
		case GUN_OFCLAW_ID :
		case GUN_BUGCLAW_ID	:
			nRet = 10000;
		break;

		default : break;
	}

	return nRet;
}


inline DDWORD GetSpawnedAmmo(RiotWeaponId nWeaponId)
{
	DDWORD dwAmmo = 1000;

	switch (nWeaponId)
	{
		case GUN_AMBEDKATOBURST_ID :
			dwAmmo = GetRandom(40,80);
		break;

		case GUN_SPIDER_ID :
			dwAmmo = GetRandom(5,10);
		break;

		case GUN_BULLGUT_ID :
			dwAmmo = GetRandom(2,10) * 4;
		break;

		case GUN_AMBEDMORT_ID :
			dwAmmo = GetRandom(2,6);
		break;

		case GUN_SNIPERRIFLE_ID :
			dwAmmo = GetRandom(40,80);
		break;

		case GUN_AMBEDSHOCHI_ID :
			dwAmmo = GetRandom(30,60);
		break;

		case GUN_SHREDDER_ID :
			dwAmmo = GetRandom(10,20);
		break;

		case GUN_AMBEDHWAR_ID :
			dwAmmo = GetRandom(1,3);
		break;


		// On-foot mode weapons...

		case GUN_AMBEDEMS_ID :
			dwAmmo = GetRandom(10,20);
		break;

		case GUN_AMBEDAGUMA_ID	:
			dwAmmo = GetRandom(10,25);
		break;

		case GUN_ASSAULTRIFLE_ID :
			dwAmmo = GetRandom(40,80);
		break;

		case GUN_ENERGYGRENADE_ID :
			dwAmmo = GetRandom(5,10);
		break;

		case GUN_AMBEDPROXIMITYMINE_ID :
			dwAmmo = GetRandom(1, 2);
		break;

		case GUN_AMBEDRASG_ID :
			dwAmmo = GetRandom(25,50);
		break;

		case GUN_AMBEDMINIGUN_ID :
			dwAmmo = GetRandom(50,100);
		break;

		case GUN_TOW_ID	:
			dwAmmo = GetRandom(1,5);
		break;

		case GUN_LASERCANNON_ID :
			dwAmmo = GetRandom(30,60);
		break;

		default : break;
	}

	return dwAmmo;
}


inline DVector GetHandWeaponScale(RiotWeaponId nWeaponId, ModelSize eSize=MS_NORMAL)
{
	DVector vRet;
	VEC_SET(vRet, 1.0f, 1.0f, 1.0f)

	switch (nWeaponId)
	{
		// On-foot mode weapons...

		case GUN_AMBEDDRILL_ID :
			VEC_SET(vRet, 2.5f, 2.5f, 2.5f);
		break;
		
		case GUN_AMBEDPROXIMITYMINE_ID :
			VEC_SET(vRet, 3.3f, 3.3f, 3.3f);
		break;

		case GUN_AMBEDEMS_ID :
			// new ev - 12/1/99 changing COLT45 to AMBED EMS
			//VEC_SET(vRet, 6.0f, 6.0f, 6.0f); 
			VEC_SET(vRet, 1.7f, 1.7f, 1.7f); 
		break;

		case GUN_AMBEDAGUMA_ID	:
			// new ev - 12/1/99 changing COLT45 to AMBED EMS
			//VEC_SET(vRet, 18.0f, 18.0f, 18.0f);
			VEC_SET(vRet, 1.55f, 1.55f, 1.55f);
		break;

		case GUN_AMBEDRASG_ID :
			// new ev - 30/11/98 changing MAC10 to AMBED RASG
			//VEC_SET(vRet, 15.0f, 15.0f, 15.0f);
			VEC_SET(vRet, 5.5f, 5.5f, 5.5f);
		break;

		case GUN_ASSAULTRIFLE_ID :
		case GUN_ENERGYGRENADE_ID :
			//VEC_SET(vRet, 1.0f, 1.0f, 1.0f);
			VEC_SET(vRet, 3.0f, 3.0f, 3.0f);
		break;

		case GUN_TOW_ID	:
			VEC_SET(vRet, 28.0f, 28.0f, 28.0f);
		break;

		case GUN_AMBEDMINIGUN_ID :
			VEC_SET(vRet, 5.0f, 5.0f, 5.0f);
		break;

		case GUN_AMBEDKATOBURST_ID :
			VEC_SET(vRet, 2.5f, 2.5f, 2.5f);
		break;

		case GUN_SHREDDER_ID	:
			VEC_SET(vRet, 25.0f, 25.0f, 25.0f);
		break;

		case GUN_AMBEDSHOCHI_ID :
			//VEC_SET(vRet, 25.0f, 25.0f, 25.0f); // new ev - 04/12/98 changing Juggernaut to Ambed DSG
			VEC_SET(vRet, 1.25f, 1.25f, 1.25f);
		break;

		case GUN_AMBEDMORT_ID	:
			// new ev - 30/11/98 changing Bullgut to AMBED MORT
			//VEC_SET(vRet, 25.0f, 25.0f, 25.0f);
			VEC_SET(vRet, 9.0f, 9.0f, 9.0f);
		break;

		case GUN_BULLGUT_ID	:
			VEC_SET(vRet, 25.0f, 25.0f, 25.0f);
		break;

		case GUN_LASERCANNON_ID:
			VEC_SET(vRet, 25.0f, 25.0f, 25.0f);
		break;

		case GUN_SPIDER_ID :
			VEC_SET(vRet, 25.0f, 25.0f, 25.0f);
		break;

		case GUN_SNIPERRIFLE_ID :
			VEC_SET(vRet, 25.0f, 25.0f, 25.0f);
		break;

		case GUN_AMBEDHWAR_ID :
			//VEC_SET(vRet, 25.0f, 25.0f, 25.0f); // new ev - 02/12/98 changing RedRiot to Ambed HWAR
			VEC_SET(vRet, 2.0, 2.0f, 2.0f);
		break;

		// claw / melee weapons
		case GUN_MCACLAW_ID :
		case GUN_OFCLAW_ID :
		case GUN_BUGCLAW_ID	:
			VEC_SET(vRet, 1.0, 1.0f, 1.0f);
		break;

		default : break;
	}

	DFLOAT fFactor = (eSize == MS_NORMAL ? 1.0f : (eSize == MS_SMALL ? 0.2f : 5.0f));
	VEC_MULSCALAR(vRet, vRet, fFactor);


	return vRet;
}

inline DVector GetHandWeaponFlashOffset(RiotWeaponId nWeaponId, ModelSize eSize=MS_NORMAL)
{
	DVector vRet;
	VEC_INIT(vRet)

	switch (nWeaponId)
	{
		case GUN_AMBEDEMS_ID :
			VEC_SET(vRet, 0.0f, 3.0f, 9.6f);
		break;

		case GUN_AMBEDAGUMA_ID	:
			VEC_SET(vRet, 0.0f, 4.5f, 33.3f);
		break;

		case GUN_ASSAULTRIFLE_ID :
		case GUN_ENERGYGRENADE_ID :
		case GUN_AMBEDPROXIMITYMINE_ID :
			// VEC_SET(vRet, 0.0f, 4.0f, 29.0f);
			// VEC_SET(vRet, 0.0f, 2.0f * scale, 8.75f * scale);
			VEC_SET(vRet, 0.0f, 6.0f, 26.25f);
		break;

		case GUN_AMBEDRASG_ID :
			VEC_SET(vRet, 0.0f, 5.5f, 30.0f);
		break;

		case GUN_TOW_ID	:
			VEC_SET(vRet, 0.0f, 2.8f, 28.0f);
		break;

		case GUN_AMBEDKATOBURST_ID :
			VEC_SET(vRet, 0.0f, 5.0f, 41.25f);
		break;

		case GUN_LASERCANNON_ID :
			VEC_SET(vRet, 0.0f, 3.75f, 43.75f);
		break;

		case GUN_SPIDER_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 43.75f);
		break;

		case GUN_AMBEDMORT_ID	:
			VEC_SET(vRet, 0.0f, 5.0f, 45.0f);
		break;

		case GUN_BULLGUT_ID	:
			VEC_SET(vRet, 0.0f, 5.0f, 45.0f);
		break;

		case GUN_SNIPERRIFLE_ID	:
			VEC_SET(vRet, 0.0f, 2.5f, 40.0f);	
		break;

		case GUN_AMBEDSHOCHI_ID :
			VEC_SET(vRet, 0.0f, 5.0f, 41.25f);
		break;

		case GUN_SHREDDER_ID	:
			VEC_SET(vRet, 0.0f, 5.0f, 41.25f);
		break;

		case GUN_AMBEDHWAR_ID	:
			VEC_SET(vRet, 0.0f, 2.5f, 42.5f);
		break;

		default : break;
	}

	DFLOAT fFactor = (eSize == MS_NORMAL ? 1.0f : (eSize == MS_SMALL ? 0.2f : 5.0f));
	VEC_MULSCALAR(vRet, vRet, fFactor);

	return vRet;
}

inline DBOOL UsesAmmo(RiotWeaponId nWeaponId)
{
	switch (nWeaponId)
	{
		case GUN_ENERGYBATON_ID :
		case GUN_ENERGYBLADE_ID :
		case GUN_KATANA_ID :
		case GUN_AMBEDKATOAXE_ID :
		case GUN_UNUSED2_ID :
		case GUN_AMBEDDRILL_ID :
		// claw / melee weapons
		case GUN_MCACLAW_ID :
		case GUN_OFCLAW_ID :
		case GUN_BUGCLAW_ID	:
		{
			return DFALSE;
		}
		break;	
	}

	return DTRUE;
}

// get the maximum range for each weapon
inline DFLOAT GetWeaponRange(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 10000.0f;

	DFLOAT s_fWeaponRange[GUN_MAX_NUMBER] =
	{
		5000.0f,	// GUN_AMBEDKATOBURST_ID
		6000.0f,	// GUN_LASERCANNON_ID
		10000.0f,	// GUN_SPIDER_ID
		10000.0f,	// GUN_BULLGUT_ID
		10000.0f,	// GUN_SNIPERRIFLE_ID
		8000.0f,	// GUN_AMBEDSHOCHI_ID
		6000.0f,	// GUN_SHREDDER_ID
		10000.0f,	// GUN_AMBEDMORT_ID
		3000.0f,	// NOT USED1 - new ev - 19/12/98 was 0.0f
		10000.0f,	// GUN_AMBEDHWAR_ID
		128.0f,		// GUN_ENERGYBATON_ID
		128.0f,		// GUN_ENERGYBLADE_ID
		128.0f,		// GUN_KATANA_ID
		128.0f,		// GUN_AMBEDKATOAXE_ID
		3000.0f,	// GUN_AMBEDEMS_ID
		2000.0f,	// GUN_AMBEDAGUMA_ID
		6000.0f,	// GUN_ASSAULTRIFLE_ID
		10000.0f,	// GUN_ENERGYGRENADE_ID
		2000.0f,		// GUN_AMBEDPROXIMITYMINE_ID
		3250.0f,	// GUN_AMBEDRASG_ID // new ev - 3/12/98 was 2000.of
		10000.0f,	// GUN_TOW_ID
		0.0f,		// NOT USED2
		4500.0f,	// GUN_AMBEDMINIGUN_ID
		96.0f,		// GUN_AMBEDDRILL_ID reduced to 96 to make it more realistic
		// claw / melee weapons
		144.0f,		// GUN_MCACLAW_ID :
		144.0f,		// GUN_OFCLAW_ID :
		128.0f		// GUN_BUGCLAW_ID	:
	};

	return s_fWeaponRange[nWeaponId];
}


// Return the average damage done by the weapon...

inline DFLOAT GetWeaponDamage(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0.0f;

	DFLOAT s_fWeaponDamage[GUN_MAX_NUMBER] =
	{	
		125.0f,		// GUN_AMBEDKATOBURST_ID
		300.0f,		// GUN_LASERCANNON_ID
		500.0f,		// GUN_SPIDER_ID
		125.0f,		// GUN_BULLGUT_ID
		100.0f,		// GUN_SNIPERRIFLE_ID
		75.0f,		// GUN_AMBEDSHOCHI_ID (made low so that it takes into account high the fire rate)
		400.0f,		// GUN_SHREDDER_ID
		125.0f,		// GUN_AMBEDMORT_ID
		100.0f,		// NOT USED1 - new ev - 19/12/98 was 0.0f
		1000.0f,	// GUN_AMBEDHWAR_ID
		600.0f,		// GUN_ENERGYBATON_ID
		800.0f,		// GUN_ENERGYBLADE_ID
		1000.0f,	// GUN_KATANA_ID
		800.0f,		// GUN_AMBEDKATOAXE_ID
		40.0f,		// GUN_AMBEDEMS_ID // new ev - 12/1/99 was 20.0f
		17.5f,		// GUN_AMBEDAGUMA_ID // new ev - 12/1/99 was 20.0f
		25.0f,		// GUN_ASSAULTRIFLE_ID
		80.0f,		// GUN_ENERGYGRENADE_ID
		120.0f,		// GUN_AMBEDPROXIMITYMINE_ID
		10.0f,		// GUN_AMBEDRASG_ID // new ev - 3/12/98 was 30.0f
		120.0f,		// GUN_TOW_ID
		0.0f,		// NOT USED2
		20.0f,		// GUN_AMBEDMINIGUN_ID
		60.0f,		// GUN_AMBEDDRILL_ID
		// claw / melee weapons
		600.0f,		// GUN_MCACLAW_ID :
		50.0f,		// GUN_OFCLAW_ID :
		10.0f		// GUN_BUGCLAW_ID	:
	};

	return s_fWeaponDamage[nWeaponId];
}


// get the maximum range for each weapon
inline int GetWeaponDamageRadius(int nWeaponId, ModelSize eSize=MS_NORMAL)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0;

	int s_nWeaponDamageRadius[GUN_MAX_NUMBER] =
	{
		100,	// GUN_AMBEDKATOBURST_ID
		0,		// GUN_LASERCANNON_ID
		300,	// GUN_SPIDER_ID
		300,	// GUN_BULLGUT_ID
		0,		// GUN_SNIPERRIFLE_ID
		200,	// GUN_AMBEDSHOCHI_ID
		150,	// GUN_SHREDDER_ID
		300,	// GUN_AMBEDMORT_ID
		200,	// NOT USED1 - new ev - 19/12/98 was 0
		600,	// GUN_AMBEDHWAR_ID
		0,		// GUN_ENERGYBATON_ID
		0,		// GUN_ENERGYBLADE_ID
		0,		// GUN_KATANA_ID
		0,		// GUN_AMBEDKATOAXE_ID
		75,		// GUN_AMBEDEMS_ID // new ev - 12/1/99 was 0
		0,		// GUN_AMBEDAGUMA_ID
		0,		// GUN_ASSAULTRIFLE_ID
		200,	// GUN_ENERGYGRENADE_ID
		250,	// GUN_AMBEDPROXIMITYMINE_ID
		0,		// GUN_AMBEDRASG_ID 
		300,	// GUN_TOW_ID
		0,		// NOT USED2
		0,		// GUN_AMBEDMINIGUN_ID
		0,		// GUN_AMBEDDRILL_ID
		// claw / melee weapons
		0,		// GUN_MCACLAW_ID :
		0,		// GUN_OFCLAW_ID :
		0		// GUN_BUGCLAW_ID	:
	};

	DFLOAT fFactor = (eSize == MS_NORMAL ? 1.0f : (eSize == MS_SMALL ? 0.2f : 5.0f));
	return int((DFLOAT)s_nWeaponDamageRadius[nWeaponId] * fFactor);
}

// get the range the weapon can be heard from
inline int GetWeaponFireSoundRadius(int nWeaponId, ModelSize eSize=MS_NORMAL)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0;

	int s_nWeaponSoundRadius[GUN_MAX_NUMBER] =
	{
		1000,	// GUN_AMBEDKATOBURST_ID
		1000,	// GUN_LASERCANNON_ID
		1000,	// GUN_SPIDER_ID
		1500,	// GUN_BULLGUT_ID
		1000,	// GUN_SNIPERRIFLE_ID
		1250,	// GUN_AMBEDSHOCHI_ID
		1500,	// GUN_SHREDDER_ID
		1250,	// GUN_AMBEDMORT_ID
		0,		// NOT USED1
		2500,	// GUN_AMBEDHWAR_ID
		0,		// GUN_ENERGYBATON_ID
		0,		// GUN_ENERGYBLADE_ID
		0,		// GUN_KATANA_ID
		0,		// GUN_AMBEDKATOAXE_ID
		500,	// GUN_AMBEDEMS_ID // new ev - 12/1/99 was 1000
		1500,	// GUN_AMBEDAGUMA_ID
		1500,	// GUN_ASSAULTRIFLE_ID
		1500,	// GUN_ENERGYGRENADE_ID
		0,		// GUN_AMBEDPROXIMITYMINE_ID
		1000,	// GUN_AMBEDRASG_ID // new ev - 12/1/99 was 500
		2000,	// GUN_TOW_ID
		0,		// NOT USED2
		2000,	// GUN_AMBEDMINIGUN_ID
		0,		// GUN_AMBEDDRILL_ID
		// claw / melee weapons
		0,		// GUN_MCACLAW_ID :
		0,		// GUN_OFCLAW_ID :
		0		// GUN_BUGCLAW_ID	:
	};

	DFLOAT fFactor = (eSize == MS_NORMAL ? 1.0f : (eSize == MS_SMALL ? 0.2f : 5.0f));
	return int((DFLOAT)s_nWeaponSoundRadius[nWeaponId] * fFactor);
}

// get the range the weapon can be heard from
inline int GetWeaponImpactSoundRadius(int nWeaponId, ModelSize eSize=MS_NORMAL)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0;

	int s_nWeaponImpactSoundRadius[GUN_MAX_NUMBER] =
	{
		300,	// GUN_AMBEDKATOBURST_ID
		300,	// GUN_LASERCANNON_ID
		1500,	// GUN_SPIDER_ID
		2000,	// GUN_BULLGUT_ID
		100,	// GUN_SNIPERRIFLE_ID
		1250,	// GUN_AMBEDSHOCHI_ID
		1500,	// GUN_SHREDDER_ID
		2000,	// GUN_AMBEDMORT_ID
		1000,	// NOT USED1 - new ev - 19/12/98 was 0
		5000,	// GUN_AMBEDHWAR_ID
		0,		// GUN_ENERGYBATON_ID
		0,		// GUN_ENERGYBLADE_ID
		0,		// GUN_KATANA_ID
		0,		// GUN_AMBEDKATOAXE_ID
		300,	// GUN_AMBEDEMS_ID
		250,	// GUN_AMBEDAGUMA_ID // new ev - 12/1/99 was 200
		700,	// GUN_ASSAULTRIFLE_ID
		1500,	// GUN_ENERGYGRENADE_ID
		300,	// GUN_AMBEDPROXIMITYMINE_ID
		200,	// GUN_AMBEDRASG_ID
		2000,	// GUN_TOW_ID
		0,		// NOT USED2
		1000,	// GUN_AMBEDMINIGUN_ID
		0,		// GUN_AMBEDDRILL_ID
		// claw / melee weapons
		0,		// GUN_MCACLAW_ID :
		0,		// GUN_OFCLAW_ID :
		0		// GUN_BUGCLAW_ID	:
	};

	DFLOAT fFactor = (eSize == MS_NORMAL ? 1.0f : (eSize == MS_SMALL ? 0.2f : 5.0f));
	return int((DFLOAT)s_nWeaponImpactSoundRadius[nWeaponId] * fFactor);
}

// get the maximum range for each weapon
inline DBYTE GetWeaponFX(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0;

	DBYTE s_nWeaponFX[GUN_MAX_NUMBER] =
	{
		/*GUN_AMBEDKATOBURST_ID*/	WFX_SPARKS | WFX_LIGHT,									
		/*GUN_LASERCANNON_ID*/		WFX_FIRESOUND | WFX_LIGHT | WFX_MARK | WFX_SMOKE,
		/*GUN_SPIDER_ID*/			WFX_MARK | WFX_SPARKS | WFX_SMOKE,
		/*GUN_BULLGUT_ID*/			WFX_MARK | WFX_SPARKS | WFX_SMOKE,
		/*GUN_SNIPERRIFLE_ID*/		WFX_FIRESOUND | WFX_LIGHT | WFX_MARK | WFX_MUZZLE | WFX_TRACER | WFX_SHELL,
		/*GUN_AMBEDSHOCHI_ID*/		WFX_FIRESOUND | WFX_LIGHT | /*WFX_MARK |*/ WFX_SPARKS /*| WFX_SMOKE | WFX_SHELL*/,	
		/*GUN_SHREDDER_ID*/			WFX_FIRESOUND | WFX_LIGHT | WFX_MARK | WFX_SPARKS | WFX_SMOKE | WFX_SHELL,
		/*GUN_AMBEDMORT_ID*/		WFX_MARK | /*WFX_SPARKS | */ WFX_SMOKE,
		/*NOT USED1*/				WFX_MARK | WFX_SPARKS | WFX_SMOKE,//0, new ev - 19/12/98 was 0
		/*GUN_AMBEDHWAR_ID*/		WFX_FIRESOUND | WFX_MARK | WFX_SPARKS | WFX_SMOKE,
		/*GUN_ENERGYBATON_ID*/		WFX_FIRESOUND | WFX_SPARKS,
		/*GUN_ENERGYBLADE_ID*/		WFX_FIRESOUND | WFX_SPARKS,
		/*GUN_KATANA_ID*/			WFX_FIRESOUND | WFX_SPARKS,
		/*GUN_AMBEDKATOAXE_ID*/		WFX_FIRESOUND | WFX_SPARKS,
		/*GUN_AMBEDEMS_ID*/			WFX_FIRESOUND | WFX_LIGHT | WFX_MARK /* | WFX_MUZZLE*/, // new ev - 12/1/99 was WFX_FIRESOUND | WFX_LIGHT | WFX_MARK | WFX_MUZZLE | WFX_TRACER | WFX_SHELL,
		/*GUN_AMBEDAGUMA_ID*/		WFX_FIRESOUND | WFX_LIGHT | WFX_MARK | WFX_MUZZLE | WFX_SHELL,
		/*GUN_ASSAULTRIFLE_ID*/		WFX_FIRESOUND | WFX_LIGHT | WFX_MARK | WFX_MUZZLE | WFX_TRACER | WFX_SHELL,
		/*GUN_ENERGYGRENADE_ID*/	WFX_MARK | WFX_SPARKS | WFX_SMOKE,
		/*GUN_AMBEDPROXIMITYMINE_ID*/	WFX_FIRESOUND,
		/*GUN_AMBEDRASG_ID*/		WFX_FIRESOUND | WFX_LIGHT | WFX_MARK | WFX_MUZZLE | WFX_TRACER | WFX_SHELL,
		/*GUN_TOW_ID*/				WFX_MARK | WFX_SPARKS | WFX_SMOKE,
		/*NOT USED2*/				0,
		/*GUN_AMBEDMINIGUN_ID*/		WFX_FIRESOUND | WFX_LIGHT | WFX_MARK | WFX_MUZZLE | WFX_TRACER | WFX_SHELL,
		/*GUN_AMBEDDRILL_ID*/		WFX_FIRESOUND | WFX_SPARKS,
		/*GUN_MCACLAW_ID*/			WFX_FIRESOUND | WFX_SPARKS,
		/*GUN_OFCLAW_ID*/			WFX_FIRESOUND | WFX_SPARKS,
		/*GUN_BUGCLAW_ID*/			WFX_FIRESOUND | WFX_SPARKS
	};

	return s_nWeaponFX[nWeaponId];
}


// get the maximum range for each weapon
inline DBYTE GetProjectileFX(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0;

	DBYTE s_nProjectileFX[GUN_MAX_NUMBER] =
	{
		/*GUN_AMBEDKATOBURST_ID*/	PFX_LIGHT | PFX_SMOKETRAIL,
		/*GUN_LASERCANNON_ID*/		0,				
		/*GUN_SPIDER_ID*/			PFX_FLYSOUND,
		/*GUN_BULLGUT_ID*/			PFX_FLYSOUND | PFX_SMOKETRAIL | PFX_FLARE | PFX_LIGHT,
		/*GUN_SNIPERRIFLE_ID*/		0,
		/*GUN_AMBEDSHOCHI_ID*/		PFX_LIGHT,//0,	
		/*GUN_SHREDDER_ID*/			0,
		/*GUN_AMBEDMORT_ID*/		PFX_FLYSOUND /*| PFX_SMOKETRAIL | PFX_FLARE | PFX_LIGHT*/, //new ev - 4/12/98
		/*GUN_MORTFRAGMENT_ID*/		PFX_SMOKETRAIL,//0, new ev - 19/12/98 was 0
		/*GUN_AMBEDHWAR_ID*/		0,
		/*GUN_ENERGYBATON_ID*/		0,
		/*GUN_ENERGYBLADE_ID*/		0,
		/*GUN_KATANA_ID*/			0,
		/*GUN_AMBEDKATOAXE_ID*/		0,
		/*GUN_AMBEDEMS_ID*/			PFX_LIGHT, //0, // new ev - 12/1/99
		/*GUN_AMBEDAGUMA_ID*/		0,
		/*GUN_ASSAULTRIFLE_ID*/		0,
		/*GUN_ENERGYGRENADE_ID*/	PFX_FLYSOUND | PFX_LIGHT,
		/*GUN_AMBEDPROXIMITYMINE_ID*/	0,//PFX_FLYSOUND,
		/*GUN_AMBEDRASG_ID*/		0,
		/*GUN_TOW_ID*/				PFX_FLYSOUND | PFX_SMOKETRAIL | PFX_FLARE | PFX_LIGHT,
		/*NOT USED2*/				0,
		/*GUN_AMBEDMINIGUN_ID*/		0,
		/*GUN_AMBEDDRILL_ID*/		0,
		/*GUN_MCACLAW_ID*/			0,
		/*GUN_OFCLAW_ID*/			0,
		/*GUN_BUGCLAW_ID*/			0
	};

	return s_nProjectileFX[nWeaponId];
}




////////////////////////////////////////////////////////////////////////////////
//
// The following functions replace data members in weapon.cpp that used to be
// sent from the server to the client....
//
/////////////////////////////////////////////////////////////////////////////////

// get the name of the flash used by this weapon
inline char* GetFlashFilename(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return DNULL;

	char* s_cFlashFilename[GUN_MAX_NUMBER] =
	{
		/*GUN_AMBEDKATOBURST_ID*/	DNULL,
		/*GUN_LASERCANNON_ID*/		DNULL,				
		/*GUN_SPIDER_ID*/			DNULL,
		/*GUN_BULLGUT_ID*/			DNULL,
		/*GUN_SNIPERRIFLE_ID*/		DNULL,
		/*GUN_AMBEDSHOCHI_ID*/		DNULL,	
		/*GUN_SHREDDER_ID*/			DNULL,
		/*GUN_AMBEDMORT_ID*/		DNULL,
		/*NOT USED1*/				DNULL,
		/*GUN_AMBEDHWAR_ID*/		DNULL,
		/*GUN_ENERGYBATON_ID*/		DNULL,
		/*GUN_ENERGYBLADE_ID*/		DNULL,
		/*GUN_KATANA_ID*/			DNULL,
		/*GUN_AMBEDKATOAXE_ID*/		DNULL,
		/*GUN_AMBEDEMS_ID*/			DNULL,// "Sprites\\Weapons\\EMSFlash.spr", // new 12/1/99 no more flash for you
		/*GUN_AMBEDAGUMA_ID*/		"Sprites\\shotgunmuzzleflash.spr",
		/*GUN_ASSAULTRIFLE_ID*/		"Sprites\\AssaultRifleMuzzleflash.spr",
		/*GUN_ENERGYGRENADE_ID*/	DNULL,
		/*GUN_AMBEDPROXIMITYMINE_ID*/	DNULL,
		/*GUN_AMBEDRASG_ID*/		"Sprites\\Weapons\\Mac10Flash.spr",
		/*GUN_TOW_ID*/				DNULL,
		/*NOT USED2*/				DNULL,
		/*GUN_AMBEDMINIGUN_ID*/		"Sprites\\AssaultRifleMuzzleflash.spr", // use this for the time being
		/*GUN_AMBEDDRILL_ID*/		DNULL,
		/*GUN_MCACLAW_ID*/			DNULL,
		/*GUN_OFCLAW_ID*/			DNULL,
		/*GUN_BUGCLAW_ID*/			DNULL
	};

	return s_cFlashFilename[nWeaponId];
}


// get the duration of the flash used by this weapon
inline DFLOAT GetFlashDuration(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0.0f;

	DFLOAT s_fFlashDuration[GUN_MAX_NUMBER] =
	{
		/*GUN_AMBEDKATOBURST_ID*/	0.0f,
		/*GUN_LASERCANNON_ID*/		0.0f,				
		/*GUN_SPIDER_ID*/			0.0f,
		/*GUN_BULLGUT_ID*/			0.0f,
		/*GUN_SNIPERRIFLE_ID*/		0.0f,
		/*GUN_AMBEDSHOCHI_ID*/		0.0f,	
		/*GUN_SHREDDER_ID*/			0.0f,
		/*GUN_AMBEDMORT_ID*/		0.0f,
		/*NOT USED1*/				0.0f,
		/*GUN_AMBEDHWAR_ID*/		0.0f,
		/*GUN_ENERGYBATON_ID*/		0.0f,
		/*GUN_ENERGYBLADE_ID*/		0.0f,
		/*GUN_KATANA_ID*/			0.0f,
		/*GUN_AMBEDKATOAXE_ID*/		0.0f,
		/*GUN_AMBEDEMS_ID*/			0.075f,
		/*GUN_AMBEDAGUMA_ID*/		0.08f,
		/*GUN_ASSAULTRIFLE_ID*/		0.075f,
		/*GUN_ENERGYGRENADE_ID*/	0.0f,
		/*GUN_AMBEDPROXIMITYMINE_ID*/	0.0f,
		/*GUN_AMBEDRASG_ID*/		0.075f,
		/*GUN_TOW_ID*/				0.0f,
		/*NOT USED2*/				0.0f,
		/*GUN_AMBEDMINIGUN_ID*/		0.1f,
		/*GUN_AMBEDDRILL_ID*/		0.0f,
		/*GUN_MCACLAW_ID*/			0.0f,
		/*GUN_OFCLAW_ID*/			0.0f,
		/*GUN_BUGCLAW_ID*/			0.0f
	};

	return s_fFlashDuration[nWeaponId];
}

// get the scale of the flash used by this weapon
inline DVector GetFlashScale(RiotWeaponId nWeaponId, ModelSize eSize=MS_NORMAL)
{
	DVector vRet;
	VEC_INIT(vRet)

	switch (nWeaponId)
	{
		case GUN_AMBEDMINIGUN_ID :
			VEC_SET(vRet, 0.1f, 0.1f, 0.0f);
		break;

		case GUN_AMBEDAGUMA_ID	:
			VEC_SET(vRet, 0.125f, 0.125f, 0.0f);
		break;

		case GUN_ASSAULTRIFLE_ID :
			VEC_SET(vRet, 0.1f, 0.1f, 0.0f);
		break;

		case GUN_AMBEDRASG_ID :
			VEC_SET(vRet, 0.1f, 0.1f, 0.0f);
		break;

		default : break;
	}

	DFLOAT fFactor = (eSize == MS_NORMAL ? 1.0f : (eSize == MS_SMALL ? 0.2f : 5.0f));
	VEC_MULSCALAR(vRet, vRet, fFactor);

	return vRet;
}


// Get the offset of the weapon from the player object...
inline DVector GetWeaponOffset(RiotWeaponId nWeaponId)
{
	DVector vRet;
	VEC_SET(vRet, 1.0f, 1.0f, 1.0f)

	switch (nWeaponId)
	{
		case GUN_AMBEDDRILL_ID :
			VEC_SET(vRet, 2.0f, -5.5f, 2.0f);
		break;

		case GUN_AMBEDPROXIMITYMINE_ID :
			VEC_SET(vRet, 3.5f, -4.5f, 8.75f);
		break;

		case GUN_AMBEDEMS_ID :
			VEC_SET(vRet, 4.75f, -5.45f, 10.25f);
		break;

		case GUN_AMBEDAGUMA_ID	:
			VEC_SET(vRet, -1.2f, -4.5f, 1.85f);
		break;

		case GUN_AMBEDRASG_ID :
			VEC_SET(vRet, -3.0f, -4.5f, -3.0f);
		break;

		case GUN_ASSAULTRIFLE_ID :
			VEC_SET(vRet, 1.4f, -2.2f, 7.3f);
		break;

		case GUN_ENERGYGRENADE_ID :
			VEC_SET(vRet, 0.4f, -1.8f, 5.7f);
		break;

		case GUN_TOW_ID	:
			VEC_SET(vRet, 1.15f, -4.5f, 8.0f);
		break;

		case GUN_AMBEDMINIGUN_ID :
			VEC_SET(vRet, 0.75f, -1.5f, 1.75f);
		break;

		case GUN_AMBEDKATOBURST_ID :
			VEC_SET(vRet, 4.75f, -4.75f, 12.0f);
		break;

		case GUN_SHREDDER_ID	:
			VEC_SET(vRet, 1.0f, -1.4f, 6.5f);
		break;

		case GUN_AMBEDSHOCHI_ID :
			VEC_SET(vRet, 0.0f, -0.5f, 0.2f);
		break;

		case GUN_AMBEDMORT_ID	:
			VEC_SET(vRet, 3.1f, -4.0f, 2.25f);//1.0f, -2.4f, 1.0f);
		break;

		case GUN_BULLGUT_ID	:
			VEC_SET(vRet, 1.0f, -2.4f, 1.0f);
		break;

		case GUN_LASERCANNON_ID:
			VEC_SET(vRet, 0.6f, -1.8f, -1.0f);
		break;

		case GUN_SPIDER_ID :
			VEC_SET(vRet, 0.5f, -1.6f, 6.0f);
		break;

		case GUN_SNIPERRIFLE_ID :
			VEC_SET(vRet, 0.9f, -2.8f, -1.2f);
		break;

		case GUN_AMBEDHWAR_ID :
			VEC_SET(vRet, 0.4f, -0.425f, 1.00f);
		break;

		case GUN_ENERGYBATON_ID :
			VEC_SET(vRet, -0.4f, -3.5f, 3.7f);
		break;

		case GUN_ENERGYBLADE_ID :
			VEC_SET(vRet, 0.8f, -1.2f, 2.9f);
		break;

		case GUN_KATANA_ID :
			VEC_SET(vRet, 0.5f, -1.9f, 0.2f);
		break;

		case GUN_AMBEDKATOAXE_ID :
			VEC_SET(vRet, 1.5f, -3.5f, 7.5f);
		break;

		default : break;
	}

	return vRet;
}


// Get the offset of the muzzle from the player object...
inline DVector GetWeaponMuzzleOffset(RiotWeaponId nWeaponId, WeaponState eState=W_FIRED)
{
	DVector vRet;
	VEC_SET(vRet, 0.0f, 0.0f, 0.0f)

	switch (nWeaponId)
	{
		case GUN_AMBEDDRILL_ID :
			VEC_SET(vRet, 2.0f, -6.5f, -1.0f);
		break;

		case GUN_AMBEDPROXIMITYMINE_ID :
			VEC_SET(vRet, 3.5f, -2.0f, 20.0f);
		break;

		case GUN_AMBEDEMS_ID :
			VEC_SET(vRet, 3.5f, -1.3f, 24.0f);
		break;

		case GUN_AMBEDAGUMA_ID	:
			//VEC_SET(vRet, 10.0f, 2.0f, 27.5f);
			VEC_SET(vRet, 10.0f, 2.0f, 27.5f);
		break;

		case GUN_AMBEDRASG_ID :
			//VEC_SET(vRet, 10.0, -1.25f, 46.5f);
			VEC_SET(vRet, 10.0, -1.25f, 46.5f);
		break;

		case GUN_ASSAULTRIFLE_ID :
			VEC_SET(vRet, 0.6f, 0.0f, 10.3f);
		break;

		case GUN_ENERGYGRENADE_ID :
			VEC_SET(vRet, 4.4f, -0.6f, 15.9f);
		break;

		case GUN_TOW_ID	:
			VEC_SET(vRet, 1.85f, 1.36f, 11.2f);
		break;

		case GUN_AMBEDMINIGUN_ID :
			VEC_SET(vRet, 3.0f, -3.0f, 18.0f);
		break;

		case GUN_AMBEDKATOBURST_ID :
			VEC_SET(vRet, 1.1f, -1.0f, 21.5f);
		break;

		case GUN_SHREDDER_ID	:
			VEC_SET(vRet, 1.6f, -2.8f, 17.1f);
		break;

		case GUN_AMBEDSHOCHI_ID :
		{
			if (eState == W_FIRED)
			{
				VEC_SET(vRet, 3.0f, -6.5f, 2.0f);
			}
			else if (eState == W_FIRED2)
			{
				VEC_SET(vRet, 6.0f, -8.0f, 2.0f);
			}
			else if (eState == W_FIRED3)
			{
				VEC_SET(vRet, 9.0f, -5.0f, 2.0f);
			}
		}
		break;

		case GUN_AMBEDMORT_ID	:
			VEC_SET(vRet, 0.0f, 0.36f, 16.6f);
		break;

		case GUN_BULLGUT_ID	:
			VEC_SET(vRet, -2.75f, 0.36f, 16.6f);
		break;

		case GUN_LASERCANNON_ID:
			VEC_SET(vRet, 0.4f, 0.0f, 21.4f);
		break;

		case GUN_SPIDER_ID :
			VEC_SET(vRet, 0.5f, 0.7f, 14.4f);
		break;

		case GUN_SNIPERRIFLE_ID :
			VEC_SET(vRet, -0.8f, 3.0f, 21.9f);
		break;

		case GUN_AMBEDHWAR_ID :
			VEC_SET(vRet, 0.2f, 0.25f, 14.3f);
		break;

		case GUN_ENERGYBATON_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		case GUN_ENERGYBLADE_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		case GUN_KATANA_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		case GUN_AMBEDKATOAXE_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		default : break;
	}

	return vRet;
}


// Get the offset of the weapon from the player object...
inline DVector GetWeaponOffset2(RiotWeaponId nWeaponId)
{
	DVector vRet;
	VEC_SET(vRet, 0.0f, 0.0f, 0.0f)

	switch (nWeaponId)
	{
		case GUN_AMBEDDRILL_ID :
			VEC_SET(vRet, 2.0f, -5.5f, -1.50f);
		break;

		case GUN_AMBEDPROXIMITYMINE_ID :
			VEC_SET(vRet, 3.5f, -4.75f, 4.75f);
		
		case GUN_AMBEDEMS_ID :
			VEC_SET(vRet, 3.0f, -5.45f, 5.0f);
		break;

		case GUN_AMBEDAGUMA_ID	:
			VEC_SET(vRet, -3.0f, -5.5f, -1.0f);
		break;

		case GUN_AMBEDRASG_ID :
			VEC_SET(vRet, -3.875f, -4.375f, -8.0f);
		break;

		case GUN_ASSAULTRIFLE_ID :
			VEC_SET(vRet, 1.4f, -2.2f, -0.5f);
		break;

		case GUN_ENERGYGRENADE_ID :
			VEC_SET(vRet, 0.4f, -1.8f, 1.0f);
		break;

		case GUN_TOW_ID	:
			VEC_SET(vRet, 1.0f, -4.5f, 3.5f);
		break;

		case GUN_AMBEDMINIGUN_ID :
			//VEC_SET(vRet, 1.0f, -1.5f, 1.75f);
			VEC_SET(vRet, 0.25f, 0.0f, -4.0f);
		break;

		case GUN_AMBEDKATOBURST_ID :
			VEC_SET(vRet, 3.75f, -3.0f, 5.0f);
		break;

		case GUN_SHREDDER_ID :
			VEC_SET(vRet, 1.0f, -1.4f, -2.5f);
		break;

		case GUN_AMBEDSHOCHI_ID :
			//VEC_SET(vRet, -0.5f, -0.5f, -1.5f);
			VEC_SET(vRet, -0.25f, -0.25f, -1.5f);
		break;

		case GUN_AMBEDMORT_ID	:
			//VEC_SET(vRet, 3.1f, -4.25f, 1.25f); // ok but see the reload
			//VEC_SET(vRet, 3.0f, -3.75f, 0.5f);
			VEC_SET(vRet, 2.5f, -3.25f, 0.0f);
		break;

		case GUN_BULLGUT_ID	:
			VEC_SET(vRet, 1.0f, -2.4f, -3.4f);
		break;

		case GUN_LASERCANNON_ID:
			VEC_SET(vRet, 0.6f, -1.8f, -8.8f);
		break;

		case GUN_SPIDER_ID :
			VEC_SET(vRet, 0.5f, -1.6f, 1.0f);
		break;

		case GUN_SNIPERRIFLE_ID :
			VEC_SET(vRet, 0.9f, -2.8f, -13.0f);
		break;

		case GUN_AMBEDHWAR_ID :
			//VEC_SET(vRet, 0.4f, -0.45f, -0.25f);
			VEC_SET(vRet, 0.4f, -0.45f, -0.5f);
		break;

		case GUN_ENERGYBATON_ID :
			VEC_SET(vRet, -0.4f, -3.5f, 3.7f);
		break;

		case GUN_ENERGYBLADE_ID :
			VEC_SET(vRet, 0.8f, -1.2f, 0.9f);
		break;

		case GUN_KATANA_ID :
			VEC_SET(vRet, 0.5f, -1.9f, -3.7f);
		break;

		case GUN_AMBEDKATOAXE_ID :
			VEC_SET(vRet, 1.5f, -3.5f, 3.75f);
		break;

		default : break;
	}

	return vRet;
}


// Get the offset of the muzzle from the player object...
inline DVector GetWeaponMuzzleOffset2(RiotWeaponId nWeaponId, WeaponState eState=W_FIRED)
{
	DVector vRet;
	VEC_SET(vRet, 0.0f, 0.0f, 0.0f)

	switch (nWeaponId)
	{
		case GUN_AMBEDDRILL_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		case GUN_AMBEDPROXIMITYMINE_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 10.0f);
		break;

		case GUN_AMBEDEMS_ID :
			VEC_SET(vRet, 3.75f, -1.3f, 10.0f);
		break;

		case GUN_AMBEDAGUMA_ID	:
			VEC_SET(vRet, 6.25f, 2.5f, 25.0f);
		break;

		case GUN_AMBEDRASG_ID :
			VEC_SET(vRet, 8.0, -1.25f, 36.5f);
		break;

		case GUN_ASSAULTRIFLE_ID :
			VEC_SET(vRet, 1.4f, -1.4f, 11.7f);
		break;

		case GUN_ENERGYGRENADE_ID :
			VEC_SET(vRet, 4.4f, -0.6f, 15.9f);
		break;

		case GUN_TOW_ID	:
			VEC_SET(vRet, 1.85f, 1.36f, 11.2f);
		break;

		case GUN_AMBEDMINIGUN_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		case GUN_AMBEDKATOBURST_ID :
			VEC_SET(vRet, 1.1f, -1.0f, 21.5f);
		break;

		case GUN_SHREDDER_ID :
			VEC_SET(vRet, 1.4f, -3.0f, 13.3f);
		break;

		case GUN_AMBEDSHOCHI_ID :
		{
			if (eState == W_FIRED)
			{
				VEC_SET(vRet, 2.0f, -4.5f, 1.0f);
			}
			else if (eState == W_FIRED2)
			{
				VEC_SET(vRet, 4.0f, -4.9f, 1.0f);
			}
			else if (eState == W_FIRED3)
			{
				VEC_SET(vRet, 6.0f, -4.1f, 1.0f);
			}
		}
		break;

		case GUN_AMBEDMORT_ID	:
			VEC_SET(vRet, -2.75f, 0.36f, 16.6f);
		break;

		case GUN_BULLGUT_ID	:
			VEC_SET(vRet, -2.75f, 0.36f, 16.6f);
		break;

		case GUN_LASERCANNON_ID :
			VEC_SET(vRet, 1.4f, -2.2f, 18.6f);
		break;

		case GUN_SPIDER_ID :
			VEC_SET(vRet, 0.5f, 0.7f, 14.4f);
		break;

		case GUN_SNIPERRIFLE_ID :
			VEC_SET(vRet, -0.5f, 2.6f, -12.8f);
		break;

		case GUN_AMBEDHWAR_ID :
			VEC_SET(vRet, 0.2f, 0.25f, 14.3f);
		break;

		case GUN_ENERGYBATON_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		case GUN_ENERGYBLADE_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		case GUN_KATANA_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		case GUN_AMBEDKATOAXE_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		default : break;
	}

	return vRet;
}

// get the scale of the flash used by this weapon
inline DVector GetFlashScale2(RiotWeaponId nWeaponId, ModelSize eSize=MS_NORMAL)
{
	DVector vRet;
	VEC_INIT(vRet)

	switch (nWeaponId)
	{
		case GUN_AMBEDMINIGUN_ID :
			VEC_SET(vRet, 0.04f, 0.04f, 0.0f);
		break;

		case GUN_AMBEDAGUMA_ID	:
			VEC_SET(vRet, 0.05f, 0.05f, 0.0f);
		break;

		case GUN_ASSAULTRIFLE_ID :
			VEC_SET(vRet, 0.10f, 0.10f, 0.0f);
		break;

		case GUN_AMBEDRASG_ID :
			VEC_SET(vRet, 0.05f, 0.05f, 0.0f);
		break;

		default : break;
	}

	DFLOAT fFactor = (eSize == MS_NORMAL ? 1.0f : (eSize == MS_SMALL ? 0.2f : 5.0f));
	VEC_MULSCALAR(vRet, vRet, fFactor);

	return vRet;
}


// Get the offset of the muzzle from the player object...
inline DVector GetWeaponRecoil(RiotWeaponId nWeaponId)
{
	DVector vRet;
	VEC_SET(vRet, 0.0f, 0.0f, 0.0f)

	switch (nWeaponId)
	{
		case GUN_AMBEDRASG_ID :
			VEC_SET(vRet, 0.03f, 0.03f, 0.15f);
		break;

		case GUN_ASSAULTRIFLE_ID :
			VEC_SET(vRet, 0.03f, 0.03f, 0.15f);
		break;

		case GUN_AMBEDAGUMA_ID :
			VEC_SET(vRet, 0.05f, 0.125f, 0.30f);

		default : break;
	}

	return vRet;
}


inline DDWORD GetExtraWeaponFlags(RiotWeaponId nWeaponId) 
{ 
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0;

	DDWORD m_dwExtraModelFlags[GUN_MAX_NUMBER] =
	{
		/*GUN_AMBEDKATOBURST_ID*/	0,
		/*GUN_LASERCANNON_ID*/		0,				
		/*GUN_SPIDER_ID*/			FLAG_ENVIRONMENTMAP,
		/*GUN_BULLGUT_ID*/			0,
		/*GUN_SNIPERRIFLE_ID*/		FLAG_ENVIRONMENTMAP,
		/*GUN_AMBEDSHOCHI_ID*/		0,	
		/*GUN_SHREDDER_ID*/			0,
		/*GUN_AMBEDMORT_ID*/		0,
		/*NOT USED1*/				0,
		/*GUN_AMBEDHWAR_ID*/		0,
		/*GUN_ENERGYBATON_ID*/		0,
		/*GUN_ENERGYBLADE_ID*/		0,
		/*GUN_KATANA_ID*/			FLAG_ENVIRONMENTMAP,
		/*GUN_AMBEDKATOAXE_ID*/		0,//    FLAG_ENVIRONMENTMAP,
		/*GUN_AMBEDEMS_ID*/			0,//	FLAG_ENVIRONMENTMAP, // new ev - 12/1/99 
		/*GUN_AMBEDAGUMA_ID*/		0,//	FLAG_ENVIRONMENTMAP, // new ev - 12/1/99 
		/*GUN_ASSAULTRIFLE_ID*/		0,
		/*GUN_ENERGYGRENADE_ID*/	0,
		/*GUN_AMBEDPROXIMITYMINE_ID*/ 0,
		/*GUN_AMBEDRASG_ID*/		FLAG_ENVIRONMENTMAP,
		/*GUN_TOW_ID*/				0,
		/*NOT USED2*/				0,
		/*GUN_AMBEDMINIGUN_ID*/		FLAG_ENVIRONMENTMAP,//0,
		/*GUN_AMBEDDRILL_ID*/		0,//FLAG_ENVIRONMENTMAP
		/*GUN_MCACLAW_ID*/			0,
		/*GUN_OFCLAW_ID*/			0,
		/*GUN_BUGCLAW_ID*/			0
	};

	return m_dwExtraModelFlags[nWeaponId]; 
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	GetWeaponFireSound()
//
//	PURPOSE:	Get fire sounds associated with this weapon
//
// ----------------------------------------------------------------------- //

inline DBOOL GetWeaponSoundDir(RiotWeaponId nId)
{
	char* pSound = DNULL;

	switch (nId)
	{
		case GUN_AMBEDKATOBURST_ID :
			pSound = "AmbedKatoBurst";
		break;

		case GUN_SPIDER_ID :
			pSound = "Spider";
		break;

		case GUN_AMBEDMORT_ID :
			pSound = "AmbedMORT";
		break;

		case GUN_BULLGUT_ID :
			pSound = "Bullgut";
		break;

		case GUN_SNIPERRIFLE_ID :
			pSound = "SniperRifle";
		break;

		case GUN_AMBEDSHOCHI_ID :
			pSound = "AmbedShochi";
		break;

		case GUN_SHREDDER_ID :
			pSound = "Shredder";
		break;

		case GUN_AMBEDHWAR_ID :
			pSound = "AmbedHWAR";
		break;

		case GUN_ENERGYBATON_ID :
			pSound = "EnergyBaton";
		break;

		case GUN_ENERGYBLADE_ID :
			pSound = "EnergyBlade";
		break;

		// LOFFIXME - give each attack a unique sound dir
		case GUN_MCACLAW_ID :
		case GUN_OFCLAW_ID :
		case GUN_BUGCLAW_ID	:
		case GUN_KATANA_ID :
			pSound = "Katana";
		break;

		case GUN_AMBEDKATOAXE_ID :
			pSound = "AmbedKatoAxe";
		break;

		case GUN_AMBEDEMS_ID :
			pSound	= "AmbedEMS";
		break;
		
		case GUN_AMBEDAGUMA_ID	:
			pSound = "Shotgun";
		break;
		
		case GUN_ASSAULTRIFLE_ID :
			pSound = "AssaultRifle";
		break;
		
		case GUN_AMBEDRASG_ID :
			pSound = "MachineGun";
		break;
	
		case GUN_ENERGYGRENADE_ID :
			pSound = "EnergyGrenade";
		break;

		case GUN_TOW_ID	:
			pSound = "TOW";
		break;

		case GUN_LASERCANNON_ID :
			pSound = "LaserCannon";
		break;

		case GUN_AMBEDMINIGUN_ID :
			pSound = "AmbedMinigun";
		break;

		case GUN_AMBEDDRILL_ID :
			pSound = "AmbedDrill";
		break;

		case GUN_AMBEDPROXIMITYMINE_ID :
			pSound = "AmbedProximityMine";
		break;

		default : 
			return DFALSE;
			break;
	}


	sprintf( s_FileBuffer, "Sounds\\Weapons\\%s\\", pSound );
	return DTRUE;
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	GetWeaponFireSound()
//
//	PURPOSE:	Get fire sounds associated with this weapon
//
// ----------------------------------------------------------------------- //

inline char* GetWeaponFireSound(RiotWeaponId nId)
{
	if (!GetWeaponSoundDir(nId)) return DNULL;

	strcat( s_FileBuffer, "fire.wav" );

	return s_FileBuffer;
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	GetWeaponDryFireSound()
//
//	PURPOSE:	Get dry fire sounds associated with this weapon
//
// ----------------------------------------------------------------------- //

inline char* GetWeaponDryFireSound(RiotWeaponId nId)
{
	if (!GetWeaponSoundDir(nId)) return DNULL;

	strcat(s_FileBuffer, "empty.wav");
	return s_FileBuffer;
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	GetWeaponFlyingSound()
//
//	PURPOSE:	Get flying sound associated with this weapon
//
// ----------------------------------------------------------------------- //

inline char* GetWeaponFlyingSound(RiotWeaponId nId)
{
	char* pRet = DNULL;

	switch (nId)
	{
		case GUN_SPIDER_ID :
			pRet = "Sounds\\Weapons\\Spider\\Timer.wav";
		break;

		case GUN_AMBEDMORT_ID :
			pRet = "Sounds\\Weapons\\Bullgut\\projectile.wav";
		break;

		case GUN_BULLGUT_ID :
			pRet = "Sounds\\Weapons\\Bullgut\\projectile.wav";
		break;

		case GUN_ENERGYGRENADE_ID :
			pRet = "Sounds\\Weapons\\EnergyGrenade\\projectile.wav";
		break;

/* new ev - 31/1/99 no longer needed as the proximity mine is a simple grenade
		case GUN_AMBEDPROXIMITYMINE_ID :
			pRet = "Sounds\\Weapons\\Spider\\Timer.wav";
			//pRet = "Sounds\\Weapons\\KatoGrenade\\projectile.wav";
		break;
*/
		case GUN_TOW_ID	:
			pRet = "Sounds\\Weapons\\Tow\\projectile.wav";
		break;

		default:
		break;
	}

	return pRet;
}


// get the duration of the flash used by this weapon
inline int GetShotsPerClip(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return -1;

	int s_nShotsPerClip[GUN_MAX_NUMBER] =
	{
		/*GUN_AMBEDKATOBURST_ID*/	25,
		/*GUN_LASERCANNON_ID*/		-1,				
		/*GUN_SPIDER_ID*/			1,
		/*GUN_BULLGUT_ID*/			4,
		/*GUN_SNIPERRIFLE_ID*/		2,
		/*GUN_AMBEDSHOCHI_ID*/		-1,	
		/*GUN_SHREDDER_ID*/			-1,
		/*GUN_AMBEDMORT_ID*/		1,
		/*NOT USED1*/				0,
		/*GUN_AMBEDHWAR_ID*/		4,
		/*GUN_ENERGYBATON_ID*/		-1,
		/*GUN_ENERGYBLADE_ID*/		-1,
		/*GUN_KATANA_ID*/			-1,
		/*GUN_AMBEDKATOAXE_ID*/		-1,
		/*GUN_AMBEDEMS_ID*/			-1,
		/*GUN_AMBEDAGUMA_ID*/		8, // new ev 19/12/98
		/*GUN_ASSAULTRIFLE_ID*/		50,
		/*GUN_ENERGYGRENADE_ID*/	6,
		/*GUN_AMBEDPROXIMITYMINE_ID*/	1,
		/*GUN_AMBEDRASG_ID*/		20,
		/*GUN_TOW_ID*/				1,
		/*NOT USED2*/				0,
		/*GUN_AMBEDMINIGUN_ID*/		50,
		/*GUN_AMBEDDRILL_ID*/		-1,
		/*GUN_MCACLAW_ID*/			-1,
		/*GUN_OFCLAW_ID*/			-1,
		/*GUN_BUGCLAW_ID*/			-1
	};

	return s_nShotsPerClip[nWeaponId];
}

// Get the number of vectors this weapon shoots...
inline int GetVectorsPerShot(int nWeaponId)
{
	if (nWeaponId == GUN_AMBEDAGUMA_ID) return 5;

	return 1;
}

// Get the number of vectors this weapon shoots...
inline DFLOAT GetWeaponSpread(int nWeaponId)
{
	if (nWeaponId == GUN_BULLGUT_ID) return 3.5f;
	if (nWeaponId == GUN_AMBEDMORT_ID) return 3.5f;

	return 0.0f;
}


// get the duration of the flash used by this weapon
inline DBYTE GetWeaponPerturbe(int nWeaponId, DFLOAT & fRPerturbe, 
							   DFLOAT & fUPerturbe)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0;

	// Important that these values are between 0 and 127...

	DBYTE s_nPerturbe[GUN_MAX_NUMBER] =
	{
		/*GUN_AMBEDKATOBURST_ID*/	3,
		/*GUN_LASERCANNON_ID*/		0,				
		/*GUN_SPIDER_ID*/			0,
		/*GUN_BULLGUT_ID*/			5,
		/*GUN_SNIPERRIFLE_ID*/		0,
		/*GUN_AMBEDSHOCHI_ID*/		10,	
		/*GUN_SHREDDER_ID*/			15,
		/*GUN_AMBEDMORT_ID*/		5,
		/*NOT USED1*/				0,
		/*GUN_AMBEDHWAR_ID*/		20,
		/*GUN_ENERGYBATON_ID*/		0,
		/*GUN_ENERGYBLADE_ID*/		0,
		/*GUN_KATANA_ID*/			0,
		/*GUN_AMBEDKATOAXE_ID*/		0,
		/*GUN_AMBEDEMS_ID*/			15, // new ev - 12/1/99 was 20
		/*GUN_AMBEDAGUMA_ID*/		50,
		/*GUN_ASSAULTRIFLE_ID*/		5,
		/*GUN_ENERGYGRENADE_ID*/	10,
		/*GUN_AMBEDPROXIMITYMINE_ID*/	10,
		/*GUN_AMBEDRASG_ID*/		30, // new ev - 3/12/98 was 40
		/*GUN_TOW_ID*/				0,
		/*NOT USED2*/				0,
		/*GUN_AMBEDMINIGUN_ID*/		10,
		/*GUN_AMBEDDRILL_ID*/		0,
		/*GUN_MCACLAW_ID*/			0,
		/*GUN_OFCLAW_ID*/			0,
		/*GUN_BUGCLAW_ID*/			0
	};

	int nPerturbe = s_nPerturbe[nWeaponId];

	// Get a random perturbe based on nPerturbe...USE nSeed!!!  Need to come
	// up with an algorithm that uses nSeed (and adjusts nSeed?) so that this
	// can be called with the same seed to generate the same "random" values...

	fRPerturbe = ((DFLOAT)GetRandom(-nPerturbe, nPerturbe))/1000.0f;
	fUPerturbe = ((DFLOAT)GetRandom(-nPerturbe, nPerturbe))/1000.0f;

	return nPerturbe;
}


inline char* GetPVModelName(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return DNULL;

	char* s_cPVModelName[GUN_MAX_NUMBER] =
	{
		/*GUN_AMBEDKATOBURST_ID*/	"Models\\PV_Weapons\\Ambed_KatoBurst_pv.abc",
		/*GUN_LASERCANNON_ID*/		"Models\\PV_Weapons\\lasercannon_pv.abc",				
		/*GUN_SPIDER_ID*/			"Models\\PV_Weapons\\Spider_pv.abc",
		/*GUN_BULLGUT_ID*/			"Models\\PV_Weapons\\Bullgut_pv.abc",
		/*GUN_SNIPERRIFLE_ID*/		"Models\\PV_Weapons\\SniperRifle_pv.abc",
		/*GUN_AMBEDSHOCHI_ID*/		"Models\\PV_Weapons\\Ambed_Shochi_pv.abc",	
		/*GUN_SHREDDER_ID*/			"Models\\PV_Weapons\\Shredder_pv.abc",
		/*GUN_AMBEDMORT_ID*/		"Models\\PV_Weapons\\Ambed_MORT_pv.abc",
		/*NOT USED1*/				DNULL,
		/*GUN_AMBEDHWAR_ID*/		"Models\\PV_Weapons\\Ambed_HWAR_pv.abc",
		/*GUN_ENERGYBATON_ID*/		"Models\\PV_Weapons\\energy_baton_pv.abc",
		/*GUN_ENERGYBLADE_ID*/		"Models\\PV_Weapons\\energy_blade_pv.abc",
		/*GUN_KATANA_ID*/			"Models\\PV_Weapons\\katana_pv.abc",
		/*GUN_AMBEDKATOAXE_ID*/		"Models\\PV_Weapons\\Ambed_KatoAxe_pv.abc",
		/*GUN_AMBEDEMS_ID*/			"Models\\PV_Weapons\\Ambed_EMS_pv.abc",
		/*GUN_AMBEDAGUMA_ID*/		"Models\\PV_Weapons\\Ambed_Aguma_pv.abc",
		/*GUN_ASSAULTRIFLE_ID*/		"Models\\PV_Weapons\\AssaultRifle_pv.abc",
		/*GUN_ENERGYGRENADE_ID*/	"Models\\PV_Weapons\\EnergyGrenade_pv.abc",
		/*GUN_AMBEDPROXIMITYMINE_ID*/	"Models\\PV_Weapons\\Ambed_ProximityMine_pv.abc",
		/*GUN_AMBEDRASG_ID*/		"Models\\PV_Weapons\\Ambed_RASG_pv.abc",
		/*GUN_TOW_ID*/				"Models\\PV_Weapons\\TOW_pv.abc",
		/*NOT USED2*/				DNULL,
		/*GUN_AMBEDMINIGUN_ID*/		"Models\\PV_Weapons\\Ambed_Minigun_pv.abc",
		/*GUN_AMBEDDRILL_ID*/		"Models\\PV_Weapons\\Ambed_Drill_pv.abc",
		/*GUN_MCACLAW_ID*/			DNULL,
		/*GUN_OFCLAW_ID*/			DNULL,
		/*GUN_BUGCLAW_ID*/			DNULL
	};

	return s_cPVModelName[nWeaponId];
}

inline char* GetPVModelSkin(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return DNULL;

	char* s_cPVModelSkin[GUN_MAX_NUMBER] =
	{
		/*GUN_AMBEDKATOBURST_ID*/	"Skins\\weapons\\Ambed_KatoBurst_pv_a.dtx",
		/*GUN_LASERCANNON_ID*/		"Skins\\weapons\\lasercannon_pv_a.dtx",				
		/*GUN_SPIDER_ID*/			"Skins\\weapons\\Spider_pv_a.dtx",
		/*GUN_BULLGUT_ID*/			"Skins\\weapons\\Bullgut_pv_a.dtx",
		/*GUN_SNIPERRIFLE_ID*/		"Skins\\weapons\\SniperRifle_pv_a.dtx",
		/*GUN_AMBEDSHOCHI_ID*/		"Skins\\weapons\\Ambed_Shochi_pv_a.dtx",	
		/*GUN_SHREDDER_ID*/			"Skins\\weapons\\Shredder_pv_a.dtx",
		/*GUN_AMBEDMORT_ID*/		"Skins\\weapons\\Ambed_MORT_pv_a.dtx",
		/*NOT USED1*/				DNULL,
		/*GUN_AMBEDHWAR_ID*/		"Skins\\weapons\\Ambed_HWAR_pv_a.dtx",
		/*GUN_ENERGYBATON_ID*/		"Skins\\weapons\\energy_baton_pv_a.dtx",
		/*GUN_ENERGYBLADE_ID*/		"Skins\\weapons\\energy_blade_pv_a.dtx",
		/*GUN_KATANA_ID*/			"Skins\\weapons\\katana_pv_a.dtx",
		/*GUN_AMBEDKATOAXE_ID*/		"Skins\\weapons\\Ambed_KatoAxe_pv_a.dtx",
		/*GUN_AMBEDEMS_ID*/			"Skins\\weapons\\Ambed_EMS_pv_a.dtx",
		/*GUN_AMBEDAGUMA_ID*/		"Skins\\weapons\\Ambed_Aguma_pv_a.dtx",
		/*GUN_ASSAULTRIFLE_ID*/		"Skins\\weapons\\AssaultRifle_pv_a.dtx",
		/*GUN_ENERGYGRENADE_ID*/	"Skins\\weapons\\EnergyGrenade_pv_a.dtx",
		/*GUN_AMBEDPROXIMITYMINE_ID*/"Skins\\weapons\\Ambed_ProximityMine_pv_a.dtx",
		/*GUN_AMBEDRASG_ID*/		"Skins\\weapons\\Ambed_RASG_pv_a.dtx",
		/*GUN_TOW_ID*/				"Skins\\weapons\\TOW_pv_a.dtx",
		/*NOT USED2*/				DNULL,
		/*GUN_AMBEDMINIGUN_ID*/		"Skins\\weapons\\Ambed_Minigun_pv_a.dtx",
		/*GUN_AMBEDDRILL_ID*/		"Skins\\weapons\\Ambed_Drill_pv_a.dtx",
		/*GUN_MCACLAW_ID*/			DNULL,
		/*GUN_OFCLAW_ID*/			DNULL,
		/*GUN_BUGCLAW_ID*/			DNULL
	};

	return s_cPVModelSkin[nWeaponId];
}


// get the velocity of this weapon (projectiles only)...

inline DFLOAT GetWeaponVelocity(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0.0f;

	DFLOAT s_fVelocity[GUN_MAX_NUMBER] =
	{
		/*GUN_AMBEDKATOBURST_ID*/	2000.0f,
		/*GUN_LASERCANNON_ID*/		0.0f,				
		/*GUN_SPIDER_ID*/			800.0f,
		/*GUN_BULLGUT_ID*/			1000.0f,
		/*GUN_SNIPERRIFLE_ID*/		0.0f,
		/*GUN_AMBEDSHOCHI_ID*/		3000.0f,	
		/*GUN_SHREDDER_ID*/			0.0f,
		/*GUN_AMBEDMORT_ID*/		1000.0f,
		/*NOT USED1*/				0.0f,
		/*GUN_AMBEDHWAR_ID*/		0.0f,
		/*GUN_ENERGYBATON_ID*/		0.0f,
		/*GUN_ENERGYBLADE_ID*/		0.0f,
		/*GUN_KATANA_ID*/			0.0f,
		/*GUN_AMBEDKATOAXE_ID*/		0.0f,
		/*GUN_AMBEDEMS_ID*/			2000.0f,//0.0f, // new ev - 12/1/99 was 0.0f,
		/*GUN_AMBEDAGUMA_ID*/		0.0f,
		/*GUN_ASSAULTRIFLE_ID*/		0.0f,
		/*GUN_ENERGYGRENADE_ID*/	2000.0f,
		/*GUN_AMBEDPROXIMITYMINE_ID*/	750.0f,
		/*GUN_AMBEDRASG_ID*/		0.0f,
		/*GUN_TOW_ID*/				2000.0f,
		/*NOT USED2*/				0.0f,
		/*GUN_AMBEDMINIGUN_ID*/		0.0f,
		/*GUN_AMBEDDRILL_ID*/		0.0f,
		/*GUN_MCACLAW_ID*/			0.0f,
		/*GUN_OFCLAW_ID*/			0.0f,
		/*GUN_BUGCLAW_ID*/			0.0f
	};

	return s_fVelocity[nWeaponId];
}


// get the lifetime of this projectile (projectiles only)...

inline DFLOAT GetWeaponLifeTime(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0.0f;

	DFLOAT s_fLifeTime[GUN_MAX_NUMBER] =
	{
		/*GUN_AMBEDKATOBURST_ID*/	5.0f,
		/*GUN_LASERCANNON_ID*/		0.0f,				
		/*GUN_SPIDER_ID*/			3.0f,
		/*GUN_BULLGUT_ID*/			6.0f,
		/*GUN_SNIPERRIFLE_ID*/		0.0f,
		/*GUN_AMBEDSHOCHI_ID*/		3.0f,	
		/*GUN_SHREDDER_ID*/			0.0f,
		/*GUN_AMBEDMORT_ID*/		0.95f, // new ev - 4/12/98 was 6.0f
		/*NOT USED1*/				0.0f,
		/*GUN_AMBEDHWAR_ID*/		0.0f,
		/*GUN_ENERGYBATON_ID*/		0.0f,
		/*GUN_ENERGYBLADE_ID*/		0.0f,
		/*GUN_KATANA_ID*/			0.0f,
		/*GUN_AMBEDKATOAXE_ID*/		0.0f,
		/*GUN_AMBEDEMS_ID*/			2.5f, // new ev - 3/12/98 was 0.0f
		/*GUN_AMBEDAGUMA_ID*/		0.0f,
		/*GUN_ASSAULTRIFLE_ID*/		0.0f,
		/*GUN_ENERGYGRENADE_ID*/	5.0f,
		/*GUN_AMBEDPROXIMITYMINE_ID*/	100000.0f, // these puppies don't go away any time soon :)
		/*GUN_AMBEDRASG_ID*/		0.0f,
		/*GUN_TOW_ID*/				3.0f,
		/*NOT USED2*/				0.0f,
		/*GUN_AMBEDMINIGUN_ID*/		0.0f,
		/*GUN_AMBEDDRILL_ID*/		0.0f,
		/*GUN_MCACLAW_ID*/			0.0f,
		/*GUN_OFCLAW_ID*/			0.0f,
		/*GUN_BUGCLAW_ID*/			0.0f
	};

	return s_fLifeTime[nWeaponId];
}


inline DBOOL CanWeaponZoom(int nWeaponId)
{
	if (nWeaponId == GUN_SNIPERRIFLE_ID || nWeaponId == GUN_ASSAULTRIFLE_ID) return DTRUE;

	return DFALSE;
}


// Calculate the adjusted path and fire position for the particular weapon...

void CalculateWeaponPathAndFirePos(RiotWeaponId nId, DVector & vPath, 
								   DVector & vFirePos, DVector & vU, DVector & vR);


#endif // __WEAPON_DEFS_H__