// ----------------------------------------------------------------------- //
//
// MODULE  : Weapons.cpp
//
// PURPOSE : Weapons container object - Implementation
//
// CREATED : 9/25/97
//
// ----------------------------------------------------------------------- //

#include "Weapons.h"
#include "RiotWeapons.h"
#include "PlayerObj.h"
#include "RiotMsgIds.h"
#include "cpp_server_de.h"
#include "BaseCharacter.h"
#include "RiotServerShell.h"

// new ev - 3/12/98 for the LOF weapons
#include "LOFWeapons.h"

extern CRiotServerShell* g_pRiotServerShellDE;

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CWeapons::CWeapons()
//
//	PURPOSE:	Initialization
//
// ----------------------------------------------------------------------- //

CWeapons::CWeapons() : Aggregate()
{
	// Clear our array...

	for (int i=GUN_FIRST_ID; i < GUN_MAX_NUMBER; i++)
	{
		m_weapons[i] = DNULL;
		// new ev - 18/1/99 adding ammo powerup support
		m_tempAmmo[i] = 0;
	}

	m_nCurWeapon = -1;
	m_eArsenal   = AT_NONE;
	m_eModelSize = MS_NORMAL;
	m_hObject	 = DNULL;
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CWeapons::~CWeapons()
//
//	PURPOSE:	Destructor - deallocate weapons
//
// ----------------------------------------------------------------------- //

CWeapons::~CWeapons()
{
	for (int i=0; i < GUN_MAX_NUMBER; i++)
	{
		if (m_weapons[i]) delete m_weapons[i];
	}
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CWeapons::Init()
//
//	PURPOSE:	Initialize weapons
//
// ----------------------------------------------------------------------- //

DBOOL CWeapons::Init(HOBJECT hObj, ModelSize eSize)
{
	m_hObject	 = hObj;
	m_eModelSize = eSize;

	return DTRUE;
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CWeapons::SetArsenal()
//
//	PURPOSE:	Set up the available weapons
//
// ----------------------------------------------------------------------- //

void CWeapons::SetArsenal(ArsenalType eType)
{
	if (m_eArsenal == eType || !m_hObject) return;

	m_eArsenal = eType;

	switch (eType)
	{
		case AT_MECHA:  // Add the Mecha weapons..
		{
			CreateMechaWeapons();
		}
		break;

		case AT_ONFOOT:
		{
			CreateOnFootWeapons();
		}
		break;

		case AT_ALL_WEAPONS:
		{
			CreateMechaWeapons();
			CreateOnFootWeapons();
		}
		break;

		default : break;
	}
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CWeapons::CreateWeapon()
//
//	PURPOSE:	Create the specified weapon
//
// ----------------------------------------------------------------------- //

void CWeapons::CreateWeapon(DBYTE nWeaponId)
{
	if (!IsValidIndex(nWeaponId) || m_weapons[nWeaponId]) return;

	CWeapon* pWeapon = DNULL;
	DBOOL	 bMelee  = DFALSE;

	switch (nWeaponId)
	{
		case GUN_AMBEDKATOBURST_ID :
			//pWeapon = new CPulseRifle; // new ev - 3/2/99 changing Pulse Rifle to Kato Burst
			pWeapon = new CAmbedKatoBurst;
		break;

		case GUN_SPIDER_ID :
			pWeapon = new CSpider;
		break;

		case GUN_AMBEDMORT_ID :
			pWeapon = new CAmbedMort;
		break;

		case GUN_BULLGUT_ID :
			pWeapon = new CBullgut; // new ev - 14/3/99 adding back Bullgut
		break;

		case GUN_SNIPERRIFLE_ID :
			pWeapon = new CSniperRifle;
		break;

		case GUN_AMBEDSHOCHI_ID :
			//pWeapon = new CJuggernaut; // new ev - 4/12/98 changing Juggernaut to CAmbedDSG
			pWeapon = new CAmbedShochi;
		break;

		case GUN_SHREDDER_ID :
			pWeapon = new CShredder;
		break;

		case GUN_AMBEDHWAR_ID :
			//pWeapon = new CRedRiot; // new ev - 3/12/98 changing RedRiotto CAmbedHWAR
			pWeapon = new CAmbedHWAR;
		break;

		case GUN_ENERGYBATON_ID :
			pWeapon = new CEnergyBaton;
			bMelee = DTRUE;
		break;

		case GUN_ENERGYBLADE_ID :
			pWeapon = new CEnergyBlade;
			bMelee = DTRUE;
		break;

		case GUN_KATANA_ID :
			pWeapon = new CKatana;
			bMelee = DTRUE;
		break;

		case GUN_AMBEDKATOAXE_ID :
			//pWeapon = new CMonoKnife; // new ev - 3/2/99 changing MonoKnife tp CAmbedKatoAxe
			pWeapon = new CAmbedKatoAxe;
			bMelee = DTRUE;
		break;

		// On-foot mode weapons...

		case GUN_AMBEDEMS_ID :
			//pWeapon = new CColt45; // new ev - 12/1/99 changing Colt45 to CAmbedEMS
			pWeapon = new CAmbedEMS;
		break;

		case GUN_AMBEDAGUMA_ID	:
			//pWeapon = new CShotgun;  // new ev - 12/1/99 changing CShotgun to CAmbedAguma
			pWeapon = new CAmbedAguma;
		break;

		case GUN_ASSAULTRIFLE_ID :
			pWeapon = new CAssaultRifle;
		break;

		case GUN_ENERGYGRENADE_ID :
			pWeapon = new CEnergyGrenade;
		break;

		case GUN_AMBEDPROXIMITYMINE_ID :
			//pWeapon = new CKatoGrenade; // new ev - 30/1/99 changing KatoGrenade to CAmbedProximityMine
			pWeapon = new CAmbedProximityMine;
		break;

		case GUN_AMBEDRASG_ID :
			//pWeapon = new CMac10; // new ev - 3/12/98 changing Mac10 to CAmbedRASG
			pWeapon = new CAmbedRASG;
		break;

		case GUN_TOW_ID	:
			pWeapon = new CTOW;
		break;

		case GUN_LASERCANNON_ID :
			pWeapon = new CLaserCannon;
		break;

		case GUN_AMBEDMINIGUN_ID :
			pWeapon = new CAmbedMinigun;
		break;

		case GUN_AMBEDDRILL_ID :
			//pWeapon = new CTanto; // new ev - 30/1/99 changing Tanto to CAmbedDrill
			pWeapon = new CAmbedDrill;
			// new ev - 6/3/99
			if (g_pRiotServerShellDE->GetGameType() != SINGLE)
			{
				// don't automatically acquire the drill for single player
				// games... you have to pickup the Drill powerup first
				bMelee = DTRUE;		
			}
		break;

		
		// NOTE: the following weapons are not for player use

		case GUN_MCACLAW_ID :
			pWeapon = new CMCAClaw;
			bMelee = DTRUE;
		break;

		case GUN_OFCLAW_ID :
			pWeapon = new COFClaw;
			bMelee = DTRUE;
		break;

		case GUN_BUGCLAW_ID :
			pWeapon = new CBugClaw;
			bMelee = DTRUE;
		break;

		default : break;
	}


	if (pWeapon)
	{
		pWeapon->Init(m_hObject, m_eModelSize);
		m_weapons[nWeaponId] = pWeapon;
	
		if (bMelee)
		{
			m_weapons[nWeaponId]->Aquire();
			m_weapons[nWeaponId]->AddAmmo(GetWeaponMaxAmmo(nWeaponId));
		}
	}
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CWeapons::CreateMechaWeapons()
//
//	PURPOSE:	Set up the mecha weapon arsenal
//
// ----------------------------------------------------------------------- //

void CWeapons::CreateMechaWeapons()
{
	// Make sure we don't already have this arsenal...

	if (m_weapons[GUN_FIRSTMECH_ID]) return;

	for (int i=GUN_FIRSTMECH_ID; i <= GUN_LASTMECH_ID; i++)
	{
		CreateWeapon(i);
	}
} 


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CWeapons::CreateOnFootWeapons()
//
//	PURPOSE:	Set up the on foot weapon arsenal
//
// ----------------------------------------------------------------------- //

void CWeapons::CreateOnFootWeapons()
{
	// Make sure we don't already have this arsenal...

	if (m_weapons[GUN_FIRSTONFOOT_ID]) return;

	
	for (int i=GUN_FIRSTONFOOT_ID; i <= GUN_LASTONFOOT_ID; i++)
	{
		CreateWeapon(i);
	}
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CWeapons::EngineMessageFn
//
//	PURPOSE:	Handle message from the engine
//
// ----------------------------------------------------------------------- //
		
DDWORD CWeapons::EngineMessageFn(LPBASECLASS pObject, DDWORD messageID, void *pData, DFLOAT lData)
{
	switch(messageID)
	{
		case MID_SAVEOBJECT:
		{
			Save((HMESSAGEWRITE)pData, (DBYTE)lData);
		}
		break;

		case MID_LOADOBJECT:
		{
			Load((HMESSAGEREAD)pData, (DBYTE)lData);
		}
		break;
	}

	return Aggregate::EngineMessageFn(pObject, messageID, pData, lData);
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CWeapons::ObjectMessageFn
//
//	PURPOSE:	Handle object-to-object messages
//
// ----------------------------------------------------------------------- //

DDWORD CWeapons::ObjectMessageFn(LPBASECLASS pObject, HOBJECT hSender, DDWORD messageID, HMESSAGEREAD hRead)
{
	switch(messageID)
	{
		case MID_ADDWEAPON:
		{	
			CServerDE* pServerDE = BaseClass::GetServerDE();
			if (!pServerDE) return DFALSE;

			DBOOL	bHadIt	  = DTRUE;
			DBOOL   bPickedUp = DFALSE;
			DBYTE	nWeapon   = pServerDE->ReadFromMessageByte(hRead);
			DFLOAT	fAmmo	  = pServerDE->ReadFromMessageFloat(hRead);

			if (IsValidIndex(nWeapon))
			{
				if (m_weapons[nWeapon] && m_weapons[nWeapon]->Have())
				{
					if( m_weapons[nWeapon]->GetAmmoCount() < m_weapons[nWeapon]->GetMaxAmmo())
					{
						bPickedUp = DTRUE;
						m_weapons[nWeapon]->AddAmmo((int)fAmmo);
					}
				}
				else
				{
					bHadIt	  = DFALSE;
					bPickedUp = DTRUE;
					// new ev - 18/1/99 adding ammo powerup support made it add the ammo in inventory
					if (m_weapons[nWeapon])
					{
						ObtainWeapon( nWeapon, (int)fAmmo + m_weapons[nWeapon]->GetAmmoCount() );
					}
					else
					{
						ObtainWeapon( nWeapon, (int)fAmmo + m_tempAmmo[nWeapon] );
						m_tempAmmo[nWeapon] = 0;
					}
				}
			}

			if (bPickedUp)
			{
				// Tell powerup it was picked up...

				HMESSAGEWRITE hWrite = pServerDE->StartMessageToObject(pObject, hSender, MID_PICKEDUP);
				pServerDE->WriteToMessageFloat(hWrite, -1.0f);
				pServerDE->EndMessage(hWrite);


				// Send the appropriate message to the client...

				HCLASS hClass = pServerDE->GetObjectClass(m_hObject);
				if (pServerDE->IsKindOf(hClass, pServerDE->GetClass("CPlayerObj")))
				{
					CPlayerObj* pPlayer = (CPlayerObj*)pServerDE->HandleToObject(m_hObject);
					if (!pPlayer) return DFALSE;

					HCLIENT hClient = pPlayer->GetClient();
					HMESSAGEWRITE hMessage = pServerDE->StartMessage(hClient, MID_PLAYER_INFOCHANGE);
					pServerDE->WriteToMessageByte(hMessage, IC_WEAPON_PICKUP_ID);
					pServerDE->WriteToMessageByte(hMessage, nWeapon);
					pServerDE->WriteToMessageFloat(hMessage, (DFLOAT)m_weapons[nWeapon]->GetAmmoCount());
					pServerDE->EndMessage(hMessage);

					
					// If there isn't currently a weapon selected, or we're
					// on our melee weapon, Select the new weapon..

					if (m_nCurWeapon < 0 || 
					   (GetCommandId(m_nCurWeapon) < GetCommandId(nWeapon) && !bHadIt) ||
						GetWeaponType((RiotWeaponId)m_nCurWeapon) == MELEE)
					{
						pPlayer->ChangeWeapon(GetCommandId(nWeapon), DTRUE);
					}
				}

			}

		}
		break;

		// new ev - 18/1/99 adding ammo powerup support
		case MID_AMMO_PICKEDUP:
		{	
			CServerDE* pServerDE = BaseClass::GetServerDE();
			if (!pServerDE) return DFALSE;

			DBOOL	bProxMine = DFALSE;
			DBOOL   bPickedUp = DFALSE;
			DBYTE	nWeapon   = pServerDE->ReadFromMessageByte(hRead);
			DFLOAT	fAmmo	  = pServerDE->ReadFromMessageFloat(hRead);

			if (IsValidIndex(nWeapon))
			{
				if (m_weapons[nWeapon])
				{
					if( m_weapons[nWeapon]->GetAmmoCount() < m_weapons[nWeapon]->GetMaxAmmo())
					{
						bPickedUp = DTRUE;
						m_weapons[nWeapon]->AddAmmo((int)fAmmo);
						// changed to allow addition of of the pv_weapon
						// when the player picks up the proximity mine
						if (nWeapon != GUN_AMBEDPROXIMITYMINE_ID)
						{
							m_weapons[nWeapon]->AddAmmo((int)fAmmo);
						}
						else
						{
							// if the player doesn't have the weapon, give them
							// the proximity mine
							if (!m_weapons[nWeapon]->Have())
							{
								ObtainWeapon( nWeapon, (int)fAmmo + m_weapons[nWeapon]->GetAmmoCount() );
								bProxMine = DTRUE;
							}
							// otherwise add some ammo
							else
							{
								m_weapons[nWeapon]->AddAmmo((int)fAmmo);
							}
						}
					}
				}
				else // this should not happen because the m_weapons should have been initialised
				{
					bPickedUp = DTRUE;
					m_tempAmmo[nWeapon] += (int)fAmmo;
				}
			}

			if (bPickedUp)
			{
				// Tell powerup it was picked up...

				HMESSAGEWRITE hWrite = pServerDE->StartMessageToObject(pObject, hSender, MID_PICKEDUP);
				pServerDE->WriteToMessageFloat(hWrite, -1.0f);
				pServerDE->EndMessage(hWrite);


				// Send the appropriate message to the client...

				HCLASS hClass = pServerDE->GetObjectClass(m_hObject);
				if (pServerDE->IsKindOf(hClass, pServerDE->GetClass("CPlayerObj")))
				{
					CPlayerObj* pPlayer = (CPlayerObj*)pServerDE->HandleToObject(m_hObject);
					if (!pPlayer) return DFALSE;

					// start a message to the appropriate client
					HCLIENT hClient = pPlayer->GetClient();
					HMESSAGEWRITE hMessage = pServerDE->StartMessage(hClient, MID_PLAYER_INFOCHANGE);

					if (!bProxMine)
					{
						// normally we just tell them the amount of ammo picked up
						pServerDE->WriteToMessageByte(hMessage, IC_AMMO_PICKUP_ID);
						pServerDE->WriteToMessageByte(hMessage, nWeapon);
						pServerDE->WriteToMessageFloat(hMessage, (DFLOAT)m_weapons[nWeapon]->GetAmmoCount());
					}
					else
					{
						// special case for the proximity mine, they get the weapon as well
						// as the ammo
						pServerDE->WriteToMessageByte(hMessage, IC_WEAPON_PICKUP_ID);
						pServerDE->WriteToMessageByte(hMessage, nWeapon);
						pServerDE->WriteToMessageFloat(hMessage, (DFLOAT)m_weapons[nWeapon]->GetAmmoCount());
					}

					pServerDE->EndMessage(hMessage);
					
					// If there isn't currently a weapon selected, or we're
					// on our melee weapon, Select the new weapon..

					//if (m_nCurWeapon < 0 || 
					//   (GetCommandId(m_nCurWeapon) < GetCommandId(nWeapon) && !bHadIt) ||
					//	GetWeaponType((RiotWeaponId)m_nCurWeapon) == MELEE)
					//{
					//	pPlayer->ChangeWeapon(GetCommandId(nWeapon), DTRUE);
					//}
				}
			}
		}
		break;

		default : break;
	}

	return Aggregate::ObjectMessageFn(pObject, hSender, messageID, hRead);
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CWeapons::ObtainWeapon()
//
//	PURPOSE:	Mark a specific weapon as aquired
//
// @parm the index of the weapon
//
// ----------------------------------------------------------------------- //

void CWeapons::ObtainWeapon(DBYTE nWeaponId, int nDefaultAmmo,
							DBOOL bNotifyClient)
{
	if (!IsValidIndex(nWeaponId)) return;

	// Create the weapon if need be...

	if (m_eArsenal == AT_AS_NEEDED)
	{
		if (!m_weapons[nWeaponId]) 
		{
			CreateWeapon(nWeaponId);
		}
	}


	// Make sure we actually have the thing...

	if (!m_weapons[nWeaponId]) return;
	

	// Give us the gun!

	m_weapons[nWeaponId]->Aquire();

	
	// Set the weapon's default ammo if appropriate...

	if (nDefaultAmmo >= 0)
	{
		m_weapons[nWeaponId]->SetAmmo(nDefaultAmmo);
	}

	
	// Notify the client if this is a player's weapon, and the flag
	// is set...

	if (bNotifyClient)
	{
		CServerDE* pServerDE = BaseClass::GetServerDE();
		if (!pServerDE) return;

		// Send the appropriate message to the client...

		HCLASS hClass = pServerDE->GetObjectClass(m_hObject);
		if (pServerDE->IsKindOf(hClass, pServerDE->GetClass("CPlayerObj")))
		{
			CPlayerObj* pPlayer = (CPlayerObj*)pServerDE->HandleToObject(m_hObject);
			if (!pPlayer) return;

			HCLIENT hClient = pPlayer->GetClient();
			HMESSAGEWRITE hMessage = pServerDE->StartMessage(hClient, MID_PLAYER_INFOCHANGE);
			pServerDE->WriteToMessageByte(hMessage, IC_WEAPON_ID);
			pServerDE->WriteToMessageByte(hMessage, nWeaponId);
			pServerDE->WriteToMessageFloat(hMessage, (DFLOAT)m_weapons[nWeaponId]->GetAmmoCount());
			pServerDE->EndMessage(hMessage);

			
			// If there isn't currently a weapon selected, or we're
			// on our melee weapon, Select the new weapon..

			if (m_nCurWeapon < 0 || GetWeaponType((RiotWeaponId)m_nCurWeapon) == MELEE)
			{
				pPlayer->ChangeWeapon(GetCommandId(nWeaponId));
			}
		}
	}
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CWeapons::GetWeaponModel()
//
//	PURPOSE:	Get the weapon model object associated with the weapon
//
// @parm the index of the weapon
//
// ----------------------------------------------------------------------- //

HOBJECT CWeapons::GetModelObject(DBYTE nWeaponId)
{
	if (IsValidIndex(nWeaponId))
	{
		if (m_weapons[nWeaponId]) return m_weapons[nWeaponId]->GetModelObject();
	}

	return DNULL;
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CWeapons::ChangeWeapon()
//
//	PURPOSE:	Change to a new weapon
//
// @parm the weapon to go to
// @rdef was the switch allowed?
//
// ----------------------------------------------------------------------- //

DBOOL CWeapons::ChangeWeapon(DBYTE nNewWeapon)
{
	if (nNewWeapon == m_nCurWeapon) return DTRUE;

//	if (!IsValidWeapon(nNewWeapon)) return DFALSE;

	CServerDE* pServerDE = BaseClass::GetServerDE();
	if (!pServerDE) return DFALSE;

	if (!IsValidWeapon(nNewWeapon)) return DFALSE;

	// Set this as our current weapon...

	m_nCurWeapon = nNewWeapon;

	// Let the base character know that a weapon change occured...

	HCLASS hClass = pServerDE->GetObjectClass(m_hObject);
	if (pServerDE->IsKindOf(hClass, pServerDE->GetClass("CBaseCharacter")))
	{
		CBaseCharacter* pChar = (CBaseCharacter*)pServerDE->HandleToObject(m_hObject);
		if (pChar) 
		{
			pChar->HandleWeaponChange();
		}
	}

	// Select the weapon...

	m_weapons[m_nCurWeapon]->Select();

	return DTRUE;
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CWeapons::DeselectCurWeapon()
//
//	PURPOSE:	Deselect the current weapon
//
// ----------------------------------------------------------------------- //

void CWeapons::DeselectCurWeapon()
{
	if (IsValidWeapon(m_nCurWeapon))
	{
		m_weapons[m_nCurWeapon]->Deselect();
	}

	m_nCurWeapon = -1;
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CWeapons::AddAmmo()
//
//	PURPOSE:	Add some ammo to a specific weapon
//
// @parm which weapon
// @parm how much ammo
//
// ----------------------------------------------------------------------- //

void CWeapons::AddAmmo(DBYTE nWeapon, int nAmmo)
{
	if (!IsValidWeapon(nWeapon)) return;

	m_weapons[nWeapon]->AddAmmo(nAmmo);
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CWeapons::GetAmmoCount()
//
//	PURPOSE:	Return the ammo amount of the currently selected weapon
//
// @rdef the ammo count
//
// ----------------------------------------------------------------------- //

int CWeapons::GetAmmoCount()
{
	if (IsValidIndex(m_nCurWeapon) && m_weapons[m_nCurWeapon])
	{
		return m_weapons[m_nCurWeapon]->GetAmmoCount();
	}

	return 0;
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CWeapons::GetAmmoCount()
//
//	PURPOSE:	Return the ammo amount of the weapon
//
// @rdef the ammo count
//
// ----------------------------------------------------------------------- //

int CWeapons::GetAmmoCount(DBYTE nWeaponID)
{
	if (IsValidIndex(nWeaponID) && m_weapons[nWeaponID])
	{
		return m_weapons[nWeaponID]->GetAmmoCount();
	}

	return 0;
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CWeapons::Save
//
//	PURPOSE:	Save the object
//
// ----------------------------------------------------------------------- //

void CWeapons::Save(HMESSAGEWRITE hWrite, DBYTE nType)
{
	CServerDE* pServerDE = BaseClass::GetServerDE();
	if (!pServerDE || !hWrite) return;

	pServerDE->WriteToLoadSaveMessageObject(hWrite, m_hObject);
	pServerDE->WriteToMessageFloat(hWrite, (DFLOAT)m_nCurWeapon);
	pServerDE->WriteToMessageByte(hWrite, m_eModelSize);
	pServerDE->WriteToMessageByte(hWrite, m_eArsenal);

	for (int i=GUN_FIRST_ID; i < GUN_MAX_NUMBER; i++)
	{
		if (m_weapons[i]) 
		{
			pServerDE->WriteToMessageByte(hWrite, DTRUE);
			m_weapons[i]->Save(hWrite, nType);
		}
		else
		{
			pServerDE->WriteToMessageByte(hWrite, DFALSE);
		}
	}
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CWeapons::Load
//
//	PURPOSE:	Load the object
//
// ----------------------------------------------------------------------- //

void CWeapons::Load(HMESSAGEREAD hRead, DBYTE nType)
{
	CServerDE* pServerDE = BaseClass::GetServerDE();
	if (!pServerDE || !hRead) return;

	pServerDE->ReadFromLoadSaveMessageObject(hRead, &m_hObject);
	m_nCurWeapon	 = (int)pServerDE->ReadFromMessageFloat(hRead);
	m_eModelSize	 = (ModelSize)pServerDE->ReadFromMessageByte(hRead);
	m_eArsenal		 = (ArsenalType)pServerDE->ReadFromMessageByte(hRead);

	for (int i=GUN_FIRST_ID; i < GUN_MAX_NUMBER; i++)
	{
		DBOOL bLoad = pServerDE->ReadFromMessageByte(hRead);

		if (bLoad)
		{
			CreateWeapon(i);

			if (m_weapons[i]) 
			{
				m_weapons[i]->Load(hRead, nType);
			}
		}
	}
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CWeapons::Reset()
//
//	PURPOSE:	Reset all the weapons (i.e., we don't have any of them)
//
// ----------------------------------------------------------------------- //

void CWeapons::Reset()
{
	for (int i=0; i < GUN_MAX_NUMBER; i++)
	{
		if (m_weapons[i] && GetWeaponType((RiotWeaponId)i) != MELEE)
		{
			m_weapons[i]->SetAmmo(0);
			m_weapons[i]->Drop();
		}
	}
}
