// ----------------------------------------------------------------------- //
//
// MODULE  : LOFAmmoPowerups.cpp
//
// PURPOSE : LOF ammo powerups / pickup items implementation
//
// CREATED : 13/1/99
//
// ----------------------------------------------------------------------- //// ----------------------------------------------------------------------- //

#include "LOFAmmoPowerups.h"
#include "RiotMsgIds.h"
#include "cpp_server_de.h"
#include "WeaponDefs.h"
#include "BaseCharacter.h"
#include "LOFWeaponDefs.h"

#define UPDATE_DELTA	0.1f

// ----------------------------------------------------------------------- //
// ----------------------------------------------------------------------- //
//
//	CLASS:		AmmoPowerup
//
//	PURPOSE:	Ammo pickup items / powerups
//
// ----------------------------------------------------------------------- //
// ----------------------------------------------------------------------- //

BEGIN_CLASS(AmmoPowerup)
	ADD_BOOLPROP_FLAG(Rotate, 0, PF_HIDDEN)
	ADD_REALPROP(RespawnTime, 30.0f)
	ADD_STRINGPROP_FLAG(SoundFile, "", PF_HIDDEN)
	ADD_LONGINTPROP( WeaponType, 0 )
	ADD_LONGINTPROP( Ammo, 0 )
	ADD_BOOLPROP( Small, 0 )
	ADD_BOOLPROP( Large, 0 )
	ADD_VECTORPROP_VAL_FLAG( Dims, 10.0f, 25.0f, 10.0f, PF_HIDDEN | PF_DIMS )
END_CLASS_DEFAULT(AmmoPowerup, Powerup, NULL, NULL)


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	AmmoPowerup::AmmoPowerup
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

AmmoPowerup::AmmoPowerup() : Powerup()
{ 
	m_dwAmmo		= 0; 
	m_iWeaponType	= 0;
	m_eModelSize	= MS_NORMAL;

	m_bInformClient	= DFALSE;
	m_bBounce = DFALSE;

//	AddAggregate(&m_bounce);
}



// ----------------------------------------------------------------------- //
//
//	ROUTINE:	AmmoPowerup::EngineMessageFn
//
//	PURPOSE:	Handle engine messages
//
// ----------------------------------------------------------------------- //

DDWORD AmmoPowerup::EngineMessageFn(DDWORD messageID, void *pData, DFLOAT fData)
{
	CServerDE* pServerDE = GetServerDE();
	if (!pServerDE) return DFALSE;

	switch(messageID)
	{
		case MID_UPDATE:
		{
			DDWORD dwRet = Powerup::EngineMessageFn(messageID, pData, fData);
			return dwRet;
		}
		break;

		case MID_PRECREATE:
		{
			DDWORD dwRet = Powerup::EngineMessageFn(messageID, pData, fData);

			if ( fData == 1.0f || fData == 2.0f )
				ReadProp(( ObjectCreateStruct * )pData );

			PostPropRead(( ObjectCreateStruct * )pData );

			return dwRet;
		}
		break;

		case MID_INITIALUPDATE:
		{
			DDWORD dwRet = Powerup::EngineMessageFn(messageID, pData, fData);

			if (fData != INITIALUPDATE_SAVEGAME)
			{
				InitialUpdate(pServerDE);
			}

			return dwRet;
		}

		case MID_SAVEOBJECT:
		{
			Save((HMESSAGEWRITE)pData, (DDWORD)fData);
		}
		break;

		case MID_LOADOBJECT:
		{
			Load((HMESSAGEREAD)pData, (DDWORD)fData);
		}
		break;

		default : break;
	}

	return Powerup::EngineMessageFn(messageID, pData, fData);
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	AmmoPowerup::ReadProp
//
//	PURPOSE:	Read object properties
//
// ----------------------------------------------------------------------- //
void AmmoPowerup::ReadProp( ObjectCreateStruct *pStruct )
{
	GenericProp genProp;

	if( g_pServerDE->GetPropGeneric( "Ammo", &genProp ) == DE_OK )
		m_dwAmmo = ( DDWORD )genProp.m_Long;
	if( g_pServerDE->GetPropGeneric( "WeaponType", &genProp ) == DE_OK)
		m_iWeaponType = ( DBYTE )genProp.m_Long;
	if( g_pServerDE->GetPropGeneric( "Small", &genProp ) == DE_OK)
		m_eModelSize = genProp.m_Bool ? MS_SMALL : m_eModelSize;
	if( g_pServerDE->GetPropGeneric( "Large", &genProp ) == DE_OK)
		m_eModelSize = genProp.m_Bool ? MS_LARGE : m_eModelSize;
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	AmmoPowerup::PostPropRead
//
//	PURPOSE:	Handle post property read engine messages
//
// ----------------------------------------------------------------------- //
void AmmoPowerup::PostPropRead( ObjectCreateStruct *pStruct )
{
	if (pStruct)
	{
		switch( m_iWeaponType )
		{
			case GUN_AMBEDKATOBURST_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Ammo\\KatoBurst_Ammo.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Ammo\\KatoBurst_Ammo_a.dtx");
				break;
			}
			case GUN_SPIDER_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\PV_Weapons\\SpiderProjectile.abc");
				SAFE_STRCPY(pStruct->m_Filename, "Skins\\Weapons\\SpiderProjectile_a.dtx");
				//SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Ammo\\Spider_Ammo_a.dtx");
				//SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Ammo\\Spider_Ammo_a.dtx");
				break;
			}
			case GUN_AMBEDMORT_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Ammo\\MORT_Ammo.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Ammo\\MORT_Ammo_a.dtx");
				break;
			}
			case GUN_SNIPERRIFLE_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Ammo\\SniperRifle_Ammo.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Ammo\\SniperRifle_Ammo_a.dtx");
				break;
			}
			case GUN_AMBEDSHOCHI_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Ammo\\Shochi_Ammo.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Ammo\\Shochi_Ammo_a.dtx");
				break;
			}
			case GUN_SHREDDER_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Ammo\\Shredder_Ammo.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Ammo\\Shredder_Ammo_a.dtx");
				break;
			}
			case GUN_AMBEDHWAR_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Ammo\\HWAR_Ammo.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Ammo\\HWAR_Ammo_a.dtx");
				break;
			}

			case GUN_AMBEDEMS_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Ammo\\EMS_Ammo.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Ammo\\EMS_Ammo_a.dtx");
				break;
			}
			case GUN_AMBEDAGUMA_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Ammo\\Aguma_Ammo.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Ammo\\Aguma_Ammo_a.dtx");
				break;
			}
			case GUN_ASSAULTRIFLE_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Ammo\\AssaultRifle_Ammo.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Ammo\\AssaultRifle_Ammo_a.dtx");
				break;
			}
			case GUN_ENERGYGRENADE_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Ammo\\EnergyGrenade.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Ammo\\EnergyGrenade_a.dtx");
				break;
			}
			case GUN_AMBEDPROXIMITYMINE_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Ammo\\Mine_Ammo.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\Weapons\\AmbedProximityProjectile_a.dtx");
				break;
			}
			case GUN_AMBEDRASG_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Ammo\\RASG_Ammo.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Ammo\\RASG_Ammo_a.dtx");
				break;
			}
			case GUN_TOW_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\PV_Weapons\\Bullgut_Projectile.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\Weapons\\Bullgut_Projectile_a.dtx");
				//SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Ammo\\TOW.abc");
				//SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Ammo\\TOW_a.dtx");
				break;
			}
			case GUN_LASERCANNON_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Ammo\\Laser_ammo.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Ammo\\Laser_ammo_a.dtx");
				break;
			}
			case GUN_AMBEDMINIGUN_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Ammo\\Minigun_Ammo.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Ammo\\Minigun_Ammo_a.dtx");
				break;
			}
			default:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Colt45.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Colt45_a.dtx");
				break;
			}
		}

		// Set up the appropriate pick up sound...

		if (m_hstrSoundFile) 
		{
			g_pServerDE->FreeString(m_hstrSoundFile);
		}

		if (GUN_FIRSTMECH_ID <= m_iWeaponType && m_iWeaponType <= GUN_LASTMECH_ID)
		{
			m_hstrSoundFile = g_pServerDE->CreateString("Sounds\\Powerups\\Weapon_mca.wav");
		}
		else
		{
			m_hstrSoundFile = g_pServerDE->CreateString("Sounds\\Powerups\\Weapon_onfoot.wav");
		}

		m_bBounce = DFALSE;
		m_bRotate = DFALSE;
	}
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	AmmoPowerup::InitialUpdate
//
//	PURPOSE:	Handle inital update engine messages
//
// ----------------------------------------------------------------------- //

void AmmoPowerup::InitialUpdate( CServerDE *pServerDE )
{
	DVector vDims, vScale, vNewDims;

	pServerDE->GetModelAnimUserDims(m_hObject, &vDims, pServerDE->GetModelAnimation(m_hObject));

	vScale = GetAmmoScale((RiotWeaponId)m_iWeaponType, m_eModelSize);
	pServerDE->ScaleObject(m_hObject, &vScale);

	// Set object dims based on scale value...

	vNewDims.x = vScale.x * vDims.x;
	vNewDims.y = vScale.y * vDims.y;
	vNewDims.z = vScale.z * vDims.z;

	g_pServerDE->SetObjectDims(m_hObject, &vNewDims);

	pServerDE->SetNextUpdate(m_hObject, UPDATE_DELTA);
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	AmmoPowerup::ObjectTouch
//
//	PURPOSE:	Add weapon powerup to object
//
// ----------------------------------------------------------------------- //

void AmmoPowerup::ObjectTouch(HOBJECT hObject)
{
	CServerDE* pServerDE = GetServerDE();
	if (!pServerDE || !hObject) return;

	// If we hit non-character objects, just ignore them...

	HCLASS hCharClass = pServerDE->GetClass("CBaseCharacter");
	HCLASS hObjClass  = pServerDE->GetObjectClass(hObject);


	if (pServerDE->IsKindOf(hObjClass, hCharClass))
	{
		CBaseCharacter* pCharObj = (CBaseCharacter*)pServerDE->HandleToObject(hObject);

		if (pCharObj && !pCharObj->IsDead() && pCharObj->CanCarryWeapon(m_iWeaponType))
		{
			HMESSAGEWRITE hMessage = pServerDE->StartMessageToObject(this, hObject, MID_AMMO_PICKEDUP);
			pServerDE->WriteToMessageByte( hMessage, m_iWeaponType );
			pServerDE->WriteToMessageFloat( hMessage, ( float )m_dwAmmo );
			pServerDE->EndMessage(hMessage);
		}
	}
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	AmmoPowerup::Save
//
//	PURPOSE:	Save the object
//
// ----------------------------------------------------------------------- //

void AmmoPowerup::Save(HMESSAGEWRITE hWrite, DDWORD dwSaveFlags)
{
	CServerDE* pServerDE = GetServerDE();
	if (!pServerDE || !hWrite) return;

	pServerDE->WriteToMessageByte(hWrite, m_iWeaponType);
	pServerDE->WriteToMessageByte(hWrite, m_eModelSize);
	pServerDE->WriteToMessageDWord(hWrite, m_dwAmmo);
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	AmmoPowerup::Load
//
//	PURPOSE:	Load the object
//
// ----------------------------------------------------------------------- //

void AmmoPowerup::Load(HMESSAGEREAD hRead, DDWORD dwLoadFlags)
{
	CServerDE* pServerDE = GetServerDE();
	if (!pServerDE || !hRead) return;

	m_iWeaponType	= pServerDE->ReadFromMessageByte(hRead);
	m_eModelSize	= (ModelSize) pServerDE->ReadFromMessageByte(hRead);
	m_dwAmmo		= pServerDE->ReadFromMessageDWord(hRead);

	// Set our object to the correct filenames...

	ObjectCreateStruct theStruct;
	INIT_OBJECTCREATESTRUCT(theStruct)
	PostPropRead(&theStruct);

	pServerDE->SetModelFilenames(m_hObject, theStruct.m_Filename, 
								 theStruct.m_SkinName);
}



BEGIN_CLASS(KatoBurstAmmo)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_AMBEDKATOBURST_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetAmmoCount(GUN_AMBEDKATOBURST_ID))
END_CLASS_DEFAULT(KatoBurstAmmo, AmmoPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	KatoBurstAmmo::KatoBurstAmmo
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

KatoBurstAmmo::KatoBurstAmmo() : AmmoPowerup()
{ 
	m_dwAmmo		= GetAmmoCount(GUN_AMBEDKATOBURST_ID); 
	m_iWeaponType	= GUN_AMBEDKATOBURST_ID;
}

BEGIN_CLASS(SpiderAmmo)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_SPIDER_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetAmmoCount(GUN_SPIDER_ID))
END_CLASS_DEFAULT(SpiderAmmo, AmmoPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	SpiderAmmo::SpiderAmmo
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

SpiderAmmo::SpiderAmmo() : AmmoPowerup()
{ 
	m_dwAmmo		= GetAmmoCount(GUN_SPIDER_ID); 
	m_iWeaponType	= GUN_SPIDER_ID;
}

BEGIN_CLASS(AmbedMORTAmmo)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_AMBEDMORT_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetAmmoCount(GUN_AMBEDMORT_ID))
END_CLASS_DEFAULT(AmbedMORTAmmo, AmmoPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	AmbedMORTAmmo::AmbedMORTAmmo
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

AmbedMORTAmmo::AmbedMORTAmmo() : AmmoPowerup()
{ 
	m_dwAmmo		= GetAmmoCount(GUN_AMBEDMORT_ID); 
	m_iWeaponType	= GUN_AMBEDMORT_ID;
}

BEGIN_CLASS(SniperRifleAmmo)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_SNIPERRIFLE_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetAmmoCount(GUN_SNIPERRIFLE_ID))
END_CLASS_DEFAULT(SniperRifleAmmo, AmmoPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	SniperRifleAmmo::SniperRifleAmmo
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

SniperRifleAmmo::SniperRifleAmmo() : AmmoPowerup()
{ 
	m_dwAmmo		= GetAmmoCount(GUN_SNIPERRIFLE_ID); 
	m_iWeaponType	= GUN_SNIPERRIFLE_ID;
}
/*
BEGIN_CLASS(AmbedDSGAmmo)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_AMBEDDSG_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetAmmoCount(GUN_AMBEDDSG_ID))
END_CLASS_DEFAULT(AmbedDSGAmmo, AmmoPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	AmbedDSGAmmo::AmbedDSGAmmo
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

AmbedDSGAmmo::AmbedDSGAmmo() : AmmoPowerup()
{ 
	m_dwAmmo		= GetAmmoCount(GUN_AMBEDDSG_ID); 
	m_iWeaponType	= GUN_AMBEDDSG_ID;
}
*/
BEGIN_CLASS(AmbedShochiAmmo)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_AMBEDSHOCHI_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetAmmoCount(GUN_AMBEDSHOCHI_ID))
END_CLASS_DEFAULT(AmbedShochiAmmo, AmmoPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	AmbedShochiAmmo::AmbedShochiAmmo
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

AmbedShochiAmmo::AmbedShochiAmmo() : AmmoPowerup()
{ 
	m_dwAmmo		= GetAmmoCount(GUN_AMBEDSHOCHI_ID); 
	m_iWeaponType	= GUN_AMBEDSHOCHI_ID;
}

BEGIN_CLASS(ShredderAmmo)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_SHREDDER_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetAmmoCount(GUN_SHREDDER_ID))
END_CLASS_DEFAULT(ShredderAmmo, AmmoPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	ShredderAmmo::ShredderAmmo
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

ShredderAmmo::ShredderAmmo() : AmmoPowerup()
{ 
	m_dwAmmo		= GetAmmoCount(GUN_SHREDDER_ID); 
	m_iWeaponType	= GUN_SHREDDER_ID;
}

BEGIN_CLASS(AmbedHWARAmmo)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_AMBEDHWAR_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetAmmoCount(GUN_AMBEDHWAR_ID))
END_CLASS_DEFAULT(AmbedHWARAmmo, AmmoPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	AmbedHWARAmmo::AmbedHWARAmmo
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

AmbedHWARAmmo::AmbedHWARAmmo() : AmmoPowerup()
{ 
	m_dwAmmo		= GetAmmoCount(GUN_AMBEDHWAR_ID); 
	m_iWeaponType	= GUN_AMBEDHWAR_ID;
}

BEGIN_CLASS(AmbedEMSAmmo)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_AMBEDEMS_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetAmmoCount(GUN_AMBEDEMS_ID))
END_CLASS_DEFAULT(AmbedEMSAmmo, AmmoPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	AmbedEMSAmmo::AmbedEMSAmmo
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

AmbedEMSAmmo::AmbedEMSAmmo() : AmmoPowerup()
{ 
	m_dwAmmo		= GetAmmoCount(GUN_AMBEDEMS_ID); 
	m_iWeaponType	= GUN_AMBEDEMS_ID;
}

BEGIN_CLASS(AmbedAgumaAmmo)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_AMBEDAGUMA_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetAmmoCount(GUN_AMBEDAGUMA_ID))
END_CLASS_DEFAULT(AmbedAgumaAmmo, AmmoPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	AmbedAgumaAmmo::AmbedAgumaAmmo
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

AmbedAgumaAmmo::AmbedAgumaAmmo() : AmmoPowerup()
{ 
	m_dwAmmo		= GetAmmoCount(GUN_AMBEDAGUMA_ID); 
	m_iWeaponType	= GUN_AMBEDAGUMA_ID;
}

BEGIN_CLASS(AssaultRifleAmmo)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_ASSAULTRIFLE_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetAmmoCount(GUN_ASSAULTRIFLE_ID))
END_CLASS_DEFAULT(AssaultRifleAmmo, AmmoPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	AssaultRifleAmmo::AssaultRifleAmmo
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

AssaultRifleAmmo::AssaultRifleAmmo() : AmmoPowerup()
{ 
	m_dwAmmo		= GetAmmoCount(GUN_ASSAULTRIFLE_ID); 
	m_iWeaponType	= GUN_ASSAULTRIFLE_ID;
}

BEGIN_CLASS(EnergyGrenadeAmmo)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_ENERGYGRENADE_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetAmmoCount(GUN_ENERGYGRENADE_ID))
END_CLASS_DEFAULT(EnergyGrenadeAmmo, AmmoPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	EnergyGrenadeAmmo::EnergyGrenadeAmmo
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

EnergyGrenadeAmmo::EnergyGrenadeAmmo() : AmmoPowerup()
{ 
	m_dwAmmo		= GetAmmoCount(GUN_ENERGYGRENADE_ID); 
	m_iWeaponType	= GUN_ENERGYGRENADE_ID;
}

BEGIN_CLASS(AmbedProximityMineAmmo)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_AMBEDPROXIMITYMINE_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetAmmoCount(GUN_AMBEDPROXIMITYMINE_ID))
END_CLASS_DEFAULT(AmbedProximityMineAmmo, AmmoPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	AmbedProximityMineAmmo::AmbedProximityMineAmmo
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

AmbedProximityMineAmmo::AmbedProximityMineAmmo() : AmmoPowerup()
{ 
	m_dwAmmo		= GetAmmoCount(GUN_AMBEDPROXIMITYMINE_ID); 
	m_iWeaponType	= GUN_AMBEDPROXIMITYMINE_ID;
}

BEGIN_CLASS(AmbedRASGAmmo)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_AMBEDRASG_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetAmmoCount(GUN_AMBEDRASG_ID))
END_CLASS_DEFAULT(AmbedRASGAmmo, AmmoPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	AmbedRASGAmmo::AmbedRASGAmmo
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

AmbedRASGAmmo::AmbedRASGAmmo() : AmmoPowerup()
{ 
	m_dwAmmo		= GetAmmoCount(GUN_AMBEDRASG_ID); 
	m_iWeaponType	= GUN_AMBEDRASG_ID;
}

BEGIN_CLASS(AmbedMinigunAmmo)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_AMBEDMINIGUN_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetAmmoCount(GUN_AMBEDMINIGUN_ID))
END_CLASS_DEFAULT(AmbedMinigunAmmo, AmmoPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	AmbedMinigunAmmo::AmbedMinigunAmmo
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

AmbedMinigunAmmo::AmbedMinigunAmmo() : AmmoPowerup()
{ 
	m_dwAmmo		= GetAmmoCount(GUN_AMBEDMINIGUN_ID); 
	m_iWeaponType	= GUN_AMBEDMINIGUN_ID;
}

BEGIN_CLASS(TOWAmmo)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_TOW_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetAmmoCount(GUN_TOW_ID))
END_CLASS_DEFAULT(TOWAmmo, AmmoPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	TOWAmmo::TOWAmmo
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

TOWAmmo::TOWAmmo() : AmmoPowerup()
{ 
	m_dwAmmo		= GetAmmoCount(GUN_TOW_ID); 
	m_iWeaponType	= GUN_TOW_ID;
}

BEGIN_CLASS(LaserCannonAmmo)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_LASERCANNON_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetAmmoCount(GUN_LASERCANNON_ID))
END_CLASS_DEFAULT(LaserCannonAmmo, AmmoPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	LaserCannonAmmo::LaserCannonAmmo
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

LaserCannonAmmo::LaserCannonAmmo() : AmmoPowerup()
{ 
	m_dwAmmo		= GetAmmoCount(GUN_LASERCANNON_ID); 
	m_iWeaponType	= GUN_LASERCANNON_ID;
}
