// ----------------------------------------------------------------------- //
//
// MODULE  : WeaponDefs.h
//
// PURPOSE : Definitions for weapon types
//
// CREATED : 9/35/97
//
// ----------------------------------------------------------------------- //

#ifndef __WEAPON_DEFS_H__
#define __WEAPON_DEFS_H__

#include "clientshell_de.h"
#include "RiotCommandIds.h"
#include "PlayerModeTypes.h"
#include "WeaponFXTypes.h"
#include "ProjectileFXTypes.h"
#include "basedefs_de.h"
#include "ModelFuncs.h"

#include <stdio.h>

#define WEAPON_MIN_IDLE_TIME	5.0f
#define WEAPON_MAX_IDLE_TIME	15.0f
#define WEAPON_KEY_FIRE			"FIRE_KEY"
#define WEAPON_KEY_SOUND		"SOUND_KEY"
#define WEAPON_SOUND_RADIUS		2000.0f
#define WEAPON_SOUND_DRYFIRE	1
#define WEAPON_SOUND_KEY		2
#define WEAPON_SOUND_FIRE		3


extern char s_FileBuffer[_MAX_PATH];

// This is defined in both "RiotObjectUtilities.h" and "ClientUtilities.h"
int GetRandom(int lo, int hi);


enum WeaponState { W_IDLE, W_BEGIN_FIRING, W_FIRING, W_FIRED, W_FIRED2, 
				   W_END_FIRING, W_RELOADING, W_FIRING_NOAMMO, W_SELECT, 
				   W_DESELECT };

enum ProjectileType { VECTOR=0, PROJECTILE, CANNON, MELEE };

// Defines....

enum RiotWeaponId {

	GUN_FIRST_ID = 0,
	GUN_FIRSTMECH_ID = 0,
	GUN_PULSERIFLE_ID = 0,
	GUN_AUTOCANNONS_ID = 1,
	GUN_DEATHCHARGE_ID = 2,
	GUN_PULSELAUNCHER_ID = 3,
	GUN_SNIPERRIFLE_ID = 4,
	GUN_MASSDRIVER_ID = 5,
	GUN_RIOTCANNON_ID = 6,
	GUN_UNUSED1_ID = 7,		// Left in to decrease impact on level designers 
	GUN_LITHKILLER_ID = 8,
	GUN_BUZZSAW_ID = 9,   // DTM BUZZSAW MELEE WEAPON
	GUN_SHOULDERCANNONS_ID = 10,
	GUN_BIGBOOT_ID = 11,
	GUN_SNIPERCLUB_ID = 12,
	GUN_HEALGUN_ID = 13,
	GUN_MACHINEGUN_ID = 14,
	GUN_ROCKETLAUNCHER_ID = 15,
	GUN_SICKGUN_ID = 16,
	GUN_BLINDER_ID = 17,
	GUN_BOMBS_ID = 18,
	GUN_RAMSET_ID = 19,
	GUN_SHOTGUN_ID = 20,
	GUN_AXE_ID = 21,
	GUN_LASTMECH_ID = 21,

	GUN_FIRSTONFOOT_ID = 22,
	GUN_COLT45_ID = 22,
	GUN_ASSAULTRIFLE_ID = 23,
	GUN_ENERGYGRENADE_ID = 24,
	GUN_KATOGRENADE_ID = 25,
	GUN_MAC10_ID = 26,
	GUN_TOW_ID = 27,
	GUN_UNUSED2_ID = 28,	// Left in to decrease impact on level designers
	GUN_SQUEAKYTOY_ID = 29,
	GUN_LASTONFOOT_ID = 29,
	GUN_MAX_NUMBER = 30,

	GUN_NONE = 50
};


inline DBOOL FiredWeapon(WeaponState eState)
{
	if (eState == W_FIRED || eState == W_FIRED2) return DTRUE;

	return DFALSE;
}


inline DBOOL IsOneHandedWeapon(RiotWeaponId nWeaponId)
{
	DBOOL bRet = DFALSE;

	switch (nWeaponId)
	{
		case GUN_SQUEAKYTOY_ID :
		case GUN_BUZZSAW_ID : // DTM BUZZSAW MELEE WEAPON
		case GUN_AUTOCANNONS_ID :
		case GUN_SNIPERCLUB_ID :
		case GUN_RAMSET_ID :
		case GUN_AXE_ID :
			bRet = DTRUE;
		break;

		default : break;
	}

	return bRet;
}


inline ProjectileType GetWeaponType(RiotWeaponId nWeaponId)
{
	ProjectileType eType = VECTOR;

	switch (nWeaponId)
	{
		case GUN_SNIPERRIFLE_ID :
		case GUN_COLT45_ID :
		case GUN_SHOTGUN_ID	:
		case GUN_ASSAULTRIFLE_ID :
		case GUN_MAC10_ID :
		case GUN_SQUEAKYTOY_ID :
		case GUN_HEALGUN_ID :
		case GUN_RIOTCANNON_ID :
		case GUN_MACHINEGUN_ID :
		case GUN_SICKGUN_ID :
		case GUN_BLINDER_ID :
			eType = VECTOR;
		break;

		case GUN_PULSERIFLE_ID :
		case GUN_PULSELAUNCHER_ID :
		case GUN_DEATHCHARGE_ID :
		case GUN_ENERGYGRENADE_ID :
		case GUN_KATOGRENADE_ID :
		case GUN_TOW_ID	:
		case GUN_LITHKILLER_ID :
		case GUN_AUTOCANNONS_ID :
		case GUN_SHOULDERCANNONS_ID :
		case GUN_ROCKETLAUNCHER_ID :
		case GUN_BOMBS_ID :
			eType = PROJECTILE;
		break;
		
		case GUN_MASSDRIVER_ID :
			eType = CANNON;
		break;

		case GUN_BUZZSAW_ID : // DTM BUZZSAW MELEE WEAPON
		case GUN_SNIPERCLUB_ID :
		case GUN_RAMSET_ID :
		case GUN_AXE_ID :
			eType = MELEE;
		break;

		default : break;
	}

	return eType;
}

// Map command to weapon id.
inline int GetWeaponId(int nCommandId, DDWORD dwPlayerMode)
{
	int nWeaponId = -1;

	DBOOL bOnFoot = DFALSE;

	switch (nCommandId)
	{
		case COMMAND_ID_WEAPON_1 :
		{
			switch (dwPlayerMode)
			{
				case PM_MODE_MCA_HEAVYW :
					nWeaponId = GUN_RIOTCANNON_ID;
				break;

				case PM_MODE_MCA_ROCKETEER :
					nWeaponId = GUN_PULSERIFLE_ID;
				break;

				case PM_MODE_MCA_DEMOMAN :
					nWeaponId = GUN_PULSELAUNCHER_ID;
				break;

				case PM_MODE_MCA_SNIPER :
					nWeaponId = GUN_SNIPERRIFLE_ID;
				break;

				case PM_MODE_MCA_SOLDIER :
					nWeaponId = GUN_SHOTGUN_ID;
				break;

				case PM_MODE_MCA_MEDIC :
					nWeaponId = GUN_SHOTGUN_ID;
				break;
//DTM
				default : break;
			}
		}
		break;
	
		case COMMAND_ID_WEAPON_2 :
		{
			switch (dwPlayerMode)
			{
				case PM_MODE_MCA_HEAVYW :
					nWeaponId = GUN_MASSDRIVER_ID;
				break;

				case PM_MODE_MCA_ROCKETEER :
					nWeaponId = GUN_BOMBS_ID;
				break;

				case PM_MODE_MCA_DEMOMAN:
					nWeaponId = GUN_DEATHCHARGE_ID;
				break;

				case PM_MODE_MCA_SNIPER :
					nWeaponId = GUN_SNIPERRIFLE_ID;
				break;

				case PM_MODE_MCA_MEDIC :
					nWeaponId = GUN_HEALGUN_ID;
				break;

				case PM_MODE_MCA_SOLDIER :
					nWeaponId = GUN_MACHINEGUN_ID;
				break;

//DTM
				default : break;
			}
		}
		break;
		case COMMAND_ID_WEAPON_3 :
		{
			switch (dwPlayerMode)
			{
				case PM_MODE_MCA_HEAVYW:
					nWeaponId = GUN_SHOULDERCANNONS_ID;
				break;
			
				case PM_MODE_MCA_MEDIC:
					nWeaponId = GUN_SICKGUN_ID;
				break;

				case PM_MODE_MCA_DEMOMAN :
					nWeaponId = GUN_LITHKILLER_ID;
				break;

				case PM_MODE_MCA_ROCKETEER:
					nWeaponId = GUN_LITHKILLER_ID;
				break;

				case PM_MODE_MCA_SOLDIER:
					nWeaponId = GUN_ROCKETLAUNCHER_ID;
				break;

				default : break;
			}
		}
		break;
		case COMMAND_ID_WEAPON_10 :
		{
			switch (dwPlayerMode)
			{
				case PM_MODE_MCA_HEAVYW :
					nWeaponId = GUN_AUTOCANNONS_ID;
				break;

				case PM_MODE_MCA_ROCKETEER :
					nWeaponId = GUN_RAMSET_ID;
				break;

				case PM_MODE_MCA_DEMOMAN :
					nWeaponId = GUN_BUZZSAW_ID; // DTM BUZZSAW MELEE WEAPON
				break;

				case PM_MODE_MCA_SNIPER :
					nWeaponId = GUN_SNIPERCLUB_ID;
				break;

				case PM_MODE_MCA_MEDIC :
					nWeaponId = GUN_AXE_ID;
				break;

				case PM_MODE_MCA_SOLDIER :
					nWeaponId = GUN_AXE_ID;
				break;

				default : break;
			}
		}
		break;

		default : 
		break;
	}

	return nWeaponId;
}

// Map weapon id to command id.
inline int GetCommandId(int nWeaponId)
{
	int nCommandId = -1;

	switch (nWeaponId)
	{
		case GUN_PULSERIFLE_ID :
		case GUN_PULSELAUNCHER_ID :
		case GUN_SNIPERRIFLE_ID :
		case GUN_RIOTCANNON_ID :
		case GUN_SHOTGUN_ID :
			nCommandId = COMMAND_ID_WEAPON_1;
			break;

		case GUN_MASSDRIVER_ID :
		case GUN_MACHINEGUN_ID :
		case GUN_DEATHCHARGE_ID :
		case GUN_HEALGUN_ID :
		case GUN_BOMBS_ID :
			nCommandId = COMMAND_ID_WEAPON_2;
			break;

		case GUN_SHOULDERCANNONS_ID :
		case GUN_LITHKILLER_ID :
		case GUN_ROCKETLAUNCHER_ID :
		case GUN_SICKGUN_ID :
			nCommandId = COMMAND_ID_WEAPON_3;
			break;

		case GUN_BUZZSAW_ID : 
		case GUN_SNIPERCLUB_ID :
		case GUN_AUTOCANNONS_ID :
		case GUN_RAMSET_ID :
		case GUN_AXE_ID :
			nCommandId = COMMAND_ID_WEAPON_10;
			break;

		default : break;
	}

	return nCommandId;
}

// get the maximum ammo for each weapon
inline DDWORD GetWeaponMaxAmmo(int nWeaponId)
{
	DDWORD nRet = 0;

	switch (nWeaponId)
	{
		case GUN_PULSERIFLE_ID :
			nRet = 300;
		break;

		case GUN_DEATHCHARGE_ID :
			nRet = 3;
		break;

		case GUN_PULSELAUNCHER_ID :
			nRet = 200;
		break;

		case GUN_SNIPERRIFLE_ID :
			nRet = 400;
		break;

		case GUN_MASSDRIVER_ID :
			nRet = 50;
		break;

		case GUN_RIOTCANNON_ID :
			nRet = 300;
		break;

		case GUN_LITHKILLER_ID :
			nRet = 10;
		break;

		case GUN_HEALGUN_ID :
			nRet = 50;
		break;
		
		case GUN_BUZZSAW_ID : // DTM BUZZSAW MELEE WEAPON
			nRet = 100;
		break;
		
		case GUN_SHOULDERCANNONS_ID :
			nRet = 70;
		break;

		case GUN_MACHINEGUN_ID :
			nRet = 300;
		break;

		case GUN_ROCKETLAUNCHER_ID :
			nRet = 50;
		break;

		case GUN_SICKGUN_ID :
			nRet = 10;
		break;

		case GUN_BLINDER_ID :
			nRet = 10;
		break;

		case GUN_BOMBS_ID :
			nRet = 200;
		break;

		case GUN_SHOTGUN_ID :
			nRet = 50;
		break;
		
		case GUN_SNIPERCLUB_ID :
		case GUN_RAMSET_ID :
		case GUN_AXE_ID :
			nRet = 10000;
		break;

		
		// On-foot mode weapons...

		case GUN_COLT45_ID :
			nRet = 200;
		break;

		case GUN_ASSAULTRIFLE_ID :
			nRet = 400;
		break;

		case GUN_ENERGYGRENADE_ID :
			nRet = 50;
		break;

		case GUN_KATOGRENADE_ID :
			nRet = 50;
		break;

		case GUN_MAC10_ID :
			nRet = 400;
		break;

		case GUN_TOW_ID	:
			nRet = 25;
		break;

		case GUN_AUTOCANNONS_ID :
			nRet = 50;
		break;

		case GUN_SQUEAKYTOY_ID :
			nRet = 10000;
		break;

		

		default : break;
	}

	return nRet;
}


inline DDWORD GetSpawnedAmmo(RiotWeaponId nWeaponId)
{
	DDWORD dwAmmo = 1000;

	switch (nWeaponId)
	{
		case GUN_PULSERIFLE_ID :
			dwAmmo = 80;
		break;

		case GUN_DEATHCHARGE_ID :
			dwAmmo = 3;
		break;

		case GUN_PULSELAUNCHER_ID :
			dwAmmo = 200;
		break;

		case GUN_SNIPERRIFLE_ID :
			dwAmmo = 80;
		break;

		case GUN_MASSDRIVER_ID :
			dwAmmo = 50;
		break;

		case GUN_RIOTCANNON_ID :
			dwAmmo = 150;
		break;

		case GUN_LITHKILLER_ID :
			dwAmmo = 10;
		break;

		case GUN_HEALGUN_ID :
			dwAmmo = 50;
		break;
		
		case GUN_SHOULDERCANNONS_ID :
			dwAmmo = 70;
		break;
		
		case GUN_BUZZSAW_ID :
			dwAmmo = 50;
		break;		
		
		case GUN_MACHINEGUN_ID :
			dwAmmo = 200;
		break;

		case GUN_ROCKETLAUNCHER_ID :
			dwAmmo = 50;
		break;

		case GUN_SICKGUN_ID :
			dwAmmo = 3;
		break;

		case GUN_BLINDER_ID :
			dwAmmo = 10;
		break;

		case GUN_BOMBS_ID :
			dwAmmo = 200;
		break;

		case GUN_SHOTGUN_ID :
			dwAmmo = 50;
		break;

			
		// On-foot mode weapons...

		case GUN_COLT45_ID :
			dwAmmo = GetRandom(20,40);
		break;

		case GUN_ASSAULTRIFLE_ID :
			dwAmmo = GetRandom(40,80);
		break;

		case GUN_ENERGYGRENADE_ID :
			dwAmmo = GetRandom(5,10);
		break;

		case GUN_KATOGRENADE_ID :
			dwAmmo = GetRandom(5,10);
		break;

		case GUN_MAC10_ID :
			dwAmmo = GetRandom(40,80);
		break;

		case GUN_TOW_ID	:
			dwAmmo = GetRandom(1,5);
		break;

		case GUN_AUTOCANNONS_ID :
			dwAmmo = 50;
		break;

		default : break;
	}

	return dwAmmo;
}


inline DVector GetHandWeaponScale(RiotWeaponId nWeaponId, ModelSize eSize=MS_NORMAL)
{
	DVector vRet;
	VEC_SET(vRet, 1.0f, 1.0f, 1.0f)

	switch (nWeaponId)
	{
		// On-foot mode weapons...

		case GUN_COLT45_ID :
			VEC_SET(vRet, 6.0f, 6.0f, 6.0f);
		break;

		case GUN_SHOTGUN_ID	:
			VEC_SET(vRet, 2.0f, 2.0f, 2.0f);
		break;

		case GUN_MAC10_ID :
			VEC_SET(vRet, 15.0f, 15.0f, 15.0f);
		break;

		case GUN_ASSAULTRIFLE_ID :
		case GUN_ENERGYGRENADE_ID :
		case GUN_KATOGRENADE_ID :
			//VEC_SET(vRet, 1.0f, 1.0f, 1.0f);
			VEC_SET(vRet, 3.0f, 3.0f, 3.0f);
		break;

		case GUN_TOW_ID	:
			VEC_SET(vRet, 28.0f, 28.0f, 28.0f);
		break;

		case GUN_SQUEAKYTOY_ID :
			VEC_SET(vRet, 15.0f, 15.0f, 15.0f);
		break;

		case GUN_PULSERIFLE_ID :
			VEC_SET(vRet, 2.0f, 2.0f, 2.0f);
		break;

		case GUN_RIOTCANNON_ID	:
			VEC_SET(vRet, 2.5f, 2.5f, 2.5f); //DTM SHRED SCALE
		break;

		case GUN_MASSDRIVER_ID :
			VEC_SET(vRet, 2.0f, 2.0f, 2.0f); //DTM JUGG SCALE
		break;

		case GUN_MACHINEGUN_ID :
			VEC_SET(vRet, 2.0f, 2.0f, 2.0f); //DTM JUGG SCALE
		break;
		
		case GUN_ROCKETLAUNCHER_ID :
			VEC_SET(vRet, 2.0f, 2.0f, 2.0f); //DTM JUGG SCALE
		break;
		
		case GUN_PULSELAUNCHER_ID	:
			VEC_SET(vRet, 2.0f, 2.0f, 2.0f);
		break;

		case GUN_AUTOCANNONS_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		case GUN_HEALGUN_ID:
			VEC_SET(vRet, 2.0f, 2.0f, 2.0f);
		break;

		case GUN_DEATHCHARGE_ID :
			VEC_SET(vRet, 2.0f, 2.0f, 2.0f);
		break;

		case GUN_SNIPERRIFLE_ID :
			VEC_SET(vRet, 2.0f, 2.0f, 2.0f); //DTM RIFLE SCALE
		break;

		case GUN_SNIPERCLUB_ID :
			VEC_SET(vRet, 2.0f, 2.0f, 2.0f); //DTM RIFLE SCALE
		break;

		case GUN_LITHKILLER_ID :
			VEC_SET(vRet, 2.0f, 2.0f, 2.0f);
		break;

		case GUN_SICKGUN_ID	:
			VEC_SET(vRet, 2.0f, 2.0f, 2.0f);
		break;

		case GUN_BLINDER_ID	:
			VEC_SET(vRet, 2.0f, 2.0f, 2.0f);
		break;
		
		case GUN_BOMBS_ID	:
			VEC_SET(vRet, 2.0f, 2.0f, 2.0f);
		break;
		
		case GUN_RAMSET_ID	:
			VEC_SET(vRet, 2.0f, 2.0f, 2.0f);
		break;

		case GUN_AXE_ID	:
			VEC_SET(vRet, 2.0f, 2.0f, 2.0f);
		break;

		default : break;
	}

	DFLOAT fFactor = (eSize == MS_NORMAL ? 1.0f : (eSize == MS_SMALL ? 0.2f : 5.0f));
	VEC_MULSCALAR(vRet, vRet, fFactor);


	return vRet;
}

inline DVector GetHandWeaponFlashOffset(RiotWeaponId nWeaponId, ModelSize eSize=MS_NORMAL)
{
	DVector vRet;
	VEC_INIT(vRet)

	switch (nWeaponId)
	{
		case GUN_COLT45_ID :
			VEC_SET(vRet, 0.0f, 3.0f, 9.6f);
		break;

		case GUN_SHOTGUN_ID	:
			VEC_SET(vRet, 0.0f, 7.5f, 83.3f);
		break;

		case GUN_ASSAULTRIFLE_ID :
		case GUN_ENERGYGRENADE_ID :
		case GUN_KATOGRENADE_ID :
			// VEC_SET(vRet, 0.0f, 4.0f, 29.0f);
			// VEC_SET(vRet, 0.0f, 2.0f * scale, 8.75f * scale);
			VEC_SET(vRet, 0.0f, 6.0f, 26.25f);
		break;

		case GUN_MAC10_ID :
			VEC_SET(vRet, 0.0f, 4.75f, 22.5f);
		break;

		case GUN_TOW_ID	:
			VEC_SET(vRet, 0.0f, 2.8f, 28.0f);
		break;

		case GUN_PULSERIFLE_ID :
			VEC_SET(vRet, 0.0f, 5.0f, 41.25f);
		break;

		case GUN_AUTOCANNONS_ID :
			VEC_SET(vRet, 0.0f, 3.75f, 43.75f);
		break;

		case GUN_HEALGUN_ID :
			VEC_SET(vRet, 0.0f, 3.75f, 43.75f);
		break;

		case GUN_DEATHCHARGE_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 43.75f);
		break;

		case GUN_PULSELAUNCHER_ID	:
			VEC_SET(vRet, 0.0f, 5.0f, 45.0f);
		break;

		case GUN_SNIPERRIFLE_ID	:
			VEC_SET(vRet, 0.0f, 2.5f, 40.0f);	
		break;

		case GUN_MASSDRIVER_ID :
			VEC_SET(vRet, 0.0f, 5.0f, 41.25f);
		break;

		case GUN_MACHINEGUN_ID :
			VEC_SET(vRet, 0.0f, -10.0f, 61.25f);
		break;

		case GUN_ROCKETLAUNCHER_ID :
			VEC_SET(vRet, 0.0f, 5.0f, 41.25f);
		break;
		
		case GUN_RIOTCANNON_ID	:
			VEC_SET(vRet, 0.0f, 10.0f, 50.25f);
		break;

		case GUN_LITHKILLER_ID	:
			VEC_SET(vRet, 0.0f, 2.5f, 42.5f);
		break;

		case GUN_SICKGUN_ID	:
			VEC_SET(vRet, 0.0f, 5.0f, 41.25f);
		break;

		case GUN_BLINDER_ID	:
			VEC_SET(vRet, 0.0f, 5.0f, 41.25f);
		break;

		case GUN_BOMBS_ID	:
			VEC_SET(vRet, 0.0f, 5.0f, 41.25f);
		break;

		case GUN_RAMSET_ID	:
			VEC_SET(vRet, 0.0f, 5.0f, 41.25f);
		break;

		case GUN_AXE_ID	:
			VEC_SET(vRet, 0.0f, 5.0f, 41.25f);
		break;


		default : break;
	}

	DFLOAT fFactor = (eSize == MS_NORMAL ? 1.0f : (eSize == MS_SMALL ? 0.2f : 5.0f));
	VEC_MULSCALAR(vRet, vRet, fFactor);

	return vRet;
}

inline DBOOL UsesAmmo(RiotWeaponId nWeaponId)
{
	switch (nWeaponId)
	{
		case GUN_SNIPERCLUB_ID :
		case GUN_SQUEAKYTOY_ID :
		case GUN_UNUSED1_ID :
		case GUN_UNUSED2_ID :
		case GUN_RAMSET_ID :
		case GUN_AXE_ID :
		{
			return DFALSE;
		}
		break;	
	}

	return DTRUE;
}

// get the maximum range for each weapon
inline DFLOAT GetWeaponRange(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 10000.0f;

	DFLOAT s_fWeaponRange[GUN_MAX_NUMBER] =
	{
		10000.0f,	// GUN_PULSERIFLE_ID
		2000.0f,	// GUN_AUTOCANNONS_ID
	     150.0f,	// GUN_DEATHCHARGE_ID // DTM RANGE OF 500 From 10000
		10000.0f,	// GUN_PULSELAUNCHER_ID
		10000.0f,	// GUN_SNIPERRIFLE_ID
		8000.0f,	// GUN_MASSDRIVER_ID
		6000.0f,	// GUN_RIOTCANNON_ID
			0.0f,	// NOT USED1
		10000.0f,	// GUN_LITHKILLER_ID
		128.0f,		// GUN_BUZZSAW_ID // DTM BUZZSAW MELEE WEAPON
		8000.0f,	// GUN_SHOULDERCANNONS_ID
		128.0f,		// GUN_BIGBOOT_ID
		128.0f,		// GUN_SNIPERCLUB_ID
		6000.0f,	// GUN_HEALGUN_ID
		2000.0f,	// GUN_MACHINEGUN_ID
		10000.0f,	// GUN_ROCKETLAUNCHER_ID
		6000.0f,	// GUN_SICKGUN_ID
		6000.0f,	// GUN_BLINDER_ID
		0.0f,		// GUN_BOMBS_ID
		128.0f,		// GUN_RAMSET_ID
		2000.0f,	// GUN_SHOTGUN_ID
		128.0f,		// GUN_AXE_ID
		3000.0f,	// GUN_COLT45_ID
		6000.0f,	// GUN_ASSAULTRIFLE_ID
		10000.0f,	// GUN_ENERGYGRENADE_ID
		5000.0f,	// GUN_KATOGRENADE_ID
		2000.0f,	// GUN_MAC10_ID
		10000.0f,	// GUN_TOW_ID
		0.0f,		// NOT USED2
		3000.0f,	// GUN_SQUEAKYTOY_ID
	};

	return s_fWeaponRange[nWeaponId];
}


// Return the average damage done by the weapon...

inline DFLOAT GetWeaponDamage(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0.0f;

	DFLOAT s_fWeaponDamage[GUN_MAX_NUMBER] =
	{
		15.0f,		// GUN_PULSERIFLE_ID
		10.0f,		// GUN_AUTOCANNONS_ID
		1500.0f,	// GUN_DEATHCHARGE_ID
		80.0f,		// GUN_PULSELAUNCHER_ID
		20.0f,		// GUN_SNIPERRIFLE_ID
		200.0f,		// GUN_MASSDRIVER_ID
		10.0f,		// GUN_RIOTCANNON_ID
		0.0f,		// NOT USED1
		200.0f,		// GUN_LITHKILLER_ID
		50.0f,		// GUN_BUZZSAW_ID // DTM BUZZSAW MELEE WEAPON
		80.0f,		// GUN_SHOULDERCANNONS_ID
		75.0f,		// GUN_BIGBOOT_ID
		40.0f,		// GUN_SNIPERCLUB_ID
		-5.0f,		// GUN_HEALGUN_ID
		15.0f,		// GUN_MACHINEGUN_ID
		80.0f,		// GUN_ROCKETLAUNCHER_ID
		12.0f,		// GUN_SICKGUN_ID
		0.0f,		// GUN_BLINDER_ID
		100.0f,		// GUN_BOMBS_ID
		75.0f,		// GUN_RAMSET_ID
		15.0f,		// GUN_SHOTGUN_ID
		20.0f,		// GUN_AXE_ID
		20.0f,		// GUN_COLT45_ID
		25.0f,		// GUN_ASSAULTRIFLE_ID
		80.0f,		// GUN_ENERGYGRENADE_ID
		50.0f,		// GUN_KATOGRENADE_ID
		30.0f,		// GUN_MAC10_ID
		120.0f,		// GUN_TOW_ID
		0.0f,		// NOT USED2
		0.0f,		// GUN_SQUEAKYTOY_ID
	};

	return s_fWeaponDamage[nWeaponId];
}


// get the maximum range for each weapon
inline int GetWeaponDamageRadius(int nWeaponId, ModelSize eSize=MS_NORMAL)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0;

	int s_nWeaponDamageRadius[GUN_MAX_NUMBER] =
	{
		100,	// GUN_PULSERIFLE_ID
		100,	// GUN_AUTOCANNONS_ID
		300,	// GUN_DEATHCHARGE_ID
		100,	// GUN_PULSELAUNCHER_ID
		0,		// GUN_SNIPERRIFLE_ID
		200,	// GUN_MASSDRIVER_ID
		0,		// GUN_RIOTCANNON_ID
		0,		// NOT USED1
		300,	// GUN_LITHKILLER_ID
		0,		// GUN_BUZZSAW_ID // DTM BUZZSAW MELEE WEAPON
		200,	// GUN_SHOULDERCANNONS_ID
		0,		// GUN_BIGBOOT_ID
		0,		// GUN_SNIPERCLUB_ID
		100,	// GUN_HEALGUN_ID
		0,		// GUN_MACHINEGUN_ID
		300,	// GUN_ROCKETLAUNCHER_ID
		0,		// GUN_SICKGUN_ID
		0,		// GUN_BLINDER_ID
		200,	// GUN_BOMBS_ID
		0,		// GUN_RAMSET_ID
		0,		// GUN_SHOTGUN_ID
		0,		// GUN_AXE_ID
		0,		// GUN_COLT45_ID
		0,		// GUN_ASSAULTRIFLE_ID
		200,	// GUN_ENERGYGRENADE_ID
		300,	// GUN_KATOGRENADE_ID
		0,		// GUN_MAC10_ID
		300,	// GUN_TOW_ID
		0,		// NOT USED2
		0,		// GUN_SQUEAKYTOY_ID
	};

	DFLOAT fFactor = (eSize == MS_NORMAL ? 1.0f : (eSize == MS_SMALL ? 0.2f : 5.0f));
	return int((DFLOAT)s_nWeaponDamageRadius[nWeaponId] * fFactor);
}

// get the range the weapon can be heard from
inline int GetWeaponFireSoundRadius(int nWeaponId, ModelSize eSize=MS_NORMAL)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0;

	int s_nWeaponSoundRadius[GUN_MAX_NUMBER] =
	{
		1000,	// GUN_PULSERIFLE_ID
		500,	// GUN_AUTOCANNONS_ID
		1000,	// GUN_DEATHCHARGE_ID
		1000,	// GUN_PULSELAUNCHER_ID
		1000,	// GUN_SNIPERRIFLE_ID
		1500,	// GUN_MASSDRIVER_ID
		1500,	// GUN_RIOTCANNON_ID
		0,		// NOT USED1
		2500,	// GUN_LITHKILLER_ID
		0,		// GUN_BUZZSAW_ID // DTM BUZZSAW MELEE WEAPON
		1500,	// GUN_SHOULDERCANNONS_ID
		0,		// GUN_BIGBOOT_ID
		0,		// GUN_SNIPERCLUB_ID
		1000,	// GUN_HEALGUN_ID
		1500,	// GUN_MACHINEGUN_ID
		2000,	// GUN_ROCKETLAUNCHER
		500,	// GUN_SICKGUN_ID
		500, 	// GUN_BLINDER_ID
		1000,	// GUN_BOMBS_ID
		0,		// GUN_RAMSET_ID
		1500,	// GUN_SHOTGUN_ID
		0,		// GUN_AXE_ID
		1000,	// GUN_COLT45_ID
		1500,	// GUN_ASSAULTRIFLE_ID
		1500,	// GUN_ENERGYGRENADE_ID
		1000,	// GUN_KATOGRENADE_ID
		500,	// GUN_MAC10_ID
		2000,	// GUN_TOW_ID
		0,		// NOT USED2
		500,	// GUN_SQUEAKYTOY_ID
	};

	DFLOAT fFactor = (eSize == MS_NORMAL ? 1.0f : (eSize == MS_SMALL ? 0.2f : 5.0f));
	return int((DFLOAT)s_nWeaponSoundRadius[nWeaponId] * fFactor);
}

// get the range the weapon can be heard from
inline int GetWeaponImpactSoundRadius(int nWeaponId, ModelSize eSize=MS_NORMAL)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0;

	int s_nWeaponImpactSoundRadius[GUN_MAX_NUMBER] =
	{
		300,	// GUN_PULSERIFLE_ID
		200,	// GUN_AUTOCANNONS_ID
		1500,	// GUN_DEATHCHARGE_ID
		1500,	// GUN_PULSELAUNCHER_ID
		100,	// GUN_SNIPERRIFLE_ID
		2000,	// GUN_MASSDRIVER_ID
		1500,	// GUN_RIOTCANNON_ID
		0,		// NOT USED1
		5000,	// GUN_LITHKILLER_ID
		0,		// GUN_BUZZSAW_ID // DTM BUZZSAW MELEE WEAPON
		2000,	// GUN_SHOULDERCANNONS_ID
		0,		// GUN_BIGBOOT_ID
		0,		// GUN_SNIPERCLUB_ID
		300,	// GUN_HEALGUN_ID
		700,	// GUN_MACHINEGUN_ID
		2000,	// GUN_ROCKETLAUNCHER_ID
		300, 	// GUN_SICKGUN_ID
		300,	// GUN_BLINDER_ID
		1500,	// GUN_BOMBS_ID
		0,		// GUN_RAMSET_ID
		200,	// GUN_SHOTGUN_ID
		0,		// GUN_AXE_ID
		300,	// GUN_COLT45_ID
		700,	// GUN_ASSAULTRIFLE_ID
		1500,	// GUN_ENERGYGRENADE_ID
		1500,	// GUN_KATOGRENADE_ID
		200,	// GUN_MAC10_ID
		2000,	// GUN_TOW_ID
		0,		// NOT USED2
		500,	// GUN_SQUEAKYTOY_ID	
	};

	DFLOAT fFactor = (eSize == MS_NORMAL ? 1.0f : (eSize == MS_SMALL ? 0.2f : 5.0f));
	return int((DFLOAT)s_nWeaponImpactSoundRadius[nWeaponId] * fFactor);
}

// get the maximum range for each weapon
inline DBYTE GetWeaponFX(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0;

	DBYTE s_nWeaponFX[GUN_MAX_NUMBER] =
	{
		/*GUN_PULSERIFLE_ID*/		WFX_SPARKS,									
		/*GUN_AUTOCANNONS_ID*/		WFX_SPARKS,
		/*GUN_DEATHCHARGE_ID*/		WFX_MARK | WFX_SPARKS | WFX_SMOKE,
		/*GUN_PULSELAUNCHER_ID*/	WFX_MARK | WFX_SPARKS | WFX_SMOKE,
		/*GUN_SNIPERRIFLE_ID*/		WFX_FIRESOUND | WFX_LIGHT | WFX_MARK | WFX_MUZZLE | WFX_TRACER | WFX_SHELL,
		/*GUN_MASSDRIVER_ID*/		WFX_FIRESOUND | WFX_LIGHT | WFX_MARK | WFX_SPARKS | WFX_SMOKE | WFX_SHELL,	
		/*GUN_RIOTCANNON_ID*/		WFX_FIRESOUND | WFX_LIGHT | WFX_MARK | WFX_MUZZLE | WFX_TRACER | WFX_SHELL,
		/*NOT USED1*/				0,
		/*GUN_LITHKILLER_ID*/		WFX_FIRESOUND | WFX_MARK | WFX_SPARKS | WFX_SMOKE,
		/*GUN_BUZZSAW_ID*/		    WFX_FIRESOUND | WFX_SPARKS, // DTM BUZZSAW MELEE WEAPON
		/*GUN_SHOULDERCANNONS_ID*/	WFX_FIRESOUND | WFX_LIGHT | WFX_MARK | WFX_SPARKS | WFX_SMOKE | WFX_SHELL,
		/*GUN_BIGBOOT_ID*/			WFX_FIRESOUND | WFX_MARK,
		/*GUN_SNIPERCLUB_ID*/		WFX_FIRESOUND | WFX_MARK,
		/*GUN_HEALGUN_ID*/			WFX_FIRESOUND | WFX_LIGHT, 
		/*GUN_MACHINEGUN_ID*/		WFX_FIRESOUND | WFX_LIGHT | WFX_MARK | WFX_MUZZLE | WFX_TRACER | WFX_SHELL,
		/*GUN_ROCKETLAUNCHER_ID*/	WFX_MARK | WFX_SPARKS | WFX_SMOKE,
		/* GUN_SICKGUN_ID */		WFX_FIRESOUND | WFX_LIGHT, 
		/* GUN_BLINDER_ID */		WFX_FIRESOUND | WFX_LIGHT, 
		/* GUN_BOMBS_ID */			WFX_MARK | WFX_SPARKS | WFX_SMOKE, 
		/* GUN_RAMSET_ID */			WFX_FIRESOUND | WFX_MARK, 
		/* GUN_SHOTGUN_ID */		WFX_FIRESOUND | WFX_LIGHT | WFX_MARK | WFX_MUZZLE | WFX_SHELL,
		/* GUN_AXE_ID */			WFX_FIRESOUND | WFX_MARK, 
		/*GUN_COLT45_ID*/			WFX_FIRESOUND | WFX_LIGHT | WFX_MARK | WFX_MUZZLE | WFX_TRACER | WFX_SHELL,
		/*GUN_ASSAULTRIFLE_ID*/		WFX_FIRESOUND | WFX_LIGHT | WFX_MARK | WFX_MUZZLE | WFX_TRACER | WFX_SHELL,
		/*GUN_ENERGYGRENADE_ID*/	WFX_MARK | WFX_SPARKS | WFX_SMOKE,
		/*GUN_KATOGRENADE_ID*/		WFX_MARK | WFX_SPARKS | WFX_SMOKE,
		/*GUN_MAC10_ID*/			WFX_FIRESOUND | WFX_LIGHT | WFX_MARK | WFX_MUZZLE | WFX_TRACER | WFX_SHELL,
		/*GUN_TOW_ID*/				WFX_MARK | WFX_SPARKS | WFX_SMOKE,
		/*NOT USED2*/				0,
		/*GUN_SQUEAKYTOY_ID*/		0,
	};

	return s_nWeaponFX[nWeaponId];
}


// get the maximum range for each weapon
inline DBYTE GetProjectileFX(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0;

	DBYTE s_nProjectileFX[GUN_MAX_NUMBER] =
	{
		/*GUN_PULSERIFLE_ID*/		PFX_LIGHT,
		/*GUN_AUTOCANNONS_ID*/		PFX_LIGHT,				
		/*GUN_DEATHCHARGE_ID*/		PFX_FLYSOUND,
		/*GUN_PULSELAUNCHER_ID*/	PFX_FLYSOUND | PFX_LIGHT,
		/*GUN_SNIPERRIFLE_ID*/		0,
		/*GUN_MASSDRIVER_ID*/		0,	
		/*GUN_RIOTCANNON_ID*/		0,
		/*NOT USED1*/				0,
		/*GUN_LITHKILLER_ID*/		PFX_FLYSOUND | PFX_SMOKETRAIL | PFX_FLARE | PFX_LIGHT,
		/*GUN_BUZZSAW_ID*/		    0, // DTM BUZZSAW MELEE WEAPON
		/*GUN_SHOULDERCANNONS_ID*/	PFX_FLYSOUND | PFX_LIGHT,
		/*GUN_BIGBOOT_ID*/			0,
		/*GUN_SNIPERCLUB_ID*/		0,
		/*GUN_HEALGUN_ID*/			0,
		/*GUN_MACHINEGUN_ID*/		0,
		/*GUN_ROCKETLAUNCHER_ID*/	PFX_FLYSOUND | PFX_SMOKETRAIL | PFX_FLARE | PFX_LIGHT,
		/*GUN_SICKGUN_ID*/			0,
		/*GUN_BLINDER_ID*/			0,
		/*GUN_BOMBS_ID*/			PFX_FLYSOUND,
		/*GUN_RAMSET_ID*/			0,
		/*GUN_SHOTGUN_ID*/			0,
		/*GUN_AXE_ID*/				0,
		/*GUN_COLT45_ID*/			0,
		/*GUN_ASSAULTRIFLE_ID*/		0,
		/*GUN_ENERGYGRENADE_ID*/	PFX_FLYSOUND | PFX_LIGHT,
		/*GUN_KATOGRENADE_ID*/		PFX_FLYSOUND | PFX_LIGHT,
		/*GUN_MAC10_ID*/			0,
		/*GUN_TOW_ID*/				PFX_FLYSOUND | PFX_SMOKETRAIL | PFX_FLARE | PFX_LIGHT,
		/*NOT USED2*/				0,
		/*GUN_SQUEAKYTOY_ID*/		0,
	};

	return s_nProjectileFX[nWeaponId];
}




////////////////////////////////////////////////////////////////////////////////
//
// The following functions replace data members in weapon.cpp that used to be
// sent from the server to the client....
//
/////////////////////////////////////////////////////////////////////////////////

// get the name of the flash used by this weapon
inline char* GetFlashFilename(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return DNULL;

	char* s_cFlashFilename[GUN_MAX_NUMBER] =
	{
		/*GUN_PULSERIFLE_ID*/		DNULL,
		/*GUN_AUTOCANNONS_ID*/		DNULL,				
		/*GUN_DEATHCHARGE_ID*/		DNULL,
		/*GUN_PULSELAUNCHER_ID*/	DNULL,
		/*GUN_SNIPERRIFLE_ID*/		DNULL,
		/*GUN_MASSDRIVER_ID*/		DNULL,	
		/*GUN_RIOTCANNON_ID*/		"Sprites\\shotgunmuzzleflash.spr",
		/*NOT USED1*/				DNULL,
		/*GUN_LITHKILLER_ID*/		DNULL,
		/*GUN_BUZZSAW_ID*/	    	DNULL, // DTM BUZZSAW MELEE WEAPON
		/*GUN_SHOULDERCANNONS_ID*/	DNULL,
		/*GUN_BIGBOOT_ID*/			DNULL,
		/*GUN_SNIPERCLUB_ID*/		DNULL,
		/*GUN_HEALGUN_ID*/			DNULL,
		/*GUN_MACHINEGUN_ID*/		DNULL, //"Sprites\\AssaultRifleMuzzleflash.spr",
		/*GUN_ROCKETLAUNCHER_ID*/	DNULL,
		/*GUN_SICKGUN_ID*/			DNULL,
		/*GUN_BLINDER_ID*/			DNULL,
		/*GUN_BOMBS_ID*/			DNULL,
		/*GUN_RAMSET_ID*/			DNULL,
		/*GUN_SHOTGUN_ID*/			DNULL, //"Sprites\\shotgunmuzzleflash.spr",
		/*GUN_AXE_ID*/				DNULL,
		/*GUN_COLT45_ID*/			"Sprites\\colt45muzzleflash.spr",
		/*GUN_ASSAULTRIFLE_ID*/		"Sprites\\AssaultRifleMuzzleflash.spr",
		/*GUN_ENERGYGRENADE_ID*/	DNULL,
		/*GUN_KATOGRENADE_ID*/		DNULL,
		/*GUN_MAC10_ID*/			"Sprites\\Weapons\\Mac10Flash.spr",
		/*GUN_TOW_ID*/				DNULL,
		/*NOT USED2*/				DNULL,
		/*GUN_SQUEAKYTOY_ID*/		DNULL,
	};

	return s_cFlashFilename[nWeaponId];
}


// get the duration of the flash used by this weapon
inline DFLOAT GetFlashDuration(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0.0f;

	DFLOAT s_fFlashDuration[GUN_MAX_NUMBER] =
	{
		/*GUN_PULSERIFLE_ID*/		0.0f,
		/*GUN_AUTOCANNONS_ID*/		0.075f,				
		/*GUN_DEATHCHARGE_ID*/		0.0f,
		/*GUN_PULSELAUNCHER_ID*/	0.0f,
		/*GUN_SNIPERRIFLE_ID*/		0.0f,
		/*GUN_MASSDRIVER_ID*/		0.0f,	
		/*GUN_RIOTCANNON_ID*/		0.075f,
		/*NOT USED1*/				0.0f,
		/*GUN_LITHKILLER_ID*/		0.0f,
		/*GUN_BUZZSAW_ID*/	    	0.0f, // DTM BUZZSAW MELEE WEAPON
		/*GUN_SHOULDERCANNONS_ID*/	0.0f,
		/*GUN_BIGBOOT_ID*/			0.0f,
		/*GUN_SNIPERCLUB_ID*/		0.0f,
		/*GUN_HEALGUN_ID*/			0.0f,
		/*GUN_MACHINEGUN_ID*/		0.075f,
		/*GUN_ROCKETLAUNCHER_ID*/	0.0f,
		/*GUN_SICKGUN_ID*/			0.0f,
		/*GUN_BLINDER_ID*/			0.0f,
		/*GUN_BOMBS_ID*/			0.0f,
		/*GUN_RAMSET_ID*/			0.0f,
		/*GUN_SHOTGUN_ID*/			0.075f,
		/*GUN_AXE_ID*/				0.0f,
		/*GUN_COLT45_ID*/			0.075f,
		/*GUN_ASSAULTRIFLE_ID*/		0.075f,
		/*GUN_ENERGYGRENADE_ID*/	0.0f,
		/*GUN_KATOGRENADE_ID*/		0.0f,
		/*GUN_MAC10_ID*/			0.075f,
		/*GUN_TOW_ID*/				0.0f,
		/*NOT USED2*/				0.0f,
		/*GUN_SQUEAKYTOY_ID*/		0.0f,
	};

	return s_fFlashDuration[nWeaponId];
}

// get the scale of the flash used by this weapon
inline DVector GetFlashScale(RiotWeaponId nWeaponId, ModelSize eSize=MS_NORMAL)
{
	DVector vRet;
	VEC_INIT(vRet)

	switch (nWeaponId)
	{
		case GUN_COLT45_ID :
			VEC_SET(vRet, 0.06f, 0.06f, 0.0f);
		break;

		case GUN_SHOTGUN_ID	:
			VEC_SET(vRet, 0.1125f, 0.1125f, 0.0f);
		break;

		case GUN_ASSAULTRIFLE_ID :
			VEC_SET(vRet, 0.1f, 0.1f, 0.0f);
		break;

		case GUN_MACHINEGUN_ID :
			VEC_SET(vRet, 0.1f, 0.1f, 0.0f);
		break;

		case GUN_MAC10_ID :
			VEC_SET(vRet, 0.1f, 0.1f, 0.0f);
		break;

		default : break;
	}

	DFLOAT fFactor = (eSize == MS_NORMAL ? 1.0f : (eSize == MS_SMALL ? 0.2f : 5.0f));
	VEC_MULSCALAR(vRet, vRet, fFactor);

	return vRet;
}


// Get the offset of the weapon from the player object...
inline DVector GetWeaponOffset(RiotWeaponId nWeaponId)
{
	DVector vRet;
	VEC_SET(vRet, 1.0f, 1.0f, 1.0f)

	switch (nWeaponId)
	{
		case GUN_RAMSET_ID :
			VEC_SET(vRet, 1.3f, -2.6f, 4.0f);
		break;

		case GUN_COLT45_ID :
			VEC_SET(vRet, -0.1f, -2.3f, 6.2f);
		break;

		case GUN_SHOTGUN_ID	:
			VEC_SET(vRet, 1.0f, -2.0f, 3.5f);
		break;

		case GUN_MAC10_ID :
			VEC_SET(vRet, .7f, -2.3f, 6.6f);
		break;

		case GUN_ASSAULTRIFLE_ID :
			VEC_SET(vRet, 1.4f, -2.2f, 7.3f);
		break;

		case GUN_ENERGYGRENADE_ID :
			VEC_SET(vRet, 0.8f, -1.8f, 5.7f);
		break;

		case GUN_KATOGRENADE_ID :
			VEC_SET(vRet, 0.8f, -1.8f, 5.7f);
		break;

		case GUN_TOW_ID	:
			VEC_SET(vRet, 1.35f, -4.5f, 8.0f);
		break;

		case GUN_SQUEAKYTOY_ID :
			VEC_SET(vRet, -1.6, -3.5f, 2.4f);
		break;

		case GUN_PULSERIFLE_ID :
			VEC_SET(vRet, 0.5f, -1.2f, 2.5f);
		break;

		case GUN_RIOTCANNON_ID	:
			VEC_SET(vRet, 1.0f, -1.4f, 6.5f);
		break;

		case GUN_MASSDRIVER_ID :
			VEC_SET(vRet, 1.0f, -2.6f, 5.5f);
		break;

		case GUN_MACHINEGUN_ID :
			VEC_SET(vRet, 1.0f, -2.6f, 5.5f);
		break;
		
		case GUN_ROCKETLAUNCHER_ID :
			VEC_SET(vRet, 1.0f, -2.6f, 5.5f);
		break;

		case GUN_SICKGUN_ID :
			VEC_SET(vRet, 1.0f, -2.6f, 5.5f);
		break;
		
		case GUN_BLINDER_ID :
			VEC_SET(vRet, 1.0f, -2.6f, 5.5f);
		break;
		
		case GUN_BOMBS_ID :
			VEC_SET(vRet, 1.0f, -2.6f, 5.5f);
		break;
		
		case GUN_AXE_ID	:
			VEC_SET(vRet, 1.0f, -2.4f, 1.0f);
		break;

		case GUN_AUTOCANNONS_ID:
			VEC_SET(vRet, 0.6f, -1.8f, -1.0f);
		break;

		case GUN_DEATHCHARGE_ID :
			VEC_SET(vRet, 0.5f, -1.6f, 6.0f);
		break;

		case GUN_SNIPERRIFLE_ID :
			VEC_SET(vRet, 0.9f, -2.8f, -1.2f);
		break;

		case GUN_LITHKILLER_ID :
			VEC_SET(vRet, 2.4f, -3.7f, 7.4f);
		break;

		case GUN_BUZZSAW_ID : // DTM BUZZSAW MELEE WEAPON
			VEC_SET(vRet, -0.4f, -3.5f, 3.7f);
		break;

		case GUN_SHOULDERCANNONS_ID :
			VEC_SET(vRet, 0.8f, -1.2f, 2.9f);
		break;

		case GUN_BIGBOOT_ID :
			VEC_SET(vRet, 0.5f, -1.9f, 0.2f);
		break;

		case GUN_SNIPERCLUB_ID :
			VEC_SET(vRet, 0.1f, -2.1f, 3.1f);
		break;

		case GUN_HEALGUN_ID:
			VEC_SET(vRet, 0.6f, -1.8f, -1.0f);
		break;

		default : break;
	}

	return vRet;
}


// Get the offset of the muzzle from the player object...
inline DVector GetWeaponMuzzleOffset(RiotWeaponId nWeaponId, WeaponState eState=W_FIRED)
{
	DVector vRet;
	VEC_SET(vRet, 0.0f, 0.0f, 0.0f)

	switch (nWeaponId)
	{
		case GUN_RAMSET_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		case GUN_COLT45_ID :
		{
			if (eState == W_FIRED)
			{
				VEC_SET(vRet, -6.2f, -1.3f, 12.2f);
			}
			else if (eState == W_FIRED2)
			{
				VEC_SET(vRet, 5.8f, -1.3f, 12.2f);
			}
		}
		break;

		case GUN_SHOTGUN_ID	:
			VEC_SET(vRet, 0.9f, 1.0f, -1.5f);
		break;

		case GUN_MAC10_ID :
			VEC_SET(vRet, 2.1f, -0.5f, 13.3f);
		break;

		case GUN_ASSAULTRIFLE_ID :
			VEC_SET(vRet, 0.6f, 0.0f, 10.3f);
		break;

		case GUN_ENERGYGRENADE_ID :
			VEC_SET(vRet, 4.4f, -0.6f, 15.9f);
		break;

		case GUN_KATOGRENADE_ID :
			VEC_SET(vRet, 4.4f, -0.6f, 15.9f);
		break;

		case GUN_TOW_ID	:
			VEC_SET(vRet, 1.85f, 1.36f, 11.2f);
		break;

		case GUN_SQUEAKYTOY_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		case GUN_PULSERIFLE_ID :
			VEC_SET(vRet, 1.1f, -1.0f, 21.5f);
		break;

		case GUN_RIOTCANNON_ID	:
			VEC_SET(vRet, 1.6f, -2.8f, 17.1f);
		break;

		case GUN_MASSDRIVER_ID :
			VEC_SET(vRet, 0.8f, 0.9f, 18.5f);
		break;

		case GUN_MACHINEGUN_ID :
			VEC_SET(vRet, 0.8f, -5.0f, 8.5f);
		break;

		case GUN_ROCKETLAUNCHER_ID :
			VEC_SET(vRet, 0.8f, 0.9f, 18.5f);
		break;

		case GUN_SICKGUN_ID :
			VEC_SET(vRet, 0.8f, 0.9f, 18.5f);
		break;

		case GUN_BLINDER_ID :
			VEC_SET(vRet, 0.8f, 0.9f, 18.5f);
		break;

		case GUN_BOMBS_ID :
			VEC_SET(vRet, 0.8f, -30.9f, -1.5f);
		break;

		case GUN_AXE_ID :
			VEC_SET(vRet, 0.8f, 0.9f, 18.5f);
		break;

		case GUN_PULSELAUNCHER_ID	:
			VEC_SET(vRet, -2.75f, 0.36f, 16.6f);
		break;

		case GUN_AUTOCANNONS_ID:
			VEC_SET(vRet, 0.4f, 0.0f, 21.4f);
		break;

		case GUN_DEATHCHARGE_ID :
			VEC_SET(vRet, 0.5f, 0.7f, 14.4f);
		break;

		case GUN_SNIPERRIFLE_ID :
			VEC_SET(vRet, -0.8f, 3.0f, 21.9f);
		break;

		case GUN_LITHKILLER_ID :
			VEC_SET(vRet, 0.2f, 0.25f, 14.3f);
		break;

		case GUN_BUZZSAW_ID : // DTM BUZZSAW MELEE WEAPON
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		case GUN_SHOULDERCANNONS_ID :
			VEC_SET(vRet, 0.4f, 0.0f, 21.4f);
		break;

		case GUN_BIGBOOT_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		case GUN_SNIPERCLUB_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		case GUN_HEALGUN_ID:
			VEC_SET(vRet, 0.4f, 0.0f, 21.4f);
		break;
		
		default : break;
	}

	return vRet;
}


// Get the offset of the weapon from the player object...
inline DVector GetWeaponOffset2(RiotWeaponId nWeaponId)
{
	DVector vRet;
	VEC_SET(vRet, 0.0f, 0.0f, 0.0f)

	switch (nWeaponId)
	{
		case GUN_RAMSET_ID :
			VEC_SET(vRet, 1.3f, -4.0f, 3.0f);
		break;

		case GUN_COLT45_ID :
			VEC_SET(vRet, -0.1f, -2.3f, 2.4f);
		break;

		case GUN_SHOTGUN_ID	:
			VEC_SET(vRet, 1.0f, -1.8f, -5.7f);
		break;

		case GUN_MAC10_ID :
			VEC_SET(vRet, .7f, -2.2f, 2.5f);
		break;

		case GUN_ASSAULTRIFLE_ID :
			VEC_SET(vRet, 1.4f, -2.2f, -0.5f);
		break;

		case GUN_ENERGYGRENADE_ID :
			VEC_SET(vRet, 0.8f, -1.8f, 0.7f);
		break;

		case GUN_KATOGRENADE_ID :
			VEC_SET(vRet, 0.8f, -1.8f, 0.7f);
		break;

		case GUN_TOW_ID	:
			VEC_SET(vRet, 1.35f, -4.5f, 0.2f);
		break;

		case GUN_SQUEAKYTOY_ID :
			VEC_SET(vRet, -1.6, -3.5f, -3.2f);
		break;

		case GUN_PULSERIFLE_ID :
			VEC_SET(vRet, 0.5f, -1.2f, -1.9f);
		break;

		case GUN_RIOTCANNON_ID :
			VEC_SET(vRet, 1.0f, -1.4f, -2.5f);
		break;

		case GUN_MASSDRIVER_ID :
			VEC_SET(vRet, 1.0f, -2.6f, -1.6f);
		break;

		case GUN_MACHINEGUN_ID :
			VEC_SET(vRet, 1.0f, -2.6f, -1.6f);
		break;

		case GUN_ROCKETLAUNCHER_ID :
			VEC_SET(vRet, 1.0f, -2.6f, -1.6f);
		break;

		case GUN_SICKGUN_ID :
			VEC_SET(vRet, 1.0f, -2.6f, -1.6f);
		break;

		case GUN_BLINDER_ID :
			VEC_SET(vRet, 1.0f, -2.6f, -1.6f);
		break;

		case GUN_BOMBS_ID :
			VEC_SET(vRet, 1.0f, -2.6f, -1.6f);
		break;

		case GUN_AXE_ID :
			VEC_SET(vRet, 1.0f, -2.6f, -1.6f);
		break;

		case GUN_PULSELAUNCHER_ID	:
			VEC_SET(vRet, 1.0f, -2.4f, -3.4f);
		break;

		case GUN_AUTOCANNONS_ID:
			VEC_SET(vRet, 0.6f, -1.8f, -8.8f);
		break;

		case GUN_DEATHCHARGE_ID :
			VEC_SET(vRet, 0.5f, -1.6f, 1.0f);
		break;

		case GUN_SNIPERRIFLE_ID :
			VEC_SET(vRet, 0.9f, -2.8f, -13.0f);
		break;

		case GUN_LITHKILLER_ID :
			VEC_SET(vRet, 2.4f, -3.7f, -5.8f);
		break;

		case GUN_BUZZSAW_ID :  // DTM BUZZSAW MELEE WEAPON
			VEC_SET(vRet, -0.4f, -3.5f, 3.7f);
		break;

		case GUN_SHOULDERCANNONS_ID :
			VEC_SET(vRet, 0.8f, -1.2f, 0.9f);
		break;

		case GUN_BIGBOOT_ID :
			VEC_SET(vRet, 0.5f, -1.9f, -3.7f);
		break;

		case GUN_SNIPERCLUB_ID :
			VEC_SET(vRet, 0.1f, -2.1f, 1.0f);
		break;

		case GUN_HEALGUN_ID:
			VEC_SET(vRet, 0.6f, -1.8f, -8.8f);
		break;

		default : break;
	}

	return vRet;
}


// Get the offset of the muzzle from the player object...
inline DVector GetWeaponMuzzleOffset2(RiotWeaponId nWeaponId, WeaponState eState=W_FIRED)
{
	DVector vRet;
	VEC_SET(vRet, 0.0f, 0.0f, 0.0f)

	switch (nWeaponId)
	{
		case GUN_RAMSET_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		case GUN_COLT45_ID :
		{
			if (eState == W_FIRED)
			{
				VEC_SET(vRet, -6.2f, -1.3f, 8.6f);
			}
			else if (eState == W_FIRED2)
			{
				VEC_SET(vRet, 5.8f, -1.3f, 8.6f);
			}
		}
		break;

		case GUN_SHOTGUN_ID	:
			VEC_SET(vRet, 2.6f, -1.0f, 15.3f);
		break;

		case GUN_MAC10_ID :
			VEC_SET(vRet, 2.1f, -0.5f, 13.3f);
		break;

		case GUN_ASSAULTRIFLE_ID :
			VEC_SET(vRet, 1.4f, -1.4f, 11.7f);
		break;

		case GUN_ENERGYGRENADE_ID :
			VEC_SET(vRet, 4.4f, -0.6f, 15.9f);
		break;

		case GUN_KATOGRENADE_ID :
			VEC_SET(vRet, 4.4f, -0.6f, 15.9f);
		break;

		case GUN_TOW_ID	:
			VEC_SET(vRet, 1.85f, 1.36f, 11.2f);
		break;

		case GUN_SQUEAKYTOY_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		case GUN_PULSERIFLE_ID :
			VEC_SET(vRet, 1.1f, -1.0f, 21.5f);
		break;

		case GUN_RIOTCANNON_ID :
			VEC_SET(vRet, 1.4f, -3.0f, 13.3f);
		break;

		case GUN_MASSDRIVER_ID :
			VEC_SET(vRet, 1.8f, -2.6f, 14.4f);
		break;
		
		case GUN_MACHINEGUN_ID :
			VEC_SET(vRet, 1.8f, -2.6f, 14.4f);
		break;
		
		case GUN_ROCKETLAUNCHER_ID :
			VEC_SET(vRet, 1.8f, -2.6f, 14.4f);
		break;

		case GUN_SICKGUN_ID :
			VEC_SET(vRet, 1.8f, -2.6f, 14.4f);
		break;

		case GUN_BLINDER_ID :
			VEC_SET(vRet, 1.8f, -2.6f, 14.4f);
		break;

		case GUN_BOMBS_ID :
			VEC_SET(vRet, 1.8f, -25.6f, 14.4f);
		break;

		case GUN_AXE_ID :
			VEC_SET(vRet, 1.8f, -2.6f, 14.4f);
		break;

		case GUN_PULSELAUNCHER_ID	:
			VEC_SET(vRet, -2.75f, 0.36f, 16.6f);
		break;

		case GUN_AUTOCANNONS_ID :
			VEC_SET(vRet, 1.4f, -2.2f, 18.6f);
		break;

		case GUN_DEATHCHARGE_ID :
			VEC_SET(vRet, 0.5f, 0.7f, 14.4f);
		break;

		case GUN_SNIPERRIFLE_ID :
			VEC_SET(vRet, -0.5f, 2.6f, -12.8f);
		break;

		case GUN_LITHKILLER_ID :
			VEC_SET(vRet, 0.2f, 0.25f, 14.3f);
		break;

		case GUN_BUZZSAW_ID : // DTM BUZZSAW MELEE WEAPON
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		case GUN_SHOULDERCANNONS_ID :
			VEC_SET(vRet, 1.4f, -2.2f, 18.6f);
		break;

		case GUN_BIGBOOT_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		case GUN_SNIPERCLUB_ID :
			VEC_SET(vRet, 0.0f, 0.0f, 0.0f);
		break;

		case GUN_HEALGUN_ID :
			VEC_SET(vRet, 1.4f, -2.2f, 18.6f);
		break;
		
		default : break;
	}

	return vRet;
}

// get the scale of the flash used by this weapon
inline DVector GetFlashScale2(RiotWeaponId nWeaponId, ModelSize eSize=MS_NORMAL)
{
	DVector vRet;
	VEC_INIT(vRet)

	switch (nWeaponId)
	{
		case GUN_COLT45_ID :
			VEC_SET(vRet, 0.04f, 0.04f, 0.0f);
		break;

		case GUN_SHOTGUN_ID	:
			VEC_SET(vRet, 0.05f, 0.05f, 0.0f);
		break;

		case GUN_ASSAULTRIFLE_ID :
			VEC_SET(vRet, 0.10f, 0.10f, 0.0f);
		break;

		case GUN_MACHINEGUN_ID :
			VEC_SET(vRet, 0.10f, 0.10f, 0.0f);
		break;

		case GUN_MAC10_ID :
			VEC_SET(vRet, 0.05f, 0.05f, 0.0f);
		break;

		default : break;
	}

	DFLOAT fFactor = (eSize == MS_NORMAL ? 1.0f : (eSize == MS_SMALL ? 0.2f : 5.0f));
	VEC_MULSCALAR(vRet, vRet, fFactor);

	return vRet;
}


// Get the offset of the muzzle from the player object...
inline DVector GetWeaponRecoil(RiotWeaponId nWeaponId)
{
	DVector vRet;
	VEC_SET(vRet, 0.0f, 0.0f, 0.0f)

	switch (nWeaponId)
	{
		case GUN_MAC10_ID :
			VEC_SET(vRet, 0.03f, 0.03f, 0.15f);
		break;

		case GUN_ASSAULTRIFLE_ID :
			VEC_SET(vRet, 0.03f, 0.03f, 0.15f);
		break;

		case GUN_MACHINEGUN_ID :
			VEC_SET(vRet, 0.03f, 0.03f, 0.15f);
		break;


		default : break;
	}

	return vRet;
}


inline DDWORD GetExtraWeaponFlags(RiotWeaponId nWeaponId) 
{ 
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0;

	DDWORD m_dwExtraModelFlags[GUN_MAX_NUMBER] =
	{
		/*GUN_PULSERIFLE_ID*/		0,
		/*GUN_AUTOCANNONS_ID*/		0,				
		/*GUN_DEATHCHARGE_ID*/		0,
		/*GUN_PULSELAUNCHER_ID*/	0,
		/*GUN_SNIPERRIFLE_ID*/		0,
		/*GUN_MASSDRIVER_ID*/		0,	
		/*GUN_RIOTCANNON_ID*/		0,
		/*NOT USED1*/				0,
		/*GUN_LITHKILLER_ID*/		0,
		/*GUN_BUZZSAW_ID*/		    FLAG_ENVIRONMENTMAP, // DTM BUZZSAW MELEE WEAPON
		/*GUN_SHOULDERCANNONS_ID*/	0,
		/*GUN_BIGBOOT_ID*/			0,
		/*GUN_SNIPERCLUB_ID*/		0,
		/*GUN_HEALGUN_ID*/			0,
		/*GUN_MACHINEGUN_ID*/		0,
		/*GUN_ROCKETLAUNCHER_ID*/	0,
		/*GUN_SICKGUN_ID*/			0,
		/*GUN_BLINDER_ID*/			0,
		/*GUN_BOMBS_ID*/			0,
		/*GUN_RAMSET_ID*/			0,
		/*GUN_SHOTGUN_ID*/			0,
		/*GUN_AXE_ID*/				FLAG_ENVIRONMENTMAP,
		/*GUN_COLT45_ID*/			0,
		/*GUN_ASSAULTRIFLE_ID*/		0,
		/*GUN_ENERGYGRENADE_ID*/	0,
		/*GUN_KATOGRENADE_ID*/		0,
		/*GUN_MAC10_ID*/			0,
		/*GUN_TOW_ID*/				0,
		/*NOT USED2*/				0,
		/*GUN_SQUEAKYTOY_ID*/		0,
	};

	return m_dwExtraModelFlags[nWeaponId]; 
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	GetWeaponFireSound()
//
//	PURPOSE:	Get fire sounds associated with this weapon
//
// ----------------------------------------------------------------------- //

inline DBOOL GetWeaponSoundDir(RiotWeaponId nId)
{
	char* pSound = DNULL;

	switch (nId)
	{
		case GUN_PULSERIFLE_ID :
			pSound = "PulseRifle";
		break;

		case GUN_DEATHCHARGE_ID :
			pSound = "DeathCharge";
		break;

		case GUN_PULSELAUNCHER_ID :
			pSound = "PulseLauncher";
		break;

		case GUN_SNIPERRIFLE_ID :
			pSound = "SniperRifle";
		break;

		case GUN_MASSDRIVER_ID :
			pSound = "MassDriver";
		break;

		case GUN_RIOTCANNON_ID :
			pSound = "RiotCannon";
		break;

		case GUN_LITHKILLER_ID :
			pSound = "LithKiller";
		break;

		case GUN_BUZZSAW_ID : // DTM BUZZSAW MELEE WEAPON
			pSound = "Buzzsaw";
		break;

		case GUN_SHOULDERCANNONS_ID :
			pSound = "ShoulderCannons";
		break;

		case GUN_BIGBOOT_ID :
			pSound = "BigBoot";
		break;

		case GUN_SNIPERCLUB_ID :
			pSound = "SniperClub";
		break;

		case GUN_COLT45_ID :
			pSound	= "Colt45";
		break;
		
		case GUN_SHOTGUN_ID	:
			pSound = "Shotgun";
		break;
		
		case GUN_ASSAULTRIFLE_ID :
			pSound = "AssaultRifle";
		break;
		
		case GUN_MAC10_ID :
			pSound = "Machinegun";
		break;
	
		case GUN_RAMSET_ID :
			pSound = "Ramset";
		break;

		case GUN_ENERGYGRENADE_ID :
			pSound = "EnergyGrenade";
		break;

		case GUN_KATOGRENADE_ID :
			pSound = "KatoGrenade";
		break;

		case GUN_TOW_ID	:
			pSound = "TOW";
		break;

		case GUN_AUTOCANNONS_ID :
			pSound = "AutoCannons";
		break;

		case GUN_SQUEAKYTOY_ID :
			pSound = "Squeakytoy";
		break;

		case GUN_HEALGUN_ID :
			pSound = "HealGun";
		break;

		case GUN_MACHINEGUN_ID :
			pSound = "MachineGun";
		break;
		
		case GUN_ROCKETLAUNCHER_ID :
			pSound = "RocketLauncher";
		break;

		case GUN_SICKGUN_ID	:
			pSound = "SickGun";
		break;

		case GUN_BLINDER_ID	:
			pSound = "Blinder";
		break;

		case GUN_BOMBS_ID	:
			pSound = "Bombs";
		break;

		case GUN_AXE_ID	:
			pSound = "Axe";
		break;


		default : 
			return DFALSE;
			break;
	}


	sprintf( s_FileBuffer, "Sounds\\Weapons\\%s\\", pSound );
	return DTRUE;
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	GetWeaponFireSound()
//
//	PURPOSE:	Get fire sounds associated with this weapon
//
// ----------------------------------------------------------------------- //

inline char* GetWeaponFireSound(RiotWeaponId nId)
{
	if (!GetWeaponSoundDir(nId)) return DNULL;

	if( nId == GUN_COLT45_ID && GetRandom(0, 1) == 1 )
	{
		strcat( s_FileBuffer, "fire2.wav" );
	}
	else
	{
		strcat( s_FileBuffer, "fire.wav" );
	}

	return s_FileBuffer;
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	GetWeaponDryFireSound()
//
//	PURPOSE:	Get dry fire sounds associated with this weapon
//
// ----------------------------------------------------------------------- //

inline char* GetWeaponDryFireSound(RiotWeaponId nId)
{
	if (!GetWeaponSoundDir(nId)) return DNULL;

	strcat(s_FileBuffer, "empty.wav");
	return s_FileBuffer;
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	GetWeaponFlyingSound()
//
//	PURPOSE:	Get flying sound associated with this weapon
//
// ----------------------------------------------------------------------- //

inline char* GetWeaponFlyingSound(RiotWeaponId nId)
{
	char* pRet = DNULL;

	switch (nId)
	{
		case GUN_DEATHCHARGE_ID :
			pRet = "Sounds\\Weapons\\DeathCharge\\Timer.wav";
		break;

		case GUN_PULSELAUNCHER_ID :
			pRet = "Sounds\\Weapons\\PulseLauncher\\projectile.wav";
		break;

		case GUN_LITHKILLER_ID :
			pRet = "Sounds\\Weapons\\LithKiller\\projectile.wav";
		break;
		
		case GUN_ENERGYGRENADE_ID :
			pRet = "Sounds\\Weapons\\EnergyGrenade\\projectile.wav";
		break;

		case GUN_KATOGRENADE_ID :
			pRet = "Sounds\\Weapons\\KatoGrenade\\projectile.wav";
		break;

		case GUN_TOW_ID	:
			pRet = "Sounds\\Weapons\\Tow\\projectile.wav";
		break;

		case GUN_ROCKETLAUNCHER_ID :
			pRet = "Sounds\\Weapons\\RocketLauncher\\projectile.wav";
		break;

		default:
		break;
	}

	return pRet;
}


// get the duration of the flash used by this weapon
inline int GetShotsPerClip(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return -1;

	int s_nShotsPerClip[GUN_MAX_NUMBER] =
	{
		/*GUN_PULSERIFLE_ID*/		30,
		/*GUN_AUTOCANNONS_ID*/		30,				
		/*GUN_DEATHCHARGE_ID*/		1,
		/*GUN_PULSELAUNCHER_ID*/	1,
		/*GUN_SNIPERRIFLE_ID*/		2,
		/*GUN_MASSDRIVER_ID*/		-1,	
		/*GUN_RIOTCANNON_ID*/		50,
		/*NOT USED1*/				0,
		/*GUN_LITHKILLER_ID*/		1,
		/*GUN_BUZZSAW_ID*/			1, // DTM BUZZSAW MELEE WEAPON
		/*GUN_SHOULDERCANNONS_ID*/	-1,
		/*GUN_BIGBOOT_ID*/			-1,
		/*GUN_SNIPERCLUB_ID*/		-1,
		/*GUN_HEALGUN_ID*/			1,
		/*GUN_MACHINEGUN_ID*/		30,
		/*GUN_ROCKETLAUNCHER_ID*/	1,
		/*GUN_SICKGUN_ID*/			1,
		/*GUN_BLINDER_ID*/			1,
		/*GUN_BOMBS_ID*/			1,
		/*GUN_RAMSET_ID*/			-1,
		/*GUN_SHOTGUN_ID*/			1,
		/*GUN_AXE_ID*/				-1,
		/*GUN_COLT45_ID*/			22,
		/*GUN_ASSAULTRIFLE_ID*/		50,
		/*GUN_ENERGYGRENADE_ID*/	6,
		/*GUN_KATOGRENADE_ID*/		1,
		/*GUN_MAC10_ID*/			30,
		/*GUN_TOW_ID*/				1,
		/*NOT USED2*/				0,
		/*GUN_SQUEAKYTOY_ID*/		-1,
	};

	return s_nShotsPerClip[nWeaponId];
}

// Get the number of vectors this weapon shoots...
inline int GetVectorsPerShot(int nWeaponId)
{
	if (nWeaponId == GUN_SHOTGUN_ID) 
		return 7;
	if (nWeaponId == GUN_AUTOCANNONS_ID)
		return 2;
	if (nWeaponId == GUN_RIOTCANNON_ID)
		return 5;

	return 1;
}

// Get the number of vectors this weapon shoots...
inline DFLOAT GetWeaponSpread(int nWeaponId)
{
	//DTM
	if (nWeaponId == GUN_SHOULDERCANNONS_ID) return 5.5f;
	
	return 0.0f;
}


// get the duration of the flash used by this weapon
inline DBYTE GetWeaponPerturbe(int nWeaponId, DFLOAT & fRPerturbe, 
							   DFLOAT & fUPerturbe)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0;

	// Important that these values are between 0 and 127...

	DBYTE s_nPerturbe[GUN_MAX_NUMBER] =
	{
		/*GUN_PULSERIFLE_ID*/		0,
		/*GUN_AUTOCANNONS_ID*/		40,				
		/*GUN_DEATHCHARGE_ID*/		0,
		/*GUN_PULSELAUNCHER_ID*/	5,
		/*GUN_SNIPERRIFLE_ID*/		0,
		/*GUN_MASSDRIVER_ID*/		10,	
		/*GUN_RIOTCANNON_ID*/		5,
		/*NOT USED1*/				0,
		/*GUN_LITHKILLER_ID*/		20,
		/*GUN_BUZZSAW_ID*/			0, // DTM BUZZSAW MELEE WEAPON
		/*GUN_SHOULDERCANNONS_ID*/	10,
		/*GUN_BIGBOOT_ID*/			0,
		/*GUN_SNIPERCLUB_ID*/		0,
		/*GUN_HEALGUN_ID*/			0,
		/*GUN_MACHINEGUN_ID*/		40,
		/*GUN_ROCKETLAUNCHER_ID*/	0,
		/*GUN_SICKGUN_ID*/			0,
		/*GUN_BLINDER_ID*/			0,
		/*GUN_BOMBS_ID*/			0,
		/*GUN_RAMSET_ID*/			0,
		/*GUN_SHOTGUN_ID*/			60,
		/*GUN_AXE_ID*/				0,
		/*GUN_COLT45_ID*/			30,
		/*GUN_ASSAULTRIFLE_ID*/		5,
		/*GUN_ENERGYGRENADE_ID*/	10,
		/*GUN_KATOGRENADE_ID*/		10,
		/*GUN_MAC10_ID*/			40,
		/*GUN_TOW_ID*/				0,
		/*NOT USED2*/				0,
		/*GUN_SQUEAKYTOY_ID*/		0,
	};

	int nPerturbe = s_nPerturbe[nWeaponId];

	// Get a random perturbe based on nPerturbe...USE nSeed!!!  Need to come
	// up with an algorithm that uses nSeed (and adjusts nSeed?) so that this
	// can be called with the same seed to generate the same "random" values...

	fRPerturbe = ((DFLOAT)GetRandom(-nPerturbe, nPerturbe))/1000.0f;
	fUPerturbe = ((DFLOAT)GetRandom(-nPerturbe, nPerturbe))/1000.0f;

	return nPerturbe;
}


inline char* GetPVModelName(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return DNULL;

	char* s_cPVModelName[GUN_MAX_NUMBER] =
	{
		/*GUN_PULSERIFLE_ID*/		"Models\\PV_Weapons\\PulseRifle_pv.abc",
		/*GUN_AUTOCANNONS_ID*/		"Models\\PV_Weapons\\AutoCannons_pv.abc",				
		/*GUN_DEATHCHARGE_ID*/		"Models\\PV_Weapons\\DeathCharge_pv.abc",
		/*GUN_PULSELAUNCHER_ID*/	"Models\\PV_Weapons\\PulseLauncher_pv.abc",
		/*GUN_SNIPERRIFLE_ID*/		"Models\\PV_Weapons\\SniperRifle_pv.abc",
		/*GUN_MASSDRIVER_ID*/		"Models\\PV_Weapons\\MassDriver_pv.abc",	
		/*GUN_RIOTCANNON_ID*/		"Models\\PV_Weapons\\RiotCannon_pv.abc",
		/*NOT USED1*/				DNULL,
		/*GUN_LITHKILLER_ID*/		"Models\\PV_Weapons\\LithKiller_pv.abc",
		/*GUN_BUZZSAW_ID*/		    "Models\\PV_Weapons\\buzzsaw_pv.abc", // DTM BUZZSAW MELEE WEAPON
		/*GUN_SHOULDERCANNONS_ID*/	"Models\\PV_Weapons\\ShoulderCannons_pv.abc",
		/*GUN_BIGBOOT_ID*/			"Models\\PV_Weapons\\BigBoot_pv.abc",
		/*GUN_SNIPERCLUB_ID*/		"Models\\PV_Weapons\\SniperRifleButt_pv.abc",
		/*GUN_HEALGUN_ID*/			"Models\\PV_Weapons\\healgun_pv.abc",
		/*GUN_MACHINEGUN_ID*/		"Models\\PV_Weapons\\MachineGun_pv.abc",
		/*GUN_ROCKETLAUNCHER*/		"Models\\PV_Weapons\\RocketLauncher_pv.abc",
		/*GUN_SICKGUN_ID*/			"Models\\PV_Weapons\\SickGun_pv.abc",
		/*GUN_BLINDER_ID*/			"Models\\PV_Weapons\\Blinder_pv.abc",
		/*GUN_BOMBS_ID*/			"Models\\PV_Weapons\\Bombs_pv.abc",
		/*GUN_RAMSET_ID*/			"Models\\PV_Weapons\\Ramset_pv.abc",
		/*GUN_SHOTGUN_ID*/			"Models\\PV_Weapons\\Shotgun_pv.abc",
		/*GUN_AXE_ID*/				"Models\\PV_Weapons\\Axe_pv.abc",
		/*GUN_COLT45_ID*/			"Models\\PV_Weapons\\Colt45_pv.abc",
		/*GUN_ASSAULTRIFLE_ID*/		"Models\\PV_Weapons\\AssaultRifle_pv.abc",
		/*GUN_ENERGYGRENADE_ID*/	"Models\\PV_Weapons\\EnergyGrenade_pv.abc",
		/*GUN_KATOGRENADE_ID*/		"Models\\PV_Weapons\\KatoGrenade_pv.abc",
		/*GUN_MAC10_ID*/			"Models\\PV_Weapons\\Machinegun_pv.abc",
		/*GUN_TOW_ID*/				"Models\\PV_Weapons\\TOW_pv.abc",
		/*NOT USED2*/				DNULL,
		/*GUN_SQUEAKYTOY_ID*/		"Models\\PV_Weapons\\squeakytoy_pv.abc",
	};

	return s_cPVModelName[nWeaponId];
}

inline char* GetPVModelSkin(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return DNULL;

	char* s_cPVModelSkin[GUN_MAX_NUMBER] =
	{
		/*GUN_PULSERIFLE_ID*/		"Skins\\weapons\\PulseRifle_pv_a.dtx",
		/*GUN_AUTOCANNONS_ID*/		"Skins\\weapons\\AutoCannons_pv_a.dtx",				
		/*GUN_DEATHCHARGE_ID*/		"Skins\\weapons\\DeathCharge_pv_a.dtx",
		/*GUN_PULSELAUNCHER_ID*/	"Skins\\weapons\\PulseLauncher_pv_a.dtx",
		/*GUN_SNIPERRIFLE_ID*/		"Skins\\weapons\\SniperRifle_pv_a.dtx",
		/*GUN_MASSDRIVER_ID*/		"Skins\\weapons\\MassDriver_pv_a.dtx",	
		/*GUN_RIOTCANNON_ID*/		"Skins\\weapons\\RiotCannon_pv_a.dtx",
		/*NOT USED1*/				DNULL,
		/*GUN_LITHKILLER_ID*/		"Skins\\weapons\\LithKiller_pv_a.dtx",
		/*GUN_BUZZSAW_ID*/			"Skins\\weapons\\buzzsaw_pv_a.dtx", // DTM BUZZSAW MELEE WEAPON
		/*GUN_SHOULDERCANNONS_ID*/	"Skins\\weapons\\ShoulderCannons_pv_a.dtx",
		/*GUN_BIGBOOT_ID*/			"Skins\\weapons\\BigBoot_pv_a.dtx",
		/*GUN_SNIPERCLUB_ID*/		"Skins\\weapons\\SniperRifleButt_pv_a.dtx",
		/*GUN_HEALGUN_ID*/			"Skins\\weapons\\healgun_pv_a.dtx", 
		/*GUN_MACHINEGUN_ID*/		"Skins\\weapons\\machinegun_pv_a.dtx",
		/*GUN_ROCKETLAUNCHER_ID*/	"Skins\\weapons\\RocketLauncher_pv_a.dtx",
		/*GUN_SICKGUN_ID*/			"Skins\\weapons\\SickGun_pv_a.dtx",
		/*GUN_BLINDER_ID*/			"Skins\\weapons\\Blinder_pv_a.dtx",
		/*GUN_BOMBS_ID*/			"Skins\\weapons\\Bombs_pv_a.dtx",
		/*GUN_RAMSET_ID*/			"Skins\\weapons\\Ramset_pv_a.dtx",
		/*GUN_SHOTGUN_ID*/			"Skins\\weapons\\Shotgun_pv_a.dtx",
		/*GUN_AXE_ID*/				"Skins\\weapons\\Axe_pv_a.dtx",
		/*GUN_COLT45_ID*/			"Skins\\weapons\\Colt45_pv_a.dtx",
		/*GUN_ASSAULTRIFLE_ID*/		"Skins\\weapons\\AssaultRifle_pv_a.dtx",
		/*GUN_ENERGYGRENADE_ID*/	"Skins\\weapons\\EnergyGrenade_pv_a.dtx",
		/*GUN_KATOGRENADE_ID*/		"Skins\\weapons\\KatoGrenade_pv_a.dtx",
		/*GUN_MAC10_ID*/			"Skins\\weapons\\Machinegun_pv_a.dtx",
		/*GUN_TOW_ID*/				"Skins\\weapons\\TOW_pv_a.dtx",
		/*NOT USED2*/				DNULL,
		/*GUN_SQUEAKYTOY_ID*/		"Skins\\weapons\\squeakytoy_pv_a.dtx",
	};

	return s_cPVModelSkin[nWeaponId];
}


// get the velocity of this weapon (projectiles only)...

inline DFLOAT GetWeaponVelocity(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0.0f;

	DFLOAT s_fVelocity[GUN_MAX_NUMBER] =
	{
		/*GUN_PULSERIFLE_ID*/		2500.0f,
		/*GUN_AUTOCANNONS_ID*/		2500.0f,				
		/*GUN_DEATHCHARGE_ID*/		400.0f,
		/*GUN_PULSELAUNCHER_ID*/	2000.0f, // DTM
		/*GUN_SNIPERRIFLE_ID*/		0.0f,
		/*GUN_MASSDRIVER_ID*/		0.0f,	
		/*GUN_RIOTCANNON_ID*/		0.0f,
		/*NOT USED1*/				0.0f,
		/*GUN_LITHKILLER_ID*/		500.0f,
		/*GUN_BUZZSAW_ID*/			0.0f,
		/*GUN_SHOULDERCANNONS_ID*/	2000.0f,  // DTM BUZZSAW MELEE WEAPON
		/*GUN_BIGBOOT_ID*/			0.0f,
		/*GUN_SNIPERCLUB_ID*/		0.0f,
		/*GUN_HEALGUN_ID*/			0.0f,
		/*GUN_MACHINEGUN_ID*/		0.0f,
		/*GUN_ROCKETLAUCNHER_ID*/	2000.0f,
		/*GUN_SICKGUN_ID*/			0.0f,
		/*GUN_BLINDER_ID*/			0.0f,
		/*GUN_BOMBS_ID*/			100.0f,
		/*GUN_RAMSET_ID*/			0.0f,
		/*GUN_SHOTGUN_ID*/			800.0f,
		/*GUN_AXE_ID*/				0.0f,
		/*GUN_COLT45_ID*/			0.0f,
		/*GUN_ASSAULTRIFLE_ID*/		0.0f,
		/*GUN_ENERGYGRENADE_ID*/	2000.0f,
		/*GUN_KATOGRENADE_ID*/		750.0f,
		/*GUN_MAC10_ID*/			0.0f,
		/*GUN_TOW_ID*/				2000.0f,
		/*NOT USED2*/				0.0f,
		/*GUN_SQUEAKYTOY_ID*/		0.0f,
	};

	return s_fVelocity[nWeaponId];
}


// get the lifetime of this projectile (projectiles only)...

inline DFLOAT GetWeaponLifeTime(int nWeaponId)
{
	if (nWeaponId < GUN_FIRST_ID || nWeaponId >= GUN_MAX_NUMBER) return 0.0f;

	DFLOAT s_fLifeTime[GUN_MAX_NUMBER] =
	{
		/*GUN_PULSERIFLE_ID*/		4.0f,
		/*GUN_AUTOCANNONS_ID*/		4.0f,				
		/*GUN_DEATHCHARGE_ID*/		5.0f,
		/*GUN_PULSELAUNCHER_ID*/	5.0f,
		/*GUN_SNIPERRIFLE_ID*/		0.0f,
		/*GUN_MASSDRIVER_ID*/		0.0f,	
		/*GUN_RIOTCANNON_ID*/		0.0f,
		/*NOT USED1*/				0.0f,
		/*GUN_LITHKILLER_ID*/		15.0f,
		/*GUN_BUZZSAW_ID*/			0.0f, // DTM BUZZSAW MELEE WEAPON
		/*GUN_SHOULDERCANNONS_ID*/	5.0f,
		/*GUN_BIGBOOT_ID*/			0.0f,
		/*GUN_SNIPERCLUB_ID*/		0.0f,
		/*GUN_HEALGUN_ID*/			0.0f,
		/*GUN_MACHINEGUN_ID*/		0.0f,
		/*GUN_ROCKETLAUNCHER_ID*/	3.0f,	
		/*GUN_SICKGUN_ID*/			0.0f,
		/*GUN_BLINDER_ID*/			0.0f,
		/*GUN_BOMBS_ID*/			4.0f,
		/*GUN_RAMSET_ID*/			0.0f,
		/*GUN_SHOTGUN_ID*/			0.0f,
		/*GUN_AXE_ID*/				0.0f,
		/*GUN_COLT45_ID*/			0.0f,
		/*GUN_ASSAULTRIFLE_ID*/		0.0f,
		/*GUN_ENERGYGRENADE_ID*/	5.0f,
		/*GUN_KATOGRENADE_ID*/		3.0f,
		/*GUN_MAC10_ID*/			0.0f,
		/*GUN_TOW_ID*/				3.0f,
		/*NOT USED2*/				0.0f,
		/*GUN_SQUEAKYTOY_ID*/		0.0f,
	};

	return s_fLifeTime[nWeaponId];
}


inline DBOOL CanWeaponZoom(int nWeaponId)
{
	if (nWeaponId == GUN_SNIPERRIFLE_ID || nWeaponId == GUN_ASSAULTRIFLE_ID) return DTRUE;

	return DFALSE;
}


// Calculate the adjusted path and fire position for the particular weapon...

void CalculateWeaponPathAndFirePos(RiotWeaponId nId, DVector & vPath, 
								   DVector & vFirePos, DVector & vU, DVector & vR);


#endif // __WEAPON_DEFS_H__