// ----------------------------------------------------------------------- //
//
// MODULE  : WeaponPowerups.cpp
//
// PURPOSE : Riot weapon powerups - Implementation
//
// CREATED : 10/7/97
//
// ----------------------------------------------------------------------- //

#include "WeaponPowerups.h"
#include "RiotMsgIds.h"
#include "cpp_server_de.h"
#include "WeaponDefs.h"
#include "BaseCharacter.h"

#define UPDATE_DELTA	0.1f

// ----------------------------------------------------------------------- //
// ----------------------------------------------------------------------- //
//
//	CLASS:		WeaponPowerup
//
//	PURPOSE:	Large Weapon powerups
//
// ----------------------------------------------------------------------- //
// ----------------------------------------------------------------------- //

BEGIN_CLASS(WeaponPowerup)
	ADD_BOOLPROP_FLAG(Rotate, 0, PF_HIDDEN)
	ADD_REALPROP(RespawnTime, 30.0f)
	ADD_STRINGPROP_FLAG(SoundFile, "", PF_HIDDEN)
	ADD_LONGINTPROP( WeaponType, 0 )
	ADD_LONGINTPROP( Ammo, 0 )
	ADD_BOOLPROP( Small, 0 )
	ADD_BOOLPROP( Large, 0 )
	ADD_VECTORPROP_VAL_FLAG( Dims, 10.0f, 25.0f, 10.0f, PF_HIDDEN | PF_DIMS )
END_CLASS_DEFAULT(WeaponPowerup, Powerup, NULL, NULL)


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	WeaponPowerup::WeaponPowerup
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

WeaponPowerup::WeaponPowerup() : Powerup()
{ 
	m_dwAmmo		= 0; 
	m_iWeaponType	= 0;
	m_eModelSize	= MS_NORMAL;

	m_bInformClient	= DFALSE;
	m_bBounce = DFALSE;

	AddAggregate(&m_bounce);
}



// ----------------------------------------------------------------------- //
//
//	ROUTINE:	WeaponPowerup::EngineMessageFn
//
//	PURPOSE:	Handle engine messages
//
// ----------------------------------------------------------------------- //

DDWORD WeaponPowerup::EngineMessageFn(DDWORD messageID, void *pData, DFLOAT fData)
{
	CServerDE* pServerDE = GetServerDE();
	if (!pServerDE) return DFALSE;

	switch(messageID)
	{
		case MID_UPDATE:
		{
			DDWORD dwRet = Powerup::EngineMessageFn(messageID, pData, fData);
			if (m_bounce.IsDoneBouncing())
			{
				pServerDE->SetNextUpdate(m_hObject, 0.0f);
			}
			else
			{
				pServerDE->SetNextUpdate(m_hObject, UPDATE_DELTA);
			}

			return dwRet;
		}
		break;

		case MID_PRECREATE:
		{
			DDWORD dwRet = Powerup::EngineMessageFn(messageID, pData, fData);

			if ( fData == 1.0f || fData == 2.0f )
				ReadProp(( ObjectCreateStruct * )pData );

			PostPropRead(( ObjectCreateStruct * )pData );

			return dwRet;
		}
		break;

		case MID_INITIALUPDATE:
		{
			DDWORD dwRet = Powerup::EngineMessageFn(messageID, pData, fData);

			if (fData != INITIALUPDATE_SAVEGAME)
			{
				InitialUpdate(pServerDE);
			}

			return dwRet;
		}

		case MID_SAVEOBJECT:
		{
			Save((HMESSAGEWRITE)pData, (DDWORD)fData);
		}
		break;

		case MID_LOADOBJECT:
		{
			Load((HMESSAGEREAD)pData, (DDWORD)fData);
		}
		break;

		default : break;
	}

	return Powerup::EngineMessageFn(messageID, pData, fData);
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	WeaponPowerup::ReadProp
//
//	PURPOSE:	Read object properties
//
// ----------------------------------------------------------------------- //
void WeaponPowerup::ReadProp( ObjectCreateStruct *pStruct )
{
	GenericProp genProp;

	if( g_pServerDE->GetPropGeneric( "Ammo", &genProp ) == DE_OK )
		m_dwAmmo = ( DDWORD )genProp.m_Long;
	if( g_pServerDE->GetPropGeneric( "WeaponType", &genProp ) == DE_OK)
		m_iWeaponType = ( DBYTE )genProp.m_Long;
	if( g_pServerDE->GetPropGeneric( "Small", &genProp ) == DE_OK)
		m_eModelSize = genProp.m_Bool ? MS_SMALL : m_eModelSize;
	if( g_pServerDE->GetPropGeneric( "Large", &genProp ) == DE_OK)
		m_eModelSize = genProp.m_Bool ? MS_LARGE : m_eModelSize;
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	WeaponPowerup::PostPropRead
//
//	PURPOSE:	Handle post property read engine messages
//
// ----------------------------------------------------------------------- //
void WeaponPowerup::PostPropRead( ObjectCreateStruct *pStruct )
{
	if (pStruct)
	{
		switch( m_iWeaponType )
		{
			case GUN_PULSERIFLE_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\PulseRifle.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\PulseRifle_a.dtx");
				break;
			}
			case GUN_DEATHCHARGE_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\DeathCharge.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\DeathCharge_a.dtx");
				break;
			}
			case GUN_PULSELAUNCHER_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\PulseLauncher.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\PulseLauncher_a.dtx");
				break;
			}
			case GUN_SNIPERRIFLE_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\SniperRifle.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\SniperRifle_a.dtx");
				break;
			}
			case GUN_MASSDRIVER_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\MassDriver.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\MassDriver_a.dtx");
				break;
			}
			case GUN_RIOTCANNON_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\RiotCannon.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\RiotCannon_a.dtx");
				break;
			}
			case GUN_LITHKILLER_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\LithKiller.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\LithKiller_a.dtx");
				break;
			}

			case GUN_COLT45_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Colt45.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Colt45_a.dtx");
				break;
			}
			case GUN_SHOTGUN_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Shotgun.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Shotgun_a.dtx");
				break;
			}
			case GUN_ASSAULTRIFLE_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\AssaultRifle.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\AssaultRifle_a.dtx");
				break;
			}
			case GUN_ENERGYGRENADE_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\EnergyGrenade.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\EnergyGrenade_a.dtx");
				break;
			}
			case GUN_BOMBS_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Bombs.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Bombs_a.dtx");
				break;
			}
			case GUN_KATOGRENADE_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\KatoGrenade.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\KatoGrenade_a.dtx");
				break;
			}
			case GUN_MAC10_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Machinegun.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Machinegun_a.dtx");
				break;
			}
			case GUN_TOW_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\TOW.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\TOW_a.dtx");
				break;
			}
			case GUN_MACHINEGUN_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\MachineGun.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\MachineGun_a.dtx");
				break;
			}
			case GUN_ROCKETLAUNCHER_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\RocketLauncher.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\RocketLauncher_a.dtx");
				break;
			}
			case GUN_AUTOCANNONS_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\autocannons.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\autocannons_a.dtx");
				break;
			}
			case GUN_HEALGUN_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\healgun.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\healgun_a.dtx");
				break;
			}
			case GUN_AXE_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\axe.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\axe_a.dtx");
				break;
			}
			case GUN_SICKGUN_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\sickgun.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\sickgun_a.dtx");
				break;
			}
			case GUN_BLINDER_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\blinder.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\blinder_a.dtx");
				break;
			}
			case GUN_SQUEAKYTOY_ID:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\SqueakyToy.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\SqueakyToy_a.dtx");
				break;
			}
			default:
			{
				SAFE_STRCPY(pStruct->m_Filename, "Models\\powerups\\Colt45.abc");
				SAFE_STRCPY(pStruct->m_SkinName, "Skins\\powerups\\Colt45_a.dtx");
				break;
			}
		}


		// Set up the appropriate pick up sound...

		if (m_hstrSoundFile) 
		{
			g_pServerDE->FreeString(m_hstrSoundFile);
		}

		if (GUN_FIRSTMECH_ID <= m_iWeaponType && m_iWeaponType <= GUN_LASTMECH_ID)
		{
			m_hstrSoundFile = g_pServerDE->CreateString("Sounds\\Powerups\\Weapon_mca.wav");
		}
		else
		{
			m_hstrSoundFile = g_pServerDE->CreateString("Sounds\\Powerups\\Weapon_onfoot.wav");
		}

		m_bBounce = DFALSE;
		m_bRotate = DFALSE;
	}
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	WeaponPowerup::InitialUpdate
//
//	PURPOSE:	Handle inital update engine messages
//
// ----------------------------------------------------------------------- //

void WeaponPowerup::InitialUpdate( CServerDE *pServerDE )
{
	DVector vDims, vScale, vNewDims;

	pServerDE->GetModelAnimUserDims(m_hObject, &vDims, pServerDE->GetModelAnimation(m_hObject));

	vScale = GetHandWeaponScale((RiotWeaponId)m_iWeaponType, m_eModelSize);
	pServerDE->ScaleObject(m_hObject, &vScale);

	// Set object dims based on scale value...

	vNewDims.x = vScale.x * vDims.x;
	vNewDims.y = vScale.y * vDims.y;
	vNewDims.z = vScale.z * vDims.z;

	g_pServerDE->SetObjectDims(m_hObject, &vNewDims);

	pServerDE->SetNextUpdate(m_hObject, UPDATE_DELTA);
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	WeaponPowerup::ObjectTouch
//
//	PURPOSE:	Add weapon powerup to object
//
// ----------------------------------------------------------------------- //

void WeaponPowerup::ObjectTouch(HOBJECT hObject)
{
	CServerDE* pServerDE = GetServerDE();
	if (!pServerDE || !hObject) return;

	// If we hit non-character objects, just ignore them...

	HCLASS hCharClass = pServerDE->GetClass("CBaseCharacter");
	HCLASS hObjClass  = pServerDE->GetObjectClass(hObject);

	if (pServerDE->IsKindOf(hObjClass, hCharClass))
	{
		CBaseCharacter* pCharObj = (CBaseCharacter*)pServerDE->HandleToObject(hObject);

		if (pCharObj && !pCharObj->IsDead() && pCharObj->CanCarryWeapon(m_iWeaponType))
		{
			HMESSAGEWRITE hMessage = pServerDE->StartMessageToObject(this, hObject, MID_ADDWEAPON);
			pServerDE->WriteToMessageByte( hMessage, m_iWeaponType );
			pServerDE->WriteToMessageFloat( hMessage, ( float )m_dwAmmo );
			pServerDE->EndMessage(hMessage);
		}
	}
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	WeaponPowerup::Save
//
//	PURPOSE:	Save the object
//
// ----------------------------------------------------------------------- //

void WeaponPowerup::Save(HMESSAGEWRITE hWrite, DDWORD dwSaveFlags)
{
	CServerDE* pServerDE = GetServerDE();
	if (!pServerDE || !hWrite) return;

	pServerDE->WriteToMessageByte(hWrite, m_iWeaponType);
	pServerDE->WriteToMessageByte(hWrite, m_eModelSize);
	pServerDE->WriteToMessageDWord(hWrite, m_dwAmmo);
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	WeaponPowerup::Load
//
//	PURPOSE:	Load the object
//
// ----------------------------------------------------------------------- //

void WeaponPowerup::Load(HMESSAGEREAD hRead, DDWORD dwLoadFlags)
{
	CServerDE* pServerDE = GetServerDE();
	if (!pServerDE || !hRead) return;

	m_iWeaponType	= pServerDE->ReadFromMessageByte(hRead);
	m_eModelSize	= (ModelSize) pServerDE->ReadFromMessageByte(hRead);
	m_dwAmmo		= pServerDE->ReadFromMessageDWord(hRead);

	// Set our object to the correct filenames...

	ObjectCreateStruct theStruct;
	INIT_OBJECTCREATESTRUCT(theStruct)
	PostPropRead(&theStruct);

	pServerDE->SetModelFilenames(m_hObject, theStruct.m_Filename, 
								 theStruct.m_SkinName);
}



BEGIN_CLASS(PulseRiflePowerup)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_PULSERIFLE_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetSpawnedAmmo(GUN_PULSERIFLE_ID))
END_CLASS_DEFAULT(PulseRiflePowerup, WeaponPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	PulseRiflePowerup::PulseRiflePowerup
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

PulseRiflePowerup::PulseRiflePowerup() : WeaponPowerup()
{ 
	m_dwAmmo		= GetSpawnedAmmo(GUN_PULSERIFLE_ID); 
	m_iWeaponType	= GUN_PULSERIFLE_ID;
}

BEGIN_CLASS(DeathChargePowerup)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_DEATHCHARGE_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetSpawnedAmmo(GUN_DEATHCHARGE_ID))
END_CLASS_DEFAULT(DeathChargePowerup, WeaponPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	DeathChargePowerup::DeathChargePowerup
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

DeathChargePowerup::DeathChargePowerup() : WeaponPowerup()
{ 
	m_dwAmmo		= GetSpawnedAmmo(GUN_DEATHCHARGE_ID); 
	m_iWeaponType	= GUN_DEATHCHARGE_ID;
}

BEGIN_CLASS(PulseLauncherPowerup)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_PULSELAUNCHER_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetSpawnedAmmo(GUN_PULSELAUNCHER_ID))
END_CLASS_DEFAULT(PulseLauncherPowerup, WeaponPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	PulseLauncherPowerup::PulseLauncherPowerup
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

PulseLauncherPowerup::PulseLauncherPowerup() : WeaponPowerup()
{ 
	m_dwAmmo		= GetSpawnedAmmo(GUN_PULSELAUNCHER_ID); 
	m_iWeaponType	= GUN_PULSELAUNCHER_ID;
}

BEGIN_CLASS(SniperRiflePowerup)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_SNIPERRIFLE_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetSpawnedAmmo(GUN_SNIPERRIFLE_ID))
END_CLASS_DEFAULT(SniperRiflePowerup, WeaponPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	SniperRiflePowerup::SniperRiflePowerup
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

SniperRiflePowerup::SniperRiflePowerup() : WeaponPowerup()
{ 
	m_dwAmmo		= GetSpawnedAmmo(GUN_SNIPERRIFLE_ID); 
	m_iWeaponType	= GUN_SNIPERRIFLE_ID;
}

BEGIN_CLASS(MassDriverPowerup)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_MASSDRIVER_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetSpawnedAmmo(GUN_MASSDRIVER_ID))
END_CLASS_DEFAULT(MassDriverPowerup, WeaponPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	MassDriverPowerup::MassDriverPowerup
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

MassDriverPowerup::MassDriverPowerup() : WeaponPowerup()
{ 
	m_dwAmmo		= GetSpawnedAmmo(GUN_MASSDRIVER_ID); 
	m_iWeaponType	= GUN_MASSDRIVER_ID;
}

BEGIN_CLASS(RiotCannonPowerup)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_RIOTCANNON_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetSpawnedAmmo(GUN_RIOTCANNON_ID))
END_CLASS_DEFAULT(RiotCannonPowerup, WeaponPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	RiotCannonPowerup::RiotCannonPowerup
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

RiotCannonPowerup::RiotCannonPowerup() : WeaponPowerup()
{ 
	m_dwAmmo		= GetSpawnedAmmo(GUN_RIOTCANNON_ID); 
	m_iWeaponType	= GUN_RIOTCANNON_ID;
}

BEGIN_CLASS(AutoCannonsPowerup)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_AUTOCANNONS_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetSpawnedAmmo(GUN_AUTOCANNONS_ID))
END_CLASS_DEFAULT(AutoCannonsPowerup, WeaponPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	AutoCannonsPowerup::AutoCannonsPowerup
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

AutoCannonsPowerup::AutoCannonsPowerup() : WeaponPowerup()
{ 
	m_dwAmmo		= GetSpawnedAmmo(GUN_AUTOCANNONS_ID); 
	m_iWeaponType	= GUN_AUTOCANNONS_ID;
}

BEGIN_CLASS(HealGunPowerup)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_HEALGUN_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetSpawnedAmmo(GUN_HEALGUN_ID))
END_CLASS_DEFAULT(HealGunPowerup, WeaponPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	HealGunPowerup::HealGunPowerup
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

HealGunPowerup::HealGunPowerup() : WeaponPowerup()
{ 
	m_dwAmmo		= GetSpawnedAmmo(GUN_HEALGUN_ID); 
	m_iWeaponType	= GUN_HEALGUN_ID;
}

BEGIN_CLASS(SickGunPowerup)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_SICKGUN_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetSpawnedAmmo(GUN_SICKGUN_ID))
END_CLASS_DEFAULT(SickGunPowerup, WeaponPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	SickGunPowerup::SickGunPowerup
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

SickGunPowerup::SickGunPowerup() : WeaponPowerup()
{ 
	m_dwAmmo		= GetSpawnedAmmo(GUN_SICKGUN_ID); 
	m_iWeaponType	= GUN_SICKGUN_ID;
}

BEGIN_CLASS(BlinderPowerup)
	ADD_LONGINTPROP_FLAG(WeaponType, GUN_BLINDER_ID, PF_HIDDEN)
	ADD_LONGINTPROP(Ammo, GetSpawnedAmmo(GUN_BLINDER_ID))
END_CLASS_DEFAULT(BlinderPowerup, WeaponPowerup, NULL, NULL)

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	BlinderPowerup::BlinderPowerup
//
//	PURPOSE:	Constructor
//
// ----------------------------------------------------------------------- //

BlinderPowerup::BlinderPowerup() : WeaponPowerup()
{ 
	m_dwAmmo		= GetSpawnedAmmo(GUN_BLINDER_ID); 
	m_iWeaponType	= GUN_BLINDER_ID;
}
