// ----------------------------------------------------------------------- //
//
// MODULE  : PlayerMode.cpp
//
// PURPOSE : PlayerObj helper class - Implementation
//
// CREATED : 9/18/97
//
// ----------------------------------------------------------------------- //

#include "PlayerMode.h"
#include "PlayerObj.h"
#include "cpp_server_de.h"
#include "RiotMsgIds.h"
#include "CharacterAttributes.h"
#include "RiotObjectUtilities.h"
#include "RiotServerShell.h"

// Used to set camera offsets...

static DVector s_vOnfootCamera;
static DVector s_vKidCamera;
//DTM
static DVector s_vMcaHeavywCamera;
static DVector s_vMcaRocketeerCamera;
static DVector s_vMcaDemomanCamera;
static DVector s_vMcaSniperCamera;
static DVector s_vMcaHeavywVehicleCamera;
static DVector s_vMcaRocketeerVehicleCamera;
static DVector s_vMcaDemomanVehicleCamera;
static DVector s_vMcaSniperVehicleCamera;
//DTM
extern CRiotServerShell* g_pRiotServerShellDE;
extern CVarTrack g_RammingDamageTrack;

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::CPlayerMode
//
//	PURPOSE:	Initialize the object
//
// ----------------------------------------------------------------------- //

CPlayerMode::CPlayerMode()
{
	// Initialize statics...

	VEC_SET(s_vOnfootCamera, 0.0, 34.0, 0.0);
	VEC_SET(s_vKidCamera, 0.0, 16.0, 0.0);
//DTM
	VEC_SET(s_vMcaHeavywCamera, 0.0, 28.0, 0.0);
	VEC_SET(s_vMcaRocketeerCamera, 0.0, 28.0, 0.0);
	VEC_SET(s_vMcaDemomanCamera, 0.0, 28.0, 0.0);
	VEC_SET(s_vMcaSniperCamera, 0.0, 28.0, 0.0);
	VEC_SET(s_vMcaHeavywVehicleCamera, 0.0, 10.0, 0.0);
	VEC_SET(s_vMcaRocketeerVehicleCamera, 0.0, 10.0, 0.0);
	VEC_SET(s_vMcaDemomanVehicleCamera, 0.0, 10.0, 0.0);
	VEC_SET(s_vMcaSniperVehicleCamera, 0.0, 10.0, 0.0);
//DTM
	m_pMyObj			= DNULL;

	m_nModelId			= MI_PLAYER_ONFOOT_ID;

	m_nMode				= PM_MODE_UNDEFINED;
	m_bBipedal			= DFALSE;
	m_fWalkVel			= PM_FOOT_WALKSPEED;
	m_fRunVel			= PM_FOOT_RUNSPEED;
	m_fJumpSpeed		= PM_FOOT_JUMPSPEED;

	m_nArmor			= PM_MCA_MOD_NONE;
	m_nVehicleHandling	= PM_MCA_MOD_NONE;

	m_FovX				= MATH_HALFPI;
	m_FovY				= m_FovX / float(640.0 / 480.0);

	VEC_INIT(m_vCameraOffset);	

	// SetMode(PM_MODE_UNDEFINED, DTRUE);
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::SetMode
//
//	PURPOSE:	Set the current mode
//
//	@cmember change the player mode
//	@parm the new mode
//	@parm is it a bipedal mode
//	@rdef the old mode
//
// ----------------------------------------------------------------------- //

DBYTE CPlayerMode::SetMode(DBYTE nNewMode, DBOOL bBipedal)
{
	int nOldMode = m_nMode;
		
	if (PM_MODE_FOOT <= nNewMode && nNewMode <= PM_MODE_KID)
	{
		m_nMode = nNewMode;
	}
	else
	{
		return m_nMode;
	}

	switch(nNewMode)
	{
// DTM
		case PM_MODE_MCA_HEAVYW : SetModeMcaHeavyw(bBipedal); break;
		case PM_MODE_MCA_ROCKETEER : SetModeMcaRocketeer(bBipedal); break;
		case PM_MODE_MCA_DEMOMAN : SetModeMcaDemoman(bBipedal); break;
		case PM_MODE_MCA_SNIPER : SetModeMcaSniper(bBipedal); break;
// DTM
		case PM_MODE_KID    : SetModeKid(); break;

		case PM_MODE_FOOT :	  
		default :			  SetModeOnFoot(); break;
	}


	// Update the Camera FOV to reflect the new player mode...

	CServerDE* pServerDE = BaseClass::GetServerDE();

	if (pServerDE && m_pMyObj)
	{
		// Update client's mode info...
		HMESSAGEWRITE hMessage = pServerDE->StartMessage(m_pMyObj->GetClient(), MID_PLAYER_MODECHANGE);
		pServerDE->WriteToMessageVector(hMessage, &m_vCameraOffset);
		pServerDE->WriteToMessageByte(hMessage, m_nMode);
		pServerDE->WriteToMessageByte(hMessage, !IsBipedalMode());
		pServerDE->WriteToMessageFloat(hMessage, m_FovX);
		pServerDE->WriteToMessageFloat(hMessage, m_FovY);
		pServerDE->EndMessage(hMessage);
	}


	return nOldMode;
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetRunVeloctiy
//
//	PURPOSE:	Get the maximum velocity when running
//
// @cmember get the maximum velocity when running
// @rdef the maximum run velocity
//
// ----------------------------------------------------------------------- //

DFLOAT CPlayerMode::GetRunVelocity() const
{
	return m_fRunVel;
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetWalkVelocity
//
//	PURPOSE:	Get the maximum velocity when walking
//
// @cmember get the maximum velocity when walking
// @rdef the maximum run velocity
//
// ----------------------------------------------------------------------- //

DFLOAT CPlayerMode::GetWalkVelocity() const
{
	return m_fWalkVel;
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::AdjustFriction
//
//	PURPOSE:	Adjust the friction based on the current mode
//
// @cmember adjust the friction based on the current mode
// @rdef the base friction coefficient
//
// ----------------------------------------------------------------------- //

float CPlayerMode::AdjustFriction(float fBaseFriction) const
{
	float fNewFriction = fBaseFriction;

	if (m_nMode == PM_MODE_FOOT)
	{
		fNewFriction = 5.0;
	}
	else if (m_bBipedal)
	{
		fNewFriction = 5.0;
	}
	else
	{
		double fFrictionAdjust[4] = { 1.0, 0.85, 1.0, 1.15 };
		fNewFriction = float(fBaseFriction * fFrictionAdjust[m_nVehicleHandling]);
	}

	return fNewFriction;
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::AdjustArmor
//
//	PURPOSE:	Adjust the armor based on the current mode
//
// @cmember adjust the armor based on the current mode
// @rdef the base armor value
//
// ----------------------------------------------------------------------- //

DFLOAT CPlayerMode::AdjustArmor(DFLOAT fBaseArmor) const
{
	DFLOAT fNewArmor = fBaseArmor;

	switch(m_nArmor)
	{
		case PM_MCA_MOD_NONE :
			fNewArmor = fBaseArmor;
		break;

		case PM_MCA_MOD_1 :
			fNewArmor = fBaseArmor * (DFLOAT)0.85;
		break;

		case PM_MCA_MOD_2 :
			fNewArmor = fBaseArmor;
		break;

		case PM_MCA_MOD_3 :
			if (fBaseArmor <= 0.0) fBaseArmor = 1;
			fNewArmor = fBaseArmor * (DFLOAT)1.15;
		break;
		
		default :			 
			fNewArmor = fBaseArmor;
	}

	return fNewArmor;
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::SetModeOnFoot
//
//	PURPOSE:	Change the player mode to on-foot
//
// @cmember change the player mode to on foot
//
// ----------------------------------------------------------------------- //

void CPlayerMode::SetModeOnFoot()
{
	CServerDE* pServerDE = BaseClass::GetServerDE();
	if (!pServerDE || !g_pRiotServerShellDE) return;

	m_nModelId = MI_PLAYER_ONFOOT_ID;

	// Make Multiplayer on-foot speed faster...

	GameType eGameType = g_pRiotServerShellDE->GetGameType();
	DFLOAT fVal = (eGameType == SINGLE ) ? 1.0f : 1.25f;

	m_fWalkVel		= (DFLOAT)(PM_FOOT_WALKSPEED * fVal);
	m_fRunVel		= (DFLOAT)(PM_FOOT_RUNSPEED * fVal);
	m_fJumpSpeed	= PM_FOOT_JUMPSPEED;

	VEC_COPY(m_vCameraOffset, s_vOnfootCamera);

	m_bBipedal			= DTRUE;
	m_nArmor			= PM_MCA_MOD_NONE;
	m_nVehicleHandling	= PM_MCA_MOD_NONE;

	m_FovX				= MATH_HALFPI;
	m_FovY				= m_FovX / float(640.0 / 480.0);
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::SetModeKid
//
//	PURPOSE:	Change the player mode to kid
//
// @cmember change the player mode to kid
//
// ----------------------------------------------------------------------- //

void CPlayerMode::SetModeKid()
{
	CServerDE* pServerDE = BaseClass::GetServerDE();
	if (!pServerDE || !g_pRiotServerShellDE) return;

	m_nModelId = MI_PLAYER_KID_ID;

	// Make Multiplayer on-foot speed faster...

	GameType eGameType = g_pRiotServerShellDE->GetGameType();
	DFLOAT fVal = (eGameType == SINGLE ) ? 1.0f : 1.25f;

	m_fWalkVel		= (DFLOAT)(PM_FOOT_WALKSPEED * fVal);
	m_fRunVel		= (DFLOAT)(PM_FOOT_RUNSPEED * fVal);
	m_fJumpSpeed	= PM_FOOT_JUMPSPEED;
	
	VEC_COPY(m_vCameraOffset, s_vKidCamera);

	m_bBipedal			= DTRUE;
	m_nArmor			= PM_MCA_MOD_NONE;
	m_nVehicleHandling	= PM_MCA_MOD_NONE;

	m_FovX				= MATH_HALFPI;
	m_FovY				= m_FovX / float(640.0 / 480.0);
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::SetModeMcaHeavyw
//
//	PURPOSE:	Change the player mode to Andra Heavyw
//
// @cmember change the player mode to Andra Heavyw
// @parm bipedal mode
//
// ----------------------------------------------------------------------- //

void CPlayerMode::SetModeMcaHeavyw(DBOOL bBipedal)
{
	m_bBipedal			= bBipedal;

	m_nArmor			= PM_MCA_MOD_3;
	m_nVehicleHandling	= PM_MCA_MOD_3;

	m_nModelId = MI_PLAYER_HEAVYW_ID;

	if (bBipedal)
	{
		m_fRunVel	 = PM_MCA_BSPEED_1;
		m_fWalkVel   = PM_MCA_BSPEED_1;
		m_fJumpSpeed = PM_MCA_JSPEED_1;
		m_FovX		 = MATH_HALFPI;
		m_FovY		 = m_FovX / float(640.0 / 480.0);

		VEC_COPY(m_vCameraOffset, s_vMcaHeavywCamera);
	}
	else
	{
		m_fRunVel	 = PM_MCA_VSPEED_1;
		m_fWalkVel   = PM_MCA_VSPEED_1;
		m_fJumpSpeed = PM_MCA_JSPEED_1;
		m_FovX		 = float(MATH_PI * .6);
		m_FovY		 = m_FovX / float(640.0 / 480.0);

		VEC_COPY(m_vCameraOffset, s_vMcaHeavywVehicleCamera);
	}
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::SetModeMcaRocketeer
//
//	PURPOSE:	Change the player mode to UCA Rocketeer
//
// @cmember change the player mode to UCA Rocketeer
// @parm bipedal mode
//
// ----------------------------------------------------------------------- //

void CPlayerMode::SetModeMcaRocketeer(DBOOL bBipedal)
{
	CServerDE* pServerDE = BaseClass::GetServerDE();
	if (!pServerDE) return;

	m_bBipedal			= bBipedal;

	m_nArmor			= PM_MCA_MOD_2;
	m_nVehicleHandling	= PM_MCA_MOD_2;

	m_nModelId = MI_PLAYER_ROCKETEER_ID;

	if (bBipedal)
	{
		m_fRunVel	 = PM_MCA_BSPEED_3;
		m_fWalkVel   = PM_MCA_BSPEED_3;
		m_fJumpSpeed = PM_MCA_JSPEED_2;
		m_FovX		 = MATH_HALFPI;
		m_FovY		 = m_FovX / float(640.0 / 480.0);

		VEC_COPY(m_vCameraOffset, s_vMcaRocketeerCamera);
	}
	else
	{
		m_fRunVel	 = PM_MCA_VSPEED_2;
		m_fWalkVel   = PM_MCA_VSPEED_2;
		m_fJumpSpeed = PM_MCA_JSPEED_2;
		m_FovX		 = float(MATH_PI * .6);
		m_FovY		 = m_FovX / float(640.0 / 480.0);
		
		VEC_COPY(m_vCameraOffset, s_vMcaRocketeerVehicleCamera);
	}
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::SetModeMcaDemoman
//
//	PURPOSE:	Change the player mode to Armacham Demoman
//
// @cmember change the player mode to Armacham Demoman
// @parm bipedal mode
//
// ----------------------------------------------------------------------- //

void CPlayerMode::SetModeMcaDemoman(DBOOL bBipedal)
{
	CServerDE* pServerDE = BaseClass::GetServerDE();
	if (!pServerDE) return;

	m_bBipedal			= bBipedal;
	m_nArmor			= PM_MCA_MOD_2;
	m_nVehicleHandling	= PM_MCA_MOD_1;

	m_nModelId = MI_PLAYER_DEMOMAN_ID;

	if (bBipedal)
	{
		m_fRunVel	 = PM_MCA_BSPEED_2;
		m_fWalkVel   = PM_MCA_BSPEED_2;
		m_fJumpSpeed = PM_MCA_JSPEED_2;
		m_FovX		 = MATH_HALFPI;
		m_FovY		 = m_FovX / float(640.0 / 480.0);

		VEC_COPY(m_vCameraOffset, s_vMcaDemomanCamera);
	}
	else
	{
		m_fRunVel	 = PM_MCA_VSPEED_3;
		m_fWalkVel   = PM_MCA_VSPEED_3;
		m_fJumpSpeed = PM_MCA_JSPEED_2;
		m_FovX		 = float(MATH_PI * .6);
		m_FovY		 = m_FovX / float(640.0 / 480.0);

		VEC_COPY(m_vCameraOffset, s_vMcaDemomanVehicleCamera);
	}
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::SetModeMcaSniper
//
//	PURPOSE:	Change the player mode to Shogo Sniper
//
// @cmember change the player mode to Shogo Sniper
// @parm bipedal mode
//
// ----------------------------------------------------------------------- //

void CPlayerMode::SetModeMcaSniper(DBOOL bBipedal)
{
	CServerDE* pServerDE = BaseClass::GetServerDE();
	if (!pServerDE) return;

	m_bBipedal			= bBipedal;
	m_nArmor			= PM_MCA_MOD_1;
	m_nVehicleHandling	= PM_MCA_MOD_3;

	m_nModelId = MI_PLAYER_SNIPER_ID;

	if (bBipedal)
	{
		m_fRunVel	 = PM_MCA_BSPEED_2;
		m_fWalkVel   = PM_MCA_BSPEED_2;
		m_fJumpSpeed = PM_MCA_JSPEED_3;
		m_FovX		 = MATH_HALFPI;
		m_FovY		 = m_FovX / float(640.0 / 480.0);

		VEC_COPY(m_vCameraOffset, s_vMcaSniperCamera);
	}
	else
	{
		m_fRunVel	 = PM_MCA_VSPEED_3;
		m_fWalkVel   = PM_MCA_VSPEED_3;
		m_fJumpSpeed = PM_MCA_JSPEED_3;
		m_FovX		 = float(MATH_PI * .6);
		m_FovY		 = m_FovX / float(640.0 / 480.0);

		VEC_COPY(m_vCameraOffset, s_vMcaSniperVehicleCamera);
	}
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::IsOnFoot
//
//	PURPOSE:	Determine if player is on foot
//
// ----------------------------------------------------------------------- //

DBOOL CPlayerMode::IsOnFoot() const
{
	return (m_nMode == PM_MODE_FOOT || m_nMode == PM_MODE_KID);
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::SetCameraOffset
//
//	PURPOSE:	Set the player's camera offset
//
// ----------------------------------------------------------------------- //

void CPlayerMode::SetCameraOffset(DVector vOffset)
{
	// Save the offset for future updates...

	switch(m_nMode)
	{
//DTM
	case PM_MODE_MCA_HEAVYW :
			if (m_bBipedal) 
			{
				VEC_COPY(s_vMcaHeavywCamera, vOffset);
			}
			else 
			{
				VEC_COPY(s_vMcaHeavywVehicleCamera, vOffset);
			}
		break;

		case PM_MODE_MCA_ROCKETEER :
			if (m_bBipedal)
			{
				VEC_COPY(s_vMcaRocketeerCamera, vOffset);
			}
			else 
			{
				VEC_COPY(s_vMcaRocketeerVehicleCamera, vOffset);
			}
		break;

		case PM_MODE_MCA_DEMOMAN : 
			if (m_bBipedal) 
			{
				VEC_COPY(s_vMcaDemomanCamera, vOffset);
			}
			else 
			{
				VEC_COPY(s_vMcaDemomanVehicleCamera, vOffset);
			}
		break;

		case PM_MODE_MCA_SNIPER : 
			if (m_bBipedal) 
			{
				VEC_COPY(s_vMcaSniperCamera, vOffset);
			}
			else 
			{
				VEC_COPY(s_vMcaSniperVehicleCamera, vOffset);
			}
		break;
// DTM
		case PM_MODE_FOOT :	  
			VEC_COPY(s_vOnfootCamera, vOffset);
		break;

		case PM_MODE_KID:	  
			VEC_COPY(s_vKidCamera, vOffset);
		break;

		default : break;
	}

	VEC_COPY(m_vCameraOffset, vOffset);
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetMass
//
//	PURPOSE:	Set the player's mass
//
// ----------------------------------------------------------------------- //

DFLOAT CPlayerMode::GetMass()
{
	DFLOAT fRet;

	switch(m_nMode)
	{
//DTM
		case PM_MODE_MCA_HEAVYW :
		case PM_MODE_MCA_ROCKETEER :
		case PM_MODE_MCA_DEMOMAN : 
		case PM_MODE_MCA_SNIPER : 
			fRet = m_bBipedal ? CA_PLAYER_MECHA_MASS : (g_RammingDamageTrack.GetFloat(1.0f) < 1.0f ? CA_PLAYER_MECHA_MASS : CA_PLAYER_MECHA_MASS * 1.5f);
		break;
//DTM
		case PM_MODE_FOOT :	  
		case PM_MODE_KID :	  
		default : 
			fRet = CA_PLAYER_ONFOOT_MASS;
		break;
	}

	return fRet;
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetSkinFilename
//
//	PURPOSE:	Set the player's skin filename
//
// ----------------------------------------------------------------------- //

char* CPlayerMode::GetSkinFilename() const 
{
	DBOOL bMulti = (g_pRiotServerShellDE->GetGameType() != SINGLE);
	return GetSkin(m_nModelId, UCA, MS_NORMAL, bMulti); 
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetMaxHitPts
//
//	PURPOSE:	Set the player's mass
//
// ----------------------------------------------------------------------- //

DFLOAT CPlayerMode::GetMaxHitPts()
{
	DFLOAT fRet;

	switch(m_nMode)
	{
//DTM
		case PM_MODE_MCA_SNIPER : 
			fRet = CA_PLAYER_SNIPER_MAX_HITPTS;
		break;

		case PM_MODE_MCA_HEAVYW :
			fRet = CA_PLAYER_HEAVYW_MAX_HITPTS;
		break;

		case PM_MODE_MCA_DEMOMAN : 
			fRet = CA_PLAYER_DEMOMAN_MAX_HITPTS;
		break;

		case PM_MODE_MCA_ROCKETEER :
			fRet = CA_PLAYER_ROCKETEER_MAX_HITPTS;
		break;
//DTM
		case PM_MODE_KID :	  
			fRet = CA_PLAYER_KID_MAX_HITPTS;
		break;

		case PM_MODE_FOOT :	  
		default : 
			fRet = CA_PLAYER_ONFOOT_MAX_HITPTS;
		break;
	}

	return fRet;
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetBaseHitPts
//
//	PURPOSE:	Set the player's starting hit points
//
// ----------------------------------------------------------------------- //

DFLOAT CPlayerMode::GetBaseHitPts()
{
	DFLOAT fRet;

	switch(m_nMode)
	{
//DTM
		case PM_MODE_MCA_SNIPER : 
			fRet = CA_PLAYER_SNIPER_HITPTS;
		break;

		case PM_MODE_MCA_HEAVYW :
			fRet = CA_PLAYER_HEAVYW_HITPTS;
		break;

		case PM_MODE_MCA_DEMOMAN : 
			fRet = CA_PLAYER_DEMOMAN_HITPTS;
		break;

		case PM_MODE_MCA_ROCKETEER :
			fRet = CA_PLAYER_ROCKETEER_HITPTS;
		break;
//DTM
		case PM_MODE_KID :	  
			fRet = CA_PLAYER_KID_HITPTS;
		break;

		case PM_MODE_FOOT :	  
		default : 
			fRet = CA_PLAYER_ONFOOT_HITPTS;
		break;
	}

	return fRet;
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetMaxArmorPts
//
//	PURPOSE:	Set the player's maximum armor points
//
// ----------------------------------------------------------------------- //

DFLOAT CPlayerMode::GetMaxArmorPts()
{
	DFLOAT fRet;

	switch(m_nMode)
	{
//DTM
		case PM_MODE_MCA_SNIPER : 
			fRet = CA_PLAYER_SNIPER_MAX_ARMOR;
		break;

		case PM_MODE_MCA_HEAVYW :
			fRet = CA_PLAYER_HEAVYW_MAX_ARMOR;
		break;

		case PM_MODE_MCA_DEMOMAN : 
			fRet = CA_PLAYER_DEMOMAN_MAX_ARMOR;
		break;

		case PM_MODE_MCA_ROCKETEER :
			fRet = CA_PLAYER_ROCKETEER_MAX_ARMOR;
		break;
//DTM
		case PM_MODE_KID :	  
			fRet = CA_PLAYER_KID_MAX_ARMOR;
		break;

		case PM_MODE_FOOT :	  
		default : 
			fRet = CA_PLAYER_ONFOOT_MAX_ARMOR;
		break;
	}

	return fRet;
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetBaseArmorPts
//
//	PURPOSE:	Set the player's starting armor points
//
// ----------------------------------------------------------------------- //

DFLOAT CPlayerMode::GetBaseArmorPts()
{
	DFLOAT fRet;

	switch(m_nMode)
	{
//DTM
		case PM_MODE_MCA_SNIPER : 
			fRet = CA_PLAYER_SNIPER_ARMOR;
		break;

		case PM_MODE_MCA_HEAVYW :
			fRet = CA_PLAYER_HEAVYW_ARMOR;
		break;

		case PM_MODE_MCA_DEMOMAN : 
			fRet = CA_PLAYER_DEMOMAN_ARMOR;
		break;

		case PM_MODE_MCA_ROCKETEER :
			fRet = CA_PLAYER_ROCKETEER_ARMOR;
		break;
//DTM
		case PM_MODE_KID :	  
			fRet = CA_PLAYER_KID_ARMOR;
		break;

		case PM_MODE_FOOT :	  
		default : 
			fRet = CA_PLAYER_ONFOOT_ARMOR;
		break;
	}

	return fRet;
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetDamageSound
//
//	PURPOSE:	Determine what damage sound to play
//
// ----------------------------------------------------------------------- //

char* CPlayerMode::GetDamageSound(DamageType eType)
{
	switch(m_nMode)
	{
		//DTM
		case PM_MODE_MCA_SNIPER : 
		case PM_MODE_MCA_HEAVYW :
		case PM_MODE_MCA_DEMOMAN : 
		case PM_MODE_MCA_ROCKETEER :
			return GetMechDamageSound(eType);
		break;
		//DTM
		case PM_MODE_KID :	  
			return GetKidDamageSound(eType);
		break;

		case PM_MODE_FOOT :	  
		default : 
			break;
		break;
	}

	return GetOnFootDamageSound(eType);
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetMechDamageSound
//
//	PURPOSE:	Determine what damage sound to play
//
// ----------------------------------------------------------------------- //

char* CPlayerMode::GetMechDamageSound(DamageType eType)
{
	SAFE_STRCPY(s_FileBuffer, "Sounds\\Player\\Mech\\");

	switch (eType)
	{
		case DT_CHOKE:
		{
			s_FileBuffer[0] = '\0';
		}
		break;

		default:
		{
			char* PainSounds[] =  { "pain1.wav", "pain2.wav" };
	
			int nSize = (sizeof(PainSounds)/sizeof(PainSounds[0])) - 1;
			strcat(s_FileBuffer, PainSounds[GetRandom(0, nSize)]);
		}
	}

	return s_FileBuffer;
}
/*
// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetAPDamageSound
//
//	PURPOSE:	Determine what AP damage sound to play
//
// ----------------------------------------------------------------------- //

char* CPlayerMode::GetAPDamageSound(DamageType eType)
{
	strcpy(s_FileBuffer, "Sounds\\Player\\Heavyw\\");

	switch (eType)
	{
		case DT_CHOKE:
		{
			s_FileBuffer[0] = '\0';
		}
		break;

		default:
		{
			char* PainSounds[] =  { "pain1.wav" };
	
			int nSize = (sizeof(PainSounds)/sizeof(PainSounds[0])) - 1;
			strcat(s_FileBuffer, PainSounds[GetRandom(0, nSize)]);
		}
	}

	return s_FileBuffer;
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetAODamageSound
//
//	PURPOSE:	Determine what AO damage sound to play
//
// ----------------------------------------------------------------------- //

char* CPlayerMode::GetAODamageSound(DamageType eType)
{
	strcpy(s_FileBuffer, "Sounds\\Player\\Demoman\\");

	switch (eType)
	{
		case DT_CHOKE:
		{
			s_FileBuffer[0] = '\0';
		}
		break;

		default:
		{
			char* PainSounds[] =  { "pain1.wav" };
	
			int nSize = (sizeof(PainSounds)/sizeof(PainSounds[0])) - 1;
			strcat(s_FileBuffer, PainSounds[GetRandom(0, nSize)]);
		}
	}

	return s_FileBuffer;
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetUEDamageSound
//
//	PURPOSE:	Determine what UE damage sound to play
//
// ----------------------------------------------------------------------- //

char* CPlayerMode::GetUEDamageSound(DamageType eType)
{
	strcpy(s_FileBuffer, "Sounds\\Player\\Rocketeer\\");

	switch (eType)
	{
		case DT_CHOKE:
		{
			s_FileBuffer[0] = '\0';
		}
		break;

		default:
		{
			char* PainSounds[] =  { "pain1.wav" };
	
			int nSize = (sizeof(PainSounds)/sizeof(PainSounds[0])) - 1;
			strcat(s_FileBuffer, PainSounds[GetRandom(0, nSize)]);
		}
	}

	return s_FileBuffer;
}
*/


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetKidDamageSound
//
//	PURPOSE:	Determine what Kid damage sound to play
//
// ----------------------------------------------------------------------- //

char* CPlayerMode::GetKidDamageSound(DamageType eType)
{
	SAFE_STRCPY(s_FileBuffer, "Sounds\\Player\\Kid\\");

	switch (eType)
	{
		case DT_CHOKE:
		{
			char* ChokeSounds[] = { "choke.wav" };
	
			int nSize = (sizeof(ChokeSounds)/sizeof(ChokeSounds[0])) - 1;
			strcat(s_FileBuffer, ChokeSounds[GetRandom(0, nSize)]);
		}
		break;

		default:
		{
			char* PainSounds[] =  { "pain1.wav", "pain2.wav" };
	
			int nSize = (sizeof(PainSounds)/sizeof(PainSounds[0])) - 1;
			strcat(s_FileBuffer, PainSounds[GetRandom(0, nSize)]);
		}
	}

	return s_FileBuffer;
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetOnFootDamageSound
//
//	PURPOSE:	Determine what OnFoot damage sound to play
//
// ----------------------------------------------------------------------- //

char* CPlayerMode::GetOnFootDamageSound(DamageType eType)
{
	SAFE_STRCPY(s_FileBuffer, "Sounds\\Player\\OnFoot\\");

	switch (eType)
	{
		case DT_CHOKE:
		{
			char* ChokeSounds[] = { "choke.wav", "choke2.wav" };
	
			int nSize = (sizeof(ChokeSounds)/sizeof(ChokeSounds[0])) - 1;
			strcat(s_FileBuffer, ChokeSounds[GetRandom(0, nSize)]);
		}
		break;

		case DT_ELECTROCUTE:
		{
			char* Sounds[] = { "playerelec1.wav", "playerelec2.wav" };
	
			int nSize = (sizeof(Sounds)/sizeof(Sounds[0])) - 1;
			strcat(s_FileBuffer, Sounds[GetRandom(0, nSize)]);
		}
		break;

		default:
		{
			char* PainSounds[] =  { "pain1.wav", "pain2.wav", "pain3.wav", "pain4.wav", "pain5.wav" };
	
			int nSize = (sizeof(PainSounds)/sizeof(PainSounds[0])) - 1;
			strcat(s_FileBuffer, PainSounds[GetRandom(0, nSize)]);
		}
	}

	return s_FileBuffer;
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetDeathSound
//
//	PURPOSE:	Determine what death sound to play
//
// ----------------------------------------------------------------------- //

char* CPlayerMode::GetDeathSound()
{
	switch(m_nMode)
	{
//DTM
		case PM_MODE_MCA_SNIPER : 
		case PM_MODE_MCA_DEMOMAN : 
		case PM_MODE_MCA_HEAVYW :
		case PM_MODE_MCA_ROCKETEER :
			return GetMechDeathSound();
		break;
//DTM
		case PM_MODE_KID :	  
			return GetKidDeathSound();
		break;

		case PM_MODE_FOOT :	  
		default : 
			break;
		break;
	}

	return GetOnFootDeathSound();
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetMechDeathSound
//
//	PURPOSE:	Determine what SA death sound to play
//
// ----------------------------------------------------------------------- //

char* CPlayerMode::GetMechDeathSound()
{
	SAFE_STRCPY(s_FileBuffer, "Sounds\\Player\\Mech\\");

	char* DeathSounds[] =  { "death1.wav", "death2.wav" };
	
	int nSize = (sizeof(DeathSounds)/sizeof(DeathSounds[0])) - 1;
	strcat(s_FileBuffer, DeathSounds[GetRandom(0, nSize)]);

	return s_FileBuffer;
}
/*
// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetAPDeathSound
//
//	PURPOSE:	Determine what AP death sound to play
//
// ----------------------------------------------------------------------- //

char* CPlayerMode::GetAPDeathSound()
{
	strcpy(s_FileBuffer, "Sounds\\Player\\Heavyw\\");

	char* DeathSounds[] =  { "death1.wav", "death2.wav" };
	
	int nSize = (sizeof(DeathSounds)/sizeof(DeathSounds[0])) - 1;
	strcat(s_FileBuffer, DeathSounds[GetRandom(0, nSize)]);

	return s_FileBuffer;
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetAODeathSound
//
//	PURPOSE:	Determine what AO death sound to play
//
// ----------------------------------------------------------------------- //

char* CPlayerMode::GetAODeathSound()
{
	strcpy(s_FileBuffer, "Sounds\\Player\\Demoman\\");

	char* DeathSounds[] =  { "death1.wav", "death2.wav" };
	
	int nSize = (sizeof(DeathSounds)/sizeof(DeathSounds[0])) - 1;
	strcat(s_FileBuffer, DeathSounds[GetRandom(0, nSize)]);

	return s_FileBuffer;
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetUEDeathSound
//
//	PURPOSE:	Determine what UE death sound to play
//
// ----------------------------------------------------------------------- //

char* CPlayerMode::GetUEDeathSound()
{
	strcpy(s_FileBuffer, "Sounds\\Player\\Rocketeer\\");

	char* DeathSounds[] =  { "death1.wav", "death2.wav" };
	
	int nSize = (sizeof(DeathSounds)/sizeof(DeathSounds[0])) - 1;
	strcat(s_FileBuffer, DeathSounds[GetRandom(0, nSize)]);

	return s_FileBuffer;
}
*/

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetKidDeathSound
//
//	PURPOSE:	Determine what Kid death sound to play
//
// ----------------------------------------------------------------------- //

char* CPlayerMode::GetKidDeathSound()
{
	SAFE_STRCPY(s_FileBuffer, "Sounds\\Player\\Kid\\");

	char* DeathSounds[] =  { "death1.wav" };
	
	int nSize = (sizeof(DeathSounds)/sizeof(DeathSounds[0])) - 1;
	strcat(s_FileBuffer, DeathSounds[GetRandom(0, nSize)]);

	return s_FileBuffer;
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetOnFootDeathSound
//
//	PURPOSE:	Determine what OnFoot death sound to play
//
// ----------------------------------------------------------------------- //

char* CPlayerMode::GetOnFootDeathSound()
{
	SAFE_STRCPY(s_FileBuffer, "Sounds\\Player\\OnFoot\\");

	char* DeathSounds[] =  { "death1.wav", "death2.wav", "death3.wav", "death4.wav", "death5.wav" };
	
	int nSize = (sizeof(DeathSounds)/sizeof(DeathSounds[0])) - 1;
	strcat(s_FileBuffer, DeathSounds[GetRandom(0, nSize)]);

	return s_FileBuffer;
}

// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::GetDimsScale
//
//	PURPOSE:	Get the dims scale for the particular mode
//
// ----------------------------------------------------------------------- //

DFLOAT CPlayerMode::GetDimsScale() const
{
	DFLOAT fScale = 1.0f;

	switch(m_nMode)
	{
//DTM
		case PM_MODE_MCA_SNIPER : 
			fScale = 0.9f;
		break;

		case PM_MODE_MCA_ROCKETEER : 
			fScale = 1.0f;
		break;

		case PM_MODE_MCA_DEMOMAN : 
			fScale = 1.1f;
		break;

		case PM_MODE_MCA_HEAVYW : 
			fScale = 1.2f;
		break;
// DTM
		case PM_MODE_KID    : 
			fScale = 1.0f;
		break;

		case PM_MODE_FOOT :	  
			fScale = 1.1f;
		break;

		default : break;
	}

	return fScale;
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::Save
//
//	PURPOSE:	Save the object
//
// ----------------------------------------------------------------------- //

void CPlayerMode::Save(HMESSAGEWRITE hWrite)
{
	CServerDE* pServerDE = BaseClass::GetServerDE();
	if (!pServerDE || !hWrite) return;

	pServerDE->WriteToMessageByte(hWrite, m_nModelId);
	pServerDE->WriteToMessageByte(hWrite, m_nMode);
	pServerDE->WriteToMessageByte(hWrite, m_nArmor);
	pServerDE->WriteToMessageByte(hWrite, m_nVehicleHandling);
	pServerDE->WriteToMessageByte(hWrite, m_bBipedal);
	pServerDE->WriteToMessageFloat(hWrite, m_fJumpSpeed);
	pServerDE->WriteToMessageFloat(hWrite, m_fRunVel);
	pServerDE->WriteToMessageFloat(hWrite, m_fWalkVel);
	pServerDE->WriteToMessageFloat(hWrite, m_FovX);
	pServerDE->WriteToMessageFloat(hWrite, m_FovY);
	pServerDE->WriteToMessageVector(hWrite, &m_vCameraOffset);
}


// ----------------------------------------------------------------------- //
//
//	ROUTINE:	CPlayerMode::Load
//
//	PURPOSE:	Load the object
//
// ----------------------------------------------------------------------- //

void CPlayerMode::Load(HMESSAGEREAD hRead)
{
	CServerDE* pServerDE = BaseClass::GetServerDE();
	if (!pServerDE || !hRead) return;

	m_nModelId			= pServerDE->ReadFromMessageByte(hRead);
	m_nMode				= pServerDE->ReadFromMessageByte(hRead);
	m_nArmor			= pServerDE->ReadFromMessageByte(hRead);
	m_nVehicleHandling	= pServerDE->ReadFromMessageByte(hRead);
	m_bBipedal			= (DBOOL)pServerDE->ReadFromMessageByte(hRead);
	m_fJumpSpeed		= pServerDE->ReadFromMessageFloat(hRead);
	m_fRunVel			= pServerDE->ReadFromMessageFloat(hRead);
	m_fWalkVel			= pServerDE->ReadFromMessageFloat(hRead);
	m_FovX				= pServerDE->ReadFromMessageFloat(hRead);
	m_FovY				= pServerDE->ReadFromMessageFloat(hRead);
	pServerDE->ReadFromMessageVector(hRead, &m_vCameraOffset);
}