//////////////////////////////////////////////////////////////////////
// FILE: Studio_utils.cpp
// PURPOSE: Implementation of the utilities functions of StudioModel class
//////////////////////////////////////////////////////////////////////
// SHORT DESCRIPTION:
// This file has all the code needed to manipulate, fill and do other things with the
// loaded model, represented by the StudioModel class
//////////////////////////////////////////////////////////////////////
// COPYRIGHTS:
// Note: Based on Valve Software's modelviewer 1.0 code. Some code licensed from iD software.
// (c)1996-1999 by the owners of the codeparts.
//
//-----[Valve header]------------
//	Copyright (c) 1998, Valve LLC. All rights reserved.
//	
//	This product contains software technology licensed from Id 
//	Software, Inc. ("Id Technology").  Id Technology (c) 1996 Id Software, Inc. 
//	All Rights Reserved.
//-----[End Valve header]--------
//
// Programmed by:
// Frans 'Otis' Bouma,
// Greg 'Ascent' Dunn,
// Volker 'Dark Yoda' Schnefeld
//
// Code by Otis is (c)1999 Solutions Design, http://www.sd.nl
// Code by Ascent is (c)1999 Greg Dunn.
// Code by Dark Yoda is (c)1999 Volker Schnefeld.
// All rights reserved.
//////////////////////////////////////////////////////////////////////
// VERSION INFORMATION.
//
// 02-april-1999
// Release 2.1
// [OTIS] Added skinbrowsing
// [OTIS] Added disable/enable code for buttons in dialog
// [OTIS] Added flickerselection for textures
//
// 01-april-1999
// [OTIS] Fixed memleak when closing application.
// [OTIS] Fixed file not closed bug when saving model.
// [OTIS] Fixed too many textures reported bug. Now the correct number of textures is reported and used.
//
// 31-mar-1999
// [OTIS] Fixed resize bug
// [OTIS] Included reinit cycle system for texture refresh
// [OTIS] Implemented texture import/export
//
// 25-mar-1999
// [OTIS] Added texture selection, chromeflag toggle
// [OTIS] Added modelsave
// [OTIS] Improved initialize/loading scheme. Now bogus files are not loaded and no view is displayed
//
// 23-mar-1999
// [OTIS] Added texture deletion when unloading models plus fixed OnDestroy bug
//
// 16-mar-1999.
//		First public release.
//
// 15-mar-1999.
// [OTIS] Bugfixes and added checks when reading params from the StudioModel.
//
// 08-feb-1999. 
//		First Internal Version 
//
////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include <stdio.h>
#include <gl\gl.h>
#include <gl\glu.h>
#include "mathlib.h"
#include "studio.h"
#include "StudioModel.h"

#pragma warning( disable : 4244 ) // double to float

////////////////////////////////////////////////////////////////////////


void StudioModel::UploadTexture(mstudiotexture_t *ptexture, byte *data, byte *pal, int g_texnum)
{
	int		i, j;
	int		row1[256], row2[256], col1[256], col2[256];
	byte	*pix1, *pix2, *pix3, *pix4;
	byte	*tex, *out;

	// convert texture to power of 2
	for (int outwidth = 1; outwidth < ptexture->width; outwidth <<= 1)
		;

	if (outwidth > 256)
		outwidth = 256;

	for (int outheight = 1; outheight < ptexture->height; outheight <<= 1)
		;

	if (outheight > 256)
		outheight = 256;

	tex = out = (byte *)malloc( outwidth * outheight * 4);

	for (i = 0; i < outwidth; i++)
	{
		col1[i] = (i + 0.25) * (ptexture->width / (float)outwidth);
		col2[i] = (i + 0.75) * (ptexture->width / (float)outwidth);
	}

	for (i = 0; i < outheight; i++)
	{
		row1[i] = (int)((i + 0.25) * (ptexture->height / (float)outheight)) * ptexture->width;
		row2[i] = (int)((i + 0.75) * (ptexture->height / (float)outheight)) * ptexture->width;
	}

	// scale down and convert to 32bit RGB
	for (i=0 ; i<outheight ; i++)
	{
		for (j=0 ; j<outwidth ; j++, out += 4)
		{
			pix1 = &pal[data[row1[i] + col1[j]] * 3];
			pix2 = &pal[data[row1[i] + col2[j]] * 3];
			pix3 = &pal[data[row2[i] + col1[j]] * 3];
			pix4 = &pal[data[row2[i] + col2[j]] * 3];

			out[0] = (pix1[0] + pix2[0] + pix3[0] + pix4[0])>>2;
			out[1] = (pix1[1] + pix2[1] + pix3[1] + pix4[1])>>2;
			out[2] = (pix1[2] + pix2[2] + pix3[2] + pix4[2])>>2;
			out[3] = 0xFF;
		}
	}
	glBindTexture( GL_TEXTURE_2D, g_texnum );
	// hint the driver that we want REAL GL_RGBA8 
	glTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA8, outwidth, outheight, 0, GL_RGBA, GL_UNSIGNED_BYTE, tex );
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	ptexture->index = g_texnum;

	free( tex );
}


void StudioModel::SetRenderMode(int mode)
{
	rendermode = mode;
}


void
StudioModel::Init()
{ 
	// this function uploads the textures
	int					i;
	byte				*pin;
	studiohdr_t			*phdr;
	mstudiotexture_t	*ptexture;

	m_pstudiohdr = (studiohdr_t *)CoreFile;

	// init the pointers to point to the data inside the corefile in memory.
	pin = CoreFile;
	phdr = (studiohdr_t *)pin;

	ptexture = (mstudiotexture_t *)(pin + phdr->textureindex);

	if (phdr->textureindex != 0)
	{
		// copy the textureindexes to savebuffer...
		pSavedTIndexes=(int *)malloc(phdr->numtextures * sizeof(int));
		for (i = 0; i < phdr->numtextures; i++)
		{
			// save index
			pSavedTIndexes[i]=ptexture[i].index;
			UploadTexture( &ptexture[i], pin + ptexture[i].index, pin + ptexture[i].width * ptexture[i].height + ptexture[i].index, m_iNumOfTextures );
			m_iNumOfTextures++;
		}
		m_iNumOfTextures=phdr->numtextures;
	}

	// preload textures
	if (m_pstudiohdr->numtextures == 0)
	{
		m_ptexturehdr = (studiohdr_t *)TextureFile;

		// init the pointers to point to the data inside the texturefile in memory.
		pin = TextureFile;
		phdr = (studiohdr_t *)pin;

		if (phdr->textureindex != 0)
		{
			// copy the textureindexes to savebuffer...
			pSavedTIndexes=(int *)malloc(phdr->numtextures * sizeof(int));

			ptexture = (mstudiotexture_t *)(pin + phdr->textureindex);
			for (i = 0; i < phdr->numtextures; i++)
			{
				// save index
				pSavedTIndexes[i]=ptexture[i].index;
				UploadTexture( &ptexture[i], pin + ptexture[i].index, pin + ptexture[i].width * ptexture[i].height + ptexture[i].index, m_iNumOfTextures );
				m_iNumOfTextures++;
			}
			m_iNumOfTextures=phdr->numtextures;
		}
	}
	else
	{
		m_ptexturehdr = m_pstudiohdr;
	}

	// all loaded... set flag
	ModelLoaded = true;

	// reset flag for upper system so there is no initialisation cycle started
	m_bDoInit=false;

	// calculate nr of bodies...
	for (i=0 ; i < m_pstudiohdr->numbodyparts ; i++) 
	{
		SetupModel( i );
	}
}


bool
StudioModel::LoadModel( char *modelname )
{ 
	// this function loads the modeldata. This can be up to a lot of files (32 sequence files, 1 core file and 1 additional texturefile)
	// the memory allocated for these files are stored separately in memoryancher pointers. All pointers in the StudioModel point inside
	// that memory. 
	// Just the loading is done. Everything else is done by the Init routine.

	FILE				*fp;
	byte				*pin;
	studiohdr_t			*phdr;
	mstudiotexture_t	*ptexture;
	char				texturename[256];
	char				seqgroupname[256];

	// load core file
	if( (fp = fopen( modelname, "rb" )) == NULL)
	{
		return false;
	}
	fseek( fp, 0, SEEK_END );
	CoreFileLength = ftell( fp );
	fseek( fp, 0, SEEK_SET );
	CoreFile = (byte *)malloc(CoreFileLength);
	fread(CoreFile, CoreFileLength, 1, fp );
	fclose(fp);

	// test for the IDST tag...
	if(memcmp(CoreFile,"IDST",4)!=0)
	{
		// bad modelfile.
		free(CoreFile);
		CoreFileLength=0;
		return false;
	}

	m_pstudiohdr = (studiohdr_t *)CoreFile;

	// init the pointers to point to the data inside the corefile in memory.
	pin = CoreFile;
	phdr = (studiohdr_t *)pin;

	ptexture = (mstudiotexture_t *)(pin + phdr->textureindex);

	// preload textures
	if (m_pstudiohdr->numtextures == 0)
	{
		strcpy( texturename, modelname );
		strcpy( &texturename[strlen(texturename) - 4], "T.mdl" );

		if( (fp = fopen( texturename, "rb" )) == NULL)
		{
			return false;
		}
		fseek( fp, 0, SEEK_END );
		TextureFileLength = ftell( fp );
		fseek( fp, 0, SEEK_SET );
		TextureFile = (byte *)malloc(TextureFileLength);
		fread(TextureFile, TextureFileLength, 1, fp );
		fclose(fp);

		m_ptexturehdr = (studiohdr_t *)TextureFile;
		m_iSkinAmount=m_ptexturehdr->numskinfamilies;
	}
	else
	{
		m_ptexturehdr = m_pstudiohdr;
		m_iSkinAmount=m_pstudiohdr->numskinfamilies;
	}

	// preload animations
	if (m_pstudiohdr->numseqgroups > 1)
	{
		for (int i = 1; i < m_pstudiohdr->numseqgroups; i++)
		{
			strcpy( seqgroupname, modelname );
			sprintf( &seqgroupname[strlen(seqgroupname) - 4], "%02d.mdl", i );

			if( (fp = fopen( seqgroupname, "rb" )) == NULL)
			{
				return false;
			}
			fseek( fp, 0, SEEK_END );
			SequenceFilesLength[i] = ftell( fp );
			fseek( fp, 0, SEEK_SET );
			SequenceFiles[i] = (byte *)malloc(SequenceFilesLength[i]);
			fread(SequenceFiles[i], SequenceFilesLength[i], 1, fp );
			fclose(fp);

			m_panimhdr[i] = (studioseqhdr_t *)SequenceFiles[i];
		}
	}
	m_iNumOfTextures = 1;
	return true;
}


// Purpose: saves the current texture to a file
// Additional info: current texture nr is stored in m_iCurrentTexture
void
StudioModel::ExportCurrentTexture()
{
	int				iResult;
	CFileDialog		*cfFileDlg;
	byte			*pColorData;
	byte			*pBitData;
	studiohdr_t		*pHdr;
	mstudiotexture_t *pTextures;
 
	cfFileDlg = new CFileDialog(FALSE,"bmp",NULL,OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT,"bitmap files (*.bmp)|*.bmp|",NULL);

	iResult = cfFileDlg->DoModal();

	if(iResult==IDOK)
	{
		// user pressed OK.
		// calculate parameters and call bitmap export routine...
		pHdr=(studiohdr_t *)CoreFile;
		if(pHdr->numtextures==0)
		{
			// textures are stored in a separated t.mdl file
			pHdr=(studiohdr_t *)TextureFile;
		}
		pTextures=(mstudiotexture_t *)((byte *)pHdr + pHdr->textureindex);
		pBitData=(byte *)pHdr + pSavedTIndexes[m_iCurrentTexture-1];
		pColorData=(byte *)pHdr + pTextures[m_iCurrentTexture-1].width * pTextures[m_iCurrentTexture-1].height + pSavedTIndexes[m_iCurrentTexture-1];
		DoExportBMP(pBitData,pColorData,cfFileDlg->GetPathName(), &pTextures[m_iCurrentTexture-1]);
	}
	delete cfFileDlg;
}


// Purpose: asks for a filename, reads the bitmap, merges the data into the modelcore or texture file and uploads the texture to
// the card.
void
StudioModel::ImportCurrentTexture()
{
	int				iResult;
	bool			bResult;
	CFileDialog		*cfFileDlg;
	byte			*pColorData;
	byte			*pBitData;
	studiohdr_t		*pHdr;
	mstudiotexture_t *pTextures;
 
	cfFileDlg = new CFileDialog(TRUE,"bmp",NULL,OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT,"bitmap files (*.bmp)|*.bmp|",NULL);

	iResult = cfFileDlg->DoModal();

	if(iResult==IDOK)
	{
		// user pressed OK.
		// calculate parameters and call bitmap export routine...
		pHdr=(studiohdr_t *)CoreFile;
		if(pHdr->numtextures==0)
		{
			// textures are stored in a separated t.mdl file
			pHdr=(studiohdr_t *)TextureFile;
		}
		pTextures=(mstudiotexture_t *)((byte *)pHdr + pHdr->textureindex);
		pBitData=(byte *)pHdr + pSavedTIndexes[m_iCurrentTexture-1];
		pColorData=(byte *)pHdr + pTextures[m_iCurrentTexture-1].width * pTextures[m_iCurrentTexture-1].height + pSavedTIndexes[m_iCurrentTexture-1];
		bResult = DoImportBMP(pBitData,pColorData,cfFileDlg->GetPathName(), &pTextures[m_iCurrentTexture-1]);
		if(bResult)
		{
			// reset parameters so a new initialisation cycle can start
			// delete the textures
			DelTextures();
			// reset indices...
			for(int i=0;i<pHdr->numtextures;i++)
			{
				pTextures[i].index=pSavedTIndexes[i];
			}
			// free mem
			free(pSavedTIndexes);
			// reset texturepointer
			m_iNumOfTextures=1;
			// set flag so there will be an initialisation cycle started the next frame.
			m_bDoInit=true;
		}
	}
	delete cfFileDlg;
}


// Purpose: return the status of the DoInit flag, so the upper application knows if we want a new initialisationcycle
bool
StudioModel::CheckDoInit()
{
	return m_bDoInit;
}


// Purpose: does the actual loading of the bitmap data
// Additional info: based on iD's LoadBMP(), which is an example of very awefull code. 
bool
StudioModel::DoImportBMP(byte *pBitData, byte *pColorData, CString sFilename, mstudiotexture_t *pTexture)
{
	int					i;
	FILE				*fp;
	BITMAPFILEHEADER	bmfh;
	BITMAPINFOHEADER	bmih;
	RGBQUAD				rgbPalette[256];
	ULONG				cbBmpBits;
	byte				*pb, *pbPal = NULL;
	ULONG				cbPalBytes;
	ULONG				biTrueWidth;
	CString				sError;

	// File exists?
	if ((fp = fopen(sFilename, "rb")) == NULL)
	{
		// open didn't succeed
		AfxMessageBox("Load failed. Could not open file.",MB_OK);
		return false;
	}
	
	// Read file header
	fread(&bmfh, sizeof bmfh, 1, fp);

	// Bogus file header check
	if (!(bmfh.bfReserved1 == 0 && bmfh.bfReserved2 == 0))
	{
		fclose(fp);
		AfxMessageBox("Invalid BMP file.",MB_OK);
		return false;
	}

	// Read info header
	fread(&bmih, sizeof bmih, 1, fp);

	// Bogus info header check
	if (!(bmih.biSize == sizeof bmih && bmih.biPlanes == 1))
	{
		fclose(fp);
		AfxMessageBox("Invalid BMP file.",MB_OK);
		return false;
	}

	// Bogus bit depth?  Only 8-bit supported.
	if (bmih.biBitCount != 8)
	{
		fclose(fp);
		AfxMessageBox("Invalid BMP file. Only 8bit (256colors) bmp files supported.",MB_OK);
		return false;
	}
	
	// Bogus compression?  Only non-compressed supported.
	if (bmih.biCompression != BI_RGB)
	{
		fclose(fp);
		AfxMessageBox("Invalid BMP file. This file is compressed. Only UNCOMPRESSED BMP files supported.",MB_OK);
		return false;
	}

	biTrueWidth = (bmih.biWidth + 3) & ~3;

	// test size. We cannot crop, because the texture has to fit exactly. If the user wants to load a smaller image,
	// the user has to stick that smaller image on a large image. It won't scale anyway.
	if((pTexture->width!=(int)biTrueWidth)||(pTexture->height!=bmih.biHeight))
	{
		// too large.
		fclose(fp);
		sError.Format("Invalid BMP file. The bitmap is not of the right size.\nWidth should be %i and height should be %i.\nTextures won't scale anyway, so it's of no use to load smaller or larger pictures", pTexture->width, pTexture->height);
		AfxMessageBox(sError,MB_OK);
		return false;
	}

	// Figure out how many entries are actually in the table
	if (bmih.biClrUsed == 0)
	{
		bmih.biClrUsed = 256;
		cbPalBytes = (1 << bmih.biBitCount) * sizeof( RGBQUAD );
	}
	else 
	{
		cbPalBytes = bmih.biClrUsed * sizeof( RGBQUAD );
	}

	// Read palette (bmih.biClrUsed entries)
	fread(rgbPalette, cbPalBytes, 1,fp);

	// convert to a packed 768 byte palette
	pbPal = pColorData;

	// Copy over used entries
	for (i = 0; i < (int)bmih.biClrUsed; i++)
	{
		*pbPal++ = rgbPalette[i].rgbRed;
		*pbPal++ = rgbPalette[i].rgbGreen;
		*pbPal++ = rgbPalette[i].rgbBlue;
	}

	// Fill in unused entires will 0,0,0
	for (i = bmih.biClrUsed; i < 256; i++) 
	{
		*pbPal++ = (byte)0;
		*pbPal++ = (byte)0;
		*pbPal++ = (byte)0;
	}

	// Read bitmap bits (remainder of file)
	cbBmpBits = bmfh.bfSize - ftell(fp);
	pb = (byte *)malloc(cbBmpBits * sizeof(byte));

	// load bitmapdata
	fread(pb, cbBmpBits, 1,fp);

	// reverse the order of the data.
	pb += (bmih.biHeight - 1) * biTrueWidth;
	for(i = 0; i < bmih.biHeight; i++)
	{
		memmove(&pBitData[biTrueWidth * i], pb, biTrueWidth);
		pb -= biTrueWidth;
	}
	pb += biTrueWidth;
	free(pb);
	fclose(fp);
	return true;
}



// Purpose: does the actual saving of the bitmap data
// Additional info: based on iD's WriteBMP(), which is an example of very awefull code. 
void
StudioModel::DoExportBMP(byte *pBitData, byte *pColorData, CString sFilename, mstudiotexture_t *pTexture)
{
	FILE				*fp;
	BITMAPFILEHEADER	bmfh;
	BITMAPINFOHEADER	bmih;
	RGBQUAD				rgbPalette[256];
	ULONG				cbBmpBits;
	BYTE*				pbBmpBits;
	byte				*pb, *pbPal = NULL;
	ULONG				cbPalBytes;
	ULONG				biTrueWidth;
	int					i;


	if((fp=fopen(sFilename,"wb"))==NULL)
	{
		// open didn't succeed...
		return;
	}
	// write data into file. First define all structures, then recalc offsets and sizes in those structures, then
	// write the data to disk..
	// we support only 8 bit, 256color, uncompressed WINDOWS bmp files. OS/2 users should go to IBM and whine.

	// fill file header

	biTrueWidth = ((pTexture->width + 3) & ~3);
	cbBmpBits = biTrueWidth * pTexture->height;
	cbPalBytes = 256 * sizeof( RGBQUAD );

	// Bogus file header check
	bmfh.bfType = MAKEWORD( 'B', 'M' );
	bmfh.bfSize = sizeof bmfh + sizeof bmih + cbBmpBits + cbPalBytes;
	bmfh.bfReserved1 = 0;
	bmfh.bfReserved2 = 0;
	bmfh.bfOffBits = sizeof bmfh + sizeof bmih + cbPalBytes;

	// Write file header
	fwrite(&bmfh, sizeof bmfh, 1, fp);

	// Size of structure
	bmih.biSize = sizeof bmih;
	// Width
	bmih.biWidth = biTrueWidth;
	// Height
	bmih.biHeight = pTexture->height;
	// Only 1 plane 
	bmih.biPlanes = 1;
	// Only 8-bit supported.
	bmih.biBitCount = 8;
	// Only non-compressed supported.
	bmih.biCompression = BI_RGB;
	bmih.biSizeImage = 0;

	// [OTIS] ahahah :) the next comment is done by a valve-programmer :) I let it stay for fun ;)
	// huh?
	bmih.biXPelsPerMeter = 0;
	bmih.biYPelsPerMeter = 0;

	// Always full palette
	bmih.biClrUsed = 256;
	bmih.biClrImportant = 0;
	
	// Write info header
	fwrite(&bmih, sizeof bmih, 1, fp);

	// convert to expanded palette
	pb = pColorData;

	// Copy over used entries
	for (i = 0; i < (int)bmih.biClrUsed; i++)
	{
		rgbPalette[i].rgbRed   = *pb++;
		rgbPalette[i].rgbGreen = *pb++;
		rgbPalette[i].rgbBlue  = *pb++;
        rgbPalette[i].rgbReserved = 0;
	}

	// Write palette (bmih.biClrUsed entries)
	cbPalBytes = bmih.biClrUsed * sizeof( RGBQUAD );
	fwrite(rgbPalette, cbPalBytes, 1, fp);

	pbBmpBits = (byte *)malloc(cbBmpBits*sizeof(byte));

	pb = pBitData;
	// reverse the order of the data.
	pb += (pTexture->height - 1) * pTexture->width;
	for(i = 0; i < bmih.biHeight; i++)
	{
		memmove(&pbBmpBits[biTrueWidth * i], pb, pTexture->width);
		pb -= pTexture->width;
	}

	// Write bitmap bits (remainder of file)
	fwrite(pbBmpBits, cbBmpBits, 1, fp);

	free(pbBmpBits);

	// done
	fclose(fp);
}


void
StudioModel::SaveModel(CString sPathFilename, CString sFileTitle)
{
	// save the modelfiles.
	FILE				*fp;
	CString				sPath;
	CString				sFilename;
	CString				sTmp;
	int					iPos;
	int					i;
	studiohdr_t			*pHdr;
	studiohdr_t			*pTmpHdr;
	byte				*pTmpFile;
	mstudiotexture_t	*pTextures;

	// get path without filename...
	sTmp=sFileTitle + ".mdl";
	iPos=sPathFilename.Find(sTmp);
	sPath=sPathFilename.Mid(0,iPos);

	// write core file
	if(CoreFileLength!=0)
	{
		// save corefile.
		sFilename = sPath + sFileTitle + ".mdl";
		// open file
		if((fp=fopen(sFilename,"wb"))==NULL)
		{
			// could not save it... 
			AfxMessageBox("Could not save core modelfile.",MB_OK);
		}
		// now, check if there are textures in this file. IF so, we first have to copy the
		// data, and have to reset the texture-indexes.
		pHdr=(studiohdr_t *)CoreFile;
		if(pHdr->numtextures!=0)
		{
			// there are textures in this datablock. copy filedata to tempbuffer, reset texture
			// indices and save that block
			pTmpFile=(byte *)malloc(CoreFileLength);
			// copy data...
			for(i=0;i<CoreFileLength;i++)
			{
				pTmpFile[i]=CoreFile[i];
			}
			// reset indices.
			pTmpHdr=(studiohdr_t *)pTmpFile;
			pTextures=(mstudiotexture_t *)((byte *)pTmpHdr + pTmpHdr->textureindex);
			for(i=0;i<pHdr->numtextures;i++)
			{
				pTextures[i].index=pSavedTIndexes[i];
			}
			// save this block
			fwrite(pTmpFile,CoreFileLength,1,fp);
			// free mem
			free(pTmpFile);
			fclose(fp);
		}
		else
		{
			// No textures here: save the corefile data...
			fwrite(CoreFile,CoreFileLength,1,fp);
			// done
			fclose(fp);
		}
	}
	// write texturefile (if any)
	if(TextureFileLength!=0)
	{
		// save texturefile.
		sFilename = sPath + sFileTitle + "t.mdl";
		// open file
		if((fp=fopen(sFilename,"wb"))==NULL)
		{
			// could not save it... 
			AfxMessageBox("Could not save texturefile.",MB_OK);
		}
		// now, check if there are textures in this file. IF so, we first have to copy the
		// data, and have to reset the texture-indexes.
		pHdr=(studiohdr_t *)TextureFile;
		if(pHdr->numtextures!=0)
		{
			// there are textures in this datablock. copy filedata to tempbuffer, reset texture
			// indices and save that block
			pTmpFile=(byte *)malloc(TextureFileLength);
			// copy data...
			for(i=0;i<TextureFileLength;i++)
			{
				pTmpFile[i]=TextureFile[i];
			}
			// reset indices.
			pTmpHdr=(studiohdr_t *)pTmpFile;
			pTextures=(mstudiotexture_t *)((byte *)pTmpHdr + pTmpHdr->textureindex);
			for(i=0;i<pHdr->numtextures;i++)
			{
				pTextures[i].index=pSavedTIndexes[i];
			}
			// save this block
			fwrite(pTmpFile,TextureFileLength,1,fp);
			// free mem
			free(pTmpFile);
			fclose(fp);
		}
		else
		{
			// No textures: save the texturefile (?) data...
			fwrite(TextureFile,TextureFileLength,1,fp);
			// done
			fclose(fp);
		}
	}
	// write sequencefiles (if any)
	for(i=0;i<32;i++)
	{
		if(SequenceFilesLength[i]>0)
		{
			// save this file...
			sTmp.Format("%02i.mdl",i);
			// save texturefile.
			sFilename = sPath + sFileTitle + sTmp;
			// open file
			if((fp=fopen(sFilename,"wb"))==NULL)
			{
				// could not save it... 
				AfxMessageBox("Could not save one of the additional sequence files.",MB_OK);
			}
			else
			{
				// save the corefile data...
				fwrite(SequenceFiles[i],SequenceFilesLength[i],1,fp);
				// done
				fclose(fp);
			}
		}
	}
}


////////////////////////////////////////////////////////////////////////

void
StudioModel::DelTextures()
{
	int					i;
	byte				*pin;
	studiohdr_t			*phdr;
	mstudiotexture_t	*ptexture;
	GLuint				*arr_iTexNames;		// never more than 32 textures

	arr_iTexNames = (GLuint *)malloc(sizeof(GLuint) * 1024);
	// deletes all binded textures.
	if(CoreFileLength>0)
	{
		// get pointer to textures
		pin = CoreFile;
		phdr=(studiohdr_t *)pin;
		ptexture = (mstudiotexture_t *)(pin + phdr->textureindex);

		if (phdr->textureindex != 0)
		{
			// there are textures... get rid of 'em!
			for (i = 0; i < phdr->numtextures; i++)
			{
				arr_iTexNames[i] = (GLuint)ptexture[i].index;
			}
			// delete them
			glDeleteTextures(phdr->numtextures,arr_iTexNames);
		}
	}
	if(TextureFileLength>0)
	{
		pin = TextureFile;
		phdr = (studiohdr_t *)pin;

		if (phdr->textureindex != 0)
		{
			ptexture = (mstudiotexture_t *)(pin + phdr->textureindex);
			for (i = 0; i < phdr->numtextures; i++)
			{
				arr_iTexNames[i] = (GLuint)ptexture[i].index;
			}
			glDeleteTextures(phdr->numtextures,arr_iTexNames);
		}
	}
	free(arr_iTexNames);
}


// Purpose: get the status of the chromeflag of current selected texture
bool
StudioModel::GetCurTextChromeFlag()
{
	mstudiotexture_t	*ptexture;

	if(ModelLoaded && (m_iCurrentTexture>0))
	{
		// get texture record.
		ptexture = (mstudiotexture_t *)((byte *)m_ptexturehdr + m_ptexturehdr->textureindex);
		// get flag
		return ptexture[m_iCurrentTexture-1].flags & STUDIO_NF_CHROME;
	}
	else
	{
		return false;
	}
}


// Purpose: toggle chrome flag of current selected texture
void
StudioModel::ToggleCurTextChromeFlag()
{
	mstudiotexture_t	*ptexture;

	if(ModelLoaded && (m_iCurrentTexture>0))
	{
		// get texture record.
		ptexture = (mstudiotexture_t *)((byte *)m_ptexturehdr + m_ptexturehdr->textureindex);
		// get flag
		if(ptexture[m_iCurrentTexture-1].flags & STUDIO_NF_CHROME)
		{
			// turn it off
			ptexture[m_iCurrentTexture-1].flags &= !STUDIO_NF_CHROME;
		}
		else
		{
			// turn it on
			ptexture[m_iCurrentTexture-1].flags |= STUDIO_NF_CHROME;
		}
	}
}


int
StudioModel::GetCurrentTextureNr()
{
	return m_iCurrentTexture;
}


void
StudioModel::SetTextureNr(int iNr)
{
	if(iNr >m_iNumOfTextures)
	{
		iNr=0;
	}
	if(iNr<0)
	{
		iNr=m_iNumOfTextures;
	}
	m_bCurrentTextureOn=false;
	m_iCurrentTexture=iNr;
}


int
StudioModel::GetMaxTextureNr()
{
	return m_iNumOfTextures;
}


int StudioModel::GetSequence( )
{
	return m_sequence;
}


int StudioModel::SetSequence( int iSequence )
{
	if (iSequence >= m_pstudiohdr->numseq)
		iSequence = 0;
	if (iSequence < 0)
		iSequence = m_pstudiohdr->numseq-1;

	m_sequence = iSequence;
	m_frame = 0;

	return m_sequence;
}


void StudioModel::ExtractBbox( float *mins, float *maxs )
{
	mstudioseqdesc_t	*pseqdesc;

	pseqdesc = (mstudioseqdesc_t *)((byte *)m_pstudiohdr + m_pstudiohdr->seqindex);
	
	mins[0] = pseqdesc[ m_sequence ].bbmin[0];
	mins[1] = pseqdesc[ m_sequence ].bbmin[1];
	mins[2] = pseqdesc[ m_sequence ].bbmin[2];

	maxs[0] = pseqdesc[ m_sequence ].bbmax[0];
	maxs[1] = pseqdesc[ m_sequence ].bbmax[1];
	maxs[2] = pseqdesc[ m_sequence ].bbmax[2];
}


// return the current playing sequence
char *StudioModel::GetSequenceLabel(void)
{
	mstudioseqdesc_t	*pseqdesc;

	if(ModelLoaded)
	{
		pseqdesc = (mstudioseqdesc_t *)((byte *)m_pstudiohdr + m_pstudiohdr->seqindex) + (int)m_sequence;
		return pseqdesc->label;
	}
	else
	{
		return "Not Yet Loaded";
	}
}


// return length of current sequence in milliseconds.
long StudioModel::GetSequenceLength(void)
{
	mstudioseqdesc_t	*pseqdesc;

	if(ModelLoaded)
	{
		pseqdesc = (mstudioseqdesc_t *)((byte *)m_pstudiohdr + m_pstudiohdr->seqindex) + (int)m_sequence;

		if (pseqdesc->numframes > 1)
		{
			return ((pseqdesc->numframes - 1)/pseqdesc->fps) * 1000;
		}
		else
		{
			return 0;
		}
	}
	else
	{
		return 0;
	}
}


int StudioModel::GetCurrentBodyNumber(void)
{
	return m_bodynum;
}


int StudioModel::GetBodyAmount(void)
{
	return maxnummodels;
}


int StudioModel::GetSeqAmount(void)
{
	if(ModelLoaded)
	{
		return m_pstudiohdr->numseq;
	}
	else
	{
		return 0;
	}
}


void StudioModel::SetBodyNr(int nr)
{
	m_bodynum=nr;
}


void StudioModel::GetSequenceInfo( float *pflFrameRate, float *pflGroundSpeed )
{
	mstudioseqdesc_t	*pseqdesc;

	pseqdesc = (mstudioseqdesc_t *)((byte *)m_pstudiohdr + m_pstudiohdr->seqindex) + (int)m_sequence;

	if (pseqdesc->numframes > 1)
	{
		*pflFrameRate = 256 * pseqdesc->fps / (pseqdesc->numframes - 1);
		*pflGroundSpeed = sqrt( pseqdesc->linearmovement[0]*pseqdesc->linearmovement[0]+ pseqdesc->linearmovement[1]*pseqdesc->linearmovement[1]+ pseqdesc->linearmovement[2]*pseqdesc->linearmovement[2] );
		*pflGroundSpeed = *pflGroundSpeed * pseqdesc->fps / (pseqdesc->numframes - 1);
	}
	else
	{
		*pflFrameRate = 256.0;
		*pflGroundSpeed = 0.0;
	}
}


float StudioModel::SetController( int iController, float flValue )
{
	mstudiobonecontroller_t	*pbonecontroller = (mstudiobonecontroller_t *)((byte *)m_pstudiohdr + m_pstudiohdr->bonecontrollerindex);

	// find first controller that matches the index
	for (int i = 0; i < m_pstudiohdr->numbonecontrollers; i++, pbonecontroller++)
	{
		if (pbonecontroller->index == iController)
			break;
	}
	if (i >= m_pstudiohdr->numbonecontrollers)
		return flValue;

	// wrap 0..360 if it's a rotational controller
	if (pbonecontroller->type & (STUDIO_XR | STUDIO_YR | STUDIO_ZR))
	{
		// ugly hack, invert value if end < start
		if (pbonecontroller->end < pbonecontroller->start)
			flValue = -flValue;

		// does the controller not wrap?
		if (pbonecontroller->start + 359.0 >= pbonecontroller->end)
		{
			if (flValue > ((pbonecontroller->start + pbonecontroller->end) / 2.0) + 180)
				flValue = flValue - 360;
			if (flValue < ((pbonecontroller->start + pbonecontroller->end) / 2.0) - 180)
				flValue = flValue + 360;
		}
		else
		{
			if (flValue > 360)
				flValue = flValue - (int)(flValue / 360.0) * 360.0;
			else if (flValue < 0)
				flValue = flValue + (int)((flValue / -360.0) + 1) * 360.0;
		}
	}

	int setting = 255 * (flValue - pbonecontroller->start) / (pbonecontroller->end - pbonecontroller->start);

	if (setting < 0) setting = 0;
	if (setting > 255) setting = 255;
	m_controller[iController] = setting;

	return setting * (1.0 / 255.0) * (pbonecontroller->end - pbonecontroller->start) + pbonecontroller->start;
}


float StudioModel::SetMouth( float flValue )
{
	mstudiobonecontroller_t	*pbonecontroller = (mstudiobonecontroller_t *)((byte *)m_pstudiohdr + m_pstudiohdr->bonecontrollerindex);

	// find first controller that matches the mouth
	for (int i = 0; i < m_pstudiohdr->numbonecontrollers; i++, pbonecontroller++)
	{
		if (pbonecontroller->index == 4)
			break;
	}

	// wrap 0..360 if it's a rotational controller
	if (pbonecontroller->type & (STUDIO_XR | STUDIO_YR | STUDIO_ZR))
	{
		// ugly hack, invert value if end < start
		if (pbonecontroller->end < pbonecontroller->start)
			flValue = -flValue;

		// does the controller not wrap?
		if (pbonecontroller->start + 359.0 >= pbonecontroller->end)
		{
			if (flValue > ((pbonecontroller->start + pbonecontroller->end) / 2.0) + 180)
				flValue = flValue - 360;
			if (flValue < ((pbonecontroller->start + pbonecontroller->end) / 2.0) - 180)
				flValue = flValue + 360;
		}
		else
		{
			if (flValue > 360)
				flValue = flValue - (int)(flValue / 360.0) * 360.0;
			else if (flValue < 0)
				flValue = flValue + (int)((flValue / -360.0) + 1) * 360.0;
		}
	}

	int setting = 64 * (flValue - pbonecontroller->start) / (pbonecontroller->end - pbonecontroller->start);

	if (setting < 0) setting = 0;
	if (setting > 64) setting = 64;
	m_mouth = setting;

	return setting * (1.0 / 64.0) * (pbonecontroller->end - pbonecontroller->start) + pbonecontroller->start;
}


float StudioModel::SetBlending( int iBlender, float flValue )
{
	mstudioseqdesc_t	*pseqdesc;

	pseqdesc = (mstudioseqdesc_t *)((byte *)m_pstudiohdr + m_pstudiohdr->seqindex) + (int)m_sequence;

	if (pseqdesc->blendtype[iBlender] == 0)
		return flValue;

	if (pseqdesc->blendtype[iBlender] & (STUDIO_XR | STUDIO_YR | STUDIO_ZR))
	{
		// ugly hack, invert value if end < start
		if (pseqdesc->blendend[iBlender] < pseqdesc->blendstart[iBlender])
			flValue = -flValue;

		// does the controller not wrap?
		if (pseqdesc->blendstart[iBlender] + 359.0 >= pseqdesc->blendend[iBlender])
		{
			if (flValue > ((pseqdesc->blendstart[iBlender] + pseqdesc->blendend[iBlender]) / 2.0) + 180)
				flValue = flValue - 360;
			if (flValue < ((pseqdesc->blendstart[iBlender] + pseqdesc->blendend[iBlender]) / 2.0) - 180)
				flValue = flValue + 360;
		}
	}

	int setting = 255 * (flValue - pseqdesc->blendstart[iBlender]) / (pseqdesc->blendend[iBlender] - pseqdesc->blendstart[iBlender]);

	if (setting < 0) setting = 0;
	if (setting > 255) setting = 255;

	m_blending[iBlender] = setting;

	return setting * (1.0 / 255.0) * (pseqdesc->blendend[iBlender] - pseqdesc->blendstart[iBlender]) + pseqdesc->blendstart[iBlender];
}



int StudioModel::SetBodygroup( int iGroup, int iValue )
{
	if (iGroup > m_pstudiohdr->numbodyparts)
		return -1;

	mstudiobodyparts_t *pbodypart = (mstudiobodyparts_t *)((byte *)m_pstudiohdr + m_pstudiohdr->bodypartindex) + iGroup;

	int iCurrent = (m_bodynum / pbodypart->base) % pbodypart->nummodels;

	if (iValue >= pbodypart->nummodels)
		return iCurrent;

	m_bodynum = (m_bodynum - (iCurrent * pbodypart->base) + (iValue * pbodypart->base));

	return iValue;
}


int
StudioModel::GetSkinNr()
{
	if(ModelLoaded)
	{
		return m_skinnum;
	}
	else
	{
		return 0;
	}
}

int
StudioModel::GetMaxSkinNr()
{
	if(ModelLoaded)
	{
		return m_iSkinAmount;
	}
	else
	{
		return 0;
	}
}


void
StudioModel::SetSkin( int iValue )
{
	if((iValue+1) > m_iSkinAmount)
	{
		m_skinnum=0;
	}
	else
	{
		m_skinnum=iValue;
	}
}


void
StudioModel::SetFogColor(float r, float g, float b)
{
	fg_r = r;
	fg_g = g;
	fg_b = b;
}


void 
StudioModel::SetLightColor(float r, float g, float b)
{
	m_lightr = r;
	m_lightg = g;
	m_lightb = b;
}
