//////////////////////////////////////////////////////////////////////
// FILE: Studio_render.cpp
// PURPOSE: Implementation of the renderfunctions of StudioModel class
//////////////////////////////////////////////////////////////////////
// SHORT DESCRIPTION:
// This file has all the code needed to render the model.
//////////////////////////////////////////////////////////////////////
// COPYRIGHTS:
// Note: Based on Valve Software's modelviewer 1.0 code. Some code licensed from iD software.
// (c)1996-1999 by the owners of the codeparts.
//
//-----[Valve header]------------
//	Copyright (c) 1998, Valve LLC. All rights reserved.
//	
//	This product contains software technology licensed from Id 
//	Software, Inc. ("Id Technology").  Id Technology (c) 1996 Id Software, Inc. 
//	All Rights Reserved.
//-----[End Valve header]--------
//
// Programmed by:
// Frans 'Otis' Bouma,
// Greg 'Ascent' Dunn,
// Volker 'Dark Yoda' Schnefeld
//
// Code by Otis is (c)1999 Solutions Design, http://www.sd.nl
// Code by Ascent is (c)1999 Greg Dunn.
// Code by Dark Yoda is (c)1999 Volker Schnefeld.
// All rights reserved.
//////////////////////////////////////////////////////////////////////
// VERSION INFORMATION.
//
// 02-april-1999
// Release 2.1
// [OTIS] Added skinbrowsing
// [OTIS] Added disable/enable code for buttons in dialog
// [OTIS] Added flickerselection for textures
//
// 31-mar-1999
// [OTIS] Fixed resize bug
// [OTIS] Included reinit cycle system for texture refresh
// [OTIS] Implemented texture import/export
//
// 25-mar-1999
// [OTIS] Added texture selection, chromeflag toggle
// [OTIS] Added modelsave
// [OTIS] Improved initialize/loading scheme. Now bogus files are not loaded and no view is displayed
//
// 23-mar-1999
// [OTIS] Added texture deletion when unloading models plus fixed OnDestroy bug
//
// 20-mar-1999
// [OTIS] added RM_ADDITIVE rendermode: bones AND transparent textures.
//
// 16-mar-1999.
//		First public release.
//
// 08-feb-1999. 
//		First Internal Version 
//
//////////////////////////////////////////////////////////////////////
#include "StdAfx.h"
#include <gl\gl.h>
#include <gl\glu.h>
#include "mathlib.h"
#include "studio.h"
#include "StudioModel.h"

#pragma warning( disable : 4244 ) // double to float
#pragma warning( disable : 4305 ) // double to float

static GLuint Indices[60] ={
		0,4,1,0,9,4,9,5,4,4,5,8,4,8,1,
		8,10,1,8,3,10,5,3,8,5,2,3,2,7,3,
		7,10,3,7,6,10,7,11,6,11,0,6,0,1,6,
		6,1,10,9,0,11,9,11,2,9,2,5,7,2,11};

void
StudioModel::SetFog(bool fogyn, float fogstrt,float fogstop)
{
	fog_enabled = fogyn;
	fog_start = fogstrt;
	fog_end = fogstop;
}


void StudioModel::CalcBoneAdj( )
{
	int					i, j;
	float				value;
	mstudiobonecontroller_t *pbonecontroller;
	
	pbonecontroller = (mstudiobonecontroller_t *)((byte *)m_pstudiohdr + m_pstudiohdr->bonecontrollerindex);

	for (j = 0; j < m_pstudiohdr->numbonecontrollers; j++)
	{
		i = pbonecontroller[j].index;
		if (i <= 3)
		{
			// check for 360% wrapping
			if (pbonecontroller[j].type & STUDIO_RLOOP)
			{
				value = m_controller[i] * (360.0/256.0) + pbonecontroller[j].start;
			}
			else 
			{
				value = m_controller[i] / 255.0;
				if (value < 0) value = 0;
				if (value > 1.0) value = 1.0;
				value = (1.0 - value) * pbonecontroller[j].start + value * pbonecontroller[j].end;
			}
			// Con_DPrintf( "%d %d %f : %f\n", m_controller[j], m_prevcontroller[j], value, dadt );
		}
		else
		{
			value = m_mouth / 64.0;
			if (value > 1.0) value = 1.0;
			value = (1.0 - value) * pbonecontroller[j].start + value * pbonecontroller[j].end;
			// Con_DPrintf("%d %f\n", mouthopen, value );
		}
		switch(pbonecontroller[j].type & STUDIO_TYPES)
		{
		case STUDIO_XR:
		case STUDIO_YR:
		case STUDIO_ZR:
			m_adj[j] = value * (Q_PI / 180.0);
			break;
		case STUDIO_X:
		case STUDIO_Y:
		case STUDIO_Z:
			m_adj[j] = value;
			break;
		}
	}
}


void StudioModel::CalcBoneQuaternion( int frame, float s, mstudiobone_t *pbone, mstudioanim_t *panim, float *q )
{
	int					j, k;
	vec4_t				q1, q2;
	vec3_t				angle1, angle2;
	mstudioanimvalue_t	*panimvalue;

	for (j = 0; j < 3; j++)
	{
		if (panim->offset[j+3] == 0)
		{
			angle2[j] = angle1[j] = pbone->value[j+3]; // default;
		}
		else
		{
			panimvalue = (mstudioanimvalue_t *)((byte *)panim + panim->offset[j+3]);
			k = frame;
			while (panimvalue->num.total <= k)
			{
				k -= panimvalue->num.total;
				panimvalue += panimvalue->num.valid + 1;
			}
			// Bah, missing blend!
			if (panimvalue->num.valid > k)
			{
				angle1[j] = panimvalue[k+1].value;

				if (panimvalue->num.valid > k + 1)
				{
					angle2[j] = panimvalue[k+2].value;
				}
				else
				{
					if (panimvalue->num.total > k + 1)
						angle2[j] = angle1[j];
					else
						angle2[j] = panimvalue[panimvalue->num.valid+2].value;
				}
			}
			else
			{
				angle1[j] = panimvalue[panimvalue->num.valid].value;
				if (panimvalue->num.total > k + 1)
				{
					angle2[j] = angle1[j];
				}
				else
				{
					angle2[j] = panimvalue[panimvalue->num.valid + 2].value;
				}
			}
			angle1[j] = pbone->value[j+3] + angle1[j] * pbone->scale[j+3];
			angle2[j] = pbone->value[j+3] + angle2[j] * pbone->scale[j+3];
		}

		if (pbone->bonecontroller[j+3] != -1)
		{
			angle1[j] += m_adj[pbone->bonecontroller[j+3]];
			angle2[j] += m_adj[pbone->bonecontroller[j+3]];
		}
	}

	if (!VectorCompare( angle1, angle2 ))
	{
		AngleQuaternion( angle1, q1 );
		AngleQuaternion( angle2, q2 );
		QuaternionSlerp( q1, q2, s, q );
	}
	else
	{
		AngleQuaternion( angle1, q );
	}
}


void StudioModel::CalcBonePosition( int frame, float s, mstudiobone_t *pbone, mstudioanim_t *panim, float *pos )
{
	int					j, k;
	mstudioanimvalue_t	*panimvalue;

	for (j = 0; j < 3; j++)
	{
		pos[j] = pbone->value[j]; // default;
		if (panim->offset[j] != 0)
		{
			panimvalue = (mstudioanimvalue_t *)((byte *)panim + panim->offset[j]);
			
			k = frame;
			// find span of values that includes the frame we want
			while (panimvalue->num.total <= k)
			{
				k -= panimvalue->num.total;
				panimvalue += panimvalue->num.valid + 1;
			}
			// if we're inside the span
			if (panimvalue->num.valid > k)
			{
				// and there's more data in the span
				if (panimvalue->num.valid > k + 1)
				{
					pos[j] += (panimvalue[k+1].value * (1.0 - s) + s * panimvalue[k+2].value) * pbone->scale[j];
				}
				else
				{
					pos[j] += panimvalue[k+1].value * pbone->scale[j];
				}
			}
			else
			{
				// are we at the end of the repeating values section and there's another section with data?
				if (panimvalue->num.total <= k + 1)
				{
					pos[j] += (panimvalue[panimvalue->num.valid].value * (1.0 - s) + s * panimvalue[panimvalue->num.valid + 2].value) * pbone->scale[j];
				}
				else
				{
					pos[j] += panimvalue[panimvalue->num.valid].value * pbone->scale[j];
				}
			}
		}
		if (pbone->bonecontroller[j] != -1)
		{
			pos[j] += m_adj[pbone->bonecontroller[j]];
		}
	}
}


void StudioModel::CalcRotations ( vec3_t *pos, vec4_t *q, mstudioseqdesc_t *pseqdesc, mstudioanim_t *panim, float f )
{
	int					i;
	int					frame;
	mstudiobone_t		*pbone;
	float				s;

	frame = (int)f;
	s = (f - frame);

	// add in programatic controllers
	CalcBoneAdj( );

	pbone		= (mstudiobone_t *)((byte *)m_pstudiohdr + m_pstudiohdr->boneindex);
	for (i = 0; i < m_pstudiohdr->numbones; i++, pbone++, panim++) 
	{
		CalcBoneQuaternion( frame, s, pbone, panim, q[i] );
		CalcBonePosition( frame, s, pbone, panim, pos[i] );
	}

	if (pseqdesc->motiontype & STUDIO_X)
		pos[pseqdesc->motionbone][0] = 0.0;
	if (pseqdesc->motiontype & STUDIO_Y)
		pos[pseqdesc->motionbone][1] = 0.0;
	if (pseqdesc->motiontype & STUDIO_Z)
		pos[pseqdesc->motionbone][2] = 0.0;
}


mstudioanim_t * StudioModel::GetAnim( mstudioseqdesc_t *pseqdesc )
{
	mstudioseqgroup_t	*pseqgroup;
	pseqgroup = (mstudioseqgroup_t *)((byte *)m_pstudiohdr + m_pstudiohdr->seqgroupindex) + pseqdesc->seqgroup;

	if (pseqdesc->seqgroup == 0)
	{
		return (mstudioanim_t *)((byte *)m_pstudiohdr + pseqgroup->data + pseqdesc->animindex);
	}

	return (mstudioanim_t *)((byte *)m_panimhdr[pseqdesc->seqgroup] + pseqdesc->animindex);
}


void StudioModel::SlerpBones( vec4_t q1[], vec3_t pos1[], vec4_t q2[], vec3_t pos2[], float s )
{
	int			i;
	vec4_t		q3;
	float		s1;

	if (s < 0) s = 0;
	else if (s > 1.0) s = 1.0;

	s1 = 1.0 - s;

	for (i = 0; i < m_pstudiohdr->numbones; i++)
	{
		QuaternionSlerp( q1[i], q2[i], s, q3 );
		q1[i][0] = q3[0];
		q1[i][1] = q3[1];
		q1[i][2] = q3[2];
		q1[i][3] = q3[3];
		pos1[i][0] = pos1[i][0] * s1 + pos2[i][0] * s;
		pos1[i][1] = pos1[i][1] * s1 + pos2[i][1] * s;
		pos1[i][2] = pos1[i][2] * s1 + pos2[i][2] * s;
	}
}


void StudioModel::AdvanceFrame( float dt )
{
	mstudioseqdesc_t	*pseqdesc;
	pseqdesc = (mstudioseqdesc_t *)((byte *)m_pstudiohdr + m_pstudiohdr->seqindex) + m_sequence;

	if (dt > 0.1)
		dt = 0.1;
	m_frame += dt * pseqdesc->fps;

	if (pseqdesc->numframes <= 1)
	{
		m_frame = 0;
	}
	else
	{
		// wrap
		m_frame -= (int)(m_frame / (pseqdesc->numframes - 1)) * (pseqdesc->numframes - 1);
	}
}


void StudioModel::SetUpBones ( void )
{
	int					i;

	mstudiobone_t		*pbones;
	mstudioseqdesc_t	*pseqdesc;
	mstudioanim_t		*panim;

	static vec3_t		pos[MAXSTUDIOBONES];
	float				bonematrix[3][4];
	static vec4_t		q[MAXSTUDIOBONES];

	static vec3_t		pos2[MAXSTUDIOBONES];
	static vec4_t		q2[MAXSTUDIOBONES];
	static vec3_t		pos3[MAXSTUDIOBONES];
	static vec4_t		q3[MAXSTUDIOBONES];
	static vec3_t		pos4[MAXSTUDIOBONES];
	static vec4_t		q4[MAXSTUDIOBONES];


	if (m_sequence >=  m_pstudiohdr->numseq) {
		m_sequence = 0;
	}

	pseqdesc = (mstudioseqdesc_t *)((byte *)m_pstudiohdr + m_pstudiohdr->seqindex) + m_sequence;

	panim = GetAnim( pseqdesc );
	CalcRotations( pos, q, pseqdesc, panim, m_frame );

	if (pseqdesc->numblends > 1)
	{
		float				s;

		panim += m_pstudiohdr->numbones;
		CalcRotations( pos2, q2, pseqdesc, panim, m_frame );
		s = m_blending[0] / 255.0;

		SlerpBones( q, pos, q2, pos2, s );

		if (pseqdesc->numblends == 4)
		{
			panim += m_pstudiohdr->numbones;
			CalcRotations( pos3, q3, pseqdesc, panim, m_frame );

			panim += m_pstudiohdr->numbones;
			CalcRotations( pos4, q4, pseqdesc, panim, m_frame );

			s = m_blending[0] / 255.0;
			SlerpBones( q3, pos3, q4, pos4, s );

			s = m_blending[1] / 255.0;
			SlerpBones( q, pos, q3, pos3, s );
		}
	}

	pbones = (mstudiobone_t *)((byte *)m_pstudiohdr + m_pstudiohdr->boneindex);

	for (i = 0; i < m_pstudiohdr->numbones; i++) {
		QuaternionMatrix( q[i], bonematrix );

		bonematrix[0][3] = pos[i][0];
		bonematrix[1][3] = pos[i][1];
		bonematrix[2][3] = pos[i][2];

		if (pbones[i].parent == -1) {
			memcpy(g_bonetransform[i], bonematrix, sizeof(float) * 12);
		} 
		else {
			R_ConcatTransforms (g_bonetransform[pbones[i].parent], bonematrix, g_bonetransform[i]);
		}
	}
}



/*
================
StudioModel::TransformFinalVert
================
*/
void StudioModel::Lighting (float *lv, int bone, int flags, vec3_t normal)
{
	float 	illum;
	float	lightcos;

	illum = g_ambientlight;

	if (flags & STUDIO_NF_FLATSHADE)
	{
		illum += g_shadelight * 0.8;
	} 
	else 
	{
		float r;
		lightcos = DotProduct (normal, g_blightvec[bone]); // -1 colinear, 1 opposite

		if (lightcos > 1.0)
			lightcos = 1;

		illum += g_shadelight;

		r = g_lambert;
		if (r <= 1.0) r = 1.0;

		lightcos = (lightcos + (r - 1.0)) / r; 		// do modified hemispherical lighting
		if (lightcos > 0.0) 
		{
			illum -= g_shadelight * lightcos; 
		}
		if (illum <= 0)
			illum = 0;
	}

	if (illum > 255) 
		illum = 255;
	*lv = illum / 255.0;	// Light from 0 to 1.0
}


void StudioModel::Chrome (int *pchrome, int bone, vec3_t normal)
{
	float n;

	if (g_chromeage[bone] != g_smodels_total)
	{
		// calculate vectors from the viewer to the bone. This roughly adjusts for position
		vec3_t chromeupvec;		// g_chrome t vector in world reference frame
		vec3_t chromerightvec;	// g_chrome s vector in world reference frame
		vec3_t tmp;				// vector pointing at bone in world reference frame
		VectorScale( m_origin, -1, tmp );
		tmp[0] += g_bonetransform[bone][0][3];
		tmp[1] += g_bonetransform[bone][1][3];
		tmp[2] += g_bonetransform[bone][2][3];
		VectorNormalize( tmp );
		CrossProduct( tmp, g_vright, chromeupvec );
		VectorNormalize( chromeupvec );
		CrossProduct( tmp, chromeupvec, chromerightvec );
		VectorNormalize( chromerightvec );

		VectorIRotate( chromeupvec, g_bonetransform[bone], g_chromeup[bone] );
		VectorIRotate( chromerightvec, g_bonetransform[bone], g_chromeright[bone] );

		g_chromeage[bone] = g_smodels_total;
	}

	// calc s coord
	n = DotProduct( normal, g_chromeright[bone] );
	pchrome[0] = (n + 1.0) * 32.0; // FIX: make this a float

	// calc t coord
	n = DotProduct( normal, g_chromeup[bone] );
	pchrome[1] = (n + 1.0) * 32.0; // FIX: make this a float
}


/*
================
StudioModel::SetupLighting
	set some global variables based on entity position
inputs:
outputs:
	g_ambientlight
	g_shadelight
================
*/
void StudioModel::SetupLighting ( )
{
	int i;
	g_ambientlight = 32;
	g_shadelight = 192;

	g_lightvec[0] = -1.0;
	g_lightvec[1] = 0.0;
	g_lightvec[2] = 0.0;

	g_lightcolor[0] = m_lightr;
	g_lightcolor[1] = m_lightg;
	g_lightcolor[2] = m_lightb;

	g_vright[0] = 1.0;
	g_vright[1] = 0;
	g_vright[2] = 0;

	// TODO: only do it for bones that actually have textures
	for (i = 0; i < m_pstudiohdr->numbones; i++)
	{
		VectorIRotate( g_lightvec, g_bonetransform[i], g_blightvec[i] );
	}
}


/*
=================
StudioModel::SetupModel
	based on the body part, figure out which mesh it should be using.
inputs:
	currententity
outputs:
	pstudiomesh
	pmdl
=================
*/

void StudioModel::SetupModel ( int bodypart )
{
	int index;

	if (bodypart > m_pstudiohdr->numbodyparts)
	{
		bodypart = 0;
	}

	mstudiobodyparts_t   *pbodypart = (mstudiobodyparts_t *)((byte *)m_pstudiohdr + m_pstudiohdr->bodypartindex) + bodypart;

	index = m_bodynum / pbodypart->base;
	index = index % pbodypart->nummodels;

	maxnummodels=pbodypart->base * pbodypart->nummodels;

	m_pmodel = (mstudiomodel_t *)((byte *)m_pstudiohdr + pbodypart->modelindex) + index;
}


/*
================
StudioModel::DrawModel
inputs:
	currententity
	r_entorigin
================
*/
void StudioModel::DrawModel( )
{
	int i;
	float	fog_color[4];

	g_smodels_total++; // render data cache cookie

	g_pxformverts = &g_xformverts[0];
	g_pvlightvalues = &g_lightvalues[0];

	if (m_pstudiohdr->numbodyparts == 0)
		return;

	glPushMatrix ();
    glTranslatef (m_origin[0],  m_origin[1],  m_origin[2]);

    glRotatef (m_angles[1],  0, 0, 1);
    glRotatef (m_angles[0],  0, 1, 0);
    glRotatef (m_angles[2],  1, 0, 0);

	// toggle flag for flickering selection of texture.
	m_bTexSelectFlickerFlag=(m_bTexSelectFlickerFlag==false);

	m_fDstAlpha = 1.0f;

	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	if(rendermode==RM_ADDITIVE)
	{
		// do additive rendering like HL gameengine does too
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA, GL_SRC_ALPHA);
	}

	if(fog_enabled)
	{
		glEnable(GL_FOG);

		// set fogcolor
		fog_color[0] = fg_r;
		fog_color[1] = fg_g;
		fog_color[2] = fg_b;
		fog_color[3] = 1.0;	

		glFogf(GL_FOG_MODE,GL_LINEAR);
		glFogfv(GL_FOG_COLOR,fog_color);

		// insert fogstartpos
		glFogf(GL_FOG_START,fog_start);

		// insert fogendpos
		glFogf(GL_FOG_END,fog_end);
	}
	else
	{
		glDisable(GL_FOG);
	}

    glClearColor( fg_r, fg_g, fg_b, 1.0 );

	SetUpBones ( );

	SetupLighting( );

	maxnummodels = 0;
	for (i=0 ; i < m_pstudiohdr->numbodyparts ; i++) 
	{
		SetupModel( i );
		DrawPoints(rendermode);
	}

	if((rendermode==RM_ADDITIVE)||(rendermode==RM_LINEDRAW))
	{
		// disable depthtest because we draw the bones on top of the pixels already there.
		glDisable(GL_DEPTH_TEST);
		DrawBones();
		glEnable(GL_DEPTH_TEST);
	}

	if(rendermode==RM_ADDITIVE)
	{
		glDisable(GL_BLEND);
	}

	glPopMatrix ();
}


void
StudioModel::DrawIcosahedron(float x,float y,float z,float size)
{
	float a = 0.525731112119133606 * size;
	float b = 0.850650808352039932 * size;

	GLfloat Vertices[12][3] = {
		{ -a,0.0,  b},{  a,0.0,  b},{ -a,0.0,  -b},{  a,0.0, -b},
		{0.0,  b,  a},{0.0,  b, -a},{0.0, -b,  a},{0.0, -b, -a},
		{  b,  a,0.0},{ -b,  a,0.0},{  b, -a,0.0},{ -b, -a,0.0}		
	};

	glPushMatrix();

	glTranslatef(x,y,z);

	int i;

	glBegin(GL_TRIANGLES);
		for(i=0;i<20;i++)
		{
			glVertex3fv(&Vertices[Indices[i*3]][0]);
			glVertex3fv(&Vertices[Indices[i*3+1]][0]);
			glVertex3fv(&Vertices[Indices[i*3+2]][0]);
		};
	glEnd();	

	glPopMatrix();
}

void
StudioModel::DrawBones()
{
	int i;
	mstudiobone_t		*pBonez = (mstudiobone_t *)((byte *)m_pstudiohdr + m_pstudiohdr->boneindex);

	glLineWidth(2.0f);

	for( i=0; i<m_pstudiohdr->numbones; i++)
	{
		// If the bone is a parent just paint a Icosahedron
		if(pBonez[i].parent == -1)
		{
			glColor4f(1.0f,0.0f,0.0f, 1.0f);
			DrawIcosahedron(g_bonetransform[i][0][3],g_bonetransform[i][1][3],g_bonetransform[i][2][3],1.0f);
		}
		else
		// else paint a Icosahedron with a line to bone's parent
		{
			glColor4f(0.25f,0.25f,1.0f,1.0f);
			DrawIcosahedron(g_bonetransform[i][0][3],g_bonetransform[i][1][3],g_bonetransform[i][2][3],0.25f);

			glBegin(GL_LINES);
				glColor4f(1.0f,1.0f,0.0f,1.0f);
				glVertex3f(g_bonetransform[pBonez[i].parent][0][3],g_bonetransform[pBonez[i].parent][1][3],g_bonetransform[pBonez[i].parent][2][3]);
				glColor4f(1.0f,1.0f,0.5f, 1.0f);
				glVertex3f(g_bonetransform[i][0][3],g_bonetransform[i][1][3],g_bonetransform[i][2][3]);
			glEnd();
		}
	}
	glLineWidth(1.0f);
}


void StudioModel::DrawPoints (int mode)
{
	int					i, j;
	mstudiomesh_t		*pmesh;
	byte				*pvertbone;
	byte				*pnormbone;
	vec3_t				*pstudioverts;
	vec3_t				*pstudionorms;
	mstudiotexture_t	*ptexture;
	float 				*av;
	float				*lv;
	float				lv_tmp;
	short				*pskinref;
	bool				bTextured;

	
	bTextured = false;

	pvertbone = ((byte *)m_pstudiohdr + m_pmodel->vertinfoindex);
	pnormbone = ((byte *)m_pstudiohdr + m_pmodel->norminfoindex);
	ptexture = (mstudiotexture_t *)((byte *)m_ptexturehdr + m_ptexturehdr->textureindex);

	pmesh = (mstudiomesh_t *)((byte *)m_pstudiohdr + m_pmodel->meshindex);

	pstudioverts = (vec3_t *)((byte *)m_pstudiohdr + m_pmodel->vertindex);
	pstudionorms = (vec3_t *)((byte *)m_pstudiohdr + m_pmodel->normindex);

	pskinref = (short *)((byte *)m_ptexturehdr + m_ptexturehdr->skinindex);
	if (m_skinnum != 0 && m_skinnum < m_ptexturehdr->numskinfamilies)
		pskinref += (m_skinnum * m_ptexturehdr->numskinref);
	else
		m_skinnum = 0;

	for (i = 0; i < m_pmodel->numverts; i++)
	{
		VectorTransform (pstudioverts[i], g_bonetransform[pvertbone[i]], g_pxformverts[i]);
	}

	//
	// clip and draw all triangles
	//

	switch(mode)
	{
		case RM_LINEDRAW:
		{
			glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
			glDisable(GL_TEXTURE_2D);
			bTextured=false;
		}; break;
		case RM_TEXTURED:
		{
			glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
			glEnable(GL_TEXTURE_2D);
			bTextured=true;
		}; break;
		case RM_ADDITIVE:
		{
			glPolygonMode(GL_FRONT,GL_LINE);
			glPolygonMode(GL_BACK,GL_FILL);
			glEnable(GL_TEXTURE_2D);
			bTextured=true;
		}; break;
		case RM_SHADED:
		{
			glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
			glDisable(GL_TEXTURE_2D);
			bTextured=false;
		}; break;
	};
	lv = (float *)g_pvlightvalues;
	for (j = 0; j < m_pmodel->nummesh; j++) 
	{
		int flags;
		flags = ptexture[pskinref[pmesh[j].skinref]].flags;
		for (i = 0; i < pmesh[j].numnorms; i++, lv += 3, pstudionorms++, pnormbone++)
		{
			Lighting (&lv_tmp, *pnormbone, flags, (float *)pstudionorms);

			if (flags & STUDIO_NF_CHROME)
				Chrome( g_chrome[(float (*)[3])lv - g_pvlightvalues], *pnormbone, (float *)pstudionorms );

			lv[0] = lv_tmp * g_lightcolor[0];
			lv[1] = lv_tmp * g_lightcolor[1];
			lv[2] = lv_tmp * g_lightcolor[2];
		}
	}

	for (j = 0; j < m_pmodel->nummesh; j++) 
	{
		float s, t;
		short		*ptricmds;

		pmesh = (mstudiomesh_t *)((byte *)m_pstudiohdr + m_pmodel->meshindex) + j;
		ptricmds = (short *)((byte *)m_pstudiohdr + pmesh->triindex);

		s = 1.0/(float)ptexture[pskinref[pmesh->skinref]].width;
		t = 1.0/(float)ptexture[pskinref[pmesh->skinref]].height;

		// now check if we have to mark the current texture as selected...
		// first turn on texturing if we've disabled that in a previous looprun
		if((bTextured)&&(!glIsEnabled(GL_TEXTURE_2D)))
		{
			glEnable(GL_TEXTURE_2D);
		}

		// default shadingmodel
		glShadeModel(GL_SMOOTH);

		if(m_iCurrentTexture==ptexture[pskinref[pmesh->skinref]].index)
		{
			// yup. if textured, switch it off to mark selected to the user
			if(glIsEnabled(GL_TEXTURE_2D))
			{
				// test if flickerflag is ON (true), then reset texturing, otherwise don't.
				if(m_bTexSelectFlickerFlag)
				{
					glDisable(GL_TEXTURE_2D);
				}
			}
			// disable shading
			glShadeModel(GL_FLAT);
		}

		glBindTexture( GL_TEXTURE_2D, ptexture[pskinref[pmesh->skinref]].index );

		if (ptexture[pskinref[pmesh->skinref]].flags & STUDIO_NF_CHROME)
		{
			while (i = *(ptricmds++))
			{
				if (i < 0)
				{
					glBegin( GL_TRIANGLE_FAN );
					i = -i;
				}
				else
				{
					glBegin( GL_TRIANGLE_STRIP );
				}


				for( ; i > 0; i--, ptricmds += 4)
				{
					// FIX: put these in as integer coords, not floats
					glTexCoord2f(g_chrome[ptricmds[1]][0]*s, g_chrome[ptricmds[1]][1]*t);
					
					lv = g_pvlightvalues[ptricmds[1]];
					glColor4f( lv[0], lv[1], lv[2], m_fDstAlpha);

					av = g_pxformverts[ptricmds[0]];
					glVertex3f(av[0], av[1], av[2]);
				}
				glEnd( );
			}	
		} 
		else 
		{
			while (i = *(ptricmds++))
			{
				if (i < 0)
				{
					glBegin(GL_TRIANGLE_FAN);
					i = -i;
				}
				else
				{
					glBegin(GL_TRIANGLE_STRIP);
				}


				for( ; i > 0; i--, ptricmds += 4)
				{
					// FIX: put these in as integer coords, not floats
					glTexCoord2f(ptricmds[2]*s, ptricmds[3]*t);
					
					lv = g_pvlightvalues[ptricmds[1]];
					glColor4f( lv[0], lv[1], lv[2],m_fDstAlpha );

					av = g_pxformverts[ptricmds[0]];
					glVertex3f(av[0], av[1], av[2]);
				}
				glEnd( );
			}	
		}
	}
}

