/*****************************************************
Copyright Notice & Disclaimer

Copyright  Ascent, Dark Yoda and Otis

Permission to use, copy, modify, and distribute this software
and its documentation for any purpose is hereby granted without
fee, provided that the above copyright notice, author statement
appear in all copies of this software and related documentation.

THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF
ANY KIND, EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT
LIMITATION, ANY WARRANTY OF MERCHANTABILITY OR FITNESS FOR A
PARTICULAR PURPOSE.

IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY
SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY
KIND, OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA
OR PROFITS, WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE,
AND ON ANY THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION
WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
******************************************************/

//////////////////////////////////////////////////////////////////////
// FILE: Mdl2View.cpp
// PURPOSE: Implementation of the Mdl2View class. 
//////////////////////////////////////////////////////////////////////
// SHORT DESCRIPTION:
// This class implements the OpenGL view on the model stored in the document attached
// to this view. Every model has 1 documentobject that holds the modeldata, every
// document has 1 view, implemented in this class
// This class used GLEnabledView class, a subclass of CView.
//////////////////////////////////////////////////////////////////////
// COPYRIGHTS:
//
// Programmed by:
// Frans 'Otis' Bouma,
// Greg 'Ascent' Dunn,
// Volker 'Dark Yoda' Schnefeld
//
// Code by Otis is (c)1999 Solutions Design, http://www.sd.nl
// Code by Ascent is (c)1999 Greg Dunn.
// Code by Dark Yoda is (c)1999 Volker Schnefeld.
// All rights reserved.
//////////////////////////////////////////////////////////////////////
// VERSION INFORMATION.
//
// 02-april-1999
// Release 2.1
// [OTIS] Added skinbrowsing
// [OTIS] Added disable/enable code for buttons in dialog
// [OTIS] Added flickerselection for textures
//
// 31-mar-1999
// [OTIS] Fixed resize bug
// [OTIS] Included reinit cycle system for texture refresh
// [OTIS] Implemented texture import/export
//
// 25-mar-1999
// [OTIS] Added texture selection, chromeflag toggle
// [OTIS] Added modelsave
// [OTIS] Improved initialize/loading scheme. Now bogus files are not loaded and no view is displayed
//
// 23-mar-1999
// [OTIS] Added texture deletion when unloading models plus fixed OnDestroy bug
//
// 20-mar-1999
// [OTIS] Rewritten code for new naming schedule. Like hungarian.
// [OTIS] Move of iD and Valve c-lib files into the code dir.
// [OTIS] Cleaned up code by removing not used code and definitions.
//
// 16-mar-1999.
//		First public release.
//
// 08-feb-1999. 
//		First Internal Version 
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "mdlv2.h"
#include "gl/gl.h"
#include "gl/glu.h"
#include "GLEnabledView.h"
#include "MdlView.h"
#include "mdlv2Doc.h"
#include "mdlv2View.h"

IMPLEMENT_DYNCREATE(CMdlv2View, CGLEnabledView )

BEGIN_MESSAGE_MAP(CMdlv2View, CGLEnabledView )
	//{{AFX_MSG_MAP(CMdlv2View)
	ON_WM_LBUTTONUP()
	ON_WM_LBUTTONDOWN()
	ON_WM_MOUSEMOVE()
	ON_WM_MOUSEWHEEL()
	ON_WM_RBUTTONDOWN()
	ON_WM_RBUTTONUP()
	ON_WM_TIMER()
	ON_WM_KEYDOWN()
	ON_WM_SETFOCUS()
	ON_WM_DESTROY()
	ON_COMMAND(ID_FILE_SAVEAS, OnFileSaveas)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


// Purpose: Constructor
CMdlv2View::CMdlv2View()
{
	m_btMouseButton = 0;
	m_bInitialized = false;
	m_bBadFile=false;
}


// Purpose: Destructor
CMdlv2View::~CMdlv2View()
{
}


#ifdef _DEBUG
void CMdlv2View::AssertValid() const
{
	CGLEnabledView ::AssertValid();
}

void CMdlv2View::Dump(CDumpContext& dc) const
{
	CGLEnabledView ::Dump(dc);
}


CMdlv2Doc* 
CMdlv2View::GetDocument() // non-debug version is inline
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CMdlv2Doc)));
	return (CMdlv2Doc*)m_pDocument;
}
#endif //_DEBUG


// Purpose: set pixeldepth for OpenGL
// Additional info: function overrides CGLEnabledView's VideoMode function
void
CMdlv2View::VideoMode(ColorsNumber & c, ZAccuracy & z, BOOL & dbuf)
{
	// ask for millions of colors...
	c=MILLIONS_WITH_TRANSPARENCY;
	// ...32 bit Z-buffer...
	z=ACCURATE;
	// ...double-buffering
	dbuf=true;
}


// Purpose: the core drawroutine. 
// Additional info: function overrides CGLEnabledView's OnDrawGL
// This routine contains also the init start calls of the model. 
void
CMdlv2View::OnDrawGL()
{
	CMdlView	*pModel;
	CRect		crWndSize;

	CMdlv2Doc *pDoc = GetDocument();

	if(pDoc==NULL)
	{
		return;
	}

	pModel = pDoc->GetMdlView();

	// check if we have to start a new initialisation cycle
	if(pModel->CheckDoInit())
	{
		// yes. so reset flag that initialisation is done. we will enter a new
		// initialisation cycle automatically
		m_bInitialized=false;
	}

	BeginGLCommands();
	if(!m_bInitialized)
	{
		if(pModel->IsModelLoaded())
		{
			if(!pModel->IsModelInitialized())
			{
				pModel->InitModel();
				pModel->SetupRenderer();
				CWnd::SetTimer(1,10,NULL);
				// size the window so we see something. 
				GetClientRect(&crWndSize);
				CGLEnabledView::OnSizeGL(crWndSize.right,crWndSize.bottom);
				m_bInitialized = true;
				pDoc->SetActive();	// set Control Center...
			}
		}
	}
	else
	{
		pModel->RenderFrame();
	}
	EndGLCommands();
}


// Purpose: WM_LBUTTONUP handler
void CMdlv2View::OnLButtonUp(UINT nFlags, CPoint point) 
{
	// forget where we clicked
	m_ptMouseDownPoint=CPoint(0,0);
	// release mouse capture
	ReleaseCapture();
	// reset flag for leftmousebutton
	m_btMouseButton &=!LMB;
}


// Purpose: WM_LBUTTONDOWN handler
void CMdlv2View::OnLButtonDown(UINT nFlags, CPoint point) 
{
	HCURSOR hcCurHandle;

	// remember where we clicked
	m_ptMouseDownPoint=point;
	// capture mouse movements even outside window borders
	SetCapture();
	// add flag for left mouse button
	m_btMouseButton |= LMB;

	hcCurHandle = AfxGetApp()->LoadCursor(IDC_CURSOR1);
	SetCursor(hcCurHandle);
}


// Purpose: WM_MOUSEMOVE handler
void CMdlv2View::OnMouseMove(UINT nFlags, CPoint point) 
{
	CMdlView *pModel;

	CMdlv2Doc *pDoc = GetDocument();
	pModel = pDoc->GetMdlView();

	// check if we have captured the mouse
	if (GetCapture()==this)
	{
		if(m_btMouseButton & LMB)
		{
			// pan
			pModel->Pan(int(point.x-m_ptMouseDownPoint.x),int(point.y-m_ptMouseDownPoint.y));
		}
		if(m_btMouseButton & RMB)
		{
			// rotate
			pModel->Rotate(int(point.x-m_ptMouseDownPoint.x),int(point.y-m_ptMouseDownPoint.y));
		}
		// redraw the view
		Invalidate(true);
		// remember the mouse point
		m_ptMouseDownPoint=point;
	}
}


// Purpose: WM_RBUTTONDOWN handler
void CMdlv2View::OnRButtonDown(UINT nFlags, CPoint point) 
{
	HCURSOR hcCurHandle;

	// remember where we clicked
	m_ptMouseDownPoint=point;
	// capture mouse movements even outside window borders
	SetCapture();
	m_btMouseButton |= RMB;

	hcCurHandle = AfxGetApp()->LoadCursor(IDC_CURSOR2);
	SetCursor(hcCurHandle);
}


// Purpose: WM_RBUTTONUP handler
void CMdlv2View::OnRButtonUp(UINT nFlags, CPoint point) 
{
	// forget where we clicked
	m_ptMouseDownPoint=CPoint(0,0);
	// release mouse capture
	ReleaseCapture();
	m_btMouseButton &= !RMB;
}


// Purpose: WM_MOUSEWHEEL handler
BOOL CMdlv2View::OnMouseWheel(UINT nFlags, short zDelta, CPoint pt) 
{
	CMdlView *pModel;

	CMdlv2Doc *pDoc = GetDocument();
	pModel = pDoc->GetMdlView();

	pModel->Zoom(zDelta/50);
	// redraw the view
	Invalidate(true);
	return TRUE;
}


// Purpose: WM_TIMER handler. 
// Additional info: this function is called every 10 milliseconds to get animation.
void CMdlv2View::OnTimer(UINT nIDEvent) 
{
	Invalidate(true);
}


// Purpose: WM_KEYDOWN handler. 
// Additional info: called when the user presses a key in actual view that has the focus
void CMdlv2View::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	CMdlView	*pModel;
	bool		bFogStatus;

	CMdlv2Doc *pDoc = GetDocument();
	pModel = pDoc->GetMdlView();

	switch(nChar)
	{
		case 0x5a: // Z key
		{
			// zoom in
			pModel->Zoom(1);
		}; break;
		case 0x58: // X key
		{
			pModel->Zoom(-1);
			// zoom out
		}; break;
		case 0x4e: // N key
		{
			// next sequence
			pModel->NextSequence();
		}; break;
		case 0x50: // P key
		{
			// previous sequence
			pModel->PrevSequence();
		}; break;
		case 0x42: // B key
		{
			// next body
			pModel->NextBody();
		}; break;
		case 0x46: // F key
		{
			// Toggle Fog
			bFogStatus = pModel->GetFogToggleStatus();
			// swap state
			bFogStatus = (bFogStatus == false);

			pModel->ToggleFog(bFogStatus);
		}; break;
		case 0x6B: // + key
		{
			// increase fog distance
			pModel->IncFogDistance();
		}; break;
		case 0x6D: // - key
		{
			// decrease fog distance
			pModel->DecFogDistance();
		}; break;
		case 0x52: // R key
		{
			// toggle rendermode
			pModel->ToggleRenderMode();
		}; break;
	}
	// update ModelControlCenter...
	pDoc->UpdateMCC();

	// update screen..
	Invalidate(true);
}


// Purpose: WM_SETFOCUS handler
// Additional info: used to sync MCC with current active modelview
void 
CMdlv2View::OnSetFocus(CWnd* pOldWnd) 
{
	CGLEnabledView ::OnSetFocus(pOldWnd);
	
	if(!m_bBadFile)
	{
		// notify the document that we're got the focus so the model control center
		// should be updated.
		CMdlv2Doc *pDoc = GetDocument();
		pDoc->SetActive();
	}
}


// Purpose: WM_DESTROY handler. 
// Additional info: used to destroy the binded textures from the OpenGL Texture Heaven.
void CMdlv2View::OnDestroy() 
{
	CMdlView	*pModel;

	CMdlv2Doc *pDoc = GetDocument();
	pModel = pDoc->GetMdlView();
	
	// get rid of the binded textures. 
	BeginGLCommands();
	pModel->DelTextures();
	EndGLCommands();

	// kill windowtimer
	CWnd::KillTimer(1);

	// bubble destroy to parentclass
	CGLEnabledView ::OnDestroyGL();
}


// Purpose: handle save as menu command...
void
CMdlv2View::OnFileSaveas() 
{
	CMdlv2Doc *pDoc = GetDocument();

	// signal doc that this model should be saved...
	pDoc->SaveModel();
}
