/*****************************************************
Copyright Notice & Disclaimer

Copyright  Ascent, Dark Yoda and Otis

Permission to use, copy, modify, and distribute this software
and its documentation for any purpose is hereby granted without
fee, provided that the above copyright notice, author statement
appear in all copies of this software and related documentation.

THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF
ANY KIND, EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT
LIMITATION, ANY WARRANTY OF MERCHANTABILITY OR FITNESS FOR A
PARTICULAR PURPOSE.

IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY
SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY
KIND, OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA
OR PROFITS, WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE,
AND ON ANY THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION
WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
******************************************************/
//////////////////////////////////////////////////////////////////////
// FILE: MdlView.cpp
// PURPOSE: Implementation of the MdlView class. 
//////////////////////////////////////////////////////////////////////
// SHORT DESCRIPTION:
// This class creates, from the model that should be loaded, a StudioModel Object. It controls and draws
// this model. Create an Object instance of this class to view it in an already created RenderingContext,
// and use the instance of this object to control the model
//////////////////////////////////////////////////////////////////////
// COPYRIGHTS:
// Note: Based on Valve Software's modelviewer 1.0 code. Some code licensed from iD software.
// (c)1996-1999 by the owners of the codeparts.
//
// Programmed by:
// Frans 'Otis' Bouma,
// Greg 'Ascent' Dunn,
// Volker 'Dark Yoda' Schnefeld
//
// Code by Otis is (c)1999 Solutions Design, http://www.sd.nl
// Code by Ascent is (c)1999 Greg Dunn.
// Code by Dark Yoda is (c)1999 Volker Schnefeld.
// All rights reserved.
//////////////////////////////////////////////////////////////////////
// VERSION INFORMATION.
//
// 02-april-1999
// Release 2.1
// [OTIS] Added skinbrowsing
// [OTIS] Added disable/enable code for buttons in dialog
// [OTIS] Added flickerselection for textures
//
// 31-mar-1999
// [OTIS] Fixed resize bug
// [OTIS] Included reinit cycle system for texture refresh
// [OTIS] Implemented texture import/export
//
// 25-mar-1999
// [OTIS] Added texture selection, chromeflag toggle
// [OTIS] Added modelsave
// [OTIS] Improved initialize/loading scheme. Now bogus files are not loaded and no view is displayed
//
// 20-mar-1999
// [OTIS] Rewritten code for new naming schedule. Like hungarian.
// [OTIS] Move of iD and Valve c-lib files into the code dir.
// [OTIS] Cleaned up code by removing not used code and definitions.
// [OTIS] added RM_ADDITIVE rendermode: bones AND transparent textures.
//
// 16-mar-1999.
//		First public release.
//
// 08-feb-1999. 
//		First Internal Version 
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "mathlib.h"
#include "studio.h"
#include "mdlv2.h"
#include "StudioModel.h"
#include "MdlView.h"

// the modelobject that is stored inside this mdlview object and where all the functions work on.
StudioModel m_smCurrentModel;


// Purpose: Constructor
CMdlView::CMdlView()
{
	// Initialize vars. For now these will do,
	m_fGlDepthMin = 0.0f;
	m_fGlDepthMax = 10.0f;
	m_iRenderMode = RM_TEXTURED;
	m_fBgColorRed=m_fBgColorGreen=m_fBgColorBlue=0.0f;
	m_fLightColorRed=m_fLightColorGreen=m_fLightColorBlue=1.0f;
	m_fTransX = 0.0f;
	m_fTransY = -.5f;
	m_fTransZ = 1.5f;
	m_fRotX=235.0f;
	m_fRotY=-90.0f;
	m_iBodyGrp = 0;
	m_bFogEnabled = false;
	m_fFogStart = 1.6f;
	m_fFogStop = 1.2f;
	m_fCurrentFrame=m_fPrevFrame=0.0f;
	m_bModelLoaded = false;
	m_bModelInitialized = false;
}


// Purpose: Destructor
CMdlView::~CMdlView()
{
}


// Purpose: return the status of the signalflag inside the model if this model wants a new initialisationcycle done
bool
CMdlView::CheckDoInit()
{
	bool	bResult;

	if(m_bModelLoaded)
	{
		bResult=m_smCurrentModel.CheckDoInit();
		if(bResult)
		{
			// there will be a new initialisation cycle done in the next frame. Reset the initialized flag, because
			// we will be REinitialized.
			m_bModelInitialized=false;
		}
	}
	return bResult;
}


// Purpose: load model stored in passed filename
// Additional info: current model in core will be cleared first. This is if you
// will reuse this mdlview object which is not recommended.
bool
CMdlView::LoadModel(char *cpFilename)
{
	bool	bResult;

	// call clear function in m_smCurrentModel to get rid of current model in core. Then load new one.
	m_smCurrentModel.Clear();

	// catch init returncode. True if everything went ok, False otherwise. 
	bResult = m_smCurrentModel.LoadModel(cpFilename);
	if(bResult)
	{
		m_bModelLoaded = true;
	}
	return bResult;
}


// Purpose: init model loaded
void
CMdlView::InitModel()
{
	m_smCurrentModel.Init();
	m_smCurrentModel.SetSequence(0);
	m_smCurrentModel.SetController(0,0.0);
	m_smCurrentModel.SetController(1,0.0);
	m_smCurrentModel.SetController(2,0.0);
	m_smCurrentModel.SetController(3,0.0);
	m_smCurrentModel.SetMouth(0);
	m_smCurrentModel.SetRenderMode(m_iRenderMode);
	m_fCurrentFrame=m_fPrevFrame=0.0;
	m_smCurrentModel.SetFog(m_bFogEnabled,m_fFogStart,m_fFogStop);
	m_smCurrentModel.SetFogColor(m_fBgColorRed,m_fBgColorGreen,m_fBgColorBlue);
	m_smCurrentModel.SetBodyNr(m_iBodyGrp);
	m_smCurrentModel.SetSkin(0);
	m_smCurrentModel.SetLightColor(m_fLightColorRed,m_fLightColorGreen,m_fLightColorBlue);
	m_bModelInitialized = true;
}


void
CMdlView::SaveModel(CString sPathFilename, CString sFileTitle)
{
	m_smCurrentModel.SaveModel(sPathFilename,sFileTitle);
}


// Purpose: sets up OpenGL viewport for this model.
void
CMdlView::SetupRenderer()
{
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(50.,1.,.1.,10.);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glClearColor( m_fBgColorRed, m_fBgColorGreen, m_fBgColorBlue, 0 );
}


// Purpose: wrapper function to advance the model 1 frame and get that frame rendered.
void
CMdlView::RenderFrame()
{
	if(!m_bModelInitialized)
	{
		return;
	}

	// draw on backbuffer.
    glDrawBuffer(GL_BACK);
	// clear the buffer and z-buffer
	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
    glPushMatrix();
		glTranslatef(m_fTransX, m_fTransY, m_fTransZ);
		glRotatef(m_fRotX, 0.0f, 1.0f, 0.0f);
		glRotatef(m_fRotY, 1.0f, 0.0f, 0.0f);
		glScalef( 0.01f, 0.01f, 0.01f );

		// re-order polys so first poly is indeed first
		glCullFace( GL_FRONT );
		glEnable( GL_DEPTH_TEST );

		// render the actual frame
		DoRenderModelFrame();
    glPopMatrix();
}


// Purpose: renders a new frame of current model in core.
// Additional info: OpenGL has to be setup.
void
CMdlView::DoRenderModelFrame()
{
	// reset depth masks
	glDepthFunc(GL_LEQUAL);
	glDepthRange(m_fGlDepthMin, m_fGlDepthMax);
	glDepthMask(1);

	// set blendingsettings
	m_smCurrentModel.SetBlending(0, 0.0f);
	m_smCurrentModel.SetBlending(1, 0.0f);

	// get currentframeticker index
	m_fCurrentFrame = GetTickCount( ) / 1000.0f;
	
	// go to next frame if we have to (on low end machines this will be later)
	m_smCurrentModel.AdvanceFrame( m_fCurrentFrame - m_fPrevFrame );

	// store current tick
	m_fPrevFrame = m_fCurrentFrame;

	// draw the model!
	m_smCurrentModel.DrawModel();
}


// Purpose: recalculate the angles where the CAMERA is rotated on. 
// Additional info: rotation is done when the model is drawn
void
CMdlView::Rotate(int iXDelta, int iYDelta) 
{
    m_fRotX += iXDelta;
    if(m_fRotX > 360.)
	{
		m_fRotX -= 360.;
	}
    else
	{
		if(m_fRotX < -360.)
		{
			m_fRotX += 360.;
		}
	}
    m_fRotY += iYDelta;
    if(m_fRotY > 360.)
	{
		m_fRotY -= 360.;
	}
    else
	{
		if (m_fRotY < -360.)
		{
			m_fRotY += 360.;
		}
	}
}


// Purpose: move CAMERA around in x/y pane.
// Additional info: actual move is done when model is drawn
void
CMdlView::Pan(int iXDelta, int iYDelta)
{
    m_fTransX += iXDelta/500.0f;
    m_fTransY -= iYDelta/500.0f;
}


// Purpose: zoom SCENE by modifying cameratransformation parameter for Z.
// Additional info: actual zooming is done when model is drawn.
void
CMdlView::Zoom(short sZDelta)
{
    m_fTransZ +=  sZDelta/20.0f;
}


// Purpose: set current sequence to the next sequence in line
void
CMdlView::NextSequence()
{
	int iSequenceNr;

	iSequenceNr = m_smCurrentModel.GetSequence();
	m_smCurrentModel.SetSequence(iSequenceNr+1);
}


// Purpose: set current sequence to the previous sequence in line
void
CMdlView::PrevSequence()
{
	int iSequenceNr;
	
	iSequenceNr = m_smCurrentModel.GetSequence();
	m_smCurrentModel.SetSequence(iSequenceNr-1);
}


// Purpose: switch fog on or off
void
CMdlView::ToggleFog(bool bFogOn)
{
	m_bFogEnabled = bFogOn;
	m_smCurrentModel.SetFog(m_bFogEnabled, m_fFogStart,m_fFogStop);
}


// Purpose: get the maximum sequence nr, i.e. the number of sequences-1
int
CMdlView::GetMaxSequenceNr()
{
	return m_smCurrentModel.GetSeqAmount();
}


// Purpose: get fogswitch status for MCC.
bool
CMdlView::GetFogToggleStatus()
{
	return m_bFogEnabled;
}


// Purpose: increase the distance between camera and the start of the fog
void
CMdlView::IncFogDistance()
{
	m_fFogStart -= 0.01f;
	m_fFogStop = m_fFogStart - 0.5f;
	m_smCurrentModel.SetFog(m_bFogEnabled, m_fFogStart,m_fFogStop);
}


// Purpose: decrease the distance between camera and start of fog
void
CMdlView::DecFogDistance()
{
	m_fFogStart += 0.01f;
	m_fFogStop = m_fFogStart - 0.5f;
	m_smCurrentModel.SetFog(m_bFogEnabled, m_fFogStart,m_fFogStop);
}


// Purpose: set current body to the next possible in line
void
CMdlView::NextBody()
{
	++m_iBodyGrp %= m_smCurrentModel.GetBodyAmount();
	m_smCurrentModel.SetBodyNr(m_iBodyGrp);
}


// purpose: cycle through the defined rendermodes
void
CMdlView::ToggleRenderMode()
{
	++m_iRenderMode%=MAXRENDERMODES;
	m_smCurrentModel.SetRenderMode(m_iRenderMode);
}


// Purpose: set background color. This color is also used for fog, otherwise you won't see a fog.
void
CMdlView::SetBGColor(int iRed, int iGreen, int iBlue)
{
	m_fBgColorRed = (float)iRed/255;
	m_fBgColorGreen = (float)iGreen/255;
	m_fBgColorBlue = (float)iBlue/255;
    glClearColor( m_fBgColorRed, m_fBgColorGreen, m_fBgColorBlue, 0 );
	m_smCurrentModel.SetFogColor(m_fBgColorRed,m_fBgColorGreen,m_fBgColorBlue);
}


// Purpose: returns Current Sequence nr. 
int
CMdlView::GetCurrentSequenceNr()
{
	return m_smCurrentModel.GetSequence();
}


// Purpose: returns Current sequence name. 
char
*CMdlView::GetCurrentSequenceName()
{
	return m_smCurrentModel.GetSequenceLabel();
}


// Purpose: returns current sequence length in milliseconds
long
CMdlView::GetCurrentSequenceLength()
{
	return m_smCurrentModel.GetSequenceLength();
}


// Purpose: returns current body nr.
int
CMdlView::GetCurrentBodyNr()
{
	return m_smCurrentModel.GetCurrentBodyNumber();
}


// Purpose: returns the total amount of bodies in this model.
int
CMdlView::GetBodyAmount()
{
	return m_smCurrentModel.GetBodyAmount();
}


// Purpose: returns the total amount of Sequences in this model.
int
CMdlView::GetSequenceAmount()
{
	return m_smCurrentModel.GetSeqAmount();
}


// Purpose: set light color.
void 
CMdlView::SetLightColor(int iRed, int iGreen, int iBlue)
{
	m_fLightColorRed = (float) iRed/255;
	m_fLightColorGreen = (float) iGreen/255;
	m_fLightColorBlue = (float) iBlue/255;
	m_smCurrentModel.SetLightColor(m_fLightColorRed,m_fLightColorGreen,m_fLightColorBlue);
}


// Purpose: deletes binded textures, so memory will be freed
void
CMdlView::DelTextures()
{
	// delete the textures
	m_smCurrentModel.DelTextures();
}


// Purpose: sets current selected texturenr
void
CMdlView::SetTextureNr(int iNr)
{
	m_smCurrentModel.SetTextureNr(iNr);
}


// Purpose: exports current selected texture to a file.
void
CMdlView::ExportCurrentTexture()
{
	m_smCurrentModel.ExportCurrentTexture();
}


// Purpose: imports a texture in a file into the current selected texture
void
CMdlView::ImportCurrentTexture()
{
	m_smCurrentModel.ImportCurrentTexture();
}


// Purpose: get current selected texturenr
int
CMdlView::GetCurrentTextureNr()
{
	return m_smCurrentModel.GetCurrentTextureNr();
}


// Purpose: get max texture nr of this model.
int
CMdlView::GetMaxTextureNr()
{
	return m_smCurrentModel.GetMaxTextureNr();
}


// Purpose: get the status of the chrome flag of current selected texture
bool
CMdlView::GetCurTextChromeFlag()
{
	return m_smCurrentModel.GetCurTextChromeFlag();
}


// Purpose: toggle the status of the chrome flag of current selected texture
void
CMdlView::ToggleCurTextChromeFlag()
{
	m_smCurrentModel.ToggleCurTextChromeFlag();
}


// Purpose: to check if there is a model loaded
bool
CMdlView::IsModelLoaded()
{
	return m_bModelLoaded;
}


// Purpose: to check if there is a model loaded
bool
CMdlView::IsModelInitialized()
{
	return m_bModelInitialized;
}


// Purpose: returns the current skinnumber
int
CMdlView::GetCurrentSkinNr()
{
	return m_smCurrentModel.GetSkinNr();
}


// Purpose: returns the max skinnumber
int
CMdlView::GetMaxSkinNr()
{
	return m_smCurrentModel.GetMaxSkinNr();
}


// Purpose: sets the current skinnumber
void
CMdlView::SetCurrentSkinNr(int iNr)
{
	m_smCurrentModel.SetSkin(iNr);
}

