/****************************************************************************

PigStats.c

PigStats: Half-Life log file analyzer and statistics generator.
Copyright (C) 2000  Eric Dennison

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*****************************************************************************/


/*
 * CHANGE LOG
 *
 * 1/19/2000 Initial release:  V 1.2
 *
 * 2/21/2000 Bug fix: 
 *
 * - Log containing player that is only killed by world causes crash.
 *
 * 2/25/2000 Enhancement:
 *
 * - 50 player limitation eliminated
 * - Multiple html file output capable
 * 
 */



#include <windows.h>
#include <stdio.h>
#include <time.h>
#include "regex.h"


char *pBanner = 
"\r\n\r\n**********************************************************\r\n**** %s\r\n**********************************************************\r\n";

#define BUFFER_INCREMENT  0x2000

#define INVALID_CHARS "\\//:*?\"<>|"


#define WEAPONEXTENSION "-weapons"
#define GAMEEXTENSION "-games"

struct PStat
{
	char *pPName;
	struct PStat *pOPStat; /* other players, related to this one */
	struct PStat *pWStat;  /* weapon stat list, related to this one */
	struct PStat *pTStat;  /* linked teams stats related to this one */
	int nk; /* excluding suicides */
	int nd; /* including suicides */
	int ns; /* num suicides */
	int nmw; /* num match wins (teams) */
	time_t duration;
	time_t start;
	time_t end;
	int team;
};


typedef enum 
{
	PARSEKILLS,
	PARSESUICIDES,
	PARSEDEATHS
} PARSESTATE;

/* global stat pointer for PLAYER stats */
struct PStat *gpPStat = NULL;
/* global stat pointer for WEAPON stats */
struct PStat *gpWStat = NULL;
/* global stat pointer for GAME stats */
struct PStat *gpGStat = NULL;
/* global stat pointer for TEAM stats */
struct PStat *gpTStat = NULL;

/* global path to log files */
char cInPath[255];
/* global path to output files */
char gcOutPath[255];
/* global root path of output files */
char gcRootPath[255];
/* global root name of output files */
char gcRootName[255];
/* global html extension text */
char gcExtPath[10];
/* global cross ref. path for weapon page */
char gcWeaponRef[255];
/* global cross ref. path for general page */
char gcGeneralRef[255];
/* global cross ref. path for game page */
char gcGameRef[255];

/* global output file type */
BOOL bIsHTML = FALSE;

struct PStat **gppPStatList=NULL;
int	gPStatListSize = 0;
int	gPStatListQty;

/* -s silent operation flag */
BOOL bSilent = FALSE;

/* -f filter out zero statistics */
BOOL bFilter = FALSE;

/* -m multiple html file output */
BOOL bMux = FALSE;

/* global time stamp data */
time_t StartTime,TimeStamp;

/* global flag indicating team play */
BOOL bTeamPlay = TRUE;


/* 
 * Morph a text string until it has only valid filename chars 
 * At this point we're going to have a name with only a-z, A-z and 0-9 
 */
void PIG_MakeStringValidFileName(char* pNameTemp, char* pDestName)
{
	int Source = 0;
	int Dest = 0;
	char c;
	int c1,c2;

	while (*(pNameTemp+Source))
	{
		c = *(pNameTemp+Source++);
		if (!isalnum(c))
		{
			c1 = c/16;
			if (c1 < 10) {c1 += '0';} else {c1 += 'A'-10;}
			c2 = c & 0x0f;
			if (c2 < 10) {c2 += '0';} else {c2 += 'A'-10;}
			*(pDestName+Dest++) = c1;
			*(pDestName+Dest++) = c2;
		}
		else
		{
			*(pDestName+Dest++) = c;
		}
	}
	*(pDestName+Dest) = 0;
}




/* initialize the pstat pointer list */
void PIG_SetupPStatList(struct PStat *pPStat)
{
	int Size;
	int Index;
	if (pPStat)
	{
		/* resize gppPStatList if necessary */
		Size=0;
		while ((pPStat+Size)->pPName)
		{
			Size++;
		}
		if (Size > gPStatListSize)
		{
			gPStatListSize = Size;
			gppPStatList = (struct PStat**)
				realloc(gppPStatList,gPStatListSize*sizeof(struct PStat*));
		}
		if (gppPStatList)
		{
			gPStatListQty = Size;
			for (Index=0; Index<Size; Index++)
			{
				*(gppPStatList+Index) = pPStat+Index;
			}
		}
	}
	else
	{
		/* no list */
		gPStatListQty = 0;
	}
	if (!gppPStatList)
	{
		printf("Allocation failure\n");
		gPStatListQty = 0;
	}
}

/* sort a pstat pointer list by Number of Match Wins */
int PIG_NMWComp( const void *arg1, const void *arg2 )
{
	struct PStat* p1 = *((struct PStat**) arg1);
	struct PStat* p2 = *((struct PStat**) arg2);
	return p2->nmw - p1->nmw;
}
void PIG_SortPStatListByNMW(void)
{
	qsort(gppPStatList,gPStatListQty,sizeof(struct PStat*),PIG_NMWComp);
}

/* sort a pstat pointer list by Kills */
int PIG_KComp( const void *arg1, const void *arg2 )
{
	struct PStat* p1 = *((struct PStat**) arg1);
	struct PStat* p2 = *((struct PStat**) arg2);
	return p2->nk - p1->nk;
}
void PIG_SortPStatListByK(void)
{
	qsort(gppPStatList,gPStatListQty,sizeof(struct PStat*),PIG_KComp);
}

/* sort a pstat pointer list by Kills and Deaths*/
int PIG_KandDComp( const void *arg1, const void *arg2 )
{
	int ret;
	struct PStat* p1 = *((struct PStat**) arg1);
	struct PStat* p2 = *((struct PStat**) arg2);
	/* sort first by kills, */
	ret = (p2->nk-p2->ns) - (p1->nk-p1->ns);
	if (!ret)
	{
		/* then by deaths */
		ret = p1->nd - p2->nd;
	}
	return ret;
}
void PIG_SortPStatListByKandD(void)
{
	qsort(gppPStatList,gPStatListQty,sizeof(struct PStat*),PIG_KandDComp);
}

/* sort a pstat pointer list by deaths */
int PIG_DComp( const void *arg1, const void *arg2 )
{
	struct PStat* p1 = *((struct PStat**) arg1);
	struct PStat* p2 = *((struct PStat**) arg2);
	return p1->nd - p2->nd;
}
void PIG_SortPStatListByD(void)
{
	qsort(gppPStatList,gPStatListQty,sizeof(struct PStat*),PIG_DComp);
}

/* sort a pstat pointer list by start time */
int PIG_STComp( const void *arg1, const void *arg2 )
{
	struct PStat* p1 = *((struct PStat**) arg1);
	struct PStat* p2 = *((struct PStat**) arg2);
	return p1->start - p2->start;
}
void PIG_SortPStatListByST(void)
{
	qsort(gppPStatList,gPStatListQty,sizeof(struct PStat*),PIG_STComp);
}

/* sort a pstat pointer list by suicides */
int PIG_SComp( const void *arg1, const void *arg2 )
{
	struct PStat* p1 = *((struct PStat**) arg1);
	struct PStat* p2 = *((struct PStat**) arg2);
	return p1->ns - p2->ns;
}
void PIG_SortPStatListByS(void)
{
	qsort(gppPStatList,gPStatListQty,sizeof(struct PStat*),PIG_SComp);
}

/* sort a pstat pointer list by kill rate */
int PIG_KRComp( const void *arg1, const void *arg2 )
{
	struct PStat* p1 = *((struct PStat**) arg1);
	struct PStat* p2 = *((struct PStat**) arg2);
	if (!p1->duration)
	{
		return -1;
	}
	if (!p2->duration)
	{
		return 1;
	}
	return (1.0*p2->nk/p2->duration) - (1.0*p1->nk/p1->duration) < 0.0 ? -1 : 1;
}
void PIG_SortPStatListByKR(void)
{
	qsort(gppPStatList,gPStatListQty,sizeof(struct PStat*),PIG_KRComp);
}

/* sort a pstat pointer list by ratio */
int PIG_RComp( const void *arg1, const void *arg2 )
{
	struct PStat* p1 = *((struct PStat**) arg1);
	struct PStat* p2 = *((struct PStat**) arg2);
	if (!p1->nd)
	{
		return -1;
	}
	if (!p2->nd)
	{
		return 1;
	}
	return ((1.0*p2->nk/p2->nd) - (1.0*p1->nk/p1->nd)) < 0 ? -1 : 1;
}

void PIG_SortPStatListByR(void)
{
	qsort(gppPStatList,gPStatListQty,sizeof(struct PStat*),PIG_RComp);
}



void PIG_StartHTML(HANDLE hOutFile, char* pTitle)
{
	DWORD dwWritten;
	BOOL bRes;
	static char* pHTML = "<html>\r\n<head>\r\n<meta http-equiv=\"Content-Type\" content=\"text/html; charset=iso-8859-1\">\r\n<title>%s</title>\r\n</head>\r\n<body>\r\n";
	static char cHeader[1000];
	if (bIsHTML)
	{
		sprintf(cHeader,pHTML,pTitle);
		bRes = WriteFile(	hOutFile, 
								cHeader, 
								strlen(cHeader),
								&dwWritten,
								NULL );
		if (!bRes)
		{
			printf("Write error %d\n",GetLastError());
		}
	}
}

void PIG_EndHTML(HANDLE hOutFile)
{
	DWORD dwWritten;
	BOOL bRes;
	static char* pHTML = "<p>PigStats Half-Life Statistics Generator is available from <a href=\"http://pigstats.50megs.com\">BetterDead</a></p></html>\r\n</body>\r\n";
	if (bIsHTML)
	{
		bRes = WriteFile(	hOutFile, 
								pHTML, 
								strlen(pHTML),
								&dwWritten,
								NULL );
		if (!bRes)
		{
			printf("Write error %d\n",GetLastError());
		}
	}
}

void PIG_PrintBanner(HANDLE hOutFile, char* pText)
{
	static char cBuff[700];
	static char cBuff2[700];
	DWORD dwWritten;
	BOOL bRes;
	static char* pHTML = "<p><a name=\"%s\"><font size=\"6\" face=\"Arial, sans-serif\"><b><i>%s</i></b></font></a></p>\r\n";
	static char* pHTMLLinks = "<p><a href=\"%s\"><font size=\"3\" \
face=\"Arial, sans-serif\">GENERAL STATISTICS</font></a><font \
size=\"3\" face=\"Arial, sans-serif\"> / </font><a \
href=\"%s\"><font size=\"3\" face=\"Arial, sans-serif\">WEAPON \
STATISTICS</font></a><font size=\"3\" face=\"Arial, sans-serif\"> / </font><a \
href=\"%s\"><font size=\"3\" face=\"Arial, sans-serif\">GAME \
STATISTICS</font></a></p>\r\n";

	cBuff[0]=0;

	if (bIsHTML)
	{
		sprintf(cBuff,pHTML,pText,pText);
		if (bMux)
		{
			sprintf(cBuff2,pHTMLLinks,gcGeneralRef,gcWeaponRef,gcGameRef);
		}
		else
		{
			sprintf(cBuff2,pHTMLLinks,"#GENERAL STATISTICS","#WEAPON STATISTICS","#GAME STATISTICS");
		}
		strcat(cBuff,cBuff2);
	}
	else
	{
		sprintf(cBuff,pBanner,pText);
	}
	bRes = WriteFile(	hOutFile, 
							cBuff, 
							strlen(cBuff),
							&dwWritten,
							NULL );
	if (!bRes)
	{
		printf("Write error %d\n",GetLastError());
	}
}


int giWidth;
int giLeft;
int giRight;
int giMoreRight;
void PIG_TableStart(HANDLE hOutFile, int Width, int LeftSize, int RightSize, int Columns, char* pHeaderText)
{
	static char cBuff[512];
	DWORD dwWritten;
	BOOL bRes;
	static char* pHTML = "<table border=\"2\" cellpadding=\"3\" cellspacing=\"0\" width=\"%d\" bgcolor=\"#FFFFFF\" bordercolor=\"#000000\" bordercolordark=\"#000000\" bordercolorlight=\"#000000\">\r\n<tr>\r\n<td colspan=\"%d\" bgcolor=\"#0000A0\"><font color=\"#FFFFFF\" face=\"Arial\"><strong>%s</strong></font>\r\n</td>\r\n</tr>\r\n";

	cBuff[0]=0;

	giWidth = Width;
	giLeft = LeftSize;
	giRight = RightSize;
	giMoreRight = Width-(LeftSize+RightSize);

	if (bIsHTML)
	{
		sprintf(cBuff,pHTML,Width,Columns,pHeaderText);
	}
	else
	{
		sprintf(cBuff, "\r\n%s\r\n", pHeaderText);
	}
	bRes = WriteFile(	hOutFile, 
							cBuff, 
							strlen(cBuff),
							&dwWritten,
							NULL );
	if (!bRes)
	{
		printf("Write error %d\n",GetLastError());
	}
}

void PIG_TableEnd(HANDLE hOutFile)
{
	DWORD dwWritten;
	BOOL bRes;
	static char* pHTML = "</table>\r\n<p><br></p>\r\n";

	if (bIsHTML)
	{
		bRes = WriteFile(	hOutFile, 
								pHTML, 
								strlen(pHTML),
								&dwWritten,
								NULL );
		if (!bRes)
		{
			printf("Write error %d\n",GetLastError());
		}
	}
}

void PIG_TableLine(HANDLE hOutFile, BOOL bXRef, char* pLeft, char* pRight, char* pMoreRight)
{
	static char cBuffl[512];
	static char cBuffr[512];
	static char cBuffTempXref[256];
	static char cBuffTempName[256];
	static char* pHTML = "<tr>\r\n<td width=\"%d\" bgcolor=\"#818181\"><font color=\"#FFFF00\" face=\"Arial\">%s</font>\r\n</td>\r\n<td width=\"%d\">\r\n<font face=\"Arial\">%s</font>\r\n</td>\r\n";
	static char* pHTMLXref = "<tr>\r\n<td width=\"%d\" bgcolor=\"#818181\"><a href=\"%s\"><font color=\"#FFFF00\" face=\"Arial\">%s</font></a>\r\n</td>\r\n<td width=\"%d\">\r\n<font face=\"Arial\">%s</font>\r\n</td>\r\n";
	static char* pHTMLEnd = "</tr>\r\n";
	static char* pHTMLMore = "<td width=\"%d\">\r\n<font face=\"Arial\">%s</font>\r\n</td>\r\n";

	DWORD dwWritten;
	BOOL bRes;
	cBuffl[0]=0;
	cBuffr[0]=0;

	if (bIsHTML)
	{
		if (bMux)
		{
			strcpy(cBuffTempXref,"./");
			PIG_MakeStringValidFileName(pLeft,cBuffTempName);
			strcat(cBuffTempXref,gcRootName);
			strcat(cBuffTempXref,"-");
			strcat(cBuffTempXref,cBuffTempName);
			strcat(cBuffTempXref,gcExtPath);
		}
		else
		{
			strcpy(cBuffTempXref,"#");
			strcat(cBuffTempXref,pLeft);
		}
		if (!pMoreRight)
		{
			if (bXRef)
			{
				sprintf(cBuffl,pHTMLXref,giLeft,cBuffTempXref,pLeft,giRight,pRight);
			}
			else
			{
				sprintf(cBuffl,pHTML,giLeft,pLeft,giRight,pRight);
			}
			sprintf(cBuffr,pHTMLEnd);
		}
		else
		{
			if (bXRef)
			{
				sprintf(cBuffl,pHTMLXref,giLeft,cBuffTempXref,pLeft,giRight,pRight);
			}
			else
			{
				sprintf(cBuffl,pHTML,giLeft,pLeft,giRight,pRight);
			}
			sprintf(cBuffr,pHTMLMore,giMoreRight,pMoreRight);
			strcat(cBuffr,pHTMLEnd);
		}
	}
	else
	{
		if (!pMoreRight)
		{
			sprintf(cBuffl,"%-20s %s\r\n",pLeft,pRight);
		}
		else
		{
			sprintf(cBuffl,"%-20s %s%s\r\n",pLeft,pRight,pMoreRight);
		}
	}
	bRes = WriteFile(	hOutFile, 
							cBuffl, 
							strlen(cBuffl),
							&dwWritten,
							NULL );
	if (!bRes)
	{
		printf("Write error %d\n",GetLastError());
	}
	bRes = WriteFile(	hOutFile, 
							cBuffr, 
							strlen(cBuffr),
							&dwWritten,
							NULL );
	if (!bRes)
	{
		printf("Write error %d\n",GetLastError());
	}
}

void	PIG_ShowOverallStats(HANDLE hOutFile)
{
	double fTemp;
	int Index;
	struct PStat* pPS;
	char cNum[255];
	/* sort by kills */
	PIG_SetupPStatList(gpPStat);
	PIG_SortPStatListByK();
	PIG_PrintBanner(hOutFile, "GENERAL STATISTICS");
	PIG_TableStart(hOutFile,500,200,300,2,"Sorted by Kills:");
	for(Index=0;Index<gPStatListQty;Index++)
	{
		pPS = *(gppPStatList+Index);
		if (!bFilter || pPS->nk)
		{
			sprintf(cNum,"%d",pPS->nk);
			PIG_TableLine(hOutFile,TRUE,pPS->pPName,cNum,NULL);
		}
	}
	PIG_TableEnd(hOutFile);

	PIG_SortPStatListByD();
	PIG_TableStart(hOutFile,500,200,300,2,"Sorted by Deaths:");
	for(Index=0;Index<gPStatListQty;Index++)
	{
		pPS = *(gppPStatList+Index);
		if (!bFilter || pPS->nd)
		{
			sprintf(cNum,"%d",pPS->nd);
			PIG_TableLine(hOutFile,TRUE,pPS->pPName,cNum,NULL);
		}
	}
	PIG_TableEnd(hOutFile);

	PIG_SortPStatListByS();
	PIG_TableStart(hOutFile,500,200,300,2,"Sorted by Suicides:");
	for(Index=0;Index<gPStatListQty;Index++)
	{
		pPS = *(gppPStatList+Index);
		if (!bFilter || pPS->ns)
		{
			sprintf(cNum,"%d",pPS->ns);
			PIG_TableLine(hOutFile,TRUE,pPS->pPName,cNum,NULL);
		}
	}
	PIG_TableEnd(hOutFile);

	PIG_SortPStatListByR();
	PIG_TableStart(hOutFile,500,200,300,2,"Sorted by Ratio:");
	for(Index=0;Index<gPStatListQty;Index++)
	{
		pPS = *(gppPStatList+Index);
		if (pPS->nd)
		{
			fTemp = 1.0*pPS->nk/pPS->nd;
			if (!bFilter || (fTemp != 0.0))
			{
				sprintf(cNum,"%f",fTemp);
				PIG_TableLine(hOutFile,TRUE,pPS->pPName,cNum,NULL);
			}
		}
		else
		{
			PIG_TableLine(hOutFile,TRUE,pPS->pPName,"N/A",NULL);
		}
	}
	PIG_TableEnd(hOutFile);

	PIG_SortPStatListByKR();
	PIG_TableStart(hOutFile,500,200,300,2,"Sorted by Kill Rate (while playing):");
	for(Index=0;Index<gPStatListQty;Index++)
	{
		pPS = *(gppPStatList+Index);
		if (pPS->duration)
		{
			fTemp = 60.0*pPS->nk/pPS->duration;
			if (!bFilter || (fTemp != 0.0))
			{
				sprintf(cNum,"%f Kills/Min",fTemp);
				PIG_TableLine(hOutFile,TRUE,pPS->pPName,cNum,NULL);
			}
		}
		else
		{
			PIG_TableLine(hOutFile,TRUE,pPS->pPName,"N/A",NULL);
		}
	}
	PIG_TableEnd(hOutFile);
}


void PIG_ShowPlayerStats(HANDLE hOutFileArg)
{
	double fTemp;
	int Index;
	struct PStat* pPS;
	struct PStat* pPSX;
	char* pName;
	static char cTemp[256];
	static char cNewName[256];
	HANDLE hOutFile = hOutFileArg;

	pPSX = gpPStat;
	while (pPSX->pPName)
	{
		/* multiple output files? */
		if (bMux)
		{
			strcpy(gcOutPath, gcRootPath);
			strcat(gcOutPath, "-");
			pName = pPSX->pPName;
			PIG_MakeStringValidFileName(pName,cNewName);
			strcat(gcOutPath, cNewName);
			strcat(gcOutPath, gcExtPath);
			/* open the output file */
			hOutFile = CreateFile(	gcOutPath,
										GENERIC_WRITE,
										0,
										NULL,
										CREATE_ALWAYS,
										FILE_ATTRIBUTE_ARCHIVE,
										NULL);
			if (hOutFile == INVALID_HANDLE_VALUE)
			{
				printf("sorry... Win32 file name error\n\n");
				return ;
			}
			PIG_StartHTML(hOutFile,"PigStats Player Statistics");
		}
		pName = pPSX->pPName;
		PIG_PrintBanner(hOutFile,pName);
		sprintf(cTemp,"General Statistics for %s",pName);
		PIG_TableStart(hOutFile,500,200,300,2,cTemp);
		sprintf(cTemp,"%d",pPSX->nk);
		PIG_TableLine(hOutFile,FALSE,"Total kills",cTemp,NULL);
		sprintf(cTemp,"%d",pPSX->nd);
		PIG_TableLine(hOutFile,FALSE,"Total deaths",cTemp,NULL);
		sprintf(cTemp,"%d",pPSX->ns);
		PIG_TableLine(hOutFile,FALSE,"Total suicides",cTemp,NULL);
		if (pPSX->nd)
		{
			sprintf(cTemp,"%f",1.0*pPSX->nk/pPSX->nd);
			PIG_TableLine(hOutFile,FALSE,"Overall kill/death",cTemp,NULL);
		}
		else
		{
			PIG_TableLine(hOutFile,FALSE,"Overall kill/death","N/A",NULL);
		}
		if (pPSX->duration)
		{
			sprintf(cTemp,"%f",60.0*pPSX->nk/pPSX->duration);
			PIG_TableLine(hOutFile,FALSE,"Kill rate (K/Min)",cTemp,NULL);
		}
		else
		{
			PIG_TableLine(hOutFile,FALSE,"Kill rate (K/Min)","N/A",NULL);
		}
		sprintf(cTemp,"%f Min",pPSX->duration/60.0);
		PIG_TableLine(hOutFile,FALSE,"Total play time",cTemp,NULL);
		PIG_TableEnd(hOutFile);

		pPS = pPSX->pOPStat;
		sprintf(cTemp,"%s most often killed",pName);
		PIG_TableStart(hOutFile,500,200,300,2,cTemp);
		/* sort by kills */
		PIG_SetupPStatList(pPS);
		PIG_SortPStatListByK();
		for(Index=0;Index<gPStatListQty;Index++)
		{
			pPS = *(gppPStatList+Index);
			sprintf(cTemp,"%d",pPS->nk);
			PIG_TableLine(hOutFile,TRUE,pPS->pPName,cTemp,NULL);
		}
		PIG_TableEnd(hOutFile);


		PIG_SortPStatListByD();
		sprintf(cTemp,"%s most often killed by",pName);
		PIG_TableStart(hOutFile,500,200,300,2,cTemp);
		for(Index=gPStatListQty;Index;Index--)
		{
			pPS = *(gppPStatList+Index-1);
			sprintf(cTemp,"%d",pPS->nd);
			PIG_TableLine(hOutFile,TRUE,pPS->pPName,cTemp,NULL);
		}
		PIG_TableEnd(hOutFile);

		PIG_SortPStatListByR();
		sprintf(cTemp,"%s kill/death ratio against",pName);
		PIG_TableStart(hOutFile,500,200,300,2,cTemp);
		for(Index=0;Index<gPStatListQty;Index++)
		{
			pPS = *(gppPStatList+Index);
			if (pPS->nd)
			{
				fTemp = 1.0*pPS->nk/pPS->nd;
				sprintf(cTemp,"%f",fTemp);
				PIG_TableLine(hOutFile,TRUE,pPS->pPName,cTemp,NULL);
			}
			else
			{
				PIG_TableLine(hOutFile,TRUE,pPS->pPName,"N/A",NULL);
			}
		}
		PIG_TableEnd(hOutFile);

		/* sort by weapon kills */
		pPS = pPSX->pWStat;
		PIG_SetupPStatList(pPS);
		PIG_SortPStatListByK();
		sprintf(cTemp,"%s most often kills with",pName);
		PIG_TableStart(hOutFile,500,200,300,2,cTemp);
		for(Index=0;Index<gPStatListQty;Index++)
		{
			pPS = *(gppPStatList+Index);
			if (pPS->nk)
			{
				sprintf(cTemp,"%d",pPS->nk);
				PIG_TableLine(hOutFile,FALSE,pPS->pPName,cTemp,NULL);
			}
		}
		PIG_TableEnd(hOutFile);

		PIG_SortPStatListByD();
		sprintf(cTemp,"%s most often killed by",pName);
		PIG_TableStart(hOutFile,500,200,300,2,cTemp);
		for(Index=gPStatListQty;Index;Index--)
		{
			pPS = *(gppPStatList+Index-1);
			if (pPS->nd)
			{
				sprintf(cTemp,"%d",pPS->nd);
				PIG_TableLine(hOutFile,FALSE,pPS->pPName,cTemp,NULL);
			}
		}
		PIG_TableEnd(hOutFile);
		if (bMux)
		{
			PIG_EndHTML(hOutFile);
			CloseHandle(hOutFile);
		}
		/* next player */
		pPSX++;
	}
}


void	PIG_ShowWeaponStats(HANDLE hOutFileArg)
{
	int	PStatListQty;
	struct PStat** ppPStatList;
	int Index;
	struct PStat* pPS;
	struct PStat* pPSFav;
	static char cTemp[256];
	static char cTemp1[256];
	HANDLE hOutFile = hOutFileArg;

	/* sort by kill usage */
	PIG_SetupPStatList(gpWStat);
	PIG_SortPStatListByK();
	/* make a local copy of the list */
	PStatListQty = gPStatListQty;
	ppPStatList = (struct PStat**)calloc(gPStatListQty,sizeof(struct PStat*));
	if (!ppPStatList)
	{
		printf("Allocation failure\n");
		return;
	}
	memcpy(ppPStatList,gppPStatList,gPStatListQty*sizeof(struct PStat*));
	/* multiple output files? */
	if (bMux)
	{
		strcpy(gcOutPath, gcRootPath);
		strcat(gcOutPath, WEAPONEXTENSION);
		strcat(gcOutPath, gcExtPath);
		/* open the output file */
		hOutFile = CreateFile(	gcOutPath,
									GENERIC_WRITE,
									0,
									NULL,
									CREATE_ALWAYS,
									FILE_ATTRIBUTE_ARCHIVE,
									NULL);
		if (hOutFile == INVALID_HANDLE_VALUE)
		{
			printf("sorry... Win32 file name error\n\n");
			return ;
		}
		PIG_StartHTML(hOutFile,"PigStats Weapon Statistics");
	}
	PIG_PrintBanner(hOutFile,"WEAPON STATISTICS");
	sprintf(cTemp,"Sorted by Kills");
	PIG_TableStart(hOutFile,500,150,100,3,cTemp);
	for(Index=0;Index<PStatListQty;Index++)
	{
		pPS = *(ppPStatList+Index);
		if (pPS->nk)
		{
			/* sort to find most common killer with this weapon */
			PIG_SetupPStatList(pPS->pOPStat);
			PIG_SortPStatListByK();
			pPSFav = *(gppPStatList);
			sprintf(cTemp,"%-6d ",pPS->nk);
			sprintf(cTemp1,"(%d by %s)",pPSFav->nk,pPSFav->pPName);
			PIG_TableLine(hOutFile,FALSE,pPS->pPName,cTemp,cTemp1);
		}
	}
	PIG_TableEnd(hOutFile);
	if (bMux)
	{
		PIG_EndHTML;
		CloseHandle(hOutFile);
	}
	free(ppPStatList);
}



/* clean up allocated memory */
void	PIG_CleanPStat(struct PStat *pPStat)
{
	int Index;
	struct PStat* pPS;

	for (Index=0;pPStat && (pPStat+Index)->pPName;Index++)
	{
		pPS = pPStat+Index;
		free(pPS->pPName);
		/* free any child pStats */
		if (pPS->pOPStat)
		{
			PIG_CleanPStat(pPS->pOPStat);
		}
		if (pPS->pWStat)
		{
			PIG_CleanPStat(pPS->pWStat);
		}
		if (pPS->pTStat)
		{
			PIG_CleanPStat(pPS->pTStat);
		}
	}
}


/* get ptr to pstat structure, and index.. create if not there yet */
struct PStat* PIG_GetPStat(struct PStat** ppPStat, char* pName, int* piIndex)
{
	int Index;
	BOOL bFound = FALSE;
	struct PStat* pPStat = NULL;

	/* allocate initial PStat, if necessary */
	if (!*ppPStat)
	{
		*ppPStat = (struct PStat*)calloc(sizeof (struct PStat),1);
	}

	if (*ppPStat)
	{
		for (Index=0; bFound == FALSE; Index++)
		{
			pPStat = *ppPStat+Index;
			if (!pPStat->pPName)
			{
				/* not found, create new */
				pPStat->pPName = (char*) malloc(strlen(pName)+1);
				strcpy(pPStat->pPName, pName);
				/* realloc the array to add a new, null entry */
				*ppPStat = (struct PStat*)realloc(*ppPStat, (sizeof(struct PStat))*(Index+2));
				/* reinitialize pPStat */
				pPStat = *ppPStat+Index;
				memset(pPStat+1,0,sizeof(struct PStat));
				bFound = TRUE;
			}
			else if (!strcmp(pPStat->pPName, pName))
			{
				/* found */
				bFound = TRUE;
			}
		}
		if (piIndex) *piIndex = Index;
	}
	return pPStat;
}


void	PIG_ShowGameStats(HANDLE hOutFileArg)
{
	int	PStatListQty;
	struct PStat** ppPStatList;
	int Index,PlayerIndex;
	struct PStat* pPS;
	struct PStat* pPSPlayer;
	static char cTemp[256];
	static char cTemp1[256];
	HANDLE hOutFile = hOutFileArg;

	/* sort by map start time usage */
	PIG_SetupPStatList(gpGStat);
	PIG_SortPStatListByST();
	/* make a local copy of the list */
	PStatListQty = gPStatListQty;
	ppPStatList = (struct PStat**)calloc(gPStatListQty,sizeof(struct PStat*));
	if (!ppPStatList)
	{
		printf("Allocation failure\n");
		return;
	}
	/* multiple output files? */
	if (bMux)
	{
		strcpy(gcOutPath, gcRootPath);
		strcat(gcOutPath, GAMEEXTENSION);
		strcat(gcOutPath, gcExtPath);
		/* open the output file */
		hOutFile = CreateFile(	gcOutPath,
									GENERIC_WRITE,
									0,
									NULL,
									CREATE_ALWAYS,
									FILE_ATTRIBUTE_ARCHIVE,
									NULL);
		if (hOutFile == INVALID_HANDLE_VALUE)
		{
			printf("sorry... Win32 file name error\n\n");
			return ;
		}
		PIG_StartHTML(hOutFile,"PigStats Game Statistics");
	}
	memcpy(ppPStatList,gppPStatList,gPStatListQty*sizeof(struct PStat*));
	PIG_PrintBanner(hOutFile,"GAME STATISTICS");
	for(Index=0;Index<PStatListQty;Index++)
	{
		pPS = *(ppPStatList+Index);
		PIG_TableStart(hOutFile,500,200,150,3,pPS->pPName);
		if (pPS->pOPStat)
		{
			/* sort to find best killers */
			PIG_SetupPStatList(pPS->pOPStat);
			PIG_SortPStatListByKandD();
			/* bump the match wins count for the best of the best */
			pPSPlayer = PIG_GetPStat(&gpPStat, (*gppPStatList)->pPName, &PlayerIndex);
			pPSPlayer->nmw++;
			for (PlayerIndex=0;PlayerIndex<gPStatListQty;PlayerIndex++)
			{
				pPSPlayer = *(gppPStatList+PlayerIndex);
				sprintf(cTemp,"%-6d ",pPSPlayer->nk-pPSPlayer->ns);
				sprintf(cTemp1,"%-6d",pPSPlayer->nd);
				PIG_TableLine(hOutFile,TRUE,pPSPlayer->pPName,cTemp,cTemp1);
			}
		}
		PIG_TableEnd(hOutFile);
	}
	/* sort players by match wins */
	PIG_TableStart(hOutFile,500,200,300,2,"Match Totals");
	PIG_SetupPStatList(gpPStat);
	PIG_SortPStatListByNMW();
	for (PlayerIndex=0;PlayerIndex<gPStatListQty;PlayerIndex++)
	{
		pPSPlayer = *(gppPStatList+PlayerIndex);
		sprintf(cTemp,"%-6d",pPSPlayer->nmw);
		PIG_TableLine(hOutFile,TRUE,pPSPlayer->pPName,cTemp,NULL);
	}
	PIG_TableEnd(hOutFile);
	if (bMux)
	{
		PIG_EndHTML(hOutFile);
		CloseHandle(hOutFile);
	}
	free(ppPStatList);
}



void	PIG_ShowTeamGameStats(HANDLE hOutFileArg)
{
	int	PStatListQty;
	int PTeamListQty;
	struct PStat** ppPStatList;
	struct PStat** ppPTeamList;
	int Index,PlayerIndex,TeamIndex,TempIndex;
	struct PStat* pPS;
	struct PStat* pPSTeam;
	struct PStat* pPSPlayer;
	struct PStat* pPSMasterPlayer;
	struct PStat* pPSTempPlayer;
	static char cTemp[256];
	static char cTemp1[256];
	static char cTemp2[256];
	HANDLE hOutFile = hOutFileArg;
	
	/* sort by map start time usage */
	PIG_SetupPStatList(gpGStat);
	PIG_SortPStatListByST();
	/* make a local copy of the list */
	PStatListQty = gPStatListQty;
	ppPStatList = (struct PStat**)calloc(gPStatListQty,sizeof(struct PStat*));
	if (!ppPStatList)
	{
		printf("Allocation failure\n");
		return;
	}
	/* multiple output files? */
	if (bMux)
	{
		strcpy(gcOutPath, gcRootPath);
		strcat(gcOutPath, GAMEEXTENSION);
		strcat(gcOutPath, gcExtPath);
		/* open the output file */
		hOutFile = CreateFile(	gcOutPath,
									GENERIC_WRITE,
									0,
									NULL,
									CREATE_ALWAYS,
									FILE_ATTRIBUTE_ARCHIVE,
									NULL);
		if (hOutFile == INVALID_HANDLE_VALUE)
		{
			printf("sorry... Win32 file name error\n\n");
			return ;
		}
		PIG_StartHTML(hOutFile,"PigStats Game Statistics");
	}
	memcpy(ppPStatList,gppPStatList,gPStatListQty*sizeof(struct PStat*));
	PIG_PrintBanner(hOutFile,"GAME STATISTICS");
	for(Index=0;Index<PStatListQty;Index++)
	{
		pPS = *(ppPStatList+Index);
		PIG_TableStart(hOutFile,500,200,150,3,pPS->pPName);
		if (pPS->pOPStat)
		{
			/* assign team ids to the players for the game */
			pPSPlayer = pPS->pOPStat;
			while (pPSPlayer && pPSPlayer->pPName)
			{
				pPSMasterPlayer = PIG_GetPStat(&gpPStat, pPSPlayer->pPName, &PlayerIndex);
				pPSPlayer->team = pPSMasterPlayer->team;
				/* figure out which team name it is */
				pPSTeam = gpTStat;
				while (pPSTeam && pPSTeam->pPName)
				{
					if (pPSTeam->team == pPSPlayer->team)
					{
						/* identified a team involved in this game, add it to the game */
						pPSTeam = PIG_GetPStat(&pPS->pTStat, pPSTeam->pPName, &TeamIndex);
						/* update team stats for this game */
						pPSTeam->nk += pPSPlayer->nk;
						pPSTeam->nd += pPSPlayer->nd;
						pPSTeam->ns += pPSPlayer->ns;
						/* and add the player to the team! */
						pPSTempPlayer = PIG_GetPStat(&pPSTeam->pOPStat, pPSPlayer->pPName, &TempIndex);
						pPSTempPlayer->nk = pPSPlayer->nk;
						pPSTempPlayer->nd = pPSPlayer->nd;
						pPSTempPlayer->ns = pPSPlayer->ns;
						break;
					}
					pPSTeam++;
				}
				pPSPlayer++;
			}

			/* sort to find best teams for this game */
			PIG_SetupPStatList(pPS->pTStat);
			PIG_SortPStatListByKandD();
			/* make a local copy of the list */
			PTeamListQty = gPStatListQty;
			ppPTeamList = (struct PStat**)calloc(gPStatListQty,sizeof(struct PStat*));
			if (!ppPTeamList)
			{
				free (ppPStatList);
				printf("Allocation failure\n");
				return;
			}
			memcpy(ppPTeamList,gppPStatList,gPStatListQty*sizeof(struct PStat*));
			/* bump the match wins count for the best of the best */
			pPSTeam = PIG_GetPStat(&gpTStat, (*ppPTeamList)->pPName, &TeamIndex);
			pPSTeam->nmw++;
			/* for each team in the game */
			for (TeamIndex=0;TeamIndex<PTeamListQty;TeamIndex++)
			{
				/* sort that team's players */
				pPSTeam = *(ppPTeamList+TeamIndex);
				sprintf(cTemp,"%-6d ",pPSTeam->nk-pPSTeam->ns);
				sprintf(cTemp2,"%-6d",pPSTeam->nd);
				PIG_TableLine(hOutFile,FALSE,pPSTeam->pPName,cTemp,cTemp2);
				PIG_SetupPStatList(pPSTeam->pOPStat);
				PIG_SortPStatListByKandD();
				for (PlayerIndex=0;PlayerIndex<gPStatListQty;PlayerIndex++)
				{
					pPSPlayer = *(gppPStatList+PlayerIndex);
					sprintf(cTemp,"%-6d ",pPSPlayer->nk-pPSPlayer->ns);
					sprintf(cTemp2,"%-6d",pPSPlayer->nd);
					sprintf(cTemp1,"..%-18s",pPSPlayer->pPName);
					PIG_TableLine(hOutFile,FALSE,cTemp1,cTemp,cTemp2);
				}
			}
		}
		PIG_TableEnd(hOutFile);
	}
	/* sort by teams by match wins */
	PIG_TableStart(hOutFile,500,200,300,2,"Match Totals");
	PIG_SetupPStatList(gpTStat);
	PIG_SortPStatListByNMW();
	for (TeamIndex=0;TeamIndex<gPStatListQty;TeamIndex++)
	{
		pPSTeam = *(gppPStatList+TeamIndex);
		sprintf(cTemp,"%-6d",pPSTeam->nmw);
		PIG_TableLine(hOutFile,FALSE,pPSTeam->pPName,cTemp,NULL);
	}
	PIG_TableEnd(hOutFile);
	if (bMux)
	{
		PIG_EndHTML(hOutFile);
		CloseHandle(hOutFile);
	}
	free(ppPStatList);
	free(ppPTeamList);
}



void PIG_UpdatePSTimeStamp(struct PStat* pStat)
{
	if (pStat->start)
	{
		/* update duration and end stamp */
		if (TimeStamp > pStat->end)
		{
			pStat->duration += TimeStamp-pStat->end;
			pStat->end = TimeStamp;
		}
	}
	else
	{
		/* initialize start and end */
		pStat->start = StartTime;
		pStat->end = StartTime;
	}
}


void PIG_ClearPSTimeStamps(void)
{
	struct PStat* pPStat = gpPStat;

	while (gpPStat && pPStat->pPName)
	{
		/* update final time for participating players */
		if (pPStat->start && (TimeStamp > pPStat->end))
		{
			pPStat->duration += TimeStamp-pPStat->end;
		}
		pPStat->start = 0;
		pPStat->end = 0;
		pPStat++;
	}
}



int PIG_UpdatePStat(time_t tstamp, char* pName, char* pVict, char* pWp, struct PStat *pGameStat, PARSESTATE eParseState)
{
	struct PStat* pKrStat;
	struct PStat* pKeStat;
	struct PStat* pKrKeStat;
	struct PStat* pKeKrStat;
	int IndexR, IndexE, Index, IndexW;
	struct PStat* pWpStat;
	struct PStat* pWpKrStat;
	struct PStat* pWpKeStat;
	struct PStat* pWFavStat;
	struct PStat* pGmKrStat;	/* game killer stat */
	struct PStat* pGmKeStat;	/* game killee stat */


	pKrStat = PIG_GetPStat(&gpPStat, pName, &IndexR);
	pGmKrStat = PIG_GetPStat(&pGameStat->pOPStat, pName, &Index);
	/* update killer time stamp data */
	PIG_UpdatePSTimeStamp(pKrStat);
	pWpStat = PIG_GetPStat(&gpWStat, pWp, &IndexW);
	switch (eParseState)
	{
	case PARSEKILLS:
		/* killer's overall kill count */
		pKrStat->nk++;
		/* and game count */
		pGmKrStat->nk++;
		/* increment killer's weapon kill count */
		pWpKrStat = PIG_GetPStat(&(pKrStat->pWStat),pWp, &Index);
		pWpKrStat->nk++;
		/* times killer killed this victim */
		pKrKeStat = PIG_GetPStat(&(pKrStat->pOPStat), pVict, &Index);
		pKrKeStat->nk++; /* times killed this victim */
		/* update killee's stats */
		pKeStat = PIG_GetPStat(&gpPStat, pVict, &IndexE);
		pGmKeStat = PIG_GetPStat(&pGameStat->pOPStat, pVict, &Index);
		pKeStat->nd++;
		pGmKeStat->nd++;
		/* update killee's time stamp data */
		PIG_UpdatePSTimeStamp(pKeStat);
		/* increment killee's weapon death count */
		pWpKeStat = PIG_GetPStat(&(pKeStat->pWStat), pWp, &Index);
		pWpKeStat->nd++;
		/* and killee's stats for killer */
		pKeKrStat = PIG_GetPStat(&(pKeStat->pOPStat), pName, &Index);
		pKeKrStat->nd++; /* times killed by this killer */
		/* update weapon info: times weapon was used to kill another player only */
		pWpStat->nk++;
		/* update weapon favorite usage info */
		pWFavStat = PIG_GetPStat(&(pWpStat->pOPStat), pName, &Index);
		pWFavStat->nk++;
		break;
	case PARSESUICIDES:
		pKrStat->ns++;
		pGmKrStat->ns++;
	case PARSEDEATHS:
		pKrStat->nd++;
		pGmKrStat->nd++;
		/* times killee has died by this weapon */
		pWpKeStat = PIG_GetPStat(&(pWpStat->pOPStat), pName, &Index);
		pWpKeStat->nd++;
		break;
	default:
		break;
	}
	return 0;
}


void PIG_FindTeams()
{
	struct PStat *pPStat;
	struct PStat *pFPStat; /* following players */
	struct PStat *pOPStat; /* other player */
	int HighestTeam = 0;
	int TeamX;
	char *pName = "";
	int Kills,TeamKills;
	int Deaths,TeamDeaths;
	int TeamSuicides;
	char TeamName[100];
	int Index;
	struct PStat *pTStat; /* team stats */

	pPStat = gpPStat;
	while (pPStat && pPStat->pPName)
	{
		
		pFPStat = pPStat + 1;
		/* for every following player */
		while (pFPStat && pFPStat->pPName)
		{
			pOPStat = pPStat->pOPStat;
			/* for every player this guy ever touched */
			while (pOPStat && pOPStat->pPName)
			{
				/* if the name matches a following player */
				/* and team ids are the same...				*/
				if (	!strcmp(pOPStat->pPName,pFPStat->pPName) &&
						(pFPStat->team == pPStat->team))
				{
					/* bump the following player's team index */
					pFPStat->team++;
					if (pFPStat->team > HighestTeam)
					{
						HighestTeam = pFPStat->team;
					}
					break;
				}
				/* next guy */
				pOPStat++;
			}
			/* next player */
			pFPStat++;
		}
		/* next player */
		pPStat++;
	}
	/* now each player has an appropriate team index */
	/* create some teams                             */
	if (HighestTeam > 0)
	{
		for (TeamX=0;TeamX<=HighestTeam;TeamX++)
		{
			pPStat = gpPStat;
			Kills = 0;
			Deaths = 0;
			TeamKills = 0;
			TeamDeaths = 0;
			TeamSuicides = 0;
			while (pPStat && pPStat->pPName)
			{
				if (pPStat->team == TeamX)
				{
					if ((pPStat->nk>Kills) || 
							((pPStat->nk==Kills) && (pPStat->nd<Deaths)) || 
							((Deaths==0) && (Kills==0)))
					{
						/* team leader so far.. */
						pName = pPStat->pPName;
						Kills = pPStat->nk;
						Deaths = pPStat->nd;
					}
					/* update team stats */
					TeamKills += pPStat->nk;
					TeamDeaths += pPStat->nd;
					TeamSuicides += pPStat->ns;
				}
				pPStat++;
			}
			TeamName[0]=0;
			strcpy(TeamName,"Team ");
			strcat(TeamName,pName);
			/* create a team! */
			pTStat = PIG_GetPStat(&gpTStat, TeamName, &Index);
			pTStat->team = TeamX;
			pTStat->nk = TeamKills;
			pTStat->nd = TeamDeaths;
			pTStat->ns = TeamSuicides;
		}
	}
	else
	{
		/* team play not detected after all! */
		bTeamPlay = FALSE;
	}

}

int PIG_ParseFile(WIN32_FIND_DATA *pFileData)
{
	char error[256];
	HANDLE hFile;
	char *pBuffer;
	DWORD dwBytesRead;
	DWORD dwTotalRead = 0;
	DWORD dwBufferSize;
	int	Index;
	BOOL bRes;
	BOOL bDone = FALSE;
	regex_t comp;
	/* reg. expression for player A kills player B */
	char *regex = "L ([0-9]*)/([0-9]*)/([0-9]*) - ([0-9]*):([0-9]*):([0-9]*): \"([^\"<]*)<[0-9]*>\" killed \"([^\"<]*)<[0-9]*>\" with ([^\r\n ]*)";
	/* reg. exp. for player A kills self */
	char *regexs = "L ([0-9]*)/([0-9]*)/([0-9]*) - ([0-9]*):([0-9]*):([0-9]*): \"([^\"<]*)<[0-9]*>\" killed (self) with ([^\r\n ]*)";
	/* reg. exp. for player A killed by world */
	char *regexd = "L ([0-9]*)/([0-9]*)/([0-9]*) - ([0-9]*):([0-9]*):([0-9]*): \"([^\"<]*)<[0-9]*>\" killed (by world) with ([^\r\n ]*)";
	/* reg. expression for time stamp */
	char *regexts = "([0-9]*)/([0-9]*)/([0-9]*) - ([0-9]*):([0-9]*):([0-9]*)";
	/* reg. expression for game/map name */
	char *regexg = "Spawning server \"([^\"]*)";
	/* reg. expression for team play status ON */
	char *regextp = "\"mp_teamplay\" = \"1\"";
	regmatch_t matches[10];
	int nomatch = 0;
	int err;
	BOOL bKillsDone;
	PARSESTATE eParseState = PARSEKILLS;
	char cFilePath[255];
	char kr[100];
	char ke[100];
	char wp[100];
	char gm[100]; /* game (map and time stamp) */
	char timedate[100];
	char mapname[100];
	int ltd,lmn;
	int lkr,lke,lwp;
	struct tm stFTime; /* formatted time */
	struct PStat *pGameStat;
	BOOL bMapDetected = FALSE;


	strcpy(cFilePath,cInPath);
	strcat(cFilePath,pFileData->cFileName);
	hFile = CreateFile(	cFilePath,
								GENERIC_READ,
								FILE_SHARE_READ,
								NULL,
								OPEN_EXISTING,
								FILE_FLAG_SEQUENTIAL_SCAN,
								NULL);
	if (hFile != INVALID_HANDLE_VALUE)
	{
		dwBufferSize = BUFFER_INCREMENT;
		pBuffer = (char*) calloc(1,dwBufferSize);
		Index = 0;
		/* file opened */
		while (pBuffer && !bDone)
		{
			bRes = ReadFile( hFile, pBuffer+Index, BUFFER_INCREMENT, &dwBytesRead, NULL);
			dwTotalRead += dwBytesRead;
			if (bRes && (dwBytesRead == BUFFER_INCREMENT))
			{
				/* jack up the buffer size */
				dwBufferSize += BUFFER_INCREMENT;
				Index += BUFFER_INCREMENT;
				pBuffer = (char*)realloc(pBuffer, dwBufferSize);
			}
			else if (bRes)
			{
				/* got our bufferload */
				bDone = TRUE;
				*(pBuffer+dwTotalRead) = 0;
			}
			else
			{
				/* error case */
				printf("Ack! I never planned for this error!\n");
			}
		}

		/* compile the regex for the first time stamp */
		if (err=regcomp(&comp,regexts,REG_EXTENDED))
		{
			regerror(err,&comp,error,255);
			printf("Error in regcomp: %s\n",error);
			return 1;
		}
		nomatch=regexec(&comp,pBuffer,10,matches,0);
		if (nomatch)
		{
			goto bail;
		}
		stFTime.tm_mon = atoi(pBuffer+matches[1].rm_so)-1;
		stFTime.tm_mday = atoi(pBuffer+matches[2].rm_so);
		stFTime.tm_year = atoi(pBuffer+matches[3].rm_so)-1900;
		stFTime.tm_hour = atoi(pBuffer+matches[4].rm_so);
		stFTime.tm_min = atoi(pBuffer+matches[5].rm_so);
		stFTime.tm_sec = atoi(pBuffer+matches[6].rm_so);
		stFTime.tm_isdst = 0;
		StartTime = mktime(&stFTime);
		ltd = matches[0].rm_eo-matches[0].rm_so;
		memcpy(timedate,pBuffer+matches[0].rm_so,ltd);
		timedate[ltd]=0;
		/* compile the regex for the map name */
		if (err=regcomp(&comp,regexg,REG_EXTENDED))
		{
			regerror(err,&comp,error,255);
			printf("Error in regcomp: %s\n",error);
			return 1;
		}
		nomatch=regexec(&comp,pBuffer,10,matches,0);
		if (nomatch)
		{
			/* no map name detected */
			strcpy(mapname,"unknown");
		}
		else
		{
			/* construct a unique game name */
			lmn = matches[1].rm_eo-matches[1].rm_so;
			memcpy(mapname,pBuffer+matches[1].rm_so,lmn);
			mapname[lmn]=0;
			bMapDetected = TRUE;
		}
		sprintf(gm,"%-20s (%s)",mapname,timedate);
		/* get a reference to its data structure */
		pGameStat = PIG_GetPStat(&gpGStat, gm, &Index);
		/* set its start time.. */
		pGameStat->start = StartTime;

		/* compile the regex for "team play" */
		if (err=regcomp(&comp,regextp,REG_EXTENDED))
		{
			regerror(err,&comp,error,255);
			printf("Error in regcomp: %s\n",error);
			return 1;
		}
		nomatch=regexec(&comp,pBuffer,10,matches,0);
		if (nomatch && bMapDetected)
		{
			/* string not found and map name was, therefore team play is OFF */
			bTeamPlay = FALSE;
		}

		/* compile the regular expression for kills */
		if (err=regcomp(&comp,regex,REG_EXTENDED))
		{
			regerror(err,&comp,error,255);
			printf("Error in regcomp: %s\n",error);
			return 1;
		}
		Index = 0;
		bKillsDone = FALSE;
		bDone = FALSE;
		while (!bDone)
		{
			/* look for text matches */
			nomatch=regexec(&comp,pBuffer+Index,10,matches,0);
			if (!nomatch)
			{
				stFTime.tm_mon = atoi(pBuffer+Index+matches[1].rm_so)-1;
				stFTime.tm_mday = atoi(pBuffer+Index+matches[2].rm_so);
				stFTime.tm_year = atoi(pBuffer+Index+matches[3].rm_so)-1900;
				stFTime.tm_hour = atoi(pBuffer+Index+matches[4].rm_so);
				stFTime.tm_min = atoi(pBuffer+Index+matches[5].rm_so);
				stFTime.tm_sec = atoi(pBuffer+Index+matches[6].rm_so);
				stFTime.tm_isdst = 0;
				TimeStamp = mktime(&stFTime);
				lkr = matches[7].rm_eo-matches[7].rm_so;
				lke = matches[8].rm_eo-matches[8].rm_so;
				lwp = matches[9].rm_eo-matches[9].rm_so;
				memcpy(kr,pBuffer+Index+matches[7].rm_so,lkr);
				kr[lkr] = 0;
				memcpy(ke,pBuffer+Index+matches[8].rm_so,lke);
				ke[lke] = 0;
				memcpy(wp,pBuffer+Index+matches[9].rm_so,lwp);
				wp[lwp] = 0;
				Index += matches[9].rm_eo;
				if (PIG_UpdatePStat(0,kr,ke,wp,pGameStat,eParseState))
				{
					return 1;
				}
			}
			else
			{
				/* no more matches, switch to next search state */
				switch (eParseState)
				{
					case PARSEKILLS:
						eParseState = PARSESUICIDES;
						/* compile the next regular expression */
						if (err=regcomp(&comp,regexs,REG_EXTENDED))
						{
							regerror(err,&comp,error,255);
							printf("Error in regcomp: %s\n",error);
							return 1;
						}
						Index = 0;
						break;
					case PARSESUICIDES:
						eParseState = PARSEDEATHS;
						/* compile the next regular expression */
						if (err=regcomp(&comp,regexd,REG_EXTENDED))
						{
							regerror(err,&comp,error,255);
							printf("Error in regcomp: %s\n",error);
							return 1;
						}
						Index = 0;
						break;
					case PARSEDEATHS:
						bDone = TRUE;
						break;
					default:
						break;
				}
			}
		} /* end while */
		/* initialize the player time stamp registers */
		PIG_ClearPSTimeStamps();
bail:
		/* free locally allocated resources */
		regfree(&comp);
		if (pBuffer) free(pBuffer);
 	}
	return 0;
}


#ifdef TEST
int test(void)
{
	char str[256],regex[256];
	regex_t comp;
	regmatch_t matches[10];

	while (1)
	{
		int nomatch,err;

		printf("Enter a string: ");
		if (fgets(str,255,stdin)==NULL) return 0;
		str[strlen(str)-1]=0;
		printf("Enter a regex: ");
		if (fgets(regex,255,stdin)==NULL) return 0;
		regex[strlen(regex)-1]=0;
		if (err=regcomp(&comp,regex,REG_EXTENDED))
		{
			char error[256];
			regerror(err,&comp,error,255);
			printf("Error in regcomp: %s\n",error);
			return 1;
		}

		nomatch=regexec(&comp,str,10,matches,0);
		if (!nomatch)
		{
			int x;

			printf("String matched regexp:\n");
			for (x=0;x<10&&(matches[x].rm_so>=0);x++)
			{
				int p=0;
				while (str[p]!=0)
				{
					if (p==matches[x].rm_so) printf("(");
					if (p==matches[x].rm_eo) printf(")");
					if (str[p]=='(') printf("\\(");
					else if (str[p]==')') printf("\\)");
					else printf("%c",str[p]);
					p++;
				}
				if (p==matches[x].rm_eo) printf(")");
				printf("\n");
			}
		} else {			
			char error[256];
			regerror(nomatch,&comp,error,255);
			printf("Error in regexec: %s\n",error);
		}
		regfree(&comp);
	}
}
#endif 


void PIG_HelpWithArgs(void)
{
	printf("Nice try, but you have to run PigStats with certain arguments:\n\n");
	printf("PigStats [-i<input path>] [-o<output path>] [-s] [-m] [-f]\n\n");
	printf("Where <input path> is the path and file name (including wild cards) of \n");
	printf("the input log file(s) and <output path> is the path and file name of \n");
	printf("the output file. \n\n");
	printf("If the extension of the output file is .htm or .html then output is in \n");
	printf("HTML, otherwise it's straight text.\n\n");
	printf("If no input path is provided PigStats looks for log files in the current\n");
	printf("directory.  If no output path is provided, PigStats generates a text file\n");
	printf("called \"stats.txt\" in the current directory.\n\n");
	printf("'-s' silent operation; default displays a startup banner and parsed \n");
	printf("     file names to the console.\n\n");
	printf("'-m' multi-file html output.  If the output file is specified as html\n");
	printf("     then this flag results in generation of individual output files \n");
	printf("     for game, weapons and player statistics.  This is helpful when a \n");
	printf("     single html file would be too large to quickly load in a browser.\n\n");
	printf("'-f' filter zero scores.  Doesn't display players with zero kills in \n");
	printf("     the general statistics section.\n");
}



int main(int argc, char* argv[])
{
	HANDLE hFoundFile;
	HANDLE hOutFile = INVALID_HANDLE_VALUE;
	WIN32_FIND_DATA stFileData;
	int iRet = 1;
	char cSearchPath[255];
	char cTemp[255];
	char cOutPath[255];
	char *pFound;
	char *pRootName;
	char *pLast=NULL;
	int Index;
	int PathLen;

#ifdef TEST
	test();
#endif

	cSearchPath[0]=0;
	cInPath[0]=0;
	cOutPath[0]=0;
	cTemp[0]=0;

	/* default search path */
	strcpy(cSearchPath,"L*.log");

	for (Index=1;Index<argc;Index++)
	{
		strcpy(cTemp,argv[Index]);
		if (!memcmp(cTemp,"-i",2))
		{
			/* user specified input path */
			strcpy(cInPath,cTemp+2);
			strcpy(cSearchPath,cInPath);
			/* trim the path down to just a path */
			pFound = strstr(cInPath,"\\");
			if (pFound)
			{
				while (pFound)
				{
					pLast = pFound;
					pFound = strstr(pFound+1,"\\");
				}
				*(pLast+1) = 0;
			}
			else
			{
				*cInPath = 0;
			}
		}
		else if (!memcmp(cTemp,"-o",2))
		{
			strcpy(cOutPath,cTemp+2);
		}
		else if (!memcmp(cTemp,"-s",2))
		{
			bSilent = TRUE;
		}
		else if (!memcmp(cTemp,"-f",2))
		{
			/* -f filter out zero statistics */
			bFilter = TRUE;
		}
		else if (!memcmp(cTemp,"-m",2))
		{
			/* -m multiple html file output */
			bMux = TRUE;
		}
		else
		{
			PIG_HelpWithArgs();
			return 1;
		}
	}
	if (!cOutPath[0])
	{
		/* no output file specified, make one */
		strcpy(cOutPath,"stats.txt");
	}
	/* is output file HTML? */
	PathLen = strlen(cOutPath);
	pFound = strstr(cOutPath,".html");
	if (pFound)
	{
		PathLen--;
	}
	else
	{
		pFound = strstr(cOutPath,".htm");
	}
	if (pFound == (cOutPath+PathLen-4))
	{
		bIsHTML = TRUE;
		strcpy(gcRootPath,cOutPath);
		/* gcRootPath is the full path, minus the '.htm(l)' */
		gcRootPath[(int)pFound - (int)cOutPath]=0;
		/* gcExtPath is simply the '.htm(l)' portion */
		strcpy(gcExtPath,pFound);
		/* locate the root name */
		pRootName = gcRootPath + strlen(gcRootPath) - 1;
		while ((pRootName != gcRootPath) && strcspn(pRootName,INVALID_CHARS))
		{
			pRootName--;
		}
		if (pRootName != gcRootPath)
		{
			/* back up one place */
			pRootName++;
		}
		/* copy the root name */
		strcpy(gcRootName,pRootName);
		/* create the cross reference paths */
		strcpy(gcWeaponRef,"./"); 
		strcat(gcWeaponRef,gcRootName);
		strcat(gcWeaponRef,WEAPONEXTENSION);
		strcat(gcWeaponRef,gcExtPath);
		strcpy(gcGeneralRef,"./");
		strcat(gcGeneralRef,gcRootName);
		strcat(gcGeneralRef,gcExtPath);
		strcpy(gcGameRef,"./");
		strcat(gcGameRef,gcRootName);
		strcat(gcGameRef,GAMEEXTENSION);
		strcat(gcGameRef,gcExtPath);
	}
	else
	{
		/* if not HTML, don't split the files */
		bMux = FALSE;
	}


	/* print the startup banner */
	if (!bSilent)
	{
		printf("PigStats v1.3 Copyright (C) 2000 Eric Dennison\n");
		printf("PigStats comes with ABSOLUTELY NO WARRANTY.  This\n");
		printf("is free software, and you are welcome to redistribute\n");
		printf("it under the terms of the GNU Public License.\n");
		printf("See www.gnu.org/copyleft/gpl.html for more details.\n");
		printf("Source code is available from better_dead@yahoo.com\n");
	}

	/* allocate a null player list */
	gppPStatList = (struct PStat**)malloc(0);
	if (!gppPStatList)
	{
		printf("Allocation failure\n");
		return 1;
	}
	/* open the output file */
	hOutFile = CreateFile(	cOutPath,
								GENERIC_WRITE,
								0,
								NULL,
								CREATE_ALWAYS,
								FILE_ATTRIBUTE_ARCHIVE,
								NULL);
	if (hOutFile == INVALID_HANDLE_VALUE)
	{
		printf("sorry... Win32 doesn't seem to like your choice for output file name\n\n");
		PIG_HelpWithArgs();
		return 1;
	}

	hFoundFile = FindFirstFile(cSearchPath, &stFileData);
	if (hFoundFile != INVALID_HANDLE_VALUE) 
	{
		if (PIG_ParseFile(&stFileData))
		{
			return 1;
		}
		if (!bSilent)
		{
			printf("Parsed: %s            \r",stFileData.cFileName);
		}
		while ((hFoundFile != INVALID_HANDLE_VALUE) && iRet)
		{
			iRet = FindNextFile(hFoundFile, &stFileData);
			if (iRet) 
			{
				if (PIG_ParseFile(&stFileData))
				{
					return 1;
				}
				if (!bSilent)
				{
					printf("Parsed: %s            \r",stFileData.cFileName);
				}
			}
		}

		/* display something */
		if (gpPStat)
		{
			/* allocate a global temporary list */
			PIG_StartHTML(hOutFile,"PigStats Half-Life Statistics");
			PIG_ShowOverallStats(hOutFile);
			PIG_ShowWeaponStats(hOutFile);
			PIG_ShowPlayerStats(hOutFile);
			if (bTeamPlay)
			{
				/* try to figure out what the teams are */
				PIG_FindTeams();
				PIG_ShowTeamGameStats(hOutFile);
			}
			else
			{
				PIG_ShowGameStats(hOutFile);
			}
			PIG_EndHTML(hOutFile);
		}
		else
		{
			printf("sorry... Win32 doesn't seem to like your choice for input file name\n\n");
			PIG_HelpWithArgs();
			return 1;
		}
	}
	else
	{
		printf("sorry... Win32 doesn't seem to like your choice for input file name\n\n");
		PIG_HelpWithArgs();
		return 1;
	}

	/* release list */
	free(gppPStatList);
	/* close output file */
	CloseHandle(hOutFile);
	/* clean up dynamically allocated memory */
	PIG_CleanPStat(gpPStat);
	PIG_CleanPStat(gpWStat);
	PIG_CleanPStat(gpGStat);
	PIG_CleanPStat(gpTStat);
	return 0;
}