// required defines
#include "extdll.h"
#include "util.h"
#include "cbase.h"
#include "monsters.h"
#include "weapons.h"
#include "nodes.h"
#include "player.h"
#include "soundent.h"
#include "gamerules.h"

// our weapons animation sequences
enum fgun_e {
        FGUN_IDLE1 = 0,
        FGUN_FIDGETSWAY,
        FGUN_FIDGETSHAKE,
        FGUN_DOWN,
        FGUN_UP,
        FGUN_SHOOT
};

// our frogs animation sequences
enum w_frog_e {
        WFROG_IDLE1 = 0,
        WFROG_FIDGET1,
        WFROG_FIDGET2,
        WFROG_FLIP,
        WFROG_JUMP,
        WFROG_WALK,
        WFROG_RUN,
        WFROG_SWIM,
        WFROG_DIE1,
        WFROG_LEGLESS,
        WFROG_ROLLOVER,
        WFROG_LEGLESS2,
        WFROG_ROLLOVER2
};

// The weapon itself, derived from CBasePlayerWeapon
class CFgun : public CBasePlayerWeapon
{
public:
        void Spawn( void );
        void Precache( void );
        int iItemSlot( void ) { return 3; }             // weapon class 3
        int GetItemInfo(ItemInfo *p);

        void PrimaryAttack( void );
        void SecondaryAttack( void );
        BOOL Deploy( void );
        void WeaponIdle( void );

		float m_flTimeWeaponIdle;
};
LINK_ENTITY_TO_CLASS( weapon_froggun, CFgun );

// our frogs
class CFrogGrenade : public CGrenade
{
        void Spawn( void );
        void Precache( void );
        int  Classify( void );
        void EXPORT SuperBounceTouch( CBaseEntity *pOther );
        void EXPORT HuntThink( void );
        int  BloodColor( void ) { return BLOOD_COLOR_YELLOW; }
        void Killed( entvars_t *pevAttacker, int iGib );
        void GibMonster( void );

        virtual int             Save( CSave &save ); 
        virtual int             Restore( CRestore &restore );
        
        static  TYPEDESCRIPTION m_SaveData[];

        static float m_flNextBounceSoundTime;

        float m_flDie;
        Vector m_vecTarget;
        float m_flNextHunt;
        float m_flNextHit;
        Vector m_posPrev;
        EHANDLE m_hOwner;
        int  m_iMyClass;
};
LINK_ENTITY_TO_CLASS( monster_frog, CFrogGrenade );

float CFrogGrenade::m_flNextBounceSoundTime = 0;

// this code defines which variables we need to save
TYPEDESCRIPTION CFrogGrenade::m_SaveData[] = 
{
        DEFINE_FIELD( CFrogGrenade, m_flDie, FIELD_TIME ),
        DEFINE_FIELD( CFrogGrenade, m_vecTarget, FIELD_VECTOR ),
        DEFINE_FIELD( CFrogGrenade, m_flNextHunt, FIELD_TIME ),
        DEFINE_FIELD( CFrogGrenade, m_flNextHit, FIELD_TIME ),
        DEFINE_FIELD( CFrogGrenade, m_posPrev, FIELD_POSITION_VECTOR ),
        DEFINE_FIELD( CFrogGrenade, m_hOwner, FIELD_EHANDLE ),
};
// implement the save/restore function
IMPLEMENT_SAVERESTORE( CFrogGrenade, CGrenade );

// our ammo
class CFrogAmmo : public CBasePlayerAmmo
{
        void Spawn( void )
        { 
                Precache( );
                SET_MODEL(ENT(pev), "models/w_isotopebox.mdl");
                CBasePlayerAmmo::Spawn( );
        }
        void Precache( void )
        {
                PRECACHE_MODEL ("models/w_isotopebox.mdl");
                PRECACHE_SOUND("items/9mmclip1.wav");
        }
        BOOL AddAmmo( CBaseEntity *pOther ) 
        { 
                if (pOther->GiveAmmo( AMMO_FROGBOX_GIVE, "frogs", FROG_MAX_CARRY ) != -1)
                {
                        EMIT_SOUND(ENT(pev), CHAN_ITEM, "items/9mmclip1.wav", 1, ATTN_NORM);
                        return TRUE;
                }
                return FALSE;
        }
};
LINK_ENTITY_TO_CLASS( ammo_frogs, CFrogAmmo );

void CFrogGrenade :: Spawn( void )
{
        Precache( );
        // motor
        pev->movetype = MOVETYPE_BOUNCE;
        pev->solid = SOLID_BBOX;

        SET_MODEL(ENT(pev), "models/chumtoad.mdl");
        UTIL_SetSize(pev, Vector( -4, -4, 0), Vector(4, 4, 8));
        UTIL_SetOrigin( pev, pev->origin );

        SetTouch( SuperBounceTouch );
        SetThink( HuntThink );
        pev->nextthink = gpGlobals->time + 0.1;
        m_flNextHunt = gpGlobals->time + 1E6;

        pev->flags |= FL_MONSTER;
        pev->takedamage         = DAMAGE_AIM;
        pev->health             = 10;
        pev->gravity            = 0.5;
        pev->friction           = 0.5;

        pev->dmg = 20;

        m_flDie = gpGlobals->time + 120.0f;

        m_flFieldOfView = 0; // 180 degrees

        if ( pev->owner )
                m_hOwner = Instance( pev->owner );

        m_flNextBounceSoundTime = gpGlobals->time;// reset each time a frog is spawned.

        pev->sequence = WFROG_WALK;
        ResetSequenceInfo( );
}

void CFrogGrenade::Precache( void )
{
        PRECACHE_MODEL("models/chumtoad.mdl");
        PRECACHE_SOUND("squeek/sqk_blast1.wav");
        PRECACHE_SOUND("common/bodysplat.wav");
        PRECACHE_SOUND("bullchicken/bc_die3.wav");
        PRECACHE_SOUND("bullchicken/bc_pain1.wav");
        PRECACHE_SOUND("bullchicken/bc_pain2.wav");
        PRECACHE_SOUND("bullchicken/bc_pain3.wav");
        PRECACHE_SOUND("bullchicken/bc_attackgrowl2.wav");
}

int CFrogGrenade :: Classify ( void )
{
        if (m_iMyClass != 0)
                return m_iMyClass; // protect against recursion

        if (m_hEnemy != NULL)
        {
                m_iMyClass = CLASS_INSECT; // no one cares about it
                switch( m_hEnemy->Classify( ) )
                {
                        case CLASS_PLAYER:
                        case CLASS_HUMAN_PASSIVE:
                        case CLASS_HUMAN_MILITARY:
                                m_iMyClass = 0;
                                return CLASS_ALIEN_MILITARY; // barney's get mad, grunts get mad at it
                }
                m_iMyClass = 0;
        }

        return CLASS_ALIEN_BIOWEAPON;
}

void CFrogGrenade::HuntThink( void )
{
        // ALERT( at_console, "think\n" );

        if (!IsInWorld())
        {
                SetTouch( NULL );
                UTIL_Remove( this );
                return;
        }
        
        StudioFrameAdvance( );
        pev->nextthink = gpGlobals->time + 0.1;

        // explode when ready
        if (gpGlobals->time >= m_flDie)
        {
                g_vecAttackDir = pev->velocity.Normalize( );
                pev->health = -1;
                Killed( pev, 0 );
                return;
        }

        // float
        if (pev->waterlevel != 0)
        {
                if (pev->movetype == MOVETYPE_BOUNCE)
                {
                        pev->movetype = MOVETYPE_FLY;
                }
                pev->velocity = pev->velocity * 0.9;
                pev->velocity.z += 8.0;
        }
        else if (pev->movetype = MOVETYPE_FLY)
        {
                pev->movetype = MOVETYPE_BOUNCE;
        }

        // return if not time to hunt
        if (m_flNextHunt > gpGlobals->time)
                return;

        m_flNextHunt = gpGlobals->time + 2.0;
        
        CBaseEntity *pOther = NULL;
        Vector vecDir;
        TraceResult tr;

        Vector vecFlat = pev->velocity;
        vecFlat.z = 0;
        vecFlat = vecFlat.Normalize( );

        UTIL_MakeVectors( pev->angles );

        if (m_hEnemy == NULL || !m_hEnemy->IsAlive())
        {
                // find target, bounce a bit towards it.
                Look( 512 );
                m_hEnemy = BestVisibleEnemy( );
        }

        // burp if it's about time blow up
        if ((m_flDie - gpGlobals->time <= 0.5) && (m_flDie - gpGlobals->time >= 0.3))
        {
                EMIT_SOUND_DYN(ENT(pev), CHAN_VOICE, "bullchicken/bc_die3.wav", 1, ATTN_NORM, 0, 100 + RANDOM_LONG(0,0x3F));
                CSoundEnt::InsertSound ( bits_SOUND_COMBAT, pev->origin, 256, 0.25 );
        }

        // higher pitch as it gets closer to detonation time
        float flpitch = 155.0 - 60.0 * ((m_flDie - gpGlobals->time) / 120.f);
        if (flpitch < 80)
                flpitch = 80;

        if (m_hEnemy != NULL)
        {
                if (FVisible( m_hEnemy ))
                {
                        vecDir = m_hEnemy->EyePosition() - pev->origin;
                        m_vecTarget = vecDir.Normalize( );
                }

                float flVel = pev->velocity.Length();
                float flAdj = 50.0 / (flVel + 10.0);

                if (flAdj > 1.2)
                        flAdj = 1.2;
                
                // ALERT( at_console, "think : enemy\n");

                // ALERT( at_console, "%.0f %.2f %.2f %.2f\n", flVel, m_vecTarget.x, m_vecTarget.y, m_vecTarget.z );

                pev->velocity = pev->velocity * flAdj + m_vecTarget * 300;
        }

        if (pev->flags & FL_ONGROUND)
        {
                pev->avelocity = Vector( 0, 0, 0 );
        }
        else
        {
                if (pev->avelocity == Vector( 0, 0, 0))
                {
                        pev->avelocity.x = RANDOM_FLOAT( -100, 100 );
                        pev->avelocity.z = RANDOM_FLOAT( -100, 100 );
                }
        }

        if ((pev->origin - m_posPrev).Length() < 1.0)
        {
                pev->velocity.x = RANDOM_FLOAT( -100, 100 );
                pev->velocity.y = RANDOM_FLOAT( -100, 100 );
        }
        m_posPrev = pev->origin;

        pev->angles = UTIL_VecToAngles( pev->velocity );
        pev->angles.z = 0;
        pev->angles.x = 0;
}


void CFrogGrenade::SuperBounceTouch( CBaseEntity *pOther )
{
        float   flpitch;

        TraceResult tr = UTIL_GetGlobalTrace( );

        // don't hit the guy that launched this grenade
        if ( pev->owner && pOther->edict() == pev->owner )
                return;

        // at least until we've bounced once
        pev->owner = NULL;

        pev->angles.x = 0;
        pev->angles.z = 0;

        // avoid bouncing too much
        if (m_flNextHit > gpGlobals->time)
                return;

        // higher pitch as it gets closer to detonation time
        flpitch = 155.0 - 60.0 * ((m_flDie - gpGlobals->time) / 120.0f);

        if ( pOther->pev->takedamage && m_flNextAttack < gpGlobals->time)
        {
                // attack!

                // make sure it's me who has touched them
                if (tr.pHit == pOther->edict())
                {
                        // and it's not another froggrenade
                        if (tr.pHit->v.modelindex != pev->modelindex)
                        {
                                // ALERT( at_console, "hit enemy\n");
                                ClearMultiDamage( );
                                // ***** 5 is the damaged caused, you may want to replace it with a skill cvar!!! ******
                                pOther->TraceAttack(pev, 5, gpGlobals->v_forward, &tr, DMG_SLASH ); 
                                // Apply the damage
                                if (m_hOwner != NULL)
                                        ApplyMultiDamage( pev, m_hOwner->pev );
                                else
                                        ApplyMultiDamage( pev, pev );

                                pev->dmg += 5; // add more explosion damage
                                // m_flDie += 2.0; // add more life (mmm tasty)

                                // make bite sound
                                EMIT_SOUND_DYN(ENT(pev), CHAN_WEAPON, "bullchicken/bc_attackgrowl2.wav", 1.0, ATTN_NORM, 0, (int)flpitch);
                                m_flNextAttack = gpGlobals->time + 0.5;
                        }
                }
                else
                {
                        // ALERT( at_console, "been hit\n");
                }
        }

        m_flNextHit = gpGlobals->time + 0.1;
        m_flNextHunt = gpGlobals->time;

        if ( g_pGameRules->IsMultiplayer() )
        {
                // in multiplayer, we limit how often frogs can make their bounce sounds to prevent overflows.
                if ( gpGlobals->time < m_flNextBounceSoundTime )
                {
                        // too soon!
                        return;
                }
        }

        if (!(pev->flags & FL_ONGROUND))
        {
                // play bounce sound
                float flRndSound = RANDOM_FLOAT ( 0 , 1 );

                if ( flRndSound <= 0.33 )
                        EMIT_SOUND_DYN(ENT(pev), CHAN_VOICE, "bullchicken/bc_pain1.wav", 1, ATTN_NORM, 0, (int)flpitch);                
                else if (flRndSound <= 0.66)
                        EMIT_SOUND_DYN(ENT(pev), CHAN_VOICE, "bullchicken/bc_pain2.wav", 1, ATTN_NORM, 0, (int)flpitch);
                else 
                        EMIT_SOUND_DYN(ENT(pev), CHAN_VOICE, "bullchicken/bc_pain3.wav", 1, ATTN_NORM, 0, (int)flpitch);
                CSoundEnt::InsertSound ( bits_SOUND_COMBAT, pev->origin, 256, 0.25 );
        }
        else
        {
                // skittering sound
                CSoundEnt::InsertSound ( bits_SOUND_COMBAT, pev->origin, 100, 0.1 );
        }

        m_flNextBounceSoundTime = gpGlobals->time + 0.5;// half second.
}

void CFrogGrenade :: Killed( entvars_t *pevAttacker, int iGib )
{
        pev->model = iStringNull;// make invisible
        SetThink( SUB_Remove );
        SetTouch( NULL );
        pev->nextthink = gpGlobals->time + 0.1;

        // since frog grenades never leave a body behind, clear out their takedamage now.
        // Frogs do a bit of radius damage when they pop, and that radius damage will
        // continue to call this function unless we acknowledge the Frog's death now.
        pev->takedamage = DAMAGE_NO;

        // play frog blast
        EMIT_SOUND_DYN(ENT(pev), CHAN_ITEM, "squeek/sqk_blast1.wav", 1, 0.5, 0, PITCH_NORM);    

        CSoundEnt::InsertSound ( bits_SOUND_COMBAT, pev->origin, SMALL_EXPLOSION_VOLUME, 3.0 );

        UTIL_BloodDrips( pev->origin, g_vecZero, BloodColor(), 80 );

        if (m_hOwner != NULL)
                RadiusDamage ( pev, m_hOwner->pev, pev->dmg, CLASS_NONE, DMG_BLAST );
        else
                RadiusDamage ( pev, pev, pev->dmg, CLASS_NONE, DMG_BLAST );

        // reset owner so death message happens
        if (m_hOwner != NULL)
                pev->owner = m_hOwner->edict();

        CBaseMonster :: Killed( pevAttacker, GIB_ALWAYS );
}

void CFrogGrenade :: GibMonster( void )
{
        EMIT_SOUND_DYN(ENT(pev), CHAN_VOICE, "common/bodysplat.wav", 0.75, ATTN_NORM, 0, 200);          
}

void CFgun::Spawn( )
{
        // Load models and sounds
        Precache( );
        // Set our weapons id
        m_iId = WEAPON_FROGGUN;
        // set our weapons model
        SET_MODEL(ENT(pev), "models/w_hgun.mdl");
        // set the default amount of ammo
        m_iDefaultAmmo = FROGGUN_DEFAULT_GIVE;
        // drop the weapon to the floor
        FallInit();
}

void CFgun::Precache( void )
{
        // Precache all the hornet guns models
        PRECACHE_MODEL("models/v_hgun.mdl");
        PRECACHE_MODEL("models/w_hgun.mdl");
        PRECACHE_MODEL("models/p_hgun.mdl");
        // Precache our frog
        UTIL_PrecacheOther("monster_frog");
        // Precache our ammo
        UTIL_PrecacheOther("ammo_frogs");
}

int CFgun::GetItemInfo(ItemInfo *p)
{
        p->pszName = STRING(pev->classname);
        p->pszAmmo1 = "frogs";
        p->iMaxAmmo1 = FROG_MAX_CARRY;
        p->pszAmmo2 = NULL;
        p->iMaxAmmo2 = -1;
        p->iMaxClip = FROGGUN_MAX_CLIP;
        p->iSlot = 2; // slot 2
        p->iPosition = 3; // position 3
        p->iId = m_iId = WEAPON_FROGGUN;
        p->iFlags = 0;
        p->iWeight = FROGGUN_WEIGHT;

        return 1;
}

void CFgun::PrimaryAttack( void )
{
        // Check we have some ammo
        if (m_pPlayer->m_rgAmmo[m_iPrimaryAmmoType] == 0)
        {
                PlayEmptySound( );
                return;
        }
        
        // reduce our ammo
        m_pPlayer->m_rgAmmo[m_iPrimaryAmmoType]--;

        // send the weapon animations
        SendWeaponAnim( FGUN_SHOOT );
        m_pPlayer->SetAnimation( PLAYER_ATTACK1 );

        // Play a sound of our frogs screaming into battle (quietly)
        m_pPlayer->m_iWeaponVolume = QUIET_GUN_VOLUME;
        EMIT_SOUND(ENT(m_pPlayer->pev), CHAN_WEAPON, "bullchicken/bc_pain2.wav", 0.8, ATTN_NORM);
 
        // make dem vectors
        UTIL_MakeVectors( m_pPlayer->pev->v_angle + m_pPlayer->pev->punchangle );
        
        // create the frog...
        CBaseEntity *pFrog = CBaseEntity::Create( "monster_frog", 
			m_pPlayer->GetGunPosition( ) + gpGlobals->v_forward * 16 + gpGlobals->v_right * 8 + gpGlobals->v_up * -12, 
			m_pPlayer->pev->v_angle, m_pPlayer->edict() );

        // ...and laucnh him wherever we are looking
        pFrog->pev->velocity = gpGlobals->v_forward * 800;
        pFrog->pev->angles = UTIL_VecToAngles (gpGlobals->v_forward * 800);
        pFrog->pev->owner = ENT(m_pPlayer->pev);
        // Tumble
        pFrog->pev->avelocity.x = RANDOM_FLOAT ( -100, -500 );

        // fire again one second from now
        m_flNextPrimaryAttack = m_flNextSecondaryAttack = gpGlobals->time + 1;
        m_flTimeWeaponIdle = gpGlobals->time + 5;// idle pretty soon after shooting.

        if (!m_pPlayer->m_rgAmmo[m_iPrimaryAmmoType])
                // HEV suit - indicate out of ammo condition
                m_pPlayer->SetSuitUpdate("!HEV_AMO0", FALSE, 0);

        // little bit of a kick for fun
        m_pPlayer->pev->punchangle.x -= 10;
}

void CFgun::SecondaryAttack( void )
{
}

BOOL CFgun::Deploy( )
{
        return DefaultDeploy( "models/v_hgun.mdl", "models/p_hgun.mdl", FGUN_UP, "hive" );
}

void CFgun::WeaponIdle( void )
{
        if (m_flTimeWeaponIdle > gpGlobals->time)
                return;

        int iAnim;
        float flRand = RANDOM_FLOAT(0, 1);
        if (flRand <= 0.75)
        {
                iAnim = FGUN_IDLE1;
                m_flTimeWeaponIdle = gpGlobals->time + 30.0 / 16 * (2);
        }
        else if (flRand <= 0.875)
        {
                iAnim = FGUN_FIDGETSWAY;
                m_flTimeWeaponIdle = gpGlobals->time + 40.0 / 16.0;
        }
        else
        {
                iAnim = FGUN_FIDGETSHAKE;
                m_flTimeWeaponIdle = gpGlobals->time + 35.0 / 16.0;
        }
        SendWeaponAnim( iAnim );
}