//
// NetProbe (Half-Life Proxy Network Monitor)
//
// written by: botman (botman@mailandnews.com)
//
// another fine Half-Life project from http://planethalflife.com/botman/
//

#define STRICT

#include <windows.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <process.h>

#include "resource.h"

#define MAX_BUF_SIZE 2048

#define WM_THREAD_EXIT  (WM_USER + 1)
#define WM_CLIENT_MSG   (WM_USER + 2)
#define WM_SERVER_MSG   (WM_USER + 3)

#define MAX_PACKET 32

typedef struct
{
   time_t time;
   int    length;
   char  buffer[MAX_BUF_SIZE];
} PROXY_PACKET;


HWND    ghNetProbeWnd;
char    gszNetProbeClass[] = {"NetProbeClass"};
TCHAR   gszAppName[] = {"NetProbe"};
TCHAR   gszPath[MAX_PATH];         // path where the running application resides
TCHAR   gszExePathName[MAX_PATH];  // full pathname of the application
TCHAR   gszHelpPathName[MAX_PATH]; // full pathname of the application's help file
TCHAR   gszIniPathName[MAX_PATH];  // full pathname of the application's ini file
HANDLE  ghSemaphore = NULL;
TCHAR   gszDialogErrorMsg[80];

BOOL    gThreadsRunning = 0;
BOOL    gWantToExit = FALSE;

SOCKET       gClientSocket = NULL;
WORD         gwClientPort = 0;
SOCKADDR_IN  gClientRecvAddr;
WORD         gwClientPackets = 0;

char         gszServer[64];
SOCKET       gServerSocket = NULL;
WORD         gwServerPort = 0;
SOCKADDR_IN  gServerSendAddr;
WORD         gwServerPackets = 0;

PROXY_PACKET client_packets[MAX_PACKET];
PROXY_PACKET server_packets[MAX_PACKET];


BOOL NEAR InitApplication( HINSTANCE hInstance );
HWND NEAR InitInstance( HINSTANCE hInstance, int nCmdShow );
LRESULT FAR PASCAL NetProbeWndProc( HWND hWnd, UINT uMsg,
                                    WPARAM wParam, LPARAM lParam );
BOOL CenterWindow (HWND hWnd);
VOID NEAR GoModalDialogBoxParam( HINSTANCE hInstance, LPCSTR lpszTemplate,
                                 HWND hWnd, DLGPROC lpDlgProc, LPARAM lParam );
BOOL FAR PASCAL SetupDlgProc( HWND hDlg, UINT uMsg,
                              WPARAM wParam, LPARAM lParam );
BOOL FAR PASCAL ErrDlgProc( HWND hWnd, UINT uMsg,
                            WPARAM wParam, LPARAM lParam );
void FormatBuffer(char *pData, int length, char *pBuffer);
UINT GetPathFromFullPathName( LPCTSTR lpFullPathName, LPTSTR lpPathBuffer,
                              UINT nPathBufferLength );
BOOL InitSockets(HWND hWnd);
DWORD WINAPI ClientThread(LPVOID hWnd);
DWORD WINAPI ServerThread(LPVOID hWnd);


int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
                   LPSTR lpszCmdLine, int nCmdShow)
{
   HANDLE hMutex;
   MSG msg;

   hMutex = CreateMutex(NULL, FALSE, gszNetProbeClass);

   if ((hMutex != NULL) && (GetLastError() == ERROR_ALREADY_EXISTS))
   {
      MessageBox (NULL, "NetProbe is already running!", NULL, MB_OK);
      CloseHandle(hMutex);
      return FALSE;
   }

   if (!hPrevInstance)
   {
      if (!InitApplication( hInstance ))
      {
         MessageBox (NULL, "InitApplication Failed!", NULL, MB_OK);
         CloseHandle(hMutex);
         return( FALSE );
      }
   }

   if (NULL == InitInstance( hInstance, nCmdShow ))
   {
      MessageBox (NULL, "InitInstance Failed!", NULL, MB_OK);
      CloseHandle(hMutex);
      return( FALSE );
   }


   while (GetMessage( &msg, NULL, 0, 0 ))
   {
      TranslateMessage( &msg );
      DispatchMessage( &msg );
   }

   CloseHandle(hMutex);

   return msg.wParam;

} // end of WinMain()


BOOL NEAR InitApplication( HINSTANCE hInstance )
{
   WNDCLASS  wndclass;
   LPCTSTR lpszIniFileExt  = TEXT( "INI" );
   LPCTSTR lpszHelpFileExt = TEXT( "HLP" );

   // register the NetProbe window class

   wndclass.style =         0;
   wndclass.lpfnWndProc =   NetProbeWndProc;
   wndclass.cbClsExtra =    0;
   wndclass.cbWndExtra =    NULL;
   wndclass.hInstance =     hInstance;
   wndclass.hIcon =         LoadIcon(hInstance, MAKEINTRESOURCE(NETPROBEICON));
   wndclass.hCursor =       NULL;
   wndclass.hbrBackground = (HBRUSH) (COLOR_WINDOW + 1);
   wndclass.lpszMenuName =  MAKEINTRESOURCE( NETPROBEMENU );
   wndclass.lpszClassName = gszNetProbeClass;

   if (!RegisterClass( &wndclass ))
      return( FALSE );

   //-- get application pathname and store the ini and help file pathname
   //   (which is located in the same directory as the application)
   GetModuleFileName( (HINSTANCE) NULL, gszExePathName,
                      sizeof(gszExePathName)/sizeof(TCHAR) );
   GetPathFromFullPathName( gszExePathName, gszPath,
                            sizeof(gszPath)/sizeof(TCHAR) );
   wsprintf( gszIniPathName,  TEXT( "%s\\%s.%s" ), gszPath,
             gszAppName, lpszIniFileExt  );
   wsprintf( gszHelpPathName, TEXT( "%s\\%s.%s" ), gszPath,
             gszAppName, lpszHelpFileExt );

   //-- retrieve stored default location from private profile data
//   return(GetPrivateProfileSettings( gszAppName, szIniPathName,
//           &Profile ));
   return TRUE;

} // end of InitApplication()


HWND NEAR InitInstance( HINSTANCE hInstance, int nCmdShow )
{

   // create the window
   ghNetProbeWnd = CreateWindow( gszNetProbeClass, gszAppName,
                           WS_OVERLAPPED | WS_CAPTION |
                           WS_SYSMENU | WS_MINIMIZEBOX,
                           CW_USEDEFAULT, 0, 600, 360,
                           NULL, NULL, hInstance, NULL );

   if (NULL == ghNetProbeWnd)
      return( NULL );

   ShowWindow( ghNetProbeWnd, nCmdShow );
   UpdateWindow( ghNetProbeWnd );

   return( ghNetProbeWnd );

} // end of InitInstance()


LRESULT FAR PASCAL NetProbeWndProc( HWND hWnd, UINT uMsg,
                                    WPARAM wParam, LPARAM lParam )
{
   HMENU hMenu;
   DWORD child_thread;
   static int client_index = 0;
   static int server_index = 0;
   static FILE *fp = NULL;
   char   out_str[MAX_BUF_SIZE*5];  // output buffer for hex dump
   char   temp_time[32];
   int    str_len;


   switch (uMsg)
   {
      case WM_CREATE:

         // disable the STOP and START menu item (until SETUP is run)
         hMenu = GetMenu( hWnd );
         EnableMenuItem( hMenu, IDM_STOP,
                         MF_GRAYED | MF_DISABLED | MF_BYCOMMAND );
         EnableMenuItem( hMenu, IDM_START,
                         MF_GRAYED | MF_DISABLED | MF_BYCOMMAND );

         break ;

      case WM_COMMAND:
      {
         switch ( LOWORD( wParam ) )
         {
            case IDM_EXIT:

               PostMessage( hWnd, WM_CLOSE, 0, 0L );
               break;

            case IDM_SETUP:

               GoModalDialogBoxParam( (HINSTANCE) GetWindowLong( hWnd, GWL_HINSTANCE ),
                                      MAKEINTRESOURCE( SETUPDLGBOX ),
                                      hWnd, SetupDlgProc, 0L );

               // if setup values are valid then enable START menu item...
               if ((gwClientPort != 0) && (gwServerPort != 0) &&
                   (gszServer[0] != 0))
               {
                  hMenu = GetMenu( (HWND)hWnd );

                  EnableMenuItem( hMenu, IDM_START, MF_ENABLED | MF_BYCOMMAND );
               }

               break;

            case IDM_START:

               if (!InitSockets(hWnd))
               {
                  WSACleanup();

                  PostMessage( hWnd, WM_CLOSE, 0, 0L );
                  break;
               }

               fp = fopen("log.txt", "w");

               // disable the START menu item and enable the STOP menu item
               hMenu = GetMenu( hWnd );
               EnableMenuItem( hMenu, IDM_START,
                               MF_GRAYED | MF_DISABLED | MF_BYCOMMAND );

               EnableMenuItem( hMenu, IDM_STOP, MF_ENABLED | MF_BYCOMMAND );


               // spawn the threads here...
               if (CreateThread(NULL, 0, ClientThread, hWnd,
                                0, &child_thread) == NULL)
               {
                  MessageBox (hWnd, "Can't create ClientThread!", NULL, MB_OK);
                  PostMessage( hWnd, WM_CLOSE, 0, 0L );
               }

               gThreadsRunning++;

               if (CreateThread(NULL, 0, ServerThread, hWnd,
                                0, &child_thread) == NULL)
               {
                  MessageBox (hWnd, "Can't create ServerThread!", NULL, MB_OK);
                  PostMessage( hWnd, WM_CLOSE, 0, 0L );
               }

               gThreadsRunning++;

               break;

            case IDM_STOP:

               if (gClientSocket != NULL)
               {
                  closesocket(gClientSocket);  // should cause thread to exit
                  gClientSocket = NULL;
               }

               if (gServerSocket != NULL)
               {
                  closesocket(gServerSocket);  // should cause thread to exit
                  gServerSocket = NULL;
               }

               break;
         }
      }
      break ;

      case WM_THREAD_EXIT:

         gThreadsRunning--;

         if (gThreadsRunning == 0)
         {
            WSACleanup();

            CloseHandle(ghSemaphore);

            // disable the STOP menu item and enable the START menu item
            hMenu = GetMenu( (HWND)hWnd );
            EnableMenuItem( hMenu, IDM_STOP,
                            MF_GRAYED | MF_DISABLED | MF_BYCOMMAND );

            EnableMenuItem( hMenu, IDM_START, MF_ENABLED | MF_BYCOMMAND );

            if (gWantToExit)
               PostMessage( hWnd, WM_CLOSE, 0, 0L );
         }

         break;

      case WM_CLIENT_MSG:

         FormatBuffer(client_packets[client_index].buffer,
                      client_packets[client_index].length, out_str);

         strcpy(temp_time, asctime(localtime(&client_packets[client_index].time)));
         str_len = strlen(temp_time);
         temp_time[str_len-6] = 0;  // chop off end of localtime string

         fprintf(fp, "client(%d)->server(%d):  %s  length=%d\n",
                 htons(gClientRecvAddr.sin_port),
                 htons(gServerSendAddr.sin_port),
                 &temp_time[11],
                 client_packets[client_index].length);

         fprintf(fp, "%s\n", out_str);

         client_index++;
         if (client_index == MAX_PACKET)
            client_index = 0;

         break;

      case WM_SERVER_MSG:

         FormatBuffer(server_packets[server_index].buffer,
                      server_packets[server_index].length, out_str);

         strcpy(temp_time, asctime(localtime(&server_packets[server_index].time)));
         str_len = strlen(temp_time);
         temp_time[str_len-6] = 0;  // chop off end of localtime string

         fprintf(fp, "server(%d)->client(%d):  %s  length=%d\n",
                 htons(gServerSendAddr.sin_port),
                 htons(gClientRecvAddr.sin_port),
                 &temp_time[11],
                 server_packets[server_index].length);

         fprintf(fp, "%s\n", out_str);

         server_index++;
         if (server_index == MAX_PACKET)
            server_index = 0;

         break;

      case WM_DESTROY:

         if (fp != NULL)
            fclose(fp);

         PostQuitMessage( 0 );
         break ;

      case WM_CLOSE:

         if (gThreadsRunning > 0)
         {
            gWantToExit = TRUE;

            if (gClientSocket != NULL)
            {
               closesocket(gClientSocket);  // should cause thread to exit
               gClientSocket = NULL;
            }

            if (gServerSocket != NULL)
            {
               closesocket(gServerSocket);  // should cause thread to exit
               gServerSocket = NULL;
            }

            return TRUE;  // don't close just yet, wait for threads...
         }

         // fall through

      default:
         return( DefWindowProc( hWnd, uMsg, wParam, lParam ) );
   }
   return 0L;

} // end of NetProbeWndProc()


UINT GetPathFromFullPathName( LPCTSTR lpFullPathName, LPTSTR lpPathBuffer,
                              UINT nPathBufferLength )
{
   UINT nLength;
   int i;

   if ((nLength = (UINT) lstrlen( lpFullPathName ) ) > nPathBufferLength)
      return( nLength );

   lstrcpy( lpPathBuffer, lpFullPathName );

   for( i = lstrlen( lpPathBuffer );
        (lpPathBuffer[i] != '\\') && (lpPathBuffer[i] != ':'); i-- )
      ;
   if (':' == lpPathBuffer[i])
      lpPathBuffer[i+1] = '\0';
   else
      lpPathBuffer[i] = '\0';

   return( (UINT) i );
}


BOOL CenterWindow (HWND hWnd)
{
    RECT    rRect, rParentRect;
    HWND    hParentWnd;
    int     wParent, hParent, xNew, yNew;
    int     w, h;

    GetWindowRect (hWnd, &rRect);
    w = rRect.right - rRect.left;
    h = rRect.bottom - rRect.top;

    if (NULL == (hParentWnd = GetParent( hWnd )))
       return( FALSE );

    GetWindowRect( hParentWnd, &rParentRect );

    wParent = rParentRect.right - rParentRect.left;
    hParent = rParentRect.bottom - rParentRect.top;

    xNew = wParent/2 - w/2 + rParentRect.left;
    yNew = hParent/2 - h/2 + rParentRect.top;

    return SetWindowPos (hWnd, NULL, xNew, yNew, 0, 0, SWP_NOSIZE | SWP_NOZORDER);
}  // end of CenterWindow


VOID NEAR GoModalDialogBoxParam( HINSTANCE hInstance, LPCSTR lpszTemplate,
                                 HWND hWnd, DLGPROC lpDlgProc, LPARAM lParam )
{
   DLGPROC  lpProcInstance ;

   lpProcInstance = (DLGPROC) MakeProcInstance( (FARPROC) lpDlgProc,
                                                hInstance ) ;

   DialogBoxParam( hInstance, lpszTemplate, hWnd, lpProcInstance, lParam ) ;

   FreeProcInstance( (FARPROC) lpProcInstance ) ;

} // end of GoModalDialogBoxParam()


BOOL FAR PASCAL SetupDlgProc( HWND hDlg, UINT uMsg,
                              WPARAM wParam, LPARAM lParam )
{
   unsigned int temp_client_port;
   unsigned int temp_server_port;
   char temp_server_addr[17];
   char temp_str[64];

   switch (uMsg)
   {
      case WM_INITDIALOG:
         if (!CenterWindow( hDlg ))
            return( FALSE );

         SetDlgItemText( hDlg, IDC_PROXYIP, TEXT("LOCAL HOST") );

         if (gwClientPort != 0)
         {
            sprintf(temp_str, "%d", gwClientPort);
            SetDlgItemText( hDlg, IDC_PROXYPORT, temp_str );
         }

         if (gszServer[0] != 0)
         {
            SetDlgItemText( hDlg, IDC_SERVERIP, gszServer );
         }

         if (gwServerPort != 0)
         {
            sprintf(temp_str, "%d", gwServerPort);
            SetDlgItemText( hDlg, IDC_SERVERPORT, temp_str );
         }

         return TRUE;

      case WM_COMMAND:
         switch ((WORD) wParam)
         {
            case IDOK:

               temp_client_port = GetDlgItemInt( hDlg, IDC_PROXYPORT, NULL, FALSE );

               if ((temp_client_port < 1) || (temp_client_port > 65535))
               {
                  lstrcpy( gszDialogErrorMsg,
                           TEXT ("Proxy Port MUST be between 1 and 65535") );

                  GoModalDialogBoxParam( (HINSTANCE) GetWindowLong( hDlg, GWL_HINSTANCE ),
                                         MAKEINTRESOURCE( ERRDLGBOX ),
                                         hDlg, (DLGPROC) ErrDlgProc, 0L );

                  return ( TRUE );
               }

               temp_server_port = GetDlgItemInt( hDlg, IDC_SERVERPORT, NULL, FALSE );

               if ((temp_server_port < 1) || (temp_server_port > 65535))
               {
                  lstrcpy( gszDialogErrorMsg,
                           TEXT ("Server Port MUST be between 1 and 65535") );

                  GoModalDialogBoxParam( (HINSTANCE) GetWindowLong( hDlg, GWL_HINSTANCE ),
                                         MAKEINTRESOURCE( ERRDLGBOX ),
                                         hDlg, (DLGPROC) ErrDlgProc, 0L );

                  return ( TRUE );
               }

               if (temp_client_port == temp_server_port)
               {
                  lstrcpy( gszDialogErrorMsg,
                           TEXT ("Proxy and Server Port CANNOT be equal") );

                  GoModalDialogBoxParam( (HINSTANCE) GetWindowLong( hDlg, GWL_HINSTANCE ),
                                         MAKEINTRESOURCE( ERRDLGBOX ),
                                         hDlg, (DLGPROC) ErrDlgProc, 0L );

                  return ( TRUE );
               }

               if (GetDlgItemText( hDlg, IDC_SERVERIP, temp_server_addr, 17) == 0)
               {
                  lstrcpy( gszDialogErrorMsg,
                           TEXT ("You must specify a Server IP Address") );

                  GoModalDialogBoxParam( (HINSTANCE) GetWindowLong( hDlg, GWL_HINSTANCE ),
                                         MAKEINTRESOURCE( ERRDLGBOX ),
                                         hDlg, (DLGPROC) ErrDlgProc, 0L );

                  return ( TRUE );
               }

               gwClientPort = temp_client_port;
               gwServerPort = temp_server_port;

               strcpy(gszServer, temp_server_addr);

               EndDialog(hDlg, TRUE);
               return ( TRUE );

            case IDCANCEL:
               EndDialog(hDlg, TRUE);
               return ( TRUE );
         }
         break;
   }

   return FALSE;

}  // end of SetupDlgProc()


BOOL FAR PASCAL ErrDlgProc( HWND hDlg, UINT uMsg,
                            WPARAM wParam, LPARAM lParam )
{
   switch (uMsg)
   {
      case WM_INITDIALOG:

         if (!CenterWindow( hDlg ))
            return( FALSE );

         SetDlgItemText( hDlg, IDC_ERR, gszDialogErrorMsg ) ;

         return TRUE;

      case WM_COMMAND:

         switch (LOWORD( wParam ))
         {
            case IDOK:
               EndDialog( hDlg, TRUE );
         }

         break;
   }

   return( FALSE );

}  // end of ErrDlgProc()


void FormatBuffer(char *pData, int length, char *pBuffer)
{
   int index;
   char *pDataPrev;

   *pBuffer = 0;
   pDataPrev = pData;

   for (index = 0; index < length; index++)
   {
      if (index % 16 == 8)
         sprintf(pBuffer, "-%02X", (unsigned char)*pData);
      else
         sprintf(pBuffer, " %02X", (unsigned char)*pData);

      pBuffer += 3;
      pData++;

      if ((index+1) % 16 == 0)
      {
         *pBuffer++ = ' ';
         *pBuffer++ = ' ';
         *pBuffer++ = '|';

         while (pDataPrev < pData)
         {
            if ((*pDataPrev >= ' ') && (*pDataPrev <= '~'))
               *pBuffer++ = *pDataPrev++;
            else
            {
               *pBuffer++ = '.';
               pDataPrev++;
            }
         }

         *pBuffer++ = '|';
         *pBuffer++ = '\n';
      }
   }

   if ((length % 16) != 0)
   {
      for (index=(length % 16); index < 16; index++)
      {
         *pBuffer++ = ' ';
         *pBuffer++ = ' ';
         *pBuffer++ = ' ';
      }

      *pBuffer++ = ' ';
      *pBuffer++ = ' ';
      *pBuffer++ = '|';

      while (pDataPrev < pData)
      {
         if ((*pDataPrev >= ' ') && (*pDataPrev <= '~'))
            *pBuffer++ = *pDataPrev++;
         else
         {
            *pBuffer++ = '.';
            pDataPrev++;
         }
      }

      for (index=(length % 16); index < 16; index++)
         *pBuffer++ = ' ';

      *pBuffer++ = '|';
      *pBuffer++ = '\n';
   }

   *pBuffer = 0;
}


BOOL InitSockets(HWND hWnd)
{
   WSADATA WSAData;
   unsigned int addr;
   struct hostent *hp;
   char  err_msg[80];

   // start up the Windows Sockets Interface...
   if (WSAStartup(0x0101, &WSAData))
   {
      MessageBox (hWnd, "Can't start Windows Sockets Interface!", NULL, MB_OK);

      return FALSE;
   }

   gClientSocket = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);

   if (gClientSocket == INVALID_SOCKET)
   {
      sprintf(err_msg, "Can't create Client socket!: %d",
              WSAGetLastError());
      MessageBox ((HWND)hWnd, err_msg, NULL, MB_OK);

      return FALSE;
   }

   memset(&gClientRecvAddr, 0, sizeof(gClientRecvAddr));

   gClientRecvAddr.sin_family = AF_INET;
   gClientRecvAddr.sin_addr.s_addr = INADDR_ANY;
   gClientRecvAddr.sin_port = htons(gwClientPort);

   if (bind(gClientSocket, (struct sockaddr *)&gClientRecvAddr,
            sizeof(gClientRecvAddr)) == SOCKET_ERROR)
   {
      sprintf(err_msg, "Can't bind to Client socket!: %d",
              WSAGetLastError());
      MessageBox ((HWND)hWnd, err_msg, NULL, MB_OK);

      return FALSE;
   }


   gServerSocket = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);

   if (gServerSocket == INVALID_SOCKET)
   {
      sprintf(err_msg, "Can't create Server socket!: %d",
              WSAGetLastError());
      MessageBox ((HWND)hWnd, err_msg, NULL, MB_OK);
      return FALSE;
   }

   // find the IP address the HL server...
   if (isalpha(gszServer[0]))
      hp = gethostbyname(gszServer);
   else
   {
      addr = inet_addr(gszServer);
      hp = gethostbyaddr((char *)&addr, 4, AF_INET);
   }

   if (hp == NULL)
   {
      sprintf(err_msg, "Can't resolve Server IP address!: %d",
              WSAGetLastError());
      MessageBox ((HWND)hWnd, err_msg, NULL, MB_OK);

      return FALSE;
   }

   memset(&gServerSendAddr, 0, sizeof(gServerSendAddr));

   gServerSendAddr.sin_family = AF_INET;
   memcpy(&(gServerSendAddr.sin_addr), hp->h_addr, hp->h_length);
   gServerSendAddr.sin_port = htons(gwServerPort);

   // create the semaphore to syncronize the threads...
   ghSemaphore = CreateSemaphore(NULL, 0, 1, NULL);

   return TRUE;
}


DWORD WINAPI ClientThread(LPVOID hWnd)
{
   int   length;
   char  buffer[MAX_BUF_SIZE];
   int   total_sent;
   int   sent;
   char  err_msg[80];
   int   recv_len;
   int   server_len;
   int   index = 0;

   recv_len = sizeof(gClientRecvAddr);
   server_len = sizeof(gServerSendAddr);

   // recv a packet from the client (and store the client's last UDP port)...
   length = recvfrom(gClientSocket, buffer, MAX_BUF_SIZE, 0,
                     (SOCKADDR *)&gClientRecvAddr, &recv_len);

   // increment the semaphore after getting a valid socket connection...
   ReleaseSemaphore(ghSemaphore, 1, NULL);


   while (length > 0)
   {
      // store this packet so main process can output to file...
      time (&client_packets[index].time);
      client_packets[index].length = length;
      memcpy(client_packets[index].buffer, buffer, length);

      index++;
      if (index == MAX_PACKET)
         index = 0;

      gwClientPackets++;  // count the total number of packets

      total_sent = 0;

      // send the received packet on to the server...
      while (total_sent < length)
      {
         sent = sendto(gServerSocket, &buffer[total_sent],
                       length-total_sent, 0,
                       (struct sockaddr *)&gServerSendAddr, server_len);

         if (sent == SOCKET_ERROR)
         {
            sprintf(err_msg, "Error sending Client data to Server: %d",
                    WSAGetLastError());
            MessageBox ((HWND)hWnd, err_msg, NULL, MB_OK);

            if (gClientSocket != NULL)
            {
               closesocket(gClientSocket);  // should cause thread to exit
               gClientSocket = NULL;
            }

            if (gServerSocket != NULL)
            {
               closesocket(gServerSocket);  // should cause thread to exit
               gServerSocket = NULL;
            }

            PostMessage( (HWND)hWnd, WM_THREAD_EXIT, 0, 0L );

            return FALSE;
         }

         total_sent += sent;
      }

      // tell the main process there's something to output to log file...
      PostMessage( (HWND)hWnd, WM_CLIENT_MSG, 0, 0L );

      // recv the next packet from the client...
      length = recvfrom(gClientSocket, buffer, MAX_BUF_SIZE, 0,
                        (SOCKADDR *)&gClientRecvAddr, &recv_len);
   }

   if (gClientSocket != NULL)
   {
      closesocket(gClientSocket);  // should cause thread to exit
      gClientSocket = NULL;
   }

   if (gServerSocket != NULL)
   {
      closesocket(gServerSocket);  // should cause thread to exit
      gServerSocket = NULL;
   }

   PostMessage( (HWND)hWnd, WM_THREAD_EXIT, 0, 0L );

   return TRUE;  // return GOOD status
}


DWORD WINAPI ServerThread(LPVOID hWnd)
{
   int   length;
   char  buffer[MAX_BUF_SIZE];
   int   total_sent;
   int   sent;
   char  err_msg[80];
   int   recv_len;
   int   server_len;
   int   index = 0;

   // wait for client thread to increment the semaphore...
   if (WaitForSingleObject(ghSemaphore, INFINITE) != WAIT_OBJECT_0)
   {
      MessageBox ((HWND)hWnd, "ServerThread failed wait for Semaphore!", NULL, MB_OK);

      PostMessage( (HWND)hWnd, WM_THREAD_EXIT, 0, 0L );

      return FALSE;
   }

   recv_len = sizeof(gClientRecvAddr);
   server_len = sizeof(gServerSendAddr);

   length = recvfrom(gServerSocket, buffer, MAX_BUF_SIZE, 0,
                     (SOCKADDR *)&gServerSendAddr, &server_len);

   while (length > 0)
   {
      // store this packet so main process can output to file...
      time (&server_packets[index].time);
      server_packets[index].length = length;
      memcpy(server_packets[index].buffer, buffer, length);

      index++;
      if (index == MAX_PACKET)
         index = 0;

      gwServerPackets++;  // count the total number of packets

      total_sent = 0;

      // send the received packet on to the client...
      while (total_sent < length)
      {
         // send data to client on last port where data was received from...

         sent = sendto(gClientSocket, &buffer[total_sent],
                       length - total_sent, 0,
                       (SOCKADDR *)&gClientRecvAddr, recv_len);

         if (sent == SOCKET_ERROR)
         {
            sprintf(err_msg, "Error sending Server data to Client: %d",
                    WSAGetLastError());
            MessageBox ((HWND)hWnd, err_msg, NULL, MB_OK);

            if (gClientSocket != NULL)
            {
               closesocket(gClientSocket);  // should cause thread to exit
               gClientSocket = NULL;
            }

            if (gServerSocket != NULL)
            {
               closesocket(gServerSocket);  // should cause thread to exit
               gServerSocket = NULL;
            }

            PostMessage( (HWND)hWnd, WM_THREAD_EXIT, 0, 0L );

            return FALSE;
         }

         total_sent += sent;
      }

      // tell the main process there's something to output to log file...
      PostMessage( (HWND)hWnd, WM_SERVER_MSG, 0, 0L );

      // recv the next packet from the server...
      length = recvfrom(gServerSocket, buffer, MAX_BUF_SIZE, 0,
                        (SOCKADDR *)&gServerSendAddr, &server_len);
   }

   if (gClientSocket != NULL)
   {
      closesocket(gClientSocket);  // should cause thread to exit
      gClientSocket = NULL;
   }

   if (gServerSocket != NULL)
   {
      closesocket(gServerSocket);  // should cause thread to exit
      gServerSocket = NULL;
   }

   PostMessage( (HWND)hWnd, WM_THREAD_EXIT, 0, 0L );

   return TRUE;  // return GOOD status
}

