// botman's Half-Life bot example (http://jump.to/botman/)

#include "extdll.h"
#include "util.h"
#include "client.h"
#include "cbase.h"
#include "player.h"
#include "items.h"
#include "effects.h"
#include "weapons.h"
#include "soundent.h"
#include "gamerules.h"
#include "animation.h"

#include "bot.h"

extern DLL_GLOBAL ULONG g_ulModelIndexPlayer;

// Set in combat.cpp.  Used to pass the damage inflictor for death messages.
extern entvars_t *g_pevLastInflictor;

extern int gmsgHealth;
extern int gmsgCurWeapon;
extern int gmsgSetFOV;

#define PLAYER_SEARCH_RADIUS (float)64

int f_Observer = 0;  // flag to indicate if player is in observer mode
int f_botskill = 3;  // default bot skill level
int f_botdontshoot = 0;
char botskin[16];
char botname[32];

#define MAX_SKINS 10

// indicate which models are currently used for random model allocation
BOOL skin_used[MAX_SKINS] = {
   FALSE, FALSE, FALSE, FALSE, FALSE,
   FALSE, FALSE, FALSE, FALSE, FALSE};

// store the names of the models...
char *bot_skins[MAX_SKINS] = {
   "barney", "gina", "gman", "gordon", "helmet",
   "hgrunt", "recon", "robo", "scientist", "zombie"};

// store the player names for each of the models...
char *bot_names[MAX_SKINS] = {
   "Barney", "Gina", "G-Man", "Gordon", "Helmet",
   "H-Grunt", "Recon", "Robo", "Scientist", "Zombie"};

// speed adjustment array based on skill (multiply by max_speed)
float speed_adjust[5] = {1.0, 0.97, 0.95, 0.90, 0.85};

// how often (out of 1000 times) the bot will pause, based on bot skill
float pause_frequency[5] = {4, 7, 10, 15, 20};

// how long the bot will delay when paused, based on bot skill
float pause_time[5][2] = {
   {0.2, 0.5}, {0.5, 1.0}, {0.7, 1.3}, {1.0, 1.7}, {1.2, 2.0}};

// weapon firing delay based on skill (min and max delay for each weapon)
float primary_fire_delay[WEAPON_SNARK+1][5][2] = {
   // WEAPON_NONE - NOT USED
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_CROWBAR
   {{0.0, 0.1}, {0.2, 0.3}, {0.3, 0.5}, {0.4, 0.6}, {0.6, 1.0}},
   // WEAPON_GLOCK (9mm)
   {{0.0, 0.1}, {0.1, 0.2}, {0.2, 0.3}, {0.3, 0.4}, {0.4, 0.5}},
   // WEAPON_PYTHON (357)
   {{0.0, 0.25}, {0.2, 0.5}, {0.4, 0.8}, {1.0, 1.3}, {1.5, 2.0}},
   // WEAPON_MP5 (9mmAR)
   {{0.0, 0.1}, {0.1, 0.3}, {0.3, 0.5}, {0.4, 0.6}, {0.5, 0.8}},
   // WEAPON_CHAINGUN - NOT USED
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_CROSSBOW
   {{0.0, 0.25}, {0.2, 0.4}, {0.5, 0.7}, {0.8, 1.0}, {1.0, 1.3}},
   // WEAPON_SHOTGUN
   {{0.0, 0.25}, {0.2, 0.5}, {0.4, 0.8}, {0.6, 1.2}, {0.8, 2.0}},
   // WEAPON_RPG
   {{1.0, 3.0}, {2.0, 4.0}, {3.0, 5.0}, {4.0, 6.0}, {5.0, 7.0}},
   // WEAPON_GAUSS
   {{0.0, 0.1}, {0.2, 0.3}, {0.3, 0.5}, {0.5, 0.8}, {1.0, 1.2}},
   // WEAPON_EGON
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_HORNETGUN
   {{0.0, 0.1}, {0.25, 0.4}, {0.4, 0.7}, {0.6, 1.0}, {1.0, 1.5}},
   // WEAPON_HANDGRENADE
   {{1.0, 3.0}, {2.0, 4.0}, {3.0, 5.0}, {4.0, 6.0}, {5.0, 7.0}},
   // WEAPON_TRIPMINE
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_SATCHEL
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_SNARK
   {{0.0, 0.1}, {0.1, 0.2}, {0.2, 0.5}, {0.5, 0.7}, {0.6, 1.0}},
   };

float secondary_fire_delay[WEAPON_SNARK+1][5][2] = {
   // WEAPON_NONE - NOT USED
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_CROWBAR - Not applicable
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_GLOCK (9mm)
   {{0.0, 0.1}, {0.0, 0.1}, {0.1, 0.2}, {0.1, 0.2}, {0.2, 0.4}},
   // WEAPON_PYTHON (357)
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_MP5 (9mmAR)
   {{0.0, 0.3}, {0.5, 0.8}, {0.7, 1.0}, {1.0, 1.6}, {1.4, 2.0}},
   // WEAPON_CHAINGUN - NOT USED
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_CROSSBOW
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_SHOTGUN
   {{0.0, 0.25}, {0.2, 0.5}, {0.4, 0.8}, {0.6, 1.2}, {0.8, 2.0}},
   // WEAPON_RPG - Not applicable
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_GAUSS
   {{0.2, 0.5}, {0.3, 0.7}, {0.5, 1.0}, {0.8, 1.5}, {1.0, 2.0}},
   // WEAPON_EGON - Not applicable
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_HORNETGUN
   {{0.0, 0.1}, {0.2, 0.3}, {0.3, 0.5}, {0.5, 0.8}, {0.7, 1.2}},
   // WEAPON_HANDGRENADE - Not applicable
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_TRIPMINE - Not applicable
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_SATCHEL
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_SNARK - Not applicable
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}}
   };   

char hgrunt_sounds[][30] = {
   "hgrunt/gr_pain1.wav",
   "hgrunt/gr_pain2.wav",
   "hgrunt/gr_pain3.wav",
   "hgrunt/gr_pain4.wav",
   "hgrunt/gr_pain5.wav"};

char barney_sounds[][30] = {
   "barney/ba_bring.wav",
   "barney/ba_pain1.wav",
   "barney/ba_pain2.wav",
   "barney/ba_pain3.wav",
   "barney/ba_die1.wav"};

char scientist_sounds[][30] = {
   "scientist/sci_fear8.wav",
   "scientist/sci_fear13.wav",
   "scientist/sci_fear14.wav",
   "scientist/sci_fear15.wav",
   "scientist/sci_pain3.wav"};

ammo_check_t ammo_check[] = {
   {"ammo_glockclip", "9mm", _9MM_MAX_CARRY},
   {"ammo_9mmclip", "9mm", _9MM_MAX_CARRY},
   {"ammo_9mmAR", "9mm", _9MM_MAX_CARRY},
   {"ammo_9mmbox", "9mm", _9MM_MAX_CARRY},
   {"ammo_mp5clip", "9mm", _9MM_MAX_CARRY},
   {"ammo_chainboxclip", "9mm", _9MM_MAX_CARRY},
   {"ammo_mp5grenades", "ARgrenades", M203_GRENADE_MAX_CARRY},
   {"ammo_ARgrenades", "ARgrenades", M203_GRENADE_MAX_CARRY},
   {"ammo_buckshot", "buckshot", BUCKSHOT_MAX_CARRY},
   {"ammo_crossbow", "bolts", BOLT_MAX_CARRY},
   {"ammo_357", "357", _357_MAX_CARRY},
   {"ammo_rpgclip", "rockets", ROCKET_MAX_CARRY},
   {"ammo_egonclip", "uranium", URANIUM_MAX_CARRY},
   {"ammo_gaussclip", "uranium", URANIUM_MAX_CARRY},
   {"", 0, 0}};


LINK_ENTITY_TO_CLASS( bot, CBot );


inline edict_t *CREATE_FAKE_CLIENT( const char *netname )
{
   return (*g_engfuncs.pfnCreateFakeClient)( netname );
}

inline char *GET_INFOBUFFER( edict_t *e )
{
   return (*g_engfuncs.pfnGetInfoKeyBuffer)( e );
}

inline char *GET_INFO_KEY_VALUE( char *infobuffer, char *key )
{
   return (g_engfuncs.pfnInfoKeyValue( infobuffer, key ));
}

inline void SET_CLIENT_KEY_VALUE( int clientIndex, char *infobuffer,
                                  char *key, char *value )
{
   (*g_engfuncs.pfnSetClientKeyValue)( clientIndex, infobuffer, key, value );
}


void BotDebug( char *buffer )
{
   // print out debug messages to the HUD of all players
   // this allows you to print messages from bots to your display
   // as you are playing the game.  Use STRING(pev->netname) in
   // buffer to see the name of the bot.

   UTIL_ClientPrintAll( HUD_PRINTNOTIFY, buffer );
}


void BotCreate(edict_t *pEntity, const char *skin, const char *name,
               const char *skill)
{
   edict_t *BotEnt;
   int skill_level;
   char c_skill[2];
   int i;
   int index;
   BOOL found = FALSE;

   if ((skin == NULL) || (*skin == 0))
   {
      // pick a random skin
      index = RANDOM_LONG(0, 9);  // there are ten possible skins

      // check if this skin has already been used...
      while (skin_used[index] == TRUE)
      {
         index++;

         if (index == MAX_SKINS)
            index = 0;
      }

      skin_used[index] = TRUE;

      // check if all skins are used...
      for (i = 0; i < MAX_SKINS; i++)
      {
         if (skin_used[i] == FALSE)
            break;
      }

      // if all skins are used, reset used to FALSE for next selection
      if (i == MAX_SKINS)
      {
         for (i = 0; i < MAX_SKINS; i++)
            skin_used[i] = FALSE;
      }

      strcpy( botskin, bot_skins[index] );
   }
   else
   {
      strncpy( botskin, skin, 15);
      botskin[15] = 0;  // make sure botskin is null terminated
   }

   for (i = 0; botskin[i] != 0; i++)
      botskin[i] = tolower( botskin[i] );  // convert to all lowercase

   index = 0;

   while ((!found) && (index < MAX_SKINS))
   {
      if (strcmp(botskin, bot_skins[index]) == 0)
         found = TRUE;
      else
         index++;
   }

   if (found == FALSE)
   {
      char err_msg[80];

      sprintf( err_msg, "model \"%s\" is unknown.\n", botskin );
      UTIL_ClientPrintAll( HUD_PRINTNOTIFY, err_msg );
      UTIL_ClientPrintAll( HUD_PRINTNOTIFY,
         "use barney, gina, gman, gordon, helmet, hgrunt,\n");
      UTIL_ClientPrintAll( HUD_PRINTNOTIFY,
         "    recon, robo, scientist, or zombie\n");
      return;
   }

   if ((name != NULL) && (*name != 0))
   {
      strncpy( botname, name, 31 );
      botname[31] = 0;  // make sure botname is null terminated
   }
   else
   {
      strcpy( botname, bot_names[index] );
   }

   skill_level = 0;

   if ((skill != NULL) && (*skill != 0))
      sscanf( skill, "%d", &skill_level );
   else
      skill_level = f_botskill;

   if ((skill_level < 1) || (skill_level > 5))
      skill_level = f_botskill;

   skill_level--;  // make 0 based for array index (now 0-4)
   sprintf( c_skill, "%d", skill_level );

   BotEnt = CREATE_FAKE_CLIENT( botname );

   if (FNullEnt( BotEnt ))
      UTIL_ClientPrintAll( HUD_PRINTNOTIFY, "Max. Players reached.  Can't create bot!\n");
   else
   {
      char ptr[128];  // allocate space for message from ClientConnect
      CBot *BotClass;
      char *infobuffer;
      int clientIndex;

      sprintf(ptr, "Creating bot \"%s\" using model %s with skill=%d\n", botname, botskin, (int)skill_level+1);
      UTIL_ClientPrintAll( HUD_PRINTNOTIFY, ptr);

      BotClass = GetClassPtr( (CBot *) VARS(BotEnt) );
      infobuffer = GET_INFOBUFFER( BotClass->edict( ) );
      clientIndex = BotClass->entindex( );

      SET_CLIENT_KEY_VALUE( clientIndex, infobuffer, "model", botskin );
      SET_CLIENT_KEY_VALUE( clientIndex, infobuffer, "skill", c_skill );

      ClientConnect( BotClass->edict( ), botname, "127.0.0.1", ptr );
      DispatchSpawn( BotClass->edict( ) );
   }
}


void CBot::Spawn( )
{
   char c_skill[2];

   pev->classname    = MAKE_STRING( "player" );
   pev->health       = 100;
   pev->armorvalue   = 0;
   pev->takedamage   = DAMAGE_AIM;
   pev->solid        = SOLID_SLIDEBOX;
   pev->movetype     = MOVETYPE_WALK;
   pev->max_health   = pev->health;
   pev->flags        = FL_CLIENT | FL_FAKECLIENT;
   pev->air_finished = gpGlobals->time + 12;
   pev->dmg          = 2;     // initial water damage
   pev->effects      = 0;
   pev->deadflag     = DEAD_NO;
   pev->dmg_take     = 0;
   pev->dmg_save     = 0;

   m_bitsHUDDamage   = -1;
   m_bitsDamageType  = 0;
   m_afPhysicsFlags  = 0;
   m_fLongJump       = FALSE; // no longjump module. 
   m_iFOV            = 0;     // init field of view.
   m_iClientFOV      = -1;    // make sure fov reset is sent

   m_flNextDecalTime = 0;     // let this player decal as soon as he spawns.

   // wait a few seconds until user-defined message registrations are recieved by all clients
   m_flgeigerDelay   = gpGlobals->time + 2.0;
   
   m_flTimeStepSound = 0;
   m_iStepLeft       = 0;

   // some monsters use this to determine whether or not the player is looking at them.
   m_flFieldOfView   = 0.5;

   m_bloodColor      = BLOOD_COLOR_RED;
   m_flNextAttack    = gpGlobals->time;
   StartSneaking( );

   m_iFlashBattery   = 99;
   m_flFlashLightTime = 1;    // force first message

   // dont let uninitialized value here hurt the player
   m_flFallVelocity  = 0;

   g_pGameRules->SetDefaultPlayerTeam( this );
   g_pGameRules->GetPlayerSpawnSpot( this );

   SET_MODEL( ENT( pev ), "models/player.mdl" );

   g_ulModelIndexPlayer = pev->modelindex;

   pev->sequence = LookupActivity( ACT_IDLE );

   if (FBitSet( pev->flags, FL_DUCKING )) 
      UTIL_SetSize( pev, VEC_DUCK_HULL_MIN, VEC_DUCK_HULL_MAX );
   else
      UTIL_SetSize( pev, VEC_HULL_MIN, VEC_HULL_MAX );

   pev->view_ofs = VEC_VIEW;

   Precache( );

   m_HackedGunPos = Vector( 0, 32, 0 );

   if (m_iPlayerSound == SOUNDLIST_EMPTY)
   {
      ALERT ( at_console, "Couldn't alloc player sound slot!\n" );
   }

   m_fNoPlayerSound = FALSE;  // normal sound behavior.

   m_pLastItem = NULL;
   m_fInitHUD = TRUE;
   m_iClientHideHUD = -1;     // force this to be recalculated
   m_fWeapon = FALSE;
   m_pClientActiveItem = NULL;
   m_iClientBattery = -1;

   // reset all ammo values to 0
   for ( int i = 0; i < MAX_AMMO_SLOTS; i++ )
   {
      // client ammo values also have to be reset
      // (the death hud clear messages does on the client side)
      m_rgAmmo[i] = 0;
      m_rgAmmoLast[i] = 0;
   }

   m_lastx = m_lasty = 0;

   g_pGameRules->PlayerSpawn( this );

   SetThink( BotThink );
   pev->nextthink = gpGlobals->time + 0.1;

   pev->ideal_yaw = pev->v_angle.y;
   pev->yaw_speed = BOT_YAW_SPEED;

   // bot starts out in "paused" state since it hasn't moved yet...
   bot_was_paused = TRUE;
   v_prev_origin = pev->origin;

   f_shoot_time = 0;

   // get bot's skill level (0=good, 4=bad)
   strcpy(c_skill, GET_INFO_KEY_VALUE(GET_INFOBUFFER(edict( )), "skill") );
   sscanf(c_skill, "%d", &bot_skill);

   f_max_speed = CVAR_GET_FLOAT("sv_maxspeed");
//   f_max_speed = f_max_speed * speed_adjust[bot_skill];

   f_speed_check_time = gpGlobals->time + 1.0;

   ladder_dir = 0;

   // pick a wander direction (50% of the time to the left, 50% to the right)
   if (RANDOM_LONG(1, 100) <= 50)
      wander_dir = WANDER_LEFT;
   else
      wander_dir = WANDER_RIGHT;

   f_pause_time = 0;
   f_find_item = 0;

   strcpy(model_name, GET_INFO_KEY_VALUE(GET_INFOBUFFER(edict( )), "model") );

   bot_model = 0;
   if ((strcmp( model_name, "hgrunt" ) == 0) ||
      (strcmp( model_name, "recon" ) == 0))
      bot_model = MODEL_HGRUNT;
   else if (strcmp( model_name, "barney") == 0)
      bot_model = MODEL_BARNEY;
   else if (strcmp( model_name, "scientist") == 0)
      bot_model = MODEL_SCIENTIST;

   for (i = 0; i < 5; i++)
      PRECACHE_SOUND (hgrunt_sounds[i]);

   for (i = 0; i < 5; i++)
      PRECACHE_SOUND (scientist_sounds[i]);

   for (i = 0; i < 5; i++)
      PRECACHE_SOUND (barney_sounds[i]);

   f_pain_time = gpGlobals->time + 5.0;

   b_use_health_station = FALSE;
   b_use_HEV_station = FALSE;
   b_use_button = FALSE;
   f_use_button_time = 0;
   b_lift_moving = FALSE;
   f_fire_gauss = -1;  // -1 means not charging gauss gun

   b_see_tripmine = FALSE;
   b_shoot_tripmine = FALSE;

   pBotEnemy = NULL;
   pBotPickupItem = NULL;
}


void CBot::Killed( entvars_t *pevAttacker, int iGib )
{
   CSound *pSound;

   g_pGameRules->PlayerKilled( this, pevAttacker, g_pevLastInflictor );

   if (m_pTank != NULL)
   {
      m_pTank->Use( this, this, USE_OFF, 0 );
      m_pTank = NULL;
   }

   // this client isn't going to be thinking for a while, so reset the sound
   // until they respawn
   pSound = CSoundEnt::SoundPointerForIndex( CSoundEnt::ClientSoundIndex(edict( )) );
   {
      if (pSound)
      {
         pSound->Reset( );
      }
   }

   SetAnimation( PLAYER_DIE );
   
   pev->modelindex = g_ulModelIndexPlayer;    // don't use eyes

#if !defined(DUCKFIX)
   pev->view_ofs      = Vector( 0, 0, -8 );
#endif
   pev->deadflag      = DEAD_DYING;
   pev->solid         = SOLID_NOT;
   pev->movetype      = MOVETYPE_TOSS;
   ClearBits( pev->flags, FL_ONGROUND );
   if (pev->velocity.z < 10)
      pev->velocity.z += RANDOM_FLOAT( 0, 300 );

   // clear out the suit message cache so we don't keep chattering
   SetSuitUpdate( NULL, FALSE, 0 );

   // send "health" update message to zero
   m_iClientHealth = 0;
   MESSAGE_BEGIN( MSG_ONE, gmsgHealth, NULL, pev );
      WRITE_BYTE( m_iClientHealth );
   MESSAGE_END( );

   // Tell Ammo Hud that the player is dead
   MESSAGE_BEGIN( MSG_ONE, gmsgCurWeapon, NULL, pev );
      WRITE_BYTE( 0 );
      WRITE_BYTE( 0xFF );
      WRITE_BYTE( 0xFF );
   MESSAGE_END( );

   // reset FOV
   m_iFOV = m_iClientFOV = 0;

   MESSAGE_BEGIN( MSG_ONE, gmsgSetFOV, NULL, pev );
      WRITE_BYTE( 0 );
   MESSAGE_END( );

   // UNDONE: Put this in, but add FFADE_PERMANENT and make fade time 8.8 instead of 4.12
   // UTIL_ScreenFade( edict( ), Vector( 128, 0, 0 ), 6, 15, 255, FFADE_OUT | FFADE_MODULATE );

   if (( pev->health < -40 && iGib != GIB_NEVER ) || iGib == GIB_ALWAYS)
   {
      GibMonster( );   // This clears pev->model
      pev->effects |= EF_NODRAW;

// TAKE THIS OUT!!! THIS HAPPENS SOMETIMES WHEN HEALTH IS < -40, THEN THE
// THINK FUNCTION DOESN'T EVER GET SET TO PlayerDeathThink
// (should we REALLY be doing this???)
//    return;
   }

   DeathSound( );
   
   pev->angles.x = 0;
   pev->angles.z = 0;

   SetThink( PlayerDeathThink );
   pev->nextthink = gpGlobals->time + 0.1;
}


void CBot::PlayerDeathThink( void )
{
   float flForward;

   pev->nextthink = gpGlobals->time + 0.1;

   if (FBitSet( pev->flags, FL_ONGROUND ))
   {
      flForward = pev->velocity.Length( ) - 20;
      if (flForward <= 0)
         pev->velocity = g_vecZero;
      else    
         pev->velocity = flForward * pev->velocity.Normalize( );
   }

   if (HasWeapons( ))
   {
      // we drop the guns here because weapons that have an area effect and can kill their user
      // will sometimes crash coming back from CBasePlayer::Killed() if they kill their owner because the
      // player class sometimes is freed. It's safer to manipulate the weapons once we know
      // we aren't calling into any of their code anymore through the player pointer.

      PackDeadPlayerItems( );
   }

   DROP_TO_FLOOR ( ENT(pev) );  // put the body on the ground

   if (pev->modelindex && (!m_fSequenceFinished) && (pev->deadflag == DEAD_DYING))
   {
      StudioFrameAdvance( );

      m_iRespawnFrames++;
      if (m_iRespawnFrames < 60)  // animations should be no longer than this
         return;
   }

   StopAnimation( );

   if (pev->deadflag == DEAD_DYING)
   {
      pev->deadflag = DEAD_DEAD;
   }

   pev->effects |= EF_NOINTERP;
   pev->framerate = 0.0;

   if (pev->deadflag == DEAD_DEAD)
   {
      if (g_pGameRules->FPlayerCanRespawn( this ))
      {
         m_fDeadTime = gpGlobals->time;
         pev->deadflag = DEAD_RESPAWNABLE;
      }
      
      return;
   }

   // check if time to respawn...
   if (gpGlobals->time > (m_fDeadTime + 5.0))
   {
      pev->button = 0;
      m_iRespawnFrames = 0;

      //ALERT( at_console, "Respawn\n" );

      respawn( pev, !(m_afPhysicsFlags & PFLAG_OBSERVER) );

      pev->nextthink = -1;
   }
}


int CBot::TakeDamage( entvars_t *pevInflictor, entvars_t *pevAttacker,
                      float flDamage, int bitsDamageType )
{
   CBaseEntity *pAttacker = CBaseEntity::Instance(pevAttacker);
   char sound[40];
   int ret_damage;

   // do the damage first...
   ret_damage = CBasePlayer::TakeDamage( pevInflictor, pevAttacker, flDamage,
                                         bitsDamageType );

   // if the bot doesn't have an enemy and someone is shooting at it then
   // turn in the attacker's direction...
   if (pBotEnemy == NULL)
   {
      // face the attacker...
      Vector v_enemy = pAttacker->pev->origin - pev->origin;
      Vector bot_angles = UTIL_VecToAngles( v_enemy );

      pev->ideal_yaw = bot_angles.y;

      // check for wrap around of angle...
      if (pev->ideal_yaw > 180)
         pev->ideal_yaw -= 360;
      if (pev->ideal_yaw < -180)
         pev->ideal_yaw += 360;

      // stop using health or HEV stations...
      b_use_health_station = FALSE;
      b_use_HEV_station = FALSE;
   }

   // check if bot model is known, attacker is not a bot,
   // time for pain sound, and bot has some of health left...

   if ((bot_model != 0) && (pAttacker->IsNetClient()) &&
       (f_pain_time <= gpGlobals->time) && (pev->health > 0))
   {
      float distance = (pAttacker->pev->origin - pev->origin).Length( );

      // check if the distance to attacker is close enough (otherwise
      // the attacker is too far away to hear the pain sounds)

      if (distance <= 400)
      {
         // speak pain sounds about 50% of the time
         if (RANDOM_LONG(1, 100) <= 50)
         {
            f_pain_time = gpGlobals->time + 5.0;

            if (bot_model == MODEL_HGRUNT)
               sprintf( sound, hgrunt_sounds[RANDOM_LONG(0,4)] );
            else if (bot_model == MODEL_BARNEY)
               sprintf( sound, barney_sounds[RANDOM_LONG(0,4)] );
            else if (bot_model == MODEL_SCIENTIST)
               sprintf( sound, scientist_sounds[RANDOM_LONG(0,4)] );

            EMIT_SOUND(ENT(pevAttacker), CHAN_VOICE, sound,
                       RANDOM_FLOAT(0.9, 1.0), ATTN_NORM);
         }
      }
   }

   return ret_damage;
}


int CBot::BotInFieldOfView(Vector dest)
{
   // find angles from source to destination...
   Vector entity_angles = UTIL_VecToAngles( dest );

   // make yaw angle 0 to 360 degrees if negative...
   if (entity_angles.y < 0)
      entity_angles.y += 360;

   // get bot's current view angle...
   float view_angle = pev->v_angle.y;

   // make view angle 0 to 360 degrees if negative...
   if (view_angle < 0)
      view_angle += 360;

   // return the absolute value of angle to destination entity
   // zero degrees means straight ahead,  45 degrees to the left or
   // 45 degrees to the right is the limit of the normal view angle

   return abs((int)view_angle - (int)entity_angles.y);
}


BOOL CBot::BotEntityIsVisible( Vector dest )
{
   TraceResult tr;

   // trace a line from bot's eyes to destination...
   UTIL_TraceLine( pev->origin + pev->view_ofs, dest, ignore_monsters,
                   ENT(pev), &tr );

   // check if line of sight to object is not blocked (i.e. visible)
   if (tr.flFraction >= 1.0)
      return TRUE;
   else
      return FALSE;
}


float CBot::BotChangeYaw( float speed )
{
   float ideal;
   float current;
   float current_180;  // current +/- 180 degrees
   float diff;

   // turn from the current v_angle yaw to the ideal_yaw by selecting
   // the quickest way to turn to face that direction
   
   current = pev->v_angle.y;
   ideal = pev->ideal_yaw;

   // find the difference in the current and ideal angle
   diff = abs(current - ideal);

   // check if the bot is already facing the ideal_yaw direction...
   if (diff <= 1)
      return diff;  // return number of degrees turned

   // check if difference is less than the max degrees per turn
   if (diff < speed)
      speed = diff;  // just need to turn a little bit (less than max)

   // here we have four cases, both angle positive, one positive and
   // the other negative, one negative and the other positive, or
   // both negative.  handle each case separately...

   if ((current >= 0) && (ideal >= 0))  // both positive
   {
      if (current > ideal)
         current -= speed;
      else
         current += speed;
   }
   else if ((current >= 0) && (ideal < 0))
   {
      current_180 = current - 180;

      if (current_180 > ideal)
         current += speed;
      else
         current -= speed;
   }
   else if ((current < 0) && (ideal >= 0))
   {
      current_180 = current + 180;
      if (current_180 > ideal)
         current += speed;
      else
         current -= speed;
   }
   else  // (current < 0) && (ideal < 0)  both negative
   {
      if (current > ideal)
         current -= speed;
      else
         current += speed;
   }

   // check for wrap around of angle...
   if (current > 180)
      current -= 360;
   if (current < -180)
      current += 360;

   pev->v_angle.y = current;

   return speed;  // return number of degrees turned
}


void CBot::BotOnLadder( float moved_distance )
{
   // moves the bot up or down a ladder.  if the bot can't move
   // (i.e. get's stuck with someone else on ladder), the bot will
   // change directions and go the other way on the ladder.

   if (ladder_dir == LADDER_UP)  // is the bot currently going up?
   {
      pev->v_angle.x = -60;  // look upwards

      // check if the bot hasn't moved much since the last location...
      if (moved_distance <= 1)
      {
         // the bot must be stuck, change directions...

         pev->v_angle.x = 60;  // look downwards
         ladder_dir = LADDER_DOWN;
      }
   }
   else if (ladder_dir == LADDER_DOWN)  // is the bot currently going down?
   {
      pev->v_angle.x = 60;  // look downwards

      // check if the bot hasn't moved much since the last location...
      if (moved_distance <= 1)
      {
         // the bot must be stuck, change directions...

         pev->v_angle.x = -60;  // look upwards
         ladder_dir = LADDER_UP;
      }
   }
   else  // the bot hasn't picked a direction yet, try going up...
   {
      pev->v_angle.x = -60;  // look upwards
      ladder_dir = LADDER_UP;
   }

   // move forward (i.e. in the direction the bot is looking, up or down)
   pev->button |= IN_FORWARD;
}


void CBot::BotUnderWater( void )
{
   // handle movements under water.  right now, just try to keep from
   // drowning by swimming up towards the surface and look to see if
   // there is a surface the bot can jump up onto to get out of the
   // water.  bots DON'T like water!

   Vector v_src, v_forward;
   TraceResult tr;
   int contents;

   // swim up towards the surface
   pev->v_angle.x = -60;  // look upwards

   // move forward (i.e. in the direction the bot is looking, up or down)
   pev->button |= IN_FORWARD;

   // set gpGlobals angles based on current view angle (for TraceLine)
   UTIL_MakeVectors( pev->v_angle );

   // look from eye position straight forward (remember: the bot is looking
   // upwards at a 60 degree angle so TraceLine will go out and up...

   v_src = pev->origin + pev->view_ofs;  // EyePosition()
   v_forward = v_src + gpGlobals->v_forward * 90;

   // trace from the bot's eyes straight forward...
   UTIL_TraceLine( v_src, v_forward, dont_ignore_monsters, ENT(pev), &tr);

   // check if the trace didn't hit anything (i.e. nothing in the way)...
   if (tr.flFraction >= 1.0)
   {
      // find out what the contents is of the end of the trace...
      contents = UTIL_PointContents( tr.vecEndPos );

      // check if the trace endpoint is in open space...
      if (contents == CONTENTS_EMPTY)
      {
         // ok so far, we are at the surface of the water, continue...

         v_src = tr.vecEndPos;
         v_forward = v_src;
         v_forward.z -= 90;

         // trace from the previous end point straight down...
         UTIL_TraceLine( v_src, v_forward, dont_ignore_monsters,
                         ENT(pev), &tr);

         // check if the trace hit something...
         if (tr.flFraction < 1.0)
         {
            contents = UTIL_PointContents( tr.vecEndPos );

            // if contents isn't water then assume it's land, jump!
            if (contents != CONTENTS_WATER)
            {
               pev->button |= IN_JUMP;
            }
         }
      }
   }
}


CBaseEntity * CBot::BotFindEnemy( void )
{
   Vector vecEnd;
   static BOOL flag=TRUE;

   if (pBotEnemy != NULL)  // does the bot already have an enemy?
   {
      vecEnd = pBotEnemy->pev->origin + pBotEnemy->pev->view_ofs;

      if (!pBotEnemy->IsAlive( ))
      {
         // the enemy is dead, jump for joy about 10% of the time
         if (RANDOM_LONG(1, 100) <= 10)
            pev->button |= IN_JUMP;
      }
      else if (BotEntityIsVisible( vecEnd ) && BotInFieldOfView( vecEnd ))
      {
         // if enemy is still visible and in field of view, keep it

         // face the enemy
         Vector v_enemy = pBotEnemy->pev->origin - pev->origin;
         Vector bot_angles = UTIL_VecToAngles( v_enemy );

         pev->ideal_yaw = bot_angles.y;

         // check for wrap around of angle...
         if (pev->ideal_yaw > 180)
            pev->ideal_yaw -= 360;
         if (pev->ideal_yaw < -180)
            pev->ideal_yaw += 360;

         return (pBotEnemy);
      }
   }

   int i;
   float nearestdistance = 1000;
   CBaseEntity *pNewEnemy = NULL;

   // search the world for players...
   for (i = 1; i <= gpGlobals->maxClients; i++)
   {
      CBaseEntity *pPlayer = UTIL_PlayerByIndex( i );

      // skip invalid players and skip self (i.e. this bot)
      if ((!pPlayer) || (pPlayer == this))
         continue;

      // skip this player if not alive (i.e. dead or dying)
      if (pPlayer->pev->deadflag != DEAD_NO)
         continue;

      // skip players that are netclients (i.e. not bot, observer mode)
      if (pPlayer->IsNetClient() && f_Observer)
         continue;

// BigGuy - start
      // is team play enabled?
      if (g_pGameRules->IsTeamplay())
      {
         // don't target your teammates if team names match...
         if (UTIL_TeamsMatch(g_pGameRules->GetTeamID(this),
                             g_pGameRules->GetTeamID(pPlayer)))
            continue;
      }
// BigGuy - end

      vecEnd = pPlayer->pev->origin + pPlayer->pev->view_ofs;

      // see if bot can see the player (within bot's field of view)...
      if (BotEntityIsVisible( vecEnd ) && BotInFieldOfView( vecEnd ))
      {
         float distance = (pPlayer->pev->origin - pev->origin).Length( );
         if (distance < nearestdistance)
         {
            nearestdistance = distance;
            pNewEnemy = pPlayer;
         }
      }
   }

   if (pNewEnemy)
   {
      // face the enemy
      Vector v_enemy = pNewEnemy->pev->origin - pev->origin;
      Vector bot_angles = UTIL_VecToAngles( v_enemy );

      pev->ideal_yaw = bot_angles.y;

      // check for wrap around of angle...
      if (pev->ideal_yaw > 180)
         pev->ideal_yaw -= 360;
      if (pev->ideal_yaw < -180)
         pev->ideal_yaw += 360;
   }

   return (pNewEnemy);
}


Vector CBot::BotBodyTarget( CBaseEntity *pBotEnemy )
{
   Vector target;

   switch (bot_skill)
   {
      case 0:
         // VERY GOOD, same as from CBasePlayer::BodyTarget (in player.h)
         target = pBotEnemy->Center( ) + pBotEnemy->pev->view_ofs * RANDOM_FLOAT( 0.5, 1.1 );
         break;
      case 1:
         // GOOD, offset a little for x, y, and z
         target = pBotEnemy->Center( ) + pBotEnemy->pev->view_ofs +
                  Vector( RANDOM_LONG(-5, 5), RANDOM_LONG(-5, 5), RANDOM_LONG(-9, 9));
         break;
      case 2:
         // FAIR, offset somewhat for x, y, and z
         target = pBotEnemy->Center( ) + pBotEnemy->pev->view_ofs +
                  Vector( RANDOM_LONG(-9, 9), RANDOM_LONG(-9, 9), RANDOM_LONG(-15, 15));
         break;
      case 3:
         // POOR, offset for x, y, and z
         target = pBotEnemy->Center( ) + pBotEnemy->pev->view_ofs +
                  Vector( RANDOM_LONG(-16, 16), RANDOM_LONG(-16, 16), RANDOM_LONG(-20, 20));
         break;
      case 4:
         // BAD, offset lots for x, y, and z
         target = pBotEnemy->Center( ) + pBotEnemy->pev->view_ofs +
                  Vector( RANDOM_LONG(-20, 20), RANDOM_LONG(-20, 20), RANDOM_LONG(-27, 27));
         break;
   }

   return target;
}


// specifing a weapon_choice allows you to choose the weapon the bot will
// use (assuming enough ammo exists for that weapon)
// BotFireWeapon will return TRUE if weapon was fired, FALSE otherwise
// primary is used to indicate whether you want primary or secondary fire
// if you have specified a weapon using weapon_choice

BOOL CBot::BotFireWeapon( Vector v_enemy, int weapon_choice, BOOL primary )
{
   CBasePlayerItem *weapon_ptr[MAX_WEAPONS];  // pointer array to weapons
   CBasePlayerItem *new_weapon;
   int primary_ammo[MAX_WEAPONS];
   int secondary_ammo[MAX_WEAPONS];
   int i;

   // check if bot can't switch weapons right now...
   if (!m_pActiveItem->CanHolster( ))
      return FALSE;

   // initialize the elements of the weapons arrays...
   for (i = 0; i < MAX_WEAPONS; i++)
   {
      weapon_ptr[i] = NULL;
      primary_ammo[i] = 0;
      secondary_ammo[i] = 0;
   }

   // find out which weapons the bot is carrying...
   for (i = 0; i < MAX_ITEM_TYPES; i++)
   {
      CBasePlayerItem *pItem = NULL;

      if (m_rgpPlayerItems[i])
      {
         pItem = m_rgpPlayerItems[i];
         while (pItem)
         {
            weapon_ptr[pItem->m_iId] = pItem;  // store pointer to item

            pItem = pItem->m_pNext;
         }
      }
   }

   // find out how much ammo of each type the bot is carrying...
   for (i = 0; i < MAX_AMMO_SLOTS; i++)
   {
      if (!CBasePlayerItem::AmmoInfoArray[i].pszName)
         continue;

      if (strcmp("9mm", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
      {
         primary_ammo[WEAPON_GLOCK] = m_rgAmmo[i];
         primary_ammo[WEAPON_MP5] = m_rgAmmo[i];
      }
      else if (strcmp("357", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
      {
         primary_ammo[WEAPON_PYTHON] = m_rgAmmo[i];
      }
      else if (strcmp("ARgrenades", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
         secondary_ammo[WEAPON_MP5] = m_rgAmmo[i];
      else if (strcmp("bolts", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
      {
         primary_ammo[WEAPON_CROSSBOW] = m_rgAmmo[i];
      }
      else if (stricmp("buckshot", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
      {
         primary_ammo[WEAPON_SHOTGUN] = m_rgAmmo[i];
      }
      else if (stricmp("rockets", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
         primary_ammo[WEAPON_RPG] = m_rgAmmo[i];
      else if (strcmp("uranium", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
      {
         primary_ammo[WEAPON_GAUSS] = m_rgAmmo[i];
         primary_ammo[WEAPON_EGON] = m_rgAmmo[i];
      }
      else if (stricmp("Hornets", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
         primary_ammo[WEAPON_HORNETGUN] = m_rgAmmo[i];
      else if (stricmp("Hand Grenade", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
         primary_ammo[WEAPON_HANDGRENADE] = m_rgAmmo[i];
      else if (stricmp("Trip Mine", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
         primary_ammo[WEAPON_TRIPMINE] = m_rgAmmo[i];
      else if (stricmp("Satchel Charge", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
         primary_ammo[WEAPON_SATCHEL] = m_rgAmmo[i];
      else if (stricmp("Snarks", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
         primary_ammo[WEAPON_SNARK] = m_rgAmmo[i];
   }

   float distance = v_enemy.Length( );  // how far away is the enemy?

   // if bot is carrying the crowbar...
   if (pev->weapons & (1<<WEAPON_CROWBAR))
   {
      // if close to enemy, and skill level is 1, 2 or 3, use the crowbar
      if (((distance <= 40) && (bot_skill <= 2) && (weapon_choice == 0)) ||
          (weapon_choice == WEAPON_CROWBAR))
      {
         new_weapon = weapon_ptr[WEAPON_CROWBAR];

         // check if the bot isn't already using this item...
         if (m_pActiveItem != new_weapon)
            SelectItem("weapon_crowbar");  // select the crowbar

         pev->button |= IN_ATTACK;  // use primary attack (whack! whack!)

         // set next time to "shoot"
         f_shoot_time = gpGlobals->time + 0.3 + 
            RANDOM_FLOAT(primary_fire_delay[WEAPON_CROWBAR][bot_skill][0],
                         primary_fire_delay[WEAPON_CROWBAR][bot_skill][1]);
         return TRUE;
      }
   }

   // if bot is carrying any hand grenades...
   if (pev->weapons & (1<<WEAPON_HANDGRENADE))
   {
      long use_grenade = RANDOM_LONG(1,100);

      // use hand grenades about 30% of the time...
      if (((distance > 250) && (distance < 750) &&
           (weapon_choice == 0) && (use_grenade <= 30)) ||
          (weapon_choice == WEAPON_HANDGRENADE))
      {
// BigGuy - start
         new_weapon = weapon_ptr[WEAPON_HANDGRENADE];

         // check if the bot isn't already using this item...
         if (m_pActiveItem != new_weapon)
            SelectItem("weapon_handgrenade");  // select the hand grenades

         pev->button |= IN_ATTACK;  // use primary attack (boom!)

         // set next time to "shoot"
         f_shoot_time = gpGlobals->time + 0.1 + 
            RANDOM_FLOAT(primary_fire_delay[WEAPON_HANDGRENADE][bot_skill][0],
                         primary_fire_delay[WEAPON_HANDGRENADE][bot_skill][1]);
         return TRUE;
// BigGuy - end
      }
   }

   // if bot is carrying any snarks (can't use underwater)...
   if ((pev->weapons & (1<<WEAPON_SNARK)) && (pev->waterlevel != 3))
   {
      long use_snark = RANDOM_LONG(1,100);

      // use snarks about 50% of the time...
      if (((distance > 150) && (distance < 500) &&
           (weapon_choice == 0) && (use_snark <= 50)) ||
          (weapon_choice == WEAPON_SNARK))
      {
// BigGuy - start
         new_weapon = weapon_ptr[WEAPON_SNARK];

         // check if the bot isn't already using this item...
         if (m_pActiveItem != new_weapon)
            SelectItem("weapon_snark");  // select the "squeak grenades"

         pev->button |= IN_ATTACK;  // use primary attack (eek! eek!)

         // set next time to "shoot"
         f_shoot_time = gpGlobals->time + 0.1 + 
            RANDOM_FLOAT(primary_fire_delay[WEAPON_SNARK][bot_skill][0],
                         primary_fire_delay[WEAPON_SNARK][bot_skill][1]);
         return TRUE;
// BigGuy - end
      }
   }

   // if the bot is carrying the egon gun (can't use underwater)...
   if ((pev->weapons & (1<<WEAPON_EGON)) && (pev->waterlevel != 3))
   {
      if ((weapon_choice == 0) || (weapon_choice == WEAPON_EGON))
      {
         new_weapon = weapon_ptr[WEAPON_EGON];

         // check if the bot has any ammo left for this weapon...
         if (primary_ammo[WEAPON_EGON] > 0)
         {
            // check if the bot isn't already using this item...
            if (m_pActiveItem != new_weapon)
               SelectItem("weapon_egon");  // select the egon gun

            pev->button |= IN_ATTACK;  // use primary attack (bang! bang!)

            // set next time to shoot
            f_shoot_time = gpGlobals->time;

            return TRUE;
         }
      }
   }

   // if the bot is carrying the gauss gun (can't use underwater)...
   if ((pev->weapons & (1<<WEAPON_GAUSS)) && (pev->waterlevel != 3))
   {
      if ((weapon_choice == 0) || (weapon_choice == WEAPON_GAUSS))
      {
         new_weapon = weapon_ptr[WEAPON_GAUSS];

         // check if the bot has any ammo left for this weapon...
         if (primary_ammo[WEAPON_GAUSS] > 1)
         {
            // check if the bot isn't already using this item...
            if (m_pActiveItem != new_weapon)
               SelectItem("weapon_gauss");  // select the gauss gun

            long use_secondary = RANDOM_LONG(1,100);

            // are we charging the gauss gun?
            if (f_fire_gauss > 0)
            {
               // is it time to fire the charged gauss gun?
               if (f_fire_gauss >= gpGlobals->time)
               {
                  // we DON'T set pev->button here to release the secondary
                  // fire button which will fire the charged gauss gun

                  f_fire_gauss = -1;  // -1 means not charging gauss gun

                  // set next time to shoot
                  f_shoot_time = gpGlobals->time + 1.0 +
                     RANDOM_FLOAT(secondary_fire_delay[WEAPON_GAUSS][bot_skill][0],
                                  secondary_fire_delay[WEAPON_GAUSS][bot_skill][1]);
               }
               else
               {
                  pev->button |= IN_ATTACK2;  // charge the gauss gun
                  f_shoot_time = gpGlobals->time;  // keep charging
               }
            }
            else if ((use_secondary <= 20) &&
                     (primary_ammo[WEAPON_GAUSS] >= 10))
            {
               // release secondary fire in 0.5 seconds...
               f_fire_gauss = gpGlobals->time + 0.5;

               pev->button |= IN_ATTACK2;  // charge the gauss gun
               f_shoot_time = gpGlobals->time; // keep charging
            }
            else
            {
               pev->button |= IN_ATTACK;  // use primary attack (bang! bang!)

               // set next time to shoot
               f_shoot_time = gpGlobals->time + 0.2 +
                  RANDOM_FLOAT(primary_fire_delay[WEAPON_GAUSS][bot_skill][0],
                               primary_fire_delay[WEAPON_GAUSS][bot_skill][1]);
            }

            return TRUE;
         }
      }
   }

   // if the bot is carrying the shotgun (can't use underwater)...
   if ((pev->weapons & (1<<WEAPON_SHOTGUN)) && (pev->waterlevel != 3))
   {
      // if close enough for good shotgun blasts...
      if (((distance > 30) && (distance < 150) && (weapon_choice == 0)) ||
          (weapon_choice == WEAPON_SHOTGUN))
      {
         new_weapon = weapon_ptr[WEAPON_SHOTGUN];

         // check if the bot has any ammo left for this weapon...
         if (primary_ammo[WEAPON_SHOTGUN] > 0)
         {
            // check if the bot isn't already using this item...
            if (m_pActiveItem != new_weapon)
               SelectItem("weapon_shotgun");  // select the shotgun

            long use_secondary = RANDOM_LONG(1,100);

            // use secondary attack about 30% of the time
            if ((use_secondary <= 30) && (primary_ammo[WEAPON_SHOTGUN] >= 2))
            {
// BigGuy - start
               pev->button |= IN_ATTACK2;  // use secondary attack (bang! bang!)

               // set next time to shoot
               f_shoot_time = gpGlobals->time + 1.5 +
                  RANDOM_FLOAT(secondary_fire_delay[WEAPON_SHOTGUN][bot_skill][0],
                               secondary_fire_delay[WEAPON_SHOTGUN][bot_skill][1]);
            }
// BigGuy - end
            else
            {
               pev->button |= IN_ATTACK;  // use primary attack (bang! bang!)

               // set next time to shoot
               f_shoot_time = gpGlobals->time + 0.75 +
                  RANDOM_FLOAT(primary_fire_delay[WEAPON_SHOTGUN][bot_skill][0],
                               primary_fire_delay[WEAPON_SHOTGUN][bot_skill][1]);
            }

            return TRUE;
         }
      }
   }

   // if the bot is carrying the 357/PYTHON, (can't use underwater)...
   if ((pev->weapons & (1<<WEAPON_PYTHON)) && (pev->waterlevel != 3))
   {
      // if close enough for 357 shot...
      if (((distance > 30) && (distance < 700) && (weapon_choice == 0)) ||
          (weapon_choice == WEAPON_PYTHON))
      {
         new_weapon = weapon_ptr[WEAPON_PYTHON];

         // check if the bot has any ammo left for this weapon...
         if (primary_ammo[WEAPON_PYTHON] > 0)
         {
            // check if the bot isn't already using this item...
            if (m_pActiveItem != new_weapon)
               SelectItem("weapon_357");  // select the 357 python

            pev->button |= IN_ATTACK;  // use primary attack (bang! bang!)

            // set next time to shoot
            f_shoot_time = gpGlobals->time + 0.75 +
               RANDOM_FLOAT(primary_fire_delay[WEAPON_PYTHON][bot_skill][0],
                            primary_fire_delay[WEAPON_PYTHON][bot_skill][1]);

            return TRUE;
         }
      }
   }

   // if the bot is carrying the hornet gun...
   if (pev->weapons & (1<<WEAPON_HORNETGUN))
   {
      // if close enough for hornet gun...
      if (((distance > 30) && (distance < 1000) && (weapon_choice == 0)) ||
          (weapon_choice == WEAPON_HORNETGUN))
      {
         new_weapon = weapon_ptr[WEAPON_HORNETGUN];

         // check if the bot has any ammo left for this weapon...
         if (primary_ammo[WEAPON_HORNETGUN] > 0)
         {
            // check if the bot isn't already using this item...
            if (m_pActiveItem != new_weapon)
               SelectItem("weapon_hornetgun");  // select the hornet gun

            long use_secondary = RANDOM_LONG(1,100);

            // use secondary attack about 50% of the time (if fully reloaded)
            if ((use_secondary <= 50) &&
                (primary_ammo[WEAPON_HORNETGUN] >= HORNET_MAX_CARRY))
            {
// BigGuy - start
               pev->button |= IN_ATTACK2;  // use secondary attack (buzz! buzz!)

               // set next time to shoot
               f_shoot_time = gpGlobals->time + 0.1 +
                  RANDOM_FLOAT(secondary_fire_delay[WEAPON_HORNETGUN][bot_skill][0],
                               secondary_fire_delay[WEAPON_HORNETGUN][bot_skill][1]);
// BigGuy - end
            }
            else
            {
               pev->button |= IN_ATTACK;  // use primary attack (buzz! buzz!)

               // set next time to shoot
               f_shoot_time = gpGlobals->time + 0.25 +
                  RANDOM_FLOAT(primary_fire_delay[WEAPON_HORNETGUN][bot_skill][0],
                               primary_fire_delay[WEAPON_HORNETGUN][bot_skill][1]);
            }

            return TRUE;
         }
      }
   }

   // if the bot is carrying the MP5 (can't use underwater)...
   if ((pev->weapons & (1<<WEAPON_MP5)) && (pev->waterlevel != 3))
   {
      long use_secondary = RANDOM_LONG(1,100);

      // use secondary attack about 10% of the time...
      if (((distance > 300) && (distance < 600) &&
           (weapon_choice == 0) && (use_secondary <= 10)) ||
          ((weapon_choice == WEAPON_MP5) && (primary == FALSE)))
      {
         // at some point we need to fire upwards in the air slightly
         // for long distance kills.  for right now, just fire the
         // grenade at the poor sucker.

// BigGuy - start
         new_weapon = weapon_ptr[WEAPON_MP5];

         // check if the bot has any ammo left for this weapon...
         if (secondary_ammo[WEAPON_MP5] > 0)
         {
            // check if the bot isn't already using this item...
            if (m_pActiveItem != new_weapon)
               SelectItem("weapon_9mmAR");  // select the 9mmAR (MP5)

            pev->button |= IN_ATTACK2;  // use secodnary attack (boom!)

            // set next time to shoot
            f_shoot_time = gpGlobals->time + 1.0 +
               RANDOM_FLOAT(secondary_fire_delay[WEAPON_MP5][bot_skill][0],
                            secondary_fire_delay[WEAPON_MP5][bot_skill][1]);

            return TRUE;
         }
// BigGuy - end
      }

      // if close enough for good MP5 shot...
      if (((distance < 250) && (weapon_choice == 0)) ||
          (weapon_choice == WEAPON_MP5))
      {
         new_weapon = weapon_ptr[WEAPON_MP5];

         // check if the bot has any ammo left for this weapon...
         if (primary_ammo[WEAPON_MP5] > 0)
         {
            // check if the bot isn't already using this item...
            if (m_pActiveItem != new_weapon)
               SelectItem("weapon_9mmAR");  // select the 9mmAR (MP5)

            pev->button |= IN_ATTACK;  // use primary attack (bang! bang!)

            // set next time to shoot
            f_shoot_time = gpGlobals->time + 0.1 +
               RANDOM_FLOAT(primary_fire_delay[WEAPON_MP5][bot_skill][0],
                            primary_fire_delay[WEAPON_MP5][bot_skill][1]);

            return TRUE;
         }
      }
   }

   // if the bot is carrying the crossbow...
   if (pev->weapons & (1<<WEAPON_CROSSBOW))
   {
      // if bot is not too close for crossbow and not too far...
      if (((distance > 100) && (distance < 1000) && (weapon_choice == 0)) ||
          (weapon_choice == WEAPON_CROSSBOW))
      {
         new_weapon = weapon_ptr[WEAPON_CROSSBOW];

         // check if the bot has any ammo left for this weapon...
         if (primary_ammo[WEAPON_CROSSBOW] > 0)
         {
            // check if the bot isn't already using this item...
            if (m_pActiveItem != new_weapon)
               SelectItem("weapon_crossbow");  // select the crossbow

            pev->button |= IN_ATTACK;  // use primary attack (bang! bang!)

            // set next time to shoot
            f_shoot_time = gpGlobals->time + 0.75 +
               RANDOM_FLOAT(primary_fire_delay[WEAPON_CROSSBOW][bot_skill][0],
                            primary_fire_delay[WEAPON_CROSSBOW][bot_skill][1]);

            return TRUE;
         }
      }
   }

   // if the bot is carrying the RPG...
   if (pev->weapons & (1<<WEAPON_RPG))
   {
      // don't use the RPG unless the enemy is pretty far away...
      if (((distance > 300) && (weapon_choice == 0)) ||
          (weapon_choice == WEAPON_RPG))
      {
         new_weapon = weapon_ptr[WEAPON_RPG];

         // check if the bot has any ammo left for this weapon...
         if (primary_ammo[WEAPON_RPG] > 0)
         {
            // check if the bot isn't already using this item...
            if (m_pActiveItem != new_weapon)
               SelectItem("weapon_rpg");  // select the RPG rocket launcher

            pev->button |= IN_ATTACK;  // use primary attack (bang! bang!)

            // set next time to shoot
            f_shoot_time = gpGlobals->time + 1.5 +
               RANDOM_FLOAT(primary_fire_delay[WEAPON_RPG][bot_skill][0],
                            primary_fire_delay[WEAPON_RPG][bot_skill][1]);

            return TRUE;
         }
      }
   }

   // if the bot is carrying the 9mm glock...
   if (pev->weapons & (1<<WEAPON_GLOCK))
   {
      // if nothing else was selected, try the good ol' 9mm glock...
      if (((distance < 1200) && (weapon_choice == 0)) ||
          (weapon_choice == WEAPON_GLOCK))
      {
         new_weapon = weapon_ptr[WEAPON_GLOCK];

         // check if the bot has any ammo left for this weapon...
         if (primary_ammo[WEAPON_GLOCK] > 0)
         {
            // check if the bot isn't already using this item...
            if (m_pActiveItem != new_weapon)
               SelectItem("weapon_9mmhandgun");  // select the trusty 9mm glock

            long use_secondary = RANDOM_LONG(1,100);

            // use secondary attack about 30% of the time
            if (use_secondary <= 30)
            {
// BigGuy - start
               pev->button |= IN_ATTACK2;  // use secondary attack (bang! bang!)

               // set next time to shoot
               f_shoot_time = gpGlobals->time + 0.2 +
                  RANDOM_FLOAT(secondary_fire_delay[WEAPON_GLOCK][bot_skill][0],
                               secondary_fire_delay[WEAPON_GLOCK][bot_skill][1]);
// BigGuy - end
            }
            else
            {
               pev->button |= IN_ATTACK;  // use primary attack (bang! bang!)

               // set next time to shoot
               f_shoot_time = gpGlobals->time + 0.3 +
                  RANDOM_FLOAT(primary_fire_delay[WEAPON_GLOCK][bot_skill][0],
                               primary_fire_delay[WEAPON_GLOCK][bot_skill][1]);
            }

            return TRUE;
         }
      }
   }

   // didn't have any available weapons or ammo, return FALSE
   return FALSE;
}


void CBot::BotShootAtEnemy( void )
{
   float f_distance;

   // aim for the head and/or body
   Vector v_enemy = BotBodyTarget( pBotEnemy ) - GetGunPosition( );

   pev->v_angle = UTIL_VecToAngles( v_enemy );

   pev->ideal_yaw = pev->v_angle.y;

   // check for wrap around of angle...
   if (pev->ideal_yaw > 180)
      pev->ideal_yaw -= 360;
   if (pev->ideal_yaw < -180)
      pev->ideal_yaw += 360;

   pev->v_angle.x = -pev->v_angle.x;  //adjust pitch to point gun

   // is it time to shoot yet?
   if (f_shoot_time <= gpGlobals->time)
   {
      // select the best weapon to use at this distance and fire...
      BotFireWeapon( v_enemy );
   }

   v_enemy.z = 0;  // ignore z component (up & down)

   f_distance = v_enemy.Length( );  // how far away is the enemy scum?

   if (f_distance > 200)      // run if distance to enemy is far
      f_move_speed = f_max_speed;
   else if (f_distance > 20)  // walk if distance is closer
      f_move_speed = f_max_speed / 2;
   else                     // don't move if close enough
      f_move_speed = 0.0;
}


void CBot::BotFindItem( void )
{
   CBaseEntity *pEntity = NULL;
   CBaseEntity *pPickupEntity = NULL;
   Vector pickup_origin;
   Vector entity_origin;
   float radius = 500;
   BOOL can_pickup;
   float min_distance;
   char item_name[40];
   TraceResult tr;
   Vector vecStart;
   Vector vecEnd;
   int angle_to_entity;

   pBotPickupItem = NULL;

   min_distance = radius + 1.0;

   while ((pEntity = UTIL_FindEntityInSphere( pEntity, pev->origin, radius )) != NULL)
   {
      can_pickup = FALSE;  // assume can't use it until known otherwise

      strcpy(item_name, STRING(pEntity->pev->classname));

      // see if this is a "func_" type of entity (func_button, etc.)...
      if (strncmp("func_", item_name, 5) == 0)
      {
         // BModels have 0,0,0 for origin so must use VecBModelOrigin...
         entity_origin = VecBModelOrigin(pEntity->pev);

         vecStart = pev->origin + pev->view_ofs;
         vecEnd = entity_origin;

         angle_to_entity = BotInFieldOfView( vecEnd - vecStart );

         // check if entity is outside field of view (+/- 45 degrees)
         if (angle_to_entity > 45)
            continue;  // skip this item if bot can't "see" it

         // trace a line from bot's eyes to func_ entity...
         UTIL_TraceLine( vecStart, vecEnd, dont_ignore_monsters, ENT(pev), &tr);

         // check if traced all the way up to the entity (didn't hit wall)
         if (strcmp(item_name, STRING(tr.pHit->v.classname)) == 0)
         {
            // find distance to item for later use...
            float distance = (vecEnd - vecStart).Length( );

            // check if entity is wall mounted health charger...
            if (strcmp("func_healthcharger", item_name) == 0)
            {
               // check if the bot can use this item and
               // check if the recharger is ready to use (has power left)...
               if ((pev->health < 100) && (pEntity->pev->frame == 0))
               {
                  // check if flag not set, facing it and close enough...
                  if ((!b_use_health_station) && (angle_to_entity <= 10) &&
                      (distance < PLAYER_SEARCH_RADIUS))
                  {
                     b_use_health_station = TRUE;
                     f_use_health_time = gpGlobals->time;
                  }

                  can_pickup = TRUE;
               }
               else
               {
                  // don't need or can't use this item...
                  b_use_health_station = FALSE;
               }
            }

            // check if entity is wall mounted HEV charger...
            else if (strcmp("func_recharge", item_name) == 0)
            {
               // check if the bot can use this item and
               // check if the recharger is ready to use (has power left)...
               if ((pev->armorvalue < MAX_NORMAL_BATTERY) &&
                   (pev->weapons & (1<<WEAPON_SUIT)) &&
                   (pEntity->pev->frame == 0))
               {
                  // check if flag not set, facing it and close enough...
                  if ((!b_use_HEV_station) && (angle_to_entity <= 10) &&
                      (distance < PLAYER_SEARCH_RADIUS))
                  {
                     b_use_HEV_station = TRUE;
                     f_use_HEV_time = gpGlobals->time;
                  }

                  can_pickup = TRUE;
               }
               else
               {
                  // don't need or can't use this item...
                  b_use_HEV_station = FALSE;
               }
            }

            // check if entity is a button...
            else if (strcmp("func_button", item_name) == 0)
            {
               // use the button about 100% of the time, if haven't
               // used a button in at least 2 seconds...
               if ((RANDOM_LONG(1, 100) <= 100) &&
                   ((f_use_button_time + 2) < gpGlobals->time))
               {
                  // check if flag not set, facing it and close enough...
                  if ((!b_use_button) && (angle_to_entity <= 10) &&
                      (distance < PLAYER_SEARCH_RADIUS))
                  {
                     b_use_button = TRUE;
                     b_lift_moving = FALSE;
                     f_use_button_time = gpGlobals->time;
                  }

                  can_pickup = TRUE;
               }
               else
               {
                  // don't need or can't use this item...
                  b_use_button = FALSE;
               }
            }
         }
      }

      // check if entity is an armed tripmine beam
      else if (strcmp("beam", item_name) == 0)
      {
         CBeam *pBeam = (CBeam *)pEntity;

         Vector v_beam_start = pBeam->GetStartPos();
         Vector v_beam_end = pBeam->GetEndPos();

         if (BotInFieldOfView( v_beam_start ) &&
             BotEntityIsVisible( v_beam_start ))
         {
//            BotDebug("I see a beam start!\n");
         }

         if (BotInFieldOfView( v_beam_end ) &&
             BotEntityIsVisible( v_beam_end ))
         {
//            BotDebug("I see a beam end!\n");
         }
      }

      else  // everything else...
      {
         entity_origin = pEntity->pev->origin;

         vecStart = pev->origin + pev->view_ofs;
         vecEnd = entity_origin;

         // find angles from bot origin to entity...
         angle_to_entity = BotInFieldOfView( vecEnd - vecStart );

         // check if entity is outside field of view (+/- 45 degrees)
         if (angle_to_entity > 45)
            continue;  // skip this item if bot can't "see" it

         // check if line of sight to object is not blocked (i.e. visible)
         if (BotEntityIsVisible( vecEnd ))
         {

            // check if entity is a weapon...
            if (strncmp("weapon_", item_name, 7) == 0)
            {
               CBasePlayerWeapon *pWeapon = (CBasePlayerWeapon *)pEntity;

               if ((pWeapon->m_pPlayer) || (pWeapon->pev->effects & EF_NODRAW))
               {
                  // someone owns this weapon or it hasn't respawned yet
                  continue;
               }

               if (g_pGameRules->CanHavePlayerItem( this, pWeapon ))
               {
                  can_pickup = TRUE;
               }
            }

            // check if entity is ammo...
            else if (strncmp("ammo_", item_name, 5) == 0)
            {
               CBasePlayerAmmo *pAmmo = (CBasePlayerAmmo *)pEntity;
               BOOL ammo_found = FALSE;
               int i;

               // check if the item is not visible (i.e. has not respawned)
               if (pAmmo->pev->effects & EF_NODRAW)
                  continue;

               i = 0;
               while (ammo_check[i].ammo_name[0])
               {
                  if (strcmp(ammo_check[i].ammo_name, item_name) == 0)
                  {
                     ammo_found = TRUE;

                     // see if the bot can pick up this item...
                     if (g_pGameRules->CanHaveAmmo( this,
                         ammo_check[i].weapon_name, ammo_check[i].max_carry))
                     {
                        can_pickup = TRUE;
                        break;
                     }
                  }

                  i++;
               }
               if (ammo_found == FALSE)
               {
                  sprintf(message, "unknown ammo: %s\n", item_name);
                  BotDebug(message);
               }
            }

            // check if entity is a battery...
            else if (strcmp("item_battery", item_name) == 0)
            {
               CItem *pBattery = (CItem *)pEntity;

               // check if the item is not visible (i.e. has not respawned)
               if (pBattery->pev->effects & EF_NODRAW)
                  continue;

               // check if the bot can use this item...
               if ((pev->armorvalue < MAX_NORMAL_BATTERY) &&
                   (pev->weapons & (1<<WEAPON_SUIT)))
               {
                  can_pickup = TRUE;
               }
            }

            // check if entity is a healthkit...
            else if (strcmp("item_healthkit", item_name) == 0)
            {
               CItem *pHealthKit = (CItem *)pEntity;

               // check if the item is not visible (i.e. has not respawned)
               if (pHealthKit->pev->effects & EF_NODRAW)
                  continue;

               // check if the bot can use this item...
               if (pev->health < 100)
               {
                  can_pickup = TRUE;
               }
            }

            // check if entity is a packed up weapons box...
            else if (strcmp("weaponbox", item_name) == 0)
            {
               can_pickup = TRUE;
            }

            // check if entity is the spot from RPG laser
            else if (strcmp("laser_spot", item_name) == 0)
            {
            }

            // check if entity is an armed tripmine
            else if (strcmp("monster_tripmine", item_name) == 0)
            {
               float distance = (pEntity->pev->origin - pev->origin).Length( );

               if (b_see_tripmine)
               {
                  // see if this tripmine is closer to bot...
                  if (distance < (v_tripmine - pev->origin).Length())
                  {
                     v_tripmine = pEntity->pev->origin;
                     b_shoot_tripmine = FALSE;

                     // see if bot is far enough to shoot the tripmine...
                     if (distance >= 375)
                     {
                        b_shoot_tripmine = TRUE;
                     }
                  }
               }
               else
               {
                  b_see_tripmine = TRUE;
                  v_tripmine = pEntity->pev->origin;
                  b_shoot_tripmine = FALSE;

                  // see if bot is far enough to shoot the tripmine...
                  if (distance >= 375)  // 375 is damage radius
                  {
                     b_shoot_tripmine = TRUE;
                  }
               }
            }

            // check if entity is an armed satchel charge
            else if (strcmp("monster_satchel", item_name) == 0)
            {
            }

            // check if entity is a snark (squeak grenade)
            else if (strcmp("monster_snark", item_name) == 0)
            {
            }

         }  // end if object is visible
      }  // end else not "func_" entity

      if (can_pickup) // if the bot found something it can pickup...
      {
         float distance = (entity_origin - pev->origin).Length( );

         // see if it's the closest item so far...
         if (distance < min_distance)
         {
            min_distance = distance;        // update the minimum distance
            pPickupEntity = pEntity;        // remember this entity
            pickup_origin = entity_origin;  // remember location of entity
         }
      }
   }  // end while loop

   if (pPickupEntity != NULL)
   {
      // let's head off toward that item...
      Vector v_item = pickup_origin - pev->origin;

      Vector bot_angles = UTIL_VecToAngles( v_item );

      pev->ideal_yaw = bot_angles.y;

      // check for wrap around of angle...
      if (pev->ideal_yaw > 180)
         pev->ideal_yaw -= 360;
      if (pev->ideal_yaw < -180)
         pev->ideal_yaw += 360;

      pBotPickupItem = pPickupEntity;  // save the item bot is trying to get
   }
}


void CBot::BotUseLift( float moved_distance )
{
   // just need to press the button once, when the flag gets set...
   if (f_use_button_time == gpGlobals->time)
   {
      pev->button = IN_USE;

      // face opposite from the button
      pev->ideal_yaw = -pev->ideal_yaw;  // rotate 180 degrees

      // check for wrap around of angle...
      if (pev->ideal_yaw > 180)
         pev->ideal_yaw -= 360;
      if (pev->ideal_yaw < -180)
         pev->ideal_yaw += 360;
   }

   // check if the bot has waited too long for the lift to move...
   if (((f_use_button_time + 2.0) < gpGlobals->time) &&
       (!b_lift_moving))
   {
      // clear use button flag
      b_use_button = FALSE;

      // bot doesn't have to set f_find_item since the bot
      // should already be facing away from the button

      f_move_speed = f_max_speed;
   }

   // check if lift has started moving...
   if ((moved_distance > 1) && (!b_lift_moving))
   {
      b_lift_moving = TRUE;
   }

   // check if lift has stopped moving...
   if ((moved_distance <= 1) && (b_lift_moving))
   {
      TraceResult tr1, tr2;
      Vector v_src, v_forward, v_right, v_left;
      Vector v_down, v_forward_down, v_right_down, v_left_down;

      b_use_button = FALSE;

      // TraceLines in 4 directions to find which way to go...

      UTIL_MakeVectors( pev->v_angle );

      v_src = pev->origin + pev->view_ofs;
      v_forward = v_src + gpGlobals->v_forward * 90;
      v_right = v_src + gpGlobals->v_right * 90;
      v_left = v_src + gpGlobals->v_right * -90;

      v_down = pev->v_angle;
      v_down.x = v_down.x + 45;  // look down at 45 degree angle

      UTIL_MakeVectors( v_down );

      v_forward_down = v_src + gpGlobals->v_forward * 100;
      v_right_down = v_src + gpGlobals->v_right * 100;
      v_left_down = v_src + gpGlobals->v_right * -100;

      // try tracing forward first...
      UTIL_TraceLine( v_src, v_forward, dont_ignore_monsters, ENT(pev), &tr1);
      UTIL_TraceLine( v_src, v_forward_down, dont_ignore_monsters, ENT(pev), &tr2);

      // check if we hit a wall or didn't find a floor...
      if ((tr1.flFraction < 1.0) || (tr2.flFraction >= 1.0))
      {
         // try tracing to the RIGHT side next...
         UTIL_TraceLine( v_src, v_right, dont_ignore_monsters, ENT(pev), &tr1);
         UTIL_TraceLine( v_src, v_right_down, dont_ignore_monsters, ENT(pev), &tr2);

         // check if we hit a wall or didn't find a floor...
         if ((tr1.flFraction < 1.0) || (tr2.flFraction >= 1.0))
         {
            // try tracing to the LEFT side next...
            UTIL_TraceLine( v_src, v_left, dont_ignore_monsters, ENT(pev), &tr1);
            UTIL_TraceLine( v_src, v_left_down, dont_ignore_monsters, ENT(pev), &tr2);

            // check if we hit a wall or didn't find a floor...
            if ((tr1.flFraction < 1.0) || (tr2.flFraction >= 1.0))
            {
               // only thing to do is turn around...
               pev->ideal_yaw += 180;  // turn all the way around
            }
            else
            {
               pev->ideal_yaw += 90;  // turn to the LEFT
            }
         }
         else
         {
            pev->ideal_yaw -= 90;  // turn to the RIGHT
         }

         // check for wrap around of angle...
         if (pev->ideal_yaw > 180)
            pev->ideal_yaw -= 360;
         if (pev->ideal_yaw < -180)
            pev->ideal_yaw += 360;
      }

      BotChangeYaw( pev->yaw_speed );

      f_move_speed = f_max_speed;
   }
}


void CBot::BotTurnAtWall( TraceResult *tr )
{
   Vector Normal;
   float Y, Y1, Y2, D1, D2, Z;

   // Find the normal vector from the trace result.  The normal vector will
   // be a vector that is perpendicular to the surface from the TraceResult.

   Normal = UTIL_VecToAngles(tr->vecPlaneNormal);

   // Since the bot keeps it's view angle in -180 < x < 180 degrees format,
   // and since TraceResults are 0 < x < 360, we convert the bot's view
   // angle (yaw) to the same format at TraceResult.

   Y = pev->v_angle.y;
   Y = Y + 180;
   if (Y > 359) Y -= 360;

   // Turn the normal vector around 180 degrees (i.e. make it point towards
   // the wall not away from it.  That makes finding the angles that the
   // bot needs to turn a little easier.

   Normal.y = Normal.y - 180;
   if (Normal.y < 0)
   Normal.y += 360;

   // Here we compare the bots view angle (Y) to the Normal - 90 degrees (Y1)
   // and the Normal + 90 degrees (Y2).  These two angles (Y1 & Y2) represent
   // angles that are parallel to the wall surface, but heading in opposite
   // directions.  We want the bot to choose the one that will require the
   // least amount of turning (saves time) and have the bot head off in that
   // direction.

   Y1 = Normal.y - 90;
   if (RANDOM_LONG(1, 100) <= 50)
      Y1 = Y1 - RANDOM_FLOAT(5.0, 20.0);
   if (Y1 < 0) Y1 += 360;

   Y2 = Normal.y + 90;
   if (RANDOM_LONG(1, 100) <= 50)
      Y2 = Y2 + RANDOM_FLOAT(5.0, 20.0);
   if (Y2 > 359) Y2 -= 360;

   // D1 and D2 are the difference (in degrees) between the bot's current
   // angle and Y1 or Y2 (respectively).

   D1 = abs(Y - Y1);
   if (D1 > 179) D1 = abs(D1 - 360);
   D2 = abs(Y - Y2);
   if (D2 > 179) D2 = abs(D2 - 360);

   // If difference 1 (D1) is more than difference 2 (D2) then the bot will
   // have to turn LESS if it heads in direction Y1 otherwise, head in
   // direction Y2.  I know this seems backwards, but try some sample angles
   // out on some graph paper and go through these equations using a
   // calculator, you'll see what I mean.

   if (D1 > D2) Z = Y1;
   else Z = Y2;

   // convert from TraceResult 0 to 360 degree format back to bot's
   // -180 to 180 degree format.

   if (Z > 180) Z -= 360;

   // set the direction to head off into...
   pev->ideal_yaw = Z;

   f_move_speed = 0;  // don't move while turning
}


BOOL CBot::BotCantMoveForward( TraceResult *tr )
{
   // use some TraceLines to determine if anything is blocking the current
   // path of the bot.

   Vector v_src, v_forward;

   UTIL_MakeVectors( pev->v_angle );

   // first do a trace from the bot's eyes forward...

   v_src = pev->origin + pev->view_ofs;  // EyePosition()
   v_forward = v_src + gpGlobals->v_forward * 40;

   // trace from the bot's eyes straight forward...
   UTIL_TraceLine( v_src, v_forward, dont_ignore_monsters, ENT(pev), tr);

   // check if the trace hit something...
   if (tr->flFraction < 1.0)
   {
      return TRUE;  // bot's head will hit something
   }

   // bot's head is clear, check at waist level...

   v_src = pev->origin;
   v_forward = v_src + gpGlobals->v_forward * 40;

   // trace from the bot's waist straight forward...
   UTIL_TraceLine( v_src, v_forward, dont_ignore_monsters, ENT(pev), tr);

   // check if the trace hit something...
   if (tr->flFraction < 1.0)
   {
      return TRUE;  // bot's body will hit something
   }

   return FALSE;  // bot can move forward, return false
}


BOOL CBot::BotCanJumpUp( void )
{
   // What I do here is trace 3 lines straight out, one unit higher than
   // the highest normal jumping distance.  I trace once at the center of
   // the body, once at the right side, and once at the left side.  If all
   // three of these TraceLines don't hit an obstruction then I know the
   // area to jump to is clear.  I then need to trace from head level,
   // above where the bot will jump to, downward to see if there is anything
   // blocking the jump.  There could be a narrow opening that the body
   // will not fit into.  These horizontal and vertical TraceLines seem
   // to catch most of the problems with falsely trying to jump on something
   // that the bot can not get onto.

   TraceResult tr;
   Vector v_jump, v_source, v_dest;

   // convert current view angle to vectors for TraceLine math...

   v_jump = pev->v_angle;
   v_jump.x = 0;  // reset pitch to 0 (level horizontally)
   v_jump.z = 0;  // reset roll to 0 (straight up and down)

   UTIL_MakeVectors( v_jump );

   // use center of the body first...

   // maximum jump height is 45, so check one unit above that (46)
   v_source = pev->origin + Vector(0, 0, -36 + 46);
   v_dest = v_source + gpGlobals->v_forward * 24;

   // trace a line forward at maximum jump height...
   UTIL_TraceLine( v_source, v_dest, dont_ignore_monsters, ENT(pev), &tr);

   // if trace hit something, return FALSE
   if (tr.flFraction < 1.0)
      return FALSE;

   // now check same height to one side of the bot...
   v_source = pev->origin + gpGlobals->v_right * 16 + Vector(0, 0, -36 + 46);
   v_dest = v_source + gpGlobals->v_forward * 24;

   // trace a line forward at maximum jump height...
   UTIL_TraceLine( v_source, v_dest, dont_ignore_monsters, ENT(pev), &tr);

   // if trace hit something, return FALSE
   if (tr.flFraction < 1.0)
      return FALSE;

   // now check same height on the other side of the bot...
   v_source = pev->origin + gpGlobals->v_right * -16 + Vector(0, 0, -36 + 46);
   v_dest = v_source + gpGlobals->v_forward * 24;

   // trace a line forward at maximum jump height...
   UTIL_TraceLine( v_source, v_dest, dont_ignore_monsters, ENT(pev), &tr);

   // if trace hit something, return FALSE
   if (tr.flFraction < 1.0)
      return FALSE;

   // now trace from head level downward to check for obstructions...

   // start of trace is 24 units in front of bot, 72 units above head...
   v_source = pev->origin + gpGlobals->v_forward * 24;

   // offset 72 units from top of head (72 + 36)
   v_source.z = v_source.z + 108;

   // end point of trace is 99 units straight down from start...
   // (99 is 108 minus the jump limit height which is 45 - 36 = 9)
   v_dest = v_source + Vector(0, 0, -99);

   // trace a line straight down toward the ground...
   UTIL_TraceLine( v_source, v_dest, dont_ignore_monsters, ENT(pev), &tr);

   // if trace hit something, return FALSE
   if (tr.flFraction < 1.0)
      return FALSE;

   // now check same height to one side of the bot...
   v_source = pev->origin + gpGlobals->v_right * 16 + gpGlobals->v_forward * 24;
   v_source.z = v_source.z + 108;
   v_dest = v_source + Vector(0, 0, -99);

   // trace a line straight down toward the ground...
   UTIL_TraceLine( v_source, v_dest, dont_ignore_monsters, ENT(pev), &tr);

   // if trace hit something, return FALSE
   if (tr.flFraction < 1.0)
      return FALSE;

   // now check same height on the other side of the bot...
   v_source = pev->origin + gpGlobals->v_right * -16 + gpGlobals->v_forward * 24;
   v_source.z = v_source.z + 108;
   v_dest = v_source + Vector(0, 0, -99);

   // trace a line straight down toward the ground...
   UTIL_TraceLine( v_source, v_dest, dont_ignore_monsters, ENT(pev), &tr);

   // if trace hit something, return FALSE
   if (tr.flFraction < 1.0)
      return FALSE;

   return TRUE;
}


BOOL CBot::BotCanDuckUnder( void )
{
   // What I do here is trace 3 lines straight out, one unit higher than
   // the ducking height.  I trace once at the center of the body, once
   // at the right side, and once at the left side.  If all three of these
   // TraceLines don't hit an obstruction then I know the area to duck to
   // is clear.  I then need to trace from the ground up, 72 units, to make
   // sure that there is something blocking the TraceLine.  Then we know
   // we can duck under it.

   TraceResult tr;
   Vector v_duck, v_source, v_dest;

   // convert current view angle to vectors for TraceLine math...

   v_duck = pev->v_angle;
   v_duck.x = 0;  // reset pitch to 0 (level horizontally)
   v_duck.z = 0;  // reset roll to 0 (straight up and down)

   UTIL_MakeVectors( v_duck );

   // use center of the body first...

   // duck height is 36, so check one unit above that (37)
   v_source = pev->origin + Vector(0, 0, -36 + 37);
   v_dest = v_source + gpGlobals->v_forward * 24;

   // trace a line forward at duck height...
   UTIL_TraceLine( v_source, v_dest, dont_ignore_monsters, ENT(pev), &tr);

   // if trace hit something, return FALSE
   if (tr.flFraction < 1.0)
      return FALSE;

   // now check same height to one side of the bot...
   v_source = pev->origin + gpGlobals->v_right * 16 + Vector(0, 0, -36 + 37);
   v_dest = v_source + gpGlobals->v_forward * 24;

   // trace a line forward at duck height...
   UTIL_TraceLine( v_source, v_dest, dont_ignore_monsters, ENT(pev), &tr);

   // if trace hit something, return FALSE
   if (tr.flFraction < 1.0)
      return FALSE;

   // now check same height on the other side of the bot...
   v_source = pev->origin + gpGlobals->v_right * -16 + Vector(0, 0, -36 + 37);
   v_dest = v_source + gpGlobals->v_forward * 24;

   // trace a line forward at duck height...
   UTIL_TraceLine( v_source, v_dest, dont_ignore_monsters, ENT(pev), &tr);

   // if trace hit something, return FALSE
   if (tr.flFraction < 1.0)
      return FALSE;

   // now trace from the ground up to check for object to duck under...

   // start of trace is 24 units in front of bot near ground...
   v_source = pev->origin + gpGlobals->v_forward * 24;
   v_source.z = v_source.z - 35;  // offset to feet + 1 unit up

   // end point of trace is 72 units straight up from start...
   v_dest = v_source + Vector(0, 0, 72);

   // trace a line straight up in the air...
   UTIL_TraceLine( v_source, v_dest, dont_ignore_monsters, ENT(pev), &tr);

   // if trace didn't hit something, return FALSE
   if (tr.flFraction >= 1.0)
      return FALSE;

   // now check same height to one side of the bot...
   v_source = pev->origin + gpGlobals->v_right * 16 + gpGlobals->v_forward * 24;
   v_source.z = v_source.z - 35;  // offset to feet + 1 unit up
   v_dest = v_source + Vector(0, 0, 72);

   // trace a line straight up in the air...
   UTIL_TraceLine( v_source, v_dest, dont_ignore_monsters, ENT(pev), &tr);

   // if trace didn't hit something, return FALSE
   if (tr.flFraction >= 1.0)
      return FALSE;

   // now check same height on the other side of the bot...
   v_source = pev->origin + gpGlobals->v_right * -16 + gpGlobals->v_forward * 24;
   v_source.z = v_source.z - 35;  // offset to feet + 1 unit up
   v_dest = v_source + Vector(0, 0, 72);

   // trace a line straight up in the air...
   UTIL_TraceLine( v_source, v_dest, dont_ignore_monsters, ENT(pev), &tr);

   // if trace didn't hit something, return FALSE
   if (tr.flFraction >= 1.0)
      return FALSE;

   return TRUE;
}


BOOL CBot::BotShootTripmine( void )
{
   if (b_shoot_tripmine != TRUE)
      return FALSE;

   // aim at the tripmine and fire the glock...

   Vector v_enemy = v_tripmine - GetGunPosition( );

   pev->v_angle = UTIL_VecToAngles( v_enemy );

   pev->ideal_yaw = pev->v_angle.y;

   // check for wrap around of angle...
   if (pev->ideal_yaw > 180)
      pev->ideal_yaw -= 360;
   if (pev->ideal_yaw < -180)
      pev->ideal_yaw += 360;

   pev->v_angle.x = -pev->v_angle.x;  //adjust pitch to point gun

   return (BotFireWeapon( v_enemy, WEAPON_GLOCK, TRUE ));
}


void CBot::Think( void )
{
   // thanks, Jehannum!

   CBasePlayer::PreThink();

   if ((gpGlobals->time >= pev->nextthink) && (m_pfnThink != NULL))
      (this->*m_pfnThink)();

   CBasePlayer::PostThink();
}


void CBot::BotThink( void )
{
   Vector v_diff;             // vector from previous to current location
   float moved_distance;      // length of v_diff vector (distance bot moved)
   float degrees_turned;


   pev->nextthink = gpGlobals->time + 0.1;

   if (!IsInWorld( ))
   {
      SetTouch( NULL );
      UTIL_Remove( this );
      return;
   }

   if (!IsAlive( ))
   {
      // if bot is dead, don't take up any space in world (set size to 0)
      UTIL_SetSize( pev, Vector(0, 0, 0), Vector(0, 0, 0));

      return;
   }

   // see if it's time to check for sv_maxspeed change...
   if (f_speed_check_time <= gpGlobals->time)
   {
      // store current sv_maxspeed setting for quick access
      f_max_speed = CVAR_GET_FLOAT("sv_maxspeed");
//      f_max_speed = f_max_speed * speed_adjust[bot_skill];

      // set next check time to 2 seconds from now
      f_speed_check_time = gpGlobals->time + 2.0;
   }

   // see how far bot has moved since the previous position...
   v_diff = v_prev_origin - pev->origin;
   moved_distance = v_diff.Length();

   v_prev_origin = pev->origin;  // save current position as previous

   pev->button = 0;  // make sure no buttons are pressed

   f_move_speed = f_max_speed;  // set to max speed unless known otherwise

   // turn towards ideal_yaw by yaw_speed degrees
   degrees_turned = BotChangeYaw( pev->yaw_speed );

   if (degrees_turned >= pev->yaw_speed)
   {
      // if bot is still turning, turn in place by setting speed to 0
      f_move_speed = 0;
   }

   else if (IsOnLadder( ))  // check if the bot is on a ladder...
   {
      BotOnLadder( moved_distance );  // go handle the ladder movement
   }

   else  // else handle movement related actions...
   {
      // bot is not on a ladder so clear ladder direction flag...
      ladder_dir = 0;

      if (f_botdontshoot == 0)  // is bot targeting turned on?
         pBotEnemy = BotFindEnemy( );
      else
         pBotEnemy = NULL;  // clear enemy pointer (no ememy for you!)

      if (pBotEnemy != NULL)  // does an enemy exist?
      {
         BotShootAtEnemy( );  // shoot at the enemy
      }
      else if (f_pause_time > gpGlobals->time)  // is bot "paused"?
      {
         // you could make the bot look left then right, or look up
         // and down, to make it appear that the bot is hunting for
         // something (don't do anything right now)

         f_move_speed = 0;
      }
      else
      {
         // no enemy, let's just wander around...

         pev->v_angle.x = 0;  // reset pitch to 0 (level horizontally)
         pev->v_angle.z = 0;  // reset roll to 0 (straight up and down)

         // check if bot should look for items now or not...
         if (f_find_item < gpGlobals->time)
         {
            BotFindItem( );  // see if there are any visible items
         }

         // check if bot sees a tripmine...
         if (b_see_tripmine)
         {
            // check if bot can shoot the tripmine...
            if ((b_shoot_tripmine) && BotShootTripmine( ))
            {
               // shot at tripmine, may or may not have hit it, clear
               // flags anyway, next BotFindItem will see it again if
               // it is still there...

               b_shoot_tripmine = FALSE;
               b_see_tripmine = FALSE;

               // pause for a while to allow tripmine to explode...
               f_pause_time = gpGlobals->time + 0.5;
            }
            else  // run away!!!
            {
               Vector tripmine_angles;

               tripmine_angles = UTIL_VecToAngles( v_tripmine - pev->origin );

               // face away from the tripmine
               pev->ideal_yaw = -tripmine_angles.y;

               // check for wrap around of angle...
               if (pev->ideal_yaw > 180)
                  pev->ideal_yaw -= 360;
               if (pev->ideal_yaw < -180)
                  pev->ideal_yaw += 360;

               b_see_tripmine = FALSE;

               f_move_speed = 0;  // don't run while turning
            }
         }

         // check if should use wall mounted health station...
         else if (b_use_health_station)
         {
            if ((f_use_health_time + 10.0) > gpGlobals->time)
            {
               f_move_speed = 0;  // don't move while using health station

               pev->button = IN_USE;
            }
            else
            {
               // bot is stuck trying to "use" a health station...

               b_use_health_station = FALSE;

               // don't look for items for a while since the bot
               // could be stuck trying to get to an item
               f_find_item = gpGlobals->time + 0.5;
            }
         }

         // check if should use wall mounted HEV station...
         else if (b_use_HEV_station)
         {
            if ((f_use_HEV_time + 10.0) > gpGlobals->time)
            {
               f_move_speed = 0;  // don't move while using HEV station

               pev->button = IN_USE;
            }
            else
            {
               // bot is stuck trying to "use" a HEV station...

               b_use_HEV_station = FALSE;

               // don't look for items for a while since the bot
               // could be stuck trying to get to an item
               f_find_item = gpGlobals->time + 0.5;
            }
         }

         else if (b_use_button)
         {
            f_move_speed = 0;  // don't move while using elevator

            BotUseLift( moved_distance );
         }

         else
         {
            TraceResult tr;

            if (pev->waterlevel == 3)  // check if the bot is underwater...
            {
               BotUnderWater( );
            }

            // check if bot is about to hit a wall.  TraceResult gets returned
            if (BotCantMoveForward( &tr ))
            {
               // ADD LATER
               // need to check if bot can jump up or duck under here...
               // ADD LATER

               BotTurnAtWall( &tr );
            }

            // check if the bot hasn't moved much since the last location
            else if ((moved_distance <= 1) && (!bot_was_paused))
            {
               // the bot must be stuck!
            
               if (BotCanJumpUp( ))  // can the bot jump onto something?
               {
                  pev->button |= IN_JUMP;  // jump up and move forward
               }
               else if (BotCanDuckUnder( ))  // can the bot duck under something?
               {
                  pev->button |= IN_DUCK;  // duck down and move forward
               }
               else
               {
                  f_move_speed = 0;  // don't move while turning

                  // turn randomly between 30 and 60 degress
                  if (wander_dir == WANDER_LEFT)
                     pev->ideal_yaw += RANDOM_LONG(30, 60);
                  else
                     pev->ideal_yaw -= RANDOM_LONG(30, 60);

                  // check for wrap around of angle...
                  if (pev->ideal_yaw > 180)
                     pev->ideal_yaw -= 360;
                  if (pev->ideal_yaw < -180)
                     pev->ideal_yaw += 360;

                  // is the bot trying to get to an item?...
                  if (pBotPickupItem != NULL)
                  {
                     // don't look for items for a while since the bot
                     // could be stuck trying to get to an item
                     f_find_item = gpGlobals->time + 0.5;
                  }
               }
            }

            // should the bot pause for a while here?
            if (RANDOM_LONG(1, 1000) <= pause_frequency[bot_skill])
            {
               // set the time that the bot will stop "pausing"
               f_pause_time = gpGlobals->time +
                  RANDOM_FLOAT(pause_time[bot_skill][0],
                               pause_time[bot_skill][1]);
            }
         }
      }
   }

   if (f_move_speed < 1)
      bot_was_paused = TRUE;
   else
      bot_was_paused = FALSE;


//   g_engfuncs.pfnRunPlayerMove(<bot edict>, <bot's view angles>,
//             <forward move>, <side move>, <up move> (does nothing?),
//             <buttons> (IN_ATTACK, etc.), <impulse number>, <msec value>);

   g_engfuncs.pfnRunPlayerMove( edict( ), pev->v_angle, f_move_speed,
                                0, 0, pev->button, 0, 50 );
}

