// botman's Half-Life bot example (http://jump.to/botman/)

#include "extdll.h"
#include "util.h"
#include "client.h"
#include "cbase.h"
#include "player.h"
#include "items.h"
#include "weapons.h"
#include "soundent.h"
#include "gamerules.h"
#include "animation.h"

#include "bot.h"

extern DLL_GLOBAL ULONG g_ulModelIndexPlayer;

// Set in combat.cpp.  Used to pass the damage inflictor for death messages.
extern entvars_t *g_pevLastInflictor;

extern int gmsgHealth;
extern int gmsgCurWeapon;
extern int gmsgSetFOV;

int f_Observer = 0;  // flag to indicate if we are in observer mode
int f_botskill = 3;  // default bot skill level
char botskin[16];
char botname[32];

#define MAX_SKINS 10

// indicate which models are currently used for random model allocation
BOOL skin_used[MAX_SKINS] = {
   FALSE, FALSE, FALSE, FALSE, FALSE,
   FALSE, FALSE, FALSE, FALSE, FALSE};

// store the names of the models...
char *bot_skins[MAX_SKINS] = {
   "barney","gina","gman","gordon","helmet",
   "hgrunt","recon","robo","scientist","zombie"};

// store the player names for each of the models...
char *bot_names[MAX_SKINS] = {
   "Barney","Gina","G-Man","Gordon","Helmet",
   "H-Grunt","Recon","Robo","Scientist","Zombie"};

// speed adjustment array based on skill (multiply by max_speed)
float speed_adjust[5] = {0.95, 0.90, 0.85, 0.80, 0.75};

// weapon firing delay based on skill (min and max delay for each weapon)
float fire_delay[WEAPON_SNARK+1][5][2] = {
   // WEAPON_NONE - NOT USED
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_CROWBAR
   {{0.0, 0.1}, {0.2, 0.3}, {0.3, 0.5}, {0.4, 0.6}, {0.6, 1.0}},
   // WEAPON_GLOCK (9mm)
   {{0.0, 0.1}, {0.1, 0.2}, {0.2, 0.3}, {0.3, 0.4}, {0.4, 0.5}},
   // WEAPON_PYTHON (357)
   {{0.0, 0.25}, {0.2, 0.5}, {0.4, 0.8}, {1.0, 1.3}, {1.5, 2.0}},
   // WEAPON_MP5 (9mmAR)
   {{0.0, 0.1}, {0.1, 0.3}, {0.3, 0.5}, {0.4, 0.6}, {0.5, 0.8}},
   // WEAPON_CHAINGUN - NOT USED
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_CROSSBOW
   {{0.0, 0.25}, {0.2, 0.4}, {0.5, 0.7}, {0.8, 1.0}, {1.0, 1.3}},
   // WEAPON_SHOTGUN
   {{0.0, 0.25}, {0.2, 0.5}, {0.4, 0.8}, {0.6, 1.2}, {0.8, 2.0}},
   // WEAPON_RPG
   {{1.0, 3.0}, {2.0, 4.0}, {3.0, 5.0}, {4.0, 6.0}, {5.0, 7.0}},
   // WEAPON_GAUSS
   {{0.0, 0.1}, {0.2, 0.3}, {0.3, 0.5}, {0.5, 0.8}, {1.0, 1.2}},
   // WEAPON_EGON
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_HORNETGUN
   {{0.0, 0.1}, {0.25, 0.4}, {0.4, 0.7}, {0.6, 1.0}, {1.0, 1.5}},
   // WEAPON_HANDGRENADE
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_TRIPMINE
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_SATCHEL
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
   // WEAPON_SNARK
   {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}}
   };   

char hgrunt_sounds[][30] = {
   "hgrunt/gr_pain1.wav",
   "hgrunt/gr_pain2.wav",
   "hgrunt/gr_pain3.wav",
   "hgrunt/gr_pain4.wav",
   "hgrunt/gr_pain5.wav"};

char barney_sounds[][30] = {
   "barney/ba_bring.wav",
   "barney/ba_pain1.wav",
   "barney/ba_pain2.wav",
   "barney/ba_pain3.wav",
   "barney/ba_die1.wav"};

char scientist_sounds[][30] = {
   "scientist/sci_fear8.wav",
   "scientist/sci_fear13.wav",
   "scientist/sci_fear14.wav",
   "scientist/sci_fear15.wav",
   "scientist/sci_pain3.wav"};

ammo_check_t ammo_check[] = {
//   {"ammo_glockclip", "9mm", _9MM_MAX_CARRY},     // NOT USED?
   {"ammo_9mmclip", "9mm", _9MM_MAX_CARRY},
   {"ammo_9mmAR", "9mm", _9MM_MAX_CARRY},
//   {"ammo_9mmbox", "9mm", _9MM_MAX_CARRY},        // NOT USED?
//   {"ammo_mp5clip", "9mm", _9MM_MAX_CARRY},       // NOT USED?
//   {"ammo_chainboxclip", "9mm", _9MM_MAX_CARRY},  // NOT USED?
//   {"ammo_mp5grenades", "ARgrenades", M203_GRENADE_MAX_CARRY},  // NOT USED?
   {"ammo_ARgrenades", "ARgrenades", M203_GRENADE_MAX_CARRY},
   {"ammo_buckshot", "buckshot", BUCKSHOT_MAX_CARRY},
   {"ammo_crossbow", "bolts", BOLT_MAX_CARRY},
   {"ammo_357", "357", _357_MAX_CARRY},
   {"ammo_rpgclip", "rockets", ROCKET_MAX_CARRY},
   {"ammo_egonclip", "uranium", URANIUM_MAX_CARRY},
   {"ammo_gaussclip", "uranium", URANIUM_MAX_CARRY},
   {"", 0, 0}};


LINK_ENTITY_TO_CLASS( bot, CBot );


inline edict_t *CREATE_FAKE_CLIENT( const char *netname )
{
   return (*g_engfuncs.pfnCreateFakeClient)( netname );
}

inline char *GET_INFOBUFFER( edict_t *e )
{
   return (*g_engfuncs.pfnGetInfoKeyBuffer)( e );
}

inline char *GET_INFO_KEY_VALUE( char *infobuffer, char *key )
{
   return (g_engfuncs.pfnInfoKeyValue( infobuffer, key ));
}

inline void SET_CLIENT_KEY_VALUE( int clientIndex, char *infobuffer,
                                  char *key, char *value )
{
    (*g_engfuncs.pfnSetClientKeyValue)( clientIndex, infobuffer, key, value );
}


void BotDebug( char *buffer )
{
   // print out debug messages to the HUD of all players
   // this allows you to print messages from bots to your display
   // as you are playing the game.

   UTIL_ClientPrintAll( HUD_PRINTNOTIFY, buffer );
}


void BotCreate(edict_t *pEntity, const char *skin, const char *name,
               const char *skill)
{
   edict_t *BotEnt;
   int skill_level;
   char c_skill[2];
   int i;
   int index;
   BOOL found = FALSE;

   if ((skin == NULL) || (*skin == 0))
   {
      // pick a random skin
      index = RANDOM_LONG(0, 9);  // there are ten possible skins

      // check if this skin has already been used...
      while (skin_used[index] == TRUE)
      {
         index++;

         if (index == MAX_SKINS)
            index = 0;
      }

      skin_used[index] = TRUE;

      // check if all skins are used...
      for (i = 0; i < MAX_SKINS; i++)
      {
         if (skin_used[i] == FALSE)
            break;
      }

      // if all skins are used, reset used to FALSE for next selection
      if (i == MAX_SKINS)
      {
         for (i = 0; i < MAX_SKINS; i++)
            skin_used[i] = FALSE;
      }

      strcpy( botskin, bot_skins[index] );
   }
   else
   {
      strncpy( botskin, skin, 15);
      botskin[15] = 0;  // make sure botskin is null terminated
   }

   for (i = 0; botskin[i] != 0; i++)
      botskin[i] = tolower( botskin[i] );  // convert to all lowercase

   index = 0;

   while ((!found) && (index < MAX_SKINS))
   {
      if (strcmp(botskin, bot_skins[index]) == 0)
         found = TRUE;
      else
         index++;
   }

   if (found == FALSE)
   {
      char err_msg[80];

      sprintf( err_msg, "model \"%s\" is unknown.\n", botskin );
      UTIL_ClientPrintAll( HUD_PRINTNOTIFY, err_msg );
      UTIL_ClientPrintAll( HUD_PRINTNOTIFY,
         "use barney, gina, gman, gordon, helmet, hgrunt,\n");
      UTIL_ClientPrintAll( HUD_PRINTNOTIFY,
         "    recon, robo, scientist, or zombie\n");
      return;
   }

   if ((name != NULL) && (*name != 0))
   {
      strncpy( botname, name, 31 );
      botname[31] = 0;  // make sure botname is null terminated
   }
   else
   {
      strcpy( botname, bot_names[index] );
   }

   skill_level = 0;

   sscanf( skill, "%d", &skill_level );

   if ((skill_level < 1) || (skill_level > 5))
      skill_level = f_botskill;

   skill_level--;  // make 0 based for array index (now 0-4)
   sprintf( c_skill, "%d", skill_level );

   BotEnt = CREATE_FAKE_CLIENT( botname );

   if (FNullEnt( BotEnt ))
      CLIENT_PRINTF( pEntity, print_console, "Max. Players reached.  Can't create bot!\n");
   else
   {
      CLIENT_PRINTF( pEntity, print_console,
         UTIL_VarArgs("Creating bot \"%s\" using model %s with skill=%d\n", botname, botskin, (int)f_botskill) );

      char ptr[128];  // allocate space for message from ClientConnect
      CBot *BotClass;
      char *infobuffer;
      int clientIndex;

      BotClass = GetClassPtr( (CBot *) VARS(BotEnt) );
      infobuffer = GET_INFOBUFFER( BotClass->edict( ) );
      clientIndex = BotClass->entindex( );

      SET_CLIENT_KEY_VALUE( clientIndex, infobuffer, "model", botskin );
      SET_CLIENT_KEY_VALUE( clientIndex, infobuffer, "skill", c_skill );

      ClientConnect( BotClass->edict( ), botname, "127.0.0.1", ptr );
      DispatchSpawn( BotClass->edict( ) );
   }
}


void CBot::Spawn( )
{
   char c_skill[2];

   pev->classname    = MAKE_STRING( "player" );
   pev->health       = 100;
   pev->armorvalue   = 0;
   pev->takedamage   = DAMAGE_AIM;
   pev->solid        = SOLID_SLIDEBOX;
   pev->movetype     = MOVETYPE_WALK;
   pev->max_health   = pev->health;
   pev->flags        = FL_CLIENT | FL_FAKECLIENT;
   pev->air_finished = gpGlobals->time + 12;
   pev->dmg          = 2;     // initial water damage
   pev->effects      = 0;
   pev->deadflag     = DEAD_NO;
   pev->dmg_take     = 0;
   pev->dmg_save     = 0;

   m_bitsHUDDamage   = -1;
   m_bitsDamageType  = 0;
   m_afPhysicsFlags  = 0;
   m_fLongJump       = FALSE; // no longjump module. 
   m_iFOV            = 0;     // init field of view.
   m_iClientFOV      = -1;    // make sure fov reset is sent

   m_flNextDecalTime = 0;     // let this player decal as soon as he spawns.

   // wait a few seconds until user-defined message registrations are recieved by all clients
   m_flgeigerDelay   = gpGlobals->time + 2.0;
   
   m_flTimeStepSound = 0;
   m_iStepLeft       = 0;

   // some monsters use this to determine whether or not the player is looking at them.
   m_flFieldOfView   = 0.5;

   m_bloodColor      = BLOOD_COLOR_RED;
   m_flNextAttack    = gpGlobals->time;
   StartSneaking( );

   m_iFlashBattery   = 99;
   m_flFlashLightTime = 1;    // force first message

   // dont let uninitialized value here hurt the player
   m_flFallVelocity  = 0;

   g_pGameRules->SetDefaultPlayerTeam( this );
   g_pGameRules->GetPlayerSpawnSpot( this );

   SET_MODEL( ENT( pev ), "models/player.mdl" );

   g_ulModelIndexPlayer = pev->modelindex;

   pev->sequence = LookupActivity( ACT_IDLE );

   if (FBitSet( pev->flags, FL_DUCKING )) 
      UTIL_SetSize( pev, VEC_DUCK_HULL_MIN, VEC_DUCK_HULL_MAX );
   else
      UTIL_SetSize( pev, VEC_HULL_MIN, VEC_HULL_MAX );

   pev->view_ofs = VEC_VIEW;

   Precache( );

   m_HackedGunPos = Vector( 0, 32, 0 );

   if (m_iPlayerSound == SOUNDLIST_EMPTY)
   {
      ALERT ( at_console, "Couldn't alloc player sound slot!\n" );
   }

   m_fNoPlayerSound = FALSE;  // normal sound behavior.

   m_pLastItem = NULL;
   m_fInitHUD = TRUE;
   m_iClientHideHUD = -1;     // force this to be recalculated
   m_fWeapon = FALSE;
   m_pClientActiveItem = NULL;
   m_iClientBattery = -1;

   // reset all ammo values to 0
   for ( int i = 0; i < MAX_AMMO_SLOTS; i++ )
   {
      // client ammo values also have to be reset
      // (the death hud clear messages does on the client side)
      m_rgAmmo[i] = 0;
      m_rgAmmoLast[i] = 0;
   }

   m_lastx = m_lasty = 0;

   g_pGameRules->PlayerSpawn( this );

   SetThink( BotThink );
   pev->nextthink = gpGlobals->time + .1;

   pev->ideal_yaw = pev->v_angle.y;
   pev->yaw_speed = BOT_YAW_SPEED;

   v_prev_origin = pev->origin;  // save previous position
   f_shoot_time = 0;

   // set bot's skill level (0=good, 4=bad)
   strcpy(c_skill, GET_INFO_KEY_VALUE(GET_INFOBUFFER(edict( )), "skill") );
   sscanf(c_skill, "%d", &bot_skill);

   f_max_speed = CVAR_GET_FLOAT("sv_maxspeed");
//   f_max_speed = f_max_speed * speed_adjust[bot_skill];

   f_speed_check_time = gpGlobals->time + 1.0;

   ladder_dir = 0;

   if (RANDOM_LONG(1, 100) & 1)
      wander_dir = WANDER_LEFT;
   else
      wander_dir = WANDER_RIGHT;

   f_pause_time = 0;
   f_find_item = 0;

   strcpy(model_name, GET_INFO_KEY_VALUE(GET_INFOBUFFER(edict( )), "model") );

   bot_model = 0;
   if ((strcmp( model_name, "hgrunt" ) == 0) ||
      (strcmp( model_name, "recon" ) == 0))
      bot_model = MODEL_HGRUNT;
   else if (strcmp( model_name, "barney") == 0)
      bot_model = MODEL_BARNEY;
   else if (strcmp( model_name, "scientist") == 0)
      bot_model = MODEL_SCIENTIST;

   for (i = 0; i < 5; i++)
      PRECACHE_SOUND (hgrunt_sounds[i]);

   for (i = 0; i < 5; i++)
      PRECACHE_SOUND (scientist_sounds[i]);

   for (i = 0; i < 5; i++)
      PRECACHE_SOUND (barney_sounds[i]);

   f_pain_time = gpGlobals->time + 5.0;

   pBotEnemy = NULL;
}


void CBot::Killed( entvars_t *pevAttacker, int iGib )
{
   CSound *pSound;

   g_pGameRules->PlayerKilled( this, pevAttacker, g_pevLastInflictor );

   if (m_pTank != NULL)
   {
      m_pTank->Use( this, this, USE_OFF, 0 );
      m_pTank = NULL;
   }

   // this client isn't going to be thinking for a while, so reset the sound
   // until they respawn
   pSound = CSoundEnt::SoundPointerForIndex( CSoundEnt::ClientSoundIndex(edict( )) );
   {
      if (pSound)
      {
         pSound->Reset( );
      }
   }

   SetAnimation( PLAYER_DIE );
   
   pev->modelindex = g_ulModelIndexPlayer;    // don't use eyes

#if !defined(DUCKFIX)
   pev->view_ofs      = Vector( 0, 0, -8 );
#endif
   pev->deadflag      = DEAD_DYING;
   pev->solid         = SOLID_NOT;
//   pev->movetype      = MOVETYPE_TOSS;
   pev->movetype      = MOVETYPE_NONE;

   ClearBits( pev->flags, FL_ONGROUND );

   if (pev->velocity.z < 10)
      pev->velocity.z += RANDOM_FLOAT( 0, 300 );

   // clear out the suit message cache so we don't keep chattering
   SetSuitUpdate( NULL, FALSE, 0 );

   // send "health" update message to zero
   m_iClientHealth = 0;

   MESSAGE_BEGIN( MSG_ONE, gmsgHealth, NULL, pev );
      WRITE_BYTE( m_iClientHealth );
   MESSAGE_END( );

   // Tell Ammo Hud that the player is dead
   MESSAGE_BEGIN( MSG_ONE, gmsgCurWeapon, NULL, pev );
      WRITE_BYTE( 0 );
      WRITE_BYTE( 0xFF );
      WRITE_BYTE( 0xFF );
   MESSAGE_END( );

   // reset FOV
   m_iFOV = m_iClientFOV = 0;

   MESSAGE_BEGIN( MSG_ONE, gmsgSetFOV, NULL, pev );
      WRITE_BYTE( 0 );
   MESSAGE_END( );

   // UNDONE: Put this in, but add FFADE_PERMANENT and make fade time 8.8 instead of 4.12
   // UTIL_ScreenFade( edict( ), Vector( 128, 0, 0 ), 6, 15, 255, FFADE_OUT | FFADE_MODULATE );

   if (( pev->health < -40 && iGib != GIB_NEVER ) || iGib == GIB_ALWAYS)
   {
      GibMonster( );   // This clears pev->model
      pev->effects |= EF_NODRAW;

// TAKE THIS OUT!!! THIS HAPPENS SOMETIMES WHEN HEALTH IS < -40, THEN THE
// THINK FUNCTION DOESN'T EVER GET SET TO PlayerDeathThink
// (should we REALLY be doing this???)
//      return;
   }

   DeathSound( );
   
   pev->angles.x = 0;  // don't let the player tilt
   pev->angles.z = 0;

   SetThink( PlayerDeathThink );
   pev->nextthink = gpGlobals->time + 0.1;
}


void CBot::PlayerDeathThink( void )
{
   float flForward;

   pev->nextthink = gpGlobals->time + 0.1;

   if (FBitSet( pev->flags, FL_ONGROUND ))
   {
      flForward = pev->velocity.Length( ) - 20;
      if (flForward <= 0)
         pev->velocity = g_vecZero;
      else    
         pev->velocity = flForward * pev->velocity.Normalize( );
   }

   if (HasWeapons( ))
   {
      // we drop the guns here because weapons that have an area effect and can kill their user
      // will sometimes crash coming back from CBasePlayer::Killed() if they kill their owner because the
      // player class sometimes is freed. It's safer to manipulate the weapons once we know
      // we aren't calling into any of their code anymore through the player pointer.

      PackDeadPlayerItems( );
   }

   DROP_TO_FLOOR ( ENT(pev) );  // put the body on the ground

   if (pev->modelindex && (!m_fSequenceFinished) && (pev->deadflag == DEAD_DYING))
   {
      StudioFrameAdvance( );

      m_iRespawnFrames++;
      if (m_iRespawnFrames < 60)  // animations should be no longer than this
         return;
   }

   StopAnimation( );

   if (pev->deadflag == DEAD_DYING)
   {
      pev->deadflag = DEAD_DEAD;
   }

   pev->effects |= EF_NOINTERP;
   pev->framerate = 0.0;

   if (pev->deadflag == DEAD_DEAD)
   {
      if (g_pGameRules->FPlayerCanRespawn( this ))
      {
         m_fDeadTime = gpGlobals->time;
         pev->deadflag = DEAD_RESPAWNABLE;
      }
      
      return;
   }

   // check if time to respawn...
   if (gpGlobals->time > (m_fDeadTime + 5.0))
   {

      pev->button = 0;
      m_iRespawnFrames = 0;

      //ALERT( at_console, "Respawn\n" );

      respawn( pev, !(m_afPhysicsFlags & PFLAG_OBSERVER) );

      pev->nextthink = -1;
   }
}


int CBot::TakeDamage( entvars_t *pevInflictor, entvars_t *pevAttacker,
                      float flDamage, int bitsDamageType )
{
   CBaseEntity *pAttacker = CBaseEntity::Instance(pevAttacker);
   char sound[40];
   int ret_damage;

   // do the damage first...
   ret_damage = CBasePlayer::TakeDamage( pevInflictor, pevAttacker, flDamage,
                                         bitsDamageType );

   // if we don't have an enemy and someone is shooting at us then
   // turn in the attacker's direction...
   if (pBotEnemy == NULL)
   {
      // face the enemy
      Vector v_enemy = pAttacker->pev->origin - pev->origin;
      Vector bot_angles = UTIL_VecToAngles( v_enemy );

      pev->ideal_yaw = bot_angles.y;

      // check for wrap around of angle...
      if (pev->ideal_yaw > 180)
         pev->ideal_yaw -= 360;
      if (pev->ideal_yaw < -180)
         pev->ideal_yaw += 360;
   }

   // check if bot model is known, attacker is not a bot,
   // time for pain sound, and bot has some of health left...

   if ((bot_model != 0) && (pAttacker->IsNetClient()) &&
       (f_pain_time <= gpGlobals->time) && (pev->health > 0))
   {
      float distance = (pAttacker->pev->origin - pev->origin).Length( );

      // check if the distance to attacker is close enough (otherwise
      // the attacker is too far away to hear the pain sounds)

      if (distance <= 400)
      {
         if (RANDOM_LONG(1, 10) <= 5)
         {
            f_pain_time = gpGlobals->time + 5.0;

            if (bot_model == MODEL_HGRUNT)
               sprintf( sound, hgrunt_sounds[RANDOM_LONG(0,4)] );
            else if (bot_model == MODEL_BARNEY)
               sprintf( sound, barney_sounds[RANDOM_LONG(0,4)] );
            else if (bot_model == MODEL_SCIENTIST)
               sprintf( sound, scientist_sounds[RANDOM_LONG(0,4)] );

            EMIT_SOUND(ENT(pevAttacker), CHAN_VOICE, sound,
                       RANDOM_FLOAT(0.9, 1.0), ATTN_NORM);
         }
      }
   }

   return ret_damage;
}


void CBot::BotChangeYaw( float speed )
{
   float ideal;
   float current;
   float current_180;  // current +/- 180 degrees
   float diff;
   
   current = pev->v_angle.y;
   ideal = pev->ideal_yaw;

   diff = abs(current - ideal);

   if (diff < 1.0)
      return;

   if (diff < speed)
      speed = diff;

   if ((current >= 0) && (ideal >= 0))
   {
      if (current > ideal)
         current -= speed;
      else
         current += speed;
   }
   else if ((current >= 0) && (ideal < 0))
   {
      current_180 = current - 180;

      if (current_180 > ideal)
         current += speed;
      else
         current -= speed;
   }
   else if ((current < 0) && (ideal >= 0))
   {
      current_180 = current + 180;
      if (current_180 > ideal)
         current += speed;
      else
         current -= speed;
   }
   else  // (current < 0) && (ideal < 0)
   {
      if (current > ideal)
         current -= speed;
      else
         current += speed;
   }

   // check for wrap around of angle...
   if (current > 180)
      current -= 360;
   if (current < -180)
      current += 360;

   pev->v_angle.y = current;

   return;
}


void CBot::BotOnLadder( void )
{
   // moves the bot up or down a ladder.  if the bot can't move
   // (i.e. get's stuck with someone else on ladder), the bot will
   // change directions and go the other way on the ladder.

   if (ladder_dir == LADDER_UP)  // are we currently going up?
   {
      pev->v_angle.x = -60;  // look upwards

      Vector v_diff = v_prev_origin - pev->origin;

      // check if we haven't moved much since the last location...
      if (v_diff.Length() <= 1)
      {
         // we must be stuck, change directions...

         pev->v_angle.x = 60;  // look downwards
         ladder_dir = LADDER_DOWN;
      }
   }
   else if (ladder_dir == LADDER_DOWN)  // are we currently going down?
   {
      pev->v_angle.x = 60;  // look downwards

      Vector v_diff = v_prev_origin - pev->origin;

      // check if we haven't moved much since the last location...
      if (v_diff.Length() <= 1)
      {
         // we must be stuck, change directions...

         pev->v_angle.x = -60;  // look upwards
         ladder_dir = LADDER_UP;
      }
   }
   else  // we haven't picked a direction yet, try going up...
   {
      pev->v_angle.x = -60;  // look upwards
      ladder_dir = LADDER_UP;
   }

   // move forward (i.e. in the direction we are looking, up or down)
   pev->button |= IN_FORWARD;

   f_move_speed = f_max_speed;
}


CBaseEntity * CBot::BotFindEnemy( void )
{
   if (pBotEnemy != NULL)  // do we already have an enemy?
   {
      if (!pBotEnemy->IsAlive( ))
      {
         // the enemy is dead, jump for joy every once in a while...
         if (RANDOM_LONG(1, 10) == 1)
            pev->button |= IN_JUMP;
      }
      else if (FVisible( pBotEnemy ) && FInViewCone( pBotEnemy ))
      {
         // if enemy is still visible and in field of view, keep it

         // face the enemy
         Vector v_enemy = pBotEnemy->pev->origin - pev->origin;
         Vector bot_angles = UTIL_VecToAngles( v_enemy );

         pev->ideal_yaw = bot_angles.y;

         // check for wrap around of angle...
         if (pev->ideal_yaw > 180)
            pev->ideal_yaw -= 360;
         if (pev->ideal_yaw < -180)
            pev->ideal_yaw += 360;

         return (pBotEnemy);
      }
   }

   int i;
   float nearestdistance = 1000;
   CBaseEntity *pNewEnemy = NULL;

   // search the world for players...
   for (i = 1; i <= gpGlobals->maxClients; i++)
   {
      CBaseEntity *pPlayer = UTIL_PlayerByIndex( i );

      // skip invalid players and skip self (i.e. this bot)
      if ((!pPlayer) || (pPlayer == this))
         continue;

      // skip this player if not alive (i.e. dead or dying)
      if (pPlayer->pev->deadflag != DEAD_NO)
         continue;

      // skip players that are netclients (i.e. not bot, observer mode)
      if (pPlayer->IsNetClient() && f_Observer)
         continue;

      // see if you can see the player (within your field of view)...
      if (FVisible( pPlayer ) && FInViewCone( pPlayer ))
      {
         float distance = (pPlayer->pev->origin - pev->origin).Length( );
         if (distance < nearestdistance)
         {
            nearestdistance = distance;
            pNewEnemy = pPlayer;
         }
      }
   }

   if (pNewEnemy)
   {
      // face the enemy
      Vector v_enemy = pNewEnemy->pev->origin - pev->origin;
      Vector bot_angles = UTIL_VecToAngles( v_enemy );

      pev->ideal_yaw = bot_angles.y;

      // check for wrap around of angle...
      if (pev->ideal_yaw > 180)
         pev->ideal_yaw -= 360;
      if (pev->ideal_yaw < -180)
         pev->ideal_yaw += 360;
   }

   return (pNewEnemy);
}


Vector CBot::BotBodyTarget( CBaseEntity *pBotEnemy )
{
   Vector target;

   switch (bot_skill)
   {
      case 0:
         // VERY GOOD, same as from CBasePlayer::BodyTarget (in player.h)
         target = pBotEnemy->Center( ) + pBotEnemy->pev->view_ofs * RANDOM_FLOAT( 0.5, 1.1 );
         break;
      case 1:
         // GOOD, offset a little for x, y, and z
         target = pBotEnemy->Center( ) + pBotEnemy->pev->view_ofs +
                  Vector( RANDOM_LONG(-5, 5), RANDOM_LONG(-5, 5), RANDOM_LONG(-9, 9));
         break;
      case 2:
         // FAIR, offset somewhat for x, y, and z
         target = pBotEnemy->Center( ) + pBotEnemy->pev->view_ofs +
                  Vector( RANDOM_LONG(-9, 9), RANDOM_LONG(-9, 9), RANDOM_LONG(-15, 15));
         break;
      case 3:
         // POOR, offset for x, y, and z
         target = pBotEnemy->Center( ) + pBotEnemy->pev->view_ofs +
                  Vector( RANDOM_LONG(-16, 16), RANDOM_LONG(-16, 16), RANDOM_LONG(-20, 20));
         break;
      case 4:
         // BAD, offset lots for x, y, and z
         target = pBotEnemy->Center( ) + pBotEnemy->pev->view_ofs +
                  Vector( RANDOM_LONG(-20, 20), RANDOM_LONG(-20, 20), RANDOM_LONG(-27, 27));
         break;
   }

   return target;
}


void CBot::BotFireWeapon( float distance )
{
   CBasePlayerItem *weapon_ptr[MAX_WEAPONS];  // pointer array to weapons
   CBasePlayerItem *new_weapon;
   int primary_ammo[MAX_WEAPONS];
   int secondary_ammo[MAX_WEAPONS];
   int i;

   // check if bot can't switch weapons right now...
   if (!m_pActiveItem->CanHolster( ))
      return;

   for (i = 0; i < MAX_WEAPONS; i++)
   {
      weapon_ptr[i] = NULL;
      primary_ammo[i] = 0;
      secondary_ammo[i] = 0;
   }

   for (i = 0; i < MAX_ITEM_TYPES; i++)
   {
      CBasePlayerItem *pItem = NULL;

      if (m_rgpPlayerItems[i])
      {
         pItem = m_rgpPlayerItems[i];
         while (pItem)
         {
            weapon_ptr[pItem->m_iId] = pItem;  // store pointer to item

            pItem = pItem->m_pNext;
         }
      }
   }

   for (i = 0; i < MAX_AMMO_SLOTS; i++)
   {
      if (!CBasePlayerItem::AmmoInfoArray[i].pszName)
         continue;

      if (strcmp("9mm", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
      {
         primary_ammo[WEAPON_GLOCK] = m_rgAmmo[i];
         secondary_ammo[WEAPON_GLOCK] = m_rgAmmo[i];
         primary_ammo[WEAPON_MP5] = m_rgAmmo[i];
      }
      else if (strcmp("357", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
      {
         primary_ammo[WEAPON_PYTHON] = m_rgAmmo[i];
         secondary_ammo[WEAPON_PYTHON] = m_rgAmmo[i];
      }
      else if (strcmp("ARgrenades", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
         secondary_ammo[WEAPON_MP5] = m_rgAmmo[i];
      else if (strcmp("bolts", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
      {
         primary_ammo[WEAPON_CROSSBOW] = m_rgAmmo[i];
         secondary_ammo[WEAPON_CROSSBOW] = m_rgAmmo[i];
      }
      else if (stricmp("buckshot", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
      {
         primary_ammo[WEAPON_SHOTGUN] = m_rgAmmo[i];
         secondary_ammo[WEAPON_SHOTGUN] = m_rgAmmo[i];
      }
      else if (stricmp("rockets", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
         primary_ammo[WEAPON_RPG] = m_rgAmmo[i];
      else if (strcmp("uranium", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
      {
         primary_ammo[WEAPON_GAUSS] = m_rgAmmo[i];
         primary_ammo[WEAPON_EGON] = m_rgAmmo[i];
      }
      else if (stricmp("Hornets", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
         primary_ammo[WEAPON_HORNETGUN] = m_rgAmmo[i];
      else if (stricmp("Hand Grenade", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
         primary_ammo[WEAPON_HANDGRENADE] = m_rgAmmo[i];
      else if (stricmp("Trip Mine", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
         primary_ammo[WEAPON_TRIPMINE] = m_rgAmmo[i];
      else if (stricmp("Satchel Charge", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
         primary_ammo[WEAPON_SATCHEL] = m_rgAmmo[i];
      else if (stricmp("Snarks", CBasePlayerItem::AmmoInfoArray[i].pszName) == 0)
         primary_ammo[WEAPON_SNARK] = m_rgAmmo[i];
   }


   // if close to enemy, and skill level is 1, 2 or 3, use the crowbar

   if ((distance <= 40) && (pev->weapons & (1<<WEAPON_CROWBAR)) &&
       (bot_skill <= 2))
   {
      new_weapon = weapon_ptr[WEAPON_CROWBAR];

      // check if we aren't already using this item...
      if (m_pActiveItem != new_weapon)
         SelectItem("weapon_crowbar");  // select the crowbar

      pev->button |= IN_ATTACK;  // use primary attack (whack! whack!)

      // set next time to "shoot"
      f_shoot_time = gpGlobals->time + 0.3 + 
         RANDOM_FLOAT(fire_delay[WEAPON_CROWBAR][bot_skill][0],
                      fire_delay[WEAPON_CROWBAR][bot_skill][1]);
      return;
   }

   // if the bot has the egon gun...

   if (pev->weapons & (1<<WEAPON_EGON))
   {
      new_weapon = weapon_ptr[WEAPON_EGON];

      // check if we have any ammo left for this weapon...
      if (primary_ammo[WEAPON_EGON] > 0)
      {
         // check if we aren't already using this item...
         if (m_pActiveItem != new_weapon)
            SelectItem("weapon_egon");  // select the egon gun

         pev->button |= IN_ATTACK;  // use primary attack (bang! bang!)

         // set next time to shoot
         f_shoot_time = gpGlobals->time;

         return;
      }
   }

   // if the bot has the gauss gun...

   if (pev->weapons & (1<<WEAPON_GAUSS))
   {
      new_weapon = weapon_ptr[WEAPON_GAUSS];

      // check if we have any ammo left for this weapon...
      if (primary_ammo[WEAPON_GAUSS] > 1)
      {
         // check if we aren't already using this item...
         if (m_pActiveItem != new_weapon)
            SelectItem("weapon_gauss");  // select the gauss gun

         pev->button |= IN_ATTACK;  // use primary attack (bang! bang!)

         // set next time to shoot
         f_shoot_time = gpGlobals->time + 0.2 +
            RANDOM_FLOAT(fire_delay[WEAPON_GAUSS][bot_skill][0],
                         fire_delay[WEAPON_GAUSS][bot_skill][1]);

         return;
      }
   }

   // if close enough for good shotgun blasts and bot has the shotgun...

   if ((distance > 30) && (distance < 150) &&
       (pev->weapons & (1<<WEAPON_SHOTGUN)))
   {
      new_weapon = weapon_ptr[WEAPON_SHOTGUN];

      // check if we have any ammo left for this weapon...
      if (primary_ammo[WEAPON_SHOTGUN] > 0)
      {
         // check if we aren't already using this item...
         if (m_pActiveItem != new_weapon)
            SelectItem("weapon_shotgun");  // select the shotgun

         pev->button |= IN_ATTACK;  // use primary attack (bang! bang!)

         // set next time to shoot
         f_shoot_time = gpGlobals->time + 0.75 +
            RANDOM_FLOAT(fire_delay[WEAPON_SHOTGUN][bot_skill][0],
                         fire_delay[WEAPON_SHOTGUN][bot_skill][1]);

         return;
      }
   }

   // if close enough for 357 shot and bot has the 357 (PYTHON)...

   if ((distance > 30) && (distance < 700) &&
       (pev->weapons & (1<<WEAPON_PYTHON)))
   {
      new_weapon = weapon_ptr[WEAPON_PYTHON];

      // check if we have any ammo left for this weapon...
      if (primary_ammo[WEAPON_PYTHON] > 0)
      {
         // check if we aren't already using this item...
         if (m_pActiveItem != new_weapon)
            SelectItem("weapon_357");  // select the 357 python

         pev->button |= IN_ATTACK;  // use primary attack (bang! bang!)

         // set next time to shoot
         f_shoot_time = gpGlobals->time + 0.75 +
            RANDOM_FLOAT(fire_delay[WEAPON_PYTHON][bot_skill][0],
                         fire_delay[WEAPON_PYTHON][bot_skill][1]);

         return;
      }
   }

   // if close enough for hornet gun and bot has the hornet gun...

   if ((distance > 30) && (distance < 1000) &&
       (pev->weapons & (1<<WEAPON_HORNETGUN)))
   {
      new_weapon = weapon_ptr[WEAPON_HORNETGUN];

      // check if we have any ammo left for this weapon...
      if (primary_ammo[WEAPON_HORNETGUN] > 0)
      {
         // check if we aren't already using this item...
         if (m_pActiveItem != new_weapon)
            SelectItem("weapon_hornetgun");  // select the shotgun

         pev->button |= IN_ATTACK;  // use primary attack (bang! bang!)

         // set next time to shoot
         f_shoot_time = gpGlobals->time + 0.25 +
            RANDOM_FLOAT(fire_delay[WEAPON_HORNETGUN][bot_skill][0],
                         fire_delay[WEAPON_HORNETGUN][bot_skill][1]);

         return;
      }
   }

   // if close enough for good MP5 shot and the bot has the MP5...

   if ((distance < 250) && (pev->weapons & (1<<WEAPON_MP5)))
   {
      new_weapon = weapon_ptr[WEAPON_MP5];

      // check if we have any ammo left for this weapon...
      if (primary_ammo[WEAPON_MP5] > 0)
      {
         // check if we aren't already using this item...
         if (m_pActiveItem != new_weapon)
            SelectItem("weapon_9mmAR");  // select the 9mmAR (MP5)

         pev->button |= IN_ATTACK;  // use primary attack (bang! bang!)

         // set next time to shoot
         f_shoot_time = gpGlobals->time + 0.1 +
            RANDOM_FLOAT(fire_delay[WEAPON_MP5][bot_skill][0],
                         fire_delay[WEAPON_MP5][bot_skill][1]);

         return;
      }
   }

   // not too close for crossbow and not too far and bot has the crossbow...

   if ((distance > 100) && (distance < 1000) &&
       (pev->weapons & (1<<WEAPON_CROSSBOW)))
   {
      new_weapon = weapon_ptr[WEAPON_CROSSBOW];

      // check if we have any ammo left for this weapon...
      if (primary_ammo[WEAPON_CROSSBOW] > 0)
      {
         // check if we aren't already using this item...
         if (m_pActiveItem != new_weapon)
            SelectItem("weapon_crossbow");  // select the crossbow

         pev->button |= IN_ATTACK;  // use primary attack (bang! bang!)

         // set next time to shoot
         f_shoot_time = gpGlobals->time + 0.75 +
            RANDOM_FLOAT(fire_delay[WEAPON_CROSSBOW][bot_skill][0],
                         fire_delay[WEAPON_CROSSBOW][bot_skill][1]);

         return;
      }
   }

   // don't use the RPG unless the enemy is pretty far away...

   if ((distance > 300) && (pev->weapons & (1<<WEAPON_RPG)))
   {
      new_weapon = weapon_ptr[WEAPON_RPG];

      // check if we have any ammo left for this weapon...
      if (primary_ammo[WEAPON_RPG] > 0)
      {
         // check if we aren't already using this item...
         if (m_pActiveItem != new_weapon)
            SelectItem("weapon_rpg");  // select the RPG rocket launcher

         pev->button |= IN_ATTACK;  // use primary attack (bang! bang!)

         // set next time to shoot
         f_shoot_time = gpGlobals->time + 1.5 +
            RANDOM_FLOAT(fire_delay[WEAPON_RPG][bot_skill][0],
                         fire_delay[WEAPON_RPG][bot_skill][1]);

         return;
      }
   }

   // if nothing else was selected, try the good ol' 9mm glock...

   if ((distance < 1200) && (pev->weapons & (1<<WEAPON_GLOCK)))
   {
      new_weapon = weapon_ptr[WEAPON_GLOCK];

      // check if we have any ammo left for this weapon...
      if (primary_ammo[WEAPON_GLOCK] > 0)
      {
         // check if we aren't already using this item...
         if (m_pActiveItem != new_weapon)
            SelectItem("weapon_9mmhandgun");  // select the trusty 9mm glock

         pev->button |= IN_ATTACK;  // use primary attack (bang! bang!)

         // set next time to shoot
         f_shoot_time = gpGlobals->time + 0.3 +
            RANDOM_FLOAT(fire_delay[WEAPON_GLOCK][bot_skill][0],
                         fire_delay[WEAPON_GLOCK][bot_skill][1]);

         return;
      }
   }

   return;
}


void CBot::BotShootAtEnemy( void )
{
   float f_distance;

   // aim for the head and/or body
   Vector v_enemy = BotBodyTarget( pBotEnemy ) - GetGunPosition( );
   pev->v_angle = UTIL_VecToAngles( v_enemy );

   pev->ideal_yaw = pev->v_angle.y;

   // check for wrap around of angle...
   if (pev->ideal_yaw > 180)
      pev->ideal_yaw -= 360;
   if (pev->ideal_yaw < -180)
      pev->ideal_yaw += 360;

   pev->v_angle.x = -pev->v_angle.x;  //adjust pitch to point gun

   v_enemy.z = 0;  // ignore z component (up & down)

   f_distance = v_enemy.Length( );  // how far away is the enemy scum?

   if (f_distance > 200)      // run if distance to enemy is far
      f_move_speed = f_max_speed;
   else if (f_distance > 20)  // walk if distance is closer
      f_move_speed = f_max_speed / 2;
   else                     // don't move if close enough
      f_move_speed = 0.0;

   // is it time to shoot yet?
   if (f_shoot_time <= gpGlobals->time)
   {
      // select the best weapon to use at this distance and fire...
      BotFireWeapon( f_distance );
   }
}


void CBot::BotFindItem( void )
{
   CBaseEntity *pEntity = NULL;
   CBaseEntity *pPickupEntity = NULL;
   float radius = 300;
   BOOL can_pickup = FALSE;
   float min_distance;   

   min_distance = radius + 1.0;

   while ((pEntity = UTIL_FindEntityInSphere( pEntity, pev->origin, radius )) != NULL)
   {
      // check to make sure that the entity is visible and in field of view...
      if (FVisible( pEntity ) && FInViewCone( pEntity ))
      {
         // check if entity is a weapon...
         if (strncmp("weapon_", STRING(pEntity->pev->classname), 7) == 0)
         {
            CBasePlayerWeapon *pWeapon = (CBasePlayerWeapon *)pEntity;

            if ((pWeapon->m_pPlayer) || (pWeapon->pev->effects & EF_NODRAW))
            {
               // someone owns this weapon or it hasn't respawned yet
               continue;
            }

            if (g_pGameRules->CanHavePlayerItem( this, pWeapon ))
            {
               can_pickup = TRUE;
            }
         }

         // check if entity is ammo...
         else if (strncmp("ammo_", STRING(pEntity->pev->classname), 5) == 0)
         {
            CBasePlayerAmmo *pAmmo = (CBasePlayerAmmo *)pEntity;
            char ammo_name[40];
            BOOL ammo_found = FALSE;
            int i;

            // check if the item is not visible (i.e. has not respawned)
            if (pAmmo->pev->effects & EF_NODRAW)
               continue;

            strcpy(ammo_name, STRING(pEntity->pev->classname));

            i = 0;
            while (ammo_check[i].ammo_name[0])
            {
               if (strcmp(ammo_check[i].ammo_name, ammo_name) == 0)
               {
                  ammo_found = TRUE;

                  // see if we can pick up this item...
                  if (g_pGameRules->CanHaveAmmo( this,
                      ammo_check[i].weapon_name, ammo_check[i].max_carry))
                  {
                     can_pickup = TRUE;
                     break;
                  }
               }

               i++;
            }
            if (ammo_found == FALSE)
            {
               sprintf(message, "unknown ammo: %s\n", ammo_name);
               BotDebug(message);
            }
         }

         // check if entity is a battery...
         else if (strcmp("item_battery", STRING(pEntity->pev->classname)) == 0)
         {
            CItem *pBattery = (CItem *)pEntity;

            // check if the item is not visible (i.e. has not respawned)
            if (pBattery->pev->effects & EF_NODRAW)
               continue;

            // check if the bot can use this item...
            if ((pev->armorvalue < MAX_NORMAL_BATTERY) &&
                (pev->weapons & (1<<WEAPON_SUIT)))
            {
               can_pickup = TRUE;
            }
         }

         // check if entity is a healthkit...
         else if (strcmp("item_healthkit", STRING(pEntity->pev->classname)) == 0)
         {
            CItem *pHealthKit = (CItem *)pEntity;

            // check if the item is not visible (i.e. has not respawned)
            if (pHealthKit->pev->effects & EF_NODRAW)
               continue;

            // check if the bot can use this item...
            if (pev->health <= 90)
            {
               can_pickup = TRUE;
            }
         }

         // check if entity is a packed up weapons box...
         else if (strcmp("weaponbox", STRING(pEntity->pev->classname)) == 0)
         {
            can_pickup = TRUE;
         }

         // check if entity is the spot from RPG laser
         else if (strcmp("laser_spot", STRING(pEntity->pev->classname)) == 0)
         {
         }

         // check if entity is an armed tripmine
         else if (strcmp("monster_tripmine", STRING(pEntity->pev->classname)) == 0)
         {
         }

         // check if entity is an armed tripmine beam
         else if (strcmp("beam", STRING(pEntity->pev->classname)) == 0)
         {
         }

         // check if entity is an armed satchel charge
         else if (strcmp("monster_satchel", STRING(pEntity->pev->classname)) == 0)
         {
         }

         // check if entity is a snark (squeak grenade)
         else if (strcmp("monster_snark", STRING(pEntity->pev->classname)) == 0)
         {
         }
      }

      if (can_pickup) // if we found something we can pickup...
      {
         float distance = (pEntity->pev->origin - pev->origin).Length( );

         // see if it's the closest item so far...
         if (distance < min_distance)
         {
            min_distance = distance;
            pPickupEntity = pEntity;  // remember this entity
         }
      }
   }

   if (pPickupEntity != NULL)
   {
      // let's head off toward that item...
      Vector v_item = pPickupEntity->pev->origin - pev->origin;

      Vector bot_angles = UTIL_VecToAngles( v_item );

      pev->ideal_yaw = bot_angles.y;

      // check for wrap around of angle...
      if (pev->ideal_yaw > 180)
         pev->ideal_yaw -= 360;
      if (pev->ideal_yaw < -180)
         pev->ideal_yaw += 360;
   }
}


void CBot::BotThink( void )
{
   if (!IsInWorld( ))
   {
      SetTouch( NULL );
      UTIL_Remove( this );
      return;
   }

   if (!IsAlive( ))
   {
      // if bot is dead, don't take up any space in world (set size to 0)
      UTIL_SetSize( pev, Vector(0, 0, 0), Vector(0, 0, 0));

      return;
   }

   // see if it's time to check for sv_maxspeed change...
   if (f_speed_check_time <= gpGlobals->time)
   {
      // store current sv_maxspeed setting for quick access
      f_max_speed = CVAR_GET_FLOAT("sv_maxspeed");
//      f_max_speed = f_max_speed * speed_adjust[bot_skill];

      // set next check time to 2 seconds from now
      f_speed_check_time = gpGlobals->time + 2.0;
   }

   pev->button = 0;  // make sure no buttons are pressed

   // turn towards ideal_yaw by yaw_speed degrees
   BotChangeYaw( pev->yaw_speed );

   if (IsOnLadder( ))  // check if the bot is on a ladder...
   {
      BotOnLadder( );  // go handle the ladder movement
   }
   else
   {
      // bot is not on a ladder so clear ladder direction flag...
      ladder_dir = 0;

      if (abs(pev->v_angle.y - pev->ideal_yaw) < pev->yaw_speed)
      {
         pBotEnemy = BotFindEnemy( );

         if (pBotEnemy != NULL)  // does an enemy exist?
         {
            BotShootAtEnemy( );  // shoot at the enemy
         }
         else if (f_pause_time > gpGlobals->time)  // are we "paused"?
         {
            // you could make the bot look left then right, or look up
            // and down, to make it appear that the bot is hunting for
            // something (don't do anything right now)

            f_move_speed = 0;
         }
         else
         {
            // no enemy, let's just wander around...

            // check if we should look for items now or not...
            if (f_find_item < gpGlobals->time)
            {
               BotFindItem( );  // see if there are any visible items
            }

            pev->v_angle.x = 0;  // reset pitch to 0 (level horizontally)
            pev->v_angle.z = 0;  // reset roll to 0 (straight up and down)

            Vector v_diff = v_prev_origin - pev->origin;

            // check if we haven't moved much since the last location...
            if (v_diff.Length() <= 1)
            {
               // we must be stuck!

               // turn randomly between 10 and 30 degress
               if (wander_dir == WANDER_LEFT)
                  pev->ideal_yaw += RANDOM_LONG(10, 30);
               else
                  pev->ideal_yaw -= RANDOM_LONG(10, 30);

               // check for wrap around of angle...
               if (pev->ideal_yaw > 180)
                  pev->ideal_yaw -= 360;
               if (pev->ideal_yaw < -180)
                  pev->ideal_yaw += 360;

               // also don't look for items for 5 seconds since we could be
               // stuck trying to get to an item
               f_find_item = gpGlobals->time + 5.0;
            }

            f_move_speed = f_max_speed;  // run like the wind!!!

            if (RANDOM_LONG(1, 1000) <= 4)  // should we pause for a while here?
            {
               // set the time that the bot will stop "pausing"
               f_pause_time = gpGlobals->time + RANDOM_FLOAT(0.5, 1.0);
            }
         }
      }
      else
      {
         // if bot is still turning, turn in place by setting speed to 0
         f_move_speed = 0;
      }
   }

   v_prev_origin = pev->origin;  // save current position as previous

//   g_engfuncs.pfnRunPlayerMove(<bot edict>, <bot's view angles>,
//             <forward move>, <side move>, <up move> (does nothing?),
//             <buttons> (IN_ATTACK, etc.), <impulse number>, <msec value>);

   g_engfuncs.pfnRunPlayerMove( edict( ), pev->v_angle, f_move_speed,
                                0, 0, pev->button, 0, 50 );
}

