/*
 * Descent II-a-like network game join menu
 * Arne de Bruijn, 1998
 */

#include "inferno.h" // for settings
#ifdef NETWORK

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "types.h"
#include "timer.h"
#include "gr.h"
#include "palette.h"
#include "inferno.h"
#include "mono.h"
#include "key.h"
#include "error.h"
#include "netmisc.h"
#include "network.h"
#include "ipx.h"
#include "game.h"
#include "multi.h"
#include "text.h"

#include "gamefont.h"
//#include "u_mem.h"

#define LINE_ITEMS 8
#define MAX_TEXT_LEN 25
// from network.c
extern int Network_games_changed;
extern netgame_info Active_games[MAX_ACTIVE_NETGAMES];
extern int num_active_games;
extern int Network_socket;
void network_listen();
void network_send_game_list_request();
// bitblt.c
void gr_bm_bitblt(int w, int h, int dx, int dy, int sx, int sy, grs_bitmap * src, grs_bitmap * dest);

typedef struct bkg {
  grs_canvas * menu_canvas;
  grs_bitmap * saved;			// The background under the menu.
  grs_bitmap * background;
} bkg;

extern grs_bitmap nm_background;

struct line_item {
	int x, y;
	int width;
	char *value;
};

void network_draw_game(const netgame_info *game) {
}

char *network_mode_text(const netgame_info *game) {
	static char *names[4]={"ANRCHY", "GTEAM", "ROBO", "COOP"};
#ifndef SHAREWARE
	if (game->gamemode >= 4 ||
	    (game->protocol_version != MULTI_PROTO_VERSION 
		 #ifdef MULTI_PROTO_D1X_VER
	     &&
	     (game->protocol_version != MULTI_PROTO_D1X_VER ||
	      game->required_subprotocol > MULTI_PROTO_D1X_MINOR)
		 #endif
	    ))
	    return "UNSUP";
#endif
	return names[game->gamemode];
}

char *network_status_text(const netgame_info *game, int activeplayers) {
	switch (game->game_status) {
		case NETSTAT_STARTING:
			return "Forming";
		case NETSTAT_PLAYING:
			if (game->game_flags & NETGAME_FLAG_CLOSED)
				return "Closed";
			else if (activeplayers == game->max_numplayers)
				return "Full";
			else
				return "Open";
		default:
			return "Between";
	}
}

static void network_update_item(const netgame_info *game, struct line_item li[]) {
	int i, activeplayers = 0;
	// get number of active players
	for (i = 0; i < game->numplayers; i++)
		if (game->players[i].connected)
			activeplayers++;
	#ifdef MULTI_PROTO_D1X_VER
	strcpy(li[1].value, game->protocol_version == MULTI_PROTO_D1X_VER ? "+" : "");
	#else
	li[1].value[0] = 0;
	#endif
	strcpy(li[2].value, game->game_name);
	strcpy(li[3].value, network_mode_text(game));

	sprintf(li[4].value, "%d/%d", activeplayers, game->max_numplayers);
#ifndef SHAREWARE
	strcpy(li[5].value, game->mission_title);
#else
	strcpy(li[5].value, "Descent: First Strike");
#endif
	if (game->levelnum < 0)
		sprintf(li[6].value, "S%d", -game->levelnum);
	else	
		sprintf(li[6].value, "%d", game->levelnum);
	strcpy(li[7].value, network_status_text(game, activeplayers));
}

static void update_items(struct line_item lis[MAX_ACTIVE_NETGAMES][LINE_ITEMS]) {
	int i, j;

	for (i = 0; i < MAX_ACTIVE_NETGAMES; i++) {
		if (i >= num_active_games) {
			for (j = 1; j < LINE_ITEMS; j++)
				lis[i][j].value[0] = 0;
		} else
			network_update_item(&Active_games[i], lis[i]);
	}
}

int network_menu_width[LINE_ITEMS] = { 10, 6, 72, 37, 38, 66, 25, 40 };
char *network_menu_title[LINE_ITEMS] = { "", "", "Game", "Mode", "#Plrs", "Mission", "Lev", "Status" };

static int selected_game;

static void draw_back(bkg *b, int x, int y, int w, int h) {
       gr_bm_bitblt(b->background->bm_w-15, h, 5, y, 5, y, b->background, &(grd_curcanv->cv_bitmap) );
}

static void draw_item(bkg *b, struct line_item *li, int is_current) {
	int i, w, h, aw, max_w, pad_w, y;
	char str[MAX_TEXT_LEN], *p;

	y = li[0].y;
	if (is_current)
		gr_set_fontcolor(BM_XRGB(31, 27, 6), -1);
	else
		gr_set_fontcolor(BM_XRGB(17, 17, 26), -1);
	gr_get_string_size(" ...", &pad_w, &h, &aw);
	draw_back(b, li[0].x, y, 
	 li[LINE_ITEMS-1].x + li[LINE_ITEMS-1].width - li[0].x, h);
	for (i = 0; i < LINE_ITEMS; i++) {
		strcpy(str, li[i].value);
		gr_get_string_size(str, &w, &h, &aw);
		if (w > li[i].width) {
			max_w = li[i].width - pad_w;
			p = str + strlen(str);
			while (p > str && w > max_w) {
				*(--p) = 0;
				gr_get_string_size(str, &w, &h, &aw);
			}
			strcpy(p, " ...");
		}
		gr_ustring(li[i].x, y, str);
	}
}

static void draw_list(bkg *bg,
 struct line_item lis[MAX_ACTIVE_NETGAMES][LINE_ITEMS]) {
	int i;
	update_items(lis);
	for (i = 0; i < MAX_ACTIVE_NETGAMES; i++) {
		draw_item(bg, lis[i], i == selected_game);
	}
}

static void init_background(bkg *bg, int x, int y, int w, int h) {
	bg->menu_canvas = gr_create_sub_canvas( &grd_curscreen->sc_canvas, x, y, w, h );
	gr_set_current_canvas( bg->menu_canvas );

	// Save the background under the menu...
	bg->saved = gr_create_bitmap( w, h );
	Assert( bg->saved != NULL );
	gr_bm_bitblt(w, h, 0, 0, 0, 0, &grd_curcanv->cv_bitmap, bg->saved );
	gr_set_current_canvas( NULL );
	nm_draw_background(x,y,x+w-1,y+h-1);
	bg->background = gr_create_sub_bitmap(&nm_background,0,0,w,h);
	gr_set_current_canvas( bg->menu_canvas );
}

static void done_background(bkg *bg) {
	gr_set_current_canvas(bg->menu_canvas);
	gr_bitmap(0, 0, bg->saved); 	
	gr_free_bitmap(bg->saved);
	free( bg->background );
	gr_free_sub_canvas( bg->menu_canvas );
}

int network_join_game_menu() {
	char menu_text[MAX_ACTIVE_NETGAMES][LINE_ITEMS][MAX_TEXT_LEN];
	struct line_item lis[MAX_ACTIVE_NETGAMES][LINE_ITEMS];
	int i, j, k;
	int old_select, old_socket, done, last_num_games;
	grs_canvas * save_canvas;
	grs_font * save_font;
	bkg bg;
	fix t, req_timer = 0;
	
	selected_game = 0;
    
	gr_palette_fade_in( gr_palette, 32, 0 );
	save_canvas = grd_curcanv;
	gr_set_current_canvas( NULL );			
	save_font = grd_curcanv->cv_font;

	init_background(&bg, 0, 7, grd_curcanv->cv_bitmap.bm_w,
	 grd_curcanv->cv_bitmap.bm_h - 14);

	game_flush_inputs();

	gr_set_curfont(Gamefonts[GFONT_BIG_1]);
	gr_string(0x8000, 22, "Netgames");

	// draw titles
	gr_set_curfont(Gamefonts[GFONT_SMALL]);
	gr_set_fontcolor(BM_XRGB(27, 27, 27), -1);
	k = 15;
	for (j = 0; j < LINE_ITEMS; j++) {
		gr_ustring(k, 61, network_menu_title[j]);
		k += network_menu_width[j];
	}

	for (i = 0; i < MAX_ACTIVE_NETGAMES; i++) {
		struct line_item *li = lis[i];

		k = 15;
		for (j = 0; j < LINE_ITEMS; j++) {
			li[j].x = k;
			li[j].y = 70 + i * 9;
			li[j].width = network_menu_width[j] - (j > 1 ? 4 : 0); // HACK!
			k += network_menu_width[j];
			li[j].value = menu_text[i][j];
		}
		sprintf(li[0].value, "%d.", i + 1); 
	}

	Network_games_changed = 1;
	old_socket = -32768;
	old_select = -1;
	last_num_games = 0;
	if ( gr_palette_faded_out ) {
	    gr_palette_fade_in( gr_palette, 32, 0 );
	}

	done = 0;
	while (!done) {
		if (Network_socket != old_socket) {
			gr_set_fontcolor(BM_XRGB(27, 27, 27), -1);
			draw_back(&bg, 30, 52, 250, 9);
			gr_printf(30, 52, "Current IPX socket is default %+d "
			 "(PgUp/PgDn to change)", Network_socket);
			if (old_socket != -32768) { /* changed by user? */
				network_listen();
				ipx_change_default_socket( IPX_DEFAULT_SOCKET + Network_socket );
				num_active_games = 0;
			}
			req_timer -= F1_0 * 5; /* force send request */
			Network_games_changed = 1;
		}
		if (Network_games_changed) {
			if (num_active_games > last_num_games) /* new game? */
				digi_play_sample(SOUND_HUD_MESSAGE, F1_0);
			last_num_games = num_active_games;
			Network_games_changed = 0;
			draw_list(&bg, lis);
		} else if (selected_game != old_select) {
			draw_item(&bg, lis[old_select], 0);
			draw_item(&bg, lis[selected_game], 1);
		}
		old_socket = Network_socket;
		old_select = selected_game;

		t = timer_get_approx_seconds();
		if (t < req_timer || t > req_timer + F1_0 * 5) {
			req_timer = timer_get_approx_seconds();
			network_send_game_list_request();
		}				
		network_listen();

		k = key_inkey();
		switch (k) {
			case KEY_PAGEUP:
			case KEY_PAD9:
				if (Network_socket < 99) Network_socket++;
				break;
			case KEY_PAGEDOWN:
			case KEY_PAD3:
				if (Network_socket > -99) Network_socket--;
				break;
			case KEY_PAD5:
				Network_socket = 0;
				break;
			case KEY_TAB + KEY_SHIFTED:
			case KEY_UP:
			case KEY_PAD8:
				if (selected_game-- == 0)
					selected_game = MAX_ACTIVE_NETGAMES - 1;
				break;
			case KEY_TAB:
			case KEY_DOWN:
			case KEY_PAD2:
				if (++selected_game == MAX_ACTIVE_NETGAMES)
					selected_game = 0;
				break;
			case KEY_ENTER:
			case KEY_PADENTER:
				done = 1;
				break;
			case KEY_ESC:
				selected_game = -1;
				done = 1;
		}
	}
	done_background(&bg);
	grd_curcanv->cv_font = save_font;
	gr_set_current_canvas( save_canvas );
	return selected_game;
}

#endif //ifdef NETWORK
