/*
THE COMPUTER CODE CONTAINED HEREIN IS THE SOLE PROPERTY OF PARALLAX
SOFTWARE CORPORATION ("PARALLAX").  PARALLAX, IN DISTRIBUTING THE CODE TO
END-USERS, AND SUBJECT TO ALL OF THE TERMS AND CONDITIONS HEREIN, GRANTS A
ROYALTY-FREE, PERPETUAL LICENSE TO SUCH END-USERS FOR USE BY SUCH END-USERS
IN USING, DISPLAYING,  AND CREATING DERIVATIVE WORKS THEREOF, SO LONG AS
SUCH USE, DISPLAY OR CREATION IS FOR NON-COMMERCIAL, ROYALTY OR REVENUE
FREE PURPOSES.  IN NO EVENT SHALL THE END-USER USE THE COMPUTER CODE
CONTAINED HEREIN FOR REVENUE-BEARING PURPOSES.  THE END-USER UNDERSTANDS
AND AGREES TO THE TERMS HEREIN AND ACCEPTS THE SAME BY USE OF THIS FILE.  
COPYRIGHT 1993-1998 PARALLAX SOFTWARE CORPORATION.  ALL RIGHTS RESERVED.
*/

#ifndef _FIX_H
#define _FIX_H

#include "types.h"

#ifndef WANT_FP_FIX

// There are things in i/o dependant on these sizes.
typedef int fix;		// 16 bits int, 16 bits frac
typedef short int fixang;	// angles

typedef struct quad {
	union {
		struct {
			unsigned int low;
			int high;
		} p;
		long long ll;
	} u;
} quad;

//Convert an int to a fix
#define i2f(i) ((fix)((i) * 65536))

//Get the int part of a fix
#define f2i(f) ((int)((f)>>16))

//Get the int part of a fix, with rounding
#define f2ir(f) (((f)+f0_5)>>16)

//Convert fix to float and float to fix
#define f2fl(f) (((float) (f)) / 65536.0)
#define fl2f(f) ((fix) ((f) * 65536))

//Find absolute value
#define fixabs   abs

//Create a quad from two 32-bit ints.
#define INIT_QUAD(q, h, l)  ((q)->u.p.low = (l), (q)->u.p.high = (h))

//Some handy constants
#define f0_0	0
#define f1_0	0x10000
#define f2_0	0x20000
#define f3_0	0x30000
#define f10_0	0xa0000

#define f0_5	0x8000
#define f0_1	0x199a

#else

#error "This doesn't actually work yet."

typedef float fix;
typedef float fixang;
typedef double quad;

//Convert an int to a fix
#define i2f(i) ((fix)(i))

//Get the int part of a fix
#define f2i(f) ((int)(f))

//Get the int part of a fix, with rounding
#define f2ir(f)  ((int) ((f) + f0_5))

//Convert fix to float and float to fix
#define f2fl(f)  ((float)(f))
#define fl2f(f)  ((fix)(f))

//Find absolute value
#define fixabs   fabsf

//Create a quad from two 32-bit ints.
#define INIT_QUAD(q, h, l)  ((q) = (quad)(h)*4294967296.0 + (quad)(l))

//Some handy constants
#define f0_0	0.0f
#define f1_0	1.0f
#define f2_0	2.0f
#define f3_0	3.0f
#define f10_0	10.0f

#define f0_5	0.5f
#define f0_1	0.1f

#endif /* !WANT_FP_FIX */

#define F0_0	f0_0
#define F1_0	f1_0
#define F2_0	f2_0
#define F3_0	f3_0
#define F10_0	f10_0

#define F0_5 	f0_5
#define F0_1 	f0_1

//multiply two fixes, return a fix
fix fixmul(fix a,fix b);

//divide two fixes, return a fix
fix fixdiv(fix a,fix b);

//multiply two fixes, then divide by a third, return a fix
fix fixmuldiv(fix a,fix b,fix c);

//multiply two fixes, and add 64-bit product to a quad
void fixmulaccum(quad *q,fix a,fix b);

//extract a fix from a quad product
fix fixquadadjust(quad *q);

//divide a quad by a long
int fixdivquadlong(uint qlow,uint qhigh,uint d);

//negate a quad
void fixquadnegate(quad *q);

//computes the square root of a long, returning a short
uint long_sqrt(long a);

//computes the square root of a quad, returning a fix
fix quad_sqrt(quad *q);

//computes the square root of a fix, returning a fix
fix fix_sqrt(fix a);

/*
	compute sine and cosine of an angle, filling in the variables
	either of the pointers can be NULL
*/
void fix_sincos(fix a,fix *s,fix *c);		//with interpolation
void fix_fastsincos(fix a,fix *s,fix *c);	//no interpolation

//compute inverse sine & cosine
fixang fix_asin(fix v); 
fixang fix_acos(fix v); 

/*
	given cos & sin of an angle, return that angle.
	parms need not be normalized, that is, the ratio of the parms
	cos/sin must equal the ratio of the actual cos & sin for the result
	angle, but the parms need not be the actual cos & sin.  
	NOTE: this is different from the standard C atan2, since it is
	left-handed.
*/
fixang fix_atan2(fix cos,fix sin); 

//for passed value a, returns 1/sqrt(a) 
fix fix_isqrt( fix a );



#if defined(WANT_FP_FIX)

#define do_fixquadnegate(q)	(*q = -*q)
#define do_fixmulaccum(q,x,y)	(*q += x * y)
#define do_fixquadadjust(q)	((fix)*q)
#define do_fixmul(x,y)		(x * y)
#define do_fixdiv(x,y)		(x / y)
#define do_fixmuldiv(x,y,z)	(x * y / z)

#else

#if defined(__i386__) && defined(ASM_FIXED_POINT)

#define do_fixquadnegate(q)	(q->u.ll = -q->u.ll)

#define do_fixmulaccum(q,x,y)						\
({									\
	int _ax, _dx;							\
	asm("imull %4\n\taddl %2,%0\n\tadcl %3,%1" 			\
	    : "=g"(q->u.p.low), "=g"(q->u.p.high), "=&a"(_ax), "=&d"(_dx)	\
	    : "rm"(y), "0"(q->u.p.low), "1"(q->u.p.high), "2"(x));	\
	(void) 0;							\
})

#define do_fixquadadjust(q)	((fix) (q->u.ll >> 16))

#define do_fixmul(x,y)				\
({						\
	int _ax, _dx;				\
	asm("imull %2\n\tshrdl %3,%1,%0"	\
	    : "=a"(_ax), "=d"(_dx)		\
	    : "rm"(y), "i"(16), "0"(x));	\
	_ax;					\
})

#define do_fixdiv(x,y)							     \
({									     \
	int _ax = x << 16, _dx = x >> 16;				     \
	asm("idivl %2" : "=a"(_ax), "=d"(_dx) : "rm"(y), "0"(_ax), "1"(_dx)); \
	_ax;								     \
})

#define do_fixmuldiv(x,y,z)						\
({									\
	int _ax, dummy;							\
	asm("imull %2\n\tidivl %3"					\
	    : "=&a"(_ax), "=&d"(dummy) : "rm"(y), "rm"(z), "0"(x));	\
	_ax;								\
})

#define do_fixdivquadlong(nl,nh,d)					   \
({									   \
	int _ax, _dx;							   \
	asm("idivl %2" : "=a"(_ax), "=d"(_dx) : "rm"(d), "0"(nl), "1"(nh)); \
	_ax;								   \
})

#define do_fixdivquadlongu(nl,nh,d)					   \
({									   \
	int _ax, _dx;							   \
	asm("divl %2" : "=a"(_ax), "=d"(_dx) : "rm"(d), "0"(nl), "1"(nh)); \
	_ax;								   \
})

#elif SIZEOF_LONG == 8

#define do_fixquadnegate(q)	(q->u.ll = -q->u.ll)
#define do_fixmulaccum(q,x,y)	(q->u.ll += (long)x * (long)y)
#define do_fixquadadjust(q)	(q->u.ll >> 16)
#define do_fixmul(x,y)		((long)x * (long)y >> 16)
#define do_fixdiv(x,y)		(((long)x << 16) / y)
#define do_fixmuldiv(x,y,z)	(((long)x * (long)y) / z)

#endif /* ARCH specific */

#endif

#if 1 // use inline versions if available
/* Create inline functions of whatever is defined above, so that parameters
   are type-checked correctly.  */

#ifdef do_fixquadnegate
extern inline void fixquadnegate(quad *q) { do_fixquadnegate(q); }
#endif
#ifdef do_fixmulaccum
extern inline void fixmulaccum(quad *q,fix x,fix y) { do_fixmulaccum(q,x,y); }
#endif
#ifdef do_fixquadadjust
extern inline fix fixquadadjust(quad *q) { return do_fixquadadjust(q); }
#endif
#ifdef do_fixmul
extern inline fix fixmul(fix x, fix y) { return do_fixmul(x,y); }
#endif
#ifdef do_fixdiv
extern inline fix fixdiv(fix x, fix y) { return do_fixdiv(x,y); }
#endif
#ifdef do_fixmuldiv
extern inline fix fixmuldiv(fix a,fix b,fix c) { return do_fixmuldiv(a,b,c); }
#endif

#ifdef do_fixdivquadlong
extern inline int fixdivquadlong(uint nl, uint nh, uint d)
{
	return do_fixdivquadlong(nl, nh, d);
}
#elif SIZEOF_LONG == 8
extern inline int fixdivquadlong(uint nl, uint nh, uint d)
{
	return (long)(((long)nh << 32) | (unsigned long)nl) / d;
}
#endif
#endif /* 0 */

#endif
