/*
THE COMPUTER CODE CONTAINED HEREIN IS THE SOLE PROPERTY OF PARALLAX
SOFTWARE CORPORATION ("PARALLAX").  PARALLAX, IN DISTRIBUTING THE CODE TO
END-USERS, AND SUBJECT TO ALL OF THE TERMS AND CONDITIONS HEREIN, GRANTS A
ROYALTY-FREE, PERPETUAL LICENSE TO SUCH END-USERS FOR USE BY SUCH END-USERS
IN USING, DISPLAYING,  AND CREATING DERIVATIVE WORKS THEREOF, SO LONG AS
SUCH USE, DISPLAY OR CREATION IS FOR NON-COMMERCIAL, ROYALTY OR REVENUE
FREE PURPOSES.  IN NO EVENT SHALL THE END-USER USE THE COMPUTER CODE
CONTAINED HEREIN FOR REVENUE-BEARING PURPOSES.  THE END-USER UNDERSTANDS
AND AGREES TO THE TERMS HEREIN AND ACCEPTS THE SAME BY USE OF THIS FILE.  
COPYRIGHT 1993-1998 PARALLAX SOFTWARE CORPORATION.  ALL RIGHTS RESERVED.
*/

#include <stdlib.h>
#include <math.h>

#include "error.h"
#include "fix.h"

extern ubyte guess_table[];
extern short sincos_table[];
extern ushort asin_table[];
extern ushort acos_table[];
extern fix isqrt_guess_table[];

#define EPSILON (F1_0/100)


void fixquadnegate(quad *q)
{
#ifdef do_fixquadnegate
	do_fixquadnegate(q);
#else
	q->u.ll = -q->u.ll;
#endif
}

void fixmulaccum(quad *q,fix a,fix b)
{
#ifdef do_fixmulaccum
	do_fixmulaccum(q,a,b);
#else
	uint aa,bb;
	uint ah,al,bh,bl;
	uint t,c=0,old;
	int neg;

	neg = ((a^b) < 0);

	aa = labs(a); bb = labs(b);

	ah = aa>>16;  al = aa&0xffff;
	bh = bb>>16;  bl = bb&0xffff;

	t = ah*bl + bh*al;

	if (neg)
		fixquadnegate(q);

	old = q->u.p.low;
	q->u.p.low += al*bl;
	if (q->u.p.low < old) q->u.p.high++;
	
	old = q->u.p.low;
	q->u.p.low += (t<<16);
	if (q->u.p.low < old) q->u.p.high++;
	
	q->u.p.high += ah*bh + (t>>16) + c;
	
	if (neg)
		fixquadnegate(q);
#endif
}


fix fixquadadjust(quad *q)
{
#ifdef do_fixquadadjust
	return do_fixquadadjust(q);
#else
	return (q->u.p.high<<16) + (q->u.p.low>>16);
#endif
}

fix fixmul(fix a, fix b)
{
#ifdef do_fixmul
	return do_fixmul(a,b);
#else
	return (fix)(((double)a*(double)b)/65536.0);
#endif
}

fix fixdiv(fix a, fix b)
{
#ifdef do_fixdiv
	return do_fixdiv(a,b);
#else
	return (fix)(((double)a * 65536.0) / (double)b);
#endif
}

fix fixmuldiv(fix a, fix b, fix c)
{
#ifdef do_fixmuldiv
	return do_fixmuldiv(a,b,c);
#else
	return (fix)(((double)a * (double) b) / (double) c);
#endif
}

int fixdivquadlong(uint nl,uint nh,uint d)
{
#ifdef do_fixdivquadlong
	return do_fixdivquadlong(nl,nh,d);
#elif SIZEOF_LONG == 8
	return (long)(((long)nh << 32) | (unsigned long)nl) / d;
#else
	int i;
	uint tmp0;
	ubyte tmp1;
	uint r;
	ubyte T,Q,M;

	r = 0;

	Q = ((nh&0x80000000)!=0);
	M = ((d&0x80000000)!=0);
	T = (M!=Q);

	if (M == 0)
	{
		for (i=0; i<32; i++ )   {
	
			r <<= 1;
			r |= T;
			T = ((nl&0x80000000L)!=0);
			nl <<= 1;
	
			switch( Q ) {
		
			case 0:
				Q = (unsigned char)((0x80000000L & nh) != 0 );
				nh = (nh << 1) | (unsigned int)T;

				tmp0 = nh;
				nh -= d;
				tmp1 = (nh>tmp0);
				if (Q == 0)
					Q = tmp1;
				else
					Q = (unsigned char)(tmp1 == 0);
				break;
			case 1:
				Q = (unsigned char)((0x80000000L & nh) != 0 );
				nh = (nh << 1) | (unsigned int)T;

				tmp0 = nh;
				nh += d;
				tmp1 = (nh<tmp0);
				if (Q == 0)
					Q = tmp1;
				else
					Q = (unsigned char)(tmp1 == 0);
				break;
			}
			T = (Q==M);
		}
	}
	else
	{
		for (i=0; i<32; i++ )   {
	
			r <<= 1;
			r |= T;
			T = ((nl&0x80000000L)!=0);
			nl <<= 1;
	
			switch( Q ) {
		
			case 0:
				Q = (unsigned char)((0x80000000L & nh) != 0 );
				nh = (nh << 1) | (unsigned int)T;

				tmp0 = nh;
				nh += d;
				tmp1 = (nh<tmp0);
				if (Q == 1)
					Q = tmp1;
				else
					Q = (unsigned char)(tmp1 == 0);
				break;
			case 1: 
				Q = (unsigned char)((0x80000000L & nh) != 0 );
				nh = (nh << 1) | (unsigned int)T;

				tmp0 = nh;
				nh = nh - d;
				tmp1 = (nh>tmp0);
				if (Q == 1)
					Q = tmp1;
				else
					Q = (unsigned char)(tmp1 == 0);
				break;
			}
			T = (Q==M);
		}
	}

	r = (r << 1) | T;

	return r;
#endif
}

unsigned int fixdivquadlongu(uint nl, uint nh, uint d)
{
#ifdef do_fixdivquadlongu
	return do_fixdivquadlongu(nl,nh,d);
#elif SIZEOF_LONG == 8
	return (((unsigned long)nh << 32) | (unsigned long)nl) / d;
#else
	return fixdivquadlong(nl, nh, d);
#endif
}


//given cos & sin of an angle, return that angle.
//parms need not be normalized, that is, the ratio of the parms cos/sin must
//equal the ratio of the actual cos & sin for the result angle, but the parms 
//need not be the actual cos & sin.  
//NOTE: this is different from the standard C atan2, since it is left-handed.

fixang fix_atan2(fix cos,fix sin)
{
#ifdef WANT_FP_FIX
	float dcos, dsin;
	dcos = f2fl(cos);
	dsin = f2fl(sin);
	return M_PI*2 - atan2f(cos, sin);
#else
	double d, dsin, dcos;
	fix m,t;

	//Assert(!(cos==0 && sin==0));

	//find smaller of two

	dsin = (double)sin;
	dcos = (double)cos;
	d = sqrt((dsin * dsin) + (dcos * dcos));

	if (d==0.0)
		return 0;

	if (labs(sin) < labs(cos)) {	//sin is smaller, use arcsin
		t = fix_asin((fix)((dsin / d) * 65536.0));
		if (cos<0)
			t = 0x8000 - t;
		return t;
	}
	else {
		t = fix_acos((fix)((dcos / d) * 65536.0));
		if (sin<0)
			t = -t;
		return t;
	}
#endif
}

fix quad_sqrt(quad *q)
{
#ifdef WANT_FP_FIX
	return sqrt(*q);
#elif SIZEOF_LONG == 8
	return long_sqrt(q->u.ll);
#else
	int high = q->u.p.high, low = q->u.p.low, i;
	int cnt;
	unsigned int r, old_r, t;
	quad tq;

	if (high<0)
		return 0;

	if (high==0 && low>=0)
		return long_sqrt(low);

	if (high & 0xff000000) {
		cnt=12+16; i = high >> 24;
	} else if (high & 0xff0000) {
		cnt=8+16; i = high >> 16;
	} else if (high & 0xff00) {
		cnt=4+16; i = high >> 8;
	} else {
		cnt=0+16; i = high;
	}
	
	r = guess_table[i]<<cnt;

	/* quad loop usually executed 4 times.  */
	/* If we have a large value, don't overflow the first calculations. */
	if (high < 0x40000000) {
		r = (fixdivquadlongu(low,high,r)+r)/2;
		r = (fixdivquadlongu(low,high,r)+r)/2;
		r = (fixdivquadlongu(low,high,r)+r)/2;
	} else {
		r = fixdivquadlongu(low,high,r)/2 + r/2;
		r = fixdivquadlongu(low,high,r)/2 + r/2;
		r = fixdivquadlongu(low,high,r)/2 + r/2;
	}

	do {
		old_r = r;
		t = fixdivquadlongu(low,high,r);
		if (t==r)
			return r;
		r = (t+r)/2;
	} while (!(r==t || r==old_r));

	t = fixdivquadlongu(low,high,r);

	INIT_QUAD(&tq, 0, 0);
	fixmulaccum(&tq,r,t);
	if (tq.u.p.low!=low || tq.u.p.high!=high)
		r++;

	return r;
#endif
}

//computes the square root of a long.
uint long_sqrt(long a)
{
	unsigned long r, old_r, t;
	long cnt, big = ~(~0UL >> 1) >> 2;

	if (a<=0)
		return 0;

#if SIZEOF_LONG == 8
	if (a & 0xffffffff00000000)
		if (a & 0xffff000000000000)
			if (a & 0xff00000000000000)
				cnt = 56;
			else
				cnt = 48;
		else
			if (a & 0x0000ff0000000000)
				cnt = 40;
			else
				cnt = 32;
	else
#endif
		if (a & 0xffff0000)
			if (a & 0xff000000)
				cnt = 24;
			else
				cnt = 16;
		else
			if (a & 0xff00)
				cnt = 8;
			else
				cnt = 0;
	r = (long)guess_table[(a>>cnt)&0xff] << (cnt / 2);
		
	//the loop nearly always executes 3 times, so we'll unroll it
	//2 times and not do any checking until after the third time.
	//By my calcutations, the loop is executed 2 times in 99.97%
	//of cases, 3 times in 93.65% of cases, four times in 16.18%
	//of cases, and five times in 0.44% of cases.  It never
	//executes more than five times.  By timing, I determined that
	//is is faster to always execute three times and not check for
	//termination the first two times through.  This means that in
	//93.65% of cases, we save 6 cmp/jcc pairs, and in 6.35% of
	//cases we do an extra divide.  In real life, these numbers
	//might not be the same.

	/* If we have a large value, don't overflow the first calculations. */
	if (a < big) {
		r = ((a/r)+r)/2;
		r = ((a/r)+r)/2;
	} else {
		r = (a/r)/2 + r/2;
		r = (a/r)/2 + r/2;
	}

	do {
		old_r = r;
		t = a/r;
		if (t==r)
			return r;
		r = (t+r)/2;
	} while (!(r==t || r==old_r));

	if (a % r)
		r++;

	return r;
}

//computes the square root of a fix, returning a fix
fix fix_sqrt(fix a)
{
#ifdef WANT_FP_FIX
	return sqrtf(a);
#else
	return ((fix) long_sqrt(a)) << 8;
#endif
}


//compute sine and cosine of an angle, filling in the variables
//either of the pointers can be NULL
//with interpolation
void fix_sincos(fix a,fix *s,fix *c)
{
#ifdef WANT_FP_FIX
	*s = sinf(a);
	*c = cosf(a);
#else
	int i,f;
	fix ss,cc;

	i = (a>>8)&0xff;
	f = a&0xff;

	ss = sincos_table[i];
	if (s) *s = (ss + (((sincos_table[i+1] - ss) * f)>>8))<<2;

	cc = sincos_table[i+64];
	if (c) *c = (cc + (((sincos_table[i+64+1] - cc) * f)>>8))<<2;
#endif
}

//compute sine and cosine of an angle, filling in the variables
//either of the pointers can be NULL
//no interpolation
void fix_fastsincos(fix a,fix *s,fix *c)
{
#ifdef WANT_FP_FIX
	*s = sinf(a);
	*c = cosf(a);
#else
	int i;

	i = (a>>8)&0xff;

	if (s) *s = sincos_table[i] << 2;
	if (c) *c = sincos_table[i+64] << 2;
#endif
}

//compute inverse sine
fixang fix_asin(fix v)
{
#ifdef WANT_FP_FIX
	return asinf(v);
#else
	fix vv;
	int i,f,aa;

	vv = labs(v);

	if (vv >= f1_0)		//check for out of range
		return 0x4000;

	i = (vv>>8)&0xff;
	f = vv&0xff;

	aa = asin_table[i];
	aa = aa + (((asin_table[i+1] - aa) * f)>>8);

	if (v < 0)
		aa = -aa;

	return aa;
#endif
}

//compute inverse cosine
fixang fix_acos(fix v)
{
#ifdef WANT_FP_FIX
	return acosf(v);
#else
	fix vv;
	int i,f,aa;

	vv = labs(v);

	if (vv >= f1_0)		//check for out of range
		return 0;

	i = (vv>>8)&0xff;
	f = vv&0xff;

	aa = acos_table[i];
	aa = aa + (((acos_table[i+1] - aa) * f)>>8);

	if (v < 0)
		aa = 0x8000 - aa;

	return aa;
#endif
}

#define TABLE_SIZE 1024

//for passed value a, returns 1/sqrt(a) 
fix fix_isqrt( fix a )
{
#ifdef WANT_FP_FIX
	return 1.0f / sqrtf(a);
#else
	int i, b = a;
	int cnt = 0;
	int r;

	if ( a == 0 ) return 0;

	while( b >= TABLE_SIZE )	{
		b >>= 1;
		cnt++;
	}

	//printf( "Count = %d (%d>>%d)\n", cnt, b, (cnt+1)/2 );
	r = isqrt_guess_table[b] >> ((cnt+1)/2);

	//printf( "Initial r = %d\n", r );

	for (i=0; i<3; i++ )	{
		int old_r = r;
		r = fixmul( ( (3*65536) - fixmul(fixmul(r,r),a) ), r) / 2;
		//printf( "r %d  = %d\n", i, r );
		if ( old_r >= r ) return (r+old_r)/2;
	}

	return r;	
#endif
}
