/* XLib graphics driver capable of MITSHM
 * Also handles 16 and 24bpp
 *
 * by Jay Kominek <jkominek@debian.org>
 *
 * Based on the plain ole Xlib graphics driver by:
 * Johannes Hirche
 */

#ifdef HAVE_X11
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/extensions/XShm.h>
#ifdef HAVE_X11_DGA
#include <X11/extensions/xf86dga.h>
#include <signal.h>
#ifdef HAVE_SIGCONTEXT_H
#include <sigcontext.h>
#endif
#endif
#include "error.h"
#include "gr.h"
#include "mono.h"
#include "key.h"
#include "timer.h"
#include "args.h"
#include "intrface.h"

/* Here are the keycodes for X->descent */
#include "xlib_keycode_to_descent.h"
#include "xlib_special_to_descent.h"

/* Prototype it so that x11_setpalette() can call it */
static void x11_update_shm(void);
static void x11_update_pix(void);
static void x11_update_nul(void);
static void x11_update_8(void);
static void x11_update_16(void);
static void x11_update_24(void);
static void x11_update_32(void);

/* Prototype it so that x11_set_mode() can call it */
static int x11_error_handler(Display *display, XErrorEvent *event);
#ifdef HAVE_X11_DGA
#ifdef HAVE_SIGCONTEXT_H
static void x11_dga_cleanup(int sig, int code, struct sigcontext *c);
#else
static void x11_dga_cleanup(int sig);
#endif
#endif

/* Locals */
static int granularity=0;

/* Xlib Variables */
static Display *display;
static Screen *screen;
static Window window;
static Colormap cmap;
static GC gc;
static XImage *image;
static unsigned long pen_fg, pen_bg;
static XVisualInfo visualinfo;
static XKeyboardState initKeyboardState;
static int x11_key_repeat;
static int x11_have_focus;

/* SHM Stuff */
static XShmSegmentInfo shminfo;
static Bool usingShm;

/* DGA Stuff */
static char *dgaaddr;
static int dgawidth, dgabank, dgaram;

#ifdef HAVE_X11_DGA
static Bool usingDGA;
#else
#define usingDGA 0
#endif

/* current colormap */
static XColor colors[256];
static unsigned int xlat_table[256];

/* Size of one frame-buffer (xsize*ysize) */
static unsigned long bytes_per_page;

/* Pointer to draw-buffer */
static unsigned char *gr_mem;

/* "Image" info */
static unsigned char *image_data;
static int image_height;
static int image_width;
static int image_line_skip;

/* Mouse info */
static int mouse_deltax, mouse_deltay, mouse_state;

/* Initialized? */
static int running;

/* How to update the server after we are done frobbing pixels.  */
static void (*update_server)(void);

/* Array of pointers to the different pages.  Not actually used by x.  */
static unsigned char *empty_pages[8];
unsigned char **gr_mem_pages = empty_pages;

/* Current bit depth */
static int bpp, depth;

/* Menu or game? */
int x11_game_mode;

/* Graphics stuff */

static void x11_mode_to_size(int mode, int *px, int *py)
{
  int x, y;
  switch(mode)
    {
    default:
    case SM_320x200C:
    case SM_320x200U: x = 320; y = 200; break;
    case SM_320x240U: x = 320; y = 240; break;
    case SM_360x200U: x = 360; y = 200; break;
    case SM_360x240U: x = 360; y = 240; break;
    case SM_376x282U: x = 376; y = 282; break;
    case SM_320x400U: x = 320; y = 400; break;
    case SM_320x480U: x = 320; y = 480; break;
    case SM_360x400U: x = 360; y = 400; break;
    case SM_360x480U: x = 360; y = 480; break;
    case SM_360x360U: x = 360; y = 360; break;
    case SM_376x308U: x = 376; y = 308; break;
    case SM_376x564U: x = 376; y = 564; break;
    case SM_640x400V: x = 640; y = 400; break;
    case SM_640x480V15:
    case SM_640x480V: x = 640; y = 480; break;
    case SM_800x600V15:
    case SM_800x600V: x = 800; y = 600; break;
    case SM_1024x768V: x = 1024; y = 768; break;
    }
  *px = x;
  *py = y;
}

static int x11_initial_open(int xsize, int ysize)
{
  static char * namestring[] = {PACKAGE,VERSION};

  XSetWindowAttributes attributes;
  XGCValues gc_val;
  unsigned long gc_valmask;
  XTextProperty windowname, iconname;
  XSizeHints hints;
  XWMHints wmhints;
  XClassHint classhints;
  XEvent event;

  /* FIXME: it only uses $DISPLAY to get the display.  */
  if(!(display = XOpenDisplay(NULL)))
    return -1;

  XSetErrorHandler(x11_error_handler);

#ifdef HAVE_X11_DGA
  if (FindArg("-UseDGA"))
    {
      int MajorVersion, MinorVersion, EventBase, ErrorBase;

      usingDGA = 0;
      if (geteuid() != 0)
	{
	  printf("Skipping DGA -- need root permissions to map fb.\n");
	}
      else if (!XF86DGAQueryExtension(display, &EventBase, &ErrorBase))
	{
	  printf("DGA extension not present.\n");
	}
      else if (!XF86DGAQueryVersion(display, &MajorVersion, &MinorVersion))
	{
	  printf("Could not query DGA version!\n");
	}
      else if (MajorVersion < 1)
	{
	  printf("Xserver is running an old XFree86-DGA version (%d.%d)\n",
		 MajorVersion, MinorVersion);
	}
      else
	{
	  int flags;
	  XF86DGAQueryDirectVideo(display, DefaultScreen(display), &flags);
	  if (!(flags & XF86DGADirectPresent))
	    {
	      printf("Xserver driver doesn't support DirectVideo\n");
	    }
#if 0 // return value of XF86DGAGetVideo seems bogus (XFree86 3.2)
	  else if (!XF86DGAGetVideo(display, DefaultScreen(display),
				    &dgaaddr, &dgawidth, &dgabank, &dgaram))
	    {
	      printf("Could not map video memory; turning off DGA.\n");
	    }
	  else
#endif	  
	    {
		  XF86DGAGetVideo(display, DefaultScreen(display),
				    &dgaaddr, &dgawidth, &dgabank, &dgaram);

		  /* XF86DGA.. calls atexit(XF86cleanup) and XF86cleanup calls _exit(), */ 
		  /* preventing our previously atexit()s from running :-( */
		  /* So install another atexit to cleanup at least the keyboard repeat */
		  atexit(x11_close_down); 

	      /* Give up root privs.  */
	      setuid(getuid());

	      printf("DGA addr %p, width %d, bank size %d, ram %d\n",
		     dgaaddr, dgawidth, dgabank, dgaram);

	      usingDGA = 1;
	    }
	}
    }
#endif /* HAVE_X11_DGA */

  if (!usingDGA)
    {
      if (!FindArg("-NoShm"))
	{
	  if ((usingShm=XShmQueryExtension(display)) != 0)
	    {
	      int major, minor;
	      Bool shared;
  
	      XShmQueryVersion(display,&major,&minor,&shared);
	      printf("Using SHM extension. Version %i.%i %s shared pixmaps\n",
		     major, minor, shared ? "with" : "without");
	    }
	  else
	    {
	      printf("SHM extension not present.\n");
	    }
	}
      else
	{
	  printf("Not using SHM extension.\n");
	  usingShm = 0;
	}
    }
    
  /* Get the default screen */
  screen = DefaultScreenOfDisplay(display);

  /* Get an appropriate visual */
  if (XMatchVisualInfo(display, DefaultScreen(display), 8, PseudoColor,
		       &visualinfo))
    {
      depth = bpp = 8;
    }
  else
    {
      XPixmapFormatValues *formats;
      int count, i;

      printf("Could not get PseudoColor visual! Trying for Truecolor.\n");

      depth = DefaultDepthOfScreen(screen);
      if (!XMatchVisualInfo(display, DefaultScreen(display),
			    depth, TrueColor, &visualinfo))
	{
	  printf("Could not get TrueColor visual!\n");
	  return -1;
	}

      formats = XListPixmapFormats(display, &count);
      for (bpp = i = 0; i < count; i++)
	if (formats[i].depth == depth)
	  {
            bpp = formats[i].bits_per_pixel;
	    break;
	  }
      XFree(formats);

      if (bpp == 0)
	{
	  printf("Could not get pixmap format for depth %d!\n", depth);
	  return -1;
	}


      if (bpp != 16 && bpp != 24 && bpp != 32)
	{
	  printf("Your bit depth (%ibpp) is not supported!\n",bpp);
	  return -1;
	}
    }

  /* A colour map.  */
  if (bpp == 8)
    cmap = XCreateColormap(display, RootWindowOfScreen(screen),
			   visualinfo.visual, AllocAll);
  else
    cmap = DefaultColormapOfScreen(screen);

  memset(&attributes, 0, sizeof attributes);
  attributes.colormap = cmap;

  if (bpp == 8)
    {
      int i;
      for (i = 0; i < 256; i++)
	colors[i].flags = DoRed|DoGreen|DoBlue;
    }

#ifdef HAVE_X11_DGA
  if (usingDGA)
    {
      /* Since we have the whole screen, grab the keyboard... */
      XGrabKeyboard(display, RootWindowOfScreen(screen), True,
		    GrabModeAsync, GrabModeAsync, CurrentTime);

      /* ... and all the mouse moves */
      XGrabPointer(display, RootWindowOfScreen(screen), True,
		   PointerMotionMask | ButtonPressMask | ButtonReleaseMask,
		   GrabModeAsync, GrabModeAsync, None,  None, CurrentTime);

      /* Prepare for the worst.  */
      signal(SIGSEGV, x11_dga_cleanup);
#ifdef SIGBUS
      signal(SIGBUS, x11_dga_cleanup);
#endif
      signal(SIGHUP, x11_dga_cleanup);
      signal(SIGFPE, x11_dga_cleanup);  
      signal(SIGABRT, x11_dga_cleanup);  

      /* Let's Go Live! */
      XF86DGADirectVideo(display, DefaultScreen(display),
			 (XF86DGADirectGraphics|
			  XF86DGADirectMouse|
			  XF86DGADirectKeyb));

      if (bpp == 8)
	XF86DGAInstallColormap(display, DefaultScreen(display), cmap);

      XF86DGASetViewPort(display, DefaultScreen(display), 0, 0);
   }
  else
#endif /* HAVE_X11_DGA */
    {
      /* Create the window */
      window = XCreateWindow(display,		/* Display */
			     RootWindowOfScreen(screen), /* Parent Window */
			     0,0,		/* Suggested pos on screen */
			     xsize, ysize,	/* Size of window */
			     4,			/* Width of borders */
			     depth,		/* Screen depth */
			     CopyFromParent,	/* Class */
			     visualinfo.visual,	/* Visual */
			     CWColormap,	/* Window attribute mask */
			     &attributes);	/* Window attributes */
      if (!window)
	{
	  /* Wow, what kind of broken system does this guy have? */
	  printf("Could not open a window!\n");
	  return -1;
	}

      /* Create the graphics context */
      gc_val.graphics_exposures = False;
      gc_val.foreground = pen_fg;
      gc_val.background = pen_bg;
      gc_valmask = GCGraphicsExposures | GCForeground | GCBackground;
      gc = XCreateGC(display, window, gc_valmask, &gc_val);

      /* Send information to the window manager. (If we actually had argc
	 and argv, this is what would make us dockable.)  This will also
	 keep all well behaved window managers from letting the user alter
	 the size of our window */

      XStringListToTextProperty(namestring,2,&windowname);
      XStringListToTextProperty(namestring,1,&iconname);
  
      hints.min_width  = hints.max_width  = hints.base_width  = xsize;
      hints.min_height = hints.max_height = hints.base_height = ysize;
      hints.flags = PSize | PMinSize | PMaxSize;
      wmhints.input = True;
      wmhints.flags = InputHint;
      classhints.res_class = classhints.res_name = PACKAGE;

      XSetWMProperties(display, window, &windowname, &iconname, Args,
		       Num_args, &hints, &wmhints, &classhints);

      /* Set out event mask */
      XSelectInput(display, window, (ExposureMask | KeyPressMask
				     | KeyReleaseMask | StructureNotifyMask
				     | FocusChangeMask));

      XMapRaised(display, window);
      XClearWindow(display, window);

      /* Wait for window to be mapped */
      do
	XNextEvent(display, &event);
      while (event.type != Expose);
    }

  /* Should check for modeinfo here. (?) */
  XGetKeyboardControl(display, &initKeyboardState);
  x11_key_repeat = 0;
  x11_have_focus = 0;
  //XAutoRepeatOff(display);

  /* Boy is this ugly... */
  granularity = 32;
  running = 1;

  return 0;
}

static int x11_alloc_image(int xsize, int ysize)
{
  if (usingDGA)
    {
      int xstart, ystart, Bpp = bpp / 8;

      /* ??? Can we use the xf86vidmode extension to change resolutions?
         Hell, or even find out how tall the screen is so we can center it.  */

      if (xsize > dgawidth)
	{
	  printf("Size too large for current mode: (%d,%d) > (%d,%d)\n",
		 xsize, ysize, dgawidth, -1);
	  return -1;
	}

      /* Clear the whole screen.  Assume 0 is always black. */
      memset (dgaaddr, 0, dgabank);

      /* Make sure our line starts on a nice boundary.  */
      xstart = ((dgawidth - xsize) / 2) & -(sizeof(long) / Bpp);

      /* Ho hum: ystart = ((dgaheight - ysize) / 2); */
      ystart = 0;

      image_width = xsize;
      image_height = ysize;
      /* Ho hum, our "width" is always the viewport size.  */
      /* image_data = dgaaddr + ((ystart * dgawidth + xstart) * Bpp); */
      image_data = dgaaddr;
      image_line_skip = (dgawidth - xsize) * Bpp;
    }
  else
    {
      switch (usingShm)
	{
	default:
	  image = XShmCreateImage(display, visualinfo.visual, depth, ZPixmap,
				  NULL, &shminfo, xsize, ysize);

	  /* Get SHM information from the image */
	  if (image)
	    {
	      shminfo.shmid = shmget(IPC_PRIVATE,
				     image->bytes_per_line*image->height,
				     IPC_CREAT|0666);
	      shminfo.shmaddr = shmat(shminfo.shmid, 0, 0);
	      image->data = shminfo.shmaddr;
	      if (image->data)
		{
		  /* We have to be able to alter the contents of the memory */
		  shminfo.readOnly = False;
		  if (XShmAttach(display, &shminfo))
		    {
		      struct shmid_ds shm_info;

		      /* Mark the memory area as to be removed on the last
			 detach.  This is helpful if we crash before
			 destroying it on shutdown.  */
		      shmctl(shminfo.shmid, IPC_RMID, &shm_info);

		      /* Whee!  All ok.  */
		      break;
		    }
		  printf("Could not attach shared memory to the X server.\n");
		}
	      else
		printf("Could not allocate shared memory.\n");
	      XDestroyImage(image);
	    }
	  else
	    printf("Failed to get SHM image.\n");
	  usingShm = False;
	  /* FALLTHRU */

	case 0:
	  image = XGetImage(display,window,0,0,xsize,ysize,AllPlanes,ZPixmap);
	  if (!image)
	    {
	      printf("Failed to get image.\n");
	      return -1;
	    }
	  break;
	}

      image_width = image->width;
      image_height = image->height;
      image_data = image->data;
      image_line_skip = 0;
    }

  /* Don't give the rest of the Descent code the real location of
     memory, since the real location is bigger for 16 and 24bpp */
  
  if (bpp != 8 || (usingDGA && image_line_skip != 0))
    {
      gr_mem = malloc(xsize*ysize);
      if (!gr_mem)
	{
	  printf("Failed to malloc() memory for the graphics buffer\n");
	  return -1;
	}
    }
  else
    {
      /* Nothing the 8bpp data does not have to be frobbed, so we
	 might as well just let the rest of the code write to the
	 real thing */
      gr_mem = image_data;
    }

  /* Shortcut the update by registering the exact function we need.  */
  if (usingDGA)
    update_server = x11_update_nul;
  else if (usingShm)
    update_server = x11_update_shm;
  else
    update_server = x11_update_pix;

  if (bpp == 8)
    {
      if (usingDGA)
	{
	  if (gr_mem == image_data)
	    {
	      printf("Yeah! Direct FB update!\n");
	      int_gr_update = x11_update_nul;
	    }
	  else
	    int_gr_update = x11_update_8;
	}
      else
	int_gr_update = update_server;
    }
  else if (bpp == 16)
    int_gr_update = x11_update_16;
  else if (bpp == 24)
    int_gr_update = x11_update_24;
  else if (bpp == 32)
    int_gr_update = x11_update_32;

  return 0;
}

static void x11_release_image(void)
{
  if (!usingDGA)
    {
      if (usingShm)
        {
          XShmDetach(display, &shminfo);
          shmdt(shminfo.shmaddr);
        }
      XDestroyImage(image);
    }

  if (gr_mem != image_data)
    free(gr_mem);
  gr_mem = NULL;
  int_gr_update = NULL;
}  

int x11_set_mode(int mode)
{
  int xsize, ysize;

  if (mode == SM_ORIGINAL)
    {
       x11_close_down();
       return 0;
    }
  
  x11_mode_to_size(mode, &xsize, &ysize);

  if (!running)
    {
      if (x11_initial_open(xsize, ysize))
	return -1;
    }
  else
    {
      x11_release_image();
      if (!usingDGA)
        XResizeWindow(display, window, xsize, ysize);
    }

  return x11_alloc_image(xsize, ysize);
}

void x11_set_palette(char *pal)
{
  register int i;

  if (!running)
    return;

  switch (bpp)
    {
    case 8:
      for (i = 0; i < 256; i++)
	{
	  colors[i].pixel = i;
	  colors[i].red   = ((*pal++)<<10);
	  colors[i].green = ((*pal++)<<10);
	  colors[i].blue  = ((*pal++)<<10);
	}
      XStoreColors(display,cmap,colors,256);
#ifdef HAVE_X11_DGA
      if (usingDGA)
	XF86DGAInstallColormap(display, DefaultScreen(display), cmap);
#endif
      break;

    case 16:
      {
	unsigned long r_mask = visualinfo.red_mask;
	unsigned long g_mask = visualinfo.green_mask;
	unsigned long b_mask = visualinfo.blue_mask;

	for (i = 0; i < 256; i++)
	  {
	    colors[i].red = ((*pal++)<<10);
	    colors[i].green = ((*pal++)<<5);
	    colors[i].blue = ((*pal++)>>1);
	    xlat_table[i] = (colors[i].red & r_mask) |
	      (colors[i].green & g_mask) | (colors[i].blue & b_mask);
	  }
	x11_update_16();
      }
      break;

    case 24:
      for (i = 0; i < 256; i++)
	{
	  colors[i].red = ((*pal++)<<2);
	  colors[i].green = ((*pal++)<<2);
	  colors[i].blue = ((*pal++)<<2);
	}
      x11_update_24();
      break;

    case 32:
      {
	int r_shift, g_shift, b_shift;
	r_shift = ffs(visualinfo.red_mask) - 1;
	g_shift = ffs(visualinfo.green_mask) - 1;
	b_shift = ffs(visualinfo.blue_mask) - 1;
	assert(0xff << r_shift == visualinfo.red_mask);
	assert(0xff << g_shift == visualinfo.green_mask);
	assert(0xff << b_shift == visualinfo.blue_mask);
	
        for (i = 0; i < 256; i++)
	  {
	    colors[i].red = ((*pal++)<<2);
	    colors[i].green = ((*pal++)<<2);
	    colors[i].blue = ((*pal++)<<2);
	    xlat_table[i] = (colors[i].red << r_shift)
			  | (colors[i].green << g_shift)
			  | (colors[i].blue << b_shift);
	  }
      }
      x11_update_32();
      break;

    default:
      break;
    }
  return;
}

void x11_get_palette(char *pal)
{
  register int i;

  if (!running)
    return;

  switch(bpp)
    {
    case 8:
      for (i = 0; i < 256; i++)
	{
	  *pal++ = (colors[i].red)   >> 10;
	  *pal++ = (colors[i].green) >> 10;
	  *pal++ = (colors[i].blue)  >> 10;
	}
      break;

    case 16:
      for (i = 0; i < 256; i++)
	{
	  *pal++ = (colors[i].red)   >> 10;
	  *pal++ = (colors[i].green) >> 5;
	  *pal++ = (colors[i].blue)  << 1;
	}
      break;

    case 24:
    case 32:
      for (i = 0; i < 256; i++)
	{
	  *pal++ = (colors[i].red)   >> 2;
	  *pal++ = (colors[i].green) >> 2;
	  *pal++ = (colors[i].blue)  >> 2;
	}
      break;

    default:
      break;
    }
  return;
}

void x11_set_drawing_page(int page)
{
}

void x11_set_viewing_page(int page)
{
}

void x11_bank(int bank)
{
}

unsigned char *x11_get_buffer()
{
  if (!running)
    return NULL;
  else
    return gr_mem;
}

int x11_get_gran()
{
  return granularity;
}

/* Here we copy the memory from gr_mem into image->data, but convert it
   to the appropriate bit depth as we go.  */

static void x11_update_shm(void)
{
  /* The rest of the code is already able to draw directly to
     image->data, so we don't have to memcpy it into the right place */

  XShmPutImage(display, window, gc, image, 0, 0, 0, 0,
	       image->width, image->height, False);
  XFlush(display);
}

static void x11_update_pix(void)
{
  /* The rest of the code is already able to draw directly to
     image->data, so we don't have to memcpy it into the right place */

  XPutImage(display, window, gc, image, 0, 0, 0, 0,
	    image->width, image->height);
  XFlush(display);
}

static void x11_update_nul(void)
{
}

static void x11_update_8(void)
{
  long y, inwidth, outwidth;
  unsigned char *in, *out;

  out = image_data;
  in = gr_mem;
  y = image_height;
  inwidth = image_width;
  outwidth = inwidth + image_line_skip;

  do
    {
      memcpy(out, in, inwidth);
      in += inwidth;
      out += outwidth;
    }
  while (--y > 0);

  /* This is only ever used with DGA -- no need to sync with the server!  */
}

static void x11_update_16(void)
{
  register long y, width, line_skip;
  register unsigned long *out, *in;

  out = (unsigned long *)image_data;
  in = (unsigned long *)gr_mem;
  y = image_height;
  width = image_width / sizeof(long);
  line_skip = image_line_skip;

  assert(((long)out & (sizeof(*out)-1)) == 0);
  assert(((long)in & (sizeof(*in)-1)) == 0);
  assert((width & (sizeof(*in)-1)) == 0);

  do
    {
      long x = width;
      do 
	{
	  unsigned long i0, i1, i2, i3, i4, i5, i6, i7;
	  unsigned long b = *in++;

	  i0 = xlat_table[b & 0xff];
	  i1 = xlat_table[(b >> 8) & 0xff];
	  i2 = xlat_table[(b >> 16) & 0xff];
	  i3 = xlat_table[(b >> 24) & 0xff];
#if SIZEOF_LONG == 8
	  i4 = xlat_table[(b >> 32) & 0xff];
	  i5 = xlat_table[(b >> 40) & 0xff];
	  i6 = xlat_table[(b >> 48) & 0xff];
	  i7 = xlat_table[(b >> 56) & 0xff];
#endif

#if SIZEOF_LONG == 8
	  i0 = (i0 | (i1 << 16)) | ((i2 << 32) | (i3 << 48));
	  i2 = (i4 | (i5 << 16)) | ((i6 << 32) | (i7 << 48));
#else
	  i0 |= i1 << 16;
	  i2 |= i3 << 16;
#endif

	  out[0] = i0;
	  out[1] = i2;
	  out += 2;
	}
      while (--x > 0);
      out = (unsigned long *)((char *)out + line_skip);
    }
  while (--y > 0);

  /* Send out the image */
  (*update_server)();
}

static void x11_update_24(void)
{
  register long wh = image->bytes_per_line/(bpp>>3)*image->height;

  /* 24 bpp code sort of by Brad Hughes <bhughes@arn.net> */
  register unsigned char *im = (unsigned char *)image->data;
  register unsigned char *buffer = gr_mem;
  do
    {
      *(im++) = (unsigned char)colors[*buffer].blue;
      *(im++) = (unsigned char)colors[*buffer].green;
      *(im++) = (unsigned char)colors[*buffer++].red;
    }
  while(--wh > 0);

  /* Send out the image */
  (*update_server)();
}

static void x11_update_32(void)
{
  register long y, width, line_skip;
  register unsigned int *out;
  register unsigned long *in;

  out = (unsigned int *)image_data;
  in = (unsigned long *)gr_mem;
  y = image_height;
  width = image_width / sizeof(long);
  line_skip = image_line_skip;

  assert(((long)out & (sizeof(*out)-1)) == 0);
  assert(((long)in & (sizeof(*in)-1)) == 0);
  assert((width & (sizeof(*in)-1)) == 0);

  do
    {
      long x = width;
      do 
	{
	  unsigned long i0, i1, i2, i3, i4, i5, i6, i7;
	  unsigned long b = *in++;

	  i0 = xlat_table[b & 0xff];
	  i1 = xlat_table[(b >> 8) & 0xff];
	  i2 = xlat_table[(b >> 16) & 0xff];
	  i3 = xlat_table[(b >> 24) & 0xff];
#if SIZEOF_LONG == 8
	  i4 = xlat_table[(b >> 32) & 0xff];
	  i5 = xlat_table[(b >> 40) & 0xff];
	  i6 = xlat_table[(b >> 48) & 0xff];
	  i7 = xlat_table[(b >> 56) & 0xff];
#endif

	  out[0] = i0;
	  out[1] = i1;
	  out[2] = i2;
	  out[3] = i3;
#if SIZEOF_LONG == 8
	  out[4] = i4;
	  out[5] = i5;
	  out[6] = i6;
	  out[7] = i7;
#endif
	  
	  out += sizeof(long);
	}
      while (--x > 0);
      out = (unsigned int *)((char *)out + line_skip);
    }
  while (--y > 0);

  /* Send out the image */
  (*update_server)();
}

inline int keysym_to_descent(KeySym keysym)
{
  if(keysym<255)
    /* no special key */
    return xlib_keycode_to_descent[keysym];
  return xlib_special_to_descent[keysym&0x00ff];
}

int x11_bm_type(void)
{
  return BM_LINEAR;
}

void x11_modex_plane(int plane)
{
  /* FIXME What is this ?? */
}

void x11_keyboard_init(void)
{
}

void x11_keyboard_update(void)
{
  int kcode, temp;
  KeySym keysym;
  XEvent event;
  
  while (XPending(display))
    {
      XNextEvent(display, &event);

      switch (event.type)
	{
	case KeyPress:
	  keysym = XLookupKeysym(&event.xkey,0);
	  kcode  = keysym_to_descent(keysym);
	  keyd_last_pressed = kcode;
	  keyd_time_when_last_pressed = timer_get_fixed_secondsX();

	  if (!keyd_pressed[kcode])
	    {
	      key_data.NumDowns[kcode]++;
	      keyd_pressed[kcode] = 1;
	      key_data.TimeKeyWentDown[kcode] = timer_get_fixed_secondsX();
	    }

	  if (keyd_pressed[KEY_LSHIFT] || keyd_pressed[KEY_RSHIFT])
	    kcode |= KEY_SHIFTED;
	  if (keyd_pressed[KEY_LALT]  || keyd_pressed[KEY_RALT])
	    kcode |= KEY_ALTED;
	  if (keyd_pressed[KEY_LCTRL] || keyd_pressed[KEY_RCTRL])
	    kcode |= KEY_CTRLED;

	  if ((temp = key_data.keytail + 1) >= KEY_BUFFER_SIZE)
	    temp = 0;
	  if (temp!=key_data.keyhead)
	    {
	      key_data.keybuffer[key_data.keytail] = kcode;
	      key_data.time_pressed[key_data.keytail]
		= keyd_time_when_last_pressed;
	      key_data.keytail = temp;
	    }
	  break;

	case KeyRelease:
	  keysym = XLookupKeysym(&event.xkey,0);
	  kcode = keysym_to_descent(keysym);
	  keyd_last_released = kcode;
	  keyd_pressed[kcode] = 0;
	  key_data.NumUps[kcode]++;
	  key_data.TimeKeyHeldDown[kcode] +=
	    timer_get_fixed_secondsX() - key_data.TimeKeyWentDown[kcode];
	  break;

	case MotionNotify:
	  mouse_deltax += event.xmotion.x_root;
	  mouse_deltay += event.xmotion.y_root;
	  break;

	case ButtonPress:
	  {
	    int m, b = event.xbutton.button;
	    m = b;
	    if (b == 2)
	      m = 4;
	    if (b == 3)
	      m = 2;
	    if (b > 3)
	      m = 1 << b;
	    mouse_state |= m;
	  }
	  break;

	case ButtonRelease:
	  {
	    int m, b = event.xbutton.button;
	    m = b;
	    if (b == 2)
	      m = 4;
	    if (b == 3)
	      m = 2;
	    if (b > 3)
	      m = 1 << b;
	    mouse_state &= ~m;
	  }
	  break;
	case FocusIn:
	  if (!x11_have_focus)
	    {
	      XKeyboardState kbstate;
	      
	      XGetKeyboardControl(display, &kbstate);
	      x11_key_repeat = kbstate.global_auto_repeat;
	      XAutoRepeatOff(display);
	      x11_have_focus = 1;
	    }
	  break;  
	case FocusOut:
	  if (x11_key_repeat)
	    {
	      XAutoRepeatOn(display);
	      x11_key_repeat = 0;
	    }
	  x11_have_focus = 0;
	  break;
	}
    }
}

void x11_keyboard_close(void)
{
}

/* Mouse stuff should be down here. */
int x11_mouse_init(void)
{
  return 0;
}

void x11_mouse_close(void) 
{
}

void x11_mouse_update(int *pdx, int *pdy, int *buttons)
{
  if (usingDGA)
    {
      x11_keyboard_update();

      *pdx = mouse_deltax;
      *pdy = mouse_deltay;
      mouse_deltax = 0;
      mouse_deltay = 0;

      *buttons = mouse_state;
    }
  else
    {
      *pdx = 0;
      *pdy = 0;
      *buttons = 0;
    }
}

/* This takes care of the X protocol error we're going to get
   when we try to attach our shared memory to a non-local system. */
int x11_error_handler(Display *display, XErrorEvent *event)
{
  /* "However, the error handler should not call any functions
     (directly or indirectly) on the display that will generate
     protocol requests or that will look for input events." */
  char string[1024];

  /* All we want to trap is the SHM errors. Everything else
     we need to print an error message for and actually exit. */
  if (event->request_code == 129)
    {
      usingShm = False;
      return 0;
    }
  else
    {
fflush(stdout);    
      XGetErrorText(display, event->error_code, string, sizeof(string));
      printf("X Error of failed request: %s\n"
	     "  Major opcode of failed request:  %i\n"
	     "  Minor opcode of failed request:  %i\n"
	     "  Serial number of failed request:  %i\n"
	     "  Current serial number in output stream:  %i\n",
	     string, event->request_code, event->minor_code,
	     (int)event->serial, (int)LastKnownRequestProcessed(display));
      exit(-1);
    }

  /* "Because this condition is not assumed to be fatal, it is
     acceptable for your error handler to return; the returned
     value is ignored." */
  return 0;
}

void x11_close_down(void)
{
  if (running)
    {
      printf("Closing down X\n");
      x11_release_image();
      if (x11_key_repeat)
        XAutoRepeatOn(display);
      if (!usingDGA)
	      XUnmapWindow(display, window);
      XCloseDisplay(display);
      display = NULL;
      running = 0;
    }
}

#ifdef HAVE_X11_DGA
#ifdef HAVE_SIGCONTEXT_H
static void x11_dga_cleanup(int sig, int code, struct sigcontext *c)
#else
static void x11_dga_cleanup(int sig)
#endif
{
  XF86DGADirectVideo(display, 0, 0);
  XSync(display, False);

  printf("Yikes! Caught signal %d (%s) -- turning off DGA\n",
	 sig, strsignal(sig));
  #ifdef HAVE_SIGCONTEXT_H
  printf("Fault at %p\n", c->sc_pc);
  #endif

  /* Turn off handling, return and core.  */
  signal(sig, SIG_DFL);
  if (sig != SIGSEGV)
    kill(getpid(), sig);
}
#endif

#endif
