//
//  HPIUtil.DLL
//  Routines for accessing HPI files

#include <windows.h>
#include <stdlib.h>
#include <stdio.h>
#include "HPIUtil.h"

#include "zlib.h"

#define HEX_HAPI 0x49504148
// 'BANK'
#define HEX_BANK 0x4B4E4142
// 'SQSH'
#define HEX_SQSH 0x48535153
#define HEX_MARKER 0x00010000

#define OUTBLOCKSIZE 16384

#define INBLOCKSIZE (65536+17)

#define TADIR_REG_ENTRY "SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Uninstall\\Total Annihilation"

#pragma pack(1)
typedef struct _HPIHEADER {
  long HPIMarker;              /* 'HAPI' */
  long SaveMarker;             /* 'BANK' if savegame */
  long DirectorySize;                /* Directory size */
  long Key;                    /* decode key */
  long Start;                  /* offset of directory */
} HPIHEADER;

typedef struct _HPIENTRY {
  int NameOffset;
  int CountOffset;
  char Flag;
} HPIENTRY;

typedef struct _HPICHUNK {
  long Marker;            /* always 0x48535153 (SQSH) */
  char Unknown1;          /* I have no idea what these mean */
	char CompMethod;				/* 1=lz77 2=zlib */
	char Encrypt;           /* Is the chunk encrypted? */
  long CompressedSize;    /* the length of the compressed data */
  long DecompressedSize;  /* the length of the decompressed data */
  long Checksum;          /* check sum */
} HPICHUNK;

typedef struct _HPIFILE {
	HANDLE f;   // handle to open file
	LPSTR d;    // pointer to decrypted directory
	int Key;	  // Key
	int Start;  // Start of directory
} HPIFILE;

typedef struct _DIRENTRY {
	struct _DIRENTRY *Next;
	struct _DIRENTRY *Prev;
	struct _DIRENTRY *FirstSub;
	struct _DIRENTRY *LastSub;
  int dirflag;
	int count;
	int *DirOffset;
	LPSTR Name;
  LPSTR FileName;
	int memflag;
} DIRENTRY;

typedef struct _TREENODE {
  struct _TREENODE *tree_l;
  struct _TREENODE *tree_r;
  int tree_b;
  void *tree_p;
} TREENODE;

typedef struct _PACKFILE {
  LPSTR FileName;
  DIRENTRY *Root;
  int DirSize;
  int Key;
  FILE *HPIFile;
  LPSTR Window;
  int WinLen;
  int WinOfs;
  int WIn;
  int WOut;
  int ChunkPtr;
  int BlockSize;
  TREENODE *SearchTree;
  HANDLE TreeHeap;
  TREENODE *TreeArray;
  int NextNode;
  HANDLE ThreadHandle;
  HPICALLBACK CallBack;
  int FileCount;
  int FileCountTotal;
  int FileBytes;
  int FileBytesTotal;
  int TotalBytes;
  int TotalBytesTotal;
  int Stop;
} PACKFILE;

#pragma pack()

HINSTANCE hThisInstance;
const char szAppTitle[] = "HPIUtil";
const char szTrailer[] = "HPIPack by Joe D (joed@cws.org) FNORD Total Annihilation Copyright 1997 Cavedog Entertainment";
char DirPath[256];
HPIENTRY *DirEntry;
HANDLE ProgHeap = NULL;

static LPVOID GetMem(int size, int zero)
{
  return HeapAlloc(ProgHeap, (zero ? HEAP_ZERO_MEMORY : 0), size); 
}

static void FreeMem(LPVOID x)
{
  HeapFree(ProgHeap, 0, x);
}

static LPSTR DupString(LPSTR x)
{
  LPSTR s = GetMem(strlen(x)+1, FALSE);
  strcpy(s, x);
  return s;
}

/*  tree stuff  */

/* as_tree - tree library for as
 * vix 14dec85 [written]
 * vix 02feb86 [added tree balancing from wirth "a+ds=p" p. 220-221]
 * vix 06feb86 [added TreeDestroy()]
 * vix 20jun86 [added TreeDelete per wirth a+ds (mod2 v.) p. 224]
 * vix 23jun86 [added Delete2 uar to add for replaced nodes]
 * vix 22jan93 [revisited; uses RCS, ANSI, POSIX; has bug fixes]
 */


/* This program text was created by Paul Vixie using examples from the book:
 * "Algorithms & Data Structures," Niklaus Wirth, Prentice-Hall, 1986, ISBN
 * 0-13-022005-1. This code and associated documentation is hereby placed
 * in the public domain.
 */

typedef int (*TREECOMPFUNC)(PACKFILE *Pack, void *k1, void *k2);

TREENODE *NewNode(PACKFILE *Pack)
{
  return Pack->TreeArray+(Pack->NextNode++);
}

static void Sprout(PACKFILE *Pack, TREENODE **ppr, void *p_data, int *pi_balance, TREECOMPFUNC TreeComp)
{
  TREENODE *p1;
  TREENODE *p2;
  int cmp;

  /* are we grounded?  if so, add the node "here" and set the rebalance
   * flag, then exit.
   */
  if (!*ppr) {
    *ppr = (TREENODE *) NewNode(Pack);
    (*ppr)->tree_l = NULL;
    (*ppr)->tree_r = NULL;
    (*ppr)->tree_b = 0;
    (*ppr)->tree_p = p_data;
    *pi_balance = TRUE;
    return;
  }

  /* compare the data using routine passed by caller.
   */
  cmp = TreeComp(Pack, p_data, (*ppr)->tree_p);

  /* if LESS, prepare to move to the left.
   */
  if (cmp < 0) {
    Sprout(Pack, &(*ppr)->tree_l, p_data, pi_balance, TreeComp);
    if (*pi_balance) {  /* left branch has grown longer */
      switch ((*ppr)->tree_b) {
        case 1: /* right branch WAS longer; balance is ok now */
          (*ppr)->tree_b = 0;
          *pi_balance = FALSE;
          break;
        case 0: /* balance WAS okay; now left branch longer */
          (*ppr)->tree_b = -1;
          break;
        case -1:
          /* left branch was already too long. rebalnce */
          p1 = (*ppr)->tree_l;
          if (p1->tree_b == -1) { /* LL */
            (*ppr)->tree_l = p1->tree_r;
            p1->tree_r = *ppr;
            (*ppr)->tree_b = 0;
            *ppr = p1;
          }
          else {      /* double LR */
            p2 = p1->tree_r;
            p1->tree_r = p2->tree_l;
            p2->tree_l = p1;
          
            (*ppr)->tree_l = p2->tree_r;
            p2->tree_r = *ppr;
          
            if (p2->tree_b == -1)
              (*ppr)->tree_b = 1;
            else
              (*ppr)->tree_b = 0;
          
            if (p2->tree_b == 1)
              p1->tree_b = -1;
            else
              p1->tree_b = 0;
            *ppr = p2;
          }
        (*ppr)->tree_b = 0;
        *pi_balance = FALSE;
        break;
      } 
    }
    return;
  }

  /* if MORE, prepare to move to the right.
   */
  if (cmp > 0) {
    Sprout(Pack, &(*ppr)->tree_r, p_data, pi_balance, TreeComp);
    if (*pi_balance) {  /* right branch has grown longer */
      switch ((*ppr)->tree_b) {
        case -1:
          (*ppr)->tree_b = 0;
          *pi_balance = FALSE;
          break;
        case 0:
          (*ppr)->tree_b = 1;
          break;
        case 1:
          p1 = (*ppr)->tree_r;
          if (p1->tree_b == 1) {  /* RR */
            (*ppr)->tree_r = p1->tree_l;
            p1->tree_l = *ppr;
            (*ppr)->tree_b = 0;
            *ppr = p1;
          }
          else {      /* double RL */
            p2 = p1->tree_l;
            p1->tree_l = p2->tree_r;
            p2->tree_r = p1;

            (*ppr)->tree_r = p2->tree_l;
            p2->tree_l = *ppr;

            if (p2->tree_b == 1)
              (*ppr)->tree_b = -1;
            else
              (*ppr)->tree_b = 0;

            if (p2->tree_b == -1)
              p1->tree_b = 1;
            else
              p1->tree_b = 0;

            *ppr = p2;
          }
        (*ppr)->tree_b = 0;
        *pi_balance = FALSE;
        break;
      } 
    }
    return;
  }

  /* not less, not more: this is the same key!  replace...
   */
  *pi_balance = FALSE;
  (*ppr)->tree_p = p_data;
}

static void BalanceLeft(PACKFILE *Pack, TREENODE **ppr_p, int *pi_balance)
{
  TREENODE  *p1, *p2;
  int b1, b2;

  switch ((*ppr_p)->tree_b) {
  case -1:
    (*ppr_p)->tree_b = 0;
    break;
  case 0:
    (*ppr_p)->tree_b = 1;
    *pi_balance = FALSE;
    break;
  case 1:
    p1 = (*ppr_p)->tree_r;
    b1 = p1->tree_b;
    if (b1 >= 0) {
      (*ppr_p)->tree_r = p1->tree_l;
      p1->tree_l = *ppr_p;
      if (b1 == 0) {
        (*ppr_p)->tree_b = 1;
        p1->tree_b = -1;
        *pi_balance = FALSE;
      } 
      else {
        (*ppr_p)->tree_b = 0;
        p1->tree_b = 0;
      }
      *ppr_p = p1;
    } else {
      p2 = p1->tree_l;
      b2 = p2->tree_b;
      p1->tree_l = p2->tree_r;
      p2->tree_r = p1;
      (*ppr_p)->tree_r = p2->tree_l;
      p2->tree_l = *ppr_p;
      if (b2 == 1)
        (*ppr_p)->tree_b = -1;
      else
        (*ppr_p)->tree_b = 0;
      if (b2 == -1)
        p1->tree_b = 1;
      else
        p1->tree_b = 0;
      *ppr_p = p2;
      p2->tree_b = 0;
    }
  }
}

static void BalanceRight(PACKFILE *Pack, TREENODE **ppr_p, int *pi_balance)
{
  TREENODE  *p1, *p2;
  int b1, b2;

  switch ((*ppr_p)->tree_b) {
  case 1: 
    (*ppr_p)->tree_b = 0;
    break;
  case 0:
    (*ppr_p)->tree_b = -1;
    *pi_balance = FALSE;
    break;
  case -1:
    p1 = (*ppr_p)->tree_l;
    b1 = p1->tree_b;
    if (b1 <= 0) {
      (*ppr_p)->tree_l = p1->tree_r;
      p1->tree_r = *ppr_p;
      if (b1 == 0) {
        (*ppr_p)->tree_b = -1;
        p1->tree_b = 1;
        *pi_balance = FALSE;
      } else {
        (*ppr_p)->tree_b = 0;
        p1->tree_b = 0;
      }
      *ppr_p = p1;
    } else {
      p2 = p1->tree_r;
      b2 = p2->tree_b;
      p1->tree_r = p2->tree_l;
      p2->tree_l = p1;
      (*ppr_p)->tree_l = p2->tree_r;
      p2->tree_r = *ppr_p;
      if (b2 == -1)
        (*ppr_p)->tree_b = 1;
      else
        (*ppr_p)->tree_b = 0;
      if (b2 == 1)
        p1->tree_b = -1;
      else
        p1->tree_b = 0;
      *ppr_p = p2;
      p2->tree_b = 0;
    }
  }
}

static void Delete3(PACKFILE *Pack, TREENODE  **ppr_r, int  *pi_balance, TREENODE **ppr_q)
{
  if ((*ppr_r)->tree_r != NULL) {
    Delete3(Pack, &(*ppr_r)->tree_r, pi_balance, ppr_q);
    if (*pi_balance)
      BalanceRight(Pack, ppr_r, pi_balance);
  } else {
    (*ppr_q)->tree_p = (*ppr_r)->tree_p;
    *ppr_q = *ppr_r;
    *ppr_r = (*ppr_r)->tree_l;
    *pi_balance = TRUE;
  }
}

static int Delete2(PACKFILE *Pack, TREENODE **ppr_p, TREECOMPFUNC TreeComp, void *p_user, int *pi_balance)
{
  TREENODE  *pr_q;
  int i_comp, i_ret;

  if (*ppr_p == NULL) {
    return FALSE;
  }

  i_comp = TreeComp(Pack, (*ppr_p)->tree_p, p_user);
  if (i_comp > 0) {
    i_ret = Delete2(Pack, &(*ppr_p)->tree_l, TreeComp, p_user, pi_balance);
    if (*pi_balance)
      BalanceLeft(Pack, ppr_p, pi_balance);
  }
  else if (i_comp < 0) {
    i_ret = Delete2(Pack, &(*ppr_p)->tree_r, TreeComp, p_user, pi_balance);
    if (*pi_balance)
      BalanceRight(Pack, ppr_p, pi_balance);
  }
  else {
    pr_q = *ppr_p;
    if (pr_q->tree_r == NULL) {
      *ppr_p = pr_q->tree_l;
      *pi_balance = TRUE;
    }
    else if (pr_q->tree_l == NULL) {
      *ppr_p = pr_q->tree_r;
      *pi_balance = TRUE;
    }
    else {
      Delete3(Pack, &pr_q->tree_l, pi_balance, &pr_q);
      if (*pi_balance)
        BalanceLeft(Pack, ppr_p, pi_balance);
    }
    //TreeFreeMem(Pack, pr_q); /* thanks to wuth@castrov.cuc.ab.ca */
    i_ret = TRUE;
  }
  return i_ret;
}


void TreeInit(PACKFILE *Pack)
{
  Pack->SearchTree = NULL;
  if (!Pack->TreeHeap) {
    Pack->TreeHeap = HeapCreate(0, 65536 * sizeof(TREENODE), 0);
  }
  if (!Pack->TreeArray)
    Pack->TreeArray = HeapAlloc(Pack->TreeHeap, 0, 65536 * sizeof(TREENODE));
  Pack->NextNode = 0;
}

void *TreeSearch(PACKFILE *Pack, TREENODE **ppr_tree, TREECOMPFUNC TreeComp, void *p_user)
{
  int i_comp;

  if (*ppr_tree) {
    i_comp = TreeComp(Pack, p_user, (**ppr_tree).tree_p);
    if (i_comp > 0)
      return TreeSearch(Pack, &(**ppr_tree).tree_r, TreeComp, p_user);
    if (i_comp < 0)
      return TreeSearch(Pack, &(**ppr_tree).tree_l, TreeComp, p_user);

    /* not higher, not lower... this must be the one.
     */
    return (**ppr_tree).tree_p;
  }

  /* grounded. NOT found.
   */
  return NULL;
}

void TreeAdd(PACKFILE *Pack, TREENODE **ppr_tree, TREECOMPFUNC TreeComp, void *p_user)
{
  int i_balance = FALSE;

  Sprout(Pack, ppr_tree, p_user, &i_balance, TreeComp);
}

int TreeDelete(PACKFILE *Pack, TREENODE  **ppr_p, TREECOMPFUNC TreeComp, void *p_user)
{
  int i_balance = FALSE;

  return Delete2(Pack, ppr_p, TreeComp, p_user, &i_balance);
}

void TreeDestroy(PACKFILE *Pack)
{
  Pack->NextNode = 0;
}


/* End tree stuff  */

static int ReadAndDecrypt(HPIFILE *hpi, int fpos, LPSTR buff, int buffsize)
{
	int count;
	int tkey;
	int result;
	
	SetFilePointer(hpi->f, fpos, NULL, FILE_BEGIN); 
  if (!ReadFile(hpi->f, buff, buffsize, &result, NULL))
		return 0;
	if (hpi->Key) {
	  for (count = 0; count < result; count++) {
  		tkey = (fpos + count) ^ hpi->Key;
      buff[count] = tkey ^ ~buff[count];
		}
	}
	return result;
}

int ZLibDecompress(char *out, char *in, HPICHUNK *Chunk)
{
  z_stream zs;
  int result;

  zs.next_in = in;
  zs.avail_in = Chunk->CompressedSize;
  zs.total_in = 0;

  zs.next_out = out;
  zs.avail_out = 65536;
  zs.total_out = 0;

  zs.msg = NULL;
  zs.state = NULL;
  zs.zalloc = Z_NULL;
  zs.zfree = Z_NULL;
  zs.opaque = NULL;

  zs.data_type = Z_BINARY;
  zs.adler = 0;
  zs.reserved = 0;

  result = inflateInit(&zs);
  if (result != Z_OK) {
    return 0;
  }

  result = inflate(&zs, Z_FINISH);
  if (result != Z_STREAM_END) {
    zs.total_out = 0;
  }

  result = inflateEnd(&zs);
  if (result != Z_OK) {
    return 0;
  }

	return zs.total_out;
}

static int LZ77Decompress(char *out, char *in, HPICHUNK *Chunk)
{
	int x;
	int work1;
	int work2;
	int work3;
	int inptr;
	int outptr;
	int count;
	int done;
	char DBuff[4096];
	int DPtr;

	done = FALSE;

  inptr = 0;
	outptr = 0;
	work1 = 1;
	work2 = 1;
	work3 = in[inptr++];
	
	while (!done) {
	  if ((work2 & work3) == 0) {
  		out[outptr++] = in[inptr];
		  DBuff[work1] = in[inptr];
		  work1 = (work1 + 1) & 0xFFF;
		  inptr++;
		}
	  else {
  		count = *((unsigned short *) (in+inptr));
			inptr += 2;
			DPtr = count >> 4;
			if (DPtr == 0) {
				return outptr;
			}
			else {
				count = (count & 0x0f) + 2;
				if (count >= 0) {
					for (x = 0; x < count; x++) {
						out[outptr++] = DBuff[DPtr];
						DBuff[work1] = DBuff[DPtr];
						DPtr = (DPtr + 1) & 0xFFF;
		        work1 = (work1 + 1) & 0xFFF;
					}

				}
			}
		}
		work2 *= 2;
		if (work2 & 0x0100) {
			work2 = 1;
			work3 = in[inptr++];
		}
	}

	return outptr;

}

static int Decompress(char *out, char *in, HPICHUNK *Chunk)
{
	int x;
	int Checksum;

	Checksum = 0;
	for (x = 0; x < Chunk->CompressedSize; x++) {
		Checksum += (unsigned char) in[x];
		if (Chunk->Encrypt)
		  in[x] = (in[x] - x) ^ x;
	}

	if (Chunk->Checksum != Checksum) {
		return 0;
	}

	switch (Chunk->CompMethod) {
		case 1 : return LZ77Decompress(out, in, Chunk);
		case 2 : return ZLibDecompress(out, in, Chunk);
		default : return 0;
	}
}



static LPSTR DecodeFileToMem(HPIFILE *hpi, HPIENTRY *Entry)
{
  HPICHUNK *Chunk;
	long *DeSize;
	int DeCount;
	int DeLen;
	int x;
	char *DeBuff;
	char *WriteBuff;
	int WriteSize;
	int WritePtr;

	int Offset;
	int Length;
	char FileFlag;

	if (!Entry)
		return NULL;

	Offset = *((int *) (hpi->d + Entry->CountOffset));
	Length = *((int *) (hpi->d + Entry->CountOffset + 4));
	FileFlag = *(hpi->d + Entry->CountOffset + 8);

	//if (FileFlag != 1)
	//	return NULL;

	WriteBuff = GlobalAlloc(GPTR, Length+1);
	if (!WriteBuff) {
		return NULL;
	}
	WriteBuff[Length] = 0;

	if (FileFlag) {

  	DeCount = Length / 65536;
  	if (Length % 65536)
		  DeCount++;
	  DeLen = DeCount * sizeof(int);

	  DeSize = GetMem(DeLen, TRUE);

	  ReadAndDecrypt(hpi, Offset, (char *) DeSize, DeLen);

  	Offset += DeLen;
	
	  WritePtr = 0;

	  for (x = 0; x < DeCount; x++) {
		  Chunk = GetMem(DeSize[x], TRUE);
		  ReadAndDecrypt(hpi, Offset, (char *) Chunk, DeSize[x]);
		  Offset += DeSize[x];

		  DeBuff = (char *) (Chunk+1);

		  WriteSize = Decompress(WriteBuff+WritePtr, DeBuff, Chunk);
		  WritePtr += WriteSize;

		  FreeMem(Chunk);
		}
  	FreeMem(DeSize);
	}
	else {
	  ReadAndDecrypt(hpi, Offset, WriteBuff, Length);
	}

	return WriteBuff;
}

void WINAPI GetTADirectory(LPSTR TADir)
{
	HKEY hk;
	int size = MAX_PATH;

	if (RegOpenKeyEx(HKEY_LOCAL_MACHINE, TADIR_REG_ENTRY, 0, KEY_READ, &hk)) {
		*TADir = 0;
	  return;
	}
	if (RegQueryValueEx(hk, "Dir", 0, NULL, TADir, &size)) {
		*TADir = 0;
	}
	RegCloseKey(hk);
}

LPVOID WINAPI HPIOpen(LPSTR FileName)
{
  HANDLE f;
	HPIHEADER Header;
	int BytesRead;
	HPIFILE *hpi;

	f = CreateFile(FileName, GENERIC_READ, (FILE_SHARE_READ | FILE_SHARE_WRITE),
		NULL, OPEN_EXISTING, (FILE_ATTRIBUTE_NORMAL | FILE_FLAG_RANDOM_ACCESS),
		NULL);

	if (f == INVALID_HANDLE_VALUE) {
		return NULL;
	}

	if (!ReadFile(f, &Header, sizeof(Header), &BytesRead, NULL)) {
		CloseHandle(f);
		return NULL;
	}

	if (BytesRead != sizeof(Header)) {
		CloseHandle(f);
		return NULL;
	}

  if (Header.HPIMarker != HEX_HAPI) {  /* 'HAPI' */
		CloseHandle(f);
		return NULL;
	}

  if (Header.SaveMarker != HEX_MARKER) {
		CloseHandle(f);
		return NULL;
	}

	hpi = GetMem(sizeof(HPIFILE), TRUE);

	hpi->f = f;
	if (Header.Key)
	  hpi->Key = ~((Header.Key * 4)	| (Header.Key >> 6));
	else
		hpi->Key = 0;

	hpi->Start = Header.Start;
	hpi->d = GetMem(Header.DirectorySize, TRUE);
	BytesRead = ReadAndDecrypt(hpi, Header.Start, hpi->d+Header.Start, Header.DirectorySize-Header.Start);
	if (BytesRead != (Header.DirectorySize-Header.Start)) {
		FreeMem(hpi->d);
		FreeMem(hpi);
		CloseHandle(f);
		return NULL;
	}
  return hpi;
}

static int ListDirectory(HPIFILE *hpi, int offset, LPSTR Path, int Next, int This)
{
	int *Entries;
	HPIENTRY *Entry;
	int count;
	char *Name;
	int *EntryOffset;
	char MyPath[1024];

	Entries = (int *) (hpi->d + offset);
	EntryOffset = Entries + 1;
	Entry = (HPIENTRY *) (hpi->d + *EntryOffset);

  for (count = 0; count < *Entries; count++) {

		Name = hpi->d + Entry->NameOffset;
    strcpy(MyPath, Path);
		if (*Path)
		  strcat(MyPath, "\\");
		strcat(MyPath, Name);

		if (This == Next) {
			if (!DirEntry) {
			  DirEntry = Entry;
				lstrcpy(DirPath, MyPath);
			}
			return This;
		}
		This++;
		if (Entry->Flag == 1)	{
			This = ListDirectory(hpi, Entry->CountOffset, MyPath, Next, This);
		}
		Entry++;
	}
	return This;
}

LRESULT WINAPI HPIGetFiles(HPIFILE *hpi, int Next, LPSTR Name, LPINT Type, LPINT Size)
{
	int result;
	int *Count;

	*Name = 0;
	*Type = 0;
	*Size = 0;

	DirEntry = NULL;
	DirPath[0] = 0;

	result = ListDirectory(hpi, hpi->Start, "", Next, 0);
	if (!DirEntry)
		return 0;

	lstrcpy(Name, DirPath);
	Count = (int *) (hpi->d + DirEntry->CountOffset);
	*Type = DirEntry->Flag;
	if (!*Type)
		Count++;
  *Size = *Count;

	return result+1;
}

static int EnumDirectory(HPIFILE *hpi, int offset, LPSTR Path, int Next, LPSTR DirName, int This)
{
	int *Entries;
	HPIENTRY *Entry;
	int count;
	int *EntryOffset;
	char MyPath[1024];

	Entries = (int *) (hpi->d + offset);
	EntryOffset = Entries + 1;
	Entry = (HPIENTRY *) (hpi->d + *EntryOffset);

	//MessageBox(NULL, Path, "Searching", MB_OK);

  for (count = 0; count < *Entries; count++) {
		if (lstrcmpi(Path, DirName) == 0) {
  		if (This >= Next) {
	  		if (!DirEntry) {
		  	  DirEntry = Entry;
	        //MessageBox(NULL, hpi->d + Entry->NameOffset, "Found", MB_OK);
			  	//lstrcpy(DirPath, MyPath);
				}
			  return This;
			}
  		This++;
		}
		if (Entry->Flag == 1)	{
      lstrcpy(MyPath, Path);
		  if (*Path)
		    lstrcat(MyPath, "\\");
		  lstrcat(MyPath, (hpi->d + Entry->NameOffset));
			//MessageBox(NULL, MyPath, "recursing", MB_OK);
			This = EnumDirectory(hpi, Entry->CountOffset, MyPath, Next, DirName, This);
			if (DirEntry)
				return This;
		}

		Entry++;
	}
	return This;
}

LRESULT WINAPI HPIDir(HPIFILE *hpi, int Next, LPSTR DirName, LPSTR Name, LPINT Type, LPINT Size)
{
	int result;
	int *Count;

	*Name = 0;
	*Type = 0;
	*Size = 0;

	DirEntry = NULL;

	result = EnumDirectory(hpi, hpi->Start, "", Next, DirName, 0);
	if (!DirEntry)
		return 0;

	lstrcpy(Name, (hpi->d + DirEntry->NameOffset));
	Count = (int *) (hpi->d + DirEntry->CountOffset);
	*Type = DirEntry->Flag;
	if (!*Type)
		Count++;
  *Size = *Count;

	return result+1;
}

LRESULT WINAPI HPIClose(HPIFILE *hpi)
{
	if (!hpi)
		return 0;

	if (hpi->f)
		CloseHandle(hpi->f);
	if (hpi->d)
		FreeMem(hpi->d);
	FreeMem(hpi);
	return 0;
}

static LPSTR SearchForFile(HPIFILE *hpi, int offset, LPSTR Path, LPSTR FName)
{
	int *Entries;
	HPIENTRY *Entry;
	int count;
	char *Name;
	int *EntryOffset;
	char MyPath[1024];
	LPSTR result;

	Entries = (int *) (hpi->d + offset);
	EntryOffset = Entries + 1;
	Entry = (HPIENTRY *) (hpi->d + *EntryOffset);

  for (count = 0; count < *Entries; count++) {

		Name = hpi->d + Entry->NameOffset;
    strcpy(MyPath, Path);
		if (*Path)
		  strcat(MyPath, "\\");
		strcat(MyPath, Name);

		if (Entry->Flag == 1)	{
			result = SearchForFile(hpi, Entry->CountOffset, MyPath, FName);
			if (result)
				return result;
		}
		else {
			if (lstrcmpi(MyPath, FName) == 0) {
				DirEntry = Entry;
        return DecodeFileToMem(hpi, Entry);
			}

		}
		Entry++;
	}
	return NULL;
}

LPSTR WINAPI HPIOpenFile(HPIFILE *hpi, LPSTR FileName)
{
	DirEntry = NULL;
	return SearchForFile(hpi, hpi->Start, "", FileName);
}

void WINAPI HPIGet(LPSTR Dest, LPSTR FileHandle, int offset, int bytecount)
{
  MoveMemory(Dest, FileHandle+offset, bytecount);
}

LRESULT WINAPI HPICloseFile(LPSTR FileHandle)
{
	if (FileHandle)
		GlobalFree(FileHandle);
	return 0;
}

LRESULT WINAPI HPIExtractFile(HPIFILE *hpi, LPSTR FileName, LPSTR ExtractName)
{
	LPSTR data;
	HANDLE f;
	int *Size;
	int Written;

	DirEntry = NULL;
	data = SearchForFile(hpi, hpi->Start, "", FileName);
	if (!data)
		return 0;

	Size = (int *) (hpi->d + DirEntry->CountOffset);
	if (!DirEntry->Flag)
		Size++;
	else {
		HPICloseFile(data);
		return 0;
	}

	f = CreateFile(ExtractName, GENERIC_WRITE, 0,
		NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL,
		NULL);

	if (f == INVALID_HANDLE_VALUE) {
		HPICloseFile(data);
		return 0;
	}

	WriteFile(f, data, *Size, &Written, NULL);
	CloseHandle(f);

	HPICloseFile(data);

	return 1;
}

LPVOID WINAPI HPICreate(LPSTR FileName, HPICALLBACK CallBack)
{
  PACKFILE *Pack;

  Pack = GetMem(sizeof(PACKFILE), TRUE);
  Pack->FileName = DupString(FileName);
  Pack->Root = GetMem(sizeof(DIRENTRY), TRUE);
  Pack->CallBack = CallBack;
  return Pack;
}

static DIRENTRY *CreatePath(DIRENTRY *dir, LPSTR FName)
{
  LPSTR p;
  char Path[MAX_PATH];
  LPSTR n;
  DIRENTRY *d;
  DIRENTRY *e;

  p = FName;
  d = dir->FirstSub;
  while (*p) {
    n = Path;
    while (*p && (*p != '\\'))
      *n++ = *p++;
    *n = 0;
    if (*p)
      p++;

    while (d) {
      if (stricmp(d->Name, Path) == 0) {
        if (d->dirflag) {
          break;
        }
        else
          return NULL; // found, but not a directory
      }
      d = d->Next;
    }
    if (d) {
      dir = d;
    }
    else {
      e = GetMem(sizeof(DIRENTRY), TRUE);
      e->dirflag = TRUE;
      e->Name = DupString(Path);
      if (dir->LastSub) {
        dir->LastSub->Next = e;
        e->Prev = dir->LastSub;
        dir->LastSub = e;
      }
      else {
        dir->FirstSub = e;
        dir->LastSub = e;
      }
      dir = e;
    }
    d = dir->FirstSub;
  }
  return dir;
}

LRESULT WINAPI HPICreateDirectory(PACKFILE *Pack, LPSTR DirName)
{
  return (CreatePath(Pack->Root, DirName) != NULL);
}

LRESULT WINAPI HPIAddFile(PACKFILE *Pack, LPSTR HPIName, LPSTR FileName)
{
  char DirName[MAX_PATH];
  LPSTR c;
  FILE *f;
  DIRENTRY *dir;
  DIRENTRY *d;
  LPSTR TName;
  int fsize;

  f = fopen(FileName, "rb");
  if (!f)
    return FALSE;
  fseek(f, 0, SEEK_END);
  fsize = ftell(f);
  fclose(f);

  strcpy(DirName, HPIName);
  c = strrchr(DirName, '\\');
  if (c) {
    TName = c+1;
    *c = 0;
    dir = CreatePath(Pack->Root, DirName);
  }
  else {
    dir = Pack->Root;
    TName = DirName;
  }

  if (!dir)
    return FALSE;
 
  d = dir->FirstSub;
  while (d) {
    if (stricmp(d->Name, TName) == 0)
      return FALSE;
    d = d->Next;
  }

  Pack->FileCountTotal++;
  Pack->TotalBytesTotal += fsize;
  d = GetMem(sizeof(DIRENTRY), TRUE);
  d->dirflag = FALSE;
  d->count = fsize;
  d->Name = DupString(TName);
  d->FileName = DupString(FileName);
	d->memflag = FALSE;
  if (dir->LastSub) {
    dir->LastSub->Next = d;
    d->Prev = dir->LastSub;
    dir->LastSub = d;
  }
  else {
    dir->FirstSub = d;
    dir->LastSub = d;
  }
  return TRUE;
}

LRESULT WINAPI HPIAddFileFromMemory(PACKFILE *Pack, LPSTR HPIName, LPSTR FileBlock, int fsize)
{
  char DirName[MAX_PATH];
  LPSTR c;
  DIRENTRY *dir;
  DIRENTRY *d;
  LPSTR TName;

  strcpy(DirName, HPIName);
  c = strrchr(DirName, '\\');
  if (c) {
    TName = c+1;
    *c = 0;
    dir = CreatePath(Pack->Root, DirName);
  }
  else {
    dir = Pack->Root;
    TName = DirName;
  }

  if (!dir)
    return FALSE;
 
  d = dir->FirstSub;
  while (d) {
    if (stricmp(d->Name, TName) == 0)
      return FALSE;
    d = d->Next;
  }

  Pack->FileCountTotal++;
  Pack->TotalBytesTotal += fsize;
  d = GetMem(sizeof(DIRENTRY), TRUE);
  d->dirflag = FALSE;
  d->count = fsize;
  d->Name = DupString(TName);
  d->FileName = FileBlock;
	d->memflag = TRUE;
  if (dir->LastSub) {
    dir->LastSub->Next = d;
    d->Prev = dir->LastSub;
    dir->LastSub = d;
  }
  else {
    dir->FirstSub = d;
    dir->LastSub = d;
  }
  return TRUE;
}

static int BuildDirectoryBlock(LPSTR Directory, DIRENTRY *dir, int DStart, int CMethod)
{
	int c = DStart;
	int nofs;
	char *n;
	DIRENTRY *d;

  *((int *)(Directory+c)) = dir->count;
	c += sizeof(int);
	*((int *)(Directory+c)) = c+sizeof(int);
	c += sizeof(int);
	nofs = c + (dir->count * sizeof(HPIENTRY));
  d = dir->FirstSub;
  while (d) {
    *((int *)(Directory+c)) = nofs;
	  c += sizeof(int);
    n = d->Name;
    while (*n)
   		Directory[nofs++] = *n++;
 	  Directory[nofs++] = 0;
    *((int *)(Directory+c)) = nofs;
	  c += sizeof(int);
 	  if (d->dirflag) {
		  Directory[c++] = 1;
	    nofs = BuildDirectoryBlock(Directory, d, nofs, CMethod);
		}
    else {
		  Directory[c++] = 0;
		  d->DirOffset = (int *) (Directory+nofs);
		  nofs += sizeof(int);
      *((int *)(Directory+nofs)) = d->count;
		  nofs += sizeof(int);
		  Directory[nofs++] = CMethod;
		}
		d = d->Next;
	}
	return nofs;
}

static int EncryptAndWrite(PACKFILE *Pack, int fpos, void *b, int buffsize)
{
	int count;
	int tkey;
	int result;
	char *buff;
	
	if (Pack->Key) {
	  buff = GetMem(buffsize, FALSE);
	  for (count = 0; count < buffsize; count++) {
  		tkey = (fpos + count) ^ Pack->Key;
      buff[count] = tkey ^ ~((char *) b)[count];
		}
	}
	else {
		buff = b;
	}
	fseek(Pack->HPIFile, fpos, SEEK_SET);
	result = fwrite(buff, 1, buffsize, Pack->HPIFile);
	if (buff != b)
	  FreeMem(buff);
	return result;
}

static int WriteChar(PACKFILE *Pack, int fpos, char *OutBlock, int *OutSize, void *data, int dsize, HPICHUNK *BlockHeader)
{
	int x;
	char *d = data;
	int o = *OutSize;
	unsigned char n;

	BlockHeader->CompressedSize += dsize;
	for (x = 0; x < dsize; x++) {
		n = (unsigned char) *d++;
		n = (n ^ o) + o;
		BlockHeader->Checksum += n;
		OutBlock[o++] = (char) n;
		if (o == OUTBLOCKSIZE) {
			EncryptAndWrite(Pack, fpos, OutBlock, o);
			fpos += o;
			o = 0;
		}
	}
	*OutSize = o;

	return fpos;
}

static int CompFunc(PACKFILE *Pack, void *d1, void *d2)
{
  unsigned int p1;
  unsigned int p2;
  char *k1;
  char *k2;
  int result;
  int i;
  int l;
  int m;

  if (d1 == d2)
    return 0;

//  if (!d1)
//    return -1;
//  if (!d2)
//    return 1;

  p1 = (unsigned int) d1;
  p2 = (unsigned int) d2;
  k1 = Pack->Window+p1;
  k2 = Pack->Window+p2;
  result = 0;
  l = 0;

  m = 17;
  for (i = 0; i < m; i++) {
    result = k1[i] - k2[i];
    if (result)
      break;
  }
  if ((Pack->WinLen <= i) && ((p2 & 0xFFF) != 0xFFF)) {
    Pack->WinLen = i;
    Pack->WinOfs = p2;
  }

  if (!result)
    result = p1-p2;

  return result;
}

static int SearchWindow(PACKFILE *Pack)
{
  Pack->WinLen = 1;
  Pack->WinOfs = 0;

  TreeSearch(Pack, &(Pack->SearchTree), CompFunc, (void *) Pack->ChunkPtr);

  if (Pack->WinLen > Pack->BlockSize)
    Pack->WinLen = Pack->BlockSize;

  return (Pack->WinLen >= 2);
}

static int LZ77CompressChunk(PACKFILE *Pack, LPSTR Chunk, HPICHUNK *BlockHeader, int fpos)
{
	char OutBlock[OUTBLOCKSIZE];
	int OutSize = 0;
	char flags;
	char data[17];
	int dptr;
	int mask;
	int olpair;
	int checksum = 0;
  int i;
  int Len;
	
  Pack->BlockSize = BlockHeader->DecompressedSize;

	mask = 0x01;
	flags = 0;
	dptr = 1;
	Pack->WIn = 0;
	Pack->WOut = 0;
  TreeInit(Pack);
	Pack->ChunkPtr = 0;
	Pack->Window = Chunk;

	do {
    //DumpMessage("\nBlockSize: 0x%X   WIn: 0x%X   WOut: 0x%X   ChunkPtr: 0x%X\n", BlockSize, WIn, WOut, ChunkPtr);
		if (SearchWindow(Pack)) {
      //DumpMessage("Matched offset: 0x%X   Length: 0x%X\n", WinOfs, WinLen);

      Pack->WinOfs &= 0xFFF;
			flags |= mask;
		  olpair = ((Pack->WinOfs+1) << 4) | (Pack->WinLen-2);
			data[dptr++] = LOBYTE(LOWORD(olpair));
			data[dptr++] = HIBYTE(LOWORD(olpair));
		}
		else {
			data[dptr++] = Chunk[Pack->ChunkPtr];
			Pack->WinLen = 1;
		}
    Len = Pack->WinLen;
  	Pack->BlockSize -= Len;

    for (i = 0; i < Len; i++) {
      TreeAdd(Pack, &(Pack->SearchTree), CompFunc, (void *) (Pack->ChunkPtr));
      Pack->ChunkPtr++;
      Pack->WOut++;
      if (Pack->WOut > 4095) {
        TreeDelete(Pack, &(Pack->SearchTree), CompFunc, (void *) Pack->WIn);
			  Pack->WIn++;
      }
    }

		if (mask == 0x80) {
			data[0] = flags;
			fpos = WriteChar(Pack, fpos, OutBlock, &OutSize, data, dptr, BlockHeader);
			mask = 0x01;
			flags = 0;
			dptr = 1;
		}
		else
			mask <<= 1;
	}
	while (Pack->BlockSize > 0);
  
	flags |= mask;
	data[dptr++] = 0;
	data[dptr++] = 0;
	data[0] = flags;

	fpos = WriteChar(Pack, fpos, OutBlock, &OutSize, data, dptr, BlockHeader);

	if (OutSize) {
		EncryptAndWrite(Pack, fpos, OutBlock, OutSize);
		fpos += OutSize;
	}

  TreeDestroy(Pack);
	return fpos;
}

static int ZLibCompressChunk(PACKFILE *Pack, char *Chunk, HPICHUNK *BlockHeader, int fpos)
{
	z_stream zs;
	int result;
	char *out;
	unsigned char n;
	int x;

	out = GetMem(131072, FALSE);

  zs.next_in = Chunk;
  zs.avail_in = BlockHeader->DecompressedSize;
  zs.total_in = 0;

  zs.next_out = out;
  zs.avail_out = 131072;
  zs.total_out = 0;

  zs.msg = NULL;
  zs.state = NULL;
  zs.zalloc = Z_NULL;
  zs.zfree = Z_NULL;
  zs.opaque = NULL;

  zs.data_type = Z_BINARY;
  zs.adler = 0;
  zs.reserved = 0;

	result = deflateInit(&zs, Z_DEFAULT_COMPRESSION);
  if (result != Z_OK) {
    return fpos;
  }

	result = deflate(&zs, Z_FINISH);
  if (result != Z_STREAM_END) {
    return fpos;
  }

	if (zs.total_out) {
		//d = (unsigned char *) out;
	  for (x = 0; (unsigned int) x < zs.total_out; x++) {
		  n = (unsigned char) out[x];
			if (BlockHeader->Encrypt)	{
  		  n = (n ^ x) + x;
				out[x] = n;
			}
		  BlockHeader->Checksum += n;
		}

		EncryptAndWrite(Pack, fpos, out, zs.total_out);

		fpos += zs.total_out;
	  BlockHeader->CompressedSize += zs.total_out;
	}

	deflateEnd(&zs);

  FreeMem(out);
	
  return fpos;
}

static int CompressFile(PACKFILE *Pack, int fpos, DIRENTRY *d, int CMethod)
{
	FILE *InFile;
	int BlockCount;
	int bpos;
	int *BlockPtr;
	int PtrBlockSize;
	HPICHUNK BlockHeader;
	int remain;
	int BlockNo;
	int hpos;
  char *InBlock;

  Pack->FileCount++;
  Pack->FileBytes = 0;
  Pack->FileBytesTotal = d->count;
  if (Pack->CallBack)
    Pack->Stop = Pack->CallBack(d->FileName, d->Name,
                   Pack->FileCount, Pack->FileCountTotal,
                   Pack->FileBytes, Pack->FileBytesTotal,
                   Pack->TotalBytes, Pack->TotalBytesTotal);
  if (Pack->Stop)
    return fpos;
	if (!d->memflag) {
    InFile = fopen(d->FileName, "rb");
	  if (!InFile)
  		return fpos;
	}

	*d->DirOffset = fpos;

	if (CMethod) {
    BlockCount = d->count / 65536;
    if ((d->count % 65536))
      BlockCount++;

	  bpos = fpos;

	  PtrBlockSize = BlockCount * sizeof(int);
	  BlockPtr = GetMem(PtrBlockSize, TRUE);

    EncryptAndWrite(Pack, bpos, BlockPtr, PtrBlockSize);

	  fpos += PtrBlockSize;

    InBlock = GetMem(INBLOCKSIZE, FALSE);

	  remain = d->count;
  	BlockNo = 0;

    while (remain) {
	  	BlockHeader.Marker = HEX_SQSH;
		  BlockHeader.Unknown1 = 0x02;
		  BlockHeader.CompMethod = CMethod;
		  BlockHeader.Encrypt = 0x01;
		  BlockHeader.CompressedSize = 0;
		  BlockHeader.DecompressedSize = (remain > 65536 ? 65536 : remain);
		  BlockHeader.Checksum = 0;

		  EncryptAndWrite(Pack, fpos, &BlockHeader, sizeof(BlockHeader));
		  hpos = fpos;

		  fpos += sizeof(BlockHeader);

		  ZeroMemory(InBlock, INBLOCKSIZE);
		  if (d->memflag)
			  MoveMemory(InBlock, d->FileName+Pack->FileBytes, BlockHeader.DecompressedSize);
		  else
		    fread(InBlock, 1, BlockHeader.DecompressedSize, InFile);

	    switch (CMethod) {
		    case LZ77_COMPRESSION :
  			  LZ77CompressChunk(Pack, InBlock, &BlockHeader, fpos);
          break;
		    case ZLIB_COMPRESSION :
			    ZLibCompressChunk(Pack, InBlock, &BlockHeader, fpos);
          break;
        default :
          break;
			}

		  BlockPtr[BlockNo] = BlockHeader.CompressedSize+sizeof(BlockHeader);

		  EncryptAndWrite(Pack, hpos, &BlockHeader, sizeof(BlockHeader));

		  fpos += BlockHeader.CompressedSize;

		  remain -= BlockHeader.DecompressedSize;

      Pack->FileBytes += BlockHeader.DecompressedSize;
      Pack->TotalBytes += BlockHeader.DecompressedSize;
      if (Pack->CallBack) {
        Pack->Stop = Pack->CallBack(d->FileName, d->Name,
                       Pack->FileCount, Pack->FileCountTotal,
                       Pack->FileBytes, Pack->FileBytesTotal,
                       Pack->TotalBytes, Pack->TotalBytesTotal);
        if (Pack->Stop)
          remain = 0;
			}

		  BlockNo++;
		}

    EncryptAndWrite(Pack, bpos, BlockPtr, PtrBlockSize);
	  FreeMem(BlockPtr);
	  FreeMem(InBlock);
	}
	else {
		if (d->memflag) {
      EncryptAndWrite(Pack, fpos, d->FileName+Pack->FileBytes, d->count);

		}
		else {
			InBlock = GetMem(d->count, FALSE);
	    fread(InBlock, 1, d->count, InFile);
      EncryptAndWrite(Pack, fpos, InBlock, d->count);
			FreeMem(InBlock);
		}
		fpos += d->count;
    if (Pack->CallBack) {
      Pack->FileBytes += d->count;
      Pack->TotalBytes += d->count;
      Pack->Stop = Pack->CallBack(d->FileName, d->Name,
                     Pack->FileCount, Pack->FileCountTotal,
                     Pack->FileBytes, Pack->FileBytesTotal,
                     Pack->TotalBytes, Pack->TotalBytesTotal);
		}

	}

	if (!d->memflag)
	  fclose(InFile);
	return fpos;
}

static int CompressAndEncrypt(PACKFILE *Pack, int fpos, DIRENTRY *dir, int CMethod)
{
	DIRENTRY *d;

	d = dir->FirstSub;

	while (d) {
		if (d->dirflag)
			fpos = CompressAndEncrypt(Pack, fpos, d, CMethod);
		else
			fpos = CompressFile(Pack, fpos, d, CMethod);
		d = d->Next;
	}

	return fpos;
}

static void FreeDir(DIRENTRY *dir)
{
  DIRENTRY *n;

  while (dir) {
    FreeDir(dir->FirstSub);
    if (dir->Name)
      FreeMem(dir->Name);
    if (dir->FileName && !dir->memflag)
      FreeMem(dir->FileName);
    n = dir->Next;
    FreeMem(dir);
    dir = n;
  }
}

static void FreePack(PACKFILE *Pack)
{
  if (Pack->FileName)
    FreeMem(Pack->FileName);
  if (Pack->HPIFile)
    FreeMem(Pack->HPIFile);
  FreeDir(Pack->Root);
  if (Pack->TreeHeap)
    HeapDestroy(Pack->TreeHeap);
  FreeMem(Pack);
}

int ScanFileNames(DIRENTRY *Start)
{
	DIRENTRY *This;
	int count;
	int dsize = 0;

	count = 0;
  This = Start->FirstSub;
	while (This) {
    count++;
		dsize += strlen(This->Name)+1;
    if (This->dirflag) {
			dsize += ScanFileNames(This);
		}
		else {
			dsize += sizeof(HPIENTRY);
		}
    This = This->Next;
	}
  Start->count = count;

  dsize += (2 * sizeof(int));
 	dsize += count * sizeof(HPIENTRY);

	return dsize;
}

LRESULT WINAPI HPIPackArchive(PACKFILE *Pack, int CMethod)
{
  HPIHEADER Header;
  int endpos;
  LPSTR Directory;

	Pack->HPIFile = fopen(Pack->FileName, "w+b");
  if (!Pack->HPIFile) {
    FreePack(Pack);
    return FALSE;
  }

  Pack->DirSize = ScanFileNames(Pack->Root);

  Pack->DirSize += sizeof(HPIHEADER);
  Directory = GetMem(Pack->DirSize, TRUE);

	Header.HPIMarker = HEX_HAPI;
	Header.SaveMarker = 0x00010000;
	Header.DirectorySize = Pack->DirSize;
  if (CMethod == LZ77_COMPRESSION)
	  Header.Key = 0x7D;
  else
    Header.Key = 0;
	Header.Start = sizeof(HPIHEADER);
	MoveMemory(Directory, &Header, sizeof(Header));

	if (Header.Key)
	  Pack->Key = ~((Header.Key * 4)	| (Header.Key >> 6));
	else
		Pack->Key = 0;

  BuildDirectoryBlock(Directory, Pack->Root, sizeof(HPIHEADER), CMethod);

	fwrite(Directory, Pack->DirSize, 1, Pack->HPIFile);

	endpos = CompressAndEncrypt(Pack, Pack->DirSize, Pack->Root, CMethod);

	fseek(Pack->HPIFile, endpos, SEEK_SET);

	fwrite(szTrailer, 1, strlen(szTrailer), Pack->HPIFile);

	EncryptAndWrite(Pack, Header.Start, Directory+Header.Start, Pack->DirSize-Header.Start);

	fclose(Pack->HPIFile);
  Pack->HPIFile = NULL;
  FreePack(Pack);
  return TRUE;
}

LRESULT WINAPI HPIPackFile(PACKFILE *Pack)
{
  return HPIPackArchive(Pack, LZ77_COMPRESSION);
}

int WINAPI DllMain(HINSTANCE hInstance, DWORD fdwReason, PVOID pvReserved)
{
  switch (fdwReason) {
    case DLL_PROCESS_ATTACH :
      hThisInstance = hInstance;
      ProgHeap = HeapCreate(0, 250000, 0);
      break;
    case DLL_THREAD_ATTACH :
      break;
    case DLL_THREAD_DETACH:
      break;
    case DLL_PROCESS_DETACH :
      if (ProgHeap) {
        HeapDestroy(ProgHeap);
        ProgHeap = NULL;
      }
      break;
  }

  return TRUE;
}
