
package org.ml.bp.totala.hpi;

// ScrambledRandomAccessFile - access the scrambled bytes of a .HPI file
//
// Copyright (C) 1997 by Barry Pederson <bpederson@geocities.com>.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
//
 
import java.io.*; 

/**
 * This class is basically a subset java.io.RandomAccessFile
 * with the addition of being able to descramble the contents of
 * the file based on a simple key (stored in the header of a HAPI file)
 * and extra methods for reading integers in Intel byte order, and
 * null-terminated ASCII strings.
 * 
 */

final class ScrambledRandomAccessFile 
	{
	private RandomAccessFile fRAF;
	private long fPosition;
	private int fScrambleKey;
	private boolean fIsScrambled;
	
/**
 * This method was created by a SmartGuide.
 * @param f java.io.File
 */
ScrambledRandomAccessFile (File f) throws IOException
	{
	fRAF = new RandomAccessFile(f, "r");
	}	
/**
 * This method was created by a SmartGuide.
 */
public void close() throws IOException
	{
	fRAF.close();
	}
/**
 * Turn off descrambling, in case you need to get at the raw file again
 */
public boolean disableKey() 
	{
	boolean oldState = fIsScrambled;
	fIsScrambled = false;
	return oldState;
	}
/**
 * Turn descrambling back on...in case you disableKey()'ed earlier
 */
public boolean enableKey() 
	{
	boolean oldState = fIsScrambled;
	fIsScrambled = true;
	return oldState;
	}
/**
 * This method was created by a SmartGuide.
 * @return long
 */
public long getFilePointer() throws IOException
	{
	return fRAF.getFilePointer();
	}
/**
 * This method was created by a SmartGuide.
 * @return long
 */
public long length() throws IOException
	{
	return fRAF.length();
	}
/**
 * This method was created by a SmartGuide.
 * @return int
 */
public int read() throws IOException
	{
	int result = fRAF.read();
	if (result == -1)
		return -1;

	if (fIsScrambled)
		result = ((fScrambleKey ^ (int)fPosition) ^(~result)) & 0x00ff;

	fPosition++;

	return result;				
	}
/**
 * This method was created by a SmartGuide.
 * @return int
 * @param b byte[]
 */
public int read(byte[] b) throws IOException
	{
	return read(b, 0, b.length);
	}
/**
 * This method was created by a SmartGuide.
 * @return int
 * @param b byte[]
 * @param off int
 * @param len int
 */
public int read(byte[] b, int off, int len) throws IOException
	{
	long initialPos = fPosition;
	int nRead = fRAF.read(b, off, len);

	if (fIsScrambled)
		{
		for (int i = 0; i < nRead; i++)
			b[i] = (byte) (((fScrambleKey ^ (i+initialPos)) ^(~b[i])) & 0x00ff);		
		}
		
	fPosition += nRead;		
	return nRead;
	}
/**
 * Read a null-terminated ASCII string
 * @return java.lang.String
 */
public String readASCIIZ() throws IOException
	{
	StringBuffer sb = new StringBuffer();
	while (true)
		{
		int i = read();
		if (i < 1)  // NULL and EOF are < 1
			break;
		sb.append((char) i);
		}
		
	return sb.toString();		
	}
/**
 * Read a 4-byte integer stored in Intel byte order
 * @return int
 */
public int readInt() throws IOException
	{
	int a = read();
	int b = read();
	int c = read();
	int d = read();

	if ((a | b | c | d) < 0)
	     throw new EOFException();

	return (d << 24) | (c << 16) | (b << 8) | a;
	}
/**
 * This method was created by a SmartGuide.
 * @param pos int
 */
public void seek(long pos) throws IOException
	{
	fRAF.seek(pos);
	fPosition = pos;
	}
/**
 * This method was created by a SmartGuide.
 * @param key int
 */
public void setKey(int key) 
	{
	fScrambleKey = ~((key << 2) | (key >> 6));
	fIsScrambled = true;
	}
}