
package org.ml.bp.totala.hpi;

// SQSHInputStream - decompress a stream of data as found in .HPI files
//
// Copyright (C) 1997 by Barry Pederson <bpederson@geocities.com>.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
//


import java.io.*;

/**
 * Decompress a stream of data as found in a .HPI file
 * 
 * Thanks to Eric DeZert (Ericd45@aol.com) - Orleans / France
 * for his WriteHPI program, which helped in understanding the 
 * algorithm for uncompressing data stored within .HPI files.
 */
 
 
final class SQSHInputStream extends java.io.FilterInputStream 
	{
	private final static int BUFFER_SIZE = 1024;
	
	private byte[] fInputBuffer;
	private int fInputLow;
	private int fInputHigh;
	
	private byte[] fOutputBuffer;
	private int fOutputLow;
	private int fOutputHigh;
	
	private byte[] fDictionary;
	private int fDictionaryPtr;
	
	private int fBitTest;
	private int fField;	
	
	private int fAvailable;
	private int fInputCount;
	
	private boolean fScrambled;
	private boolean fNoMore;
	
/**
 * SQSHInputStream constructor comment.
 * @param in java.io.InputStream
 */
protected SQSHInputStream(java.io.InputStream in) throws IOException
	{
	super(in);
	
	int magic = readInt();
	if (magic != 0x48535153)
		throw new IOException("Invalid signature in SQSH header: 0x" + Integer.toHexString(magic));

	int unknown = in.read();	// usually 2
	unknown = in.read();		// usually 1
	fScrambled = (in.read() == 1);		// usually 1 in HPI, 0 in Saved games

	int compressedSize = readInt(); // not actually used in this Java code
	fAvailable = readInt();
	unknown = readInt();  		// CRC? (as Eric says)
	
	fDictionary = new byte[4096]; // max 12-bit codes
	fDictionaryPtr = 1;

	fInputBuffer = new byte[BUFFER_SIZE];
	fInputHigh = fInputLow = 0;
	
	fOutputBuffer = new byte[BUFFER_SIZE];
	fOutputHigh = fOutputLow = 0;

	fBitTest = 1;
	fField = nextByte(); // get an initial flag byte	
	}
/**
 * This method was created by a SmartGuide.
 * @return int
 */
public int available() 
	{
	return fAvailable;
	}
/**
 * This method was created by a SmartGuide.
 */
public void close() throws IOException
	{
	in.close();
	fDictionary = null;
	fOutputBuffer = null;
	fInputBuffer = null;
	}
/**
 * When this method is called, the fOutputBuffer will always be empty
 * and fAvailable reflects how many bytes we -should- be able to decompress
 */
private int fillBuffer(byte[] dest, int offset, int destLength) throws IOException
	{
	// I'm reversing the highs and lows somewhat here..be careful not to get confused
	int destLow = offset;
	int destHigh = destLow + Math.min(destLength, fAvailable);

	fOutputLow = fOutputHigh = 0; // in case we need to overflow;
	
	while (destLow < destHigh)
		{	
		if ((fField & fBitTest) == 0)
			{
			byte b = nextByte();
			dest[destLow++] = b;
	
			// append to dictionary
			fDictionary[fDictionaryPtr++] = b;
			fDictionaryPtr &= 0x0fff; // dictionary wraps around at 4k			
			}
		else
			{
			int s1 = nextByte() & 0x00ff;
			int s2 = nextByte() & 0x00ff;
			int dictionaryOffset = (s2 << 4) | (s1 >> 4);
			int repetitions = (s1 & 0x0f) + 2;
			
			// not sure about this bit
			if (repetitions <= 0)
				return destLow - offset;

			// if the number of repetitions is more
			// than the destination buffer will handle, put
			// the repetitions in the output buffer
			if ((repetitions > (destHigh - destLow)) && (dest != fOutputBuffer))
				{
				destHigh = destLow; // so we exit after this
				for (int j = 0; j < repetitions; j++)	
					{
					// fetch from dictionary
					byte b = fDictionary[dictionaryOffset++];
					dictionaryOffset &= 0x0fff;
					
					// copy to output buffer
					fOutputBuffer[fOutputHigh++] = b;
		
					// append back into dictionary
					fDictionary[fDictionaryPtr++] = b;
					fDictionaryPtr &= 0x0fff; // dictionary wraps around at 4k
					}
				}
			else	// do a straightforward copy
				{								
				for (int j = 0; j < repetitions; j++)	
					{
					// fetch from dictionary
					byte b = fDictionary[dictionaryOffset++];
					dictionaryOffset &= 0x0fff;
					
					// copy to destination buffer
					dest[destLow++] = b;
		
					// append back into dictionary
					fDictionary[fDictionaryPtr++] = b;
					fDictionaryPtr &= 0x0fff; // dictionary wraps around at 4k
					}				
				}					
			}	

		// rotate the bitTest mask
		fBitTest <<= 1;			
		if (fBitTest == 0x0100)
			{
			// reset the mask
			fBitTest = 1;
			fField = nextByte();  // get a flag byte
			}		
						
		}	
		
	return destLow - offset;						
	}
/**
 * This method was created by a SmartGuide.
 */
public synchronized void mark(int readlimit)
	{
	}
/**
 * This method was created by a SmartGuide.
 * @return boolean
 */
public boolean markSupported() 
	{
	return false;
	}
/**
 * This method was created by a SmartGuide.
 * @return byte
 */
private byte nextByte() throws IOException
	{
	if (fInputLow < fInputHigh)
		return fInputBuffer[fInputLow++];
		
	// refill the input buffer;		
	fInputLow = 0;		
	fInputHigh = in.read(fInputBuffer, 0, BUFFER_SIZE);		
	
	if (fScrambled)
		{
		for (int i = 0; i < fInputHigh; i++)
			{
			fInputBuffer[i] = (byte) (((fInputBuffer[i] - fInputCount) ^ fInputCount) & 0x00ff);
			fInputCount++;
			}
		}
	
	// try again				
	if (fInputLow < fInputHigh)
		return fInputBuffer[fInputLow++];
	else
		fNoMore = true;
	return 0;		
	}
/**
 * @return Next byte in the input stream, -1 on EOF
 */
public int read() throws IOException 
	{
	// take from the output buffer if we can
	if (fOutputLow < fOutputHigh)
		{
		fAvailable--;
		return fOutputBuffer[fOutputLow++] & 0x00ff;
		}

	// guess not..crank out some more bytes
	fOutputLow = 0;
	fOutputHigh = fillBuffer(fOutputBuffer, 0, BUFFER_SIZE-32);

	// try again
	if (fOutputLow < fOutputHigh)
		{
		fAvailable--;
		return fOutputBuffer[fOutputLow++] & 0x00ff;
		}

	// must be EOF		
	return -1;		
	}
/**
 * Typical wrapper for reading arrays
 * @return int
 * @param b byte[]
 */
public int read(byte[] b) throws IOException
	{
	return read(b, 0, b.length);
	}
/**
 * This method was created by a SmartGuide.
 * @return Number of bytes read.  -1 on EOF
 * @param b byte[]
 * @param off int
 * @param len int
 */
public int read(byte[] buffer, int offset, int len) throws IOException
	{
	int bufferLow = offset;
	int bufferHigh = offset + len;
	
	// take from output buffer if possible
	if (fOutputLow < fOutputHigh)
		{
		int nToCopy = Math.min(fOutputHigh - fOutputLow, bufferHigh - bufferLow);
		System.arraycopy(fOutputBuffer, fOutputLow, buffer, bufferLow, nToCopy);
		fOutputLow += nToCopy;
		bufferLow += nToCopy;
		fAvailable -= nToCopy;

		// is that all we need, or can get?		
		if ((bufferLow >= bufferHigh) || (fAvailable < 1))
			return bufferLow - offset;
		}		
		
	// crank out some new data		
	int nRead = fillBuffer(buffer, bufferLow, bufferHigh - bufferLow);		
	bufferLow += nRead;
	fAvailable -= nRead;
	
	// Is there a little more we need and can get from the output buffer
	if ((bufferLow < bufferHigh) && (fOutputLow < fOutputHigh))
		{
		int nToCopy = Math.min(fOutputHigh - fOutputLow, bufferHigh - bufferLow);
		System.arraycopy(fOutputBuffer, fOutputLow, buffer, bufferLow, nToCopy);
		fOutputLow += nToCopy;
		bufferLow += nToCopy;
		fAvailable -= nToCopy;
		}		

	int result = bufferLow - offset;
	return (result < 1 ? -1 : result);
	}
/**
 * Read a 4-byte integer stored in Intel byte order
 * @return int
 */
private int readInt() throws IOException
	{
	int a = in.read();
	int b = in.read();
	int c = in.read();
	int d = in.read();

	if ((a | b | c | d) < 0)
	     throw new EOFException();

	return (d << 24) | (c << 16) | (b << 8) | a;
	}
/**
 * This method was created by a SmartGuide.
 */
public synchronized void reset() throws IOException
	{
	throw new IOException("mark/reset not supported");
	}
/**
 * This method was created by a SmartGuide.
 * @return long
 * @param n long
 */
public long skip(long n) throws IOException
	{
	long i;
	for (i = 0; (i < n) && (read() >= 0); i++)
		;
		
	return i;
	}
}