
package org.ml.bp.totala.hpi;

// HPIFile - accesses .HPI and .UFO archive file
//
// Copyright (C) 1997 by Barry Pederson <bpederson@geocities.com>.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
//

import java.io.*;
import java.util.Vector;

/**
 * Instances of this object represent a .HPI archive file. The
 * design is somewhat similar to java.util.zip.ZipFile<p>
 * <i>.UFO unit files also have the same format</i><p>
 *
 * Thanks to Eric DeZert (Ericd45@aol.com) - Orleans / France
 * for his WriteHPI program, which helped in understanding the 
 * .HPI format and the algorithm for uncompressing data stored 
 * within those files.
 *
 * @author Barry Pederson &lt;<a href="mailto:bpederson@geocities.com">bpederson@geocities.com</a>&gt;
 */
  
public class HPIFile 
	{
	private static ScrambledRandomAccessFile fSRAF;
	private HPIEntry fContents;
	private HPIEntry fFlatList[];
	private Vector fTempVector;
	
/**
 * Opens a HPI file for reading given the specified File object.
 * @param f the HPI file to be opened for reading
 * @exception IOException if an I/O error has occurred
 */
public HPIFile (File f) throws IOException
	{
	fSRAF = new ScrambledRandomAccessFile(f);

	if (fSRAF.length() < 20)
		{
		fSRAF.close();
		throw new IOException("Too short to be a HAPI file");
		}
				
	int magic = fSRAF.readInt();
	if (magic != 0x49504148) // the first 4 bytes were 'H' 'A' 'P' 'I'
		{
		fSRAF.close();
		throw new IOException("Invalid signature for a HAPI file");
		}
		
	int subtype = fSRAF.readInt();
	if (subtype == 0x4b4e4142) // the second 4 bytes were 'B' 'A' 'N' 'K'
		{
		fSRAF.close();
		throw new IOException("Sorry, saved games not supported");
		}
		
	int contentOffset = fSRAF.readInt();	 // not actually used in this Java code
	int key = fSRAF.readInt();
	int rootDirectoryOffset = fSRAF.readInt();
	
	// switch to scrambled mode
	fSRAF.setKey(key);	
	
	// create our lists of contents
	fTempVector = new Vector();
	fContents = readDirectoryInfo(null, null, rootDirectoryOffset);
	fFlatList = new HPIEntry[fTempVector.size()];
	fTempVector.copyInto(fFlatList);
	fTempVector = null;
	}
/**
 * Opens a HPI file for reading given the specified name.
 * @param name the name of the HPI file to be opened for reading
 * @exception IOException if an I/O error has occurred
 */
public HPIFile (String name) throws IOException 
	{
	this(new File(name));
	}
/**
 * Close the HPI file
 */
public void close() throws IOException
	{
	fSRAF.close();
	}
/**
 * Get a list of only the top-level entries. 
 * (These may include directory entries which have their own sub-entries)<p>
 *
 * @return HAPIFileEntry[] Array of top-level files and subdirectories
 */
public HPIEntry[] getEntries() 
	{
	return fContents.fSubDir;
	}
/**
 * Get a flattened list of just the files stored in this archive. 
 * (Directory names are still available in each HPIEntry)<p>
 *
 * <i>java.util.zip.ZipEntry has a similar method "entries()" that returns an enumeration,
 * which seemed inconvenient - Since you just end up casting the elements of
 * the enumeration to a known class, why not just return the class you 
 * know you'll be working with?</i>
 *
 * @return HAPIFileEntry[] An array of just the file entries
 */
public HPIEntry[] getEntriesFlattened() 
	{
	return fFlatList;
	}
/**
 * Access the contents of one of the entries stored in this archive for reading. <p>
 *
 * <i>It should be safe to open and read multiple InputStreams on a given HPI file simultaneously.</i>
 *
 * @return InputStream for reading
 * @param he HPIEntry representing a file stored within the HPI file
 */
public InputStream getInputStream(HPIEntry he) throws IOException 
	{
	if (he.fFile != this)
		throw new IOException("The HPIEntry doesn't correspond to this HPIFile");

	if (!he.isFile())
		throw new IOException("This HPIEntry doesn't represent a file (it may be a directory)");		
		
	Vector chunks = new Vector();

	// find out how many chunks are out there
	synchronized (fSRAF)
		{
		fSRAF.seek(he.fDataOffset);
		while (true)
			{
			int i = fSRAF.readInt();
			if (i == 0x48535153) // "SQSH"
				break;
			else 
				chunks.addElement(new Integer(i));				
			}
		}
		
	int chunkOffset = he.fDataOffset + (chunks.size() * 4);		
	Vector streams = new Vector(chunks.size());
	for (int i = 0; i < chunks.size(); i++)
		{
		int chunkSize = ((Integer) chunks.elementAt(i)).intValue();
		SubsetInputStream sis = new SubsetInputStream(fSRAF, chunkOffset, chunkSize);
		streams.addElement(new SQSHInputStream(sis));
		chunkOffset += chunkSize;
		}
									
	return new SequenceInputStream(streams.elements());
	}
/**
 * This method was created by a SmartGuide.
 * @param offset int
 *
 * Each directory info entry consists of a 
 *   4-byte count items
 *   4-bytes of unknown info
 *   for each item
 *       {
 *		4-byte offset of ASCIIZ item name
 *		4-byte offset of item info
 *		1-byte item type flag (0 = file, 1 = directory)
 *		}
 */
private HPIEntry readDirectoryInfo(String parentName, String directoryName, int offset) throws IOException
	{
	Vector v = new Vector();
	String newParent;
	if (parentName == null)
		newParent = directoryName;
	else
		newParent = parentName + File.separator + directoryName;
	
	fSRAF.seek(offset);
	int nEntries = fSRAF.readInt();
	int unknown = fSRAF.readInt();
				
	for (int i = 0; i < nEntries; i++)
		{
		int itemNameOffset = fSRAF.readInt();
		int itemInfoOffset = fSRAF.readInt();
		int entryType = fSRAF.read(); // this field is just a single byte

		long currentPos = fSRAF.getFilePointer();
		fSRAF.seek(itemNameOffset);				
		String itemName = fSRAF.readASCIIZ();
		fSRAF.seek(itemInfoOffset);
		switch (entryType)
			{
			case 0:
				v.addElement(readFileInfo(newParent, itemName, itemInfoOffset));
				break;
			case 1:
				v.addElement(readDirectoryInfo(newParent, itemName, itemInfoOffset));
				break;
			default:
				System.out.println(newParent + File.separator + itemName + " (Unknown entry type: " + entryType + ")");
				break;											
			}		
		fSRAF.seek(currentPos);			
		}
		
	HPIEntry[] ha = new HPIEntry[v.size()];
	v.copyInto(ha);
	v = null;
	
	HPIEntry result = new HPIEntry(this, parentName, directoryName, 0, 0);
	result.fIsDirectory = true;
	result.fSubDir = ha;
	return result;
	}
/**
 * This method was created by a SmartGuide.
 * @param name java.lang.String
 * @param offset int
 *
 * Each file info entry consists of a 
 *   4-byte file offset of where the compressed data is stored
 *   4-byte length of uncompressed file
 *   1-byte of unknown info
 */
private HPIEntry readFileInfo(String parentName, String name, int offset) throws IOException
	{
	int dataOffset = fSRAF.readInt();
	// this dataOffset points to a spot in the file that contains
	// 1 or more 4-byte integers representing the sizes of
	// SQSH chunks, followed by those SQSH chunks back-to-back
	// So far, the only way I can see to tell how many chunks
	// are involved is to scan ahead looking for the first "SQSH" 
	// header
	//
	// I'm guessing each SQSH chunk is < 64k in size..so when
	// scanning, the highest two bytes of a chunksize will always be 0x00000,
	// so if you see a "SQ" then it's probably safe to figure you found the first chunk
	// Eric seems to do it this way too
	//
	// I would also speculate that each SQSH chunk other than the last
	// expands to something very close to 64k
	// (so when they wrote the file, individual entries were broken
	// into certain-sized chunks, and each chunk was SQSH'ed)
	
	
	int dataSize = fSRAF.readInt();
//	int unknown = fSRAF.read(); // just a single byte field, always seems to be 0x01, doesn't seem to indicate the number of SQSH chunks
	HPIEntry result = new HPIEntry(this, parentName, name, dataOffset, dataSize);
	fTempVector.addElement(result);
	return result;
	}
}	