
// UnHPI - access the contents of a .HPI (or .UFO) file
//
// Copyright (C) 1997 by Barry Pederson <bpederson@geocities.com>.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
//

 
import java.io.*;
import org.ml.bp.misc.Wildcard;
import org.ml.bp.totala.hpi.*;

/**
 * Command-line utility for viewing the contents of, and extracting files
 * from a .HPI file
 * 
 */
 
public class UnHPI 
	{
	private final static int BUFFER_SIZE = 16384; // size of our copy buffer
	private static byte[] fBuffer; // for copying files.
	private static boolean fAbort;
	
/**
 * This method was created by a SmartGuide.
 * @param hpi org.ml.bp.totala.hpi.HPIFile
 * @param e org.ml.bp.totala.hpi.HPIEntry
 */
private static void extractFile(HPIFile hpi, HPIEntry e, boolean junkDirectories) 
	{
	String filename;

	if (junkDirectories)
		filename = e.getName();
	else 
		{		
		File dir = new File(e.getParent());
		if (!dir.exists())
			dir.mkdirs();
		filename = e.getPath();
		}

	File f = new File(filename);
		
	if (f.exists())
		{
		System.err.print(filename + " already exists, replace? (y/N/q) ");
		DataInputStream dis = new DataInputStream(System.in);
		try
			{
			String answer = dis.readLine();
			if ((answer == null) || (answer.length() < 1))
				{
				System.out.println("  Skipped: " + filename);					
				return;
				}
			if (answer.equalsIgnoreCase("q"))
				{
				System.out.println("  Quitting");
				fAbort = true;
				return;
				}
			if (!answer.equalsIgnoreCase("y"))				
				{
				System.out.println("  Skipped: " + filename);					
				return;
				}
			}
		catch (IOException ex) // this probably won't happen
			{
			ex.printStackTrace();
			}							
		}

	System.out.print("Extracting: " + filename + " ");
	try
		{
		FileOutputStream fos = new FileOutputStream(f);
		InputStream is = hpi.getInputStream(e);
		int nRead;
		while ((nRead = is.read(fBuffer)) > 0)
			{
			fos.write(fBuffer, 0, nRead);
			System.out.print('.');
			}
		is.close();
		fos.close();		
		System.out.println(" (" + e.length() + " bytes) done");					
		}
	catch (IOException ex)
		{
		System.err.println("Error writing " + filename + " : " + ex.getMessage());		
		return;
		}		
	}	
/**
 * main entrypoint - starts the application
 * @param args java.lang.String[]
 */
public static void main(java.lang.String[] args) throws IOException
	{
	System.out.println("UnHPI v1.1  1997-12-12  Barry Pederson <bpederson@geocities.com>");
	System.out.println("");

	if (args.length < 1)
		{
		System.out.println("Usage: UnHPI [option] HPI-file [file file file ...]");
		System.out.println("    -v     View contents");
		System.out.println("    -j     Junk pathnames and extract files");
		System.out.println("    -d     Extract with directories (default)");
		System.out.println("");
		System.out.println("Wildcards * and ? are permitted when you specify files to extract");
		System.out.println("(filenames are case-insensitive)");
		return;		
		}

	boolean junkDirectories = false;
	boolean viewContents = false;
	int hpiPointer = 0; // which argument contains the name of the HPI file
	
	if (args[0].charAt(0) == '-')
		{
		if (args[0].length() != 2)
			{
			System.out.println("Illegal option: " + args[0]);
			return;
			}
			
		hpiPointer = 1;
		switch (args[0].charAt(1))
			{
			case 'v': viewContents = true; break;
			case 'j': junkDirectories = true; break;
			case 'd': break;
			default: System.out.println("Unknown option: " + args[0]); return;
			}
		}			

	File f = new File(args[hpiPointer]);
				
	if (!f.canRead())
		{
		System.err.println("Can't read " + f.getPath());
		return;
		}
	
	try
		{
		HPIFile hpi = new HPIFile(f);
		HPIEntry[] entries = hpi.getEntriesFlattened();
		
		if (viewContents)
			{
			System.out.println("   Length   Name");
			System.out.println("   ------   ----");
			}
		else
			fBuffer = new byte[BUFFER_SIZE];
					
		
		int totalSize = 0;
		int totalCount = 0;
		fAbort = false;
			
		for (int i = 0; (!fAbort) && (i < entries.length); i++)
			{
			boolean match = args.length == (hpiPointer + 1); // match everything if no files are specified
			for (int j = hpiPointer+1; (!match) && (j < args.length); j++)
				match = Wildcard.match((junkDirectories ? entries[i].getName() : entries[i].getPath()), args[j]);
				
			if (!match)
				continue;
										
			if (viewContents)
				System.out.println(rightAlign(entries[i].length(), 9) + "   " + entries[i].getPath());
			else				
				extractFile(hpi, entries[i], junkDirectories);

			totalCount++;
			totalSize += entries[i].length();				
			}

		if (viewContents)
			{
			System.out.println("   ------   ----");
			System.out.println(rightAlign(totalSize, 9) + "   " + totalCount);
			}			

		hpi.close();			
		}
	catch (IOException e)
		{
		System.err.println("Error reading HPI file: " + e.getMessage());
		return;
		}				
	}
/**
 * This method was created by a SmartGuide.
 * @return java.lang.String
 * @param s java.lang.String
 * @param size int
 */
private static String rightAlign(int i, int size) 
	{
	StringBuffer sb = new StringBuffer();

	sb.append(i);

	while (sb.length() < size)
		sb.insert(0, ' ');
		
	return sb.toString();
	}
}