/*
**
**  QSOCKET implementation and support routines; encapsulates socket IO
**
**  (c) mike warren, 1997
**  mikeBot
**
*/

#include "defines.h"
#include "qsocket.h"
#include "qpacket.h"
#include <stdio.h>

#if !WIN32
#include <sys/types.h>
#include <sys/socket.h>
#endif

/*
**  convert string to an actual address, "0.0.0.0" and "www.crap.com" forms
**  accepted. straight out of sockets faq ( www.auroraonline.com/sock-faq )
**
** !NOTE: ONLY the address returned from THE LAST CALL IS VALID
*/

struct in_addr * qsocket::atoaddr( char * address )
{
	struct hostent * host;
	static struct in_addr saddr;

			// no address at all...

	if( !address || !*address )
	  return NULL;

			// check for "1.2.3.4" etc.

	saddr.s_addr = inet_addr( address );
	if( saddr.s_addr != (unsigned long)-1 )
	{
#if (DEBUG & DQS)
	  printf("qsocket::atoaddr(): using `%s' directly\n", address); 
#endif
	  return &saddr;
	}

			// no, so try "www.foo.com"

	printf("qsocket::atoaddr(): DNS lookup on `%s'\n", address);

	host = gethostbyname( address );
	if( host != NULL )
	{
#if (DEBUG & DQS)
	  printf("qsocket::atoaddr(): lookup on `%s' succeeded\n", address);
#endif
	  char * b = host->h_addr_list[0];
	  printf("qsocket::atoaddr(): DNS answer `%u.%u.%u.%u'\n",b[0],b[1],b[2],b[3] );
	  return (struct in_addr *) *host->h_addr_list;
	}

	fprintf(stderr, "qsocket::atoaddr(): DNS lookup failed\n");

	return NULL;

}

/*
**  ctor : checks addresses (if passed) and sets ports, etc. on local and
**         server, and connects and binds local port.
**
*/


qsocket::qsocket( char * addr, int port )
{

	struct in_addr * temp;

	memset( (char *)&remote, 0, sizeof(remote) );
	memset( (char *)&local, 0, sizeof(local) );

	winStartup();

	if ( (temp = atoaddr( addr )) == NULL )
	  {
#if QTHROW
	    throw( "qsocket::qsocket(): invalid address" );
#else
	    fprintf(stderr, "qsocket::qsocket() : invalid address `%s'\n", addr);
	    exit( -1 );
#endif
	  }

	remote.sin_port = htons( port );
	remote.sin_family = AF_INET;
	remote.sin_addr.s_addr = temp->s_addr;

	local.sin_family = AF_INET;
	local.sin_addr.s_addr = INADDR_ANY;
	local.sin_port = 0;//htons( port );

#if WIN32
	if( (fd = socket( PF_INET, SOCK_DGRAM, 0 )) == INVALID_SOCKET )
#else
	if ( (fd = socket( AF_INET, SOCK_DGRAM, 0 )) < 0 )
#endif
	{
#if QTHROW
	  throw( "qsocket::qsocket(): could not obtain socket\n" );
#else
	  perror( "qsocket::qsocket(): socket()" );
	  exit( -1 );
#endif
	}
#if (DEBUG & DQS)
	else
	  printf("qsocket::qsocket(): fd = %d\n", fd );
#endif

	if ((bind( fd, (struct sockaddr *)&local, sizeof(local)) ) < 0)
	{
#if QTHROW
	  throw( "qsocket::qsocket(): bind failed\n" );
#else
	  perror( "qsocket::qsocket(): bind()" );
	  exit( -1 );
#endif
	}
#if (DEBUG & DQS)
	else
	  printf("qsocket::qsocket(): bind succeded\n");
#endif

#if !WIN32
#if AIX
	if( (fcntl( fd, F_SETFL, O_NONBLOCK )) < 0 )
#else
	if( (fcntl( fd, F_SETFL, O_NDELAY )) < 0 )
#endif
#else		// windows
	if( (ioctlsocket( fd, FIONBIO, NULL )) != 0 )
#endif
	   {
#if QTHROW
	     throw("qsocket::qsocket(): fcntl");
#else
	     perror("qsocket::qsocket(): fcntl");
	     exit( -1 );
#endif
	   }

}

/*
**  binds to local address ONLY
**
*/

qsocket::qsocket( struct sockaddr_in * ia, struct sockaddr_in * ra, int port )
{

	memset( (char *)&remote, 0, sizeof(remote) );
	memset( (char *)&local, 0, sizeof(local) );

	remote.sin_port = htons( port );
	remote.sin_family = AF_INET;
	if( !ra )
	  remote.sin_addr.s_addr = htonl( INADDR_ANY );
	else
	  remote.sin_addr.s_addr = ra->sin_addr.s_addr;

	local.sin_family = AF_INET;
	if( ia )
	  local.sin_addr.s_addr = ia->sin_addr.s_addr;
	else
	  local.sin_addr.s_addr = htonl( INADDR_ANY );
	local.sin_port = 0;//htons( port );

	if ( (fd = socket( AF_INET, SOCK_DGRAM, 0 )) < 0 )
	{
#if QTHROW
	  throw( "qsocket::qsocket(): could not obtain socket\n" );
#else
	  perror( "qsocket::qsocket(): socket()" );
	  exit( -1 );
#endif
	}
#if (DEBUG & DQS)
	else
	  printf("qsocket::qsocket(): fd = %d\n", fd );
#endif

	if ((bind( fd, (struct sockaddr *)&local, sizeof(local)) ) < 0)
	{
#if QTHROW
	  throw( "qsocket::qsocket(): bind failed\n" );
#else
	  perror( "qsocket::qsocket(): bind()" );
	  exit( -1 );
#endif
	}
#if (DEBUG & DQS)
	else
	  printf("qsocket::qsocket(): bind succeded\n");
#endif

#if !WIN32
#if AIX
	if( (fcntl( fd, F_SETFL, O_NONBLOCK )) < 0 )
#else
	if( (fcntl( fd, F_SETFL, O_NDELAY )) < 0 )
#endif
#else   /// windows
	if( (ioctlsocket( fd, FIONBIO, NULL )) != 0 )
#endif
	   {
#if QTHROW
	     throw("qsocket::qsocket(): fcntl");
#else
	     perror("qsocket::qsocket(): fcntl");
	     exit( -1 );
#endif
	   }

}

/*
**  ctor 3
**
**  Usefull for "listen" sockets...
**
*/

qsocket::qsocket( int port, char * addr )
{

	struct in_addr * temp;

	memset( (char *)&remote, 0, sizeof(remote) );
	memset( (char *)&local, 0, sizeof(local) );

	winStartup();

	if ( (temp = atoaddr( addr )) == NULL )
	  {
#if QTHROW
	    throw( "qsocket::qsocket(): invalid address" );
#else
	    fprintf(stderr, "qsocket::qsocket() : invalid address `%s'\n", addr);
	    exit( -1 );
#endif
	  }

	remote.sin_port = htons( port );
	remote.sin_family = AF_INET;
	remote.sin_addr.s_addr = INADDR_ANY;

	local.sin_family = AF_INET;
	local.sin_addr.s_addr = temp->s_addr;
	local.sin_port = htons( port );

#if WIN32
	if( (fd = socket( PF_INET, SOCK_DGRAM, 0 )) == INVALID_SOCKET )
#else
	if ( (fd = socket( AF_INET, SOCK_DGRAM, 0 )) < 0 )
#endif
	{
#if QTHROW
	  throw( "qsocket::qsocket(): could not obtain socket\n" );
#else
	  perror( "qsocket::qsocket(): socket()" );
	  exit( -1 );
#endif
	}
#if (DEBUG & DQS)
	else
	  printf("qsocket::qsocket(): fd = %d\n", fd );
#endif

	if ((bind( fd, (struct sockaddr *)&local, sizeof(local)) ) < 0)
	{
#if QTHROW
	  throw( "qsocket::qsocket(): bind failed\n" );
#else
	  perror( "qsocket::qsocket(): bind()" );
	  exit( -1 );
#endif
	}
#if (DEBUG & DQS)
	else
	  printf("qsocket::qsocket(): bind succeded\n");
#endif

#if !WIN32
#if AIX
	if( (fcntl( fd, F_SETFL, O_NONBLOCK )) < 0 )
#else
	if( (fcntl( fd, F_SETFL, O_NDELAY )) < 0 )
#endif
#else		// windows
	if( (ioctlsocket( fd, FIONBIO, NULL )) != 0 )
#endif
	   {
#if QTHROW
	     throw("qsocket::qsocket(): fcntl");
#else
	     perror("qsocket::qsocket(): fcntl");
	     exit( -1 );
#endif
	   }

}





/*
**  dtor : disconnects (just a close, but hey)
*/

qsocket::~qsocket()
{
#if WIN32
	closesocket( fd );
#else
	close( fd );
#endif
	fd = 0;
#if (DEBUG & DQS)
	printf("qsocket::~qsocket(): socket closed\n");
#endif
}

/*
**
**  send() : sends the (entire) buffer passed in to the current remote
**	     bytes sent is returned, or <= 0 on error
**
*/

int qsocket::send( unsigned char * b, int s )
{
  return sendto(fd, (char *)b, s, 0, (struct sockaddr *)&remote, sizeof(remote));
}

	  
/*
**
**  receive() : returns bytes read into buffer (or -1 on error)
**
*/

int qsocket::receive( unsigned char * buff, int max )
{
#if WIN32
  int s = sizeof(remote);
  int r = recvfrom( fd, (char *)buff, max, 0, (struct sockaddr *)&remote, &s);
  if( r == SOCKET_ERROR )
	  return -1;
  return r;
#else
#if !AIX
  int s = sizeof(remote);
  return recvfrom( fd, buff, max, 0, (struct sockaddr *)&remote, &s);
#else
  size_t s = sizeof(remote);
  return recvfrom( fd, buff, max, 0, (struct sockaddr *)&remote, &s);
#endif
#endif
}


int qsocket::receive( qpacket & qp )
{
	int r = receive( buffer, QP_MAX_PACKET_DATA );
	if( r > QP_MAX_PACKET_DATA )
		qp.updateData( buffer, QP_MAX_PACKET_DATA );
	else if( r > 0 )
		qp.updateData( buffer, r );

	return r;
}




/*
**  changePort : changes local and remote ports and rebind()s local
**
*/

void qsocket::changePort( int port )
{
  remote.sin_port = htons( port );
  local.sin_port = htons( port );
}


/*
**  winStartup : sets up for winsock calls
**
*/

void qsocket::winStartup()
{
#if WIN32
	WORD wVersionRequested; 
	WSADATA wsaData; 
	int err; 
	wVersionRequested = MAKEWORD(1, 1); 
	 
	err = WSAStartup(wVersionRequested, &wsaData); 
	 
	if (err != 0) 
	{
		fprintf( stderr, "qsocket::winStartup(): invalid WINSOCK.DLL\n");
		winError();
		exit( -1 );
	}
	 
	/* Confirm that the Windows Sockets DLL supports 1.1.*/ 
	/* Note that if the DLL supports versions greater */ 
	/* than 1.1 in addition to 1.1, it will still return */ 
	/* 1.1 in wVersion since that is the version we */ 
	/* requested. */ 
	 
	if ( LOBYTE( wsaData.wVersion ) != 1 || 
			HIBYTE( wsaData.wVersion ) != 1 ) 
	{ 
		fprintf( stderr, "qsocket::winStartup(): WINSOCK.DLL not correct version\n");
		WSACleanup(); 
		winError();
		exit( -1 );
	} 

	printf("qsocket::winStartup(): %s\n", wsaData.szDescription );
#endif
}

/*
**  winShutdown : WSACleanup();
**
*/

void qsocket::winShutdown()
{
#if WIN32
	WSACleanup();
#endif
}

/*
**  winError : prints error message
**
*/

void qsocket::winError()
{
#if WIN32
	printf("qsocket::winError():");

	printf("\n");
#endif
}