

#include "defines.h"
#include "qproxy.h"

#if QPROXY

/*
**  ctor 
**
*/

qproxy::qproxy( int max, int port )
{
  try 
    {
      sock = new qsocket( (sockaddr_in *)0, (sockaddr_in *)0, port );
    }
  catch ( char * err )
    {
      fprintf(stderr,"qproxy::qproxy(): %s\n",err );
	  sock = (qsocket *)0;
    }

#if (DEBUG & DQX)
  printf("qproxy::qproxy(): socket opened fd=%d\n", sock->getFD() );
#endif

  maxClients=max;
  connectedClients=0;
  cc=0;
  masterClient=-1;

  numPrespawns = 0;
  clientIP[0] = 0;
}

/*
**  update : tries to read port, replies appropriatly if data
**
*/

void qproxy::update()
{
  if( !sock )
 	return;

  int r;
  int i;

  r = receive( inPacket.getBuffer(), inPacket.getMaxSize() );

  if( r < 0 )
    {
#if WIN32
	  if( WSAGetLastError() == WSAEWOULDBLOCK )
#else
      if( errno == EWOULDBLOCK )
#endif
	errno = 0;
      else
	perror("qproxy::update(): receive()");
    }
  else if( r == 0 )
    {
      printf("qproxy::update(): disconnected socket\n");
    }
  else
    {
      inPacket.init();
      decodePacket( inPacket );
    }

  for( i=0; i < QX_MAX_CLIENTS; i++ )
    {
      cc=i;
      if( clients[cc].sock )
	{
	  r = clientReceive( inPacket.getBuffer(), 1024 );

#if !WIN32
	  if( r < 0 )
	    {
	      if( errno == EWOULDBLOCK )
		errno = 0;
	      else
		perror("qproxy::update(): receive()");
	    }
	  else if( r == 0 )
	    {
	      printf("qproxy::update(): disconnected socket %d\n", cc);
	      clients[cc].reset();
	      if( cc==masterClient )
		{
		  masterClient=-1;
		  printf("qproxy::update(): no more master\n");
		}
	    }
	  else
	    {
	      inPacket.init();
	      decodePacket( inPacket );
	    }
#endif
	  clients[cc].ackQueue.resend( clients[cc].sock );
	}
    }

}


/*
**  decodePacket : decodse all packet types (calls appropriate decode fn)
**
*/

int qproxy::decodePacket( qpacket & qp )
{
  switch( qp.getType() )
    {
    case qpacket::control:
      return decodeControl( qp );
      break;

    case qpacket::reliableFragment:
    case qpacket::reliableEnd:
      return decodeReliable( qp );
      break;

    case qpacket::unreliable:
      return decodeUnreliable( qp );
      break;

    case qpacket::acknowledge:
      return decodeAcknowledge( qp );
      break;

    case qpacket::unknown:
    default:
      fprintf(stderr, "qproxy::decodePacket(): unknown type\n");
      break;
    }
  
  return FALSE;
}

/* 
**  decodeAcknowledge
**
*/

int qproxy::decodeAcknowledge( qpacket & qp )
{
  int i = qp.readBEint();

#if DEBUG & DQX
  printf("qproxy::decodeAcknowledge(): got %d from client %d\n", i,cc );
#endif

  return clients[cc].ackQueue.gotAck( i, clients[cc].sock );
}
  

/*
**  decodeReliable
**
*/

int qproxy::decodeReliable( qpacket & qp )
{
  int pnum = qp.readBEint();
  
#if DEBUG & DQX
  printf("qproxy::decodeReliable(): packet #%d from client %d\n", pnum,cc);
#endif

  outPacket.reset();
  outPacket.addBEint( pnum );
  outPacket.changeType( qpacket::acknowledge );
  outPacket.send( clients[cc].sock );

  int opcode = qp.readByte();
  char * ts;

  if( opcode == CS_CONSOLE )
    {
      ts = qp.readString();
      if( !strncmp( ts, "say", 3 ) )
	{
	  printf("qproxy::decodeReliable(): client %d: %s\n", cc, ts );
	}
      else if ( !strncmp( ts, "begin", 5 ) )
	{
	  clients[cc].sendable=1;
	  printf("qproxy::decodeReliable(): client %d fully connected\n",cc);
	}
      else if ( !strncmp( ts, "name", 4 ) )
	{
	  strncpy( clients[cc].name, strtok( &ts[6],"\"\n" ), Q_MAX_STRING );
	  parseName( (unsigned char *)clients[cc].name );
	  printf("qproxy::decodeReliable(): client %d name `%s'\n",cc, clients[cc].name );
	}
      else
	printf("qproxy::decodeReliable(): console %s\n", ts );

      delete ts;
    }
  else if ( opcode == CS_KEEPALIVE )
    {
      printf("qproxy::decodeReliable(): %d: client-server keepalive\n",cc);
      outPacket.reset();
      outPacket.addBEint( clients[cc].outgoingUnreliable++ );
      outPacket.addByte( SC_NOP );
      outPacket.changeType( qpacket::unreliable );
      outPacket.send( clients[cc].sock );
    }
  else
    printf("qproxy::decodeReliable(): unknown opcode `%02x' from client %d\n", opcode,cc);

  return TRUE;
}

/*
**  decodeControl : decodes client to server control packets
**
*/

int qproxy::decodeControl( qpacket & qp )
{
  char c = qp.readByte();
  char * tempString;
  int i;

  switch( c )
    {
    case CCREQ_CONNECT:
      tempString = qp.readString();
      printf("qproxy::decodeControl(): connect request '%s'\n", tempString);
      delete tempString;

      if( !freeClient() )
	{
	  outPacket.reset();
	  outPacket.addByte( CCREP_REJECT );
	  outPacket.addString( "\001\n\nmikeBot proxy server full\nfor mor information on mikeBot, visit:\n\nhttp://www.ucalgary.ca/~mbwarren/mikeBot/\n\n\n" );
          outPacket.changeType( qpacket::control );
	  outPacket.send( sock );
	  printf("qproxy::decodeControl(): request rejected\n");
	}
      else
	{
	  i = findOpenClient();
	  clients[ i ].haveConnection=TRUE;

	  client = sock->lastPacketIP();
	  clientName = gethostbyaddr( (char *)&client, sizeof(client), AF_INET );
	  if( !clientName )
	    perror("qproxy::gethostbyaddr()");
	  else	
	    strncpy( clientIP, clientName->h_name, Q_MAX_STRING );
	  printf("qproxy::decodeControl(): new connection %s:%d\n", clientIP, sock->lastPacketPort() );

	  connectedClients++;
	  if( connectedClients == 1 )
	    {
	      printf("qproxy::decodeControl(): client %d is now master\n",i);
	      masterClient=i;
	    }

	  try
	    {
	      clients[ i ].sock = new qsocket( sock->getLocalAddress(), sock->getRemoteAddress(), sock->lastPacketPort() );
	    }
	  catch( char * err )
	    {
	      fprintf(stderr,"qproxy::decodeControl(): %s\n",err );
	      connectedClients--;
	      clients[i].reset();

	      outPacket.reset();
	      outPacket.addByte( CCREP_REJECT );
	      outPacket.addString( "\001Connection error\n\nplease try again...\n");
	      outPacket.changeType( qpacket::control );
	      outPacket.send( sock );
	      return FALSE;
	    }

	  clients[i].port = sock->lastPacketPort();

	  outPacket.reset();
	  outPacket.addByte( CCREP_ACCEPT );
	  outPacket.addLEshort( (short)sock->lastPacketPort() );
	  outPacket.addLEshort( (short)0 );
          outPacket.changeType( qpacket::control );
	  outPacket.send( sock );


	  printf("qproxy::decodeControl(): request accepted on port %d using client struct %d\n", sock->lastPacketPort(), i);

	  qpacket tp;

	  spawn1.setReadPos( 0 );
	  splitAndSend( i, spawn1 );

	  spawn2.setReadPos( 0 );
	  splitAndSend( i, spawn2 );

	  spawn3.setReadPos( 0 );
	  splitAndSend( i, spawn3 );

	  outPacket.reset();
	  outPacket.addBEint( clients[i].outgoingReliable++ );
	  outPacket.addByte( SC_PRINT );
	  if( i == masterClient )
	    outPacket.addString( "Welcome to mikeBot proxyVision\n\nYOU ARE THE MASTER CLIENT\nand therefore have these impulses:\n\nimpulse 60 \216 status bar on/off\nimpulse 61 \216 target particles on/off\nimpulse 62 \216 locked view on/off\n\n" );
	  else
	    outPacket.addString( "Welcome to mikeBot proxyVision\n\nA master is already present.\n\n");
	    
	  outPacket.changeType( qpacket::reliableEnd );
	  clients[ i ].ackQueue.addPacket( outPacket, clients[ i ].sock );
	}
      break;

    case CCREQ_SERVER_INFO:
      outPacket.reset();
      outPacket.addString("www.ucalgary.ca/~mbwarren/mikeBot/");
      outPacket.addString("mikeBot proxy server");
      outPacket.addString("www.ucalgary.ca/~mbwarren/mikeBot/");
      outPacket.addByte( (char)connectedClients );
      outPacket.addByte( (char)QX_MAX_CLIENTS );
      outPacket.addByte( (char)3 );
      outPacket.changeType( qpacket::control );
      outPacket.send( sock );
      break;

    default:
      fprintf(stderr, "qproxy::decodeControl(): unknown opcode '%c'\n", c);
      break;
    }

  return TRUE;
}


/*
**  splitAndSend : used for packets > 1024 bytes
**
*/

int qproxy::splitAndSend( int cl, qpacket & qp )
{
  qpacket tp;
  int added;

#if DEBUG & DQX
  printf("qproxy::splitAndSend(): client %d, read=%d\n", cl,qp.getReadPos() );
#endif

  while( qp.getReadPos() < qp.getSize() )
    {
      tp.reset();
      tp.addBEint( clients[cl].outgoingReliable++ );
      for( added=0; added < 1024; added++ )
	{
	  if( qp.getSize() == qp.getReadPos() )
	    break;
	  tp.addByte( qp.readByte() );
	}
      if( added==0 )
	break;
#if DEBUG & DQX
      printf("qproxy::splitAndSend(): added %d to fragments\n",added);
#endif
      if( added < 1024 )
	{
	  tp.changeType( qpacket::reliableEnd );
	  clients[cl].ackQueue.addPacket( tp, clients[cl].sock );
	  break;
	}
      else
	{
	  tp.changeType( qpacket::reliableFragment );
	  clients[cl].ackQueue.addPacket( tp, clients[cl].sock );
	}
    }

  return TRUE;


}

/*
**  sendToAll : sends to all clients
**
*/

int qproxy::sendToAll( qpacket & qp )
{
  for( int i=0; i < QX_MAX_CLIENTS; i++ )
    if( clients[i].sendable )
      sendToClient( i, qp );

  return TRUE;
}

/*
**  sendToClient : sends to specific client
**
*/

int qproxy::sendToClient( int cl, qpacket & qp )
{
  if( !clients[cl].sendable )
    return -1;

  if( qp.getSize() <= 1024 ) 
    return qp.send( clients[cl].sock ); 

  return splitAndSend( cl, qp );
}

/*
** send
**
*/

int qproxy::send( qpacket & qp )
{ 
  return sendToAll( qp ); 
} 

/*
**  sendReliable : sends packet reliably to all clients
**
*/

int qproxy::sendReliable( qpacket & qp )
{
  
  for( int i=0; i < QX_MAX_CLIENTS; i++ )
    if( clients[i].sendable )
      {
	if( qp.getSize() <= 1024 )
	  {
//	    qp.changeType( qpacket::reliableEnd );
//	    qp.changeNumber( clients[i].outgoingReliable++ );
	    qp.changeType( qpacket::unreliable );
	    qp.changeNumber( clients[i].outgoingUnreliable++ );
	    qp.send( clients[i].sock );
//	    clients[i].ackQueue.addPacket( qp, clients[i].sock );
	  }
	else
	  {
	    qp.setReadPos( 4 );
	    splitAndSend( i, qp );
	  }
      }

  return TRUE;
}



/*
**  decodeUnreliable : gets clients facing, etc
**
*/

int qproxy::decodeUnreliable( qpacket & qp )
{

  char opcode;
  float roll,pitch,yaw;
  int impulse;
  char actions;
  char temp[ 80 ];

  clients[cc].incomingUnreliable = qp.readBEint();

  opcode = qp.readByte();

  if( opcode == CS_ACTION )
    {
      qp.readLEfloat();		// timestamp
      pitch = qp.readAngle();
      yaw = qp.readAngle();
      roll = qp.readAngle();
      qp.readLEshort();
      qp.readLEshort();		// strafe
      qp.readLEshort();		// verticle
      actions = qp.readByte();	      
      impulse = qp.readByte();

      if( impulse )
	{
	  printf("qproxy::decodeUnreliable(): client %d: impulse %d\n", cc, impulse );
	  switch( impulse )
	    {
	    case 60:
	      clients[cc].cl_statusbar=!clients[cc].cl_statusbar;
	      sprintf( temp, "\360\362\357\370\371: statusbar %s\n", clients[cc].cl_statusbar?"ON":"OFF");
	      outPacket.reset();
	      outPacket.addBEint( clients[cc].outgoingReliable++ );
	      outPacket.addByte( SC_PRINT );
	      outPacket.addString( temp );
	      outPacket.changeType( qpacket::reliableEnd );
	      clients[cc].ackQueue.addPacket( outPacket, clients[cc].sock );
	      break;
	      
	    case 61:
	      clients[cc].cl_particle=!clients[cc].cl_particle;
	      sprintf( temp, "\360\362\357\370\371: target particles %s\n", clients[cc].cl_particle?"ON":"OFF");
	      outPacket.reset();
	      outPacket.addBEint( clients[cc].outgoingReliable++ );
	      outPacket.addByte( SC_PRINT );
	      outPacket.addString( temp );
	      outPacket.changeType( qpacket::reliableEnd );
	      clients[cc].ackQueue.addPacket( outPacket, clients[cc].sock );
	      break;
	      
	    case 62:
	      clients[cc].cl_lockview=!clients[cc].cl_lockview;
	      sprintf( temp, "\360\362\357\370\371: lockview %s\n", clients[cc].cl_lockview?"ON":"OFF");
	      outPacket.reset();
	      outPacket.addBEint( clients[cc].outgoingReliable++ );
	      outPacket.addByte( SC_PRINT );
	      outPacket.addString( temp );
	      outPacket.changeType( qpacket::reliableEnd );
	      clients[cc].ackQueue.addPacket( outPacket, clients[cc].sock );
	      break;

	    default:
	      printf("qproxy::decodeUnreliable(): unknown impulse %d\n", impulse );
	      break;
	    }
	}
      return TRUE;
    }
  else if (opcode == CS_DISCONNECT )
    {
      printf("qproxy::decodeUnreliable(): disconnected client %d\n",cc);
      clients[cc].reset();
      connectedClients--;
      if( cc == masterClient )
	{
	  masterClient = -1;
	  printf("qproxy::decodeUnreliable: master disconnected\n");
	}
    }
  else
    printf("qproxy::decodeUnreliable(): unknown opcode %02x\n", opcode);

  return FALSE;
}


/*
**  sayClient : sends the string (reliably) to all clients
**
*/

void qproxy::sayClient( char * x )
{
  char temp[ Q_MAX_STRING ];

  sprintf(temp, "\001proxyVision: %s\n", x );

  for( int i=0; i < QX_MAX_CLIENTS; i++ )
    if( clients[i].sock )
      {
	outPacket.reset();
	outPacket.addBEint( clients[i].outgoingReliable++ );
	outPacket.addByte( SC_PRINT );
	outPacket.addString( temp );
	outPacket.changeType( qpacket::reliableEnd );
	clients[i].ackQueue.addPacket( outPacket, clients[i].sock );
      }
}

/*
**  dumpClients : prints info on all clients
**
*/

void qproxy::dumpClients()
{
  printf("qproxy clients:\n\n");
  printf(" #  internet address               port name\n");

  for( int i=0; i < QX_MAX_CLIENTS; i++ )
    {
      if( clients[i].sock )
	{
	  client = clients[i].sock->lastPacketIP();
	  clientName = gethostbyaddr( (char *)&client, sizeof(client), AF_INET );
	  if( !clientName )
	    {
	      strcpy( clientIP, "error" );
	      errno=0;
	    }
	  else	
	    strncpy( clientIP, clientName->h_name, Q_MAX_STRING );
	}
      else
	strcpy( clientIP, "not connected" );

      printf("%02d  %-30s %04d %s\n",i,clientIP,clients[i].port,clients[i].name );
    }

}


/*
**  kickClient : disconnects a client
**
*/

void qproxy::kickClient( int c )
{
  if( !clients[c].haveConnection )
    return;

  outPacket.reset();
  outPacket.addBEint( clients[c].outgoingReliable++ );
  outPacket.addByte( SC_PRINT );
  outPacket.addString( "proxyVision kicked you...\n");
  outPacket.addByte( SC_STUFFTEXT );
  outPacket.addString( "disconnect\n" );
  outPacket.changeType( qpacket::reliableEnd );
  clients[c].ackQueue.addPacket( outPacket, clients[c].sock );
}

/*
**  destroyClient : not as nice as above; simply stops sending stuff...
**
*/

void qproxy::destroyClient( int x )
{
  connectedClients--;
  if( x == masterClient )
    {
      masterClient=-1;
      printf("qproxy::destroyClient(): master destroyed\n");
    }
  clients[x].reset();
}

#endif

      


