/*
**
**  QPACKET handles packet construction/deconstruction, with endianess 
**  considerations.
**
**  (c) mike warren, 1997
**  mikeBot
**
*/

#include "defines.h"
#include "qpacket.h"
#include "vector.h"
#include "qsocket.h"

/*
**  getNumber : returns the number of the packet. WARNING - return value is
**              only valid if the packet is unreliable or relibale (not 
**              control)
**
*/

int qpacket::getNumber()
{
  int temp = readPos;
  int r=0;

  readPos = 4;
  r = readBEint();
  readPos = temp;

  return r;
}

int qpacket::changeNumber( int x )
{
	int temp = writePos;
	
	writePos = 4;
	addBEint( x );
	writePos = temp;
	size -= 4;

	return TRUE;
}
	
  
  

/*
**  send : sends a packet through the socket specified
**
*/

int qpacket::send( qsocket * s )
{
  update();
#if DEBUG & DQP
  if( !s )
    return -1;
#endif
  return s->send( data, size+4 );
}

/*
**  copy constructor
**
*/

void qpacket::copy( qpacket & qp )
{
	size = qp.size;
	maxSize = qp.maxSize;

	delete[] data;
	data = new unsigned char[ maxSize ];
	memcpy( data, qp.data, maxSize );

	readPos = qp.readPos;
	writePos = qp.writePos;
	pType = qp.pType;
}


/*
** operator += : glues two packets together, discarding headers
**
*/

qpacket & qpacket::operator += ( qpacket & qp )
{
	if( size+qp.size > maxSize )
	{
#if DEBUG & DQP
	printf("****************  REALLOCING PACKET **********************\n");
#endif
	  unsigned char * t = new unsigned char[ size+qp.size+1024 ];
	  maxSize = size+qp.size+1024;
	  memcpy( t, data, size );
	  memcpy( &t[size], &qp.data[8], qp.size-4 );
	  size += qp.size - 4;
	  delete data;
	  data = t;
	}
	else
	{
	  memcpy( &data[ size ], &qp.data[8], qp.size-4 );
	  size += qp.size-4;
#if DEBUG & DQP
	printf("qpacket::+=(): size now %d\n", size );
#endif
	}

	return *this;
}


/*
** ctor
**
*/

qpacket::qpacket( int x )
{
  data = new unsigned char[ x ];
  if( !data )
  {
#if QTHROW
	  throw( "qpacket::qpacket():allocation error\n");
#else
	  printf("qpacket::qpacket(): allocation error\n"); 
	  exit(-1 );
#endif
  }

  writePos = 4;			// leave room for header
  readPos = 4;
  size = 0;
  maxSize = x;
  pType = unknown;
}

/*
**  init : updates the pType and size FROM THE DATA IN *DATA
**
*/

int qpacket::init()
{
  int temp;

  if( data[0] & 0x80 )
    pType = control;
  else if (data[0] == 0x00 )
    {
      if( data[1] == 0x09 )
	pType = reliableEnd;
      else if (data[1] == 0x10)
	pType = unreliable;
      else if (data[1] == 0x01)
	pType = reliableFragment;
      else if (data[1] == 0x02)
	pType = acknowledge;
      else
	{
	  pType = unknown;
	  fprintf(stderr, "qpacket::init(): unknown type %02x%02x\n", data[0],data[1]);
	}
    }
  else
    {
      pType = unknown;
      fprintf(stderr, "qpacket::init(): unknown type %02x%02x\n", data[0],data[1]);
    }

  readPos = 2;
  temp = (int)readBEshort();

#if DEBUG & DQP
  printf("qpacket::init(): size is %d\n", temp);
#endif

  writePos = temp;
  size = temp-4;

  if( writePos >= maxSize )
  {
	  writePos = maxSize-1;
	  size = maxSize-4;
	  fprintf(stderr, "qpacket::init(): data size exceeds buffer (%d)\n", temp );
  }

  return TRUE;
  
}
  

/*
**  update : adds the packet length and type (header) to top of packet
**           --> pType had better be correct on entry <--
**
*/

int qpacket::update()
{
  switch( pType )
    {
    case control:
      data[0] = (unsigned char)0x80;
      data[1] = (unsigned char)0x00;
      break;
      
    case unreliable:
      data[0] = (unsigned char)0x00;
      data[1] = (unsigned char)0x10;
      break;

    case reliableFragment:
      data[0] = (unsigned char)0x00;
      data[1] = (unsigned char)0x01;
#if DEBUG & DQP
      printf("qpacket::update(): size is %d\n", size+4 );
#endif
      break;

    case reliableEnd:
      data[0] = (unsigned char)0x00;
      data[1] = (unsigned char)0x09;
#if DEBUG & DQP
      printf("qpacket::update(): size is %d\n", size+4 );
#endif
      break;

    case acknowledge:
      data[0] = (unsigned char)0x00;
      data[1] = (unsigned char)0x02;
      break;

    default:
      data[0] = (unsigned char)0x00;
      data[1] = (unsigned char)0x00;
      fprintf(stderr, "qpacket::update(): WARNING unknown packet type (size=%d)\n", size+4);
      break;
    }

  int temp = writePos;
  writePos = 2;
  addBEshort( (short)(size+4) );
  size -= 2;			// just added a short, but not really...
  writePos = temp;

  return TRUE;

}


/*
 **  write : writes the data w/o header, unaltered to the open file stream
 **	    passed in (for recording demos). TRUE/FALSE returned
 **
 **   this is rather un-elegant :) the if(x) things are because when i glue
 **   the reliableFragments togethere, they start at 0, not 8, as in 
 **   unreliable packets (i.e. headers are already stripped)
 **
 ** TODO : error checking correct? errno? inline it, maybe. How about packet 
 **        numbers? do i write those? (it doesn't, currently)
 */

int qpacket::write( FILE * fp, int x, vector & angles )
{
  static unsigned char h[ 16 ];
  static unsigned char s[ 16 ];
  int i=0;
  
  if( x )
    *((int *)h) = size-4;
  else
    *((int *)h) = size;

  *((float *) &h[4]) = (float)angles.getPitch();
  *((float *) &h[8]) = (float)angles.getYaw();
  *((float *) &h[12]) = (float)0.0;

#ifdef QBIG_ENDIAN
  for( i=0; i < 16; i += 4 )
    {
      s[i+0] = h[i+3];
      s[i+1] = h[i+2];
      s[i+2] = h[i+1];
      s[i+3] = h[i+0];  
    }
#else
  for( i=0; i < 16; i++ )
    s[i] = h[i];
#endif  
  
  fwrite( s, 1, 16, fp );
  if( x )
    fwrite( &data[ x ], 1, size-4, fp );
  else
    fwrite( &data[ x ], 1, size, fp );

  return TRUE;
}


/*
 **
 **  readLE* functions : return the * data type ready-to-use, after converting
 **			from LE in the network stream to the #define'd host 
 **			system. other functions similar.
 **
 */

int qpacket::readLEint()
{
#ifdef QBIG_ENDIAN
  t1[3] = readByte();
  t1[2] = readByte();
  t1[1] = readByte();
  t1[0] = readByte();
#else
  t1[0] = readByte();
  t1[1] = readByte();
  t1[2] = readByte();
  t1[3] = readByte();
#endif
  return *((int *) t1 );
}


int qpacket::readBEint()
{
#ifdef QBIG_ENDIAN
  t1[0] = readByte();
  t1[1] = readByte();
  t1[2] = readByte();
  t1[3] = readByte();
#else
  t1[3] = readByte();
  t1[2] = readByte();
  t1[1] = readByte();
  t1[0] = readByte();
#endif
  return *((int *) t1 );
}


short qpacket::readLEshort()
{
#ifdef QBIG_ENDIAN
  t1[1] = readByte();
  t1[0] = readByte();
#else
  t1[0] = readByte();
  t1[1] = readByte();
#endif
  return *((short *) t1 );
}


short qpacket::readBEshort()
{
#ifdef QBIG_ENDIAN
  t1[0] = readByte();
  t1[1] = readByte();
#else
  t1[1] = readByte();
  t1[0] = readByte();
#endif
  return *((short *) t1 );
}


float qpacket::readLEfloat()
{
#ifdef QBIG_ENDIAN
  t1[3] = readByte();
  t1[2] = readByte();
  t1[1] = readByte();
  t1[0] = readByte();
#else
  t1[0] = readByte();
  t1[1] = readByte();
  t1[2] = readByte();
  t1[3] = readByte();
#endif
  return *((float *) t1 );
}


float qpacket::readBEfloat()
{
#ifdef QBIG_ENDIAN
  t1[0] = readByte();
  t1[1] = readByte();
  t1[2] = readByte();
  t1[3] = readByte();
#else
  t1[3] = readByte();
  t1[2] = readByte();
  t1[1] = readByte();
  t1[0] = readByte();
#endif
  return *((float *) t1 );
}

char * qpacket::readString()
{
	char * s = new char[ Q_MAX_STRING ];
	
	for( int i=0; i < Q_MAX_STRING-1; i++ )
	  if( (s[ i ] = readByte()) == '\0' )
	    break;

	s[ Q_MAX_STRING-1 ] = '\0';		// safety first ;)

	return s;
}

float qpacket::readAngle()
{
	return (float)readByte() / (float)256.0 * (float)360.0;
}

float qpacket::readCoord()
{
	return (float)readLEshort() * (float)0.125;
}

//
//  addLE* functions take the parameter and add it to the current position,
//  assuming it is required to be in little endian order in the buffer. Pos.
//  is incremented
//
//  TODO : maybe change to addByte() instead of memcpy() (just 'cause ;) )
//

void qpacket::addLEint( int x )
{
  *((int *)t1) = x;
#ifdef QBIG_ENDIAN
  addByte( t1[3] );
  addByte( t1[2] );
  addByte( t1[1] );
  addByte( t1[0] );
#else
  addByte( t1[0] );
  addByte( t1[1] );
  addByte( t1[2] );
  addByte( t1[3] );
#endif
}
  

void qpacket::addBEint( int x )
{
  *((int *)t1) = x;
#ifdef QBIG_ENDIAN
  addByte( t1[0] );
  addByte( t1[1] );
  addByte( t1[2] );
  addByte( t1[3] );
#else
  addByte( t1[3] );
  addByte( t1[2] );
  addByte( t1[1] );
  addByte( t1[0] );
#endif
}


void qpacket::addLEshort( short x )
{
  *((short *)t1) = x;
#ifdef QBIG_ENDIAN
  addByte( t1[1] );
  addByte( t1[0] );
#else
  addByte( t1[0] );
  addByte( t1[1] );
#endif
}  


void qpacket::addBEshort( short x )
{
  *((short *)t1) = x;
#ifdef QBIG_ENDIAN
  addByte( t1[0] );
  addByte( t1[1] );
#else
  addByte( t1[1] );
  addByte( t1[0] );
#endif
}  


void qpacket::addLEfloat( float x )
{
  *((float *)t1) = x;
#ifdef QBIG_ENDIAN
  addByte( t1[3] );
  addByte( t1[2] );
  addByte( t1[1] );
  addByte( t1[0] );
#else
  addByte( t1[0] );
  addByte( t1[1] );
  addByte( t1[2] );
  addByte( t1[3] );
#endif
}


void qpacket::addBEfloat( float x )
{
  *((float *)t1) = x;
#ifdef QBIG_ENDIAN
  addByte( t1[0] );
  addByte( t1[1] );
  addByte( t1[2] );
  addByte( t1[3] );
#else
  addByte( t1[3] );
  addByte( t1[2] );
  addByte( t1[1] );
  addByte( t1[0] );
#endif
}


void qpacket::addString( char * s )
{

  if( !s )
    {
      addByte( 0 );
      return;
    }

  int i=0;
  do {
    addByte( s[i] );
  } while( s[i++] );

}

void qpacket::addCoord( float x )
{
  short a;

  a = (short)( x / 0.125 );
  addLEshort( a );

}

void qpacket::addAngle( float x )
{
  int a;

  a = (int)(x * 256.0 / 360.0);
  a = a % 256;

  addByte( (unsigned char)a );
}

void qpacket::addData( unsigned char * d, int s )
{
  unsigned char * t;

  if ( maxSize < s + size )
    {
#if DEBUG & DQP
	printf("qpacket::addData(): added %d bytes (realloced)\n", s);
#endif
      t = new unsigned char[ s+size+1024 ];
      memcpy( t, data, size );
      memcpy( &data[size], d, s );
      delete[] data;
      data = t;
      size = size+s;
      maxSize = size+s+1024;
    }
  else
    {
      memcpy( &data[size], d, s );
      size += s;
#if DEBUG & DQP
	printf("qpacket::addData(): added %d bytes, size now %d\n", s, size );
#endif
    }

}


/*
**  updateData : copies data to the internal qpacket buffer 
**
*/

void qpacket::updateData( unsigned char * buff, int max )
{
	int m = (max > maxSize) ? maxSize : max;

	for( int i=0; i < m; i++ )
		data[ i ] = buff[ i ];
}




		

	


