/*
**
**  Mbfire class 
**
**  (c) 1997, mike warren
**  mikeBot
**
**
**  SKELTON AI CLASS:
**
**  . This class is responsible for selecting a fire-control target and
**    eliminating it. It works completely independantly from MBNAV; the
**    low-level movement routines take care of the fire-navigation seperation
**    calculations.
**
**  . This skeleton class selects the closest other player and targets
**    it; it doesn't shoot if a wall is in the way. It selects the best
**    weapon it has which isn't out of ammo and uses it.
**
**  . Suggestions for improvement:
**      . Account for walls when selecting targets; don't target a player
**        if a wall is in the way, even if they're closer.
**      . Account for your best weapon when selecting targets; select
**        medium-distance targets if you have the RL, close targets for
**        the double-shotgun, lightning gun, etc.
**      . Try and guess the other player's weapon and attempt to eliminate
**        the player with the best weapon first.
**
*/

#include "defines.h"
#include "mbfire.h"
#include <math.h>
#if WIN32
#include <float.h>
#endif

/*
**  ctor
**
*/

mbfire::mbfire()
{
	mbf_target = 0;		// the entity number of the current target
	mbf_oldTarget = 0;	// enitity number of last frame's target
	
	mbf_shoot = 2;		// 2 == inactive, 1 = shoot, 0 = send 0
	mbf_impulse = 0;	// if impulse != 0, it is sent, then set=0

	mbf_facing.set( (float)0.0, (float)0.0, (float)0.0 );
}

/*
**  aimAt
**
**  Makes the bot aim at the specified entity number. FALSE is returned
**  if the facing was not changed, for the following reasons:
**
**     +  the entity is dead (or non-living; i.e. health, ammo).
**     +  the entity is not visible.
**     +  the entity number is invalid (==0, or > QCS_MAX_ENTITIES)
**
*/

int mbfire::aimAt( int targetEntity )
{
  if( targetEntity > QCS_MAX_ENTITIES || targetEntity < 1 )
    return FALSE;
  else if ( !isVisible( targetEntity ) )
    return FALSE;
  else if ( isDead( targetEntity ) )
    return FALSE;
  else if ( players[targetEntity-1].hate < 0 )
    return FALSE;

  mbf_oldTarget = mbf_target;
  mbf_target=targetEntity;
  predictPosition( mbf_target );

  mbf_facing = entities[myEntityNumber].origin - mbf_predictedPosition;

  return TRUE;
}

/*
**  update
**
**  Performs any targeting calculations, etc. This is called before the
**  MBNAV::update() method and thus allows navigation to check out what
**  firecontrol is targeting when doing its thing. This is called on every
**  frame before the movement packets are sent.
**
*/

void mbfire::update()
{
	int i;
	float bestRate=(float)0;
	float rate=(float)0;
	int best=0;

    for( i=1; i <= info.maxPlayers; i++ )
      if( isVisible( i ) && !isDead( i ) && i != myEntityNumber )
		{
		  rate = (float)1000000.0;
		  rate /= (float)(entities[i].origin - entities[myEntityNumber].origin).length();

		  predictPosition( i );

		  if( rate > bestRate )
		    {
		      bestRate = rate;
		      best = i;
		    }
		}

	if( best )
	  mbf_target = best;

	if( !aimAt( mbf_target ) )
	  {
	    mbf_target = 0;

					//
					//  the 'target-record' option only writes demo
					//  packets if a target is being tracked
					//

	    if( opts.get( "target-record" ) )
	      pauseDemo();
	  }
	else
	{
	  selectWeapon();
	  unpauseDemo();

	  chIsVisible = FALSE;

	  if( !opts.get( "no-fire" ) )
		{
			fire();

					//
					//  this shit is a hack for the crosshair-model;
					//  it should really just reference a real entity
					//  in the already-present entity array
					//

			chIsVisible = TRUE;
			chQEntity.origin = entities[ mbf_target ].origin;
			chQEntity.facing = entities[ mbf_target ].facing;


			chQEntity.frame++;
			if( chQEntity.frame > 35 )
				chQEntity.frame = 0;
		}
	}

	return;
}

/*
**  selectWeapon 
**
**  Picks the best weapon which isn't out of ammo. Won't use the LG
**  if the bot is in a water-leaf. Only uses the RL if the map-file 
**  has been loaded, to avoid shooting walls.
**
*/

void mbfire::selectWeapon()
{
	int newWeapon=0;

  if( haveLG() && cells && map.getCurrentLeafType() != 2 )
	newWeapon = 8;
  else if ( haveRL() && rockets && map.isLoaded() )
	newWeapon = 7;
  else if ( haveSNG() && nails )
    newWeapon = 5;
  else if ( haveSSG() && shells )
    newWeapon = 3;
  else if ( haveNG() && nails )
    newWeapon = 4;
  else if ( shells )
    newWeapon = 2;
  else						// axe it is
    newWeapon = 1;

  if( newWeapon != weapon )
	  mbf_impulse = newWeapon;

}
  
      



/*
**  predictPosition
**
**  Does some jim-dandy entity position prediction, based on the
**  algorithm used by StoogeBot (http://www-graphics.stanford.edu/~quake)
**
*/

void mbfire::predictPosition( int x )
{

  float weapVel;

  switch( weapon )
    {
    case 0:			// axd
    case 1:			// SG
    case 2:			// SSG
    case 64:			// LG
      weapVel=(float)0.0;
      break;

    case 4:			// NG
    case 8:			// SNG
    case 32:			// RL
      weapVel=(float)1000.0;
      break;

    default:
      weapVel=(float)500.0;
    }

  if( opts.get( "no-predict" ) )
    {
      mbf_predictedPosition = entities[ x ].origin;
      return;
    }

  if( weapVel == 0.0 )
    {
      mbf_predictedPosition = entities[ x ].origin;
      return;
    }

  float time=(float)0.0,guess,err;
  vector tv;
  float enemyVel = (float)entities[ x ].velocity.length();
  int times=0;

  do {
    tv = entities[ x ].velocity;
    tv.normalize();
    tv *= (enemyVel*time);
    tv += entities[ x ].origin;
    guess = (float)((entities[myEntityNumber].origin - tv).length() / weapVel) + ((float)opts.get( "latency" )/100);
    err = guess - time;
    if ( err < 0.0 )
      err = -err;
    time = guess;
    times++;
  }while( err > 0.01 && times < 6);

#if WIN32
  if( !_isnan( guess ) )
#else
  if( !isnan( guess ) )
#endif
    mbf_predictedPosition = tv;
  else
    mbf_predictedPosition = entities[ x ].origin;

}


/*
**  cmd
**
**  see QCS::cmd()
**
*/

int mbfire::cmd( char * x )
{
	switch( x[0] )
	{
	case 'i':
		if( connected() )
		{
			printf("Impulse %d\n", atoi( &x[2] ) );
			sendImpulse( atoi( &x[2] ) );
			return TRUE;
		}
		break;
	}
	return mbtalk::cmd( x );
}


