/*
**  graph class
**
**  (c) 1997 mike warren
**  mikeBot
**
**
**  builds a directed graph of the current level as the bot moves through
**  the level. as position is updates, this class figures out where the bot
**  is and if that new node has a floor polygon or is lava/slime/water, it
**  is added to the graph.  use the public: interface functions to get info
**  about the level. Provides line of sight, leaf type, etc
**
**  TODO:
**     . lava check when linking polygons
**     . path finding working right?
**
*/



#include "graph.h"
#include <memory.h>
#include <string.h>
#include "mpack.h"
#include "m_stack.h"

extern packList packFileList;

/*
**  newMap 
**
**  Loads new map, if possible, after saving the level graph for the
**  current map. Returns FALSE on failure, TRUE on success.
**
*/

int bspGraph::newMap( char * fname, int )
{
  if( map )
    {
      delete map;		// erase bspFile
    }

  printf("loading BSP from %s...", fname);
  fflush( stdout );

			//
			//  try to find the file in the list of PACK files
			//

  mFile * mf;

  mf = packFileList.getFile( fname );

  if( !mf )
  {
	  try 
	  {
		  mf = new mFile( fname );
	  }
	  catch( char * err )
	  {
		  printf("bspGraph::newMap(): open error `%s'\n", err );
		  delete map;
		  return FALSE;
	  }
  }
  else
  {
  	  try 
	  {
		  map = new BSPlevel( *mf );
	  }
	  catch( char * err )
	  {
		  fprintf(stderr, "\n%s\nbspGraph::newMap(): level not loaded\n",err);
		  delete map;
      map=0;
      return FALSE;
	  }
  }

  return TRUE;
}


/*
**  getCurrentLeafType
**
**  Which type of leaf is the bot in (water, slime, lava, normal)
**
*/

m_BSPnode::type_t bspGraph::getCurrentLeafType()
{
  if( !map || !currentNode )
	  return m_BSPnode::unknown;

	return currentNode->type();
}

/*
**  getLeafType
**
**  Same as above, but uses vector to find leaf first
**
*/

m_BSPnode::type_t bspGraph::getLeafType( vector pos )
{
  if( !map )
	  return m_BSPnode::unknown;

  m_BSPnode * leaf = findLeaf( pos );

  if( !leaf ) 
	  return m_BSPnode::unknown;

  return leaf->type();
}

/*
**  isShotBlocked 
**
**  return TRUE if there's a wall between bot's current position
**  and the one passed in. if *hp is not null, the origin of the
**  intersection is also returned.
**
*/

int bspGraph::isShotBlocked( vector origin, vector * hp )
{
	return isLineBlocked( position, origin, hp );
}


/*
**  findLeaf
**
**  Determines which leaf the vector is in.
**
*/

m_BSPnode * bspGraph::findLeaf( vector origin )
{
	if( !map )
		return (m_BSPnode *)0;

	m_BSPnode * pNode = map->worldspawnRoot();
	double dist;

	if( !pNode ) return (m_BSPnode *)0;

	while( pNode )
	{
		if( !pNode->isLeaf() )
		{
			dist = (origin * pNode->plane().normal()) - pNode->plane().distance();
			pNode = (dist > 0.0) ? pNode->front() : pNode->back();
		}
		else
		{
			return pNode;
		}
	}

	fprintf( debugFile, "bspGraph::findLeaf(): NULL child node\n" );
	fflush( debugFile );
	return (m_BSPnode *)0;

}

/*
**  isLineBlocked
**
**  returns TRUE iff the line segment intersects the BSP (not including
**  water, slime or lava) if hitPoint is not NULL, the origin of the 
**  intersection is returned (used for fireParticle). If plane is not
**  NULL, the plane which the intersection point lies on is returned.
**
*/

#define R_E 0.001		// rounding error

int bspGraph::isLineBlocked( vector sv, vector ev, vector * hitPoint )
{
	if( !map ) return FALSE;

	double e, s;
	vector start, end, tv;
	m_BSPnode * pNode, * pEnd, * pStart;

	static m_stack< vector > vectorStack;
	static m_stack< m_BSPnode * > nodeStack;

	vectorStack.empty();
	nodeStack.empty();

	start = sv;
	end = ev;

	pNode = map->worldspawnRoot();

	if( !pNode ) 
	{
		fprintf( debugFile, "bspGraph::isLineBlocked(): 0th model root is NULL\n" );
	}


	while( pNode )
	{
		if( !pNode->isLeaf() )
		{
			s = (start * pNode->plane().normal()) - pNode->plane().distance();
			e = (end * pNode->plane().normal()) - pNode->plane().distance();

			pEnd = (e > 0.0) ? pNode->front() : pNode->back();
			pStart=(s > 0.0) ? pNode->front() : pNode->back();

			if( ((s < -R_E) && (e > R_E)) || ((s > R_E) && (e < -R_E)) )
			{
				vectorStack.push( end );
				nodeStack.push( pEnd );
			
				tv = start;
				tv *= e;
				end *= s;
				end = end - tv;
				end /= (s-e);
			}

			pNode = pStart;
		}
		else
		{
			if( pNode->type() == m_BSPnode::solid ||
				pNode->type() == m_BSPnode::unknown )
			{
				if( hitPoint ) *hitPoint = start;
				return TRUE;
			}

			start = end;
			if( nodeStack.isEmpty() ) 
			{
				pNode = (m_BSPnode *)0;
			}
			else
			{
				end = vectorStack.pop();
				pNode = nodeStack.pop();
			}
		}
	}


	if( hitPoint ) *hitPoint = start;
	return FALSE;

}



/*
**  isLineBlockedFluid
**
**  Returns TRUE if the line is blocked by lava or slime, or water 
**  if water==1.
**
*/

int bspGraph::isLineBlockedFluid( vector sv, vector ev, vector * hitPoint, int water )
{
	if( !map ) return FALSE;

	double e, s;
	vector start, end, tv;
	m_BSPnode * pNode, * pEnd, * pStart;

	static m_stack< vector > vectorStack;
	static m_stack< m_BSPnode * > nodeStack;

	vectorStack.empty();
	nodeStack.empty();

	start = sv;
	end = ev;

	pNode = map->worldspawnRoot();

	if( !pNode ) 
	{
		fprintf( debugFile, "bspGraph::isLineBlocked(): 0th model root is NULL\n" );
	}


	while( pNode )
	{
		if( !pNode->isLeaf() )
		{
			s = (start * pNode->plane().normal()) - pNode->plane().distance();
			e = (end * pNode->plane().normal()) - pNode->plane().distance();

			pEnd = (e > 0.0) ? pNode->front() : pNode->back();
			pStart=(s > 0.0) ? pNode->front() : pNode->back();

			if( ((s < -R_E) && (e > R_E)) || ((s > R_E) && (e < -R_E)) )
			{
				vectorStack.push( end );
				nodeStack.push( pEnd );
			
				tv = start;
				tv *= e;
				end *= s;
				end = end - tv;
				end /= (s-e);
			}

			pNode = pStart;
		}
		else
		{
			if( pNode->type() == m_BSPnode::water && water ) 
			{
				if( hitPoint ) *hitPoint = start;
				return TRUE;
			}

			if( pNode->type() == m_BSPnode::lava || pNode->type() == m_BSPnode::slime ) 
			{
				if( hitPoint ) *hitPoint = start;
				return TRUE;
			}

			if( pNode->type() == m_BSPnode::solid || pNode->type() == m_BSPnode::unknown )
			{
				if( hitPoint ) *hitPoint = start;
				return FALSE;
			}

			start = end;
			if( nodeStack.isEmpty() ) 
			{
				pNode = (m_BSPnode *)0;
			}
			else
			{
				end = vectorStack.pop();
				pNode = nodeStack.pop();
			}
		}
	}


	if( hitPoint ) *hitPoint = start;
	return FALSE;
}


/*
**  modelBlocksLine 
**
**  Checks if a particular model is blocking the line segment.
**
**  TODO:
**    implement me.
**
*/

int bspGraph::modelBlocksLine( vector end, vector delta, int )
{
  return 0;
}


