/*
**  bspLevel
**  mike warren 1997
**
**  Implementation of "nice" bsp-file class
**
*/



#include "bsplevel.h"
#include "mfile.h"
#include "bsp_misc.h"


/*
**  ctor
**
**  Loads all relevent BSP information
**
*/

BSPlevel::BSPlevel( mFile & file )
{

	BSPhead head( file );

	if( head.version() != 29 && head.version() != 28 )
	{
		fprintf( debugFile, "BSPlevel::BSPlevel(): BSP version was %d\n", head.version() );
		fflush( debugFile );
#if QTHROW
		throw( "BSPlevel::BSPlevel(): .BSP version not 29 or 28\n" );
#else
		fprintf( stderr, "BSPlevel::BSPlevel(): Wrong BSP version (%d)\n", head.version() );
#endif
	}

	BSPentry ent;
	vector tv;
	int index, max, i, j;

				//
				//  load vertices
				//

	ent = head.get( BSPhead::vertices );

	file.setReadPos( ent.offset() );
	m_vertices = new vector[ ent.size()/BSP_VERTEX_SIZE ];

	fprintf( debugFile, "BSPlevel::BSPlevel(): loading %d vertices\n", ent.size()/BSP_VERTEX_SIZE );

	for( i=0; i < ent.size()/BSP_VERTEX_SIZE; i++ )
	{
		m_vertices[ i ].setx( file.readLEfloat() );
		m_vertices[ i ].sety( file.readLEfloat() );
		m_vertices[ i ].setz( file.readLEfloat() );
		
//		fprintf( debugFile, "  %04d: (%f, %f, %f)\n", i, m_vertices[i].getx(), m_vertices[i].gety(), m_vertices[i].getz() );
	}

	fprintf( debugFile, "BSPlevel::BSPlevel(): finished vertices\n" );
	fflush( debugFile );


				//
				//  load edges
				//

	ent = head.get( BSPhead::edges );

	file.setReadPos( ent.offset() );
	m_edges = new m_line[ ent.size()/BSP_EDGE_SIZE ];

	fprintf( debugFile, "BSPlevel::BSPlevel(): loading %d edges\n", ent.size()/BSP_EDGE_SIZE );

	for( i=0; i < ent.size()/BSP_EDGE_SIZE; i++ )
	{
		index = (int)file.readLEushort();
		max = (int)file.readLEushort();

		m_edges[ i ].first( &(m_vertices[ index ]) );
		m_edges[ i ].second( &(m_vertices[ max ]) );

//		fprintf( debugFile, "  %04d: %d --> %d\n", i, index, max );
	}

	fprintf( debugFile, "BSPlevel::BSPlevel(): finished edges\n" );
	fflush( debugFile );

				//
				//  load planes
				//

	ent = head.get( BSPhead::planes );

	file.setReadPos( ent.offset() );
	m_aPlanes = new m_plane[ ent.size()/BSP_PLANE_SIZE ];
	m_dNumPlanes = ent.size() / BSP_PLANE_SIZE;

	fprintf( debugFile, "BSPlevel::BSPlevel(): loading %d planes\n", ent.size()/BSP_PLANE_SIZE );

	for( i=0; i < ent.size()/BSP_PLANE_SIZE; i++ )
	{
		tv.setx( file.readLEfloat() );
		tv.sety( file.readLEfloat() );
		tv.setz( file.readLEfloat() );

		m_aPlanes[ i ].normal( tv );
		m_aPlanes[ i ].distance( file.readLEfloat() );

		switch( file.readLEint() ) 
		{ 
		case 0:
		case 3: 
			m_aPlanes[ i ].type( m_plane::x );
			break; 
		case 1:
		case 4: 
			m_aPlanes[ i ].type( m_plane::y );
			break; 
		case 2: 
		case 5: 
			m_aPlanes[ i ].type( m_plane::z );
			break; 
		default: 
			m_aPlanes[ i ].type( m_plane::unknown );
			fprintf( debugFile, "BSPlevel::BSPlevel(): unknown plane orientation (plane #%d)\n", i);
			break; 
		}

//		fprintf( debugFile, "  %04d: (%f, %f, %f).%f  [%d]\n", i, tv.getx(), tv.gety(), tv.getz(), m_aPlanes[i].distance(), (int)m_aPlanes[i].type() );
	}

	fprintf( debugFile, "BSPlevel::BSPlevel(): finished planes\n" );
	fflush( debugFile );

				//
				//  temporary edge list
				//

	ent = head.get( BSPhead::edgelist );

	file.setReadPos( ent.offset() );
	int * edgelist = new int[ ent.size()/BSP_EDGELIST_SIZE ];

	fprintf( debugFile, "BSPlevel::BSPlevel(): loading %d temporary edge list entries\n", ent.size()/BSP_EDGELIST_SIZE );

	for( i=0; i < ent.size()/BSP_EDGELIST_SIZE; i++ )
	{
		edgelist[ i ] = file.readLEint();
		
//		fprintf( debugFile, "  %04d: %d\n", i, edgelist[i] );
	}

	fprintf( debugFile, "BSPlevel::BSPlevel(): finished edgelist\n" );
	fflush( debugFile );

				//
				//  load faces (polygons)
				//

	ent = head.get( BSPhead::faces );

	file.setReadPos( ent.offset() );
	m_aPolygons = new m_polygon[ ent.size()/BSP_FACE_SIZE ];

	m_dNumPolygons = ent.size() / BSP_FACE_SIZE;
	
	fprintf( debugFile, "BSPlevel::BSPlevel(): loading %d polygons (faces)\n", ent.size()/BSP_FACE_SIZE );

	for( i=0; i < ent.size()/BSP_FACE_SIZE; i++ )
	{
		m_aPolygons[ i ].number( i );
		m_aPolygons[ i ].plane( &(m_aPlanes[ file.readLEushort() ]) );
		m_aPolygons[ i ].onFront( !file.readLEushort() );

		index = file.readLEint();	// edgelist index, num
		max = (int)file.readLEushort();
		file.readLEushort();		// texture index (unused)
		file.readLEint();			// lights (unused)
		file.readLEint();			// lightmap index (unused)

		for( j = index; j < index+max; j++ )
		{
			if( edgelist[ j ] > 0 )
			{
				if( !m_edges[ edgelist[j] ].pFirst() )
					fprintf( debugFile, "BSPlevel::BSPlevel(): WARNING: NULL first vertex (edgelist %d, edge %d)\n", j, edgelist[j] );
				else
					m_aPolygons[ i ].addPoint( m_edges[ edgelist[j] ].pFirst() );
			}
			else if( edgelist[ j ] < 0 )
			{
				if( !m_edges[ -edgelist[j] ].pSecond() )
					fprintf( debugFile, "BSPlevel::BSPlevel(): WARNING: NULL second vertex (edgelist %d, edge %d)\n", j, -edgelist[j] );
				else
					m_aPolygons[ i ].addPoint( m_edges[ -edgelist[j] ].pSecond() );
			}
			else
			{
				fprintf( debugFile, "BSPlevel::BSPlevel(): WARNING: edgelist indexed 0th edge\n" );
			}
		}

//		fprintf( debugFile, "  %04d: plane (%f, %f, %f), onFront %d, edgelistId %d, edgelistNum %d\n", i, m_aPolygons[i].plane().normal().getx(), m_aPolygons[i].plane().normal().gety(), m_aPolygons[i].plane().normal().getz(), m_aPolygons[ i ].onFront(), index, max );
	}

	delete [] edgelist;

	fprintf( debugFile, "BSPlevel::BSPlevel(): finished polygons\n" );
	fflush( debugFile );


					//
					//  load temporary face list
					//

	ent = head.get( BSPhead::facelist );
	int * facelist = new int[ ent.size() / BSP_FACELIST_SIZE ];

	file.setReadPos( ent.offset() );

	
	fprintf( debugFile, "BSPlevel::BSPlevel(): loading %d temporary facelist entries\n", ent.size()/BSP_FACELIST_SIZE );

	for( i=0; i < ent.size() / BSP_FACELIST_SIZE; i++ )
	{
		facelist[ i ] = (int)file.readLEushort();

//		fprintf( debugFile, "  %04d: %d\n", i, facelist[i] );
	}

	fprintf( debugFile, "BSPlevel::BSPlevel(): finished facelist\n" );
	fflush( debugFile );

					//
					//  load and build the BSP tree (nodes, leaves)
					//

					//  allocate space for leaves AND nodes

	ent = head.get( BSPhead::leaves );
	numberOfLeavesAndNodes = ent.size() / BSP_LEAF_SIZE;

	m_dNumLeaves = numberOfLeavesAndNodes;

	ent = head.get( BSPhead::nodes );
	numberOfLeavesAndNodes += ent.size() / BSP_NODE_SIZE;

	m_dLeafOffset = numberOfLeavesAndNodes - m_dNumLeaves;

	m_pBSPnodes = new m_BSPnode* [ numberOfLeavesAndNodes ];
	unsigned short * fronts = new unsigned short[ numberOfLeavesAndNodes ];
	unsigned short * backs = new unsigned short[ numberOfLeavesAndNodes ];

					//  load nodes

	file.setReadPos( ent.offset() );

	int faceid, facenum;

	fprintf( debugFile, "BSPlevel::BSPlevel(): loading %d nodes\n", ent.size()/BSP_NODE_SIZE );

	for( i=0; i < ent.size() / BSP_NODE_SIZE; i++ )
	{
		m_pBSPnodes[ i ] = new m_BSPnode();

		index = file.readLEint();

		if( index < 0 || index >= m_dNumPlanes )
		{
			fprintf( debugFile, "BSPlevel::BSPlevel(): WARNING: plane index out-of-range (%d)\n", index );
		}

		m_pBSPnodes[ i ]->plane( &m_aPlanes[ index ] );

		fronts[i] = file.readLEushort();
		backs[i] = file.readLEushort();

		if( fronts[i] & 0x8000 ) 
		{
			fronts[i] = ~fronts[i];
			fronts[i] += m_dLeafOffset;
		}
		if( backs[i] & 0x8000 )	
		{
			backs[i] = ~backs[i];
			backs[i] += m_dLeafOffset;
		}

		tv.setx( file.readLEshort() );
		tv.sety( file.readLEshort() );
		tv.setz( file.readLEshort() );
		m_pBSPnodes[ i ]->bboxMin( tv );

		tv.setx( file.readLEshort() );
		tv.sety( file.readLEshort() );
		tv.setz( file.readLEshort() );
		m_pBSPnodes[ i ]->bboxMax( tv );

		faceid = (int)file.readLEushort();
		facenum = (int)file.readLEushort();

//		fprintf( debugFile, "  %04d: faceId %d, faceNum %d\n", i, faceid, facenum );

		for( j=0; j < facenum; j++ )
		{
			m_pBSPnodes[ i ]->addPolygon( &(m_aPolygons[ j+faceid ]) );
		}

		m_pBSPnodes[ i ]->isLeaf( FALSE );

	}

	fprintf( debugFile, "BSPlevel::BSPlevel(): done nodes\n" );
	fflush( debugFile );

					//  load leaves

	ent = head.get( BSPhead::leaves );
	file.setReadPos( ent.offset() );

	fprintf( debugFile, "BSPlevel::BSPlevel(): loading %d leaves (leaf offset=%d)\n", ent.size()/BSP_LEAF_SIZE, m_dLeafOffset );

	for( ; i < numberOfLeavesAndNodes; i++ )
	{
		m_pBSPnodes[ i ] = new m_BSPnode();

		m_pBSPnodes[ i ]->isLeaf( TRUE );

		switch( file.readLEint() )
		{
		case -1: 
			m_pBSPnodes[ i ]->type( m_BSPnode::normal );
			break; 
		
		case -2: 
			m_pBSPnodes[ i ]->type( m_BSPnode::solid );
			break; 
		
		case -3: 
			m_pBSPnodes[ i ]->type( m_BSPnode::water );
			break; 
		
		case -4: 
			m_pBSPnodes[ i ]->type( m_BSPnode::slime );
			break; 
		
		case -5: 
			m_pBSPnodes[ i ]->type( m_BSPnode::lava );
			break; 
		
		case -6: 
			m_pBSPnodes[ i ]->type( m_BSPnode::sky );
			break; 
		
		default: 
			m_pBSPnodes[ i ]->type( m_BSPnode::unknown );
			break;
		}

		file.readLEint();		// visibility list index

		tv.setx( file.readLEshort() );
		tv.sety( file.readLEshort() );
		tv.setz( file.readLEshort() );
		m_pBSPnodes[ i ]->bboxMin( tv );

		tv.setx( file.readLEshort() );
		tv.sety( file.readLEshort() );
		tv.setz( file.readLEshort() );
		m_pBSPnodes[ i ]->bboxMax( tv );

		faceid = (int)file.readLEushort();
		facenum = (int)file.readLEushort();

		for( j=0; j < facenum; j++ )
		{
			m_pBSPnodes[ i ]->addPolygon( &(m_aPolygons[ facelist[j+faceid] ]) );
		}

		file.readLEint();		// sounds (ambient)

		m_pBSPnodes[ i ]->front( 0 );
		m_pBSPnodes[ i ]->back( 0 );

	}

	fprintf( debugFile, "BSPlevel::BSPlevel(): finished leaves\n" );
	fprintf( debugFile, "BSPlevel::BSPlevel(): Loaded %d nodes and leaves\n", numberOfLeavesAndNodes );
	fflush( debugFile );

					//  fix FRONT and BACK links for nodes

	ent = head.get( BSPhead::nodes );

	fprintf( debugFile, "BSPlevel::BSPlevel(): fixing front/back links for nodes\n" );

	for( i=0; i < ent.size() / BSP_NODE_SIZE; i++ )
	{
		if( fronts[i] > numberOfLeavesAndNodes || backs[i] > numberOfLeavesAndNodes )
		{
			fprintf( debugFile, "BSPlevel::BSPlevel(): %d/%d back/front reference out of range\n", backs[i], fronts[i] );
			fflush( debugFile );
		}
		if( fronts[i] < 0 || backs[i] < 0 )
		{
			fprintf( debugFile, "BSPlevel::BSPlevel(): %d/%d back/front reference out of range\n", backs[i], fronts[i] );
			fflush( debugFile );
		}
		if( !m_pBSPnodes[ fronts[i] ] )
			fprintf( debugFile, "BSPlevel::BSPlevel(): node %d is NULL\n", fronts[i] );
		else
			m_pBSPnodes[ i ]->front( m_pBSPnodes[ fronts[i] ] );

		if( !m_pBSPnodes[ backs[i] ] )
			fprintf( debugFile, "BSPlevel::BSPlevel(): node %d is NULL\n", backs[i] );
		else
			m_pBSPnodes[ i ]->back( m_pBSPnodes[ backs[i] ] );
	}

	delete [] fronts;
	delete [] backs;

	fprintf( debugFile, "BSPlevel::BSPlevel(): finished building tree\n" );
	fflush( debugFile );

					//
					//  load model roots
					//

	ent = head.get( BSPhead::models );
	file.setReadPos( ent.offset() );

	fprintf( debugFile, "BSPlevel::BSPlevel(): loading %d models\n", ent.size() / BSP_MODEL_SIZE );

	m_apModelRoots = new m_BSPnode* [ ent.size() / BSP_MODEL_SIZE ];

	for( i=0; i < ent.size() / BSP_MODEL_SIZE; i++ )
	{

					// bbox min

		file.readLEfloat();
		file.readLEfloat();
		file.readLEfloat();

					//  bbox max

		file.readLEfloat();
		file.readLEfloat();
		file.readLEfloat();

					//  origin

		file.readLEfloat();
		file.readLEfloat();
		file.readLEfloat();

		index = file.readLEint();

		if( !m_pBSPnodes[ index ] )
			fprintf( debugFile, "BSPlevel()::BSPlevel(): WARNING: node %d is NULL\n", index );
		else
			m_apModelRoots[ i ] = m_pBSPnodes[ index ];

		file.readLEint();		// clip node id
		file.readLEint();		// second clip node id ?
		file.readLEint();		// unknown; node id?
		file.readLEint();		// number of leaves
		file.readLEint();		// face id
		file.readLEint();		// number of faces

	}

	fprintf( debugFile, "BSPlevel::BSPlevel(): Completed loading .BSP\n" );
	fflush( debugFile );

	
}


/*
**  dtor
**
*/

BSPlevel::~BSPlevel()
{
	delete [] m_aPlanes;
	delete [] m_apModelRoots;
	delete [] m_aPolygons;

	for( int i=0; i < numberOfLeavesAndNodes; i++ )
	{
		delete m_pBSPnodes[ i ];
	}

	delete [] m_pBSPnodes;
}
