/*
**
**  QCS handles client-server communications
**
**  (c) mike warren, 1997
**  mikeBot
**
**  packet decoding functions
**
*/

#include "defines.h"
#include "qcs.h"

/*
 **  decodeReliable : puts reliableFragments into the fragmentBuffer (qcs::fb)
 **		     `till a reliableEnd is received,then they are decoded with
 **		     qcs::decodeMessages().
 **
 */

int qcs::decodeReliable( qpacket & qp )
{
  if( !connected() )
    {
      fprintf(stderr, "qcs::decodeReliable(): received reliable when not connected\n");
      return FALSE;
    }

  int pnum = qp.readBEint();
	
#if DEBUG & DQCS
  printf("qcs::decodeReliable(): packet #%d\n", pnum);
#endif

			//
			// build an acknowledge packet and send it
			//

  outPacket.reset();
  outPacket.addBEint( pnum );
  outPacket.changeType( qpacket::acknowledge );
  send( outPacket );

  if( pnum < incomingReliable )
    {
      fprintf(stderr, "qcs::decodeReliable(): packet discarded %d\n",pnum);
      return FALSE;
    }

  incomingReliable++;


  if( qp.getType() == qpacket::reliableFragment )
    {
      *fragments += qp;
#if QPROXY
      if( proxy && proxy->connected() )
	proxy->sendReliable( qp );
#endif
    }
  else			// qpacket::reliableEnd
    {
      *fragments += qp;
#if QPROXY
      if( proxy && proxy->connected() )
	proxy->sendReliable( qp );
#endif
      fragments->setReadPos( 0 );
      if( recording() && !demoPauseState )
	fragments->write( demoFP, 0, facing);
      decodeMessages( *fragments );
      fragments->reset();
    }

  return TRUE;

}



/*
 **  decodeAcknowledge : calls qack::gotAck()
 **
 */

int qcs::decodeAcknowledge( qpacket & qp )
{
  if( !connected() )
    {
      fprintf(stderr, "qcs::decodeAcknowledge(): got ack when not connected\n");
      return FALSE;
    }

  int i = qp.readBEint();

#if DEBUG & DQCS
  printf("qcs::decodeAcknowledge(): got %d\n", i );
#endif

  return ackQueue.gotAck( i, sock );

}

/*
 **  decodeUnreliable : most game data comes through this (i.e. unreliable).
 **
 */

int qcs::decodeUnreliable( qpacket & qp )
{
  if( !connected() )
    {
      fprintf(stderr, "qcs::decodeUnreliable(): got unreliable when not connected\n");
      return FALSE;
    }

  int pnum = qp.readBEint();	

#if DEBUG & DQCS
  printf("qcs::decodeUnreliable(): # %d\n", pnum);
#endif

  if( pnum < incomingUnreliable )
    {
      fprintf(stderr, "qcs::decodeUnreliable(): out-of-order packet %d\n", pnum);
      incomingUnreliable++;
      return FALSE;
    }

  incomingUnreliable++;

  qp.ffwd();		// fast-forward to end of packet...

#if QPROXY
  if( proxy && proxy->clientStatus() )
    {
      qp.addByte( SC_CENTERPRINT );
      qp.addString( qcs_message );
    }

				// shoot some particles from mbnav's
				// target

  if( proxy && proxy->clientParticle() )
    {
      qp.addByte( SC_PARTICLE );
      qp.addCoord( navParticle.getx() );	// origin(xyz)
      qp.addCoord( navParticle.gety() );
      qp.addCoord( navParticle.getz() );
      qp.addByte( (char)(rand()%256) ); // XYZ velocitiyes
      qp.addByte( (char)(rand()%256) );
      qp.addByte( (char)(rand()%256) );
      qp.addByte( (char)73 );	// color; LG
      qp.addByte( (char)100 );	// count; timeout?

				// and some from mbfire's
      
      qp.addByte( SC_PARTICLE );
      qp.addCoord( fireParticle.getx() );	// origin(xyz)
      qp.addCoord( fireParticle.gety() );
      qp.addCoord( fireParticle.getz() );
      qp.addByte( (char)(rand()%256) ); // XYZ velocitiyes
      qp.addByte( (char)(rand()%256) );
      qp.addByte( (char)(rand()%256) );
      qp.addByte( (char)73 );	// color; LG
      qp.addByte( (char)100 );	// count; timeout?
    }
      
				// write packet to filestream
#endif

#if QPROXY
  if( proxy && proxy->connected() )
    {
      if( proxy->clientLockview() )
	{
#else
	  qp.addByte( SC_SETANGLE );
	  qp.addAngle( facing.getPitch() );
	  qp.addAngle( facing.getYaw() );
	  qp.addAngle( (float)0.0 );
#endif
#if QPROXY
	}
      proxy->send( qp );
    }
#endif

  if( recording() && !demoPauseState )
    {
#if QPROXY
      if( !(proxy && proxy->clientStatus()) )
	{
#endif
	  qp.addByte( SC_CENTERPRINT );
	  qp.addString( qcs_message );
#if QPROXY
	}

      if( !(proxy && proxy->clientParticle()) )
	{
#endif
	  qp.addByte( SC_PARTICLE );
	  qp.addCoord( navParticle.getx() );	// origin(xyz)
	  qp.addCoord( navParticle.gety() );
	  qp.addCoord( navParticle.getz() );
	  qp.addByte( (char)(rand()%256) ); // XYZ velocitiyes
	  qp.addByte( (char)(rand()%256) );
	  qp.addByte( (char)(rand()%256) );
	  qp.addByte( (char)73 );	// color; LG
	  qp.addByte( (char)100 );	// count; timeout?

	  // and some from mbfire's
      
	  qp.addByte( SC_PARTICLE );
	  qp.addCoord( fireParticle.getx() );	// origin(xyz)
	  qp.addCoord( fireParticle.gety() );
	  qp.addCoord( fireParticle.getz() );
	  qp.addByte( (char)(rand()%256) ); // XYZ velocitiyes
	  qp.addByte( (char)(rand()%256) );
	  qp.addByte( (char)(rand()%256) );
	  qp.addByte( (char)73 );	// color; LG
	  qp.addByte( (char)100 );	// count; timeout?
#if QPROXY
	}
#endif

      qp.write( demoFP, 8, facing);

    }
  
  return decodeMessages( qp );

}


/*
 **  decodeQPacket : takes the passed packet and (tries) to decode it
 **
 */

int qcs::decodeQPacket( qpacket & qp )
{

  switch( qp.getType() )
    {
    case qpacket::control:
      return decodeControl( qp );
      break;

    case qpacket::reliableFragment:
    case qpacket::reliableEnd:
      return decodeReliable( qp );
      break;

    case qpacket::unreliable:
      return decodeUnreliable( qp );
      break;

    case qpacket::acknowledge:
      return decodeAcknowledge( qp );
      break;

    default:
      fprintf(stderr, "qcs::decodeQPacket(): can't decode packet!!!\n");
      break;

    case qpacket::unknown:
      fprintf(stderr, "qcs::decodeQPacket(): `qpacket::unknown' type!!!\n");
      break;
    }

  return FALSE;

}

/*
 **  decodeControl : decodes only control (type 0x8000, or qpacket::control)
 **		     returns FALSE if the packet wasn't decoded
 **
 */

int qcs::decodeControl( qpacket & qp )
{

  if( connected() )
    {
      printf("qcs::decodeControl(): got control when connected; ignoring\n");
      return FALSE;
    }

  int opcode = (int) qp.readByte();
  int i = 0;
  char * tempString = 0;

  switch( opcode )
    {
    case CCREP_SERVER_INFO:
      if( info.address )
	delete[] info.address;
      if( info.name )
	delete[] info.name;
      if( info.level )
	delete[] info.level;
      info.address = qp.readString();
      info.name = qp.readString();
      info.level = qp.readString();
      info.players = (int)qp.readByte();
      info.maxPlayers = (int)qp.readByte();
      return TRUE;
      break;

    case CCREP_ACCEPT:
      if ( result )
	delete[] result;
      result = (char *)0;
      i = (int) qp.readLEshort();
//#if DEBUG & DQCS
      printf("qcs::decode(): got port # %d\n", i );
//#endif
      sock->changePort( i );
      haveConnection = 1;
      return TRUE;
      break;

    case CCREP_REJECT:
      if( result )
	delete[] result;
      result = qp.readString();
      return TRUE;
      break;

    case CCREP_PLAYER_INFO:
      printf("%d : ", (int)qp.readByte() );
      tempString = qp.readString();
      parseString( tempString );
      printf("%-25s : ", tempString);
      delete[] tempString;
      qp.readLEint();		// colors
      printf("%-3d : ", qp.readLEint() );
      printf("%-6d secs : ", qp.readLEint() );
      tempString = qp.readString();

      printf("%s\n", tempString );

      delete[] tempString;
      return TRUE;
      break;

    case CCREP_RULE_INFO:
      if( qcs_lastrule )
	delete[] qcs_lastrule;

      if( qp.getSize() > 5 )
	qcs_lastrule = qp.readString();
      else
	{
	  qcs_lastrule=0;
	  return TRUE;
	}

      tempString = qp.readString();
      opts.set( qcs_lastrule, atoi( tempString ) );
      delete[] tempString;

      return TRUE;
      break;

    default:
      fprintf(stderr, "qcs::decodeControl(): unknown opcode %02x\n", opcode );
      return FALSE;
      break;
    }

  return FALSE;

}



/*
**  decodeMessages : the biggie; decodes the .DEM format messages. many thanks
**		     (again) to Uwe Girlich! :)
**
*/

int qcs::decodeMessages( qpacket & qp )
{
  float f1, f2, f3;
  int a, c, i;
  char x, y;
  short s1, s2;
  char * tempString;
  unsigned char opcode;
  vector tv;
  char temp[ 80 ];

#if DEBUG & DQCS
  printf("qcs::decodeMessages(): size is %d\n", qp.getSize() );
#endif

  while( qp.getReadPos() < qp.getSize() )
    {
      opcode = qp.readByte();
      switch( opcode )
	{
	case SC_BAD:
	  printf("qcs::decodeMessages(): got SC_BAD\n");
	  return FALSE;
	  break;
	  
	case SC_NOOP:
	  printf("qcs::decodeMessages(): server - client keepalive\n");
	  break;
	  
	case SC_DISCONNECT:
	  fprintf(stderr, "qcs::decodeMessages(): got disconnect\n");
	  disconnect();
	  return FALSE;
	  break;
	  
	case SC_UPDATESTAT:
	  a = (int)qp.readByte();
	  c = (int)qp.readLEint();
	  if (a < 32 && a >= 0 )
	    {	
	      playerstateUpdate( a, c );
#if DEBUG & DQCS
	      printf("playerstate  %d = %d\n", a, c );
#endif
	    }
	  else
	    fprintf(stderr,"qcs:decodeMessages():playerstate index wrong\n");
	  break;
	  
	case SC_VERSION:
#if DEBUG & DQCS
	  printf("qcs:decodeMessages(): version %d\n", qp.readBEint());
#else
	  qp.readBEint();
#endif
	  break;
	  
	case SC_SETVIEW:
	  myEntityNumber = qp.readLEshort();
	  players[myEntityNumber-1].hate = -1000;
	  printf("Bot is entity #%d\n", myEntityNumber);
#if DEBUG & DQCS
	  printf("bot is playernumber %d\n", myEntityNumber );
#endif
	  break;

		// FIXME : call virtual void heardSound()
	  
	case SC_SOUND:
	  x = qp.readByte();
	  f1 = (float)((x & 0x01) ? qp.readByte()/255.0 : 1.0);
	  f2 = (float)((x & 0x02) ? qp.readByte()/255.0 : 1.0);
	  s2 = qp.readLEshort();
	  y = qp.readByte();		//sound number
	  qp.readLEshort();	//x
	  qp.readLEshort();	//y
	  qp.readLEshort();	//z
	  break;
	  
	case SC_TIME:
	  oldTimestamp = newTimestamp;
	  newTimestamp = qp.readLEfloat();
#if DEBUG & DQCS
#if 0
	  printf("timestamp = %f\n", newTimestamp );
#endif
#endif
	  if( newTimestamp > oldTimestamp )
	    timestampChanged( newTimestamp );
	  break;
	  
	case SC_PRINT:
	  tempString = qp.readString();
#if DEBUG & DQCS
	  printf("`%s'\n", tempString );
#endif
	  gotSayMessage( tempString );
	  delete[] tempString;
	  break;

	  
	case SC_STUFFTEXT:
	  tempString = qp.readString();

	  parseName( (unsigned char *)tempString );
	  consoleCommand( tempString );

	  delete[] tempString;
	  break;
	  
	case SC_SETANGLE:	// set view angle; skip
	  f1 = qp.readAngle();
	  f2 = qp.readAngle();
	  f3 = qp.readAngle();
	  break;
	  
	case SC_SERVERINFO:
	  a = qp.readLEint();		// server version; should be 0x0F

	  if( a != 0xf )
	    fprintf(stderr, "qcs::decodeMessages(): server version %d, not 15\n", a );

	  info.maxPlayers = (int)qp.readByte();
	  qp.readByte();		// multiplayer 1, single 0

#if DEBUG & DQCS
	  printf("serverinfo version = %d\n", a );
	  printf("max players %d\n", info.maxPlayers);
#endif

	  if( info.level )
	    delete[] info.level;
	  info.level = qp.readString();

#if DEBUG & DQCS
	  printf("mapname = `%s'\n", info.level );
#else
	  printf("\nlevel is `%s'\n", info.level );
#endif

	  deleteTables();		// deletes any non-NULL entries in 
					// modeltable, soundtable

	  c=1;		
	  do {
	    modeltable[ c ] = qp.readString();
#if DEBUG & DQCS
	    printf("model : `%s'\n", modeltable[ c ] );
#endif
	    if ( c > QCS_MAX_MODELS )
	    {
	      fprintf(stderr, "qcs::decodeMessage(): too many models\n");
	      c--;
	    }
	  } while (*modeltable[c++]);
	  c=1;
	  do {
	    soundtable[ c ]=qp.readString();
#if DEBUG & DQCS
	    printf("sound : `%s'\n", soundtable[ c ] );
#endif
	    if ( c > QCS_MAX_SOUNDS )
	    {
	      fprintf(stderr, "qcs::decodeMessage(): too many sounds\n");
	      c--;
	    }
	  } while (*soundtable[c++]);

	  updateTableTypes();

	  break;

	  
	case SC_LIGHTSTYLE:	// don't need lightstyles
	  x = qp.readByte();
	  tempString = qp.readString();
#if DEBUG & DQCS
	  printf("lightstyle %d = `%s'\n", (int)x, tempString);
#endif
	  delete[] tempString;
	  break;
	  
	case SC_UPDATENAME:
	  x = qp.readByte();
	  if ((int)x > 15 || (int)x < 0 )
	    fprintf(stderr, "qcs::decodeMessages(): playernumber invalid %d\n", (int)x );
	  else
	  {
	    if( players[ x ].name )
	      delete[] players[ x ].name;

  	    players[ x ].name = qp.readString();
	    parseName( (unsigned char *)players[ x ].name );
	    if( x != myEntityNumber-1 )
	      players[ x ].hate = 0;
	    players[ x ].frags = 0;
	    players[ x ].index = 0;
	    players[ x ].shirt = 0;
	    players[ x ].pants = 0;
#if DEBUG & DQCS
	    printf("updatename : %d = `%s'\n", (int)x, players[x].name );
#endif
	  }
	  break;
	  
	case SC_UPDATEFRAGS:
	  a = qp.readByte();
	  s1 = qp.readLEshort();
	  if ((int)a > 15 || (int)a < 0 )
	    fprintf(stderr, "qcs::decodeMessages(): playernumber invalid %d\n",(int)a);
	  else
	  {
	    players[ a ].frags = (int) s1;
#if DEBUG & DQCS
	  printf("%d now has %d frags\n", (int)a, s1 );
#endif
	  }
	  break;

	  /// TODO : change these to readAngles()'s, etc.

	case SC_CLIENTDATA:
	  s1 = qp.readLEshort();		// bitmask
	  if (s1 & 0x0001)
	    qp.readByte();			// view ofset; don't need
	  if (s1 & 0x0002)
	    qp.readByte();			// ditto
	  f1 = (float)((s1 & 0x0004) ? (float)(qp.readByte() / (float)256.0*(float)360.0) : (float)(0.0));
	  (s1 & 0x0020)? qp.readByte() : 0; // velocity x
	  f2 = (float)((s1 & 0x0008) ? (float)(qp.readByte() / 256.0*360.0) : (float)(0.0));
	  (s1 & 0x0040) ? qp.readByte(): 0; // velocit y
	  f3 = (float)((s1 & 0x0010) ? (float)(qp.readByte() / 256.0*360.0) : (float)(0.0));
	  (s1 & 0x0080) ? qp.readByte(): 0; // velocity z
	  updateItems( (s1 & 0x200) ? qp.readLEint() : 0x4001 );

	  (s1 & 0x1000) ? qp.readByte() : 0; // weapon frame
	  updateArmour( (int)(s1 & 0x2000) ? qp.readByte() : 0 );
	  (int)(s1 & 0x4000) ? qp.readByte() : 0;//weapon model
	  updateHealth( (int)qp.readLEshort() );
	  (int)qp.readByte();	// current ammo
	  updateShells( (int)qp.readByte() );
	  updateNails( (int)qp.readByte() );
	  updateRockets( (int)qp.readByte() );
	  updateCells( (int)qp.readByte() );
	  updateWeapon( (int)qp.readByte() );

	  break;

	case SC_STOPSOUND:	// needed for "ears"?
	  s1 = qp.readLEshort();
	  break;

	case SC_UPDATECOLORS:
	  x = qp.readByte();
	  y = qp.readByte();
	  if( x > info.maxPlayers || x < 0)
	    fprintf(stderr, "qcs::decodeMessages(): player number invalid\n");
	  else
	    {
	      players[ (int)x ].pants = y & 0x0F;
	      players[ (int)x ].shirt = (y>>4) & 0x0F;
	    }
	  break;

	case SC_PARTICLE:	// starts a particle; not needed
	  qp.readLEshort();
	  qp.readLEshort();
	  qp.readLEshort();
	  qp.readByte();
	  qp.readByte();
	  qp.readByte();
	  x = qp.readByte();
	  qp.readByte();
	  break;


	case SC_DAMAGE:
	  x = qp.readByte();
	  y = qp.readByte();
	  tv.setx( qp.readCoord() );
	  tv.sety( qp.readCoord() );
	  tv.setz( qp.readCoord() );
	
	  receivedDamage( x+y, tv );
	
#if DEBUG & DQCS
	  printf("damage : save = %d, take = %d\n", (int)x, (int)y );
#endif
	  break;


	case SC_SPAWNSTATIC:
	  (int)qp.readByte();
	  (int)qp.readByte();
	  qp.readByte();				// skip colormap
	  (int)qp.readByte();
	  for( i=0; i < 3; i++)
	    {
	      qp.readLEshort();  // coord
	      qp.readByte();	  // angle
	    }
	  break;

	case SC_SPAWNBINARY:	// obsolete message
	  fprintf(stderr, "qcs::decodeMessages(): got SC_SPAWNBINARY\n");
	  break;
	  

	case SC_SPAWNBASELINE:
	  s1 = qp.readLEshort();
	  if( s1 >= QCS_MAX_ENTITIES || s1 < 0 )
	    {
	      fprintf(stderr, "qcs::decodeMessages(): entity reference invalid\n");
	      break;
	    }
	  entities[ s1 ].default_index = (int)qp.readByte();
	  entities[ s1 ].frame = ( (int)qp.readByte() );
	  qp.readByte();			// skip colormap
	  entities[ s1 ].skin = ( (int)qp.readByte() );
	  entities[ s1 ].origin.setx( qp.readCoord() );
	  entities[ s1 ].facing.setx( qp.readAngle() );
	  entities[ s1 ].origin.sety( qp.readCoord() );
	  entities[ s1 ].facing.sety( qp.readAngle() );
	  entities[ s1 ].origin.setz( qp.readCoord() );
	  entities[ s1 ].facing.setz( qp.readAngle() );

#if DEBUG & DQCS
	  printf("`%s' ", modeltable[ entities[s1].index ] );
	  entities[s1].origin.print();
	  entities[s1].facing.print();
	  printf("\n");
#endif
	  break;

				// 
				// TODO : use these?
				// 

	case SC_TEMPENTITY:
	  x = qp.readByte();
	  if ((int)x > 11)
	    fprintf(stderr, "qcs::decodeMessages(): bad TEMPENTITY type\n");
	  switch ((int)x)
	    {
	    case 0:
	    case 1:
	    case 2:
	    case 3:
	    case 4:
	    case 7:
	    case 8:
	    case 10:
	    case 11:
	      for( i=0; i < 3; i++ )
		qp.readLEshort();  // coord
	      break;
	    case 5:
	    case 6:
	    case 9:
	      s1 = qp.readLEshort();  // entity number?
	      for( i=0; i < 3; i++)
		qp.readLEshort();   // start coords
	      for ( i=0; i < 3; i++)
		qp.readLEshort();	// end coords
	      break;
	    }
	  break;

	case SC_SETPAUSE:
	  qp.readByte();
	  break;

	case SC_SIGNON:
	  signonLevel = (int)qp.readByte();
#if DEBUG & DQCS
	  printf("qcs::decodeMessages(): signon = %d\n", signonLevel);
#endif
	  switch( signonLevel )
	    {
	    case 1:
	      prespawns1.reset();
	      prespawns1.copy( qp );
#if QPROXY	
	      if( proxy )
		proxy->prespawn1( qp );
#endif

	
				// send NO-OP

	      outPacket.reset();
	      outPacket.addBEint( outgoingReliable++ );
	      outPacket.addByte( CS_KEEPALIVE );
	      outPacket.changeType( qpacket::reliableEnd );
	      send( outPacket );

			// send "prespawn" console command

	      outPacket.reset();
	      outPacket.addBEint( outgoingReliable++ );
	      outPacket.addByte( CS_CONSOLE );
	      outPacket.addString("prespawn\n");
	      outPacket.changeType( qpacket::reliableEnd );
	      send( outPacket );
	      break;

	    case 2:

	      prespawns2.reset();
	      prespawns2.copy( qp );
#if QPROXY
	      if( proxy )
		proxy->prespawn2( qp );
#endif
	
			// send name, colors, "spawn" console command

	      outPacket.reset();
	      outPacket.addBEint( outgoingReliable++ );
	      outPacket.addByte( CS_CONSOLE );
	      sprintf( temp, "name \"%s\"\n", qcs_botName );
	      outPacket.addString( temp );
	      outPacket.addByte( CS_CONSOLE );
	      sprintf( temp, "color \"%d %d\"\n", qcs_botShirt, qcs_botPants );
	      outPacket.addString( temp );
	      outPacket.addByte( CS_CONSOLE );
	      outPacket.addString("spawn\n");
	      outPacket.changeType( qpacket::reliableEnd );
	      send( outPacket );
	      break;
	      
	    case 3:

	      prespawns3.reset();
	      prespawns3.copy( qp );
#if QPROXY
	      if( proxy )
		proxy->prespawn3( qp );
#endif

			// send "begin" console command

	      outPacket.reset();
	      outPacket.addBEint( outgoingReliable++ );
	      outPacket.addByte( CS_CONSOLE );
	      outPacket.addString("begin\n");
	      outPacket.changeType( qpacket::reliableEnd );
	      send( outPacket );

	      changedLevel( modeltable[1] );

	      break;

	    default:
	      fprintf(stderr,"qcs::decodeMessages(): unknown signon level %d\n", signonLevel);
	      break;
	    }

	  break;

	case SC_CENTERPRINT:
	  tempString = qp.readString();

	  centerPrint( tempString );

	  delete[] tempString;
	  break;

	case SC_KILLEDMONSTER:
	case SC_FOUNDSECRET:
	  break;

				// 
				// static sounds : skip 'em
				// 

	case SC_SPAWNSTATICSOUND: 
	  for( i=0; i < 3; i++)
	    qp.readLEshort();
	  qp.readByte();		
	  qp.readByte();		
	  qp.readByte();		
	  break;

	case SC_INTERMISSION:
	  printf("rankings: (intermission)\n");
	  rankings();
	  break;

	case SC_FINALE:
	  tempString = qp.readString();
	  delete[] tempString;
	  break;

	case SC_CDTRACK:
	  qp.readByte();	// start track
	  qp.readByte();	// end track
	  break;

	case SC_SELLSCREEN:
	  printf("buyme, ya bastard\n");
	  break;

	default:
	  s1 = opcode & 0x07F;
	  if( s1 & 0x0001) 
	    s1 |= ((short)qp.readByte() << 8);
	  s2 = (short)((s1 & 0x4000) ? qp.readLEshort() : (unsigned char)qp.readByte());
	  if (!(s2 < QCS_MAX_ENTITIES && s2 >= 0))
	    {
	      fprintf(stderr, "qcs::decodeMessages(): entity ref. invalid %d\n", s2);
	      break;
	    }

	  if (s1 & 0x0400)
	    {
	      entities[ s2 ].index = (int)qp.readByte();
	      if( entities[s2].index < 0 )
		entities[s2].index = 0;
	    }
	  else
	    entities[s2].index = entities[s2].default_index;

	  if (s1 & 0x0040)
	    entities[ s2 ].frame = (int)qp.readByte();

	  if (s1 & 0x0800)
	    qp.readByte();		//skip;colormap
	  if (s1 & 0x1000)
	    entities[ s2 ].skin = (int)qp.readByte();

				// 
				// TODO : use "effects" (glowing)
				// 

	  if (s1 & 0x2000)
	    qp.readByte();		//attack-state, effects

	  tv = entities[ s2 ].origin;

	  if (s1 & 0x0002)
	    tv.setx( qp.readCoord() );
	  if (s1 & 0x0100)
	    entities[ s2 ].facing.setx( qp.readAngle() );
	  if (s1 & 0x0004)
	    tv.sety( qp.readCoord() );
	  if (s1 & 0x0010)
	    entities[ s2 ].facing.sety( qp.readAngle() );
	  if (s1 & 0x0008)
	    tv.setz( qp.readCoord() );
	  if (s1 & 0x0200)
	    entities[ s2 ].facing.setz( qp.readAngle() );
	  if (s1 & 0x0020)	// got new info (model index, etc...?)
	    {
	      entityChanged( (int)s2 );
	      printf("QCS::new model %s\n", modeltable[entities[s2].index]);
	    }

	  if( s2 < info.maxPlayers )
	    tv.setz( tv.getz()+8 );

          entities[ s2 ].updateOrigin( tv,entities[s2].lastTime,newTimestamp );
	  entities[ s2 ].lastTime = newTimestamp;

	  entityUpdated( (int)s2 );

	  break;
	}

    }

  return TRUE;

}





