/*
**
**  QCS handles client-server communications
**
**  (c) mike warren, 1997
**  mikeBot
**
**  ``core'' functions; connect, disconnect, etc.
**
*/



#include "defines.h"
#include "qcs.h"

#include <sys/types.h>
#include <sys/time.h>

/*
**  reinit : deletes everything, then calls ctor
**
*/

void qcs::reinit()
{

  if( recording() )
    stopDemo();

  if( info.name )
    delete info.name;
  info.name = (char *)0;
  if( info.level )
    delete info.level;
  info.level = (char *)0;
  if( info.address )
    delete info.address;
  info.address = (char *)0;

  fragments->reset();

  outgoingReliable = 0;
  outgoingUnreliable = 0;
  incomingReliable = 0;
  incomingUnreliable = 0;

  haveConnection = 0;
  newTimestamp = 0.0;
  oldTimestamp = 0.0;
  signonLevel = 0;

  myEntityNumber = 0;

  int i;
  for( i=0; i < QCS_MAX_ENTITIES; i++ )
    {
      entities[i].origin.set(0,0,0);
      entities[i].velocity.set(0,0,0);
      entities[i].facing.set(0,0,0);
      entities[i].index = 0;
      entities[i].default_index = 0;
      entities[i].frame = 0;
      entities[i].skin = 0;
      entities[i].lastTime = 0.0;
      entities[i].rate = 0;
    }

  for( i=0; i < QCS_MAX_PLAYERS; i++ )
    {
      players[i].hate = 0;
      players[i].index = 0;
      players[i].frags=0;
      players[i].name = 0;
      players[i].shirt = 0;
      players[i].pants = 0;
    }

  deleteTables();
  initTables();			// make all entries NULL

  inPacket.reset();
  outPacket.reset();
  prespawns1.reset();
  prespawns2.reset();
  prespawns3.reset();

  ackQueue.clear();

  for( i=0; i < QCS_MAX_PLAYERS; i++ )
    {
      if( players[i].name )
	delete players[i].name;
      players[i].name = 0;
    }
}


/*
**  ctor : simply inits variables
*/

qcs::qcs()
{
  sock = (qsocket *)0; 
  demoFP = (FILE *)0;
  
#if QPROXY
  proxy = new qproxy( 25999 );
#endif
  info.name = (char *)0;
  info.level = (char *)0;
  info.address = (char *)0;

  fragments = new qpacket( QCS_FRAGMENT_BUFFER );

  outgoingReliable = 0;
  outgoingUnreliable = 0;
  incomingReliable = 0;
  incomingUnreliable = 0;

  haveConnection = 0;
  newTimestamp = 0.0;
  oldTimestamp = 0.0;
  signonLevel = 0;

  myEntityNumber = 0;

  int i;
  for( i=0; i < QCS_MAX_ENTITIES; i++ )
    {
      entities[i].origin.set(0,0,0);
      entities[i].velocity.set(0,0,0);
      entities[i].facing.set(0,0,0);
      entities[i].index = 0;
      entities[i].default_index = 0;
      entities[i].frame = 0;
      entities[i].skin = 0;
      entities[i].lastTime = 0.0;
      entities[i].rate = 0;
    }

  for( i=0; i < QCS_MAX_PLAYERS; i++ )
    {
      players[i].hate = 0;
      players[i].index = 0;
      players[i].frags=0;
      players[i].name = 0;
      players[i].shirt = 0;
      players[i].pants = 0;
    }

  initTables();			// make all entries NULL

}


/*
 **  recordDemo : opens the filename passed, writes the header (<cd track>\n)
 **		 and all subsequent update()'s write to the file.
 **
 **  TODO : allow recording part-way through a level
 */

int qcs::recordDemo( char * filename )
{
  if ( recording() )		// already recording
    return FALSE;		

  if( connected() )
    printf("qcs::record(): joining game already in progress...\n");

  demoFP = fopen( filename, "wb" );

  if( !demoFP )			// open failed
    {
      fprintf(stderr, "qcs::recordDemo(): couldn't open `%s'\n", filename);
      return FALSE;
    }
  else
    printf("opened `%s' for writing\n", filename);

			// write "header", such as it is (cd track; -1=none)

  char h[3];
  
  h[0] = 0x2d;
  h[1] = 0x31;
  h[2] = 0x0A;

  fwrite( h, 1, 3, demoFP );

  vector tv;
  if( connected() )
    {
      prespawns1.write( demoFP, 0, tv );
      prespawns2.write( demoFP, 0, tv );
      prespawns3.write( demoFP, 0, tv );
    }
	  
  return TRUE;
}

/*
 **  stopDemo : stops recording demo ( if any ) and closes file. TRUE if succes
 **
 */

int qcs::stopDemo()
{
  if( !recording() )
    return FALSE;

  char t[ 17 ];
  int i=0;
  for( i=0; i < 16; i++ )
    t[i]=0;

#ifdef QBIG_ENDIAN
  t[0] = 1;
#else
  t[4] = 1;
#endif

  t[16] = SC_DISCONNECT;

  fwrite( t, 1, 17, demoFP );

  fclose( demoFP );
  demoFP = (FILE *)0;
  
  return TRUE;
}


/*
 **  Server : releases (if any) the current socket, and obtains a new one, 
 **	      pointed at the specified ip. returns FALSE on error
 **
 */


int qcs::server( char * ip, int port )
{
  
  if( connected() )
    disconnect();	
  
  if (sock)
    {
      delete sock;
      sock = (qsocket *)0;
    }

  for( int i=0; i < 10; i++ )
    if( !strcmp( ip, servers[i] ) )
      {
	printf("qcs::server(): using `%s' for `%s'\n", address[i],ip);
	ip = address[i];
	break;
      }

#if QTHROW
  try
    {
      sock = new qsocket( ip, port );
    }
  catch( char * err )
    {
      printf("qcs::server(): `%s'\n", err );
      sock = (qsocket *)0;
      return FALSE;
    }
#else
      sock = new qsocket( ip, port );
#endif

  strcpy( currentServerIP, ip );
  outgoingReliable=0;
  incomingReliable=0;
  outgoingUnreliable=0;
  incomingUnreliable=0;
  newTimestamp=0.0;
  oldTimestamp=0.0;

  return TRUE;
}


/*
 **  update : call this ONLY WHEN THERE IS DATA TO BE READ FROM THE SOCKET 
 **	      all decoding, etc, is then handled
 **	     
 */

int qcs::update()
{	

  static int calls=0;

  if ( (sock->receive( inPacket.getBuffer(), inPacket.getMaxSize())) <= 0 )
    {
#if UNIX
      if( errno != EWOULDBLOCK )
#endif
        fprintf(stderr,"qcs::update(): read error\n");
      return FALSE;
    }

  inPacket.init();			// updates type, size from header
  decodeQPacket( inPacket );		// decodes it

  ackQueue.resend( sock );		// update the acknowledge queue

  if( signonLevel == 3 )
    {	
      calls=0;
      sendMovement();
    }

#if QPROXY
    proxy->update();
#endif

  return TRUE;

}

/*
 **  connect : returns (char *)0 if connect was successful, else returns the
 **	      reject string. (i.e. "Server is full.\n")
 **
 */

char * qcs::connect()
{

  result = "No responce.\n";

  outPacket.reset();	
  outPacket.addByte( (char)0x01 );
  outPacket.addString( "QUAKE" );
  outPacket.addByte( (char)0x03 );
  outPacket.changeType( qpacket::control );
  send( outPacket );

  int i=0;

  while( i++ < QCS_MAX_TIMEOUTS )
  {
    printf("qcs::connect(): trying...\n");
    if( !waitForPacket() )
      {
	printf( result );
	send( outPacket );
      }
    else if ( connected() )
      break;
  }

  if( connected() )
    return (char *)0;
  else
    return result;

}

/*
 **  disconnect : deallocs qsocket, sends disconnect
 */

void qcs::disconnect()
{
  
  printf("qcs::disconnect(): disconnecting...\n");

  if( connected() )		// send 3 times 'cause itt
  {
    sendDisconnect();
    sendDisconnect();
    sendDisconnect();
  }
  if( recording() )
    stopDemo();
  
  haveConnection = 0;

  reinit();

  delete sock;
  sock = (qsocket *)0;		// yes, this *is* important
  server( currentServerIP, 26000 ); // aquire new socket

}

/*
**  waitForPacket : waits to receive a packet; return TRUE if one received, 
**		    FALSE on timeout. NOTE that update() *IS* CALLED!
**
**	!!!->	    this is meant to be used to wait for control packets (only)
**
*/

int qcs::waitForPacket()
{

  fd_set a;
  struct timeval tv;

  tv.tv_sec = QCS_MAX_WAIT;
  tv.tv_usec = QCS_MAX_WAIT * 1000;

  FD_ZERO( &a );
  FD_SET( sock->getFD(), &a );

  if ( (select( sock->getFD()+1, &a, NULL, NULL, &tv )) == -1 )
    {
      perror("qcs::waitForPacket(): select()");
      return FALSE;
    }

#if DEBUG & DQCS
  printf("qcs::waitForPacket(): %s data\n", FD_ISSET(sock->getFD(),&a)?"got some":"no");
#endif

  
  if( FD_ISSET( sock->getFD(), &a ) )
    update();
  else
    return FALSE;

  return TRUE;
  
}



/*
**  initTables : makes each entry in modeltable, soundtable == (char *)0
**
*/

void qcs::initTables()
{
	int i=0;

	for(i=0; i < QCS_MAX_MODELS; i++ )
	  modeltable[ i ] = (char *)0;
	
	for(i=0; i < QCS_MAX_SOUNDS; i++ )
	  soundtable[ i ] = (char *)0;

}

/*
**  deleteTables : deletes each entry in soundtable, modeltable (if they are
**		   not already NULL)
**
*/

void qcs::deleteTables()
{
	int i=0;

	for(i=0; i < QCS_MAX_MODELS; i++)
	  if( modeltable[i] ) 
	    delete modeltable[i];

	for(i=0; i < QCS_MAX_SOUNDS; i++)
	  if( soundtable[i] )
	    delete soundtable[i];

	initTables();
}


/*
**  updateTableTypes : slow function which determines what "modeltype" each
**                     of the table entries is.
**
*/

void qcs::updateTableTypes()
{

  int i;

  for( i=1; i < QCS_MAX_MODELS; i++ )
    if( modeltable[i] )
      {

	modeltypes[i] = 0;
				// first, determine major types

	if( !strncmp( modeltable[i], "progs/g_", 8 ) )
	  {
	    modeltypes[ i ] = ET_WEAPON;
	    if( !strcmp( modeltable[i], "progs/g_shot.mdl" ) )
	      {
		modeltypes[i] |= ET_W_SSG;
	      }
	    else if ( !strcmp( modeltable[i], "progs/g_rock.mdl" ))
	      {
		modeltypes[i] |= ET_W_GL;
	      }
	    else if ( !strcmp( modeltable[i], "progs/g_rock2.mdl" ))
	      {
		modeltypes[i] |= ET_W_RL;
	      }
	    else if ( !strcmp( modeltable[i], "progs/g_nail.mdl" ))
	      {
		modeltypes[i] |= ET_W_NG;
	      }
	    else if ( !strcmp( modeltable[i], "progs/g_nail2.mdl" ))
	      {
		modeltypes[i] |= ET_W_SNG;
	      }
	    else if ( !strcmp( modeltable[i], "progs/g_light.mdl" ))
	      {
		modeltypes[i] |= ET_W_LG;
	      }
	  }
	else if ( !strcmp( modeltable[i], "progs/missile.mdl" ) )
	  modeltypes[i] = ET_PROJECTILE | ET_P_MISSILE;
	else if ( !strcmp( modeltable[i], "progs/grenade.mdl" ) )
	  modeltypes[i] = ET_PROJECTILE | ET_P_GRENADE;
	else if ( !strcmp( modeltable[i], "progs/s_spike.mdl" ) )
	  modeltypes[i] = ET_PROJECTILE | ET_P_NAIL;
	else if ( !strcmp( modeltable[i], "progs/spike.mdl" ) )
	  modeltypes[i] = ET_PROJECTILE | ET_P_NAIL;
//	else if ( !strcmp( modeltable[i], "progs/v_spike.mdl" ) )
//	  modeltypes[i] = ET_PROJECTILE | ET_P_NAIL;
	else if ( !strncmp( modeltable[i], "maps/b_bh", 9 ) )
	  {
	    modeltypes[i] = ET_HEALTH;
	    if( !strcmp( modeltable[i], "maps/b_bh100.bsp"))
	      {
		modeltypes[i] |= ET_H_100;
	      }
	    else if( !strcmp( modeltable[i], "maps/b_bh25.bsp"))
	      {
		modeltypes[i] |= ET_H_25;
	      }
	    else if( !strcmp( modeltable[i], "maps/b_bh10.bsp"))
	      {
		modeltypes[i] |= ET_H_15;
	      }
	  }
	else if ( !strcmp( modeltable[i], "progs/armor.mdl" ) )
	  modeltypes[i] = ET_ARMOR;
	else if ( !strncmp( modeltable[i], "maps/b", 6 ) )
	  {
	    modeltypes[i] = ET_AMMO;
	    if( !strncmp( modeltable[i], "maps/b_shell", 12 ) )
	      modeltypes[i] |= ET_A_SHELLS;
	    else if( !strncmp( modeltable[i], "maps/b_batt", 11 ) )
	      modeltypes[i] |= ET_A_CELLS;
	    else if( !strncmp( modeltable[i], "maps/b_rock", 11 ) )
	      modeltypes[i] |= ET_A_ROCKETS;
	    else if( !strncmp( modeltable[i], "maps/b_nail", 11 ) )
	      modeltypes[i] |= ET_A_NAILS;
	  }
	else 
	  {
	    modeltypes[i] = ET_MISC;
	    if ( !strncmp( modeltable[i], "progs/gib", 9))
	      modeltypes[i] |= ET_BODY;
	    else if ( !strcmp( modeltable[i], "progs/h_player.mdl" ))
	      modeltypes[i] |= ET_BODY;
	    else if ( *modeltable[i]=='*' )
	      ;
	    else if ( !strcmp( modeltable[i], "progs/backpack.mdl" ))
	      modeltypes[i] = ET_ITEM | ET_I_BACKPACK;
	    else if ( !strcmp( modeltable[i], "progs/quaddama.mdl" ))
	      modeltypes[i] = ET_ITEM | ET_I_QUAD;
	    else if ( !strcmp( modeltable[i], "progs/invisibl.mdl" ))
	      modeltypes[i] = ET_ITEM | ET_I_INVISIBLE;
	    else if ( !strcmp( modeltable[i], "progs/invulner.mdl" ))
	      modeltypes[i] = ET_ITEM | ET_I_POP;
	    else if ( !strcmp( modeltable[i], "progs/suit.mdl" ))
	      modeltypes[i] = ET_ITEM | ET_I_BIOSUIT;
	    else
	      printf(" `%s' is unknown\n", modeltable[i] );

	  }

      }


				// 
				// TODO : do this for sounds, too
				// 

      
}


