

#include "graph.h"
#include "stack.h"
#include <memory.h>


/*
**  newMap : loads new map, if possible
**
*/

int bspGraph::newMap( char * fname )
{
  if( map )
    delete map;

  printf("loading BSP...");
  fflush( stdout );

  try {
    map = new bspFile( fname );
  }
  catch( char * err )
    {
      fprintf(stderr, "\n%s\nbspGraph::newMap(): level not loaded\n",err);
      delete map;
      map=0;
      return FALSE;
    }

  eraseGraph();

  int i;

  listSize = map->leaves->getNum();
  adjList = new adjNode*[ listSize+1 ];
  for( i=0; i <= listSize; i++ )
    if( !(adjList[ i ] = new adjNode(-1)) )
      fprintf(stderr, "bspGraph::newMap(): node failed\n");;
  t = new table[ listSize+1 ];


  return TRUE;

}

/*
**  updatePosition : call this everytime the bot moves...figures out where
**		     it is and possibly starts remaking paths
**
*/

void bspGraph::updatePosition( vector newPos )
{
  oldNode = currentNode;
  currentNode = findLeaf( newPos );
  position=newPos;
}


/*
**  getLeafOrigin : return the center of the bbox for the specified leaf
**
*/

vector bspGraph::getLeafOrigin( int x )
{
  vector r;
  bspLeaf * l;

  if( !map )
    return r;

  l = map->leaves->getLeaf( x );
  if( !l )
    return r;

  r = l->getBboxMin();
  vector r2 = l->getBboxMax();
  r += r2;
  r /= 2.0;

  return r;
}
  

/*
**  getCurrentLeafType : which type of leaf is the bot in
**
*/

int bspGraph::getCurrentLeafType()
{
  if( !map )
    return -100;

  return map->leaves->getLeaf( currentNode )->getType();

}

int bspGraph::getLeafType( vector pos )
{
  if( !map )
    return -100;

  return map->leaves->getLeaf( findLeaf(pos) )->getType();
}

/*
**  isShotBlocked : return TRUE if there is a wall between bots current pos
**                  and the one passed in
**
*/

int bspGraph::isShotBlocked( vector origin, vector * hp )
{

  if( isLineBlocked( position, origin, hp ) )
    return TRUE;
  else
    return FALSE;

}


/*
**  findLeaf : determines which leaf the pos vector is in
**
*/

int bspGraph::findLeaf( vector origin )
{

  int node;
  int isNode = 1;
  float dist;

  if( !map )
    return 0;
  
  node = map->models->getModel( 0 )->getBspRoot();

  while( 1 )
    {
      if( isNode )
	{
	  dist = ((map->planes->getPlane( map->nodes->getNode(node)->getPlaneId() )->getNormal()) *
	    (origin)) - 
	      (map->planes->getPlane( map->nodes->getNode(node)->getPlaneId() )->getDistance());
	  if( dist > 0.0 )
	    {
	      isNode = !(map->nodes->getNode(node)->isFrontLeaf());
	      node = map->nodes->getNode(node)->getFront();
	    }
	  else
	    {
	      isNode = !(map->nodes->getNode(node)->isBackLeaf());
	      node = map->nodes->getNode(node)->getBack();
	    }
	}
      else
	break;
    }

#if DEBUG & MBM
//  printf("bspGraph::findLeaf(): leaf %d\n", node);
#endif

  return node;
}
	  


/*
**  isLineBlocked : returns 0 if the line is not blocked, or the index
**	            of the leaf which blocks it.
**
*/

int bspGraph::isLineBlocked( vector sv, vector ev, vector * hitPoint )
{

  stack< vector > st;
  stack< int > st1;
  vector start=sv, end=ev, tv;
  float s,e;
  int node, startNode, endNode;
  int isNode=1;

  if( !map )
    return 0;
  
  node = map->models->getModel( 0 )->getBspRoot();

  while ( 1 )
    {
      if( node == -1 )
	break;

      if( isNode )
	{
	  // e = distance from end to plane
	  // s = distance from start to plane

	  e = ((map->planes->getPlane( map->nodes->getNode(node)->getPlaneId() )->getNormal()) * (end)) - (map->planes->getPlane( map->nodes->getNode(node)->getPlaneId() )->getDistance());
	  s = ((map->planes->getPlane( map->nodes->getNode(node)->getPlaneId() )->getNormal()) * (start)) - (map->planes->getPlane( map->nodes->getNode(node)->getPlaneId() )->getDistance());
      
#define RE 0.001

	  if( e > 0.0 )
	    endNode = map->nodes->getNode(node)->getFront();
	  else
	    endNode = map->nodes->getNode(node)->getBack();

	  if( s > 0.0 )
	    {
	      isNode = !(map->nodes->getNode(node)->isFrontLeaf());
	      startNode = map->nodes->getNode(node)->getFront();
	    }
	  else
	    {
	      isNode = !(map->nodes->getNode(node)->isBackLeaf());
	      startNode = map->nodes->getNode(node)->getBack();
	    }

	  if( (s < -RE && e > RE) || (s > RE && e < -RE) )
	    {
	      st.push( end );
	      st1.push( endNode );
	      tv = start;
	      tv *= e;
	      end *= s;
	      end = end - tv;
	      end /= (s-e);
	    }
	  node = startNode;
	}
      else
	{
	  if( map->leaves->getLeaf( node )->getType() != bspLeaf::normal )
	    return map->leaves->getLeaf(node)->getType();

	  start = end;
	  if( st.isEmpty() )
	    return FALSE;

	  end = st.pop();
	  node = st1.pop();
	  isNode=1;
	}
    }

  if( hitPoint )
    *hitPoint = end;

  return FALSE;
  
}


/*
**  lineHitsLava : returns TRUE if the line passes throgh lava
**
*/

int bspGraph::lineHitsLava( vector sv, vector ev )
{

  stack< vector > st;
  stack< int > st1;
  vector start=sv, end=ev, tv;
  float s,e;
  int node, startNode, endNode;
  int isNode=1;

  if( !map )
    return 0;
  
  node = map->models->getModel( 0 )->getBspRoot();

  while ( 1 )
    {
      if( node == -1 )
	break;

      if( isNode )
	{
	  // e = distance from end to plane
	  // s = distance from start to plane

	  e = ((map->planes->getPlane( map->nodes->getNode(node)->getPlaneId() )->getNormal()) * (end)) - (map->planes->getPlane( map->nodes->getNode(node)->getPlaneId() )->getDistance());
	  s = ((map->planes->getPlane( map->nodes->getNode(node)->getPlaneId() )->getNormal()) * (start)) - (map->planes->getPlane( map->nodes->getNode(node)->getPlaneId() )->getDistance());
      
	  if( e > 0.0 )
	    endNode = map->nodes->getNode(node)->getFront();
	  else
	    endNode = map->nodes->getNode(node)->getBack();

	  if( s > 0.0 )
	    {
	      isNode = !(map->nodes->getNode(node)->isFrontLeaf());
	      startNode = map->nodes->getNode(node)->getFront();
	    }
	  else
	    {
	      isNode = !(map->nodes->getNode(node)->isBackLeaf());
	      startNode = map->nodes->getNode(node)->getBack();
	    }

	  if( (s < -RE && e > RE) || (s > RE && e < -RE) )
	    {
	      st.push( end );
	      st1.push( endNode );
	      tv = start;
	      tv *= e;
	      end *= s;
	      end = end - tv;
	      end /= (s-e);
	    }
	  node = startNode;
	}
      else
	{
	  if( map->leaves->getLeaf( node )->getType() == bspLeaf::lava )
	    return TRUE;

	  start = end;
	  if( st.isEmpty() )
	    return FALSE;

	  end = st.pop();
	  node = st1.pop();
	  isNode=1;
	}
    }

  return FALSE;
  
}


/*
**  leavesAdjacent : return TRUE iff leaf B is reachable from leaf A in a
**	             straight-line-no-jumping kinda sence
**
*/

int bspGraph::leavesAdjacent( int a, int b )
{
  return FALSE;

  if( a > map->leaves->getNum() || b > map->leaves->getNum() )
    return FALSE;

  vector abmin, abmax, bbmin, bbmax;
  bspLeaf *al, *bl;
  float x,y,z;
  float xx,yy,zz;

  al = map->leaves->getLeaf( a );
  bl = map->leaves->getLeaf( b );

  abmin = al->getBboxMin();
  abmax = al->getBboxMax();
  bbmin = bl->getBboxMin();
  bbmax = bl->getBboxMax();

  x = abmax.getx() - abmin.getx();
  y = abmax.gety() - abmin.gety();
  z = abmax.getz() - abmin.getz();
  xx = bbmax.getx() - bbmin.getx();
  yy = bbmax.gety() - bbmin.gety();
  zz = bbmax.getz() - bbmin.getz();

  if( bbmin.getx() > abmin.getx() && 
     bbmin.gety() > abmin.gety() &&
     bbmin.getz() > abmin.getz() &&
     bbmin.getx() < abmin.getx()+x &&
     bbmin.gety() < abmin.gety()+y &&
     bbmin.getz() < abmin.getz()+z )
    return TRUE;

  if( bbmin.getx()+xx > abmin.getx() && 
     bbmin.gety() > abmin.gety() &&
     bbmin.getz() > abmin.getz() &&
     bbmin.getx()+xx < abmin.getx()+x &&
     bbmin.gety() < abmin.gety()+y &&
     bbmin.getz() < abmin.getz()+z )
    return TRUE;

  if( bbmin.getx() > abmin.getx() && 
     bbmin.gety()+yy > abmin.gety() &&
     bbmin.getz() > abmin.getz() &&
     bbmin.getx() < abmin.getx()+x &&
     bbmin.gety()+yy < abmin.gety()+y &&
     bbmin.getz() < abmin.getz()+z )
    return TRUE;

  if( bbmin.getx() > abmin.getx() && 
     bbmin.gety() > abmin.gety() &&
     bbmin.getz()+zz > abmin.getz() &&
     bbmin.getx() < abmin.getx()+x &&
     bbmin.gety() < abmin.gety()+y &&
     bbmin.getz()+zz < abmin.getz()+z )
    return TRUE;

  if( bbmin.getx()+xx > abmin.getx() && 
     bbmin.gety()+yy > abmin.gety() &&
     bbmin.getz() > abmin.getz() &&
     bbmin.getx()+xx < abmin.getx()+x &&
     bbmin.gety()+yy < abmin.gety()+y &&
     bbmin.getz() < abmin.getz()+z )
    return TRUE;

  if( bbmin.getx()+xx > abmin.getx() && 
     bbmin.gety() > abmin.gety() &&
     bbmin.getz()+zz > abmin.getz() &&
     bbmin.getx()+xx < abmin.getx()+x &&
     bbmin.gety() < abmin.gety()+y &&
     bbmin.getz()+zz < abmin.getz()+z )
    return TRUE;

  return FALSE;
}


/*
**  updateTable
**
*/

void bspGraph::updateTable( int a )
{
  static queue q;
  int v, w;
  adjNode * p;

  t[ a ].dist = 0;		// start node
  q.add( a );

  while( !q.isEmpty() )
    {
      v = q.get();
      p = adjList[ v ];
      while( p->next )
	{
	  w = p->next->data;
	  p = p->next;
	  if( t[w].dist == -1 )
	    {
	      t[w].dist = t[v].dist + 1;
	      t[w].p = v;
	      q.add( w );
	    }
	}
    }

}


/*
**  findPath : finds a path from node A to B. returns a pointer to a stack
**             or NULL if no path
**
*/

stack<int> * bspGraph::findPath( int a, int b )
{
  int v;

  if( t[a].dist != 0 )
    updateTable( a );


  if( t[b].p == -1 )
    return 0;

  stack<int> * r = new stack<int>;


  v = b;
  r->push( v );
  
  int length=0;
  while( v != a )
    {
      if( length++ > listSize )
	{
	  delete r;
	  return 0;
	}
      v = t[ v ].p;
      r->push( v );
    }

  return r;
      

}

/*
**  removeLink : takes out link from A to B
**
*/

int bspGraph::removeLink( int a, int b )
{

 if( a > listSize || b > listSize )
    return FALSE;

 adjNode * p = adjList[ a ];
 adjNode * q;

 while( p->next )
   {
     if( p->next->data == b )
       {
	 q = p->next->next;
	 delete p->next;
	 p->next = q;
	 return TRUE;
       }
     p = p->next;
   }

 return FALSE;

}

/*
**  isLink : returns TRUE iff A --> B is a link
**
*/

int bspGraph::isLink( int a, int b )
{
  if( a >= listSize || b >= listSize )
    return TRUE;
  if( a < 0 || b < 0 )
    return TRUE;

  adjNode * p = adjList[ a ];

  if( !p )
    {
      fprintf(stderr, "bspGraph::isLink(): problem\n");
      return FALSE;
    }

  while( p->next )
    {
      if( p->next->data == b )
	return TRUE;
      p = p->next;
    }

  return FALSE;

}

/*
**  addLink : links node A to B
**
*/

int bspGraph::addLink( int a, int b )
{
  if( a > listSize || b > listSize )
    return FALSE;
  
  if( isLink( a,b ) )
    return FALSE;

  adjNode * p = adjList[ a ];
  int i=0;

  if( !p )
    {
      fprintf(stderr,"bspGraph::addLink(): problem\n");
      return FALSE;
    }
  
  while( p->next )
    {
      p = p->next;
      i++;
    }

  printf("%d links from node %d\n", i, a );

  p->next = new adjNode( b );

  return TRUE;

}


/*
**  eraseGraph : deletes directed-graph stuff if it exists
**
*/

void bspGraph::eraseGraph()
{
  if( listSize <= 0 )
    return;
  
  adjNode *p, *q;
  int i;
  
  delete[] t;
  for( i=0; i < listSize; i++ )
    {
      p = adjList[i];
      while( p->next )
	{
	  q = p;
	  p = p->next;
	  delete q;
	}
      adjList[i]=0;
    }
  
  delete[] adjList;

  adjList=0;
  listSize=0;
  t = 0;
  
}

