/**********************************************************
Quake2 Cluster Project
An addition and modification of iD Software's Quake2 shared 
library sources which enable Quake2 servers running this 
library to interconnect other servers which speak the 
protocol developed under the Quake2 Cluster Project.
Copyright (c) 1998 Justin Randall and Todd Bliss

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; version 2
of the License.

This program is distributed in the hope that it will be 
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more 
details. You should have received a copy of the GNU General 
Public License along with this program; if not, write to 
the Free Software Foundation, Inc., 
59 Temple Place - Suite 330, 
Boston, MA  02111-1307, USA.
**********************************************************/
/**********************************************************
*	$Log: cluster_links.c,v $
*	Revision 1.5  1998/01/31 09:53:58  logic
*	Added RULES_ClientUserInfoChanged for access to client variables.
*	Fixed the password bug
*	Fixed the looping respawn/telefrag bug
*	Added Invincibility functions to RULES
*	Added ClientThink2 to RULES
*	Added RULES_ClientCmd
*	Added cmd invincible to RULES
*	Added ()'s around player names on broadcast messages
*	Added MOTD motd-mapname.txt
*	Changed invincibility fx
*
*	Revision 1.4  1998/01/29 23:26:32  logic
*	Fixed gibbable head bug.
*
*	Revision 1.3  1998/01/28 04:03:09  logic
*	I think the Big Fucking Memory Leak bug has been irradicated.
*
*	Fixed:
*	  ClusterListen, a persistant thread, was spinning ClusterMiniServer,
*	  a temporary worker thread, which would also spin some temporary and
*	  some semi-persistant threads. If those threads died, and ClusterMiniServer
*	  was no longer available, the allocated system resources would not clean
*	  up --hence the hosed servers in the morning.
*
*	  ToDo:
*	  Fix ClusterExit relinks from an entrance that is still sending keepalives.
*	   if the exit sends a keepalive to an exit that is not linked to it, then
*	   the exit should return a NACK to tell the damned thing to shutup and quit
*	   spamming the network.
*
*	  Fix the serverlist generation: it's sloppy, piggish, and doesn't take
*	    dead systems very well. Also slow and nowhere near real time. Flooding
*	    is a bad idea. I need to devise some link-state status update, perhaps
*	    by arrangning some peering capabilities --something like the "directly
*	    connected host" concept of OSPF --if a directly connect host dies, his
*	    peer(s) check their other directly connected hosts to see if they too
*	    are having problems, then a link-state advertisement (heh, LSA) can be
*	    propogated to the rest of the network advising the rest of the cluster
*	    servers that the system is not available. Likewise, when a peering
*	    arrangement is established, another LSA can advise the cluster that
*	    a new server is available. (Who said network geeks make bad programmers?)
*
*	  Add some RULES callbacks for cluster mod authors
*
*	  Add ClusterPlayerLocate() function to identify the location of a player
*	    in the cluster. (perhaps tracking his last known exit rather than
*	    flooding for his location?)
*
*	Revision 1.2  1998/01/26 23:49:52  logic
*	Autolog commenting added to original source files.
*
*	Found bugs but not yet fixed:
*	Windows95 system resources are consumed over a period of time.
*	Links die after a period of time ( > 12 hours)
*
**********************************************************/

/**********************************************************
	cluster_links.c
	handles aspects specific to link entities: think 
	functions, etc...
**********************************************************/

#include "udp.h"
#include "windows.h"
#include <stdlib.h>
#include "g_local.h"
#include "cluster_globs.h"

/**********************************************************
SP_LinkEntity(administrator, link name, exit or entrance)

	Creates a new link entity in the game. Usually in 
	response to a cmd spawn <exit|entrance>
**********************************************************/
void SP_LinkEntity(edict_t *admin, char *named, char link_type) {
	
	edict_t *newLink;	// New link entity 
	edict_t *dup;		// temp var to check for duplicate names
	vec3_t	forward, right, offset;	// used when spawning the entity
	trace_t trace;		// used when dropping the entity
	int err;			// generic error holder
	char master[64] = {"\0"};	// am I still using this?
	char myaddr[64] = {"\0"};	// and this?
	char *myname;		// need to malloc the space for the name...
	unsigned char linktype;	// exit or entrance?

	// Check to see if the name is too long
	if(strlen(named) > 15) {
		gi.cprintf(admin, PRINT_HIGH, "Link entity names must be\n less than 16 characters.\n");
		err = -3;
	}

	// Ahh yes, just in case something is really wrong with the world
	myname = (char *)malloc(16);
	if(myname == NULL) {
		_sleep(50);
		myname = (char *)malloc(16);
		if(myname == NULL) {
			printf("cluster_links.c:SP_LinkEntity():MALLOC FAILED!\n");
			err = -4;
		}
	}

	// Do it
	memset(myname, 0 , 16);
	if(ValidateAdmin(admin)) {
		err = 0;
		
		// Check for entities with the same name
		dup = NULL;
		while ((dup = G_Find (dup, FOFS(classname), "cluster_exit")) != NULL)  {
			if(stricmp(dup->targetname, named) == 0) {
				err = -2;
			}
		}
		while ((dup = G_Find (dup, FOFS(classname), "cluster_entrance")) != NULL) {
			if(stricmp(dup->targetname, named) == 0) {
				err = -2;
			}
		}
		strncpy(myname, named, strlen(named));	

		// Make the entity
		if(err == 0) {
			newLink = G_Spawn();
			VectorSet(newLink->mins, -15, -15, -15);
			VectorSet(newLink->maxs, 15, 15, 15);
			newLink->timeout = 0;
			newLink->inuse = TRUE;
			newLink->takedamage = DAMAGE_NO;
			newLink->mass = 0;
			newLink->movetype = MOVETYPE_NONE;
			newLink->solid = SOLID_NOT;
			newLink->target = NULL;
			newLink->targetname = myname;
			VectorCopy(admin->s.angles, newLink->s.angles);
			AngleVectors(admin->client->v_angle, forward, right, NULL);
			VectorSet(offset, 24, 0, -16);
			G_ProjectSource(admin->s.origin, offset, forward, right, newLink->s.origin);
			trace = gi.trace(admin->s.origin, newLink->mins, newLink->maxs, newLink->s.origin, admin, CONTENTS_SOLID);
			VectorCopy(trace.endpos, newLink->s.origin);
			if(link_type == LINK_EXIT) {
				linktype = PKT_EXIT_SET;
				newLink->s.modelindex = 0;
				newLink->classname = "cluster_exit";
				newLink->s.effects = 0;
				newLink->think = ClusterExit_think;
				newLink->nextthink = level.time + 5; 
			} else if (link_type == LINK_ENTRANCE) {
				// newLink->think = ClusterEntrance_think;
				linktype = PKT_ENTER_SET;
				newLink->classname = "cluster_entrance";
				newLink->s.modelindex = 0;
				
			}
			gi.linkentity (newLink);
			gi.centerprintf(admin, "Created link named %s\n", newLink->targetname);
			ClusterRegisterLink(newLink, linktype);
			if(SaveClusterLinksToDisk())
				printf("SERVER: Interlink status saved\n");
		} else if(err == -1) {
			gi.centerprintf(admin,"usage: \ncmd cluster_exit <named>\n\nIt must be linked to a \ncluster_enter with \nsame name on destination server\n");
		} else if(err == -2) {
			gi.centerprintf(admin, "An exit with that name\nalready exists!\nPlease delete it before \ncreating a new one with that name\n");
		}
	} else {
		gi.cprintf(admin, PRINT_HIGH, "You are not authenticated.\n");
	}
	return;
}

/**********************************************************
ClusterListLinks(client requesting the list)
**********************************************************/
void ClusterListLinks(edict_t *ent) {
	edict_t *link;
	long ltime;

	link = NULL;
	time(&ltime);
	if(ValidateAdmin(ent)) {  // Verify that the client is an administrator
		while((link = G_Find (link, FOFS(classname), "cluster_exit")) != NULL) {
			if(link->target != NULL) {
				if((link->timeout + 5 < ltime) || (link->timeout == 0)) {
					gi.cprintf(ent, PRINT_HIGH, "Exit link %s is waiting for it's peer entrance at %s to send a heartbeat\n", link->targetname, link->target);
				} else {
					gi.cprintf(ent, PRINT_HIGH, "Exit link %s to %s\n", link->targetname, link->target);
				}
			} else {
				gi.cprintf(ent, PRINT_HIGH, "Exit link %s is not yet linked to another server\n", link->targetname);
			}
		}
		 while((link = G_Find (link, FOFS(classname), "cluster_entrance")) != NULL) {
			gi.cprintf(ent, PRINT_HIGH, "Entrance link %s is ready to receive.\n", link->targetname);
		}

	}
}

/**********************************************************
ClusterLinkTo(who links, what, to what ip addr)

	This establishes a link between an exit and a receptive
	entrance on another server.
**********************************************************/
void ClusterLinkTo(edict_t *ent, char *name, char *dest) {
	char *mydest;
	int err, i;
	edict_t *link;
	ServerReqPacket ServerReq;
	HOSTENT *heAddr;
	struct sockaddr_in saAddr;
	
	err = 0;

	for(i=0;i<(int)strlen(dest); i++) {
		if(isalpha(dest[i])) {
			err = 1;
			break;
		}
	}
	
	if(err) {
		heAddr = gethostbyname(dest);
		if(heAddr == NULL) {
		} else {
			memcpy(&saAddr.sin_addr, heAddr->h_addr, heAddr->h_length);
			strcpy(dest, inet_ntoa(saAddr.sin_addr));
		}
	}

	link = NULL;
	err = 0;
	mydest = (char *)malloc(16);
	if(mydest == NULL) {
		_sleep(50);
		mydest = (char *)malloc(16);
		if(mydest == NULL) {
			printf("cluster_links.c:ClusterLinkTo():MALLOC FAILURE!\n");
			err = 1;
		}
	}
	memset(mydest, 0, 17);
	if(!err) {
		// Find the link entity
		err = 1;
		while((link = G_Find (link, FOFS(classname), "cluster_exit")) != NULL) {
			if(stricmp(link->targetname, name) ==0) {
				// Found it!
				strcpy(mydest, dest);
				link->target = mydest;
				// Registration
				/*	1. Send request to main referal server at destination
						a. include local comm miniserver port
					2. local miniserver receives an ACK
					3. miniserver calls a function that completes the link and
						activates this entity
				*/
				ServerReq.packetid = PKT_ENTER_GET;
				ClusterIntToBytes(G_LocalComPort, ServerReq.ms_port);
				strncpy(ServerReq.link_dest, link->targetname, 16);
				ClusterSendMSG(link->target, G_ReferalPort, (unsigned char *)&ServerReq, sizeof(ServerReqPacket), G_ClusterKey);
				err = 0;
				gi.cprintf(ent, PRINT_HIGH, "Exit %s linked to %s\n", link->targetname, link->target);
				if(SaveClusterLinksToDisk())
					printf("SERVER: Interlink status saved\n");
				ClusterServerListRequest();
				break;
			}
		}
	}
	if(err) {
		gi.cprintf(ent, PRINT_HIGH, "Could not find an exit named %s\n", name);
	}
	return;
}

/**********************************************************
	This is for exit entities that have a .target
	already set. They will attempt to relink to their 
	destination addres:entrance with the same targetname
	when this function is called. This packet is sent
	to a referal server, which then asks the local Q2
	server that contains the matching entrance to 
	send heartbeats to this exit via the network.
***********************************************************/
void ClusterReLink(edict_t *ent) {
	ServerReqPacket ServerReq;
	
	ServerReq.packetid = PKT_ENTER_GET;
	ClusterIntToBytes(G_LocalComPort, ServerReq.ms_port);
	strcpy(ServerReq.link_dest, ent->targetname);
	ClusterSendMSG(ent->target, G_ReferalPort, (unsigned char *)&ServerReq, sizeof(ServerReq), G_ClusterKey);
	return;
}

/**********************************************************
	ClusterEnterHeartBeat()

	This function is spun as a thread for each exit that
	links to it. It wakes up periodically and sends
	an update to a remote exit, telling it that it's alive
	and whether it will accept a player.
**********************************************************/
void ClusterEnterHeartBeat(RecvMSG *request) {
	ReqLinkHeartbeatPacket	remote;
	unsigned char AcceptPlayers;
	EntranceKeepAlive	heartbeat;
	char ipaddr[17];
	edict_t *entrance;
	int RegistrationCycle, err;

	RegistrationCycle = 0;
	err = 1;
	entrance = NULL;
	memcpy(&remote, request->cData, sizeof(ReqLinkHeartbeatPacket));
	heartbeat.packet_id = PKT_ENTER_KEEPALIVE;
	ClusterIntToBytes(G_LocalComPort, heartbeat.entrance_ms_port);
	ClusterIntToBytes(G_ServerPort, heartbeat.entrance_q2_port);
	strcpy(heartbeat.name, remote.name);
	ClusterBytesToDDIP(remote.addr, ipaddr);
	while((entrance = G_Find (entrance, FOFS(classname), "cluster_entrance")) != NULL) {
		if(!stricmp(entrance->targetname, remote.name)) {
			// Found the entrance
			err = 0;
			break;
		}
	}
	if(err) {
		G_ActiveThreads--;
		ExitThread(0);	// Didn't find a matching exit
	}
	while(!err) {
		if(!entrance->inuse) {
			err=1;
		}
		if(level.players < game.maxclients) {
			AcceptPlayers = 1;
		} else {
			AcceptPlayers = 0;
		}

		heartbeat.accept_players = AcceptPlayers;
		ClusterSendMSG(ipaddr, ClusterCharToUS(remote.remote_ms_port), (unsigned char *)&heartbeat, sizeof(EntranceKeepAlive), G_ClusterKey);
		RegistrationCycle++;
		if(RegistrationCycle > 5) {
			RegistrationCycle = 0;
			if(entrance->inuse) {
				ClusterRegisterLink(entrance, PKT_ENTER_SET);
			} else {
				err=1;
			}
		}
		_sleep(2000);
	}
	G_ActiveThreads--;
	ExitThread(0);
}

/**********************************************************
ClusterReceiveEnterKeepAlive

	This server has received an entrance's keepalive 
	datagram. Update our coresponding exit.
**********************************************************/
void ClusterReceiveEnterKeepAlive(RecvMSG *Packet) {
	// Update an exit entitie's message queue
	edict_t *ent;
	EntranceKeepAlive	heartbeat;
	long ltime;

	time(&ltime);
	memcpy(&heartbeat, Packet->cData, sizeof(EntranceKeepAlive));
	ent = NULL;
	while((ent = G_Find (ent, FOFS(classname), "cluster_exit")) != NULL) {
		if(stricmp(ent->targetname, heartbeat.name) == 0) {
			// Found our match
			break;
		}
	}
	if(ent != NULL && ent->target) {
		// Set WILL ACCEPT flag
		ent->timeout = ltime;	// If, when checking the status of the exit, the last heartbeat received is
								// > 30 seconds ago, then the exit shuts down.
		ent->remote_ms_port = ClusterCharToUS(heartbeat.entrance_ms_port);
		ent->remote_q2_port = ClusterCharToUS(heartbeat.entrance_q2_port);
	}

}


/**********************************************************
	CluisterExit_think(edict_t *)

	The meat of the cluster_exit entity work happens here.
	Special FX, detecting a client coming within range,
	sending messages to the Master to let it know that
	it is alive, etc...

	See comments in source for detailed info.
**********************************************************/
void ClusterExit_think(edict_t *self) {
	edict_t *ent;	// entites within range to affect
	int mset;	// Toggle to know whether to set model or not
	long ltime;
	// Check to see if our endpoint is alive
	time(&ltime);
	if((self->timeout + 5 < ltime) || (self->timeout == 0)) {
		// Not alive on the other side
		self->nextthink = level.time + 5;
		// If the exit is still showing, make it disappear
		if(self->s.modelindex == gi.modelindex("sprites/s_bfg1.sp2")) {
			gi.WriteByte (svc_temp_entity);
			gi.WriteByte (TE_BFG_BIGEXPLOSION);
			gi.WritePosition (self->s.origin);
			gi.multicast (self->s.origin, MULTICAST_PVS);
			self->s.modelindex = 0;
			self->s.effects = 0;
		}
		// Send a relink request every 10 seconds if we have a destination
		if(self->timeout + 15 < ltime && self->target != NULL) {
			ClusterReLink(self);
			self->timeout = ltime - 5; // reset timer
		}

		return;
	}

	self->nextthink = level.time + .3;	// This doesn't have to think hard
	ent = NULL;
	mset = 0;

	// Anybody within zapping range?
	while ((ent = findradius(ent, self->s.origin, 50)) != NULL) {
		if(ent->classname == "player") {	// did we find a player?
			// FX, search player entities, if they aren't close, then
			// turn of BFG effects, otherwise, set the effect
			ent->s.effects |= EF_BFG;
			ent->timeout = ltime + 5;
			ent->solid = SOLID_NOT;
			ent->takedamage = DAMAGE_NO;
			// Try to send player to my destination
			ClusterSendPlayer(ent, self); 
			gi.bprintf(PRINT_HIGH, "%s exited the level\n", ent->client->pers.netname);
		}
	}
	// Undo FX on players who's timeoute has exceeded
	while((ent = G_Find (ent, FOFS(classname), "player")) != NULL) {
		if(ent->timeout < ltime && !RULES_CheckInvincibility(ent)) {
			ent->s.effects = 0;
			ent->solid = SOLID_BBOX;
			ent->takedamage = DAMAGE_AIM;
		}
	}
	// Anybody in the area?
	while ((ent = findradius(ent, self->s.origin, 350)) != NULL) {
		if(ent->classname == "player") {	// is it a player?
			mset = 1;	// Toggle mset var to switch models
		}
	}
	
	// Did we find someone in the area?
	if(mset == 1) {
		// Yes, we did, and I need to switch models
		if(self->s.modelindex == 0) {	// If I am already seen, don't spam the clients with WriteBytes
			self->s.modelindex = gi.modelindex ("sprites/s_bfg1.sp2");
			gi.sound(self, CHAN_AUTO, gi.soundindex("misc/bigtele.wav"), 1, ATTN_NORM, 0);
			self->s.effects |= EF_BFG | EF_ANIM_ALLFAST;
			gi.WriteByte (svc_temp_entity);
			gi.WriteByte (TE_BFG_BIGEXPLOSION);
			gi.WritePosition (self->s.origin);
			gi.multicast (self->s.origin, MULTICAST_PVS);
		}
	} else {
		// No one is around, be invisible
		if(self->s.modelindex == gi.modelindex("sprites/s_bfg1.sp2")) {
			gi.WriteByte (svc_temp_entity);
			gi.WriteByte (TE_BFG_BIGEXPLOSION);
			gi.WritePosition (self->s.origin);
			gi.multicast (self->s.origin, MULTICAST_PVS);
		}
		self->s.modelindex = 0;
		self->s.effects = 0;
	}
}


/**********************************************************
	ClusterRemoveLink(edict_t *name)

	Removes a link entity from a server
**********************************************************/
void ClusterRemoveLink(char *name) {
	edict_t *ent;

	ent = NULL;
	while((ent = G_Find (ent, FOFS(classname), "cluster_exit")) != NULL) {
		if(!stricmp(ent->targetname, name)) {
			// Found the exit to remove
			G_FreeEdict(ent);
			if(SaveClusterLinksToDisk())
				printf("SERVER: Interlink status saved\n");

		}
	}
	while((ent = G_Find (ent, FOFS(classname), "cluster_entrance")) != NULL) {
		if(!stricmp(ent->targetname, name)) {
			// Found the entrance to remove
			// Set timeout to -1 so the heartbeat thread can
			// terminate cleanly --no need to spam the net with a 
			// dead entrance

			G_FreeEdict(ent);
			if(SaveClusterLinksToDisk())
				printf("SERVER: Interlink status saved\n");
		}
	}
	return;
}
