/**********************************************************
Quake2 Cluster Project
An addition and modification of iD Software's Quake2 shared 
library sources which enable Quake2 servers running this 
library to interconnect other servers which speak the 
protocol developed under the Quake2 Cluster Project.
Copyright (c) 1998 Justin Randall and Todd Bliss

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; version 2
of the License.

This program is distributed in the hope that it will be 
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more 
details. You should have received a copy of the GNU General 
Public License along with this program; if not, write to 
the Free Software Foundation, Inc., 
59 Temple Place - Suite 330, 
Boston, MA  02111-1307, USA.
**********************************************************/

/**********************************************************
*	$Log: cluster.c,v $
*	Revision 1.5  1998/01/31 09:53:56  logic
*	Added RULES_ClientUserInfoChanged for access to client variables.
*	Fixed the password bug
*	Fixed the looping respawn/telefrag bug
*	Added Invincibility functions to RULES
*	Added ClientThink2 to RULES
*	Added RULES_ClientCmd
*	Added cmd invincible to RULES
*	Added ()'s around player names on broadcast messages
*	Added MOTD motd-mapname.txt
*	Changed invincibility fx
*
*	Revision 1.4  1998/01/29 23:26:30  logic
*	Fixed gibbable head bug.
*
*	Revision 1.3  1998/01/28 04:03:07  logic
*	I think the Big Fucking Memory Leak bug has been irradicated.
*
*	Fixed:
*	  ClusterListen, a persistant thread, was spinning ClusterMiniServer,
*	  a temporary worker thread, which would also spin some temporary and
*	  some semi-persistant threads. If those threads died, and ClusterMiniServer
*	  was no longer available, the allocated system resources would not clean
*	  up --hence the hosed servers in the morning.
*
*	  ToDo:
*	  Fix ClusterExit relinks from an entrance that is still sending keepalives.
*	   if the exit sends a keepalive to an exit that is not linked to it, then
*	   the exit should return a NACK to tell the damned thing to shutup and quit
*	   spamming the network.
*
*	  Fix the serverlist generation: it's sloppy, piggish, and doesn't take
*	    dead systems very well. Also slow and nowhere near real time. Flooding
*	    is a bad idea. I need to devise some link-state status update, perhaps
*	    by arrangning some peering capabilities --something like the "directly
*	    connected host" concept of OSPF --if a directly connect host dies, his
*	    peer(s) check their other directly connected hosts to see if they too
*	    are having problems, then a link-state advertisement (heh, LSA) can be
*	    propogated to the rest of the network advising the rest of the cluster
*	    servers that the system is not available. Likewise, when a peering
*	    arrangement is established, another LSA can advise the cluster that
*	    a new server is available. (Who said network geeks make bad programmers?)
*
*	  Add some RULES callbacks for cluster mod authors
*
*	  Add ClusterPlayerLocate() function to identify the location of a player
*	    in the cluster. (perhaps tracking his last known exit rather than
*	    flooding for his location?)
*
*	Revision 1.2  1998/01/26 23:49:50  logic
*	Autolog commenting added to original source files.
*
*	Found bugs but not yet fixed:
*	Windows95 system resources are consumed over a period of time.
*	Links die after a period of time ( > 12 hours)
*
**********************************************************/
#include "udp.h"
#include "g_local.h"
#include "windows.h"
#include <stdlib.h>
#include "cluster_globs.h"


/**********************************************************
	stuffcmd(edict_t *, char *)

	Extremely simple function to send a command to a client
	for execution. This is called primarily for level
	transfers -- 
	stuffcmd(client_ent, "connect quake2.logic...")
**********************************************************/
void stuffcmd(edict_t *e, char *s) {
	gi.WriteByte(11);
	gi.WriteString(s);
	gi.unicast(e, true);
}


/**********************************************************
	This code is used to enable the default commands for 
	Perecli Manole's awesome hook code.

**********************************************************/
void StuffDefault(edict_t *ent) {
	stuffcmd(ent, "alias +hook \"cmd hook action; wait; cmd hook shrink\"\n");
	stuffcmd(ent, "alias -hook \"cmd hook stop\"\n");
	stuffcmd(ent, "alias +shrink \"cmd hook shrink\"\n");
	stuffcmd(ent, "alias -shrink \"cmd hook stop\"\n");
	stuffcmd(ent, "alias +grow \"cmd hook grow\"\n");
	stuffcmd(ent, "alias -grow \"cmd hook stop\"\n");
	stuffcmd(ent, "bind SHIFT +hook\n");
	stuffcmd(ent, "bind w +grow\n");
	stuffcmd(ent, "bind r +shrink\n");
}

/**********************************************************
	SaveClusterLinksToDisk()

	saves all of the cluster_exit entities to a file called
	interlinks-xxx.ent in the game directory. Usefull for 
	reloading by ClusterLoadLinks().

	This function is called whenever a cluster_exit is 
	updated (removed, added, etc...)
	Returns 1 on sucess, 0 on failure.
**********************************************************/
int SaveClusterLinksToDisk(void) {
	// Every time there is a change in the cluster_exit and cluster_enter
	// entities, save this stuff to a file so reloads keep cluster 
	// information
	edict_t *exit;
	FILE *links;
	int retval;
	char filename[64] = {"\0"};
	char map[32] = {"\0"};	// The name of the map we're running
	int t1, t2, t3;			// The various sizes of the data we are saving, prepended before the data
	if(level.mapname != NULL) {
		strcpy(map, level.mapname);
	} else {
		gi.bprintf(PRINT_HIGH, "Map not defined. Assuming base1\n");	// This should *NEVER* happen!
		strcpy(map, "base1");
	}
	if(gi.cvar("gamedir", NULL, 0) != NULL) {
		sprintf(filename, "%s/interlinks-%s.ent", gi.cvar("gamedir", NULL, 0)->string, map);
	} else {
		sprintf(filename, "cluster/interlinks-%s.txt", map);
	}
	retval = 0;
	exit = NULL;

	links = fopen(filename, "w");
	retval = 1;

	// Save the exits
	while ((exit = G_Find (exit, FOFS(classname), "cluster_exit")) != NULL) {
		// Got one, write it.
		if(exit->inuse == true) {	// Is it still inuse?
			t1 = strlen(exit->targetname);
			if(exit->target !=NULL)
				t2 = strlen(exit->target);
			else
				t2 = 1;
			t3 = strlen(exit->classname);
			fwrite(&t1, sizeof(int), 1, links);
			fwrite(exit->targetname, strlen(exit->targetname), 1, links);
			fwrite(&t2, sizeof(int), 1, links);
			if(t2 > 1)
				fwrite(exit->target, strlen(exit->target), 1, links);
			else
				fwrite("0", t2, 1, links);
			fwrite(&t3, sizeof(int), 1, links);
			fwrite(exit->classname, t3, 1, links);
			fwrite(exit->s.origin, sizeof(vec3_t), 1, links);
			fwrite(exit->s.angles, sizeof(vec3_t), 1, links);
		}
		
	}

	// now save the entrances
	while ((exit = G_Find (exit, FOFS(classname), "cluster_entrance")) != NULL) {
		// Got one, write it.
		if(exit->inuse == true) {	// Is it still inuse?
			t1 = strlen(exit->targetname);
			if(exit->target !=NULL)
				t2 = strlen(exit->target);
			else
				t2 = 1;
			t3 = strlen(exit->classname);
			fwrite(&t1, sizeof(int), 1, links);
			fwrite(exit->targetname, strlen(exit->targetname), 1, links);
			fwrite(&t2, sizeof(int), 1, links);
			if(t2 > 1)
				fwrite(exit->target, strlen(exit->target), 1, links);
			else
				fwrite("0", t2, 1, links);
			fwrite(&t3, sizeof(int), 1, links);
			fwrite(exit->classname, t3, 1, links);
			fwrite(exit->s.origin, sizeof(vec3_t), 1, links);
			fwrite(exit->s.angles, sizeof(vec3_t), 1, links);
		}
		
	}
	fclose(links);
	return retval;
};

/**********************************************************
	ClusterLoadLinks()

	Used to read a file called interkinl-xxx.ent from the
	game directory, where xxx is the name of the map these
	exits should live in.

	Called when a server initially loads to recover the 
	cluster_exit entities after a shutdown/crash

	Example (for doing loads on the fly), g_cmds.c perhaps?

    if(Q_stricmp (cmd, "load_exits") == 0 && ValidateAdmin(ent))
		ClusterLoadLinks();
	...
**********************************************************/
int ClusterLoadLinks(void) {
	edict_t *newLink;
	FILE *links;
	int retval;
	char filename[64] = {"\0"};
	char map[32] = {"\0"};
	unsigned char linktype;
	printf("SERVER: Loading Link Entities\n");
	if(level.mapname != NULL) {
		strcpy(map, level.mapname);
	} else {
		sprintf(map, "base1");
	}
	if(gi.cvar("gamedir", NULL, 0) != NULL) {
		sprintf(filename, "%s/interlinks-%s.ent", gi.cvar("gamedir", NULL, 0)->string, map);
	} else {
		sprintf(filename, "cluster/interlinks-%s.txt", map);
	}
	retval = 0;
	newLink = NULL;
	links = NULL;
	links = fopen(filename, "r");
	retval = 1;
	if(links != NULL) {
		while(! feof(links)) {
			if((newLink = PopulateExit(links)) != NULL) {
				if(!stricmp(newLink->classname, "cluster_exit")) {
					linktype = PKT_EXIT_SET;
					newLink->think = ClusterExit_think;
					newLink->nextthink = level.time + 5;	// let admin get away from exit and let 
														// this guy register with the master
					newLink->s.effects = 0;
					newLink->s.modelindex = 0;
				} else if(!stricmp(newLink->classname, "cluster_entrance")) {
					linktype = PKT_ENTER_SET;
				}
				newLink->takedamage = DAMAGE_NO;
				newLink->mass = 0;
				newLink->movetype = MOVETYPE_NONE;
				newLink->timeout = 0;
				newLink->solid = SOLID_NOT;
				gi.linkentity(newLink);

				ClusterRegisterLink(newLink, linktype);
				if(linktype = LINK_EXIT && newLink->target != NULL) {
					ClusterReLink(newLink);
				}
			}
		}
		fclose(links);
	} else {
		retval = 0;
	}
	return retval;
};

/**********************************************************
edict_t *PopulateExit(FILE *f)

  Returns NULL if the file does not contain a 
	cluster_exit entity, otherwise allocates the entity and
	returns a pointer to the allocated entity.

	It is currently implemented in ClusterLoadLinks to
	restore all of the cluster_exits defined before a 
	server restart.
	
	Example:
	
	FILE *f;
	edict_t *temp_edict;

	f = fopen("myexits.dat","r");
	if((temp_edict = PopulateExit(f)) != NULL) {
		// do some stuff with the exit
	}
	...
**********************************************************/
edict_t *PopulateExit(FILE *f) {
	edict_t *ent;
	int t1, t2, t3;
	char *target;
	char *targetname;
	char *classname;

	target = (char *)malloc(64);
	if(target == NULL) {
		_sleep(50);
		target = (char *)malloc(64);
		if(target == NULL) {
			printf("cluster.c:PopulateExit:MEMORY ALLOCATION FAILURE\n");
			return NULL; // Yes, I know, bad form
		}
	}
	
	targetname = (char *)malloc(64);
	if(targetname == NULL) {
		_sleep(50);
		targetname = (char *)malloc(64);
		if(targetname == NULL) {
			printf("cluster.c:PopulateExit:MEMORY ALLOCATION FAILURE\n");
			free(target);
			return NULL; // see above
		}
	}
	
	classname = (char *)malloc(64);
	if(classname == NULL) {
		printf("cluster.c:PopulateExit:MEMORY ALLOC FAILURE!\n");
		free(target);
		free(targetname);
		return NULL;
	}
	ent = NULL;
	ent = G_Spawn();
	fread(&t1, sizeof(int), 1, f);
	fread(targetname, t1, 1, f);
	fread(&t2, sizeof(int),1, f);
	fread(target, t2, 1, f);
	fread(&t3, sizeof(int), 1, f);
	fread(classname, t3, 1, f);
	fread(ent->s.origin, sizeof(vec3_t), 1, f);
	fread(ent->s.angles, sizeof(vec3_t), 1, f);
	
	ent->targetname = targetname;
	if(t2 == 1) {
		ent->target = NULL;
	} else {
		ent->target = target;
	}
	ent->classname = classname;
	ent->inuse = true;
	if(strlen(ent->targetname) > 0) {
		return ent;
	}
	free(target);
	free(targetname);
	G_FreeEdict (ent);
	return NULL;
}

/**********************************************************
	ValidateAdmin(edict_t *)

	This function checks if the client is an adminstrator.
	It looks at the client's password variable, and matches
	it against the servers admin_password variable. Returns
	1 if they match, 0 if they don't.
	
	Example:
	if(ValidateAdmin(client)) {
	  // Doing Admin stuff
	}
**********************************************************/
int ValidateAdmin(edict_t *admin) {
	char c_password[32] = {"\0"};
	cvar_t *admin_password;
	char s_password[32] = {"\0"};
	int retval;

	retval = 0;
	if(gi.cvar("admin_password", NULL, 0) == NULL) {
		gi.cvar_set("admin_password", "logic");
	}
	admin_password = gi.cvar("admin_password", NULL, 0);
	strcpy(s_password, admin_password->string);
	sprintf(c_password, "%s\0", admin->client->pers.password);
	if(strlen(c_password) && c_password != NULL) {
		if(stricmp(c_password, s_password) == 0) {
				retval = 1;
		}
	}
	return retval;
}

void ClusterCopying(void) {
	printf ("=====================================================================\n");
	printf ("===          -=Quake2 Cluster Project, Version 0.3=-              ===\n");
	printf ("===       Some portions Copyright(c) 1998 Justin Randall          ===\n");
	printf ("=== See COPYING.TXT, included with this distribution, for details ===\n");
	printf ("===             http://www.planetquake.com/logic                  ===\n");
	printf ("=====================================================================\n");
	printf ("Build %s\n\n", G_ClusterBuild);
}

void StringToNull(char *buf) {
	memset(buf, '\0', strlen(buf));
};


void ClusterInit(void) {
	char	temp[64];
	HOSTENT *heAddr;
	int i;

	// Set local IP Address
	// The first Q2 Server Spawned starts a registration/referral server,
	// All exits in Q2 servers on this machine register with
	// the referal server.
	G_LastRequestID = 0;	// Initialize the list request identifier
	// Initialize the server list database
	for(i = 0; i <= CLUSTER_MAX_SERVERS; i++) {
		G_ServerList[i].ms_port[0] = 0;
		G_ServerList[i].ms_port[1] = 0;
	}
	gethostname(temp, 64);
	if(gethostbyname(temp) != NULL) {
		heAddr = gethostbyname(temp);
		memcpy(&G_ServerAddr, heAddr->h_addr, sizeof(G_ServerAddr));	
	}
	// Get port 
	if(gi.cvar("cluster_port", NULL, 0) != NULL) {
		G_ReferalPort = atoi(gi.cvar("cluster_port", NULL, 0)->string);
	} else {
		G_ReferalPort = SERVER_PORT;
	}
	if(gi.cvar("port", NULL, 0) != NULL) {
		G_ServerPort = atoi(gi.cvar("port", NULL, 0)->string);
	} else {
		G_ServerPort = 27910;
	}

	// Set up our id/password keypair
	G_ClusterId = (char *)malloc(64);
	if(G_ClusterId == NULL) {
		_sleep(50);
		G_ClusterId = (char *)malloc(64);
		if(G_ClusterId == NULL) {
			printf("cluster.c:ClusterInit:MALLOC FAILURE!\n");
			exit(-1);
		}
	}
	G_ClusterKey = (char *)malloc(64);
	if(G_ClusterKey == NULL) {
		_sleep(50);
		G_ClusterKey = (char *)malloc(64);
		if(G_ClusterKey == NULL) {
			printf("cluster.c:ClusterInit:MALLOC FAILURE\n");
			exit(-1);
		}
	}
	if(gi.cvar("cluster_id", NULL, 0) != NULL) {
		strcpy(G_ClusterId, gi.cvar("cluster_id", NULL, 0)->string);
	} else {
		strcpy(G_ClusterId, "default");
		printf("SECURITY: Warning: cluster_id was unset, value is now \"%s\"\n", G_ClusterId);
	}
	if(gi.cvar("cluster_key", NULL, 0) != NULL) {
		strcpy(G_ClusterKey, gi.cvar("cluster_key", NULL, 0)->string);
	} else {
		strcpy(G_ClusterKey, "changeme");
		printf("SECURITY: Warning: cluster_key was unset, value is now \"%s\"\n", G_ClusterKey);
	}
	gi.cvar_set("cluster_id", G_ClusterId);
	gi.cvar_set("cluster_key", G_ClusterKey);

	SpinDBThread();
	ClusterSpinNetServer();
	_sleep(6000);	// Let database initialize
	ClusterLoadLinks();

}

int ClusterRegisterLink(edict_t *exit, char pkt_id) {
	char	ServerAddr[64] = {"\0"};	// Registration Server Addr
	ClusterLinkRegPacket	Packet;

	memset(&Packet, 0 , sizeof(Packet));
	Packet.packetid = pkt_id;
	// calculate port bytes (worried about cross-platform stuff here)
	ClusterIntToBytes(G_ServerPort, Packet.q2_port);
	ClusterIntToBytes(G_LocalComPort, Packet.ms_port);
	strcpy(ServerAddr, inet_ntoa(G_ServerAddr));
	strncpy(Packet.name, exit->targetname, strlen(exit->targetname));
	ClusterSendMSG(ServerAddr, G_ReferalPort, (unsigned char *)&Packet, sizeof(ClusterLinkRegPacket), G_ClusterKey);
	return 1;
};


/**********************************************************
DEBUG_PrintBytes(binary array of data, size of the array)

	I use this to take a look at exactly what data is 
	in a binary chunk of data, typically in the protocl, 
	strictly for debugging.
**********************************************************/
void DEBUG_PrintBytes(unsigned char *bytes, long len) {
	long i;
	printf("DEBUG:bytes\n");
	for(i=0; i<=len; i++) {
		printf("[%d]\t", bytes[i]);
	}
	printf("\n\n");
}

/**********************************************************
ClusterSelectSpawnPoint(origin, angles, exit that sent client)

	Used to find the corresponding entrance entity that
	this player should spawn at. If it's not found, then
	just select a DM spawn point.
**********************************************************/
void ClusterSelectSpawnPoint(vec3_t spawn_origin, vec3_t spawn_angles, char *ExitName) {
	edict_t *entrance;

	entrance = NULL;
	while ((entrance = G_Find(entrance, FOFS(classname), "cluster_entrance")) != NULL) {
		if(!stricmp(entrance->targetname, ExitName)) {
			VectorCopy (entrance->s.origin, spawn_origin);
			spawn_origin[2] += 9;
			VectorCopy (entrance->s.angles, spawn_angles);
			break;
		}
	}
}

/**********************************************************
ClusterCMDListServers(client receiving list)

	Runs through the in-memory server listing and tells the
	client which servers are on this cluster.
**********************************************************/
void ClusterCMDListServers(edict_t *ent) {
	int i, q2port, msport;
	char addr[17];

	for(i = 0; i <= CLUSTER_MAX_SERVERS; i++) {
		if(G_ServerList[i].q2_port[0] > 0 || G_ServerList[i].q2_port[1] > 0) {
			q2port = ClusterCharToUS(G_ServerList[i].q2_port);
			msport = ClusterCharToUS(G_ServerList[i].ms_port);
			ClusterBytesToDDIP(G_ServerList[i].addr, addr);
			gi.cprintf(ent, PRINT_HIGH, "%s:%d:%d\n", addr, q2port, msport);
		}
	}
}