// drOpenGL.dll
// The Doomsday graphics library driver for OpenGL
//
// list.c : List rendering

#include "drOpenGL.h"


// This is only for solid, non-masked primitives.
void renderList(rendlist_t *rl)
{
	int			i;
	float		tcleft, tcright, tctop, tcbottom;
	rendquad_t	*cq;

	if(!rl->numquads) return;	// The list is empty.

	// Bind the right texture.
	glBindTexture(GL_TEXTURE_2D, currentTex=rl->tex);

	// Check what kind of primitives there are on the list.
	// There can only be one kind on each list.
	if(rl->quads->flags & RQF_FLAT)	// Check the first primitive.
	{
		// There's only triangles here, I see.				
		glBegin(GL_TRIANGLES);
		for(i=0; i<rl->numquads; i++)
		{
			cq = rl->quads+i;
			if(cq->flags & RQF_MISSING_WALL)
			{
				// This triangle is REALLY a quad that originally had no 
				// texture. We have to render it as two triangles.
				tcright = (tcleft=0) + cq->len/cq->texw;
				tcbottom = (tctop=0) + (cq->top - cq->bottom)/cq->texh;
				
				glColor3ubv(cq->color[0].rgb);
				glTexCoord2f(tcleft, tctop);
				glVertex3f(cq->v1[VX], cq->top, cq->v1[VY]);

				glColor3ubv(cq->color[1].rgb);
				glTexCoord2f(tcright, tctop);
				glVertex3f(cq->v2[VX], cq->top, cq->v2[VY]);

				glTexCoord2f(tcright, tcbottom);
				glVertex3f(cq->v2[VX], cq->bottom, cq->v2[VY]);

				// The other triangle.
				glTexCoord2f(tcright, tcbottom);
				glVertex3f(cq->v2[VX], cq->bottom, cq->v2[VY]);

				glColor3ubv(cq->color[0].rgb);
				glTexCoord2f(tcleft, tcbottom);
				glVertex3f(cq->v1[VX], cq->bottom, cq->v1[VY]);

				glTexCoord2f(tcleft, tctop);
				glVertex3f(cq->v1[VX], cq->top, cq->v1[VY]);
				continue;
			}

			// The vertices.
			glColor3ubv(cq->color[0].rgb);
			glTexCoord2f((cq->v1[VX]+cq->texoffx)/cq->texw,
				(cq->v1[VY]+cq->texoffy)/cq->texh);
			glVertex3f(cq->v1[VX], cq->top, cq->v1[VY]);

			glColor3ubv(cq->color[1].rgb);
			glTexCoord2f((cq->v2[VX]+cq->texoffx)/cq->texw,
				(cq->v2[VY]+cq->texoffy)/cq->texh);
			glVertex3f(cq->v2[VX], cq->top, cq->v2[VY]);

			glColor3ubv(cq->color[2].rgb);
			glTexCoord2f((cq->v3[VX]+cq->texoffx)/cq->texw,
				(cq->v3[VY]+cq->texoffy)/cq->texh);
			glVertex3f(cq->v3[VX], cq->top, cq->v3[VY]);
		}
		glEnd();
	}
	else
	{
		// Render quads.
		glBegin(GL_QUADS);
		for(i=0; i<rl->numquads; i++)
		{
			cq = rl->quads+i;

			// Calculate relative texture coordinates.
			tcright = (tcleft=cq->texoffx/(float)cq->texw) + cq->len/cq->texw;
			tcbottom = (tctop=cq->texoffy/cq->texh) + (cq->top - cq->bottom)/cq->texh;

			// The vertices.
			glColor3ubv(cq->color[0].rgb);
			glTexCoord2f(tcleft, tcbottom);
			glVertex3f(cq->v1[VX], cq->bottom, cq->v1[VY]);

			glTexCoord2f(tcleft, tctop);
			glVertex3f(cq->v1[VX], cq->top, cq->v1[VY]);

			glColor3ubv(cq->color[1].rgb);
			glTexCoord2f(tcright, tctop);
			glVertex3f(cq->v2[VX], cq->top, cq->v2[VY]);

			glTexCoord2f(tcright, tcbottom);
			glVertex3f(cq->v2[VX], cq->bottom, cq->v2[VY]);
		}
		glEnd();
	}
}

// Masked lists only include quads.
void renderMaskedList(rendlist_t *mrl)
{
	int			i;
	float		tcleft, tcright, tctop, tcbottom;
	rendquad_t	*cq;
	
	if(!mrl->numquads) return;	// No quads to render, I'm afraid.

	// Curtex is used to keep track of the current texture.
	// Zero also denotes that no glBegin() has yet been called.
	currentTex = 0;

	// Render quads.
	for(i=mrl->numquads-1; i>=0; i--)	// Render back to front.
	{
		cq = mrl->quads+i;

		// Calculate relative texture coordinates.
		tcright = (tcleft=cq->texoffx/cq->texw) + cq->len/cq->texw;
		tcbottom = (tctop=cq->texoffy/cq->texh) + (cq->top - cq->bottom)/cq->texh;

		// Is there a need to change the texture?
		if(currentTex != cq->masktex)	
		{
			if(currentTex) glEnd();	// Finish with the old texture.
			glBindTexture(GL_TEXTURE_2D, currentTex=cq->masktex);
			glBegin(GL_QUADS);	// I love OpenGL.
		}
		
		// The vertices.
		glColor3ubv(cq->color[0].rgb);
		glTexCoord2f(tcleft, tcbottom);
		glVertex3f(cq->v1[VX], cq->bottom, cq->v1[VY]);

		glTexCoord2f(tcleft, tctop);
		glVertex3f(cq->v1[VX], cq->top, cq->v1[VY]);

		glColor3ubv(cq->color[1].rgb);
		glTexCoord2f(tcright, tctop);
		glVertex3f(cq->v2[VX], cq->top, cq->v2[VY]);

		glTexCoord2f(tcright, tcbottom);
		glVertex3f(cq->v2[VX], cq->bottom, cq->v2[VY]);
	}
	if(currentTex) glEnd();	// If something was drawn, finish with it.
}

void renderSkyMaskLists(rendlist_t *smrl, rendlist_t *skyw)
{
	int			i;
	rendquad_t	*cq;

	if(!smrl->numquads && !skyw->numquads) return; // Nothing to do here.

	// Nothing gets written to the color buffer.
	//glBindTexture(GL_TEXTURE_2D, 0);
	// Instead of setting texture 0 we'll temporarily disable texturing.
	glDisable(GL_TEXTURE_2D);
	// This will effectively disable color buffer writes.
	glBlendFunc(GL_ZERO, GL_ONE);

	if(smrl->numquads)
	{
		glBegin(GL_TRIANGLES);
		for(i=0; i<smrl->numquads; i++)
		{
			cq = smrl->quads+i;
			// ONLY the vertices, please.
			glVertex3f(cq->v1[VX], cq->top, cq->v1[VY]);
			glVertex3f(cq->v2[VX], cq->top, cq->v2[VY]);
			glVertex3f(cq->v3[VX], cq->top, cq->v3[VY]);
		}
		glEnd();		
	}

	// Then the walls.
	if(skyw->numquads)
	{
		glBegin(GL_QUADS);
		for(i=0; i<skyw->numquads; i++)
		{
			cq = skyw->quads + i;
			// Only the verts.
			glVertex3f(cq->v1[VX], cq->bottom, cq->v1[VY]);
			glVertex3f(cq->v1[VX], cq->top, cq->v1[VY]);
			glVertex3f(cq->v2[VX], cq->top, cq->v2[VY]);
			glVertex3f(cq->v2[VX], cq->bottom, cq->v2[VY]);
		}
		glEnd();
	}
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_TEXTURE_2D);
}

void renderDynLightLists(rendlist_t *frl, rendlist_t *wrl)
{
	int			i;
	rendquad_t	*cq;
	GLfloat		projmat[16];

	if(!frl->numquads && !wrl->numquads) return; // Nothing to do.

	// Adjust the projection matrix for Z bias.
	glGetFloatv(GL_PROJECTION_MATRIX, projmat);
	glMatrixMode(GL_PROJECTION);
	//glPushMatrix();
	//glLoadIdentity();
	//glTranslatef(0, 0, -.00009f);
	glScalef(.99f, .99f, .99f);
	//glMultMatrixf(projmat);

	// Setup the correct rendering state.	
	// Disable fog.
	if(usefog) glDisable(GL_FOG);

	// This'll allow multiple light quads to be rendered on top of each other.
	glDepthMask(GL_FALSE); 
	glDepthFunc(GL_LEQUAL);
	// Set up addition blending. The source is added to the destination.
	if(!dlBlend) // The normal setting.
		glBlendFunc(GL_DST_COLOR, GL_ONE);
	else if(dlBlend == 1)
		glBlendFunc(GL_ONE, GL_ONE);

	// The light texture.
	glBindTexture(GL_TEXTURE_2D, currentTex = lightTex);

	// The flats.
	if(frl->numquads)
	{
		glBegin(GL_TRIANGLES);
		for(i=0; i<frl->numquads; i++)
		{
			cq = frl->quads + i;
			// Set the color.
			glColor3ubv(cq->color[0].rgb);

			// The vertices.			
			glTexCoord2f((cq->texoffx - cq->v1[VX])/cq->texw,
				(cq->texoffy - cq->v1[VY])/cq->texh);
			glVertex3f(cq->v1[VX], cq->top, cq->v1[VY]);

			glTexCoord2f((cq->texoffx - cq->v2[VX])/cq->texw,
				(cq->texoffy - cq->v2[VY])/cq->texh);
			glVertex3f(cq->v2[VX], cq->top, cq->v2[VY]);

			glTexCoord2f((cq->texoffx - cq->v3[VX])/cq->texw,
				(cq->texoffy - cq->v3[VY])/cq->texh);
			glVertex3f(cq->v3[VX], cq->top, cq->v3[VY]);
		}
		glEnd();
	}	

	// The walls.
	if(wrl->numquads)
	{
		float tctl[2], tcbr[2];	// Top left and bottom right.
		glBegin(GL_QUADS);
		for(i=0; i<wrl->numquads; i++)
		{
			cq = wrl->quads + i;
			// Set the color.
			glColor3ubv(cq->color[0].rgb);

			// Calculate the texture coordinates.
			tcbr[VX] = (tctl[VX]=-cq->texoffx/cq->texw) + cq->len/cq->texw;
			tcbr[VY] = (tctl[VY]=cq->texoffy/cq->texh) + (cq->top - cq->bottom)/cq->texh;

			// The vertices.
			glTexCoord2f(tctl[VX], tcbr[VY]);
			glVertex3f(cq->v1[VX], cq->bottom, cq->v1[VY]);

			glTexCoord2f(tctl[VX], tctl[VY]);
			glVertex3f(cq->v1[VX], cq->top, cq->v1[VY]);

			glTexCoord2f(tcbr[VX], tctl[VY]);
			glVertex3f(cq->v2[VX], cq->top, cq->v2[VY]);

			glTexCoord2f(tcbr[VX], tcbr[VY]);
			glVertex3f(cq->v2[VX], cq->bottom, cq->v2[VY]);
		}
		glEnd();
	}

//	glPopMatrix();
	glLoadMatrixf(projmat);

	// Restore the normal rendering state.
	if(usefog) glEnable(GL_FOG);
	glDepthMask(GL_TRUE); 
	glDepthFunc(GL_LESS);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
}
